/*
 * @Author: jack.lu
 * @Date: 2020-4-21 10:10:20
 * @Last Modified by: jack.lu
 * @Last Modified time: 2020-4-21 15:01:41
 */

import GoEasy from 'goeasy';
import RestApi from './restapi';

function Friend(uuid, username, avatar) {
    this.uuid = uuid;
    this.username = username;
    this.avatar = avatar;
    this.online = false;
    this.unReadMessage = 0;
}

function CurrentUser(uuid,username,avatar){
    this.uuid = uuid;
    this.username = username;
    this.avatar = avatar;
}

function ChatMessage(senderUUID, content) {
    this.senderUUID = senderUUID;
    this.content = content;
}


function IMService() {
    this.goeasy = null;

    this.currentUser = null;

    this.friendList = [];

    this.messages = [];
}

/**
 * 初始化
 */
IMService.prototype.initialGoEasy = function () {
    this.goeasy = new GoEasy({
        appkey: "",
        host: "hangzhou.goeasy.io",
        userId: this.currentUser.uuid,
        userData: '{"username":"' + this.currentUser.username + '","avatar":"' + this.currentUser.avatar + '"}',
        onConnected: function () {
            console.log("GoEasy connect successfully.")
        },
        onDisconnected: function () {
            console.log("GoEasy disconnected.")
        },
        onConnectFailed: function (error) {
            console.log("GoEasy连接失败，请确认imservice.js文件48行appkey和host配置正确.");
        }
    });
    this.initialFriendList();
    this.subscribePresence();
    this.subscribe();
};

/**
 * 初始化好友列表
 */
IMService.prototype.initialFriendList = function () {
    //好友列表来源后台数据，这儿临时模拟
    let friends = RestApi.findFriends(this.currentUser.uuid);
    let friendList = [];
    friends.forEach((friendItem) => {
        let friend = new Friend(friendItem.friendUUID, friendItem.username, friendItem.avatar);
        friendList.push(friend);
    });
    this.friendList = friendList;
    this.initialFriendOnlineStatus();
};

/**
 * 初始化好友在线状态
 */
IMService.prototype.initialFriendOnlineStatus = function () {
    let self = this;
    this.goeasy.hereNowByUserIds({
        userIds: getFriendUUIDs(this.friendList)
    }, function (result) {
        if (result.code == 200) {
            result.content.forEach(function (user) {
                let userIndex = self.friendList.findIndex(item => item.uuid == user.userId);
                self.friendList[userIndex].online = true;
            });
        }
        if (result.code == 401) {
            console.log("您还没有高级功能的权限，付费用户请联系GoEasy开通")
        }
    });
};

/**
 * 接收消息
 */
IMService.prototype.subscribe = function () {
    let self = this;
    this.goeasy.subscribe({
        channel: this.currentUser.uuid, //替换为您自己的channel
        onMessage: function (message) {
            let content = JSON.parse(message.content);
            let userIndex = self.friendList.findIndex(item => item.uuid == content.senderUUID);
            self.friendList[userIndex].unReadMessage++;
            //todo:事实上不推荐在前端收到时保存, 一个用户开多个窗口，会导致重复保存, 建议所有消息都是都在发送时在服务器端保存，这里只是为了演示
            RestApi.saveChatMessage(self.currentUser.uuid,content.senderUUID, content);
            let myMessage = new ChatMessage(content.senderUUID, content.content);
            self.messages.push(myMessage)
        }
    });
};

/**
 * 监听所有好友上下线
 */
IMService.prototype.subscribePresence = function () {
    let self = this;
    this.goeasy.subscribePresence({
        channels: getFriendUUIDs(self.friendList),
        onPresence: function (presenceEvents) {
            presenceEvents.events.forEach(function (user) {
                let userIndex = self.friendList.findIndex(item => item.uuid == user.userId);
                if (user.action == "join" || user.action == "online") {
                    self.friendList[userIndex].online = true;
                } else {
                    self.friendList[userIndex].online = false;
                }
            });
        },
        onSuccess: function () {
            console.log("监听上下线成功.");
        }
    });
};

/**
 * 发送消息
 * @param message
 */
IMService.prototype.sendMessage = function (message) {
    let chatMessage = new ChatMessage(message.currentUserUUID, message.content);
    this.messages.push(chatMessage);
    RestApi.saveChatMessage(message.currentUserUUID,message.friendUUID, chatMessage);
    let chatMessageAsJsonString = JSON.stringify(chatMessage);
    this.goeasy.publish({
        channel: message.friendUUID,
        message: chatMessageAsJsonString
    });
};

/**
 * 校验用户
 * @param username
 * @param password
 * @returns {{password: string, avatar: *, uuid: string, username: string}|{password: string, avatar: *, uuid: string, username: string}|{password: string, avatar: *, uuid: string, username: string}|{password: string, avatar: *, uuid: string, username: string}|null}
 */
IMService.prototype.login = function(username, password) {
    let user = RestApi.findUser(username, password);
    this.currentUser = new CurrentUser(user.uuid, user.username, user.avatar);
    return user
};

/**
 * 加载历史数据
 * @param friend
 */
IMService.prototype.loadHistoryMessage = function (friend) {
    this.messages  = RestApi.findChatHistory(this.currentUser.uuid, friend.uuid)
};

/**
 * 重置未读消息
 * @param friendUUID
 */
IMService.prototype.resetFriendUnReadMessage = function (friendUUID) {
    let friendIndex = this.friendList.findIndex(item => item.uuid == friendUUID);
    if(friendIndex > -1){
        this.friendList[friendIndex].unReadMessage = 0;
    }
};

/**
 * 获取好友UUID
 * @param friendList
 * @returns {[]}
 */
function getFriendUUIDs (friendList) {
    let friendUUIDs = [];
    friendList.forEach(function (friend) {
        friendUUIDs.push(friend.uuid);
    });
    return friendUUIDs;
}


export default IMService;