// Copyright (C) 2021 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.gerrit.acceptance;

import com.google.gerrit.metrics.Counter0;
import com.google.gerrit.metrics.Description;
import com.google.gerrit.metrics.DisabledMetricMaker;
import com.google.inject.Singleton;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang.mutable.MutableLong;

/**
 * {@link com.google.gerrit.metrics.MetricMaker} to be bound in tests.
 *
 * <p>Records how often {@link Counter0} metrics are invoked. Metrics of other types are not
 * recorded.
 *
 * <p>Allows test to check how much a {@link Counter0} metrics is increased by an operation.
 *
 * <p>Example:
 *
 * <pre>
 * public class MyTest extends AbstractDaemonTest {
 *   @Inject private TestMetricMaker testMetricMaker;
 *
 *   ...
 *
 *   @Test
 *   public void testFoo() throws Exception {
 *     testMetricMaker.reset();
 *     doSomething();
 *     assertThat(testMetricMaker.getCount("foo/bar_count")).isEqualsTo(1);
 *   }
 * }
 * </pre>
 */
@Singleton
public class TestMetricMaker extends DisabledMetricMaker {
  private final Map<String, MutableLong> counts = new HashMap<>();

  public long getCount(String counter0Name) {
    return get(counter0Name).longValue();
  }

  public void reset() {
    counts.clear();
  }

  private MutableLong get(String counter0Name) {
    return counts.computeIfAbsent(counter0Name, name -> new MutableLong(0));
  }

  @Override
  public Counter0 newCounter(String name, Description desc) {
    return new Counter0() {
      @Override
      public void incrementBy(long value) {
        get(name).add(value);
      }

      @Override
      public void remove() {}
    };
  }
}
