// Copyright (C) 2016 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.gerrit.server.project;

import com.google.auto.value.AutoValue;

/**
 * Stable identifier for options passed to a particular submit rule evaluator.
 *
 * <p>Used to test whether it is ok to reuse a cached list of submit records. Does not include a
 * change or patch set ID; callers are responsible for checking those on their own.
 */
@AutoValue
public abstract class SubmitRuleOptions {
  private static final SubmitRuleOptions defaults =
      new AutoValue_SubmitRuleOptions.Builder().recomputeOnClosedChanges(false).build();

  public static SubmitRuleOptions defaults() {
    return defaults;
  }

  public static Builder builder() {
    return defaults.toBuilder();
  }

  /**
   * True if the submit rules should be recomputed even when the change is already closed (merged).
   */
  public abstract boolean recomputeOnClosedChanges();

  public abstract Builder toBuilder();

  @AutoValue.Builder
  public abstract static class Builder {
    public abstract SubmitRuleOptions.Builder recomputeOnClosedChanges(boolean allowClosed);

    public abstract SubmitRuleOptions build();
  }
}
