/**
 * @license
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * This is a command-line tool to calculate licenses for files in npm packages.
 * The tool expected the following inputs:
 *
 * config.js - information about licenses for packages. The js file must be a module
 *   and must have array of {@link PackageInfo} as a default export.
 *
 * node-modules-files.txt - list of files (one per line) to calculate license.
 *   Each line must be a path to a file inside node_module directory
 *   (full path or path relative to current work dir)
 *
 * shared-licenses.txt - list of files (one per line) with licenses texts. Each line is a path
 *   to a file with license text. The files can be referenced by {@link SharedFileLicenseInfo}
 *
 * json-output.json - output file name. The {@link LicensesMap} defines the schema of this file.
 *
 * Note: It is expected, that config is compiled from .ts file end has default export.
 * Typescript compiler checks that .ts file matches PackageInfo interface, so no additional
 * validations are needed.
 */

import * as path from "path";
import * as fs from "fs";
import {PackageInfo} from "./package-license-info";
import {fail, readMultilineParamFile} from "./utils";
import {LicenseMapGenerator} from "./licenses-map";
import {SharedLicensesProvider} from "./shared-licenses-provider";

interface LicenseMapCommandLineArgs {
  generatorParams: LicenseMapGeneratorParameters;
  outputJsonPath: string;
}

export interface LicenseMapGeneratorParameters {
  sharedLicensesFiles: string[];
  nodeModulesFiles: string[];
  packages: PackageInfo[];
}

function parseArguments(argv: string[]): LicenseMapCommandLineArgs {
  if(argv.length < 6) {
    fail("Invalid command line parameters\n" +
        "\tUsage:\n\tnode license-map-generator config.js node-modules-files.txt shared-licenses.txt json-output.json");
  }
  const packages: PackageInfo[] = require(path.join(process.cwd(), argv[2])).default;
  const nodeModulesFiles = readMultilineParamFile(argv[3]);
  const sharedLicensesFiles = readMultilineParamFile(argv[4]);

  return {
    generatorParams: {
      packages,
      nodeModulesFiles,
      sharedLicensesFiles,
    },
    outputJsonPath: argv[5]
  }
}

function main() {
  const args = parseArguments(process.argv);
  const generator = new LicenseMapGenerator(args.generatorParams.packages, new SharedLicensesProvider(args.generatorParams.sharedLicensesFiles));
  const licenseMap = generator.generateMap(args.generatorParams.nodeModulesFiles);
  // JSON is quite small, so there are no reasons to minify it.
  // Write it as multiline file with tabs (spaces).
  fs.writeFileSync(args.outputJsonPath, JSON.stringify(licenseMap, null, 2), "utf-8");
}

main();
