/**
 * @license
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import {LicenseName, LicenseTypeName, PackageName, FilePath} from "./base-types";

export interface LicenseType {
  /** Name of license type (GPL, MIT, etc...) - can appear in Documentation*/
  name: LicenseTypeName;
  /** If true, the file with this license type can be used.
   *  If false, build shouldn't use this file */
  allowed: boolean;
}

/** Type to describe a license*/
export type LicenseInfo = SharedFileLicenseInfo | PackageFileLicenseInfo;

interface LicenseInfoBase {
  /** Name of license - can appear in Documentation*/
  name: LicenseName;
  /** Type of license */
  type: LicenseType;
}

/** Use SharedFileLicenseInfo if license text must be obtained from a list of predefined texts */
export interface SharedFileLicenseInfo extends LicenseInfoBase {
  /** Name of the file with license text. The path to the file is not important, only the filename is used */
  sharedLicenseFile: string;
}

/** Use PackageFileLicenseInfo if license text must be obtained from package's file */
export interface PackageFileLicenseInfo extends LicenseInfoBase {
  /** Relative path to a file inside package*/
  packageLicenseFile: string;
}

export function isSharedFileLicenseInfo(licenseInfo: LicenseInfo): licenseInfo is SharedFileLicenseInfo {
  return (licenseInfo as SharedFileLicenseInfo).sharedLicenseFile !== undefined;
}

export function isPackageFileLicenseInfo(licenseInfo: LicenseInfo): licenseInfo is PackageFileLicenseInfo {
  return (licenseInfo as PackageFileLicenseInfo).packageLicenseFile !== undefined;
}

export type FilesFilter = (fileName: FilePath) => boolean;

/** Describes license for a whole package or to some files inside package */
export interface PackageInfo {
  /** Package name, as it appears in dependencies proprty of package.json */
  name: PackageName;
  /** Information about license*/
  license: LicenseInfo;
  /** If versions are set, then license applies only to a specific versions */
  versions?: string[];
  /** Predicate to select files to apply license. */
  filesFilter?: FilesFilter;
  /** List of nested directories with package.json files, that are not real packages*/
  nonPackages?: string[];
}
