<?php

namespace Imi\Bean;

use Imi\Aop\Annotation\Aspect;
use Imi\Aop\Annotation\PointCut;
use Imi\Aop\PointCutType;
use Imi\Bean\Annotation\AnnotationManager;
use Imi\Bean\Parser\PartialParser;
use Imi\Util\Imi;

abstract class BeanFactory
{
    /**
     * 计数器.
     *
     * @var int
     */
    private static $counter = 0;

    /**
     * 类名映射.
     *
     * @var array
     */
    private static $classNameMap = [];

    /**
     * 实例化.
     *
     * @param string $class
     * @param mixed  ...$args
     *
     * @return mixed
     */
    public static function newInstance($class, ...$args)
    {
        $classNameMap = &static::$classNameMap;
        if (isset($classNameMap[$class]))
        {
            $className = $classNameMap[$class];
        }
        else
        {
            $ref = ReflectionContainer::getClassReflection($class);
            $className = static::getNewClassName($ref->getShortName());
            $tpl = static::getTpl($ref, $className);
            Imi::eval($tpl);
            $classNameMap[$class] = $className;
        }
        $object = new $className(...$args);
        static::initInstance($object, $args);

        return $object;
    }

    /**
     * 实例化，但不初始化.
     *
     * @param string $class
     * @param mixed  ...$args
     *
     * @return object
     */
    public static function newInstanceNoInit($class, ...$args)
    {
        $classNameMap = &static::$classNameMap;
        if (isset($classNameMap[$class]))
        {
            $className = $classNameMap[$class];
        }
        else
        {
            $ref = ReflectionContainer::getClassReflection($class);
            $className = static::getNewClassName($ref->getShortName());
            $tpl = static::getTpl($ref, $className);
            Imi::eval($tpl);
            $classNameMap[$class] = $className;
        }

        return new $className(...$args);
    }

    /**
     * 初始化Bean对象
     *
     * @param object $object
     * @param array  $args
     *
     * @return void
     */
    public static function initInstance($object, $args = [])
    {
        $ref = ReflectionContainer::getClassReflection(\get_class($object));
        $beanProxy = $ref->getProperty('beanProxy');
        $beanProxy->setAccessible(true);
        $beanProxy->getValue($object)
                  ->injectProps($object);
        if ($ref->hasMethod('__init'))
        {
            $ref->getMethod('__init')->invoke($object, ...$args);
        }
    }

    /**
     * 获取新的类名.
     *
     * @param string $className
     *
     * @return string
     */
    private static function getNewClassName($className)
    {
        return $className . '__Bean__' . (++static::$counter);
    }

    /**
     * 获取类模版.
     *
     * @param \ReflectionClass $ref
     * @param string           $newClassName
     *
     * @return string
     */
    private static function getTpl($ref, $newClassName)
    {
        $class = $ref->getName();
        $methodsTpl = static::getMethodsTpl($ref);
        $construct = '';
        $constructMethod = $ref->getConstructor();
        if (null !== $constructMethod)
        {
            $paramsTpls = static::getMethodParamTpls($constructMethod);
            $constructDefine = $paramsTpls['define'];
            $construct = "parent::__construct({$paramsTpls['call']});";
            if (static::hasAop($ref, '__construct'))
            {
                $aopConstruct = <<<TPL
        \$__args__ = func_get_args();
        {$paramsTpls['set_args']}
        \$__result__ = \$beanProxy->call(
            \$this,
            '__construct',
            function({$paramsTpls['define']}){
                \$__args__ = func_get_args();
                {$paramsTpls['set_args']}
                return parent::__construct(...\$__args__);
            },
            \$__args__
        );

TPL;
            }
            else
            {
                $aopConstruct = $construct;
            }
        }
        else
        {
            $constructDefine = '...$args';
            $aopConstruct = '';
        }
        // partial 处理
        $classes = class_parents($class);
        if (isset($classes[1]))
        {
            $classes = array_reverse($classes);
        }
        $classes[] = $class;

        $partialData = PartialParser::getInstance()->getData();
        $traits = [];
        foreach ($classes as $currentClass)
        {
            if (isset($partialData[$currentClass]))
            {
                $traits[] = $partialData[$currentClass];
            }
        }
        if ($traits)
        {
            $traits = array_unique(array_merge(...$traits));
            $traitsTpl = 'use ' . implode(',', $traits) . ';';
        }
        else
        {
            $traitsTpl = '';
        }

        $parentClone = $ref->hasMethod('__clone') ? 'parent::__clone();' : '';
        // 类模版定义
        $tpl = <<<TPL
class {$newClassName} extends {$class} implements \Imi\Bean\IBean
{
    {$traitsTpl}

    protected \$beanProxy;

    public function __construct({$constructDefine})
    {
        \$this->beanProxy = \$beanProxy = new \Imi\Bean\BeanProxy(\$this);
        {$aopConstruct}
    }

    public function __clone()
    {
        \$this->beanProxy = \$beanProxy = new \Imi\Bean\BeanProxy(\$this);
        \$beanProxy->injectProps(\$this);
        {$parentClone}
    }
{$methodsTpl}
}
TPL;

        return $tpl;
    }

    /**
     * 获取方法模版.
     *
     * @param \ReflectionClass $ref
     *
     * @return string
     */
    private static function getMethodsTpl($ref)
    {
        $tpl = '';
        foreach ($ref->getMethods(\ReflectionMethod::IS_PUBLIC | \ReflectionMethod::IS_PROTECTED) as $method)
        {
            $methodName = $method->name;
            if ($method->isStatic() || '__construct' === $methodName || $method->isFinal() || !static::hasAop($ref, $method))
            {
                continue;
            }
            $paramsTpls = static::getMethodParamTpls($method);
            $methodReturnType = static::getMethodReturnType($method);
            $returnsReference = $method->returnsReference() ? '&' : '';
            // @phpstan-ignore-next-line
            $returnContent = $method->hasReturnType() && 'void' === $method->getReturnType()->getName() ? '' : 'return $__result__;';
            $tpl .= <<<TPL
    public function {$returnsReference}{$methodName}({$paramsTpls['define']}){$methodReturnType}
    {
        \$__args__ = func_get_args();
        {$paramsTpls['set_args']}
        \$__result__ = \$this->beanProxy->call(
            \$this,
            '{$methodName}',
            function({$paramsTpls['define']}){
                \$__args__ = func_get_args();
                {$paramsTpls['set_args']}
                return parent::{$methodName}(...\$__args__);
            },
            \$__args__
        );
        {$paramsTpls['set_args_back']}
        {$returnContent}
    }

TPL;
        }

        return $tpl;
    }

    /**
     * 获取方法参数模版们.
     *
     * @param \ReflectionMethod $method
     *
     * @return array
     */
    private static function getMethodParamTpls(\ReflectionMethod $method)
    {
        $args = $define = $call = [];
        $setArgs = $setArgsBack = '';
        $result = [
            'args'          => &$args,
            'define'        => &$define,
            'call'          => &$call,
            'set_args'      => &$setArgs,
            'set_args_back' => &$setArgsBack,
        ];
        foreach ($method->getParameters() as $i => $param)
        {
            // 数组参数，支持可变传参
            if (!$param->isVariadic())
            {
                $args[] = static::getMethodParamArgsTpl($param);
            }
            // 方法参数定义
            $define[] = static::getMethodParamDefineTpl($param);
            // 调用传参
            $call[] = static::getMethodParamCallTpl($param);
            // 引用传参
            if ($param->isPassedByReference())
            {
                $paramName = $param->name;
                $setArgs .= '$__args__[' . $i . '] = &$' . $paramName . ';';
                $setArgsBack .= '$' . $paramName . ' = $__args__[' . $i . '];';
            }
        }
        foreach ($result as &$item)
        {
            if (\is_array($item))
            {
                $item = implode(',', $item);
            }
        }
        // 调用如果参数为空处理
        if ([] === $call)
        {
            $call = '...func_get_args()';
        }

        return $result;
    }

    /**
     * 获取方法参数模版.
     *
     * @param \ReflectionParameter $param
     *
     * @return string
     */
    private static function getMethodParamArgsTpl(\ReflectionParameter $param)
    {
        $reference = $param->isPassedByReference() ? '&' : '';

        return $reference . '$' . $param->name;
    }

    /**
     * 获取方法参数定义模版.
     *
     * @param \ReflectionParameter $param
     *
     * @return string
     */
    private static function getMethodParamDefineTpl(\ReflectionParameter $param)
    {
        $result = '';
        // 类型
        $paramType = $param->getType();
        if ($paramType)
        {
            // @phpstan-ignore-next-line
            $paramType = $paramType->getName();
        }
        if (null !== $paramType && $param->allowsNull())
        {
            $paramType = '?' . $paramType;
        }
        $result .= null === $paramType ? '' : ((string) $paramType . ' ');
        if ($param->isPassedByReference())
        {
            // 引用传参
            $result .= '&';
        }
        elseif ($param->isVariadic())
        {
            // 可变参数...
            $result .= '...';
        }
        // $参数名
        $result .= '$' . $param->name;
        // 默认值
        if ($param->isDefaultValueAvailable())
        {
            $result .= ' = ' . var_export($param->getDefaultValue(), true);
        }

        return $result;
    }

    /**
     * 获取方法参数调用模版.
     *
     * @param \ReflectionParameter $param
     *
     * @return string
     */
    private static function getMethodParamCallTpl(\ReflectionParameter $param)
    {
        return ($param->isVariadic() ? '...' : '') . '$' . $param->name;
    }

    /**
     * 获取方法返回值模版.
     *
     * @param \ReflectionMethod $method
     *
     * @return string
     */
    private static function getMethodReturnType(\ReflectionMethod $method)
    {
        if (!$method->hasReturnType())
        {
            return '';
        }
        $returnType = $method->getReturnType();

        // @phpstan-ignore-next-line
        return ': ' . ($returnType->allowsNull() ? '?' : '') . $returnType->getName();
    }

    /**
     * 获取对象类名.
     *
     * @param string|object $object
     *
     * @return string
     */
    public static function getObjectClass($object)
    {
        if (\is_object($object))
        {
            if ($object instanceof IBean)
            {
                $parentClass = get_parent_class($object);
                // @phpstan-ignore-next-line
                if (false !== $parentClass)
                {
                    return $parentClass;
                }
            }

            return \get_class($object);
        }
        else
        {
            return (string) $object;
        }
    }

    /**
     * 是否有Aop注入当前方法.
     *
     * @param \ReflectionClass $class
     * @param string           $method
     *
     * @return bool
     */
    private static function hasAop($class, $method)
    {
        $aspects = AnnotationManager::getAnnotationPoints(Aspect::class);
        $className = $class->getName();
        if ('__construct' === $method)
        {
            foreach ($aspects as $item)
            {
                $pointCutsSet = AnnotationManager::getMethodsAnnotations($item->getClass(), PointCut::class);
                foreach ($pointCutsSet as $pointCuts)
                {
                    foreach ($pointCuts as $pointCut)
                    {
                        switch ($pointCut->type)
                        {
                            case PointCutType::CONSTRUCT:
                                // 构造方法
                                foreach ($pointCut->allow as $allowItem)
                                {
                                    if (Imi::checkRuleMatch($allowItem, $className))
                                    {
                                        return true;
                                    }
                                }
                                break;
                            case PointCutType::ANNOTATION_CONSTRUCT:
                                // 注解构造方法
                                $classAnnotations = AnnotationManager::getClassAnnotations($className);
                                foreach ($pointCut->allow as $allowItem)
                                {
                                    foreach ($classAnnotations as $annotation)
                                    {
                                        if ($annotation instanceof $allowItem)
                                        {
                                            return true;
                                        }
                                    }
                                }
                                break;
                        }
                    }
                }
            }
        }
        else
        {
            // @phpstan-ignore-next-line
            $methodAnnotations = AnnotationManager::getMethodAnnotations($className, $method->getName());
            foreach ($aspects as $item)
            {
                // 判断是否属于当前类的切面
                $pointCutsSet = AnnotationManager::getMethodsAnnotations($item->getClass(), PointCut::class);
                foreach ($pointCutsSet as $pointCuts)
                {
                    foreach ($pointCuts as $pointCut)
                    {
                        switch ($pointCut->type)
                        {
                            case PointCutType::METHOD:
                                foreach ($pointCut->allow as $allowItem)
                                {
                                    if (Imi::checkClassRule($allowItem, $className))
                                    {
                                        return true;
                                    }
                                }
                                break;
                            case PointCutType::ANNOTATION:
                                foreach ($pointCut->allow as $allowItem)
                                {
                                    foreach ($methodAnnotations as $annotation)
                                    {
                                        if ($annotation instanceof $allowItem)
                                        {
                                            return true;
                                        }
                                    }
                                }
                                break;
                        }
                    }
                }
            }
        }

        return false;
    }
}
