<?php
  
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File; 
use Image;

use App\User;
use App\Role;
use App\Page;
use App\Settings;
use App\Advertisements;
use App\Questions;
use App\Answers;
use App\FakeQuestions;
use App\Reports;
use Carbon\Carbon;
  
class Admin extends Controller
{
    
    public function __construct()
    {
        $this->middleware('role:admin');
    }
    
    /**
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('admin.index')->with([
            'title' => 'Dashboard - '.Settings::first()->website_name,
            'desc' => 'Dashboard',
        ]);
    }
    
    /**
     * Users
     */
    public function users()
    {
        $users = User::latest()
            ->orderByDesc('id')
            ->paginate(25);
        
        return view('admin.users.users')->with([
            'title' => 'Users - '.Settings::first()->website_name,
            'desc' => 'Dashboard',
            'users' => $users,
        ]);
        
    }
    
    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\User  $user
     * @return \Illuminate\Http\Response
     */
    public function delete_user($id)
    {
        $item = User::find($id);
        // delete uploaded avatar
        if(is_file('dist/img/avatar/'.$item->avatar)){
            $delete_image = File::delete('dist/img/avatar/'.$item->avatar);
        }
        
        $delete = DB::table('users')->where('id', $id)->delete();
        if($delete == true){
            return redirect('/admin/users')
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()->with('danger', __('app.text_120'));
        }
    }
    
    /**
     * Add New user
     *
     */
    public function add_user()
    {

        return view('admin.users.add_user')
            ->with([
                'title' => 'Add User - '.Settings::first()->website_name,
                'desc' => 'Dashboard',
                'roles' => Role::get()
            ]);
    }
    
    public function store_user(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'status' => ['required'],
        ]);
        
        $user = User::create([
            'name'     => request('name'),
            'email'    => request('email'),
            'password' => bcrypt(request('password')),
            'status'    => request('status'),
        ]);
        
        $user
            ->roles()
            ->attach(Role::where('name', request('role'))->first());
        
        if($user == true)
        {
            return redirect('/admin/users')
                ->with('success', __('app.text_120'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    /**
     * Edit user
     * @id
     */
    public function edit_user($id)
    {
        $user = User::findOrFail($id);
        $roles = Role::all();
        $userRole = $user->roles->pluck('name','name')->all();
        
        return view('admin.users.edit_user')
            ->with([
                'title' => 'Edit User - '.Settings::first()->website_name,
                'desc' => 'Edit user',
                'item' => $user,
                'roles' => $roles,
                'userRole' => $userRole
            ]);
    }
    
    public function store_edit_user(Request $request, $id)
    {
        
        if(!empty(request('new_password'))){
            $this->validate($request, [
                'new_password' => 'required|min:8',
                'new_confirm_password' => 'same:new_password',
            ]);
            
            $update = User::where('id', $id)
                ->update([
                    'password' => bcrypt(request('new_password'))
                ]);
            
            if($update == true)
            {
                return redirect('/admin/users')
                    ->with('success', __('app.text_120'));
            } else 
            {
                return back()
                    ->with('danger', __('app.text_120'));
            }
            
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|unique:users,email,'.$id,
            //'password' => 'string|min:8|confirmed',
            'status' => 'required',
        ]);
        
        $update = User::where('id', $id)
            ->update([
            'name'     => request('name'),
            'email'    => request('email'),
            //'password' => bcrypt(request('password')),
            'status'    => request('status')
        ]);

        User::find($id)->roles()->detach();
        User::find($id)->roles()->attach(Role::where('name', request('role'))->first());
            
        if($update == true)
        {
            return redirect('/admin/users')
                ->with('success', __('app.text_120'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    public function confirm($user_id)
    {
        $user = User::find($user_id)->first();
        if(!empty($user->email_verified_at)){
            $update = User::where('id', $user_id)
                ->update([
                    'email_verified_at' => Carbon::now(),
                ]);
            return redirect('admin/users')
                ->with('success', __('app.text_120'));
        } else {
            return redirect('admin/users')
                ->with('danger', __('app.text_120'));
        }
    }
    
    /**
     * Pages
     */
    public function pages()
    {
        $pages = Page::latest()
            ->orderByDesc('id')
            ->paginate(25);
        
        return view('admin.pages.pages')->with([
            'title' => 'Pages - '.Settings::find(1)->website_name,
            'desc' => 'Dashboard',
            'pages' => $pages,
        ]);
        
    }
    
    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Page  $pages
     * @return \Illuminate\Http\Response
     */
    public function delete_page($id)
    {

        $delete = DB::table('pages')->where('id', $id)->delete();
        
        if($delete == true){
            return redirect('/admin/pages')
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()->with('danger', __('app.text_120'));
        }
        
    }
    
    /**
     * Add New page
     *
     */
    public function add_page()
    {

        return view('admin.pages.add_page')
            ->with([
                'title' => 'Add Page - '.Settings::find(1)->website_name,
                'desc' => 'Dashboard',
            ]);
    }
    
    public function store_page(Request $request)
    {
        $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'body' => ['required'],
            'status' => ['required'],
        ]);
        
        $page = Page::create([
            'title'     => request('title'),
            'body'    => request('body'),
            'slug' => Str::of(request('title').' '.Str::random(8))->slug('-'),
            'status'    => request('status'),
        ]);

        if($page == true)
        {
            return redirect('/admin/pages')
                ->with('success', __('app.text_120'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    /**
     * Edit page
     * @id
     */
    public function edit_page($id)
    {
        
        return view('admin.pages.edit_page')
            ->with([
                'title' => 'Edit Page - '.Settings::find(1)->website_name,
                'desc' => 'Edit page',
                'item' => Page::findOrFail($id)
            ]);
        
    }
    
    public function store_edit_page(Request $request, $id)
    {

        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required',
            'status' => 'required',
        ]);

        $update = Page::where('id', $id)
            ->update([
            'title'     => request('title'),
            'body'    => request('body'),
            //'slug' => Str::slug(request('title').Str::random(12), '-'),
            'status'    => request('status')
        ]);

        if($update == true)
        {
            return back()
                ->with('success', __('app.text_120'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    /**
     * Settings
     */
    public function settings()
    {
        
        return view('admin.settings.index')->with([
            'title' => 'Settings - '.Settings::first()->website_name,
            'desc' => 'Settings',
            'settings' => Settings::first(),
            'advertisements' => Advertisements::where('status', 1)->get(),
        ]);
        
    }
    
    public function update_settings(Request $request)
    {
        $request->validate([
            'website_name' => 'required|string|max:255',
            'website_tagline' => 'required|string|max:255',
            'website_desc' => 'nullable|max:255',
            'items_result' => 'required|numeric',
            'photo_upload' => 'required',
            'analytics' => 'nullable',
            'adv_1' => 'nullable',
            'adv_2' => 'nullable',
            'adv_3' => 'nullable',
            'min_question' => 'required|numeric',
            'max_question' => 'required|numeric',
            'min_answer' => 'required|numeric',
            'max_answer' => 'required|numeric',
        ]);
        
        if(!empty(request('logo'))){
            
            $request->validate([
                'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            ]);
            
            if(!empty(Settings::first()->logo))
            {
                $delete = File::delete('dist/img/logo/'.Settings::first()->logo);
            }
            
            $image                   =       $request->file('logo');
            $input['imagename']      =       time().'.'.$image->extension();
            $destinationPath         =       'dist/img/logo';
            $img                     =       Image::make($image->path());
        
            // --------- [ Resize Image ] ---------------

            $img->resize(function($constraint) {
                $constraint->aspectRatio();
            })->save($destinationPath.'/'.$input['imagename']);
            
            $upload_logo = Settings::first()
                ->update([
                    'logo' => $input['imagename'],
                ]);
            
            if($upload_logo == true)
            {
                return back()
                    ->with('success', __('app.text_124'));
            } else 
            {
                return back()
                    ->with('danger', __('app.text_120'));
            }
            
        }

        $update = Settings::first()
            ->update([
                'website_name' => request('website_name'),
                'website_tagline' => request('website_tagline'),
                'website_desc' => request('website_desc'),
                'items_result' => request('items_result'),
                'photo_upload' => request('photo_upload'),
                'analytics' => request('analytics'),
                'adv_1' => request('adv_1'),
                'adv_2' => request('adv_2'),
                'adv_3' => request('adv_3'),
                'min_question' => request('min_question'),
                'max_question' => request('max_question'),
                'min_answer' => request('min_answer'),
                'max_answer' => request('max_answer'),
        ]);
        
        if($update == true)
        {
            return redirect('/admin/settings')
                ->with('success', __('app.text_122'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    public function delete_logo()
    {
        
        if(!empty(Settings::first()->logo))
        {
            // delete file 
            File::delete('dist/img/logo/'.Settings::first()->logo);
            
            $delete_logo = Settings::first()
                ->update([
                    'logo' => NULL,
                ]);
            
            if($delete_logo == true)
            {
                return back()
                    ->with('success', __('app.text_123'));
            } else 
            {
                return back()
                    ->with('danger', __('app.text_120'));
            }
        } 
        else 
        {
            return back()
                ->with('danger', __('app.text_125'));
        }
        
    }
    
   
    /**
     * Advertisements
     */
    public function advertisements()
    {
        $advertisements = Advertisements::latest()
            ->orderByDesc('id')
            ->paginate(25);
        
        return view('admin.advertisements.index')->with([
            'title' => 'Advertisements :: '.Settings::first()->website_name,
            'desc' => 'Dashboard',
            'advertisements' => $advertisements,
        ]);
        
    }
    
    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Page  $pages
     * @return \Illuminate\Http\Response
     */
    public function delete_adv($id)
    {

        $delete = DB::table('advertisements')->where('id', $id)->delete();
        
        if($delete == true){
            return back()
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()->with('danger', __('app.text_120'));
        }
        
    }
    
    /**
     * Add New advertisement
     *
     */
    public function add_adv()
    {

        return view('admin.advertisements.add_adv')
            ->with([
                'title' => 'Add ADV :: '.Settings::first()->website_name,
                'desc' => 'Dashboard',
            ]);
    }
    
    public function store_adv(Request $request)
    {
        $request->validate([
            'title' => 'required|min:3|unique:advertisements,title',
            'adv' => 'required',
        ]);
        
        $advertisements = Advertisements::create([
            'title'     => request('title'),
            'adv' => request('adv'),
            'status' => request('status'),
        ]);

        if($advertisements == true)
        {
            return back()
                ->with('success', __('app.text_120'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    /**
     * Edit advertisement
     * @id
     */
    public function edit_adv($id)
    {
        
        return view('admin.advertisements.edit_adv')
            ->with([
                'title' => 'Edit Advertisements :: '.Settings::first()->website_name,
                'desc' => 'Edit advertisements...',
                'item' => Advertisements::findOrFail($id)
            ]);
        
    }
    
    public function store_edit_adv(Request $request, $id)
    {

        $request->validate([
            'title' => 'required|min:3|unique:advertisements,title,'.$id,
            'adv' => 'required',
        ]);

        $update = Advertisements::where('id', $id)
            ->update([
                'title'     => request('title'),
                'adv'     => request('adv'),
                'status' => request('status'),
        ]);

        if($update == true)
        {
            return back()
                ->with('success', __('app.text_120'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    //***
    // Questions
    //***
    public function questions()
    {
        return view('admin.questions.index')->with([
            'title' => __('app.text_56').' - '.Settings::first()->website_name,
            'desc' => 'List questions',
            'items' => Questions::latest()->orderByDesc('id')->paginate(25)
        ]);
    }
    
    /**
     */
    public function delete_question($id)
    {
        $item = Questions::find($id);
        $delete = DB::table('questions')
            ->where('id', $id)
            ->delete();
        
        if($delete == true){
            return back()
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    //***
    // Answers
    //***
    public function answers()
    {
        return view('admin.answers.index')->with([
            'title' => __('app.text_58').' - '.Settings::first()->website_name,
            'desc' => 'List questions',
            'items' => Answers::latest()->orderByDesc('id')->paginate(25)
        ]);
    }
    
     /**
     */
    public function delete_answer($id)
    {
        
        $item = Answers::find($id);
        $delete = DB::table('answers')->where('id', $id)->delete();
        if($delete == true){
            
            // if you delete the answer, the question goes back to the inbox to answer
            $update = Questions::where('id', $item->id_question)
                ->update([
                    'replied'  => 2,
                ]);
            
            // if there is an image, delete it
            // delete uploaded image 
            if(is_file('dist/img/answers/'.$item->photo)){
                $delete_image = File::delete('dist/img/answers/'.$item->photo);
            }

            return back()
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()->with('danger', __('app.text_120'));
        }
        
    }
    
    //***
    // Fake Questions
    //***
    public function fake_questions()
    {
        return view('admin.fake_questions.index')->with([
            'title' => __('app.text_57').' - '.Settings::first()->website_name,
            'desc' => 'Fake Questions.',
            'items' => FakeQuestions::latest()->orderByDesc('id')->paginate(25)
        ]);
    }
    
    public function new_fake_question(Request $request)
    {
        
        $request->validate([
            'fake_question' => 'required|max:'.Settings::first()->max_question.'|min:'.Settings::first()->min_question,
        ]);
        
        $fakeQuestion = new FakeQuestions;
        $fakeQuestion->question = request('fake_question');
        
        if($fakeQuestion->save() == true)
        {
            return back()
                ->with('success', __('app.text_41'));

        } else 
        {
            return back()
                ->with('danger', __('app._text_42'));
        }
        
    }
    
    public function delete_fake_question($id)
    {
        $item = FakeQuestions::find($id);
        $delete = DB::table('fake_questions')
            ->where('id', $id)
            ->delete();
        
        if($delete == true){
            return back()
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
    
    //***
    // Reports
    //***
    public function reports()
    {
        return view('admin.reports.index')->with([
            'title' => __('app.text_59').' - '.Settings::first()->website_name,
            'desc' => 'Reports.',
            'items' => Reports::latest()->orderByDesc('id')->paginate(25)
        ]);
    }
    
    /**
     */
    public function delete_report($id)
    {
        $item = Reports::find($id);
        $delete = DB::table('reports')
            ->where('id', $id)
            ->delete();
        
        if($delete == true){
            return back()
                ->with('success', __('app.text_120'));
        }
        else 
        {
            return back()
                ->with('danger', __('app.text_120'));
        }
        
    }
   
}