<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Builder;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File; 
use Image;

use App\Questions;
use App\FakeQuestions;
use App\Answers;
use App\Likes;
use App\User;
use App\Role;
use App\Page;
use App\Settings;
use App\Advertisements;
use App\Notifications;
use App\Reports;
use Carbon\Carbon;
use Cookie;
use Mail;

class HomeController extends Controller
{
    
    /**
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $items = Questions::where('status', 1)
            ->where('replied', 1)
            ->orderByDesc('updated_at')
            ->paginate(Settings::first()->items_result);
        
        $rankings = DB::table('likes')
            ->selectRaw("COUNT(id) as total_id, sent_to")
            ->groupBy('sent_to')
            ->orderByRaw('total_id DESC')
            ->limit(10)
            ->get();
        
        return view('index')->with([
            'title' => Settings::first()->website_name.' - '.Settings::first()->website_tagline,
            'desc' => Settings::first()->website_desc,
            'items' => $items,
            'users' => User::orderByDesc('created_at')->limit(8)->get(),
            'rankings' => $rankings
        ]);
    }
    
    public function inbox()
    {
        $items = Questions::where('status', 1)
            ->where('sent_to', Auth::id())
            ->where('replied', 2)
            ->orderByDesc('id')
            ->paginate(Settings::first()->items_result);
        
        $rankings = DB::table('likes')
            ->selectRaw("COUNT(id) as total_id, sent_to")
            ->groupBy('sent_to')
            ->orderByRaw('total_id DESC')
            ->limit(10)
            ->get();
        
        return view('inbox')->with([
            'title' => __('app.text_24').' - '.Settings::first()->website_name,
            'desc' => Settings::first()->website_desc,
            'items' => $items,
            'rankings' => $rankings
        ]);
    }
    
    /**
     * Pages
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function page($slug)
    {
          
        if($slug == null){
            abort(404);
        }
        
        $item = Page::where('slug', $slug)->first();
        
        if($item == null){
            abort(404);
        }
        
        if($item->status == 2){
            abort(404, 'This content is currently disabled!');
        }

        return view('page')->with([
            'title' => $item->title.' - '.Settings::first()->website_name,
            'desc' => Str::of($item->body)->words(45, '...'),
            'item' => $item
        ]);
        
    }
    
    /**
     * Contacts
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function contacts(Request $request)
    {
        
        return view('contacts')->with([
            'title' => __('app.text_8').' - '.Settings::first()->website_name,
            'desc' => '',
        ]);
        
    }
    
    /**
     * postContacts
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function postContacts(Request $request)
    {
        
        $request->validate([
            'name' => 'required',
            'email' => 'required|email',
            'message' => 'required',
        ]);
        
        //  Send mail to admin
        \Mail::send('post_contacts', array(
            'name' => $request->get('name'),
            'email' => $request->get('email'),
            'user_query' => $request->get('message'),
        ), function($message) use ($request){
            $message->from($request->email);
            $message->to('digambersingh126@gmail.com', 'Admin')->subject('Information requests');
        });

        return back()
            ->with('success', __('app.text_43')); 
    }
     
    /**
     * Search
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    
    public function search(Request $request)
    {
        
        $request->validate([
            'key' => 'required',
        ]);

        if($request->has('key')) {
            $items = Questions::when($request->key, function ($query, $key) {
                return $query->where('question', 'like', "%{$key}%");
            })->where('replied', 1)->where('replied', 1)
                ->orderByDesc('id')
                ->paginate(Settings::first()->items_result); 
        }
        
        $rankings = DB::table('likes')
            ->selectRaw("COUNT(id) as total_id, sent_to")
            ->groupBy('sent_to')
            ->orderByRaw('total_id DESC')
            ->limit(10)
            ->get();
        
        return view('search')->with([
            'title' => request('key').' - '.Settings::first()->website_name,
            'desc' => '',
            'items' => $items,
            'keywords' => request('key'),
            'rankings' => $rankings
        ]);
    }

    // Show user profile
    public function user($username)
    {
          
        if($username == null){
            abort(404);
        }

        $user = DB::table('users')
            ->where('name', $username)
            ->first();
        
        if($user == null){
            abort(404);
        }
        
        $items = Questions::where('status', 1)
            ->where('sent_to', $user->id)
            ->where('replied', 1)
            ->orderByDesc('updated_at')
            ->paginate(Settings::first()->items_result);
        
        return view('user')->with([
            'title' => $user->name.' - '.Settings::first()->website_name,
            'desc' => Settings::first()->website_desc,
            'user' => $user, 
            'items' => $items
        ]);
        
    }
    
    /**
     * Edit user
     * @id
     */
    public function edit_user()
    {
        
        $user = User::findOrFail(Auth::id());
        
        return view('edit_user')
            ->with([
                'title' => __('app.text_40_1').' - '.Settings::first()->website_name,
                'desc' => 'Edit user',
                'item' => $user,
            ]);
    }
    
    public function store_edit_user(Request $request)
    {
        
        if(!empty(request('new_password'))){
            $this->validate($request, [
                'new_password' => 'required|min:8',
                'new_confirm_password' => 'same:new_password',
            ]);

            $update = User::where('id', Auth::id())
                ->update([
                    'password' => bcrypt(request('new_password'))
                ]);

            if($update == true)
            {
                return back()
                    ->with('success', __('app.text_41'));
            } else 
            {
                return back()
                    ->with('danger', __('app.text_42'));
            }

        }

        $request->validate([
            'email' => 'required|unique:users,email,'.Auth::id(),
            'bio' => 'nullable|max:200|min:5|',
        ]);
        
        if(!empty(request('avatar'))){

            $request->validate([
                'avatar' => 'nullable|image|dimensions:min_width=200,min_height=200|mimes:jpeg,png,jpg|max:2048',
            ]);

            if(!empty(Auth::user()->avatar))
            {
                $delete = File::delete('dist/img/avatar/'.Auth::user()->avatar);
            }

            $image                   =       $request->file('avatar');
            $input['imagename']      =       time().'.'.$image->extension();
            $destinationPath         =       'dist/img/avatar';
            $img                     =       Image::make($image->path());

            // --------- [ Resize Image ] ---------------

            $img->resize(function($constraint) {
                $constraint->aspectRatio();
            })->save($destinationPath.'/'.$input['imagename']);

            $upload_avatar = User::where('id', Auth::id())
                ->update([
                    'avatar' => $input['imagename'],
                ]);

            if($upload_avatar == true)
            {
                return back()
                    ->with('success', __('app.text_41'));
            } else 
            {
                return back()
                    ->with('danger', __('app.text_42'));
            }
        }

        $update = User::where('id', Auth::id())
            ->update([
                'email'    => request('email'),
                'bio' => request('bio'),

        ]);

        if($update == true)
        {
            return back()
                ->with('success', __('app.text_41'));
        } else 
        {
            return back()
                ->with('danger', __('app.text_42'));
        }

    }
    
    public function delete_avatar()
    {
        
        if(!empty(Auth::user()->avatar))
        {
            // delete avatar 
            File::delete('dist/img/avatar/'.Auth::user()->avatar);
            
            $delete_avatar = User::where('id', Auth::id())
                ->update([
                    'avatar' => NULL,
                ]);
            
            if($delete_avatar == true)
            {
                return back()
                    ->with('success', __('app.text_41'));
            } else 
            {
                return back()
                    ->with('danger', __('app.text_42'));
            }
        } 
        else 
        {
            return back()
                ->with('danger', __('app.text_42'));
        }
        
    }
    
    public function delete_account()
    {

        $delete = DB::table('users')
            ->where('id', Auth::id())
            ->delete();
        
        if($delete == true){
            
            // delete avatar 
            File::delete('dist/img/avatar/'.Auth::user()->avatar);
            
            return redirect('register')
                ->with('success', __('auth.account_deleted'));
        } else {
            return redirect('register')
                ->with('danger', __('app.text_42'));
        }
        
    }
    
    // Delete Question
    // The user deletes the question individually
    //
    public function delete_question($id)
    {
        
        $question = DB::table('questions')
            ->where('id', $id)
            ->first();
        
        if(!$question->sent_to == Auth::id()){
            return back()
                ->with('danger', __('app.text_44'));
        }
        
        $delete = DB::table('questions')
            ->where('id', $id)
            ->where('sent_to', Auth::id())
            ->delete();
        
        if($delete == true){
            return back()->with('success', __('app.text_41'));
        } else {
            return back()->with('danger', __('app.text_42'));
        }
        
    }
    
    // Bulk Delete Questions
    // The user bulk deletes all questions they have not answered
    //
    public function bulk_delete_questions()
    {
        
        $bulkDelete = DB::table('questions')
            ->where('sent_to', Auth::id())
            ->where('replied', 2)
            ->delete();
        
        if($bulkDelete == true){
            return back()->with('success', __('app.text_41'));
        } else {
            return back()->with('danger', __('app.text_42'));
        }
        
    }
    
    // Delete Answer
    // The user deletes the answer individually
    //
    public function delete_answer($id)
    {
        
        $answer = DB::table('answers')
            ->where('id', $id)
            ->first();
        
        if(!$answer->sent_to == Auth::id()){
            return back()
                ->with('danger', __('app.text_44'));
        }
        
        // if you delete the answer, the question goes back to the inbox to answer
        $update = Questions::where('id', $answer->id_question)
            ->update([
                'replied'  => 2,
            ]);
        
        if($update == true){
            
            // if there is an image, delete it
            // delete uploaded image 
            if(is_file('dist/img/answers/'.$answer->photo)){
                $delete_image = File::delete('dist/img/answers/'.$answer->photo);
            }
            
            DB::table('answers')
                ->where('id', $id)
                ->where('sent_by', Auth::id())
                ->delete();
            
            return back()->with('success', __('app.text_41'));
        } else {
            return back()->with('danger', __('app.text_42'));
        }
        
    }
    
    public function markNotification()
    {
        $notifications = Notifications::where('sent_to', Auth::id())
            ->where('read_at', null)
            ->update([
                'read_at'  => now(),
            ]);
        
        if($notifications == true)
        {
            return back();
        } else
        {
            return back();
        }
        
    }
    
    function save_like(Request $request){
        
        if($request->answer == null){
            abort(404);
        }
        
        $checkItem = DB::table('answers')->where('id', $request->answer)->first();
        if($checkItem == null){
            abort(404);
        }
        
        $check = DB::table('likes')
            ->where('sent_by', Auth::id())
            ->where('answer_id', $request->answer)
            ->get();
        
        if($check->count() < 1){
            
            $data = new Likes;
            $data->sent_by = Auth::id();
            $data->sent_to = $request->touser;
            $data->question_id = $request->post;
            $data->answer_id = $request->answer;
            $data->save();
            
            // add a like notification
            $notification = new Notifications;
            $notification->sent_by = Auth::id();
            $notification->sent_to = $request->touser;
            $notification->item_id = $request->post;
            $notification->reply_id = $request->answer;
            $notification->type = "like";
            $notification->save();
            
            return response()->json([
                'bool'=>true
            ]);
        } else
        {
            DB::table('likes')->where('sent_by', Auth::id())->where('answer_id', $request->answer)->delete();
            DB::table('notifications')
                ->where('sent_by', Auth::id())
                ->where('sent_to', $request->touser)
                ->where('reply_id', $request->answer)
                ->where('type', "like")
                ->delete();
            
                return response()->json([
                    'bool'=>false
                ]);
            
        }
        
    }
    
    public function show_question($id)
    {
        
        $quest = Questions::where('id', $id)->where('replied', 1)->firstOrFail();
        
        return view('show_question')->with([
            'title' => $quest->question.' - '.Settings::first()->website_name,
            'desc' => Str::of($quest->question)->words(45, '...'),
            'quest' => $quest,
        ]);
        
    }
    
    // saves the question sent from user to user in the database
    //
    public function new_question(Request $request)
    {
        // you can't ask yourself questions
        if(request('sent_to') == Auth::id()) {
            return back()
                ->with('danger', __('app.text_49_1'));
        }
        
        $questions = new Questions;
        
        $request->validate([
            'question' => 'required|max:'.Settings::first()->max_question.'|min:'.Settings::first()->min_question,
            'sent_to' => 'required',
        ]);

        $questions->question = request('question')."?";
        $questions->sent_to = request('sent_to');
        $questions->sent_by = Auth::id();
        
        
        if(isset($request->private)) { $questions->private = "1"; } else { $questions->private = "2"; }
        
        $questions->replied = 2;
        
        if($questions->save() == true)
        {
            if(Settings::first()->items_status == 1){
                return back()->with('success', __('app.text_41'));
            } else {
                return back()->with('success', __('app.text_41'));
            }   
        } else 
        {
            return back()->with('danger', __('app._text_42'));
        }
        
    }
    
    // answer a question posted by a user
    //
    public function answer_the_question(Request $request)
    {

        $answers = new Answers;
        
        $request->validate([
            'answer' => 'required|max:'.Settings::first()->max_answer.'|min:'.Settings::first()->min_answer,
        ]);

        $answers->reply = request('answer');
        $answers->id_question = request('id_question');
        $answers->sent_by = Auth::id();
        $answers->sent_to = request('sent_to');
        
        if(Settings::first()->photo_upload == 1){
            if(!empty(request('image'))){
                $request->validate([
                    'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                ]);

                $image                   =       $request->file('image');
                $input['imagename']      =       time().'.'.$image->extension();
                $destinationPath         =       'dist/img/answers';
                $img                     =       Image::make($image->path());

                // --------- [ Resize Image ] ---------------

                $img->resize(function($constraint) {
                    $constraint->aspectRatio();
                })->save($destinationPath.'/'.$input['imagename']);

                $upload = $answers->save([
                        $answers->photo = $input['imagename'],
                    ]);
            }
        }

        if($answers->save() == true)
        {
            $update = Questions::where('id', request('id_question'))
                ->update([
                    'replied' => 1,
            ]);

            // add a reply notification
            $notification = new Notifications;
            $notification->sent_by = Auth::id();
            $notification->sent_to = request('sent_to');
            $notification->item_id = request('id_question');
            $notification->reply_id = $answers->id;
            $notification->type = "reply";
            $notification->save();
                    
            return back()->with('success', __('app.text_41'));
        } else 
        {
            return back()->with('danger', __('app._text_42'));
        }
        
    }
    
    public function new_random_question()
    {
        $fakeQuestions = FakeQuestions::where('status', 1)->get()->random();
        
        $userQuestions = Questions::where('question', 'LIKE', '%'.$fakeQuestions->question.'%')
            ->where('sent_to', Auth::id())
            ->count();
        
        if($userQuestions == 1){
            return back()
                ->with('danger', __('app.text_45'));
        } else {
            
            $questions = new Questions;
            $questions->question = $fakeQuestions->question;
            $questions->sent_by = Auth::id();
            $questions->sent_to = Auth::id();
            $questions->private = 1;
            $questions->replied = 2;
            $questions->status = 1;

            if($questions->save() == true){
                return back()->with('success', __('app.text_46'));
            } else {
                return back()->with('danger', __('app._text_42'));
            }
            
        }
         
    }
    
    // report question - answer
    public function reportQA(Request $request)
    {
        $request->validate([
            'user_id' => 'required',
            'question_id' => 'required',
            'answer_id' => 'required',
        ]);
        
        // record the report
        $report = new Reports;
        $report->user_id = Auth::id();
        $report->question_id = request('question_id');
        $report->answer_id = request('answer_id');

        if($report->save() == true) {
            return back()
                ->with('success', __('app.text_47'));
        } else {
            return back()
                ->with('danger', __('app._text_42'));
        }
        
    }
    
}