/**
 * Copyright (c) 2016-2019, Michael Yang 杨福海 (fuhai999@gmail.com).
 * <p>
 * Licensed under the GNU Lesser General Public License (LGPL) ,Version 3.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.gnu.org/licenses/lgpl-3.0.txt
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.jpress.model;

import io.jboot.db.annotation.Table;
import io.jboot.utils.StrUtil;
import io.jpress.commons.pay.PayStatus;
import io.jpress.model.base.BasePaymentRecord;

import java.util.HashMap;
import java.util.Map;

/**
 * Generated by Jboot.
 */
@Table(tableName = "payment_record", primaryKey = "id")
public class PaymentRecord extends BasePaymentRecord<PaymentRecord> {
    private static final long serialVersionUID = 1L;

    /**
     * 业务类型
     */
    public static final String TRX_TYPE_RECHARGE = "recharge"; //用户充值
    public static final String TRX_TYPE_ORDER = "order"; //订单支付
    public static final String TRX_TYPE_MEMBER = "member"; //购买会员

    public static final Map<String, String> trxTypeTexts = new HashMap<>();

    static {
        trxTypeTexts.put(TRX_TYPE_RECHARGE, "用户充值");
        trxTypeTexts.put(TRX_TYPE_ORDER, "订单支付");
        trxTypeTexts.put(TRX_TYPE_MEMBER, "购买会员");
    }


    /**
     * 支付类型
     */
    public static final String PAY_TYPE_AMOUNT = "amount"; //余额支付
    public static final String PAY_TYPE_WECHAT = "wechat"; //微信在线支付
    public static final String PAY_TYPE_WECHATX = "wechatx"; //微信好友转账支付（包括红包等）
    public static final String PAY_TYPE_ALIPAY = "alipay"; //支付宝在线支付
    public static final String PAY_TYPE_ALIPAYX = "alipayx"; //支付宝转账
    public static final String PAY_TYPE_PAYPAL = "paypal"; //paypal在线支付
    public static final String PAY_TYPE_OFFLINE = "offline"; //线下支付
    public static final String PAY_TYPE_OTHER = "other"; //其他支付

    public static final Map<String, String> payTypeTexts = new HashMap<>();

    static {
        payTypeTexts.put(PAY_TYPE_AMOUNT, "余额支付");
        payTypeTexts.put(PAY_TYPE_WECHAT, "微信在线支付");
        payTypeTexts.put(PAY_TYPE_WECHATX, "微信好友转账支付");
        payTypeTexts.put(PAY_TYPE_ALIPAY, "支付宝在线支付");
        payTypeTexts.put(PAY_TYPE_ALIPAYX, "支付宝转账支付");
        payTypeTexts.put(PAY_TYPE_PAYPAL, "paypal在线支付");
        payTypeTexts.put(PAY_TYPE_OFFLINE, "线下支付");
        payTypeTexts.put(PAY_TYPE_OTHER, "其他方式支付");
    }


    /**
     * payment 状态
     */
    public static final int STATUS_PAY_PRE = 1; //预支付
    public static final int STATUS_PAY_FAILURE = 2; //支付失败
    public static final int STATUS_PAY_SUCCESS = 9; //支付成功
    public static final int STATUS_REFUND_PRE = 11; //预退款
    public static final int STATUS_REFUNDING = 12; //退款中
    public static final int STATUS_REFUND_FAILURE = 13; //退款失败
    public static final int STATUS_REFUND_SUCCESS = 19; //退款成功

    public static final Map<Integer, String> statusTexts = new HashMap<>();

    static {
        statusTexts.put(STATUS_PAY_PRE, "未支付");
        statusTexts.put(STATUS_PAY_FAILURE, "支付失败");
        statusTexts.put(STATUS_PAY_SUCCESS, "支付成功");
        statusTexts.put(STATUS_REFUND_PRE, "申请退款");
        statusTexts.put(STATUS_REFUNDING, "退款中");
        statusTexts.put(STATUS_REFUND_FAILURE, "退款失败");
        statusTexts.put(STATUS_REFUND_SUCCESS, "退款成功");
    }


    /**
     * 是否可以对支付记录进行手动入账
     *
     * @return
     */
    public boolean isCanUpdateByUser() {
        Integer payStatus = getPayStatus();
        String payType = getPayType();

        return payStatus != null
                && payStatus == PayStatus.UNPAY.getStatus()
                && (PAY_TYPE_WECHATX.equals(payType) || PAY_TYPE_ALIPAYX.equals(payType));
    }


    /**
     * 是否支付成功
     *
     * @return
     */
    public boolean isPaySuccess() {
        return getPayStatus() != null && getPayStatus() > PayStatus.SUCCESS_ALIPAY.getStatus();
    }


    /**
     * 是否处于预支付状态
     *
     * @return
     */
    public boolean isPayPre() {
        return getPayStatus() != null && PayStatus.UNPAY.getStatus() == getPayStatus();
    }


    public String getPayTypeStr() {
        return payTypeTexts.get(getPayType());
    }

    public String getPayStatusStr() {
        return PayStatus.getTextByInt(getPayStatus());
    }

    public String getStatusStr() {
        return statusTexts.get(getStatus());
    }

    public String getTrxTypeStr() {
        String str = trxTypeTexts.get(getTrxType());
        return StrUtil.isBlank(str) ? getTrxType() : str;
    }
}
