<?php
// +----------------------------------------------------------------------
// | ThinkPHP 5 [ WE CAN DO IT JUST THINK IT ]
// +----------------------------------------------------------------------
// | Copyright (c) 2018 http://www.qilecms.com All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: Byron Sampson <xiaobo.sun@gzzstudio.net>
// +----------------------------------------------------------------------
namespace think;

use think\Db;
// use think\facade\Config;
// use think\facade\Session;
use think\Request;
use think\Loader;

/**
 * 权限认证类
 * 功能特性：
 * 1，是对规则进行认证，不是对节点进行认证。用户可以把节点当作规则名称实现对节点进行认证。
 *      $auth=new Auth();  $auth->check('规则名称','用户id')
 * 2，可以同时对多条规则进行认证，并设置多条规则的关系（or或者and）
 *      $auth=new Auth();  $auth->check('规则1,规则2','用户id','and')
 *      第三个参数为and时表示，用户需要同时具有规则1和规则2的权限。 当第三个参数为or时，表示用户值需要具备其中一个条件即可。默认为or
 * 3，一个用户可以属于多个角色组(think_auth_role_access表 定义了用户所属角色组)。我们需要设置每个角色组拥有哪些规则(think_auth_role 定义了角色组权限)
 *
 * 4，支持规则表达式。
 *      在think_auth_rule 表中定义一条规则时，如果type为1， condition字段就可以定义规则表达式。 如定义{score}>5  and {score}<100  表示用户的分数在5-100之间时这条规则才会通过。
 */
//数据库
/*
-- ----------------------------
-- think_auth_rule，规则表，
-- id:主键，name：规则唯一标识, title：规则中文名称 status 状态：为1正常，为0禁用，condition：规则表达式，为空表示存在就验证，不为空表示按照条件验证
-- ----------------------------
DROP TABLE IF EXISTS `think_auth_rule`;
CREATE TABLE `think_auth_rule` (
    `id` mediumint(8) unsigned NOT NULL AUTO_INCREMENT,
    `name` char(80) NOT NULL DEFAULT '',
    `title` char(20) NOT NULL DEFAULT '',
    `type` tinyint(1) NOT NULL DEFAULT '1',
    `status` tinyint(1) NOT NULL DEFAULT '1',
    `condition` char(100) NOT NULL DEFAULT '',  # 规则附件条件,满足附加条件的规则,才认为是有效的规则
    PRIMARY KEY (`id`),
    UNIQUE KEY `name` (`name`)
) ENGINE=MyISAM  DEFAULT CHARSET=utf8;
-- ----------------------------
-- think_auth_role 角色组表，
-- id：主键， role_name:角色组中文名称， rules：角色组拥有的规则id， 多个规则","隔开，status 状态：为1正常，为0禁用
-- ----------------------------
DROP TABLE IF EXISTS `think_auth_role`;
    CREATE TABLE `think_auth_role` (
    `role_id` mediumint(8) unsigned NOT NULL AUTO_INCREMENT,
    `role_name` char(100) NOT NULL DEFAULT '',
    `status` tinyint(1) NOT NULL DEFAULT '1',
    `rules` char(80) NOT NULL DEFAULT '',
    PRIMARY KEY (`id`)
) ENGINE=MyISAM  DEFAULT CHARSET=utf8;
-- ----------------------------
-- think_auth_role_access 角色组明细表
-- uid:用户id，role_id：角色组id
-- ----------------------------
DROP TABLE IF EXISTS `think_auth_role_access`;
    CREATE TABLE `think_auth_role_access` (
    `uid` mediumint(8) unsigned NOT NULL,
    `role_id` mediumint(8) unsigned NOT NULL,
    UNIQUE KEY `uid_role_id` (`uid`,`role_id`),
    KEY `uid` (`uid`),
    KEY `role_id` (`role_id`)
) ENGINE=MyISAM DEFAULT CHARSET=utf8;
 */

class Auth
{
    /**
     * @var object 对象实例
     */
    protected static $instance;
    /**
     * 当前请求实例
     * @var Request
     */
    protected $request;


    //默认配置
    protected $config = [
        'auth_on' => 1, // 权限开关
        'auth_type' => 1, // 认证方式，1为实时认证；2为登录认证。
        'auth_role' => 'auth_role', // 角色组数据表名
        'auth_role_access' => 'auth_role_access', // 用户-角色组关系表
        'auth_rule' => 'auth_rule', // 权限规则表
        'auth_user' => 'users', // 用户信息表
    ];


    /**
     * 类架构函数
     * Auth constructor.
     */
    public function __construct()
    {

        //可设置配置项 auth, 此配置项为数组。
        if ($auth = Config('auth.')) {
            $this->config = array_merge($this->config, $auth);
        }
       // print_r(request());
        // 初始化request
         $this->request = request();
    }

    /**
     * 初始化
     * @access public
     * @param array $options 参数
     * @return \think\Request
     */
    public static function instance($options = [])
    {
        // if (is_null(self::$instance)) {
        //     self::$instance = new static($options);
        // }
        // return self::$instance;

    }

    /**
     * 检查权限
     * @param $name string|array  需要验证的规则列表,支持逗号分隔的权限规则或索引数组
     * @param $uid  int           认证用户的id
     * @param int $type 认证类型
     * @param string $mode 执行check的模式
     * @param string $relation 如果为 'or' 表示满足任一条规则即通过验证;如果为 'and'则表示需满足所有规则才能通过验证
     * @return bool               通过验证返回true;失败返回false
     */
    public function check($name, $uid, $type = 1, $mode = 'url', $relation = 'or')
    {

 
        if (!$this->config['auth_on']) {
            return true;
        }

        // 获取用户需要验证的所有有效规则列表
        $authList = $this->getAuthList($uid,$type);

        //具有超级管理员权限
        if($authList == 'all'){
           return true;
        }

        if (is_string($name)) {
           //传入字符串
            $name = strtolower($name);
            if (strpos($name, ',') !== false) {
                $name = explode(',', $name);
            } else {
                $name = [$name];
            }
        }

        $list = []; //保存验证通过的规则名
        if ('url' == $mode) {
            $REQUEST = unserialize(strtolower(serialize($this->request->param())));
        }

        foreach ($authList as $auth) {

            $query = preg_replace('/^.+\?/U', '', $auth);
            if ('url' == $mode && $query != $auth) {
                parse_str($query, $param); //解析规则中的param
                $intersect = array_intersect_assoc($REQUEST, $param);
                $auth = preg_replace('/\?.*$/U', '', $auth);
              // return $auth;
                if (in_array($auth, $name) && $intersect == $param) {
                    //如果节点相符且url参数满足
                    $list[] = $auth;
                }
            } else {
                if (in_array($auth, $name)) {
                    $list[] = $auth;
                }
            }
        }
        if ('or' == $relation && !empty($list)) {
            return true;
        }
        $diff = array_diff($name, $list);
        if ('and' == $relation && empty($diff)) {
            return true;
        }

        return false;
    }

    /**
     * 根据用户id 获取角色组,返回值为数组
     * @param  $uid int     用户id
     * @return array       用户所属的角色组 array(
     *     array('uid'=>'用户id','role_id'=>'角色组id','role_name'=>'角色组名称','rules'=>'角色组拥有的规则id,多个,号隔开'),
     *     ...)
     */
    public function getRoles($uid)
    {
        static $roles = [];
        if (isset($roles[$uid])) {
            return $roles[$uid];
        }
        // 转换表名
        $auth_role_access = $this->config['auth_role_access'];
        $auth_role = $this->config['auth_role'];
        // 执行查询
        $user_groups = Db::view($auth_role_access, 'uid,role_id')
            ->view($auth_role, 'role_name,rules', "{$auth_role_access}.role_id={$auth_role}.role_id", 'LEFT')
            ->where("{$auth_role_access}.uid='{$uid}' and {$auth_role}.status='1'")
            ->select();
        $roles[$uid] = $user_groups ?: [];

        return $roles[$uid];
    }

    /**
     * 获得权限列表
     * @param integer $uid 用户id
     * @param integer $type
     * @return array
     */
    public  function getAuthList($uid, $type)
    {
        static $_authList = []; //保存用户验证通过的权限列表
        $t = implode(',', (array)$type);
        if (isset($_authList[$uid . $t])) {
            return $_authList[$uid . $t];
        }
        if (2 == $this->config['auth_type'] && Session::has('_auth_list_' . $uid . $t)) {
            return Session::get('_auth_list_' . $uid . $t);
        }
        //读取用户所属角色组
        $roles = $this->getRoles($uid);
        if($roles[0]['rules'] == 'all'){
          return 'all';
        }
        $ids = []; //保存用户所属角色组设置的所有权限规则id
        foreach ($roles as $g) {
            $ids = array_merge($ids, explode(',', trim($g['rules'], ',')));
        }
        $ids = array_unique($ids);
        if (empty($ids)) {
            $_authList[$uid . $t] = [];
            return [];
        }
        
        $map[] = ['id','in',$ids];
        $map[] = ['type','=',$type];
  
        //读取角色组所有权限规则
        $rules = Db::name($this->config['auth_rule'])->where($map)->field('condition,name')->select();
        //循环规则，判断结果。
        $authList = []; //
        foreach ($rules as $rule) {
            if (!empty($rule['condition'])) {
                //根据condition进行验证
                $user = $this->getUserInfo($uid); //获取用户信息,一维数组
                $command = preg_replace('/\{(\w*?)\}/', '$user[\'\\1\']', $rule['condition']);
                //dump($command); //debug
                @(eval('$condition=(' . $command . ');'));
                if ($condition) {
                    $authList[] = strtolower($rule['name']);
                }
            } else {
                //只要存在就记录
                $authList[] = strtolower($rule['name']);
            }
        }
        $_authList[$uid . $t] = $authList;
        if (2 == $this->config['auth_type']) {
            //规则列表结果保存到session
            Session::set('_auth_list_' . $uid . $t, $authList);
        }

        return array_unique($authList);
    }

    /**
     * 获取权限组对应的权限列表
     * @param $gid
     * @param int $type
     * @return array|mixed
     */
   protected function getGroupAuthList($gid, $type = 1)
    {
        // 转换表名
        $auth_role = $this->config['auth_role'];
        $auth_rule = $this->config['auth_rule'];
        // 执行查询
        $rules = Db::name($auth_role)->where(['status'=>1, 'id'=>$gid])->value('rules');
        // 格式化access表id
        $ids = explode(',', trim($rules, ','));
        $ids = array_unique($ids);

        $map[] = ['id','in',$ids];
        $map[] = ['type','=',$type];
        $map[] = ['status','=',1];
        
        //读取角色组所有权限规则
        $rules = Db::name($auth_rule)->where($map)->column('title,name,condition');

        return $rules;
    }

    /**
     * 获得用户资料,根据自己的情况读取数据库
     */
    protected function getUserInfo($uid)
    {
        static $userinfo = [];

        $user = Db::name($this->config['auth_user']);
        // 获取用户表主键
        $_pk = is_string($user->getPk()) ? $user->getPk() : 'admin_uid';
        if (!isset($userinfo[$uid])) {
            $userinfo[$uid] = $user->where($_pk, $uid)->find();
        }

        return $userinfo[$uid];
    }
}