﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;

using System.Collections;
using System.Web;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Blog;
using SpaceBuilder.Common;
using SpaceBuilder.Utils;
using System.IO;
using System.Web.Routing;
using SpaceBuilder.Web;
using SpaceBuilder.LuceneSearch;
using SpaceBuilder.News;

namespace SpaceBuilder.Blog.Controllers
{
    /// <summary>
    /// 个人空间博客相关Controller
    /// </summary>
    [HttpCompress()]
    public class BlogController : UserDomainControllerBase
    {
        private int BlogApplicationID = ApplicationIDs.Instance().Blog();
        /// <summary>
        /// 空间博客默认首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Home(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            BlogSection blogSection = GetBlogSection(userDomainName);

            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            bool summaryShow = userCookie.GetBlogThreadListsMode();

            #region 计数 标题

            //用户博客浏览统计
            AddWebCount<BlogSectionCounter>(blogSection.OwnerUserID);
            string title = GetResourceString("Title_HisBlog", BlogApplicationID);
            if (IsUserDomainOwner(userDomainName))
                title = GetResourceString("Title_MyBlog", BlogApplicationID);
            SetPageTitle(userDomainName, title);
            //BreadCrumb.AddBreadCrumbItem(title, string.Empty);

            //添加rss自动发现
            Header.AddRssAutoDiscovery(SiteUrls.Instance().BlogRss(currentDomainUser.UserName));
            Header.AddRSDDiscovery(SiteUrls.Instance().Blog_RSD(blogSection.OwnerUserID));
            Header.AddWLWManifestDiscovery(SiteUrls.Instance().Blog_WLWManifest(blogSection.OwnerUserID));

            #endregion
            //if (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator)
            //    ViewData["isManage"] = true;
            return this.View("Pages/Blogs/Home.aspx", "Masters/Blog.master");
        }

        /// 博客全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty);
            parameters["SearchScopeUserID"] = currentDomainUser.UserID.ToString();

            return RedirectToAction("Search", "ChannelBlog", parameters);
        }


        #region BlogThread

        #region Show

        /// <summary>
        /// 根据标签显示文章列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListBlogThreadsByUserTag(string userDomainName, string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserTag userTag = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTag(tagName, currentDomainUser.UserID, true);

            if (userTag == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserTagNoFind", BlogApplicationID), tagName)));
            }

            #region 计数 标题
            AddWebCount<BlogSectionCounter>(GetBlogSection(userDomainName).OwnerUserID);
            SetPageTitle(userDomainName, userTag.TagName);
            //string pageName = string.Format(GetResourceString("Common_UserTag") + "：{0}", userTag.TagName);
            //BreadCrumb.AddBreadCrumbItem(pageName, string.Empty);
            #endregion

            return View("Pages/Blogs/ListBlogThreadsByUserTag.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 根据用户分类显示文章列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListBlogThreadsByUserCategory(string userDomainName, int categoryID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserCategory userCategory = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).GetUserCategory(categoryID, false);

            #region 验证 计数 标题
            AddWebCount<BlogSectionCounter>(GetBlogSection(userDomainName).OwnerUserID);
            SetPageTitle(userDomainName, userCategory.CategoryName);
            //string pageName = string.Format(GetResourceString("Common_Category") + "：{0}", userCategory.CategoryName);
            //BreadCrumb.AddBreadCrumbItem(pageName, string.Empty);
            #endregion


            return View("Pages/Blogs/ListBlogThreadsByUserCategory.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 按天显示文章列表
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListBlogThreadsByYMD(string userDomainName, int year, int? month, int? day)
        {
            DateTime dateTime = new DateTime(year, (month.HasValue ? (int)month : 1), (day.HasValue ? (int)day : 1));
            #region 计数 标题
            AddWebCount<BlogSectionCounter>(GetBlogSection(userDomainName).OwnerUserID);
            string title = string.Empty;
            if (day != null && (int)day > 0)
            {
                if (month == null || (!((int)month > 0)))
                //  throw new ArgumentException("请填写月份");
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_InputMonth", BlogApplicationID)));
                }
                switch ((int)month)
                {
                    case 1:
                    case 3:
                    case 5:
                    case 7:
                    case 8:
                    case 10:
                    case 12:
                        if ((int)day > 31)
                        {
                            return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_MonthNoFind", BlogApplicationID), (int)day)));
                        }
                        break;
                    case 2:
                    case 4:
                    case 6:
                    case 9:
                    case 11:
                        if ((int)day > 30)
                        {
                            return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_MonthNoFind", BlogApplicationID), (int)day)));
                        }
                        break;
                    default:
                        {
                            return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindMonth", BlogApplicationID), (int)month)));
                        }
                }
                if ((int)month == 2)
                {
                    if (((year % 4 == 0 || month % 100 == 0) && year % 400 != 0) && (int)day > 28)
                    {
                        return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_LeapYearFebruary", BlogApplicationID), (int)day)));
                    }
                }
                title = string.Format(GetResourceString("Title_Pattern_YearMonthDay", BlogApplicationID), year, (int)month, (int)day);
            }
            else if (month != null && (int)month > 0)
            {
                if ((int)month > 12)
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_MonthOneToTwelve", BlogApplicationID)));
                }
                title = string.Format(GetResourceString("Title_Pattern_YearMonth", BlogApplicationID), year, (int)month);
            }
            else
            {
                title = string.Format(GetResourceString("Title_Pattern_Year", BlogApplicationID), year);
            }
            SetPageTitle(userDomainName, title);
            //BreadCrumb.AddBreadCrumbItem(title, string.Empty);
            #endregion

            return View("Pages/Blogs/ListBlogThreadsByYMD.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 列表显示朋友的博客文章
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFriendsBlogThreads(string userDomainName, int? pageIndex)
        {
            #region 标题
            string title = GetResourceString("Title_FriendBlog", BlogApplicationID);
            SetPageTitle(userDomainName, title);
            //BreadCrumb.AddBreadCrumbItem(title, string.Empty);
            #endregion
            return View("Pages/Blogs/ListFriendsBlogThreads.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 显示博客文章内容
        /// </summary>
        /// <param name="userDomainName">博客所有者用户名</param>
        /// <param name="postID">博客文章的PostID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowBlogThread(string userDomainName, int threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            BlogSection blogSection = GetBlogSection(userDomainName);
            bool hasManagePermission = HasManagePermission(currentDomainUser);
            ViewData["hasManagePermission"] = hasManagePermission;
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, hasManagePermission);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }
            if (blogThread.OwnerUserID != currentDomainUser.UserID)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserNoReleaseArticle", BlogApplicationID), currentDomainUser.DisplayName, blogThread.Subject)));
            }

            List<Attachment> attachments = BlogAttachmentManager.Instance().GetAttachmentsByAssociateID(blogThread.ThreadID, hasManagePermission);
            ViewData["attachements"] = attachments;

            #region 处理审核
            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Blog()).AuditingStatusForPublicDisplay;
            if ((int)blogThread.AuditingStatus < auditingStatusValue)
            {
                if (blogThread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_ArticleNotAuditing", BlogApplicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_ArticleAuditing", BlogApplicationID);
                ViewData["errorAuditingStatus"] = true;
            }
            #endregion

            bool ismyFriend = false;

            if (CurrentUser == null || (CurrentUser != null && !IsUserDomainOwner(userDomainName) && !CurrentUser.IsContentAdministrator))
            {
                switch (blogThread.PrivacyStatus)
                {
                    case PrivacyStatuses.NeedPassword:
                        UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                        ViewData["passwordRight"] = userCookie.AuthorizeBlogThreadID(threadID);

                        break;
                    case PrivacyStatuses.OnlyFriend:
                        //仅朋友可见
                        ismyFriend = Friends.IsFriend(CurrentUser != null ? CurrentUser.UserID : 0, currentDomainUser.UserID);
                        if (!ismyFriend)
                        {
                            ViewData["ismyFriend"] = ismyFriend;
                            return View("Pages/Blogs/ShowBlogThread.aspx", "Masters/Blog.master", blogThread);
                        }
                        break;
                    case PrivacyStatuses.Privacy:
                        // 私密
                        return View("Pages/Blogs/ShowBlogThread.aspx", "Masters/Blog.master", blogThread);
                    case PrivacyStatuses.Public:
                        // 公开
                        break;
                    default:
                        break;
                }
            }



            #region 相关文章

            //List<BlogThread> correlativeBlogThreads = new List<BlogThread>();
            //List<int> correlativeBlogThreadIDs = new List<int>();
            //SearchResultDataSet<BlogThread> srs;

            //BlogThreadFullTextQuery fullTextQuery = new BlogThreadFullTextQuery();
            //fullTextQuery.PageSize = 11;
            //if (blogThread.Tags != null && blogThread.Tags.Count > 0)
            //{
            //    string[] tagNames = new string[blogThread.Tags.Count];
            //    for (int i = 0; i < blogThread.Tags.Count; i++)
            //    {
            //        tagNames[i] = blogThread.Tags[i].TagName;
            //    }
            //    fullTextQuery.TagNames = tagNames;

            //    try
            //    {
            //        srs = BlogSearchManager.Instance().Search(fullTextQuery);
            //        if (srs != null)
            //        {
            //            correlativeBlogThreads.AddRange(srs.Records);
            //            foreach (var correlativeBlogThread in correlativeBlogThreads)
            //            {
            //                correlativeBlogThreadIDs.Add(correlativeBlogThread.ThreadID);
            //            }
            //        }
            //    }
            //    catch { }
            //}

            ////如果不足6条相关文章，则从用户类别中查找其他文章
            //if (correlativeBlogThreads.Count < 6 && blogThread.UserCategoryID > 0)
            //{
            //    fullTextQuery.TagNames = null;
            //    fullTextQuery.UserCategoryID = blogThread.UserCategoryID;
            //    try
            //    {
            //        srs = BlogSearchManager.Instance().Search(fullTextQuery);
            //        if (srs != null)
            //        {

            //            foreach (var correlativeBlogThread in srs.Records)
            //            {
            //                if (!correlativeBlogThreadIDs.Contains(correlativeBlogThread.ThreadID))
            //                {
            //                    correlativeBlogThreads.Add(correlativeBlogThread);
            //                    correlativeBlogThreadIDs.Add(correlativeBlogThread.ThreadID);
            //                }

            //                if (correlativeBlogThreads.Count > 11)
            //                    break;
            //            }
            //        }
            //    }
            //    catch { }
            //}

            ////如果不足6条相关文章，则根据标题关键字查找其他文章
            //if (correlativeBlogThreads.Count < 6)
            //{
            //    fullTextQuery.TagNames = null;
            //    fullTextQuery.UserCategoryID = -1;
            //    fullTextQuery.Keyword = blogThread.Subject;
            //    try
            //    {
            //        srs = BlogSearchManager.Instance().Search(fullTextQuery);
            //        if (srs != null)
            //        {
            //            foreach (var correlativeBlogThread in srs.Records)
            //            {
            //                if (!correlativeBlogThreadIDs.Contains(correlativeBlogThread.ThreadID))
            //                {
            //                    correlativeBlogThreads.Add(correlativeBlogThread);
            //                    correlativeBlogThreadIDs.Add(correlativeBlogThread.ThreadID);
            //                }

            //                if (correlativeBlogThreads.Count > 11)
            //                    break;
            //            }
            //        }
            //    }
            //    catch { }
            //}


            #endregion

            #region 计数 标题
            AddWebCount<BlogSectionCounter>(blogSection.OwnerUserID);
            AddWebCount<BlogThreadCounter>(blogThread.ThreadID);
            base.SetPageTitle(userDomainName, blogThread.Subject);
            #endregion

            #region 设置Meta标签

            //设置Meta标签：Description和Keywords
            string metaDescription = string.Empty;
            if (string.IsNullOrEmpty(blogThread.Excerpt))
                metaDescription = string.Format("{0}：{1}", blogThread.Subject, HtmlUtils.TrimHtml(blogThread.GetBody(false), 150));
            else
                metaDescription = string.Format("{0}：{1}", blogThread.Subject, StringUtils.Trim(blogThread.GetBody(false), 150));
            Header.AddMetaKeywords(blogThread.Keywords);
            Header.AddMetaDescription(metaDescription);

            #endregion

            bool isRecommended = false;

            IList<RecommendedItem> recommendItems = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Blog(), null, ValueHelper.GetSafeSqlInt(int.MaxValue));
            if (recommendItems != null && recommendItems.Count > 0)
            {
                foreach (RecommendedItem recommendedItem in recommendItems)
                {
                    if (recommendedItem.ItemID == blogThread.ThreadID)
                    {
                        isRecommended = true;
                        break;
                    }
                }
            }

            bool isManager = false;
            bool isSupManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManager = true;
            if (CurrentUser != null && CurrentUser.IsContentAdministrator)
                isSupManager = true;

            IList<Rating> ratings = BlogThreads.GetRatings(threadID, 30);
            ViewData["ratings"] = ratings;

            ViewData["isRecommended"] = isRecommended;
            ViewData["isManager"] = isManager;
            ViewData["isSupManager"] = isSupManager;
            ViewData["topSpecialOrder"] = topSpecialOrder;
            //ViewData["RelatedBlogThreads"] = correlativeBlogThreads;
            ViewData["ismyFriend"] = ismyFriend;
            //BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(blogThread.Subject, 10), string.Empty);
            return View("Pages/Blogs/ShowBlogThread.aspx", "Masters/Blog.master", blogThread);
        }

        /// <summary>
        /// 相关文章
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="threadID"></param>
        /// <param name="title"></param>
        /// <returns></returns>
        public ActionResult Control_CorrelativeBlogThreads(string userDomainName, int threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool hasManagePermission = HasManagePermission(currentDomainUser);
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, hasManagePermission);
            ICollection<BlogThread> correlativeBlogThreads = null;
            try
            {
                correlativeBlogThreads = BlogSearchManager.Instance().GetCorrelativeThreads(blogThread, 10);
            }
            catch
            {
                correlativeBlogThreads = new List<BlogThread>();
            }
            return PartialView("Controls/Blogs/CorrelativeBlogThreads.ascx", correlativeBlogThreads);
        }

        /// <summary>
        /// 显示我的朋友的最新博客
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ListFriendsBlogThreads(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (!pageIndex.HasValue)
                pageIndex = 0;
            PagingDataSet<BlogSection> friendsBlogSections = BlogSections.GetFirendRecentBlogThreads(currentDomainUser.UserID, pageIndex.Value, 10);

            return PartialView("Controls/Blogs/ListFriendsBlogThreads.ascx", friendsBlogSections);
        }

        /// <summary>
        /// 管理博客文章
        /// </summary>
        public ActionResult Control_ManageBlogThreads(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool hasPermission = HasManagePermission(userDomainName);
            PagingDataSet<BlogThread> blogThreads = BlogThreads.GetMyThreads(currentDomainUser.UserID, string.Empty, null, pageIndex ?? 1);
            ViewData["isFriend"] = CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID);
            ViewData["hasPermission"] = hasPermission;
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTags(currentDomainUser.UserID, true);
            List<UserCategory> userCategoryList = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).GetUserCategories(currentDomainUser.UserID, true);
            List<SiteCategory> allSiteCategoryList = SiteCategories.Instance(BlogApplicationID).GetAll(false);
            ViewData["userTags"] = userTags;
            ViewData["userCategoryList"] = userCategoryList;
            ViewData["allSiteCategoryList"] = allSiteCategoryList;

            return PartialView("Controls/Blogs/ManageBlogThreads.ascx", blogThreads);
        }



        /// <summary>
        /// 列表显示博客文章
        /// </summary>
        public ActionResult Control_ListBlogThreads(string userDomainName, bool? isSummary, int? categoryID, string tagName, int? year, int? month, int? day, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool hasPermission = HasManagePermission(userDomainName);
            PagingDataSet<BlogThread> blogThreads = null;

            if (year.HasValue && year.Value > 0 || month.HasValue && month.Value > 0 || day.HasValue && day.Value > 0)
            {
                if (day.HasValue && (int)day > 0)
                    blogThreads = BlogThreads.GetThreadsForArchive(currentDomainUser.UserID, BlogThreadArchiveTypes.Day, new DateTime(year.Value, month.Value, day.Value), hasPermission, pageIndex ?? 1);
                else if (month.HasValue && (int)month > 0)
                    blogThreads = BlogThreads.GetThreadsForArchive(currentDomainUser.UserID, BlogThreadArchiveTypes.Month, new DateTime(year.Value, month.Value, 1), hasPermission, pageIndex ?? 1);
                else
                    blogThreads = BlogThreads.GetThreadsForArchive(currentDomainUser.UserID, BlogThreadArchiveTypes.Year, new DateTime(year.Value, 1, 1), hasPermission, pageIndex ?? 1);
            }
            else
            {
                if (hasPermission)
                    blogThreads = BlogThreads.GetMyThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex ?? 1);
                else
                    blogThreads = BlogThreads.GetThreads(currentDomainUser.UserID, tagName, categoryID, pageIndex ?? 1);
            }
            ViewData["isFriend"] = CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID);
            ViewData["hasPermission"] = hasPermission;
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTags(currentDomainUser.UserID, true);
            List<UserCategory> userCategoryList = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).GetUserCategories(currentDomainUser.UserID, true);
            List<SiteCategory> allSiteCategoryList = SiteCategories.Instance(BlogApplicationID).GetAll(false);
            ViewData["userTags"] = userTags;
            ViewData["userCategoryList"] = userCategoryList;
            ViewData["allSiteCategoryList"] = allSiteCategoryList;

            string boxTitle = string.Empty;
            if (categoryID > 0)
                boxTitle = GetResourceString("Common_Category") + "：" + UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).GetUserCategory(categoryID ?? 0, false).CategoryName;
            else if (!string.IsNullOrEmpty(tagName))
                boxTitle = GetResourceString("Common_UserTag") + "：" + tagName;
            else if (year != null && (month == null || month <= 0) && (day == null || day <= 0))
                boxTitle = string.Format(GetResourceString("Title_Pattern_Year", BlogApplicationID), year);
            else if (year != null && month != null && (day == null || day <= 0))
                boxTitle = string.Format(GetResourceString("Title_Pattern_YearMonth", BlogApplicationID), year, (int)month);
            else if (year != null && month != null && day != null)
                boxTitle = string.Format(GetResourceString("Title_Pattern_YearMonthDay", BlogApplicationID), year, (int)month, (int)day);

            ViewData["boxTitle"] = boxTitle;

            if (isSummary.HasValue && isSummary.Value)
                return PartialView("Controls/Blogs/SummaryBlogThreads.ascx", blogThreads);
            else
                return PartialView("Controls/Blogs/ListBlogThreads.ascx", blogThreads);
        }

        /// <summary>
        /// 显示日历
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ShowCalendar(string userDomainName, int? year, int? month, int? day)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            DateTime nowShowDateTime = DateTime.Now;
            if (day != null && (int)day > 0)
            {
                if (year != null && month != null && (int)year > 0 && (int)month > 0)
                    nowShowDateTime = new DateTime((int)year, (int)month, (int)day);
            }
            else if (month != null && (int)month > 0)
            {
                if (year != null && (int)year > 0)
                    nowShowDateTime = new DateTime((int)year, (int)month, 1);
            }
            else if (year != null && (int)year > 0)
            {
                nowShowDateTime = new DateTime((int)year, 1, 1);
            }
            Dictionary<int, ArchiveDataItem> dictionary = BlogThreads.GetStatsByMonth(currentDomainUser.UserID, nowShowDateTime);
            Dictionary<int, string> calendarDictionary = new Dictionary<int, string>();

            foreach (int setDay in dictionary.Keys)
            {
                calendarDictionary.Add(setDay, string.Format(GetResourceString("Description_Pattern_AllArticle", BlogApplicationID), dictionary[setDay].Count));
            }

            ViewData["calendarDictionary"] = calendarDictionary;
            ViewData["nowShowDateTime"] = nowShowDateTime;
            return PartialView("Controls/ShowCalendar.ascx");
        }


        /// <summary>
        /// 显示输入密码框
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditPassWord(string userDomainName)
        {
            ViewData["actionUrl"] = SPBUrlHelper.Action("SetBlogThreadsPrivacyStatus", "Blog", new RouteValueDictionary { { "userDomainName", userDomainName }, { "stause", PrivacyStatuses.NeedPassword } });
            return PartialView("Controls/Common/EditPassWord.ascx");
        }



        #endregion

        #region Manage

        /// <summary>
        /// 新建博客文章
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult NewBlogThread(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ReleaseArticl", BlogApplicationID));
            #endregion

            List<BlogThreadType> blogThreadTypes = BlogThreads.GetAllBlogThreadTypes(false);
            BlogSection currentBlogSection = BlogSections.GetSection(currentDomainUser.UserID, false);
            ViewData["blogTypes"] = blogThreadTypes;
            ViewData["CurrentDomainUserType"] = currentDomainUser.UserType;
            BlogThread thread = null;
            if (TempData["BlogThread"] != null)
                thread = TempData.Get<BlogThread>("BlogThread");
            if (thread == null)
            {
                thread = new BlogThread();
                thread.PrivacyStatus = currentBlogSection.DefaultPrivacyStatus;
                thread.Password = currentBlogSection.DefaultPrivacyPassword;
            }
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Button_ReleaseArticle", BlogApplicationID), string.Empty);
            return this.View("Pages/Blogs/EditBlogThread.aspx", "Masters/Blog.master", thread);
        }

        /// <summary>
        /// 创建博客文章
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateBlogThread(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull", BlogApplicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            BlogSection currentBlog = GetBlogSection(userDomainName);
            string metaKeyWords = string.Empty;

            BlogThread blogThread = new BlogThread();

            blogThread.OwnerUserID = currentDomainUser.UserID;
            blogThread.Author = currentDomainUser.DisplayName;

            //设置文章主题内容和摘要
            blogThread.Subject = Request.Form.GetString("PostSubject", string.Empty).Trim();
            blogThread.BlogThreadTypeID = Request.Form.GetInt("BlogThreadType", -1);
            if (BlogSectionConfiguration.Instance().EableUserEntrySiteCategory)
                blogThread.SiteCategoryID = Request.Form.GetInt("blogSiteCategoryDDL", 0);
            blogThread.UserCategoryID = Request.Form.Get<int>("blogUserCategoryDDL", 0);
            blogThread.BodyForDataTransmission = Request.Form.GetString("PostBody", string.Empty);
            blogThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("setBlogThreadsPirvacyStatus", (int)PrivacyStatuses.Public);
            blogThread.Password = Request.Form.GetString("setBlogThreadsPirvacyStatus_Password", string.Empty);
            blogThread.Excerpt = Request.Form.GetString("postExcerpt", string.Empty);
            blogThread.Keywords = Request.Form.GetString("metaKeyWords", string.Empty);

            //设置文章的YesOrNo的一些属性
            PopulateFromYesNo(blogThread);
            blogThread.ModerationType = (CommentModerationType)Enum.Parse(typeof(CommentModerationType), Request.Form["CommentModerationDropDownList"], false);

            //设置标题图片            
            blogThread.UserFeaturedImageAttachmentID = Request.Form.GetInt("featurePicAttachmentID", -1);

            if (string.IsNullOrEmpty(blogThread.Subject))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_ArticleTextNotNull", BlogApplicationID);
                TempData["BlogThread"] = blogThread;
                return RedirectToAction("NewBlogThread", "Blog", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            if (string.IsNullOrEmpty(blogThread.BodyForDataTransmission))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_ArticleContentNotNull", BlogApplicationID);
                TempData["BlogThread"] = blogThread;
                return RedirectToAction("NewBlogThread", "Blog", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }


            #region 处理标签
            string selectedTagNames = Request.Form.GetString("userTags", string.Empty);
            List<UserTag> selectedTags = null;
            if (!string.IsNullOrEmpty(selectedTagNames))
            {
                string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (tagNames != null && tagNames.Length > 0)
                {
                    selectedTags = new List<UserTag>();
                    foreach (string tagName in tagNames)
                    {
                        if (!string.IsNullOrEmpty(tagName.Trim()))
                        {
                            UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTag(tagName, currentDomainUser.UserID, true);
                            if (tag != null)
                            {
                                selectedTags.Add(tag);
                            }
                            else
                            {
                                UserTag userTag = new UserTag();
                                userTag.UserID = currentDomainUser.UserID;
                                userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).CreateUserTag(userTag);
                                if (isSuccess)
                                    selectedTags.Add(userTag);
                            }
                        }
                    }
                }
            }
            #endregion

            if (string.IsNullOrEmpty(blogThread.Keywords))
                blogThread.Keywords = string.Format("{0},{1},{2}", string.IsNullOrEmpty(selectedTagNames) ? (selectedTagNames + ",") : string.Empty, blogThread.Subject + ",", blogThread.Author);

            BlogThread newBlogThread = BlogThreads.CreateThread(blogThread, selectedTags);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_CreateArticleSucces", BlogApplicationID);

            //设置标签
            if (selectedTags != null && selectedTags.Count > 0)
            {
                UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).UpdateUserTagsOfItem(blogThread.ThreadID, currentDomainUser.UserID, selectedTags.Select(n => n.TagName).ToList());
            }
            if (Request.Form.GetBool("isTranspondToNews", false))
            {
                Post post = new Post();
                post.Subject = blogThread.Subject;
                post.BodyForDataTransmission = blogThread.BodyForDataTransmission;
                post.Excerpt = blogThread.Excerpt;
                post.OwnerUserID = blogThread.OwnerUserID;
                post.Author = blogThread.Author;
                post.FromUrl = Globals.GetFullUrlInMainSite(SiteUrls.Instance().ShowBlogThread(userDomainName, blogThread));
                post.Attachments = BlogAttachmentManager.Instance().GetAttachmentsByAssociateID(blogThread.ThreadID, true);

                TempData["post"] = post;

                //SpaceBuilder.News.NewsThread newsThread = new SpaceBuilder.News.NewsThread();

                //newsThread.Title = blogThread.Subject;
                //newsThread.BodyForDataTransmission = blogThread.BodyForDataTransmission;
                //newsThread.Excerpt = blogThread.Excerpt;
                //newsThread.Keywords = blogThread.Keywords;
                ////将文件的附件拷贝到资讯的临时附件中
                //List<Attachment> attachments = BlogAttachmentManager.Instance().GetAttachmentsByAssociateID(blogThread.ThreadID, true);
                //foreach (var attach in attachments)
                //{
                //    if (attach.File == null)
                //        continue;
                //    Attachment attachment = new Attachment(NewsAttachmentManager.Instance().StorageProviderName, null);
                //    attachment.UserID = attach.UserID;
                //    attachment.ContentSize = attach.ContentSize;
                //    attachment.ContentType = attach.ContentType;
                //    attachment.DateCreated = attach.DateCreated;
                //    attachment.FriendlyFileName = attach.FriendlyFileName;

                //    NewsAttachmentManager.Instance().CreateTemporaryAttachment(attachment, attach.File.OpenReadStream());
                //}


                ////设置标签

                //TempData["TagNames"] = selectedTagNames;
                //TempData["newsThread"] = newsThread;
                return RedirectToAction("BlogToNews", "News", new RouteValueDictionary { { "userDomainName", userDomainName } });


            }
            else
                return RedirectToAction("ShowBlogThread", "Blog", new RouteValueDictionary { { "threadID", blogThread.ThreadID }, { "userDomainName", userDomainName } });

        }

        /// <summary>
        /// 编辑博客文章
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult EditBlogThread(int threadID, string userDomainName)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, true);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_EditNoteNoFind", BlogApplicationID), threadID)));
            }
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ValidateItemOwner(blogThread.OwnerUserID);

            #region 标题
            SetPageTitle(userDomainName, string.Format(GetResourceString("Title_Pattern_Edit", BlogApplicationID), blogThread.Subject));
            #endregion

            List<BlogThreadType> blogTypes = BlogThreads.GetAllBlogThreadTypes(false);
            ViewData["blogTypes"] = blogTypes;
            ViewData["CurrentDomainUserType"] = currentDomainUser.UserType;
            ViewData["SelectedUserTags"] = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTagsOfItem(blogThread.ThreadID, HasManagePermission(currentDomainUser));
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditBlogArticle", BlogApplicationID), string.Empty);

            return this.View("Pages/Blogs/EditBlogThread.aspx", "Masters/Blog.master", blogThread);
        }

        /// <summary>
        /// 更新博客文章
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateBlogThread(int threadID, string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull", BlogApplicationID));
            User blogOwner = GetCurrentDomainUser(userDomainName);
            BlogSection currentBlog = GetBlogSection(userDomainName);
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, false);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }

            ValidateItemOwner(blogThread.OwnerUserID);

            //设置文章主题内容和摘要
            blogThread.Subject = Request.Form.GetString("PostSubject", string.Empty).Trim();
            blogThread.BlogThreadTypeID = Request.Form.GetInt("BlogThreadType", -1);
            if (BlogSectionConfiguration.Instance().EableUserEntrySiteCategory)
                blogThread.SiteCategoryID = Request.Form.GetInt("blogSiteCategoryDDL", 0);
            int oldUserCategoryID = blogThread.UserCategoryID;
            blogThread.UserCategoryID = Request.Form.Get<int>("blogUserCategoryDDL", 0);
            blogThread.BodyForDataTransmission = Request.Form.GetString("PostBody", string.Empty);
            blogThread.Excerpt = Request.Form.GetString("postExcerpt", string.Empty);
            blogThread.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("setBlogThreadsPirvacyStatus", (int)PrivacyStatuses.Public);
            blogThread.Password = Request.Form.GetString("setBlogThreadsPirvacyStatus_Password", string.Empty);
            blogThread.Keywords = Request.Form.GetString("metaKeyWords", string.Empty);

            //设置标题图片            
            blogThread.UserFeaturedImageAttachmentID = Request.Form.GetInt("featurePicAttachmentID", -1);

            if (string.IsNullOrEmpty(blogThread.Subject))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_ArticleTextNotNull", BlogApplicationID);
                return RedirectToAction("EditBlogThread", "Blog", new RouteValueDictionary { { "threadID", threadID }, { "userDomainName", userDomainName } });
            }
            else if (string.IsNullOrEmpty(blogThread.BodyForDataTransmission))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_ArticleContentNotNull", BlogApplicationID);
                return RedirectToAction("EditBlogThread", "Blog", new RouteValueDictionary { { "threadID", threadID }, { "userDomainName", userDomainName } });
            }
            else
            {
                //设置标签
                string selectedTagNames = Request.Form.Get<string>("userTags", string.Empty);
                List<UserTag> allTags = null;
                if (!string.IsNullOrEmpty(selectedTagNames))
                {
                    string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                    if (tagNames != null && tagNames.Length > 0)
                    {
                        allTags = new List<UserTag>();
                        foreach (string tagName in tagNames)
                        {
                            if (!string.IsNullOrEmpty(tagName.Trim()))
                            {
                                UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTag(tagName, blogThread.OwnerUserID, true);
                                if (tag != null)
                                {
                                    allTags.Add(tag);
                                }
                                else
                                {
                                    UserTag userTag = new UserTag();
                                    userTag.UserID = blogThread.OwnerUserID;
                                    userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                    bool isCreated = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).CreateUserTag(userTag);
                                    if (isCreated)
                                    {
                                        allTags.Add(userTag);
                                    }
                                }
                            }
                        }
                    }
                }

                //设置文章的YesOrNo的一些属性
                PopulateFromYesNo(blogThread);


                blogThread.ModerationType = (CommentModerationType)Enum.Parse(typeof(CommentModerationType), Request.Form["CommentModerationDropDownList"], false);
                //设置文章所推荐到的群组
                //SettingCommendClub(blogThread, blogOwner, blogOwner, Request.Form.Get<string>("blogClubCommedCheckBox"));

                BlogThreads.UpdateThread(blogThread, oldUserCategoryID, allTags);
                UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).UpdateUserTagsOfItem(blogThread.ThreadID, blogThread.OwnerUserID, allTags == null ? new List<string>() : allTags.Select(n => n.TagName).ToList());

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdateArticleSucces", BlogApplicationID);
                if (Request.Form.GetBool("isTranspondToNews", false))
                {
                    Post post = new Post();
                    post.Subject = blogThread.Subject;
                    post.BodyForDataTransmission = blogThread.BodyForDataTransmission;
                    post.Excerpt = blogThread.Excerpt;
                    post.OwnerUserID = blogThread.OwnerUserID;
                    post.Author = blogThread.Author;
                    post.FromUrl = Globals.GetFullUrlInMainSite(SiteUrls.Instance().ShowBlogThread(userDomainName, blogThread));
                    post.Attachments = BlogAttachmentManager.Instance().GetAttachmentsByAssociateID(blogThread.ThreadID, true);

                    TempData["post"] = post;


                    return RedirectToAction("BlogToNews", "News", new RouteValueDictionary { { "userDomainName", userDomainName } });

                }
                else
                    return RedirectToAction("ShowBlogThread", "Blog", new RouteValueDictionary { { "threadID", threadID }, { "userDomainName", userDomainName } });
            }
        }

        /// <summary>
        /// 删除某篇博客文章
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteBlogThread(int threadID, string userDomainName, bool isManage)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, true);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }

            ValidateItemOwner(blogThread.OwnerUserID);
            BlogThreads.DeleteThread(threadID);
            bool isShowBlogThread = false;
            isShowBlogThread = Request.QueryString.Get<bool>("isShowBlogThread", false);
            if (isShowBlogThread)
                return RedirectToAction("Home", new RouteValueDictionary { { "userDomainName", userDomainName } });
            else if (!isManage)
                return RedirectToAction("Control_ListBlogThreads", CurrentRouteValueDictionary);
            else
                return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 删除某篇博客文章（不重新加载页面）
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public JsonResult AjaxDeleteBlogThread(int threadID, string userDomainName)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, true);
            if (blogThread == null)
            {
                return Json(new { Fail = true, error404 = SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)) }, JsonRequestBehavior.AllowGet);
            }

            ValidateItemOwner(blogThread.OwnerUserID);
            //BlogThreads.DeleteThread(threadID);
            return Json(new { Success = true }, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 删除所有选中的博客文章
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteBlogThreads(string userDomainName)
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> deleteBlogThreadIDs = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        int id = int.Parse(postID);
                        BlogThread blogThread = BlogThreads.GetThread(id, null, true);
                        if (blogThread != null)
                        {
                            ValidateItemOwner(blogThread.OwnerUserID);
                            deleteBlogThreadIDs.Add(id);
                        }
                    }
                    BlogThreads.DeleteThreads(deleteBlogThreadIDs);
                }
            }
            return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置 博客隐私设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetBlogThreadsPrivacyStatus(string userDomainName, PrivacyStatuses stause)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                PrivacyStatuses setPrivacyStatus = stause;
                string setPassword = string.Empty;
                if (stause == PrivacyStatuses.NeedPassword)
                    setPassword = Request.Form.GetString("ThreadsPirvacyStatus_Password", string.Empty);
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    List<int> setBlogThreadIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        BlogThread blogThread = BlogThreads.GetThread(id, null, false);
                        if (blogThread != null)
                        {
                            ValidateItemOwner(blogThread.OwnerUserID);
                            setBlogThreadIDs.Add(id);
                        }
                    }
                    BlogThreads.UpdatePrivacyStatus(setBlogThreadIDs, setPrivacyStatus, setPassword, currentDomainUser.UserID);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "修改成功！";

                }
            }
            if (stause == PrivacyStatuses.NeedPassword)
                return RedirectToAction("Control_EditPassWord");
            else
                return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置标签
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetBlogThreadsTags(string userDomainName)
        {
            User blogOwner = GetCurrentDomainUser(userDomainName);
            string[] setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (setPostIDs != null && setPostIDs.Length > 0)
            {
                string selectedTagNames = Request.Form.GetString("setUserTagss", string.Empty);
                if (string.IsNullOrEmpty(selectedTagNames))
                    return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
                string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                List<string> tNames = null;
                if (tagNames != null && tagNames.Length > 0)
                {
                    tNames = new List<string>();
                    foreach (string tagName in tagNames)
                    {
                        UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTag(tagName, blogOwner.UserID, false);
                        if (tag != null)
                            tNames.Add(tag.TagName);
                    }
                }

                foreach (string postID in setPostIDs)
                {
                    int id = int.Parse(postID);
                    BlogThread blogThread = BlogThreads.GetThread(id, null, true);
                    if (blogThread != null)
                    {
                        UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).UpdateUserTagsOfItem(blogThread.ThreadID, blogThread.OwnerUserID, tNames);
                    }
                }
            }
            return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置用户分类
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetBlogThreadsUserCategory(string userDomainName, int? userCategoryID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (!userCategoryID.HasValue && !string.IsNullOrEmpty(Request.Form.GetString("categoryName", string.Empty).Trim()))
            {
                UserCategory userCategory = new UserCategory();
                userCategory.UserID = currentDomainUser.UserID;

                userCategory.CategoryName = Request.Form.GetString("categoryName", string.Empty).Trim();
                userCategory.CategoryName = HtmlUtils.TrimHtml(userCategory.CategoryName, 20);
                userCategory.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(userCategory.CategoryName);

                UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).CreateUserCategory(userCategory);
                userCategoryID = userCategory.CategoryID;
            }

            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    //int UserCategoryID = Request.Form.GetInt("blogUserCategoryDDL", 0);
                    List<int> setBlogThreadIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        BlogThread blogThread = BlogThreads.GetThread(id, null, true);
                        if (blogThread != null)
                        {
                            setBlogThreadIDs.Add(id);
                        }
                    }
                    BlogThreads.UpdateThreadsUserCategory(setBlogThreadIDs, userCategoryID ?? 0, currentDomainUser.UserID);
                }
            }
            return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量设置站点分类
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetBlogThreadsSiteCategory(string userDomainName, int siteCategoryID)
        {
            string[] setPostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("ThreadCheckBoxGroup", string.Empty)))
            {
                //int siteCategoryID = Request.Form.GetInt("blogSiteCategoryDDL", 0);
                setPostIDs = Request.Form.GetString("ThreadCheckBoxGroup", string.Empty).Split(',');
                if (setPostIDs != null && setPostIDs.Length > 0)
                {
                    List<int> setBlogThreadIDs = new List<int>();
                    foreach (string postID in setPostIDs)
                    {
                        int id = int.Parse(postID);
                        BlogThread blogThread = BlogThreads.GetThread(id, null, true);
                        if (blogThread != null)
                        {
                            setBlogThreadIDs.Add(id);
                        }
                    }
                    BlogThreads.UpdateThreadsSiteCategory(setBlogThreadIDs, siteCategoryID);
                }
            }
            return RedirectToAction("Control_ManageBlogThreads", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 设置为精华
        /// </summary>
        public ActionResult SetEssential(string userDomainName, int threadID)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, true);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ArticleNoFound", BlogApplicationID), threadID)));
            }
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Exception_AdminSetHighLimit"));
            }

            blogThread.IsEssential = true;
            BlogThreads.SetEssential(blogThread);

            return new EmptyResult();
        }

        /// <summary>
        /// 取消精华设置
        /// </summary>
        public ActionResult CancelEssential(string userDomainName, int threadID)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, true);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ArticleNoFound", BlogApplicationID), threadID)));
            }
            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
            {
                return new EmptyResult();
                //throw new ArgumentException("只有管理员才有取消精华的权限！");
            }

            blogThread.IsEssential = false;
            BlogThreads.SetEssential(blogThread);
            return new EmptyResult();
        }

        /// <summary>
        /// 设为置顶
        /// </summary>
        /// <param name="threadID">需要置顶的ThreadID</param>
        public ActionResult SetTop(string userDomainName, int threadID)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, PrivacyStatuses.Public, true);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ArticleNoFound", BlogApplicationID), threadID)));
            }
            if (CurrentUser != null && CurrentUser.IsContentAdministrator)
            {
                blogThread.SpecialOrder = this.topSpecialOrder;
                BlogThreads.SetSticky(blogThread);
            }
            else
            {
                return new EmptyResult();
                //throw new ArgumentException(GetResourceString("Exception_AdministratorSetTop", BlogApplicationID));
            }
            return new EmptyResult();
        }

        /// <summary>
        /// 取消置顶
        /// </summary>
        /// <param name="threadID">取消置顶的ThreadID</param>
        public ActionResult CancelTop(string userDomainName, int threadID)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, true);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ArticleNoFound", BlogApplicationID), threadID)));
            }
            List<int> threadIDs = new List<int>();
            threadIDs.Add(threadID);

            if (CurrentUser != null && CurrentUser.IsContentAdministrator)
            {
                blogThread.SpecialOrder = 0;
                BlogThreads.SetSticky(blogThread);
            }
            // else
            //throw new ArgumentException(GetResourceString("Exception_AdministratorSetTop", BlogApplicationID));
            return new EmptyResult();
        }

        #endregion

        #endregion

        #region BlogComment

        #region Show

        /// <summary>
        /// 博客最新评论
        /// </summary>
        /// <param name="userDomainName">博客用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_NewBlogComment(string userDomainName, int threadID, int pageIndex, bool? isReplay)
        {
            BlogSection blogSection = GetBlogSection(userDomainName);
            ViewData["BlogSection"] = blogSection;
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            ViewData["rememberMyName"] = userCookie.GetRememberMeInfor_name();
            ViewData["rememberMyUrl"] = userCookie.GetRememberMeInfor_url();
            return PartialView("Controls/Blogs/NewBlogComment.ascx");
        }
        /// <summary>
        /// 博客最新评论
        /// </summary>
        /// <param name="userDomainName">博客用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_NewBlogComments(string userDomainName)
        {
            BlogSection blogSection = GetBlogSection(userDomainName);
            IList<BlogComment> blogComments = BlogComments.GetTopComments(blogSection.OwnerUserID, 10);
            return PartialView("Controls/Blogs/NewBlogComments.ascx", blogComments);
        }

        /// <summary>
        /// 博客文章评论列表以及评论
        /// </summary>
        /// <param name="postID">博客文章的PostID</param>
        /// <param name="pageIndex">当前页数</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_BlogThreadComments(string userDomainName, int threadID, int? pageIndex)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, false);
            bool isManager = false;
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(String.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }

            if (CurrentUser != null && (blogThread.OwnerUserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator))
                isManager = true;
            if (!pageIndex.HasValue)
                pageIndex = CommentsPageIndex(userDomainName, blogThread, pageIndex ?? 1);
            PagingDataSet<BlogComment> blogComments = BlogComments.GetCommentsOfThread(blogThread.ThreadID, pageIndex ?? 1, isManager);

            ViewData["isManager"] = isManager;
            ViewData["blogThread"] = blogThread;

            return PartialView("Controls/Blogs/BlogThreadComments.ascx", blogComments);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 添加博客文章评论
        /// </summary>
        /// <param name="threadID">博客文章ThreadID</param>
        /// <param name="postID">博客文章PostID</param>
        /// <param name="pageIndex">返回显示页面</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult CreateBlogThreadComment(string userDomainName, int threadID, int pageIndex, bool? isReplay)
        {
            int parentID = Request.QueryString.GetInt("parentID", 0);
            string commentBody = Request.Form.GetString("commentBody", string.Empty);
            string url = Request.Form.GetString("url", string.Empty);
            string author = Request.Form.GetString("author", string.Empty);
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();
                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    if (CurrentUser == null)
                    {
                        TempData["url"] = url;
                        TempData["author"] = author;
                    }
                    TempData["commentBody"] = commentBody;
                    TempData["parentID"] = parentID;
                    return RedirectToAction("Control_BlogThreadComments", CurrentRouteValueDictionary);
                }
            }

            BlogThread thread = BlogThreads.GetThread(threadID, null, true);
            if (thread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_AddComment", BlogApplicationID));
            #endregion

            BlogComment entry = new BlogComment();
            entry.TitleUrl = Request.Form.GetString("CommenterUrl", string.Empty).Trim();
            entry.Body = commentBody;
            entry.ParentID = parentID;
            entry.Subject = "re:" + thread.Subject;
            if (string.IsNullOrEmpty(entry.Body))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_InputComment", BlogApplicationID);
                TempData["parentID"] = entry.ParentID;
                return RedirectToAction("Control_BlogThreadComments", CurrentRouteValueDictionary);
            }
            entry.ThreadID = thread.ThreadID;
            entry.OwnerUserID = thread.OwnerUserID;
            entry.PostDate = DateTime.Now;

            switch (thread.BlogSection.ModerationType)
            {
                case CommentModerationType.All:
                    entry.IsApproved = false;
                    break;
                case CommentModerationType.Anonymous:
                    if (CurrentUser == null)
                        entry.IsApproved = false;
                    if (thread.ModerationType == CommentModerationType.All)
                        entry.IsApproved = false;
                    break;
                case CommentModerationType.None:
                    switch (thread.ModerationType)
                    {
                        case CommentModerationType.All:
                            entry.IsApproved = false;
                            break;
                        case CommentModerationType.Anonymous:
                            if (CurrentUser == null)
                                entry.IsApproved = false;
                            break;
                        case CommentModerationType.None:
                            entry.IsApproved = true;
                            break;
                        default:
                            break;
                    }
                    break;
                default:
                    break;
            }
            if (CurrentUser != null)
            {
                entry.Author = CurrentUser.DisplayName;
                entry.UserID = CurrentUser.UserID;
                entry.TitleUrl = SiteUrls.Instance().UserDomainHome(CurrentUser);
            }
            else
            {
                entry.Author = Request.Form.GetString("Author", GetResourceString("Common_AnonymousDisplayName"));
                entry.UserID = 0;
                entry.TitleUrl = Request.Form.GetString("CommenterUrl", string.Empty);

                // If it is empty, set it to nothing And Make sure the URL starts with http://
                if (!string.IsNullOrEmpty(entry.TitleUrl) && !entry.TitleUrl.StartsWith("http"))
                    entry.TitleUrl = "http://" + entry.TitleUrl;

                UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                if (Request.Form.Get<int>("RememberMe", -1) > 0)
                {
                    userCookie.SaveRememberMeInfor_name(entry.Author);
                    userCookie.SaveRememberMeInfor_url(entry.TitleUrl);
                }
                else
                {
                    userCookie.SaveRememberMeInfor_name(string.Empty);
                    userCookie.SaveRememberMeInfor_url(string.Empty);
                }
            }
            if (entry.ParentID > 0)
            {
                BlogComment parentComment = BlogComments.GetComment(entry.ParentID, null, true);
                if (parentComment != null)
                {
                    entry.ParentAuthor = parentComment.Author;
                    entry.ParentBody = parentComment.Body;
                    entry.ParentUserID = parentComment.UserID;
                    entry.ParentPostDate = parentComment.PostDate;
                    if (entry.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (entry.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length)
                        {
                            entry.Body = entry.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentComment.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_InputReplyContent", BlogApplicationID);
                            TempData["commentBody"] = entry.Body;
                            TempData["parentID"] = entry.ParentID;
                            return RedirectToAction("Control_BlogThreadComments", CurrentRouteValueDictionary);
                        }
                    }
                }
            }

            try
            {
                #region 检查发帖时间,防灌水
                Globals.CheckPostDate();
                #endregion

                entry = BlogComments.CreateComment(entry);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Blog());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_BlogThreadComments", CurrentRouteValueDictionary);
            }
            if (entry == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_CommentFailed", BlogApplicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                if (thread.BlogSection.ModerationType == CommentModerationType.All || thread.ModerationType == CommentModerationType.All)
                    TempData["StatusMessageContent"] = GetResourceString("Message_CommentSuccessNeedAuditing", BlogApplicationID);
                else if ((thread.BlogSection.ModerationType == CommentModerationType.Anonymous || thread.ModerationType == CommentModerationType.Anonymous) && Globals.GetCurrentUser() == null)
                    TempData["StatusMessageContent"] = GetResourceString("Message_CommentSuccessNeedAuditing", BlogApplicationID);
                else
                    TempData["StatusMessageContent"] = GetResourceString("Message_CommentSuccess", BlogApplicationID);
            }


            RouteValueDictionary currentRouteValueDictionary = CurrentRouteValueDictionary;

            if (currentRouteValueDictionary.ContainsKey("pageIndex"))
            {
                thread.CommentCount++;
                if (isReplay.HasValue && !isReplay.Value)
                    currentRouteValueDictionary["pageIndex"] = CommentsPageIndex(userDomainName, thread, pageIndex);
            }

            return RedirectToAction("Control_BlogThreadComments", currentRouteValueDictionary);
        }

        /// <summary>
        /// 删除某篇文章的评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult DeleteBlogThreadComment(string userDomainName, int postID, int threadID, int pageIndex)
        {
            BlogComment blogComment = BlogComments.GetComment(postID, null, true);
            if (blogComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogCommentNoFound", BlogApplicationID), postID)));
            }
            ValidateCommentDeleteAuthority(blogComment.OwnerUserID, blogComment.UserID);

            BlogComments.DeleteComment(postID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");
            return RedirectToAction("Control_BlogThreadComments", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 批量删除博客文章评论
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteBlogComments(string userDomainName)
        {
            BlogSection currentBlogSection = GetBlogSection(userDomainName);

            string selectedIDsString = Request.Form["BlogCommentCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {

                    List<int> selectedIDsInt = new List<int>();
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int ID = int.Parse(IDString);
                        if (ID > 0)
                        {
                            BlogComment blogComment = BlogComments.GetComment(ID, null, true);
                            if (blogComment != null)
                            {
                                ValidateCommentDeleteAuthority(blogComment.OwnerUserID, blogComment.UserID);
                                selectedIDsInt.Add(ID);
                            }
                        }

                    }
                    if (selectedIDsInt.Count > 0)
                    {
                        BlogComments.DeleteComments(selectedIDsInt);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");
                    }
                }
            }
            ApprovalStatuses? approvalStatus = null;
            if (Request.Form["approveDropDownList"] != null)
            {
                if (int.Parse(Request.Form["approveDropDownList"]) == 0)
                    approvalStatus = ApprovalStatuses.NotApprove;
                else if (int.Parse(Request.Form["approveDropDownList"]) == 1)
                    approvalStatus = ApprovalStatuses.IsApproved;
            }
            return Control_ManageBlogComments(userDomainName, null, approvalStatus);
        }

        /// <summary>
        /// 管理用户评论
        /// </summary>
        /// <param name="userDomainName">用户名称</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageBlogComments(string userDomainName, int? pageIndex)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageBlogComment", BlogApplicationID));
            #endregion
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ManageBlogComment", BlogApplicationID), string.Empty);

            return View("Pages/Blogs/ManageBlogComments.aspx", "Masters/Blog.master");
        }


        /// <summary>
        /// 管理用户博客
        /// </summary>
        /// <param name="userDomainName">用户名称</param>
        /// <param name="pageIndex">当前页码</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageBlogThreads(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_BlogManage", BlogApplicationID));
            #endregion
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_BlogManage", BlogApplicationID), string.Empty);


            return View("Pages/Blogs/ManageBlogThreads.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 按状态显示管理博客评论
        /// </summary>
        /// <param name="userDomainName">当前空间用户名</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult ManageBlogCommentsByApprovalStatus(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageBlogComment", BlogApplicationID));
            #endregion

            int approvalStatus = int.Parse(Request.Form["approveDropDownList"]);
            ApprovalStatuses getApprovalStatus = ApprovalStatuses.IsApproved;
            if (approvalStatus == 0)
                getApprovalStatus = ApprovalStatuses.NotApprove;
            else if (approvalStatus == 1)
                getApprovalStatus = ApprovalStatuses.IsApproved;
            else
                return Control_ManageBlogComments(userDomainName, null, null);

            return Control_ManageBlogComments(userDomainName, null, getApprovalStatus);
        }

        /// <summary>
        /// 设置博客评论状态
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        /// <param name="isApproved">是否批准评论</param>
        /// <param name="pageIndex">当前显示页码</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetBlogCommentsApproveStatus(string userDomainName, bool isApproved)
        {
            BlogSection currentBlogSection = GetBlogSection(userDomainName);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageBlogComment", BlogApplicationID));
            #endregion

            int getApprovalStatus = int.Parse(Request.Form["approveDropDownList"]);
            ApprovalStatuses? approvalStatus = null;
            if (getApprovalStatus == 0)
                approvalStatus = ApprovalStatuses.NotApprove;
            else if (getApprovalStatus == 1)
                approvalStatus = ApprovalStatuses.IsApproved;

            string selectedIDsString = Request.Form["BlogCommentCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    List<int> selectedIDsInt = new List<int>();
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int ID = int.Parse(IDString);
                        if (ID > 0)
                        {
                            BlogComment blogComment = BlogComments.GetComment(ID, null, true);
                            if (blogComment != null)
                            {
                                ValidateCommentDeleteAuthority(blogComment.OwnerUserID, blogComment.UserID);
                                selectedIDsInt.Add(ID);
                            }
                        }

                    }
                    BlogComments.ApprovePosts(currentBlogSection.OwnerUserID, selectedIDsInt, isApproved);

                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "修改成功！";

                }
            }

            return Control_ManageBlogComments(userDomainName, 0, approvalStatus);
        }

        /// <summary>
        /// 设置博客评论状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetBlogCommentApproveStatus(string userDomainName, int threadID, int postID, bool isApproved, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            BlogComment blogComment = BlogComments.GetComment(postID, null, true);
            if (blogComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogCommentNoFound", BlogApplicationID), postID)));
            }
            List<int> blogCommentIDs = new List<int>();
            blogCommentIDs.Add(postID);
            BlogComments.ApprovePosts(currentDomainUser.UserID, blogCommentIDs, isApproved);
            return RedirectToAction("Control_BlogThreadComments", CurrentRouteValueDictionary);
        }

        /// <summary>
        /// 博客评论批量处理（设置为批准或未批准）
        /// </summary>
        /// <param name="userDomainName">当前用户空间用户名称</param>
        /// <param name="pageIndex">当前页码</param>
        /// <param name="approvalStatus">显示状态状态</param>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ManageBlogComments(string userDomainName, int? pageIndex, ApprovalStatuses? approvalStatus)
        {
            BlogSection blogSection = GetBlogSection(userDomainName);
            //获取查询返回的博客评论集合，该集合是封装成用于分页的数据集合
            PagingDataSet<BlogComment> pds = BlogComments.GetMyBlogComments(blogSection.OwnerUserID, approvalStatus, pageIndex ?? 1);
            ViewData["ApprovalStatus"] = approvalStatus;
            return PartialView("Controls/Blogs/ManageBlogComments.ascx", pds);
        }

        #endregion

        #endregion

        #region BlogSection

        /// <summary>
        /// 管理用户博客设置
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageBlogSettings(string userDomainName)
        {
            BlogSection blogSection = GetBlogSection(userDomainName);

            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageBlogSet", BlogApplicationID));
            #endregion

            ViewData["currentBlogSection"] = blogSection;
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ManageBlogSet", BlogApplicationID), string.Empty);

            return View("Pages/Blogs/ManageBlogSettings.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 更新博客设置
        /// </summary>
        /// <param name="userDomainName">博客所属用户名</param>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateBlogSettings(string userDomainName)
        {
            BlogSection BlogSection = GetBlogSection(userDomainName);

            //if (weblog.ApplicationKey != AppKey.Text.ToLower())
            //{
            //    string formattedKey = null;
            //    Globals.ValidateApplicationKey(AppKey.Text, out formattedKey);

            //    string oldPath = SiteUrls.Instance().Locations["BlogSections"] + BlogSection.ApplicationKey;
            //    string newPath = SiteUrls.Instance().Locations["BlogSections"] + formattedKey;

            //    BlogSection.ApplicationKey = formattedKey;
            //}

            BlogSection.EnableAnonymousPosting = Request.Form.GetBool("EnableAnonymousPosting", true);
            BlogSection.SectionName = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.Get<string>("BlogName", GetResourceString("Title_Blog", BlogApplicationID))).Trim();
            if (string.IsNullOrEmpty(BlogSection.SectionName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            }
            else
            {
                BlogSection.EnableComments = Request.Form.GetBool("EnableCommentes", true);
                BlogSection.EnableTrackback = Request.Form.GetBool("EnableTrackBack", true);
                BlogSection.EnableRating = Request.Form.GetBool("EnableRating", true);

                //设置文章评论审核类型
                BlogSection.ModerationType = (CommentModerationType)Enum.Parse(typeof(CommentModerationType), Request.Form["ModerationDDL"], false);
                //BlogSection.DefaultPrivacyStatus = (PrivacyStatuses)Enum.Parse(typeof(PrivacyStatuses), Request.Form["setBlogThreadsPirvacyStatus"], false);
                if (BlogSection.DefaultPrivacyStatus == PrivacyStatuses.NeedPassword)
                    BlogSection.DefaultPrivacyPassword = Request.Form.GetString("setBlogThreadsPirvacyStatus_Password", string.Empty);

                //更新博客
                if (BlogSections.Update(BlogSection) > 0)
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                else
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            }
            return RedirectToAction("ManageBlogSettings", "Blog", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #region BlogThreadCategory

        #region Show

        /// <summary>
        /// 用分类列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult ListBlogUserCategories(string userDomainName, int? pageIndex)
        {
            #region 计数 标题

            SetPageTitle(userDomainName, GetResourceString("Title_CategoryList", BlogApplicationID));

            #endregion

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool isManager = HasManagePermission(userDomainName);
            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).GetUserCategories(currentDomainUser.UserID, isManager);

            ViewData["isManager"] = isManager;
            ViewData["userDomainName"] = userDomainName;
            return PartialView("Controls/Blogs/ListBlogUserCategories.ascx", userCategories);
        }

        /// <summary>
        /// 博客的用户分类下拉列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_BlogThreadCategoryDropdownList(string userDomainName, int? currentCategoryID, string controlName, string cssClass)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            List<UserCategory> userCategories = UserCategoryManager.Instance(UserCategoryItemKeys.Instance().BlogThread()).GetUserCategories(currentDomainUser.UserID, true);
            if (currentCategoryID != null && currentCategoryID > 0)
            {
                ViewData["currentCategoryID"] = (int)currentCategoryID;
            }
            else
                ViewData["currentCategoryID"] = 0;
            ViewData["controlName"] = controlName;
            ViewData["cssClass"] = cssClass;
            return PartialView("Controls/Blogs/BlogThreadCategoryDropdownList.ascx", userCategories);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 管理博客的用户分类
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageBlogUserCategories(string userDomainName, int? pageIndex)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageBlogUserCategory", BlogApplicationID));
            #endregion
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ManageBlogUserCategory", BlogApplicationID), string.Empty);

            return View("Pages/Blogs/ManageBlogUserCategories.aspx", "Masters/Blog.master");
        }

        #endregion

        #endregion

        #region BlogThreadTag

        /// <summary>
        /// 管理用户博客标签
        /// </summary>
        /// <param name="userDomainName">当前空间用户用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageBlogUserTags(string userDomainName)
        {
            #region 计数 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageBlogTag", BlogApplicationID));
            #endregion
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ManageBlogTag", BlogApplicationID), string.Empty);

            return View("Pages/Blogs/ManageBlogUserTags.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 博客用户标签
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListBlogUserTags(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTags(currentDomainUser.UserID, HasManagePermission(currentDomainUser));
            bool isManger = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManger = true;
            ViewData["isManager"] = isManger;
            return PartialView("Controls/Blogs/ListBlogUserTags.ascx", userTags);
        }

        #endregion

        #region FootPrint

        /// <summary>
        /// 显示足迹列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult ListFootPrints(string userDomainName, int threadID, int showCount)
        {
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, false);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }
            List<FootPrint> footPrints = new List<FootPrint>();
            footPrints = FootPrints.GetFootPrints(threadID, false);
            ViewData["threadID"] = blogThread.ThreadID;
            ViewData["showCount"] = showCount;
            ViewData["userDomainName"] = userDomainName;
            return PartialView("Controls/Blogs/ListFootPrints.ascx", footPrints);
        }

        /// <summary>
        /// 创建足迹
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult CreateFootPrint(string userDomainName, int threadID, int showCount)
        {
            if (CurrentUser == null)
                return new EmptyResult();
            BlogThread blogThread = BlogThreads.GetThread(threadID, null, false);
            if (blogThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogArticleNoFind", BlogApplicationID), threadID)));
            }
            int userID = Globals.GetCurrentUser().UserID;
            string author = Globals.GetCurrentUser().DisplayName;
            bool isExistsFoot = FootPrints.IsExistsFootPrint(threadID, userID);
            if (isExistsFoot)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_AlreadySaveFootPrint", BlogApplicationID);
            }
            else
            {
                FootPrint footPrint = new FootPrint();
                footPrint.Author = author;
                footPrint.UserID = userID;
                footPrint.ThreadID = threadID;
                FootPrints.CreateFootPrint(footPrint);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_SuccessSaveFootPrint", BlogApplicationID);
            }

            return RedirectToAction("ListFootPrints", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", threadID }, { "showCount", showCount } });
        }

        /// <summary>
        /// 删除脚印
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFootPrint(string userDomainName, int footPrintID, int threadID, int showCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool isExistsFootPrint = FootPrints.IsExistsFootPrint(threadID, currentDomainUser.UserID);
            if (isExistsFootPrint)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_FootPrintNoFound", BlogApplicationID), footPrintID)));
            }
            //TODO:验证没有加
            FootPrints.DeleteFootPrint(threadID, footPrintID);
            return ListFootPrints(userDomainName, threadID, showCount);
        }


        #endregion

        #region 公用控件

        /// <summary>
        /// 博客存档
        /// </summary>
        /// <param name="userDomainName">博客所属用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ArchiveList(string userDomainName)
        {
            BlogSection blogSection = GetBlogSection(userDomainName);
            List<ArchiveDataItem> list = BlogThreads.GetStatByMonths(blogSection.OwnerUserID);

            return PartialView("Controls/Blogs/ArchiveList.ascx", list);
        }

        #endregion

        #region 辅助方法


        private int topSpecialOrder = 1;

        /// <summary>
        /// 设置文章YesOrNo属性，是否公开，是否启用评价，是否允许回复等
        /// </summary>
        private void PopulateFromYesNo(BlogThread blogThread)
        {
            //是否公开
            //blogThread.IsPublic = Request.Form.GetBool("ynPublic", true);

            //是否允许回复

            blogThread.IsLocked = (!Request.Form.GetBool("ynRepaly", true));

            //是否启用评价
            blogThread.EnableRatings = Request.Form.GetBool("ynEnableRatings", true);


            ////Rss中显示的摘要
            //if (Request.Form["ynSyndicateExcerpt"] == "true" || Request.Form["ynSyndicateExcerpt"] == "True")
            //    blogThread.SyndicateExcerpt = true;
            //else
            //    blogThread.SyndicateExcerpt = false;
        }

        /// <summary>
        /// 获取博客实体
        /// </summary>
        /// <param name="userDomainName">博客所属用户用户名</param>
        private BlogSection GetBlogSection(string userDomainName)
        {
            User user = GetCurrentDomainUser(userDomainName);
            BlogSection blogSection = BlogSections.GetSection(user.UserID, false);
            if (blogSection == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_BlogNotFound", BlogApplicationID), userDomainName)));
            }
            //throw new ArgumentException(String.Format("{0}博客不存在", userDomainName));
            return blogSection;
        }

        /// <summary>
        /// 设置页面标题
        /// </summary>
        protected override void SetPageTitle(string userDomainName, string text)
        {
            base.SetPageTitle(userDomainName, text);
            BlogSection currentBlogSection = GetBlogSection(userDomainName);
            if (currentBlogSection == null)
                return;
            if (string.IsNullOrEmpty(text))
                Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, currentBlogSection.SectionName, GetCurrentDomainUser(userDomainName).DisplayName, Globals.SiteName));
            else
                Header.AddTitle(string.Format("{1}{0}{2}{0}{3}{0}{4}", Globals.BrowserTitleSeparator, text, currentBlogSection.SectionName, GetCurrentDomainUser(userDomainName).DisplayName, Globals.SiteName));
        }

        private int CommentsPageIndex(string userDoMainName, BlogThread thread, int pageIndex)
        {

            RouteValueDictionary currentRouteValueDictionary = CurrentRouteValueDictionary;

            if (thread.CommentCount > 0)
            {

                int commentCount = BlogComments.GetCommentcount(thread.ThreadID, null, HasManagePermission(userDoMainName));
                if (commentCount > BlogSectionConfiguration.Instance().CommentsPerPage)
                {
                    pageIndex = commentCount / BlogSectionConfiguration.Instance().CommentsPerPage;

                    if ((commentCount % BlogSectionConfiguration.Instance().CommentsPerPage) > 0)
                    {
                        pageIndex++;
                    }
                }
            }

            return pageIndex;
        }


        ///// <summary>
        ///// 获取连接字符串中的日期
        ///// </summary>
        //private DateTime GetDate(int? year, int? month, int? day)
        //{
        //    int getYear = year != null ? (int)year : 2000;
        //    int getMonth = month != null ? (int)month : 1;
        //    int getDay = day != null ? (int)day : 1;
        //    return new DateTime(getYear, getMonth, getDay);

        //}

        #endregion

        #region 更新博客审核状态

        public ActionResult updateBlogAuditingStatuses(int threadID, string userDomainName)
        {
            List<int> threadIds = new List<int>();
            threadIds.Add(threadID);
            BlogThreads.UpdateAuditingStatus(threadIds, AuditingStatuses.Success);
            return RedirectToAction("ShowBlogThread", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", threadID } });
        }

        #endregion

        /// <summary>
        /// 验证隐私密码
        /// </summary>
        [HttpGet]
        public ActionResult Control_PrivacyPassword(string userDomainName, int threadID)
        {

            return PartialView("Controls/Blogs/PrivacyPassword.ascx");
        }
        /// <summary>
        /// 验证隐私密码
        /// </summary>
        [HttpPost]
        public ActionResult AuthorizePrivacyPassword(string userDomainName, int threadID)
        {
            BlogThread thread = BlogThreads.GetThread(threadID, null, true);
            if (thread == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, string.Format("找不到CategoryID为{0}的文章", threadID));
                return RedirectToAction("Control_PrivacyPassword", CurrentRouteValueDictionary);
            }

            string password = Request.Form.GetString("password", string.Empty).Trim();
            bool isSuccess = BlogThreads.ValidPrivacyPassword(password, thread.Password);
            if (isSuccess)
            {
                UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);

                userCookie.SaveAuthorizedBlogThreadIDs(threadID, password);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "通过验证");
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "密码错误");
            }
            return RedirectToAction("Control_PrivacyPassword", CurrentRouteValueDictionary);
        }


        /// <summary>
        /// 博客附件管理
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ManageBlogAttachments()
        {
            if (CurrentUser == null)
                return new EmptyResult();

            List<Attachment> attachments = BlogAttachmentManager.Instance().GetAttachmentsByOwnerID(CurrentUser.UserID);
            return PartialView("Controls/Blogs/ManageBlogAttachments.ascx", attachments);
        }
    }

    /// <summary>
    /// 空间博客二级菜单
    /// </summary>
    public enum BlogUserDomainSubMenus
    {
        /// <summary>
        /// 我的博客
        /// </summary>
        myBlog = 1,

        /// <summary>
        /// 朋友的博客
        /// </summary>
        friendBlog = 10,

        /// <summary>
        /// 站点中的博客
        /// </summary>
        siteBlog = 20
    }
}
