﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;

using SpaceBuilder.Blog;
using SpaceBuilder.Common;

using SpaceBuilder.Common.Controllers;
using SpaceBuilder.LuceneSearch;
using System.Web.Routing;
using SpaceBuilder.Web;
using SpaceBuilder.Utils;

namespace SpaceBuilder.Blog.Controllers
{
    [HttpCompress]
    public class ChannelBlogController : ChannelControllerBase
    {
        private int BlogApplicationID = ApplicationIDs.Instance().Blog();
        /// <summary>
        /// 总站中博客首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_Blog", BlogApplicationID));
            List<SiteCategory> siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).GetRoots(false);

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Blog", BlogApplicationID), string.Empty);

            return this.View("Pages/Blogs/Home.aspx", "Masters/Blog.master", siteCategories);
        }

        #region BlogThread

        /// <summary>
        /// 当前标签下的文章列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListBlogThreadsByTag(string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");

            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, string.Format(GetResourceString("Title_Pattern_TagArticle", BlogApplicationID), tagName), GetResourceString("Title_Blog", BlogApplicationID), Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem(string.Format(GetResourceString("Title_Pattern_TagArticle", BlogApplicationID), tagName), string.Empty);
            ViewData["tagName"] = tagName;

            return View("Pages/Blogs/ListBlogThreadsByTag.aspx", "Masters/Blog.master");
        }

        /// <summary>
        /// 博客排行
        /// </summary>
        /// <param name="sortBy">排序条件</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListBlogRanks(BlogSectionSortBy? sortBy, int? pageIndex)
        {
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_BlogTop", BlogApplicationID), GetResourceString("Title_Blog", BlogApplicationID), Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_BlogTop", BlogApplicationID), string.Empty);

            //获取博客列表
            PagingDataSet<BlogSection> sections = BlogSections.GetPagingSectionsSortBy(sortBy ?? BlogSectionSortBy.Points, 20, pageIndex ?? 1);
            ViewData["sortBy"] = sortBy;
            return View("Pages/Blogs/BlogRanks.aspx", "Masters/Blog.master", sections);
        }

        /// <summary>
        /// 文章排行
        /// </summary>
        /// <param name="sortBy">排序条件</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult BlogThreadRanks(BlogThreadSortBy? rankBy, int categoryID, string tagName, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_ArticleTop", BlogApplicationID), GetResourceString("Title_Blog", BlogApplicationID), Globals.SiteName));
            if (categoryID > 0)
            {
                ShowCategoryBreadCrumb(categoryID);
            }
            else if (!string.IsNullOrEmpty(tagName))
            {
                BreadCrumb.AddBreadCrumbItem(tagName, string.Empty);
            }
            else
            {
                BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ArticleTop", BlogApplicationID), string.Empty);
            }
            ViewData["sortBy"] = rankBy ?? BlogThreadSortBy.StickyAndMostRecent;

            return View("Pages/Blogs/BlogThreadRanks.aspx", "Masters/Blog.master");
        }


        /// <summary>
        /// 根据博客类别聚合博客
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListBlogThreadsByCategory(int categoryID)
        {
            SiteCategory siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get(categoryID, false);
            if (siteCategory == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_CategoryNoFind", BlogApplicationID), categoryID)));
            }

            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, siteCategory.GetMultilingualCategoryName(CurrentUserLanguage), GetResourceString("Title_Blog", BlogApplicationID), Globals.SiteName));
            #region 处理类别面包屑
            BreadCrumb.AddBreadCrumbItem(siteCategory.GetMultilingualCategoryName(CurrentUserLanguage), SPBUrlHelper.Action("ListBlogThreadsByCategory", "ChannelBlog", new RouteValueDictionary { { "categoryID", siteCategory.CategoryID } }));

            //ShowCategoryBreadCrumb(categoryID);
            #endregion

            List<SiteCategory> childSiteCategories = new List<SiteCategory>();
            childSiteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).GetAllChilds(categoryID, false);

            BlogThread topBlogThreadInCategory = null;

            IList<BlogThread> blogThreads = BlogThreads.GetTopThreadsSortBy(BlogThreadSortBy.StickyAndMostRecent, categoryID, 1);
            if (blogThreads != null && blogThreads.Count > 0)
                topBlogThreadInCategory = blogThreads[0];

            ViewData["currentSiteCategory"] = siteCategory;
            ViewData["childSiteCategories"] = childSiteCategories;
            ViewData["topBlogThreadInCategory"] = topBlogThreadInCategory;

            return View("Pages/Blogs/ListBlogThreadsByCategory.aspx", "Masters/Blog.master");
        }
        /// <summary>
        /// 生成博客类别面包屑
        /// </summary>
        private void ShowCategoryBreadCrumb(int siteCategoryID)
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Blog", BlogApplicationID), SiteUrls.Instance().ChannelBlogHome());
            Stack<SiteCategory> stack = new Stack<SiteCategory>();
            SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get(siteCategoryID, false);
            while (currentCategory != null)
            {
                stack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get(currentCategory.ParentID, false);
            }
            while (stack.Count > 0)
            {
                currentCategory = stack.Pop();
                BreadCrumb.AddBreadCrumbItem(currentCategory.GetMultilingualCategoryName(CurrentUserLanguage), SPBUrlHelper.Action("ListBlogThreadsByCategory", "ChannelBlog", new RouteValueDictionary { { "categoryID", currentCategory.CategoryID } }));
            }
        }

        /// <summary>
        /// Tab页中显示文章列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogThreadsInTab(BlogThreadSortBy? sortBy, int? siteCategoryID, int pageSize)
        {
            ViewData["blogThreadSortBy"] = sortBy;
            ViewData["categoryID"] = siteCategoryID.HasValue ? (int)siteCategoryID : 0;
            IList<BlogThread> blogThreads = BlogThreads.GetTopThreadsSortBy(sortBy ?? BlogThreadSortBy.MostRecent, siteCategoryID, pageSize);
            return PartialView("Controls/Blogs/ListBlogThreadsInTab.ascx", blogThreads);
        }


        /// <summary>
        /// 文章列表模块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogThreadsBlock(BlogThreadSortBy? sortBy, int? siteCategoryID, int pageSize)
        {
            ViewData["blogThreadSortBy"] = sortBy;
            ViewData["categoryID"] = siteCategoryID.HasValue ? (int)siteCategoryID : 0;
            IList<BlogThread> blogThreads = BlogThreads.GetTopThreadsSortBy(sortBy ?? BlogThreadSortBy.MostRecent, siteCategoryID, pageSize);
            return PartialView("Controls/Blogs/ListBlogThreadsBlock.ascx", blogThreads);
        }

        /// <summary>
        /// 类别博客列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogBlock(BlogThreadSortBy? sortBy, string viewName, string titleName, int? categoryID, int topNumber)
        {
            SiteCategory siteCategory = null;
            if (categoryID.HasValue)
                siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get(categoryID.Value, false);
            if (siteCategory == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_BlogCategoryNoFind", BlogApplicationID), categoryID.Value)));
            }

            ViewData["siteCategory"] = siteCategory;
            ViewData["titleName"] = string.IsNullOrEmpty(titleName) ? siteCategory.GetMultilingualCategoryName(CurrentUserLanguage) : titleName;
            IList<BlogThread> blogThreads = BlogThreads.GetTopThreadsSortBy(sortBy ?? BlogThreadSortBy.MostRecent, categoryID, topNumber);
            return PartialView("Controls/Blogs/" + viewName + ".ascx", blogThreads);
        }


        /// <summary>
        /// 博客文章摘要显示
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_SummaryBlogThreads(BlogThreadSortBy? sortBy, int pageIndex, int? categoryID, string tagName)
        {
            tagName = WebUtils.UrlDecode(tagName);
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            if (categoryID != null && (int)categoryID > 0)
            {
                SiteCategory siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get((int)categoryID, false);
                if (siteCategory == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_CategoryNoFind", BlogApplicationID), categoryID)));
                }
            }
            PagingDataSet<BlogThread> blogThreads = null;
            if (!string.IsNullOrEmpty(tagName))
            {
                blogThreads = BlogThreads.GetPagingThreadsInSiteTag(tagName, BlogSectionConfiguration.Instance().ThreadsPerPage, pageIndex);
            }
            else
            {
                blogThreads = BlogThreads.GetPagingThreadsSortBy(sortBy ?? BlogThreadSortBy.MostRecent, categoryID, BlogSectionConfiguration.Instance().ThreadsPerPage, pageIndex);
            }
            ViewData["currentSortBy"] = sortBy;
            ViewData["categoryID"] = categoryID;
            ViewData["tagName"] = tagName;


            return PartialView("Controls/Blogs/SummaryBlogThreads.ascx", blogThreads);
        }

        /// <summary>
        /// 博客文章列表显示
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogThreads(BlogThreadSortBy? sortBy, int pageIndex, int? categoryID, string tagName)
        {
            tagName = WebUtils.UrlDecode(tagName);
            if (!string.IsNullOrEmpty(tagName))
                tagName = tagName.Replace('<', ' ').Replace('>', ' ').Replace("%3C", "&lt;").Replace("%3E", "&gt;");
            if (categoryID != null && (int)categoryID > 0)
            {
                SiteCategory siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get((int)categoryID, false);
                if (siteCategory == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_CategoryNoFind", BlogApplicationID), categoryID)));
                }
            }
            PagingDataSet<BlogThread> blogThreads = null;
            if (!string.IsNullOrEmpty(tagName))
            {
                blogThreads = BlogThreads.GetPagingThreadsInSiteTag(tagName, BlogSectionConfiguration.Instance().ThreadsPerPage, pageIndex);
            }
            else
                blogThreads = BlogThreads.GetPagingThreadsSortBy(sortBy ?? BlogThreadSortBy.MostRecent, categoryID, BlogSectionConfiguration.Instance().ThreadsPerPage, pageIndex);
            ViewData["currentSortBy"] = sortBy;
            ViewData["categoryID"] = categoryID;
            ViewData["tagName"] = tagName;


            return PartialView("Controls/Blogs/ListBlogThreads.ascx", blogThreads);
        }

        /// <summary>
        /// 显示顶置文章列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListStickyBlogThreads(int? categoryID)
        {
            if (categoryID != null && (int)categoryID > 0)
            {
                SiteCategory siteCategory = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get((int)categoryID, false);
                if (siteCategory == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Message_Pattern_CategoryNoFind", BlogApplicationID), categoryID)));
                }
                //throw new ArgumentException(string.Format("{0}站点类别不存在", categoryID));
            }

            IList<BlogThread> blogThreads = BlogThreads.GetTopThreadsSortBy(BlogThreadSortBy.StickyAndMostRecent, categoryID, 7);

            IList<RecommendedItem> recommendItems = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Blog(), 111001, 9);
            IList<BlogThread> recommendBlogThreads = new List<BlogThread>();
            foreach (RecommendedItem item in recommendItems)
            {
                if (item != null)
                {
                    BlogThread nowThread = BlogThreads.GetThread(item.ItemID, PrivacyStatuses.Public, false);
                    if (nowThread != null)
                        recommendBlogThreads.Add(nowThread);
                }
            }

            ViewData["commendBlogThreads"] = recommendBlogThreads;

            return PartialView("Controls/Blogs/ListStickyBlogThreads.ascx", blogThreads);
        }

        /// <summary>
        /// 首页热门文章
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListStickyBlogThreadsBlock(int displayCount)
        {
            IList<BlogThread> blogThreads = BlogThreads.GetTopThreadsSortBy(BlogThreadSortBy.StickyAndMostRecent, null, displayCount);
            bool isShow = false;
            if (Applications.IsEnabled(ApplicationIDs.Instance().Blog()))
                isShow = true;
            ViewData["isShow"] = isShow;

            return PartialView("Controls/Blogs/ListStickyBlogThreadsBlock.ascx", blogThreads);
        }

        #endregion

        #region BlogSection

        /// <summary>
        /// 最新博客列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogsBlock(int pageSize, BlogSectionSortBy? sortBy)
        {
            IList<BlogSection> sections = BlogSections.GetTopSectionsSortBy(sortBy ?? BlogSectionSortBy.CreateDate, pageSize);
            ViewData["sortBy"] = sortBy;
            return PartialView("Controls/Blogs/ListBlogsBlock.ascx", sections);
        }

        /// <summary>
        /// 博客排行
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult BlogRanks(BlogSectionSortBy? sortBlogsBy, int? pageIndex)
        {
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_BlogTop", BlogApplicationID), GetResourceString("Title_Blog", BlogApplicationID), Globals.SiteName));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Blog", BlogApplicationID), SiteUrls.Instance().ChannelBlogHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_BlogTop", BlogApplicationID), string.Empty);

            ViewData["sortBy"] = sortBlogsBy;
            PagingDataSet<BlogSection> blogSections = BlogSections.GetPagingSectionsSortBy(sortBlogsBy ?? BlogSectionSortBy.CreateDate, 20, pageIndex ?? 1);
            return View("Pages/Blogs/BlogRanks.aspx", "Masters/Blog.master", blogSections);
        }

        /// <summary>
        /// Tab页中博客列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogsInTab(BlogSectionSortBy? sortBy, int pageSize)
        {
            ViewData["sortBy"] = sortBy;
            IList<BlogSection> blogSections = BlogSections.GetTopSectionsSortBy(sortBy ?? BlogSectionSortBy.CreateDate, pageSize);
            return PartialView("Controls/Blogs/ListBlogsInTab.ascx", blogSections);
        }

        #endregion

        #region SiteCategory

        /// <summary>
        /// 获取所有的根类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        //[ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_SubMenus()
        {
            List<SiteCategory> siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).GetRoots(false);
            StringBuilder sb = new StringBuilder();
            foreach (SiteCategory rootCategory in siteCategories)
            {
                sb.AppendFormat("<li class=\"spb-nav-item\"><a {0} href=\"{1}\"><span>{2}</span></a>", rootCategory.ChildCount > 0 ? "class=\"sf-with-ul\"" : "", Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListBlogThreadsByCategory", "ChannelBlog", new RouteValueDictionary { { "categoryID", rootCategory.CategoryID } })), rootCategory.GetMultilingualCategoryName(CurrentUserLanguage));
                if (rootCategory.ChildCount > 0)
                    sb.Append(GetChildCategoriesTag(rootCategory.CategoryID));
                sb.Append("</li>");
            }
            return PartialView("Controls/Blogs/SubMenus.ascx", sb.ToString());
        }

        /// <summary>
        /// 递归返回子类的标签
        /// </summary>
        /// <param name="categoryID"></param>
        /// <returns></returns>
        private string GetChildCategoriesTag(int categoryID)
        {
            SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).Get(categoryID, false);
            StringBuilder sb = new StringBuilder();
            sb.Append("<ul class=\"spb-nav-submenu\" style=\"disply:none;visibility:hidden\">");
            if (category.ChildCount > 0)
            {
                foreach (SiteCategory childCategory in category.Childs)
                {
                    sb.AppendFormat("<li class=\"spb-nav-item\"><a {0} href=\"{1}\"><span>{2}</span></a>", childCategory.ChildCount > 0 ? "class=\"sf-with-ul\"" : "", Globals.GetFullUrlInMainSite(SPBUrlHelper.Action("ListBlogThreadsByCategory", "ChannelBlog", new RouteValueDictionary { { "categoryID", childCategory.CategoryID } })), childCategory.GetMultilingualCategoryName(CurrentUserLanguage));
                    if (childCategory.ChildCount > 0)
                        sb.Append(GetChildCategoriesTag(childCategory.CategoryID));
                    sb.Append("</li>");
                }
            }
            sb.Append("</ul>");

            return sb.ToString();
        }



        #endregion

        #region SiteTag

        /// <summary>
        /// 标签地图
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult BlogTagMap()
        {
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_TagMap"), GetResourceString("Title_Blog", BlogApplicationID), Globals.SiteName));
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Blog", BlogApplicationID), SiteUrls.Instance().ChannelBlogHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TagMap", BlogApplicationID), string.Empty);

            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetTopSiteTagsSortBy(SiteTagsSortBy.ItemCount, 300);

            return View("Pages/Blogs/BlogTagMap.aspx", "Masters/Blog.master", siteTags);
        }

        /// <summary>
        /// 频道标签列表显示
        /// </summary>
        /// <param name="pageSize">显示标签数量</param>
        /// <param name="tagType">标签类型</param>
        /// <param name="sortBy">按文章数或用户数排序</param>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListBlogSiteTags(int pageSize, SiteTagsSortBy? sortBy)
        {

            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetTopSiteTagsSortBy(sortBy ?? SiteTagsSortBy.ItemCount, pageSize);
            return PartialView("Controls/Blogs/ListBlogSiteTags.ascx", siteTags);
        }

        #endregion

        #region Rating

        /// <summary>
        /// 对博客主题进行星级评价
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Rate()
        {
            //博客ThreadID
            int itemID = Request.QueryString.GetInt("itemID", -1);

            //评价的星级
            int value = Request.QueryString.GetInt("value", -1);


            ContentResult result = new ContentResult();
            result.ContentType = "text/html";
            if (!this.HttpContext.User.Identity.IsAuthenticated)
            {
                result.Content = GetResourceString("Message_PleaceLogin");
            }
            else
            {
                if (CurrentUser != null && itemID > 0 && value > 0)
                {
                    Rating rating = new Rating();
                    rating.UserID = CurrentUser.UserID;
                    rating.Author = CurrentUser.DisplayName;
                    rating.ItemID = itemID;
                    rating.Rate = value;
                    BlogThreads.Rate(rating);
                    result.Content = GetResourceString("Message_ThanksForRating");
                }
                else
                {
                    result.Content = GetResourceString("Message_ParameterIncorrect");
                }
            }
            return result;
        }
        /// <summary>
        /// 显示评价人列表
        /// </summary>
        public ActionResult Control_ListRatingUsers(int itemID, int? topNumber)
        {
            IList<Rating> ratings = BlogThreads.GetRatings(itemID, topNumber ?? 30);
            return PartialView("Controls/Common/ListRatingUsers.ascx", ratings);
        }

        #endregion


        #region BlogAttachment
        /// <summary>
        /// 管理博客主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_UploadBlogAttachment(int? threadID, bool? isImage, string queueCompleteCallback)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            UploadAttachmentSettings settings = new UploadAttachmentSettings();
            BlogAttachmentSettings attachmentSettings = (BlogAttachmentSettings)BlogAttachmentManager.Instance().Settings;
            string uploadPhotoTypes = BlogAttachmentManager.Instance().Settings.Extensions;
            IList<string> imageExtensions = MimeTypeConfiguration.GetImageExtensions();
            StringBuilder newUploadPhotoTypes = new StringBuilder();
            string[] types = uploadPhotoTypes.Split(',');
            foreach (string type in types)
            {
                if (!string.IsNullOrEmpty(type))
                {
                    if (isImage.HasValue && isImage.Value == imageExtensions.Contains(type.ToLower()))
                        newUploadPhotoTypes.AppendFormat("*.{0};", type);
                }
            }
            settings.AssociateID = threadID ?? 0;
            settings.FileTypes = newUploadPhotoTypes.ToString();
            settings.UploadUrl = Globals.GetFullUrlInFileHandlerSite(BlogAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Services/Blog/UploadAttachments.ashx"));
            settings.FlashUrl = Globals.GetFullUrlInFileHandlerSite(BlogAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Utility/UploadFiles/swfupload.swf"));
            settings.FileQueryLimit = attachmentSettings.MaxFileCountATime;
            settings.FileSizeLimit = attachmentSettings.MaxAttachmentSize.ToString();
            settings.QueueCompleteCallback = queueCompleteCallback;
            settings.IsImage = isImage ?? true;

            ViewData["UploadAttachmentSettings"] = settings;
            return View("Controls/UploadAttachmentInEditer.ascx", settings);
        }

        /// <summary>
        /// 管理博客主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ManageThreadAttachments(int? threadID, bool? isImage)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            ManageThreadAttachmentsModel model = new ManageThreadAttachmentsModel();
            model.IsImage = isImage ?? true;

            if (threadID.HasValue && threadID.Value > 0)
            {
                BlogThread blogThread = BlogThreads.GetThread(threadID.Value, null, true);
                if (blogThread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_Pattern_VisitBlogArticleNoFound", BlogApplicationID)));
                }

                IList<Attachment> attachments = BlogAttachmentManager.Instance().GetAttachmentsByAssociateID(threadID.Value, true);
                model.Attachments = attachments;
                model.UserFeaturedImageAttachmentID = blogThread.UserFeaturedImageAttachmentID;
            }
            else
            {
                IList<Attachment> attachments = BlogAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(CurrentUser.UserID);
                model.Attachments = attachments;
            }
            model.GetAttachmentUrl = SiteUrls.Instance().DownloadBlogAttachment;
            model.DeleteAttachmentUrl = SiteUrls.Instance().DeleteBlogAttachmentUrl;
            model.ShowAttachmentThumbnail = SPBHtmlHelper.Instance().ShowBlogAttachmentThumbnail;
            return PartialView("Controls/ManageThreadAttachmentsInEditer.ascx", model);
        }

        public ActionResult UserContentSelector_UpdateBlogAttachments(int? threadID, bool isImage)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            return PartialView("Controls/Blogs/UserContentSelector_UpdateBlogAttachments.ascx");
        }

        #region Manage

        /// <summary>
        /// 删除博客附件
        /// </summary>
        public ActionResult DeleteBlogAttachment(int attachmentID, bool? isImage)
        {

            Attachment attachment = BlogAttachmentManager.Instance().GetAttachment(attachmentID, true);
            if (attachment == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (CurrentUser == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请先登录");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (attachment.AssociateID > 0)
            {
                BlogThread blogThread = BlogThreads.GetThread(attachment.AssociateID, null, true);
                if (blogThread == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件所属的文章");
                    return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
                }
            }

            User attachmentOwner = Users.GetUser(attachment.UserID, false);
            if (attachmentOwner == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "找不到附件的上传者");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            if (CurrentUser.UserID != attachment.UserID && !CurrentUser.IsContentAdministrator)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "没有权限，只有上传者和网站管理员才能删除附件");
                return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
            }

            BlogAttachmentManager.Instance().DeleteAttachment(attachmentID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "删除成功");
            return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", attachment.AssociateID }, { "isImage", isImage } });
        }

        #endregion


        #endregion


        #region 用户内容选择器

        /// <summary>
        /// 在用户内容选择器中显示用户的照片
        /// </summary>
        /// <param name="categoryID">类别ID</param>
        public ActionResult UserContentSelector_ListBlogAttachments()
        {
            List<Attachment> attachments = null;

            User currentUser = Globals.GetCurrentUser();
            if (currentUser != null)
            {
                attachments = BlogAttachmentManager.Instance().GetAttachmentsByOwnerID(currentUser.UserID);
            }
            else
            {
                attachments = new List<Attachment>();
            }

            return PartialView("Controls/UserContentSelector_ListBlogAttachments.ascx", attachments);
        }

        #endregion



        #region FullTextSearch

        /// <summary>
        /// 博客全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_BlogSearch", BlogApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_BlogSearch", BlogApplicationID), string.Empty);
            BlogThreadFullTextQuery query = new BlogThreadFullTextQuery();
            query.ConvertFromQueryString(Request);
            query.PageSize = 10;

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<BlogThread> results = null;
            if (query.IsValid())
            {
                results = BlogSearchManager.Instance().Search(query);
            }

            ViewData["Query"] = query;
            return this.View("Pages/Blogs/Search.aspx", "Masters/Blog.master", results);
        }

        /// 博客全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["Keyword"] = Request.Form.GetString("Keyword", string.Empty);

            int searchScopeUserID = Request.Form.GetInt("SearchScopeUserID", -1);
            if (searchScopeUserID > 0)
                parameters["SearchScopeUserID"] = searchScopeUserID.ToString();

            return RedirectToAction("Search", parameters);
        }

        #endregion


        #region 推荐博文
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_RecommendBlogThreadBlock(int? commendTypeID, int? displayCount)
        {
            IList<RecommendedItem> recommendItems = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Blog(), commendTypeID, displayCount ?? 9);
            //IList<BlogThread> recommendBlogThreads = new List<BlogThread>();
            //foreach (RecommendedItem item in recommendItems)
            //{
            //    if (item != null)
            //    {
            //        BlogThread nowThread = BlogThreads.GetThread(item.ItemID, PrivacyStatuses.Public, false);
            //        if (nowThread != null)
            //            recommendBlogThreads.Add(nowThread);
            //    }
            //}
            //ViewData["commendBlogThreads"] = recommendBlogThreads;
            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID ?? 0, false);

            return PartialView("Controls/Blogs/RecommendBlogThreadBlock.ascx", recommendItems);
        }

        #endregion


    }
}
