﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;

using System.Web;
using SpaceBuilder.Blog;
using SpaceBuilder.Common.Handlers;
using SpaceBuilder.Rss;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;

namespace SpaceBuilder.Blog.Handlers
{
    /// <summary>
    /// 用于生成博客评论RSS
    /// </summary>
    /// <remarks>
    /// 例如：生成博客最新评论,获取某篇文章的评论
    /// </remarks>
    public class BlogCommentRssHandler : BaseRssHandler
    {
        private User blogOwner = null;
        /// <summary>
        /// 博客主人
        /// </summary>
        public User BlogOwner
        {
            get { return blogOwner; }
            set { blogOwner = value; }
        }

        private int threadID = -1;
        /// <summary>
        /// 博客主题ID
        /// </summary>
        public int ThreadID
        {
            get { return threadID; }
            set { threadID = value; }
        }

        /// <summary>
        /// 当前博客
        /// </summary>
        public BlogSection CurrentBlog
        {
            get
            {
                return BlogSections.GetSection(BlogOwner.UserID, false);
            }
        }

        /// <summary>        
        /// 获取完整的URL（例如： http://www.spacebuilder.cn/a/b.aspx）
        /// </summary>
        protected override string BaseUrl
        {
            get { return WebUtils.HostPath(Context.Request.Url); }
        }

        /// <summary>
        /// 缓存时间
        /// </summary>
        protected override int CacheTime
        {
            get { return BlogSectionConfiguration.Instance().RssCacheWindowInSeconds; }
        }

        /// <summary>
        /// 缓存标识
        /// </summary>
        protected override string CacheKey
        {
            get
            {
                if (ThreadID > 0)
                    return string.Format("Rss_BlogComments:{0}:ThreadID:{1}", BlogOwner.UserID, ThreadID);
                else
                    return string.Format("Rss_BlogComments:{0}:MostRecent", BlogOwner.UserID);
            }
        }

        /// <summary>
        /// 处理Feed
        /// </summary>
        protected override void ProcessFeed()
        {
            int ownerUserID = this.Context.Request.QueryString.GetInt("OwnerUserID", 0);
            if (ownerUserID > 0)
            {
                BlogOwner = Users.GetUser(ownerUserID, false);
            }

            if (BlogOwner == null)
            {
                BlogOwner = Users.GetUser(WebUtils.UrlDecode(this.Context.Request.QueryString.GetString("UserDomainName", string.Empty)), false);
            }

            if (string.IsNullOrEmpty(this.Context.Request.QueryString["ThreadID"]))
                int.TryParse(this.Context.Request.QueryString["ThreadID"], out this.threadID);

            base.ProcessFeed();
        }

        /// <summary>
        /// 创建Feed
        /// </summary>
        protected override CachedFeed BuildFeed()
        {
            if (CurrentBlog == null)
                return new CachedFeed(DateTime.Now, null, string.Empty);

            IList<BlogComment> blogComments;
            if (this.ThreadID > 0)
            {
                blogComments = BlogComments.GetCommentsOfThread(this.ThreadID, 1, false).Records;
            }
            else
            {
                blogComments = BlogComments.GetTopComments(CurrentBlog.OwnerUserID, BlogSectionConfiguration.Instance().RssDefaultThreadsPerFeed);
            }

            RssDocument rssDoc = new RssDocument();
            rssDoc.UseExternalNamespaces = true;

            if (!string.IsNullOrEmpty(StyleSheet))
                rssDoc.StyleSheet = FormatUrl(WebUtils.ResolveUrl(StyleSheet));

            RssChannel rssChannel = new RssChannel();
            rssChannel.Title = CurrentBlog.SectionName;
            rssChannel.Link = FormatUrl(SiteUrls.Instance().UserChannelHome(blogOwner, ApplicationIDs.Instance().Blog()));
            rssChannel.Description = CurrentBlog.Description;
            //rssChannel.Language = currentBlog.Langugage;
            rssChannel.Generator = SPBConfig.Instance().SpaceBuilderVersionInfo;

            List<RssItem> rssItems = new List<RssItem>();

            foreach (BlogComment comment in blogComments)
            {
                RssItem item = new RssItem();


                item.Author = comment.Author;
                item.PubDate = comment.PostDate;

                RssGuid guid = new RssGuid();
                guid.IsPermaLink = false;
                guid.Text = SiteSettingsManager.GetSiteSettings().SiteKey + ":BlogSection:" + comment.PostID;
                item.Guid = guid;

                BlogThread thread = BlogThreads.GetThread(comment.ParentID, PrivacyStatuses.Public, false);
                if (thread != null)
                {
                    item.Title = WebUtils.HtmlDecode("re: " + thread.Subject);
                    item.Link = FormatUrl(SiteUrls.Instance().ShowBlogThread(BlogOwner.UserName, thread, comment.PostID));
                    item.Comments = FormatUrl(SiteUrls.Instance().ShowBlogThread(BlogOwner.UserName, thread));
                }

                item.Description = comment.Body;
                rssItems.Add(item);
            }

            rssDoc.Channel = rssChannel;
            rssChannel.Items = rssItems;

            DateTime dt = blogComments.Count > 0 ? blogComments[0].PostDate : DateTime.Now;

            return new CachedFeed(dt, null, rssDoc.ToXml());
        }


    }
}
