﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;

using System.Web;
using SpaceBuilder.Blog;
using SpaceBuilder.Common.Handlers;
using SpaceBuilder.Rss;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;

namespace SpaceBuilder.Blog.Handlers
{
    /// <summary>
    /// 用于生成博客文章rss
    /// </summary>
    public class BlogThreadRssHandler : BaseRssHandler
    {

        private string tagName = null;
        /// <summary>
        /// 标签名称
        /// </summary>
        public string TagName
        {
            get { return tagName; }
            set { tagName = value; }
        }


        private User blogOwner = null;
        /// <summary>
        /// 博客主人
        /// </summary>
        public User BlogOwner
        {
            get { return blogOwner; }
            set { blogOwner = value; }
        }

        /// <summary>
        /// 当前博客
        /// </summary>
        public BlogSection CurrentBlog
        {
            get
            {
                return BlogSections.GetSection(BlogOwner.UserID, false);
            }
        }

        /// <summary>        
        /// 获取完整的URL（例如： http://www.spacebuilder.cn/a/b.aspx）
        /// </summary>
        protected override string BaseUrl
        {
            get { return WebUtils.HostPath(Context.Request.Url); }
        }

        /// <summary>
        /// 缓存时间
        /// </summary>
        protected override int CacheTime
        {
            get { return BlogSectionConfiguration.Instance().RssCacheWindowInSeconds; }
        }

        /// <summary>
        /// 缓存标识
        /// </summary>
        protected override string CacheKey
        {
            get
            {
                if (!string.IsNullOrEmpty(TagName))
                    return string.Format("Rss_BlogThreads:{0}:TagName:{1}", BlogOwner.UserID, TagName);
                else
                    return string.Format("Rss_BlogThreads:{0}", BlogOwner.UserID);
            }
        }

        /// <summary>
        /// 处理Feed
        /// </summary
        protected override void ProcessFeed()
        {
            int ownerUserID = this.Context.Request.QueryString.GetInt("OwnerUserID", 0);
            if (ownerUserID > 0)
            {
                BlogOwner = Users.GetUser(ownerUserID, false);
            }

            if (BlogOwner == null)
            {
                BlogOwner = Users.GetUser(WebUtils.UrlDecode(this.Context.Request.QueryString.GetString("UserDomainName", string.Empty)), false);
            }

            this.TagName = this.Context.Request.QueryString["TagName"];

            base.ProcessFeed();
        }

        /// <summary>
        /// 创建Feed
        /// </summary>
        protected override CachedFeed BuildFeed()
        {
            if (CurrentBlog == null)
                return new CachedFeed(DateTime.Now, null, string.Empty);

            User blogOwner = Users.GetUser(CurrentBlog.OwnerUserID, false);

            PagingDataSet<BlogThread> pdsBlogThreads = BlogThreads.GetThreads(blogOwner.UserID, null, null, 1);
            IList<BlogThread> blogThreads = pdsBlogThreads.TotalRecords > BlogSectionConfiguration.Instance().RssDefaultThreadsPerFeed ? pdsBlogThreads.Records : pdsBlogThreads.Records;
            if (blogThreads == null || blogThreads.Count == 0)
                return new CachedFeed(DateTime.Now, null, string.Empty);

            RssDocument rssDoc = new RssDocument();
            rssDoc.UseExternalNamespaces = true;
            if (!string.IsNullOrEmpty(StyleSheet))
                rssDoc.StyleSheet = FormatUrl(WebUtils.ResolveUrl(StyleSheet));

            RssChannel rssChannel = new RssChannel();
            rssChannel.Title = CurrentBlog.SectionName;
            rssChannel.Link = FormatUrl(SiteUrls.Instance().UserChannelHome(blogOwner, ApplicationIDs.Instance().Blog()));
            rssChannel.Description = CurrentBlog.Description;
            //rssChannel.Language = currentBlog.Langugage;
            rssChannel.Generator = SPBConfig.Instance().SpaceBuilderVersionInfo;

            List<RssItem> rssItems = new List<RssItem>();

            foreach (BlogThread thread in blogThreads)
            {
                RssItem item = new RssItem();
                item.Title = WebUtils.HtmlDecode(thread.Subject);
                item.Link = FormatUrl(SiteUrls.Instance().ShowBlogThread(blogOwner.UserName, thread));
                item.Author = thread.Author;
                item.PubDate = thread.PostDate;

                RssGuid guid = new RssGuid();
                guid.IsPermaLink = false;
                guid.Text = SiteSettingsManager.GetSiteSettings().SiteKey + ":BlogSection:" + thread.ThreadID;
                item.Guid = guid;

                item.Comments = FormatUrl(SiteUrls.Instance().ShowBlogThread(blogOwner.UserName, thread));

                if (BlogSectionConfiguration.Instance().TruncateAggregatePost)
                    item.Description = HtmlUtils.TrimHtml(thread.GetBody(false), BlogSectionConfiguration.Instance().AggregatePostSize);
                else
                    item.Description = thread.GetBody(false);

                //是否启用rss阅读计数
                if (CurrentBlog.EnableAggBugs)
                {
                    item.Description = string.Format("{0}<img src=\"{1}\" width=\"1\" height=\"1\">", item.Description, FormatUrl(SiteUrls.Instance().AggView(typeof(BlogThreadCounter), thread.ThreadID)));
                }

                item.ReplyCount = thread.CommentCount;

                rssItems.Add(item);
            }

            rssDoc.Channel = rssChannel;
            rssChannel.Items = rssItems;

            DateTime dt = blogThreads.Count > 0 ? blogThreads[0].PostDate : DateTime.Now;

            return new CachedFeed(dt, null, rssDoc.ToXml());
        }


    }
}
