﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using Lucene.Net.Analysis;
using Lucene.Net.Documents;
using Lucene.Net.Index;
using Lucene.Net.QueryParsers;
using Lucene.Net.Search;
using SpaceBuilder.Blog;
using SpaceBuilder.Common;
using SpaceBuilder.Utils;


namespace SpaceBuilder.LuceneSearch
{
    /// <summary>
    /// 博客全文检索
    /// </summary>
    public class BlogSearchManager : SearchManagerBase<BlogThread>
    {
        //博客索引文件夹
        private static readonly string BlogIndexFileDirectory = "Blog";

        //用volatile修饰后的变量不允许有不同于“主”内存区域的变量拷贝。换句话说，
        //一个变量经volatile修饰后在所有线程中必须是同步的；任何线程中改变了它的值，
        //所有其他线程立即获取到了相同的值。理所当然的，volatile修饰的变量存取时比一般变量消耗的资源要多一点，因为线程有它自己的变量拷贝更为高效
        //(volatile 修饰符通常用于由多个线程访问而不使用 lock 语句（C# 参考） 语句对访问进行序列化的字段)
        private static volatile BlogSearchManager _self = null;
        //private static BlogSearchManager _self = null;
        private static readonly object lockObject = new object();

        private BlogSearchManager(string indexFileDirectory)
            : base(indexFileDirectory)
        {
            //if (searchClient == null)
            //    searchClient = new BlogSearchClient("BlogSearchService");
        }

        public static BlogSearchManager Instance()
        {
            if (_self == null)
            {
                lock (lockObject)
                {
                    if (_self == null)
                    {
                        _self = new BlogSearchManager(BlogIndexFileDirectory);
                    }
                }
            }

            return _self;
        }

        /// <summary>
        /// 搜索
        /// </summary>
        /// <param name="query">BlogThreadFullTextQuery</param>
        /// <returns></returns>
        public SearchResultDataSet<BlogThread> Search(BlogThreadFullTextQuery query)
        {
            //索引文件不存在时，返回null
            if (!IsIndexFilesExists)
                return new SearchResultDataSet<BlogThread>();

            BooleanQuery currentQuery = new BooleanQuery();
            BooleanQuery queryForFilter = new BooleanQuery();
            
            if (query.SearchScopeUserID > 0)
            {
                Term userIDTerm = new Term(BlogThreadIndexFields.UserID, query.SearchScopeUserID.ToString());
                Query userIDQuery = new TermQuery(userIDTerm);
                queryForFilter.Add(userIDQuery, BooleanClause.Occur.MUST);
            }

            if (query.SiteCategoryID > 0)
            {
                if (query.IncludeSiteCategoryDescendant)
                {
                    QueryParser siteCategoryIDQueryParser = new QueryParser(CurrentLuceneVersion, BlogThreadIndexFields.SiteCategoryID, new WhitespaceAnalyzer());

                    List<SiteCategory> childSiteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Blog()).GetAllChilds(query.SiteCategoryID, false);
                    StringBuilder siteCategoryIDs = new StringBuilder();
                    siteCategoryIDs.Append(query.SiteCategoryID.ToString());
                    if (childSiteCategories != null)
                    {
                        foreach (SiteCategory siteCategory in childSiteCategories)
                        {
                            siteCategoryIDs.Append(" " + siteCategory.CategoryID);
                        }
                    }

                    Query siteCategoryIDQuery = siteCategoryIDQueryParser.Parse(siteCategoryIDs.ToString());
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
                else
                {
                    Term siteCategoryIDTerm = new Term(BlogThreadIndexFields.SiteCategoryID, query.SiteCategoryID.ToString());
                    Query siteCategoryIDQuery = new TermQuery(siteCategoryIDTerm);
                    currentQuery.Add(siteCategoryIDQuery, BooleanClause.Occur.MUST);
                }
            }

            if (query.PrivacyStatusForDisplay.HasValue)
            {
                if (query.PrivacyStatusForDisplay != null)
                {
                    switch (query.PrivacyStatusForDisplay)
                    {
                        case PrivacyStatusesForDisplay.Privacy:
                        case PrivacyStatusesForDisplay.NeedPassword:
                        case PrivacyStatusesForDisplay.OnlyFriend:
                        case PrivacyStatusesForDisplay.Public:
                            Term term = new Term(BlogThreadIndexFields.PrivacyStatus, ((int)query.PrivacyStatusForDisplay).ToString());
                            TermQuery termQuery = new TermQuery(term);
                            queryForFilter.Add(termQuery, BooleanClause.Occur.MUST);
                            break;
                        case PrivacyStatusesForDisplay.NeedPassword_GreaterThanOrEqual:
                            string greaterOrEqualPasswordstrRange = ((int)PrivacyStatusesForDisplay.Public).ToString() + " " + ((int)PrivacyStatusesForDisplay.OnlyFriend).ToString() + " " + ((int)PrivacyStatusesForDisplay.NeedPassword).ToString();
                            queryForFilter.Add(new QueryParser(CurrentLuceneVersion, BlogThreadIndexFields.PrivacyStatus, new WhitespaceAnalyzer()).Parse(greaterOrEqualPasswordstrRange), BooleanClause.Occur.MUST);
                            break;
                        case PrivacyStatusesForDisplay.OnlyFriend_GreaterThanOrEqual:
                            string greaterOrEqualOnlyPassword = ((int)PrivacyStatusesForDisplay.Public).ToString() + " " + ((int)PrivacyStatusesForDisplay.OnlyFriend).ToString();
                            queryForFilter.Add(new QueryParser(CurrentLuceneVersion, BlogThreadIndexFields.PrivacyStatus, new WhitespaceAnalyzer()).Parse(greaterOrEqualOnlyPassword), BooleanClause.Occur.MUST);
                            break;
                        default:
                            break;
                    }
                }
            }
                        
            if (!string.IsNullOrEmpty(query.Keyword))
            {
                Query postKeywordQuery = null;
                query.Keyword = StringUtilsForLucene.LuceneKeywordsScrubber(query.Keyword.ToLower());
                string keywordSegments = SegmentForQueryParser(query.Keyword);

                if (!string.IsNullOrEmpty(query.Keyword))
                {
                    string[] searchFieldsForKeyword = new string[4];
                    searchFieldsForKeyword[0] = BlogThreadIndexFields.Subject;
                    searchFieldsForKeyword[1] = BlogThreadIndexFields.Body;
                    searchFieldsForKeyword[2] = BlogThreadIndexFields.Tags;
                    searchFieldsForKeyword[3] = BlogThreadIndexFields.Author;

                    MultiFieldQueryParser keywordParser = new MultiFieldQueryParser(CurrentLuceneVersion, searchFieldsForKeyword, GetChineseAnalyzerOfUnTokenized());
                    keywordParser.SetLowercaseExpandedTerms(true);
                    keywordParser.SetDefaultOperator(QueryParser.OR_OPERATOR);
                    postKeywordQuery = keywordParser.Parse(keywordSegments);
                    currentQuery.Add(postKeywordQuery, BooleanClause.Occur.MUST);
                }
            }

            if (!string.IsNullOrEmpty(query.TagName))
            {
                string[] tagSegments = SegmentForPhraseQuery(query.TagName);
                if (tagSegments != null && tagSegments.Length > 0)
                {
                    PhraseQuery tagQuery = new PhraseQuery();
                    foreach (var tagSegment in tagSegments)
                        tagQuery.Add(new Term(BlogThreadIndexFields.Tags, tagSegment));

                    tagQuery.SetSlop(PhraseQuerySlop);
                    tagQuery.SetBoost((float)Math.Pow(3, 5));
                    currentQuery.Add(tagQuery, BooleanClause.Occur.MUST);
                }
            }
            
            SortField[] sortFields;
            switch (query.SortBy)
            {
                case FullTextQueryBlogThreadsSortBy.DateCreated:
                    sortFields = new SortField[] { new SortField(BlogThreadIndexFields.ThreadID, SortField.INT, true) };
                    break;
                default:
                    sortFields = new SortField[] { SortField.FIELD_SCORE };
                    break;
            }

            Filter filter = null;
            if (queryForFilter.Clauses().Count > 0)
                filter = new QueryWrapperFilter(queryForFilter);

            SearchResultDataSet<BlogThread> pds = Search(currentQuery,filter, sortFields, query.PageIndex, query.PageSize);
            foreach (var item in pds.Records)
            {
                item.Subject = HighlighterForKeyWord(item.Subject, query.Keyword);
                item.ForcedExcerpt = HighlighterForKeyWord(item.ForcedExcerpt, query.Keyword);
            }
            return pds;

        }

        /// <summary>
        /// 获取相关文章（已经排除自身）
        /// </summary>
        /// <param name="thread"></param>
        /// <param name="topNumber">获取的最大记录数</param>
        /// <returns></returns>
        public ICollection<BlogThread> GetCorrelativeThreads(BlogThread thread, int topNumber)
        {
            //索引文件不存在时，返回null
            if (!IsIndexFilesExists || thread == null)
                return new List<BlogThread>();

            BooleanQuery currentQuery = new BooleanQuery();

            if (thread.Tags != null && thread.Tags.Count > 0)
            {
                foreach (var tag in thread.Tags)
                {
                    string[] tagSegments = SegmentForPhraseQuery(tag);
                    if (tagSegments != null && tagSegments.Length > 0)
                    {
                        PhraseQuery tagQuery = new PhraseQuery();
                        foreach (var tagSegment in tagSegments)
                            tagQuery.Add(new Term(BlogThreadIndexFields.Tags, tagSegment));

                        tagQuery.SetSlop(PhraseQuerySlop);
                        tagQuery.SetBoost((float)Math.Pow(3, 5));
                        currentQuery.Add(tagQuery, BooleanClause.Occur.SHOULD);
                    }
                }
            }

            string subjectSegments = SegmentForQueryParser(StringUtilsForLucene.LuceneKeywordsScrubber(thread.Subject));
            QueryParser subjectQueryParser = new QueryParser(CurrentLuceneVersion, BlogThreadIndexFields.Subject, GetChineseAnalyzerOfUnTokenized());
            currentQuery.Add(subjectQueryParser.Parse(subjectSegments), BooleanClause.Occur.SHOULD);
            QueryParser bodyQueryParser = new QueryParser(CurrentLuceneVersion, BlogThreadIndexFields.Body, GetChineseAnalyzerOfUnTokenized());
            currentQuery.Add(bodyQueryParser.Parse(subjectSegments), BooleanClause.Occur.SHOULD);

            //限定为公开的文章
            Term privacyStatusTerm = new Term(BlogThreadIndexFields.PrivacyStatus, ((int)PrivacyStatusesForDisplay.Public).ToString());
            TermQuery queryForFilter = new TermQuery(privacyStatusTerm);

            ICollection<BlogThread> threads = Search(currentQuery, null, null, topNumber + 1);
            BlogThread self = threads.Where(n => n.ThreadID == thread.ThreadID).SingleOrDefault();
            if (self != null)
                threads.Remove(self);
            else
                threads.Remove(threads.Last());

            return threads;
        }


        #region 插入删除更新索引已经提取到父类

        ///// <summary>
        ///// 加入索引
        ///// </summary>
        //public bool Insert(IList<BlogThread> threads)
        //{
        //    if (!IsIndexDirectoryExists)
        //    {
        //        try
        //        {
        //            System.IO.Directory.CreateDirectory(PhysicalIndexDirectory);
        //            isIndexDirectoryExists = true;
        //        }
        //        catch
        //        {
        //            throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
        //        }
        //    }

        //    return Insert(threads, PhysicalIndexDirectory, !IsIndexFilesExists);
        //}

        ///// <summary>
        ///// 加入索引
        ///// </summary>
        ///// <param name="createIndexFile">是否创建索引文件</param>
        //public bool Insert(IList<BlogThread> threads, string indexPath, bool createIndexFile)
        //{
        //    if (threads == null || threads.Count == 0)
        //        return false;

        //    bool result = false;
        //    lock (lockObject)
        //    {
        //        // 第一个参数是存放索引目录有FSDirectory（存储到磁盘上）和RAMDirectory（存储到内存中）， 第二个参数是使用的分词器， 第三个：true，建立全新的索引，false,建立增量索引，第四个是.字段的最长文本，如果超过这个最长的文本，就切除超过的部分。如果你有些字段的文本长度有可能超过10000，那就要需要改动！ 为了性能，完全没必要。
        //        IndexWriter fsWriter = new IndexWriter(FSDirectory.Open(new DirectoryInfo(indexPath)), GetChineseAnalyzer(), true, IndexWriter.MaxFieldLength.UNLIMITED);
        //        // 索引合并因子
        //        // SetMergeFactor（合并因子）   
        //        // SetMergeFactor是控制segment合并频率的，其决定了一个索引块中包括多少个文档，当硬盘上的索引块达到多少时，   
        //        // 将它们合并成一个较大的索引块。当MergeFactor值较大时，生成索引的速度较快。MergeFactor的默认值是10，建议在建立索引前将其设置的大一些。
        //        fsWriter.SetMergeFactor(SearchConfiguration.Instance().MergeFactor);
        //        // SetMaxBufferedDocs（最大缓存文档数）   
        //        // SetMaxBufferedDocs是控制写入一个新的segment前内存中保存的document的数目，   
        //        // 设置较大的数目可以加快建索引速度，默认为10。  
        //        fsWriter.SetMaxBufferedDocs(SearchConfiguration.Instance().MinMergeDocs);
        //        // SetMaxMergeDocs（最大合并文档数）   
        //        // SetMaxMergeDocs是控制一个segment中可以保存的最大document数目，值较小有利于追加索引的速度，默认Integer.MAX_VALUE，无需修改。   
        //        // 在创建大量数据的索引时，我们会发现索引过程的瓶颈在于大量的磁盘操作，如果内存足够大的话，   
        //        // 我们应当尽量使用内存，而非硬盘。可以通过SetMaxBufferedDocs来调整，增大Lucene使用内存的次数。   
        //        fsWriter.SetMaxMergeDocs(SearchConfiguration.Instance().MaxMergeDocs);
        //        // SetUseCompoundFile这个方法可以使Lucene在创建索引库时，会合并多个 Segments 文件到一个.cfs中。   
        //        // 此方式有助于减少索引文件数量，对于将来搜索的效率有较大影响。   
        //        // 压缩存储（True则为复合索引格式）   
        //        fsWriter.SetUseCompoundFile(true);

        //        //FSDirectory fsDir;
        //        //if (createIndexFile)
        //        //{
        //        //fsDir = FSDirectory.GetDirectory(indexPath, true);
        //        //    fsWriter = new IndexWriter(fsDir, GetChineseAnalyzer(), true);
        //        //}
        //        //else
        //        //{
        //        //fsDir = FSDirectory.GetDirectory(indexPath, false);
        //        //    fsWriter = new IndexWriter(fsDir, GetChineseAnalyzer(), false);
        //        //}


        //        try
        //        {
        //            foreach (BlogThread thread in threads)
        //            {
        //                if (thread != null)
        //                {
        //                    Document doc = ConvertBlogThreadToDocument(thread);
        //                    if (doc != null)
        //                        fsWriter.AddDocument(doc);
        //                }
        //            }
        //            result = true;
        //            //优化索引，使多个Segments变成一个Segments  
        //            //optimize()  
        //            //指定最大Segments的数量  
        //            //optimize(int maxNumSegments)  
        //            //前面的方面都是优化完成之后再返回，这个方法的参数如果是FALSE的话，就直接返回，再开一个线程来优化  
        //            //optimize(boolean doWait)  
        //            //前面两个参数的组合
        //            //optimize(int maxNumSegments, boolean doWait) 
        //            fsWriter.Optimize();
        //        }
        //        finally
        //        {
        //            fsWriter.Close();
        //        }
        //    }
        //    return result;
        //}

        //private static readonly object delLockObject = new object();
        ///// <summary>
        ///// 删除索引
        ///// </summary>
        //public bool Delete(string[] ids)
        //{
        //    return Delete(ids, true);
        //}

        ///// <summary>
        ///// 删除索引
        ///// </summary>
        //public bool Delete(string[] ids, bool needOptimize)
        //{
        //    if (ids == null && ids.Length == 0)
        //        return false;

        //    if (!IsIndexFilesExists)
        //        return false;

        //    bool result = false;
        //    lock (delLockObject)
        //    {
        //        IndexWriter iw = null;
        //        try
        //        {
        //            iw = new IndexWriter(FSDirectory.Open(new DirectoryInfo(PhysicalIndexDirectory)), GetChineseAnalyzer(), false, IndexWriter.MaxFieldLength.UNLIMITED);
        //            for (int i = 0; i < ids.Length; i++)
        //            {
        //                if (ids[i] != null)
        //                {
        //                    Term term = new Term(BlogThreadIndexFields.ThreadID, ids[i]);
        //                    iw.DeleteDocuments(term);
        //                }
        //            }
        //            if (needOptimize)
        //                iw.Optimize();
        //            result = true;
        //        }
        //        finally
        //        {
        //            if (iw != null)
        //                iw.Close();
        //        }
        //    }
        //    return result;
        //}


        //private static readonly object updateLockObject = new object();
        ///// <summary>
        ///// 更新索引
        ///// </summary>
        //public bool Update(IList<BlogThread> threads)
        //{
        //    if (threads == null || threads.Count == 0)
        //        return false;

        //    bool result = false;
        //    lock (updateLockObject)
        //    {
        //        string[] ids = Array.ConvertAll(threads.Select(n => n.ThreadID).ToArray(), n => n.ToString());
        //        result = Delete(ids, BlogThreadIndexFields.ThreadID, false);

        //        if (result || !IsIndexFilesExists)
        //            result = Insert(threads);
        //    }
        //    return result;
        //}

        #endregion

        #region Help Methods

        ///// <summary>
        ///// 从BlogThread转换为PostSearchItem
        ///// </summary>
        //private static PostSearchItem PopulatePostSearchItemFromBlogThread(BlogThread blogThread)
        //{
        //    PostSearchItem postSearchItem = null;

        //    if (blogThread == null)
        //        return null;
        //    else
        //    {
        //        BlogSection blogSection = blogThread.BlogSection;
        //        if (blogSection != null)
        //        {
        //            postSearchItem = new PostSearchItem();
        //            postSearchItem.ApplicationID = ApplicationIDs.Instance().Blog();
        //            postSearchItem.UniqueID = string.Format("{0}-{1}", postSearchItem.ApplicationID, blogThread.ThreadID);
        //            postSearchItem.UserID = blogThread.OwnerUserID;
        //            postSearchItem.SectionName = blogSection.SectionName;
        //            postSearchItem.ThreadID = blogThread.ThreadID;
        //            postSearchItem.Author = blogThread.Author;
        //            postSearchItem.Subject = HtmlUtils.StripAllTags(blogThread.Subject, false);
        //            postSearchItem.Body = HtmlUtils.StripAllTags(blogThread.Body, false);
        //            postSearchItem.SubjectPrefixID = blogThread.BlogThreadTypeID;
        //            postSearchItem.UserCategoryID = blogThread.UserCategoryID;

        //            if (blogThread.UserCategory != null)
        //                postSearchItem.UserCategoryName = blogThread.UserCategory.CategoryName;
        //            else
        //                postSearchItem.UserCategoryName = string.Empty;

        //            postSearchItem.SiteCategoryID = blogThread.SiteCategoryID;

        //            if (blogThread.Tags.Count > 0)
        //            {
        //                string[] tagNames = new string[blogThread.Tags.Count];
        //                for (int i = 0; i < blogThread.Tags.Count; i++)
        //                {
        //                    tagNames[i] = blogThread.Tags[i].TagName;
        //                }
        //                postSearchItem.Tags = tagNames;
        //            }

        //            postSearchItem.PrivacyStatus = blogThread.PrivacyStatus;
        //            postSearchItem.IsEssential = blogThread.IsEssential;
        //            postSearchItem.AuditingStatus = blogThread.AuditingStatus;
        //            postSearchItem.PostDate = blogThread.LastRepliedDate;
        //            postSearchItem.LastUpdatedDate = blogThread.LastRepliedDate;
        //        }
        //    }

        //    return postSearchItem;
        //}

        #endregion

        #region Base

        /// <summary>
        /// 初始化索引
        /// </summary>
        public override void InitializeIndex(string indexPath)
        {
            if (!System.IO.Directory.Exists(indexPath))
            {
                try
                {
                    System.IO.Directory.CreateDirectory(indexPath);
                }
                catch
                {
                    throw new ApplicationException(string.Format("create Directory '{0}' failed", PhysicalIndexDirectory));
                }
            }

            #region 索引Blog

            int indexPageSize = 2000;
            bool createIndexFile = true;
            PagingDataSet<BlogThread> pds = BlogThreads.GetThreadsForAdmin(indexPageSize, 1, Applications.GetApplication(ApplicationIDs.Instance().Blog()).AuditingStatusForPublicDisplay, -1, string.Empty);
            double tIndex = Convert.ToDouble(pds.TotalRecords) / Convert.ToDouble(indexPageSize);
            int indexPageIndex = (int)Math.Ceiling(tIndex);

            if (pds.TotalRecords > 0)
            {
                //分多次进行索引
                for (int i = 1; i <= indexPageIndex; i++)
                {
                    if (i != 1)
                        pds = BlogThreads.GetThreadsForAdmin(indexPageSize, i, Applications.GetApplication(ApplicationIDs.Instance().Blog()).AuditingStatusForPublicDisplay, -1, string.Empty);
                    Insert(pds.Records, indexPath, createIndexFile);
                    if (createIndexFile)
                        createIndexFile = false;
                }
            }

            #endregion
        }

        /// <summary>
        /// 将博客主题实体转换为Document
        /// </summary>
        /// <param name="thread"></param>
        /// <returns></returns>
        protected override Document ConvertObjToDocument(BlogThread thread)
        {
            if (thread == null)
                return null;

            //待审核及未通过审核不允许加入索引
            if (thread.AuditingStatus == AuditingStatuses.Pending || thread.AuditingStatus == AuditingStatuses.Fail)
                return null; 

            // 增加索引字段
            //         
            // 在Field中有三个内部类：Field.Index,Field.Store,Field.termVector，而构造函数也用到了它们。   
            // 参数说明：   
            // Field.Store：
            // Field.Store.NO：表示该Ｆield不需要存储。   
            // Field.Store.Yes：表示该Ｆield需要存储。   
            // Field.Store.COMPRESS：表示使用压缩方式来存储。   
            // Field.Index：
            // Field.Index.NO：表示该Ｆield不需要索引。   
            // Field.Index.ANALYZED ：表示该Ｆield先被分词再索引。   
            // Field.Index.NOT_ANALYZED：表示不对该Ｆield进行分词，但要对其索引。   
            // TermVector这个参数也不常用，它有五个选项。
            //                Field.TermVector.NO表示不索引Token的位置属性；
            //                Field.TermVector.WITH_OFFSETS表示额外索引Token的结束点；
            //                Field.TermVector.WITH_POSITIONS表示额外索引Token的当前位置；
            //                Field.TermVector.WITH_POSITIONS_OFFSETS表示额外索引Token的当前和结束位置；
            //                Field.TermVector.YES则表示存储向量。

            // 增加文档 Field相当于增加数据库字段一样检索,获取都需要的内容,直接放index中,不过这样会增大index,保存文件的txt内容
            /**
             * Field.Store 表示“是否存储”，即该Field内的信息是否要被原封不动的保存在索引中。
             * Field.Index 表示“是否索引”，即在这个Field中的数据是否在将来检索时需要被用户检索到，一个“不索引”的Field通常仅是提供辅助信息储存的功能。
             * Field.TermVector 表示“是否切词”，即在这个Field中的数据是否需要被切词。
             */

            Document doc = new Document();
            Field field;

            field = new Field(BlogThreadIndexFields.UserID, thread.OwnerUserID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.ThreadID, thread.ThreadID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.Author, thread.Author.ToLower(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.Subject, thread.Subject.ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            field.SetBoost(2.0F);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.Body, HtmlUtils.StripAllTags(thread.GetBody(false)).ToLower(), Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.YES);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.UserCategoryID, thread.UserCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.SiteCategoryID, thread.SiteCategoryID.ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            if (thread.Tags != null)
            {
                foreach (var tag in thread.Tags)
                {
                    field = new Field(BlogThreadIndexFields.Tags, tag.ToLower(), Field.Store.YES, Field.Index.ANALYZED);
                    field.SetBoost(2.0F);
                    doc.Add(field);
                }
            }

            field = new Field(BlogThreadIndexFields.PrivacyStatus, ((int)thread.PrivacyStatus).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.AuditingStatus, ((int)thread.AuditingStatus).ToString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            field = new Field(BlogThreadIndexFields.PostDate, DateTools.DateToString(thread.PostDate, DateTools.Resolution.DAY), Field.Store.YES, Field.Index.NOT_ANALYZED);
            doc.Add(field);

            return doc;
        }

        /// <summary>
        /// Document转化成BlogThread
        /// </summary>
        protected override BlogThread ConvertDocumentToObj(Document doc)
        {
            BlogThread blogThread = new BlogThread();
            int userID;
            int.TryParse(doc.Get(BlogThreadIndexFields.UserID), out userID);
            blogThread.OwnerUserID = userID;

            blogThread.Author = doc.Get(BlogThreadIndexFields.Author);

            int threadID;
            int.TryParse(doc.Get(BlogThreadIndexFields.ThreadID), out threadID);
            blogThread.ThreadID = threadID;

            blogThread.Subject = doc.Get(BlogThreadIndexFields.Subject);
            blogThread.ForcedExcerpt = doc.Get(BlogThreadIndexFields.Body);

            int userCategoryID;
            int.TryParse(doc.Get(BlogThreadIndexFields.UserCategoryID), out userCategoryID);
            blogThread.UserCategoryID = userCategoryID;

            int siteCategoryID;
            int.TryParse(doc.Get(BlogThreadIndexFields.SiteCategoryID), out siteCategoryID);
            blogThread.SiteCategoryID = siteCategoryID;

            int privacyStatus;
            int.TryParse(doc.Get(BlogThreadIndexFields.PrivacyStatus), out privacyStatus);
            blogThread.PrivacyStatus = (PrivacyStatuses)privacyStatus;

            int auditingStatusValue = (int)AuditingStatuses.Success;
            int.TryParse(doc.Get(BlogThreadIndexFields.AuditingStatus), out auditingStatusValue);
            blogThread.AuditingStatus = (AuditingStatuses)auditingStatusValue;

            try
            {
                blogThread.PostDate = DateTools.StringToDate(doc.Get(BlogThreadIndexFields.PostDate));
            }
            catch { }

            return blogThread;
        }

        #endregion
    }
}
