﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using CookComputing.XmlRpc;
using System.IO;
using System.Collections.Specialized;
using SpaceBuilder.Blog.MetaWeblogAPI;
using SpaceBuilder.Common;
using SpaceBuilder.Utils;
using System.Linq;


namespace SpaceBuilder.Blog.Handlers
{
    public class MetaWeblog : XmlRpcService, IMetaWeblog
    {

        #region IMetaWeblog 成员

        /// <summary>
        /// 创建博客文章
        /// </summary>
        /// <param name="blogid">博客ID</param>
        /// <param name="username">用户名</param>
        /// <param name="password">密码</param>
        /// <param name="post">文章实体</param>
        /// <param name="publish">是否公开</param>
        public string newPost(string blogid, string username, string password, SpaceBuilder.Blog.MetaWeblogAPI.Post post, bool publish)
        {
            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser == null)
                    throw new XmlRpcFaultException(0, "User does not exist");

                int sectionID = -1;
                int.TryParse(blogid, out sectionID);
                BlogSection blogSection = BlogSections.GetSection(sectionID, false);

                //检查当前用户是否具有发布文章权限
                PostPermissionCheck(blogSection, currentUser);

                BlogThread postToAdd = new BlogThread();
                postToAdd.OwnerUserID = currentUser.UserID;
                postToAdd.Author = currentUser.DisplayName;

                // Set Default Values
                postToAdd.EnableRatings = blogSection.EnableRating;
                postToAdd.EnableTrackBacks = blogSection.EnableTrackback;
                postToAdd.ModerationType = blogSection.ModerationType;
                postToAdd.IsLocked = !blogSection.EnableComments;

                if (publish)
                    postToAdd.PrivacyStatus = PrivacyStatuses.Public;
                else
                    postToAdd.PrivacyStatus = PrivacyStatuses.Privacy;

                postToAdd.BodyForDataTransmission = post.description;
                postToAdd.Subject = WebUtils.HtmlDecode(post.title);

                List<UserTag> tags = SafeTags(currentUser, post.categories, post.mt_keywords);

                try
                {
                    if (post.dateCreated != DateTime.MinValue)
                        postToAdd.PostDate = ValueHelper.GetSafeSqlDateTime(post.dateCreated);
                    else
                        postToAdd.PostDate = DateTime.Now;
                }
                catch { postToAdd.PostDate = DateTime.Now; }

                if (!string.IsNullOrEmpty(post.mt_excerpt))
                    postToAdd.Excerpt = post.mt_excerpt;

                postToAdd = BlogThreads.CreateThread(postToAdd, tags);

                if (postToAdd.ThreadID > 0)
                    return postToAdd.ThreadID.ToString();
                else
                    throw new XmlRpcFaultException(0, "The post could not be added");
            }

            throw new XmlRpcFaultException(0, "User does not exist");
        }

        /// <summary>
        /// 编辑博客文章
        /// </summary>
        public object editPost(string postid, string username, string password, SpaceBuilder.Blog.MetaWeblogAPI.Post post, bool publish)
        {
            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser == null)
                    throw new XmlRpcFaultException(0, "User does not exist");

                int threadID = 0;
                int.TryParse(postid, out threadID);

                BlogThread blogThread = null;
                if (threadID > 0)
                    blogThread = BlogThreads.GetThread(threadID, null, true);
                else
                    throw new XmlRpcFaultException(0, "BlogThead does not exist");

                BlogSection blogSection = null;
                if (blogThread != null)
                    blogSection = BlogSections.GetSection(blogThread.OwnerUserID, false);

                //检查当前用户是否具有发布文章权限
                PostPermissionCheck(blogSection, currentUser);


                //处理文章Category
                List<UserTag> tags = SafeTags(currentUser, post.categories, post.mt_keywords);

                blogThread.BodyForDataTransmission = post.description;
                blogThread.Subject = WebUtils.HtmlDecode(post.title);

                if (publish)
                    blogThread.PrivacyStatus = PrivacyStatuses.Public;
                else
                    blogThread.PrivacyStatus = PrivacyStatuses.Privacy;

                if (string.IsNullOrEmpty(post.mt_excerpt))
                    blogThread.Excerpt = string.Empty;
                else
                    blogThread.Excerpt = post.mt_excerpt;

                // Get UserTime safely (some clients pass in a DateTime that is not valid)
                try
                {
                    if (post.dateCreated != DateTime.MinValue)
                        blogThread.PostDate = post.dateCreated;
                }
                catch { }


                UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).UpdateUserTagsOfItem(blogThread.ThreadID, blogThread.OwnerUserID, tags == null ? new List<string>() : tags.Select(n => n.TagName).ToList());

                //更新博客文章
                BlogThreads.UpdateThread(blogThread, blogThread.UserCategoryID, tags);

                return true;
            }

            throw new XmlRpcFaultException(0, "User does not exist");
        }

        public CategoryInfo[] getCategories(string blogid, string username, string password)
        {
            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser == null)
                    throw new XmlRpcFaultException(0, "User does not exist");

                ICollection<UserTag> blogTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTags(currentUser.UserID, false);

                CategoryInfo[] categories = new CategoryInfo[blogTags.Count];
                int i = 0;
                foreach (var userTag in blogTags)
                {
                    CategoryInfo ci = new CategoryInfo();
                    ci.categoryid = userTag.TagName;
                    ci.title = userTag.TagName;
                    ci.description = userTag.TagName;
                    ci.htmlUrl = WebUtils.FullPath(SiteUrls.Instance().ListBlogThreadsByUserTag(currentUser.UserName, userTag.TagName));
                    ci.rssUrl = WebUtils.FullPath(SiteUrls.Instance().BlogRss(currentUser.UserName));
                    categories[i] = ci;
                    i++;
                }

                return categories;
            }
            throw new XmlRpcFaultException(0, "User does not exist");
        }

        /// <summary>
        /// 获取博客文章
        /// </summary>
        public SpaceBuilder.Blog.MetaWeblogAPI.Post getPost(string postid, string username, string password)
        {
            if (ValidateUser(username, password))
            {
                int threadID = 0;
                int.TryParse(postid, out threadID);

                BlogThread blogThread = BlogThreads.GetThread(threadID, null, false);

                if (blogThread == null)
                    throw new XmlRpcFaultException(0, "BlogThead does not exist");

                BlogSection blogSection = BlogSections.GetSection(blogThread.OwnerUserID, false);
                User currentUser = Users.GetUser(username, false);
                if (currentUser == null)
                    throw new XmlRpcFaultException(0, "User does not exist");

                //检查当前用户是否具有发布文章权限
                PostPermissionCheck(blogSection, currentUser);

                return ConvertToPost(blogThread);
            }

            throw new XmlRpcFaultException(0, "User does not exist");
        }

        /// <summary>
        /// 获取最新numberOfPosts个博客文章
        /// </summary>
        public SpaceBuilder.Blog.MetaWeblogAPI.Post[] getRecentPosts(string blogid, string username, string password, int numberOfPosts)
        {
            //numberOfPosts目前没起作用

            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser == null)
                    throw new XmlRpcFaultException(0, "User does not exist");
                BlogSection blogSection = BlogSections.GetSection(currentUser.UserID, false);

                //检查当前用户是否具有发布文章权限
                PostPermissionCheck(blogSection, currentUser);

                PagingDataSet<BlogThread> blogThreads = BlogThreads.GetMyThreads(blogSection.OwnerUserID, null, null, 1);

                SpaceBuilder.Blog.MetaWeblogAPI.Post[] posts = new SpaceBuilder.Blog.MetaWeblogAPI.Post[blogThreads.Records.Count];
                for (int i = 0; i < blogThreads.Records.Count; i++)
                {
                    posts[i] = ConvertToPost(blogThreads.Records[i]);
                }
                return posts;
            }
            throw new XmlRpcFaultException(0, "User does not exist");
        }

        //由于博客附件必须与文章关联因此当前不能实现附件上传
        //public MediaObjectInfo newMediaObject(string blogid, string username, string password, MediaObject fileData)
        //{
        //    if (ValidateUser(username, password))
        //    {
        //        User currentUser = Users.GetUser(username);
        //        BlogSection weblog = BlogSections.GetBlogSection(currentUser.UserID);

        //        //检查当前用户是否具有发布文章权限
        //        PostPermissionCheck(weblog, currentUser);

        //        //if (!BlogSectionConfiguration.Instance().AttachmentSettings.EnableFileSystemStorage)
        //        //    throw new XmlRpcFaultException(0, "Blog file storage not allowed");

        //        try
        //        {
        //            if (fileData.bits != null && !string.IsNullOrEmpty(fileData.name))
        //            {
        //                AttachmentSettings fs = BlogSectionConfiguration.Instance().AttachmentSettings;
        //                if ((fileData.bits.Length / 1024) < fs.MaxAttachmentSize)
        //                {
        //                    string fileName = string.Empty;
        //                    if (fileData.name.LastIndexOf("/") > -1)
        //                        fileName = fileData.name.Substring(fileData.name.LastIndexOf("/") + 1);
        //                    else
        //                        fileName = fileData.name;

        //                    string contentType;

        //                    //Zoundry1.04 fileData.type=null
        //                    if (string.IsNullOrEmpty(fileData.type))
        //                        contentType = MimeTypeConfiguration.GetMimeType(fileName);
        //                    else
        //                        contentType = fileData.type;

        //                    MemoryStream stream = new MemoryStream(fileData.bits);

        //                    PostAttachment pa = new PostAttachment(fileName, contentType, stream);
        //                    pa.FriendlyFileName = fileName;
        //                    pa.CreateDate = DateTime.Now;
        //                    pa.UserID = currentUser.UserID;
        //                    pa.PostID = 0;
        //                    pa.SectionID = weblog.SectionID;
        //                    pa.ApplicationType = ApplicationType.Blog;

        //                    PostAttachments.Add(pa, ApplicationType.Blog);
        //                    fs.SaveAttachmentToDisk(pa, pa.SectionID, pa.PostID);
        //                    pa.Content.Close();

        //                    MediaObjectInfo urlData = new MediaObjectInfo();
        //                    urlData.url = WebUtils.FullPath(BlogUrls.Instance().DownloadPostAttachment(weblog.ApplicationKey, pa.AttachmentID));
        //                    return urlData;

        //                }
        //                else
        //                {
        //                    throw new XmlRpcFaultException(0, "file too big");
        //                }
        //            }
        //        }
        //        catch
        //        {
        //        }

        //        throw new XmlRpcFaultException(0, "Invalid file");
        //    }

        //    throw new XmlRpcFaultException(0, "User does not exist");
        //}

        //不支持
        //public UrlData newMediaObject(string blogid, string username, string password, FileData fileData)
        //{
        //    if (ValidateUser(username, password))
        //    {
        //        BlogSection weblog = BlogSections.GetBlogSection(int.Parse(blogid), true, true);
        //        Permissions.AccessCheck(weblog, Permission.Post, SBContext.Current.User);

        //        //if (!BlogSectionConfiguration.Instance().AttachmentSettings.EnableFileSystemStorage)
        //        //    throw new XmlRpcFaultException(0, "Blog file storage not allowed");

        //        try
        //        {
        //            if (fileData.bits != null && !ValueHelper.IsNullOrEmpty(fileData.name))
        //            {
        //                AttachmentSettings fs = BlogSectionConfiguration.Instance().AttachmentSettings;
        //                if ((fileData.bits.Length / 1024) < fs.MaxAttachmentSize)
        //                {
        //                    PostAttachment pa = new PostAttachment();
        //                    MemoryStream stream = new MemoryStream(fileData.bits);
        //                    pa.Content = stream;
        //                    pa.Length = fileData.bits.Length;
        //                    pa.ContentType = fileData.type;
        //                    pa.FileName = fileData.name;
        //                    pa.FriendlyFileName = fileData.name;

        //                    pa.CreateDate = DateTime.Now;
        //                    pa.UserID = currentUser.UserID;
        //                    pa.PostID = 0;
        //                    pa.SectionID = weblog.SectionID;
        //                    pa.ApplicationType = ApplicationType.Blog;

        //                    PostAttachments.Add(pa, ApplicationType.Blog);
        //                    fs.SaveAttachmentToDisk(pa, pa.SectionID, pa.PostID);

        //                    UrlData urlData = new UrlData();
        //                    urlData.url = WebUtils.FullPath(BlogUrls.Instance().DownloadPostAttachment(weblog.ApplicationKey, pa.AttachmentID));
        //                    return urlData;

        //                }
        //                else
        //                {
        //                    throw new XmlRpcFaultException(0, "file too big");
        //                }
        //            }
        //        }
        //        catch
        //        {
        //        }

        //        throw new XmlRpcFaultException(0, "Invalid file");
        //    }

        //    throw new XmlRpcFaultException(0, "User does not exist");
        //}


        #region LiveWriter API
        public UserBlog[] getUsersBlogs(string appKey, string username, string password)
        {
            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser != null )
                {
                    BlogSection blogSection = BlogSections.GetSection(currentUser.UserID, false);
                    if (blogSection != null)
                    {
                        UserBlog ub = new UserBlog();
                        ub.blogid = currentUser.UserID.ToString();
                        ub.blogName = blogSection.SectionName;
                        ub.url = WebUtils.FullPath(SiteUrls.Instance().UserChannelHome(blogSection.Owner, ApplicationIDs.Instance().Blog()));
                        return new UserBlog[] { ub };
                    }
                }
            }
            throw new XmlRpcFaultException(0, "User does not exist");
        }

        public UserInfo getUserInfo(string appKey, string username, string password)
        {
            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser != null)
                {
                    BlogSection blogSection = BlogSections.GetSection(currentUser.UserID, false);
                    UserInfo ui = new UserInfo();

                    if (blogSection != null)
                    {
                        ui.blogid = blogSection.OwnerUserID.ToString();
                        ui.blogName = blogSection.SectionName;
                    }

                    ui.firstname = currentUser.CommonName;
                    ui.lastname = "";

                    if (string.IsNullOrEmpty(currentUser.PublicEmail))
                        ui.email = currentUser.PublicEmail;
                    else
                        ui.email = "";

                    if (string.IsNullOrEmpty(currentUser.NickName))
                        ui.nickname = currentUser.NickName;
                    else
                        ui.nickname = "";

                    ui.url = WebUtils.FullPath(SiteUrls.Instance().UserDomainHome(currentUser));

                    return ui;
                }
            }
            throw new XmlRpcFaultException(0, "User does not exist");
        }

        public bool deletePost(string appKey, string postid, string username, string password, [XmlRpcParameter(Description = "Where applicable, this specifies whether the blogSection should be republished after the post has been deleted.")] bool publish)
        {
            if (ValidateUser(username, password))
            {
                User currentUser = Users.GetUser(username, false);
                if (currentUser == null)
                    throw new XmlRpcFaultException(0, "User does not exist");
                int threadID = 0;
                int.TryParse(postid, out threadID);

                BlogThread blogThread = BlogThreads.GetThread(threadID, null, false);

                if (blogThread == null)
                    throw new XmlRpcFaultException(0, "blogSection post does not exist");
                else
                {
                    BlogSection blogSection = BlogSections.GetSection(blogThread.OwnerUserID, false);

                    //检查当前用户是否具有发布文章权限
                    PostPermissionCheck(blogSection, currentUser);

                    BlogThreads.DeleteThread(blogThread.ThreadID);
                    return true;
                }
            }

            throw new XmlRpcFaultException(0, "User does not exist");
        }


        #endregion

        #endregion


        #region Private Helper Methods

        /// <summary>
        /// 验证用户身份
        /// </summary>
        protected bool ValidateUser(string userName, string password)
        {
            try
            {
                LoginUserStatuses status = Users.ValidUser(userName, password);
                if (status == LoginUserStatuses.Success)
                {
                    return true;
                }

            }
            catch { }
            return false;
        }

        /// <summary>
        /// 当前用户是否具有在blogSection中发帖的权限
        /// </summary>
        protected bool PostPermissionCheck(BlogSection blogSection, User currentUser)
        {
            if (blogSection == null || currentUser == null )
            {
                throw new XmlRpcFaultException(0, "User no permission");
            }

            if (currentUser.IsContentAdministrator || currentUser.UserID == blogSection.OwnerUserID)
                return true;
            else
                throw new XmlRpcFaultException(0, "User no permission");
        }

        protected SpaceBuilder.Blog.MetaWeblogAPI.Post ConvertToPost(BlogThread blogThread)
        {
            SpaceBuilder.Blog.MetaWeblogAPI.Post p = new SpaceBuilder.Blog.MetaWeblogAPI.Post();
            if (blogThread.Tags != null && blogThread.Tags.Count > 0)
            {
                string[] tags = new string[blogThread.Tags.Count];
                for (int i = 0; i < blogThread.Tags.Count; i++)
                {
                    tags[i] = blogThread.Tags[i];
                }
                p.categories = tags;
            }
            else
            {
                p.categories = null;
            }

            p.dateCreated = blogThread.PostDate;
            p.description = blogThread.GetBody(true);
            p.link = SiteUrls.Instance().ShowBlogThread(UserID2UserNameDictionary.GetUserName(blogThread.OwnerUserID), blogThread);
            p.permalink = p.link;
            p.postid = blogThread.ThreadID.ToString();
            p.title = blogThread.Subject;
            return p;
        }

        private static List<UserTag> SafeTags(User currentUser, string[] existingTags, string keywords)
        {
            NameValueCollection nvc = new NameValueCollection();
            if (existingTags != null && existingTags.Length > 0)
                foreach (string s in existingTags)
                    nvc[s.Trim()] = s.Trim();

            if (!string.IsNullOrEmpty(keywords))
                foreach (string s in keywords.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries))
                    nvc[s.Trim()] = s.Trim();

            List<UserTag> tags = new List<UserTag>();
            if (nvc.AllKeys.Length > 0)
            {
                ICollection<UserTag> blogTags = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).GetUserTags(currentUser.UserID, false);
                foreach (string category in nvc.AllKeys)
                {
                    bool exist = false;
                    foreach (UserTag userTag in blogTags)
                    {
                        if (category.Equals(userTag.TagName, StringComparison.CurrentCultureIgnoreCase))
                        {
                            tags.Add(userTag);
                            exist = true;
                            break;
                        }
                    }

                    if (!exist)
                    {
                        UserTag userTag = new UserTag();
                        userTag.UserID = currentUser.UserID;
                        userTag.TagName = category;

                        bool isCreated = UserTagManager.Instance(UserTagItemKeys.Instance().BlogThread()).CreateUserTag(userTag);
                        if (isCreated)
                            tags.Add(userTag);
                    }
                }
            }

            return tags;
        }

        #endregion

    }
}
