﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;

namespace SpaceBuilder.Blog
{
    /// <summary>
    /// Blog相关Url
    /// </summary>
    public static class SiteUrlsExtensions
    {

        #region Blog

        /// <summary>
        /// 返回引用跟踪检测Url
        /// </summary>
        /// <param name="postID">博客文章的PostID</param>
        public static string BlogTrackbackPingUrl(this SiteUrls siteUrls, int postID)
        {
            return WebUtils.ResolveUrl(string.Format("~/Services/Blog/TrackBack.ashx?PostID={0}", postID));
        }

        /// <summary>
        /// 下载或查看blog附件
        /// </summary>
        public static string DownloadBlogAttachment(this SiteUrls siteUrls, int attachmentID)
        {
            //如果启用URL直连 则直接使用直连URL
            if (FileStorageProvider.Instance(BlogAttachmentManager.Instance().StorageProviderName).EnableDirectlyUrl)
            {
                Attachment attachment = BlogAttachmentManager.Instance().GetAttachment(attachmentID, false);
                if (attachment == null)
                    return string.Empty;

                if (attachment.IsImage)
                    return BlogAttachmentManager.Instance().GetDirectlyUrl(attachment, BlogPhotoTypes.Original);
            }

            return Globals.GetFullUrlInFileHandlerSite(BlogAttachmentManager.Instance().StorageProviderName, string.Format("~/Services/BlogAttachment.ashx?AttachmentID={0}", attachmentID));
        }
        /// <summary>
        /// 删除博客附件Url
        /// </summary>
        public static string DeleteBlogAttachmentUrl(this SiteUrls siteUrls, int attachmentID, bool? isImage)
        {
            return SPBUrlHelper.RouteUrl("Common", new RouteValueDictionary { { "controller", "ChannelBlog" }, { "action", "DeleteBlogAttachment" }, { "attachmentID", attachmentID }, { "isImage", isImage } });
        }

        /// <summary>
        /// 根据AttachmentID获取标题图片Url
        /// </summary>
        public static string BlogPhotoUrl(this SiteUrls siteUrls, int attachmentID, BlogPhotoTypes photoType)
        {
            //如果启用URL直连 则直接使用直连URL
            if (FileStorageProvider.Instance(BlogAttachmentManager.Instance().StorageProviderName).EnableDirectlyUrl)
            {
                Attachment attachment = BlogAttachmentManager.Instance().GetAttachment(attachmentID, false);
                if (attachment == null)
                    return string.Empty;

                //默认仅预生成标题图
                if (photoType == BlogPhotoTypes.Original || photoType == BlogPhotoTypes.Title)
                    return BlogAttachmentManager.Instance().GetDirectlyUrl(attachment, photoType);
            }
            return Globals.GetFullUrlInFileHandlerSite(BlogAttachmentManager.Instance().StorageProviderName, string.Format("~/Services/Blog/BlogPhoto.ashx?AttachmentID={0}&PhotoType={1}", attachmentID, photoType));
        }

        #region service

        /// <summary>
        /// 返回RSS订阅某博客所有评论的Url
        /// </summary>
        /// <param name="applicationKey">博客功能模块标识</param>
        public static string BlogCommentRss(this SiteUrls siteUrls, string userDomainName)
        {
            return WebUtils.ResolveUrl(string.Format("~/Services/blog/CommentRss.ashx?UserDomainName={0}",WebUtils.UrlEncode(userDomainName)));
        }

        /// <summary>
        /// 返回RSS订阅某博客文章的Url
        /// </summary>
        /// <param name="applicationKey">博客功能模块标识</param>
        public static string BlogRss(this SiteUrls siteUrls, string userDomainName)
        {
            return WebUtils.ResolveUrl(string.Format("~/Services/blog/rss.ashx?UserDomainName={0}", WebUtils.UrlEncode(userDomainName)));
        }

        /// <summary>
        /// 返回客户端博客编辑工具链接的Url
        /// </summary>
        /// <returns>客户端博客编辑工具链接的Url</returns>
        public static string MetaBlog(this SiteUrls siteUrls)
        {
            return WebUtils.ResolveUrl("~/Services/Blog/metablog.ashx");
        }

        /// <summary>
        /// 获取返回客户端博客编辑器的链接地址协议等信息的Url
        /// </summary>
        /// <param name="ownerUserID">博客拥有者UserID</param>
        /// <returns>返回客户端博客编辑器的链接地址协议等信息的Url</returns>
        public static string Blog_RSD(this SiteUrls siteUrls, int ownerUserID)
        {
            return WebUtils.ResolveUrl(string.Format("~/Services/blog/rsd.ashx?OwnerUserID={0}", ownerUserID));
        }

        /// <summary>
        /// 获取返回客户端博客编辑设置的Url
        /// </summary>
        /// <param name="ownerUserID">博客拥有者UserID</param>
        /// <returns>返回客户端博客编辑设置的Url</returns>
        public static string Blog_WLWManifest(this SiteUrls siteUrls, int ownerUserID)
        {
            return WebUtils.ResolveUrl(string.Format("~/Services/blog/wlwmanifest.ashx?OwnerUserID={0}", ownerUserID));
        }

        #endregion


        #endregion


        #region 重复配置的URL

        /// <summary>
        /// 博客频道首页
        /// </summary>
        public static string ChannelBlogHome(this SiteUrls siteUrls)
        {
            return SPBUrlHelper.RouteUrl("Channel_Blog_Home");
        }

        /// <summary>
        /// 根据站点标签列表显示文章
        /// </summary>
        public static string ListBlogThreadsBySiteTag(this SiteUrls siteUrls, string tagName)
        {
            return SPBUrlHelper.RouteUrl("Channel_Blog_ListBlogThreadsByTag", new RouteValueDictionary { { "tagName", WebUtils.UrlEncode(tagName) } });
        }
        /// <summary>
        /// 获取博客文章显示的URL
        /// </summary>
        /// <returns></returns>
        public static string ShowBlogThread(this SiteUrls siteUrls, string userDomainName, BlogThread blogThread)
        {
            return SPBUrlHelper.RouteUrl("UserDomain_Blog_ShowThread", new RouteValueDictionary() { { "UserDomainName", userDomainName }, { "threadID", blogThread.ThreadID } });
        }

        /// <summary>
        /// 获取博客评论分页列表Url
        /// </summary>
        /// <param name="siteUrls"></param>
        /// <param name="blogThread"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public static string ShowBlogCommentsPaged(this SiteUrls siteUrls, BlogThread blogThread, int pageIndex)
        {
            return SPBUrlHelper.RouteUrl("UserDomain_Blog_ShowThread", new RouteValueDictionary() { { "UserDomainName", blogThread.Owner.UserName }, { "threadID", blogThread.ThreadID }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 获取博客评论分页列表指定位置Url
        /// </summary>
        /// <param name="siteUrls"></param>
        /// <param name="blogThread"></param>
        /// <param name="pageIndex"></param>
        /// <param name="anchorPostID"></param>
        /// <returns></returns>
        public static string ShowBlogCommentsPaged(this SiteUrls siteUrls, BlogThread blogThread, int pageIndex, int anchorPostID)
        {
            string strPpageIndex = string.Format("{0}#{1}", pageIndex, anchorPostID);
            return SPBUrlHelper.RouteUrl("UserDomain_Blog_ShowThread", new RouteValueDictionary() { { "UserDomainName", blogThread.Owner.UserName }, { "threadID", blogThread.ThreadID }, { "pageIndex", strPpageIndex } });
        }

        /// <summary>
        /// 显示博客主题和评论并定位到最后一页评论
        /// </summary>
        public static string ShowBlogThreadGotoLastPost(this SiteUrls siteUrls, int threadID)
        {
            return WebUtils.ResolveUrl(String.Format("~/Services/Blog/BlogCommentUrl.ashx?ThreadID={0}", threadID));
        }


        /// <summary>
        /// 显示博客主题和评论并定位到最后一条评论
        /// </summary>
        public static string ShowBlogThreadGotoLastPost(this SiteUrls siteUrls, int threadID, int anchorPostID)
        {
            return WebUtils.ResolveUrl(String.Format("~/Services/Blog/BlogCommentUrl.ashx?ThreadID={0}&anchorPostID={1}", threadID, anchorPostID));
        }

        /// <summary>
        /// 根据用户标签列表显示文章
        /// </summary>
        public static string ListBlogThreadsByUserTag(this SiteUrls siteUrls, string userDomainName, string tagName)
        {
            return SPBUrlHelper.RouteUrl("UserDomain_Blog_Newthread", new RouteValueDictionary { { "UserDomainName", userDomainName }, { "action", "ListBlogThreadsByUserTag" }, { "tagName", WebUtils.UrlEncode(tagName) } });
        }

        /// <summary>
        /// 获取博客文章显示的URL
        /// </summary>
        /// <returns></returns>
        public static string ShowBlogThread(this SiteUrls siteUrls, string userDomainName, BlogThread blogThread, int childPostID)
        {
            return ShowBlogThread(siteUrls, userDomainName, blogThread) + "#" + childPostID;
        }

        /// <summary>
        /// 获取博客文章显示的URL
        /// </summary>
        /// <returns></returns>
        public static string ShowBlogThread(this SiteUrls siteUrls, string userDomainName, BlogThread blogThread, int pageIndex, int childPostID)
        {
            return ShowBlogThread(siteUrls, userDomainName, blogThread) + "?pageIndex=" + pageIndex + "#" + childPostID;
        }

        /// <summary>
        /// 显示博客,并定位评论所在页数
        /// </summary>
        public static string ShowBlogThreadGotoPageIndex(this SiteUrls siteUrls, string userDomainName, int threadID, int parentID, int anchorPostID)
        {
            return WebUtils.ResolveUrl(String.Format("~/Services/Blog/BlogUrl.ashx?UserDomainName={0}&ThreadID={1}&anchorPostID={2}&parentID={3}", WebUtils.UrlEncode(userDomainName), threadID, anchorPostID, parentID));
        }

        /// <summary>
        /// 创建文章
        /// </summary>
        public static string NewBlogThread(this SiteUrls siteUrls, string userDomainName)
        {
            return SPBUrlHelper.RouteUrl("UserDomain_Blog_Newthread", new RouteValueDictionary { { "UserDomainName", userDomainName }, { "action", "NewBlogThread" } });
        }

        /// <summary>
        /// 管理文章
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public static string ManageBlogThreads(this SiteUrls siteUrls, string userDomainName)
        {
            return SPBUrlHelper.RouteUrl("UserDomain_Blog_Newthread", new RouteValueDictionary { { "UserDomainName", userDomainName }, { "action", "ListBlogThreads" } });
        }

        #endregion

    }

}
