﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Web;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;
using System.Web.Routing;
using System.Text.RegularExpressions;
using SpaceBuilder.Forum;

namespace SpaceBuilder.Club.Controllers
{
    [HttpCompress]
    public class ChannelClubController : ChannelControllerBase
    {
        int clubApplicationID = ApplicationIDs.Instance().Club();

        /// <summary>
        /// 总站中群组首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_Club", clubApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Club", clubApplicationID), string.Empty);
            return View("pages/Clubs/Home.aspx", "Masters/Club.master");
        }

        #region Club

        /// <summary>
        /// 群组排行
        /// </summary>
        /// <returns></returns>
        public ActionResult ClubRanks(ClubsSortBy? sortBy, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_ClubRank", clubApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ClubRank", clubApplicationID), string.Empty);
            PagingDataSet<Club> pds = Clubs.GetPagingClubsSortBy(sortBy ?? ClubsSortBy.CreateDate, 7, pageIndex ?? 1);
            ViewData["sortBy"] = sortBy ?? ClubsSortBy.CreateDate;
            return View("Pages/Clubs/ClubRanks.aspx", "Masters/Club.master", pds);
        }

        /// <summary>
        /// 同城群组
        /// </summary>
        /// <returns></returns>
        public ActionResult ListClubsByArea(ClubsSortBy? sortBy, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_SameCityClub", clubApplicationID));
            string areaCode = Request.QueryString.GetString("areaCode", string.Empty);
            List<ClubAreaStatistics> ClubAreaStatisticss = null;
            string rootAreaCode = ClubDomainConfiguration.Instance().RootAreaCode;
            ClubAreaStatistics rootAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(rootAreaCode);
            if (!string.IsNullOrEmpty(areaCode))
            {
                int maxDepth = rootAreaStatistics.Depth + ClubDomainConfiguration.Instance().AreaLevel;

                ClubAreaStatistics currentAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(areaCode);
                if (currentAreaStatistics != null)
                {
                    if (currentAreaStatistics.Depth == maxDepth)
                    {
                        ClubAreaStatistics parentAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(currentAreaStatistics.ParentCode);
                        if (parentAreaStatistics != null)
                        {
                            ViewData["ParentAreaStatistics"] = parentAreaStatistics;
                            ClubAreaStatisticss = parentAreaStatistics.Childs;
                        }
                        else
                            ClubAreaStatisticss = ClubAreaStatisticses.GetRootClubAreaStatisticses();
                    }
                    else if (currentAreaStatistics.Depth < maxDepth)
                    {
                        ViewData["ParentAreaStatistics"] = currentAreaStatistics;
                        ClubAreaStatisticss = currentAreaStatistics.Childs;
                    }
                }
            }
            else
            {
                if (rootAreaStatistics != null)
                {
                    ViewData["ParentAreaStatistics"] = rootAreaStatistics;
                    ClubAreaStatisticss = rootAreaStatistics.Childs;
                }
                else
                    ClubAreaStatisticss = ClubAreaStatisticses.GetRootClubAreaStatisticses();
            }
            ViewData["ClubsAreaStatisticss"] = ClubAreaStatisticss;

            #region 处理地区面包屑
            if (!string.IsNullOrEmpty(areaCode))
            {
                List<Area> parentAreas = new List<Area>();
                Area area = Areas.GetArea(areaCode, true);
                Stack<Area> stack = new Stack<Area>();
                while (area != null && area.AreaCode != rootAreaCode)
                {
                    stack.Push(area);
                    area = area.Parent;
                }
                BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Club", clubApplicationID), SiteUrls.Instance().ChannelClubHome());
                BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_AreaSkim", clubApplicationID), SPBUrlHelper.Action("ListClubsByArea", "ChannelClub"));

                while (stack.Count > 0)
                {
                    area = stack.Pop();
                    BreadCrumb.AddBreadCrumbItem(area.GetMultilingualAreaName(CurrentUserLanguage), SiteUrls.Instance().ListClubsByArea(area.AreaCode));
                }
            }

            #endregion

            ViewData["sortBy"] = sortBy ?? ClubsSortBy.CreateDate;
            return View("Pages/Clubs/ListClubsByArea.aspx", "Masters/Club.master");
        }

        /// <summary>
        /// 同类群组
        /// </summary>
        /// <param name="categoryID"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult ListClubsByCategory(ClubsSortBy? sortBy, int categoryID, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_SameCategory", clubApplicationID));
            if (categoryID > 0)
            {
                SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Club()).Get(categoryID, false);

                BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Club", clubApplicationID), SiteUrls.Instance().ChannelClubHome());
                if (category.ParentID > 0)
                {
                    SiteCategory parentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Club()).Get(category.ParentID, false);
                    BreadCrumb.AddBreadCrumbItem(parentCategory.CategoryName, SiteUrls.Instance().ListClubsByCategory(parentCategory.CategoryID));
                }

                BreadCrumb.AddBreadCrumbItem(category.CategoryName, string.Empty);
                ViewData["category"] = category;
                ViewData["ControlTitle"] = category.CategoryName;
            }
            else
                ViewData["category"] = null;

            List<SiteCategory> rootSiteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetRoots(false);
            ViewData["rootSiteCategories"] = rootSiteCategories;
            ViewData["sortBy"] = sortBy ?? ClubsSortBy.CreateDate;
            return View("Pages/Clubs/ListClubsByCategory.aspx", "Masters/Club.master");
        }

        /// <summary>
        /// 群组列表显示
        /// </summary>
        /// <param name="categoryID"></param>
        /// <param name="areaCode"></param>
        /// <returns></returns>
        public ActionResult Control_ListClubs(ClubsSortBy? sortBy, int? categoryID, string areaCode, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(areaCode))
            {
                ViewData["isCategory"] = false;
            }
            else if (categoryID.HasValue && categoryID.Value > 0)
            {
                ViewData["isCategory"] = true;
            }
            else if (categoryID.HasValue)
                ViewData["isCategory"] = true;
            PagingDataSet<Club> pds = Clubs.GetPagingClubsSortBy(sortBy ?? ClubsSortBy.CreateDate, categoryID, areaCode, 10, pageIndex.HasValue ? pageIndex.Value : 1);
            ViewData["sortBy"] = sortBy ?? ClubsSortBy.CreateDate;
            return PartialView("Controls/Clubs/ListClubs.ascx", pds);
        }

        /// <summary>
        /// 显示群组模块
        /// </summary>
        public ActionResult Control_ListClubsBlock(ClubsSortBy? sortBy, int pageSize, string moduleTitle, string areaCode)
        {
            IList<Club> list = Clubs.GetTopClubsSortBy(sortBy ?? ClubsSortBy.CreateDate, pageSize);
            ViewData["sortBy"] = sortBy;
            ViewData["moduleTitle"] = moduleTitle;
            ViewData["areaCode"] = areaCode;
            return PartialView("Controls/Clubs/ListClubsBlock.ascx", list);
        }

        /// <summary>
        /// 同城群组
        /// </summary>
        public ActionResult Control_ListClubsBySameArea(ClubsSortBy? sortBy, int pageSize, string areaCode)
        {
            IList<Club> list = Clubs.GetPagingClubsSortBy(sortBy ?? ClubsSortBy.CreateDate, null, areaCode, pageSize, 1).Records;
            ViewData["sortBy"] = sortBy;
            ViewData["areaCode"] = areaCode;
            return PartialView("Controls/Clubs/ListClubsBySameArea.ascx", list);
        }

        /// <summary>
        /// 同城群组
        /// </summary>
        public ActionResult Control_ListNewClubs(int pageSize)
        {
            IList<Club> list = Clubs.GetPagingClubsSortBy(ClubsSortBy.CreateDate, null, string.Empty, pageSize, 1).Records;

            return PartialView("Controls/Clubs/ListNewClubs.ascx", list);
        }

        /// <summary>
        /// 总站首页显示的群组列表模块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_ListClubsHomeBlock(ClubsSortBy? sortBy, int? displayCount)
        {
            IList<Club> clubs = Clubs.GetTopClubsSortBy(sortBy ?? ClubsSortBy.CreateDate, displayCount ?? 6);
            return PartialView("Controls/Clubs/ListClubsHomeBlock.ascx", clubs);
        }

        /// <summary>
        /// 新建群组
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ChannelGuideInstallApplication(ApplicationID = 231)]
        public ActionResult NewClub()
        {
            if (CurrentUser == null)
            {
                Response.Redirect(SiteUrls.Instance().Login(true));
            }
            if (!ClubPermission.AccessCheck(null, CurrentUser, PermissionItemKeys.Instance().CreateClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            SetPageTitle(GetResourceString("Title_CreateClub", clubApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_CreateClub", clubApplicationID), string.Empty);
            ClubDomainConfiguration clubDomainConfiguration = ClubDomainConfiguration.Instance();
            ViewData["SiteCategoryLevel"] = clubDomainConfiguration.SiteCategoryLevel;
            ViewData["AreaLevel"] = clubDomainConfiguration.AreaLevel;
            ViewData["RootAreaCode"] = clubDomainConfiguration.RootAreaCode;

            return View("Pages/Clubs/CreateClub.aspx", "Masters/Club.master");
        }

        /// <summary>
        /// 创建群组
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [ChannelGuideInstallApplication(ApplicationID = 231)]
        public ActionResult CreateClub()
        {
            Club newclub = new Club();
            if (!ClubPermission.AccessCheck(newclub, CurrentUser, PermissionItemKeys.Instance().CreateClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            newclub.IsEnabled = true;
            newclub.ClubOwnerUserID = CurrentUser.UserID;
            newclub.ClubName = Request.Form.GetString("clubName", string.Empty).Trim();
            newclub.DomainName = Request.Form.GetString("clubDomain", string.Empty).Trim();
            newclub.Brief = Request.Form.GetString("brief", string.Empty).Trim();
            newclub.CategoryID = Request.Form.GetInt("categoryID", 0);
            newclub.AreaCode = Request.Form.GetString("areaCode", string.Empty);
            newclub.IsPublic = Request.Form.Get<int>("clubSecrecyType", 0) > 0 ? true : false;
            newclub.JoinNotice = Request.Form.GetString("joinNotice", string.Empty).Trim();
            if (newclub.IsPublic)
            {
                newclub.IsMemberNeedAuthorize = Request.Form.Get<int>("needApproveCheckbox", 0) > 0 ? true : false;
            }
            else
            {
                newclub.IsMemberNeedAuthorize = false;
            }
            if (string.IsNullOrEmpty(newclub.ClubName))
            {
                TempData["StatusMessage"] = GetResourceString("Message_ClubNameNotNull", clubApplicationID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                return RedirectToAction("NewClub");
            }
            else if (string.IsNullOrEmpty(newclub.DomainName))
            {
                TempData["StatusMessage"] = GetResourceString("Message_ClubUrlNotNull", clubApplicationID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                return RedirectToAction("NewClub");
            }
            else if (Clubs.DomainNameIsExist(newclub.DomainName))
            {
                TempData["newclub"] = newclub;
                TempData["StatusMessage"] = GetResourceString("Message_ClubUrlNotExist", clubApplicationID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                return RedirectToAction("NewClub");
            }

            Clubs.CreateClub(CurrentUser.UserID, newclub);

            #region 创建版块组

            //创建默认板块组
            ForumSectionGroup group = new ForumSectionGroup();
            group.OwnerID = newclub.ClubID;
            group.GroupName = newclub.ClubName;
            group.PresentAreaID = PresentAreaIDs.Instance().ClubDomain();
            ForumSectionGroups.CreateGroup(group);

            //创建默认板块
            ForumSection forumSection = new ForumSection();
            forumSection.GroupID = group.GroupID;
            forumSection.SectionName = "默认版块";
            ForumSections.Create(forumSection, null);

            #endregion

            switch (newclub.AuditingStatus)
            {
                case AuditingStatuses.Again:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessage"] = GetResourceString("Message_ClubCreateSuccess", clubApplicationID);
                    break;
                case AuditingStatuses.Fail:

                    break;
                case AuditingStatuses.Pending:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessage"] = GetResourceString("Message_ClubCreateSuccessButNotDisplayInChannel", clubApplicationID);
                    break;
                case AuditingStatuses.Success:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessage"] = GetResourceString("Message_ClubCreateSuccessNowInAdminView", clubApplicationID);
                    break;
                default:
                    break;
            }

            return RedirectToAction("Home", "ClubManage", new RouteValueDictionary { { "clubDomainName", newclub.DomainName } });
        }

        #region 分享到群组

        public ActionResult ShareToMyClubs(int applicationID, string contentUrl)
        {

            ViewData["applicationID"] = applicationID;
            ViewData["contentSubject"] = Request.QueryString.Get<string>("contentSubject", string.Empty);
            ViewData["contentBody"] = Request.QueryString.Get<string>("contentBody", string.Empty);

            ViewData["contentImgUrl"] = Request.QueryString.Get<string>("contentImgUrl", string.Empty);
            ViewData["contentUrl"] = contentUrl;

            ViewData["contentAuthor"] = Request.QueryString.Get<string>("contentAuthor", string.Empty);
            ViewData["contentUserID"] = Request.QueryString.Get<int>("contentUserID", 0);

            ViewData["shareAuthor"] = CurrentUser.DisplayName;
            ViewData["shareAuthorID"] = CurrentUser.UserID;

            PagingDataSet<Club> currentUserClubs = null;
            List<int> commendClubIDs = null;
            User currentUser = Globals.GetCurrentUser();

            currentUserClubs = Clubs.GetClubs(currentUser.UserID, true, 1);
            commendClubIDs = ClubSharedContents.GetSharedClubIDs(contentUrl);

            ViewData["currentUserClubs"] = currentUserClubs;
            ViewData["commendClubIDs"] = commendClubIDs;
            return PartialView("Controls/Clubs/ShareToMyClubs.ascx");
        }

        public ActionResult ShareToMyClubsAction(int applicationID, string contentUrl, int shareAuthorID, int authorUserID)
        {
            ClubSharedContent shareContent = new ClubSharedContent();
            string shareClubs = Request.Form.Get<string>("shareToMyClubsControl", string.Empty);
            string contentSubject = Request.Form.Get<string>("contentSubject", string.Empty);
            string contentBody = Request.Form.Get<string>("contentBody", string.Empty);
            string contentAuthor = Request.Form.Get<string>("contentAuthor", string.Empty);
            //string contentUrl = Request.Form.Get<string>("contentUrl",string.Empty);
            string contentImgUrl = Request.Form.Get<string>("contentImgUrl", string.Empty);
            string shareAuthor = Request.Form.Get<string>("shareAuthor", string.Empty);

            shareContent.ApplicationID = applicationID;
            shareContent.Author = contentAuthor;
            shareContent.AuthorUserID = authorUserID;
            shareContent.Body = Formatter.FormatPlainTextComment(contentBody);
            shareContent.DetailsUrl = contentUrl;
            shareContent.Sharer = shareAuthor;
            shareContent.SharerUserID = shareAuthorID;
            shareContent.Subject = contentSubject;
            shareContent.ThumbnailUrl = contentImgUrl;

            if (!string.IsNullOrEmpty(shareClubs))
            {
                string[] selectedClubIDs = shareClubs.Split(',');
                List<int> lastSelectedClubIDs = new List<int>();
                if (selectedClubIDs != null && selectedClubIDs.Length > 0)
                {
                    foreach (string clubIDString in selectedClubIDs)
                    {
                        if (string.IsNullOrEmpty(clubIDString))
                            continue;

                        int id = 0;
                        int.TryParse(clubIDString, out id);
                        if (id > 0)
                            lastSelectedClubIDs.Add(id);
                    }

                    if (lastSelectedClubIDs == null || !(lastSelectedClubIDs.Count > 0))
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_PleaseChooseClub", clubApplicationID));
                    }
                    else
                    {
                        ClubSharedContents.ShareContents(shareContent, lastSelectedClubIDs);
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_ShareToClubSuccess", clubApplicationID));
                    }
                }
                else
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_PleaseChooseClub", clubApplicationID));

            }
            else
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_PleaseChooseClub", clubApplicationID));

            return RedirectToAction("ShareToMyClubs", "ChannelClub", new RouteValueDictionary { { "applicationID", applicationID }, { "contentUrl", contentUrl } });
        }

        #endregion

        /// <summary>
        /// 处理验证群组域名
        /// </summary>
        public JsonResult ValidateClubDomainName(string clubDomainName)
        {
            if (Clubs.DomainNameIsExist(clubDomainName))
            {
                return Json(new { isSuccess = false, messageContent = GetResourceString("Message_SorryClubUrlRegistered", clubApplicationID) }, JsonRequestBehavior.AllowGet);
            }
            else
            {
                Regex regex = new Regex(ClubDomainConfiguration.Instance().ClubDomainNameRegex);
                if (regex.IsMatch(clubDomainName))
                    return Json(new { isSuccess = true, messageContent = GetResourceString("Message_CongratulationsClunUrlNotRegistered", clubApplicationID) }, JsonRequestBehavior.AllowGet);
                else
                    return Json(new { isSuccess = false, messageContent = GetResourceString("Message_SorryClubUrlNotConform", clubApplicationID) }, JsonRequestBehavior.AllowGet);
            }
        }

        /// <summary>
        /// 群组站点类别列表
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.Stable)]
        public ActionResult Control_ListCategoriesBlock(int childCategoryCount)
        {
            List<SiteCategory> siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetRoots(false);
            ViewData["childCategoryCount"] = childCategoryCount;
            return PartialView("Controls/Clubs/ListCategoriesBlock.ascx", siteCategories);
        }

        /// <summary>
        /// 地区模块
        /// </summary>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.Stable)]
        public ActionResult Control_ListAreasBlock()
        {
            List<ClubAreaStatistics> areaList = ClubAreaStatisticses.GetChildClubAreaStatisticses(ClubDomainConfiguration.Instance().RootAreaCode, ClubAreaStatisticsSortBy.CumulateTotalCount, SortOrder.Descending);
            return View("Controls/Clubs/ListAreasBlock.ascx", areaList);
        }

        /// <summary>
        /// 群组类别
        /// </summary>
        /// <returns></returns>
        public ActionResult ClubCategories()
        {
            SetPageTitle(GetResourceString("Title_ClubCategory", clubApplicationID));
            List<SiteCategory> siteCategories = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetRoots(false);
            return View("Pages/Clubs/ClubCategories.aspx", "Masters/Club.master", siteCategories);
        }

        /// <summary>
        /// 显示推荐群组列表
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_RecommendClubBlock(int commendTypeID, int? displayCount)
        {
            PagingDataSet<RecommendedItem> recommendedClubItems = RecommendedItems.GetRecommendedItems<RecommendedItem>(displayCount ?? 4, 1, ApplicationIDs.Instance().Club(), commendTypeID);

            //List<Club> recommendedClubs = new List<Club>();
            //foreach (RecommendedItem recommendItem in recommendedClubItems.Records)
            //{
            //    Club recommendedClub = Clubs.GetClub(recommendItem.ItemID, false);
            //    if (recommendedClub != null)
            //        recommendedClubs.Add(recommendedClub);

            //}

            ViewData["recommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            ViewData["commendTypeID"] = commendTypeID;
           // ViewData["recommendedClubs"] = recommendedClubs;
            return PartialView("Controls/Clubs/RecommendClubBlock.ascx", recommendedClubItems);
        }

        /// <summary>
        /// 显示群组搜索结果
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_ClubSearch", clubApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ClubSearch", clubApplicationID), string.Empty);
            string keyword = Request.QueryString.GetString("keyword", string.Empty).Trim();
            int categoryID = Request.QueryString.GetInt("categoryID", 0);
            string areaCode = Request.QueryString.GetString("areaCode", string.Empty).Trim();
            PagingDataSet<Club> pds = Clubs.GetClubsForAdmin(20, pageIndex.HasValue ? pageIndex.Value : 1, AuditingStatusesForDisplay.Success, -1, keyword, categoryID, true, areaCode);

            #region Category

            Stack<SiteCategory> categoryStack = null;
            List<SiteCategory> siteCategories = GetChildCategories(categoryID, out categoryStack);
            ViewData["siteCategories"] = siteCategories;
            ViewData["categoryStack"] = categoryStack;

            if (categoryID > 0)
            {
                SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Club()).Get(categoryID, false);
                if (currentCategory != null)
                    ViewData["CurrentCategoryName"] = currentCategory.GetMultilingualCategoryName(CurrentUserLanguage);
            }

            #endregion

            #region Area

            Stack<ClubAreaStatistics> areaStatisticsStack = null;
            List<ClubAreaStatistics> areaStatisticss = GetChildAreas(areaCode, out areaStatisticsStack);
            ViewData["areas"] = areaStatisticss;
            ViewData["areaStack"] = areaStatisticsStack;

            if (!string.IsNullOrEmpty(areaCode))
            {
                Area area = Areas.GetArea(areaCode, false);
                if (area != null)
                    ViewData["CurrentAreaName"] = area.GetMultilingualAreaName(CurrentUserLanguage);
            }

            #endregion

            return this.View("Pages/Clubs/Search.aspx", "Masters/Club.master", pds);
        }

        /// 搜索群组
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["keyword"] = Request.Form.GetString("keyword", string.Empty).Trim();
            parameters["categoryID"] = Request.QueryString.GetInt("categoryID", 0);
            parameters["areaCode"] = Request.QueryString.GetString("areaCode", string.Empty);

            return RedirectToAction("Search", parameters);
        }

        /// 高级搜索群组
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult AdvancedSearch()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["keyword"] = Request.Form.GetString("keyword", string.Empty).Trim();
            parameters["categoryID"] = Request.Form.GetInt("categoryID", -1);
            parameters["areaCode"] = Request.Form.GetString("areaCode", string.Empty);
            return RedirectToAction("Search", parameters);
        }



        /// <summary>
        /// 返回某站点类别统计下的所有子类别统计列表,使用json格式传输
        /// </summary>
        /// <param name="parentID">父类别ID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public JsonResult GetClubSiteCategories_StatisticsNavigations(int? parentID)
        {
            List<SiteCategory> list = null;
            if (parentID.HasValue && parentID.Value > 0)
                list = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetAllChilds(parentID.Value, false);
            else
                list = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetRoots(false);
            List<StatisticsNavigation> navs = new List<StatisticsNavigation>();
            foreach (SiteCategory siteCategory in list)
            {
                StatisticsNavigation navigation = new StatisticsNavigation();
                navigation.ID = siteCategory.CategoryID.ToString();
                navigation.Name = siteCategory.GetMultilingualCategoryName(CurrentUserLanguage);
                navigation.Url = SiteUrls.Instance().ListClubsByCategory(siteCategory.CategoryID);
                navigation.Count = siteCategory.CumulateItemCount;
                navs.Add(navigation);
            }
            return Json(navs, JsonRequestBehavior.AllowGet);
        }

        public JsonResult GetSuggestClubsJosn(string contentUrl)
        {
            //群组名称过滤条件
            string clubNameFilter = Request.QueryString.GetString("q", string.Empty);

            if (string.IsNullOrEmpty(clubNameFilter))
                return Json(new { }, JsonRequestBehavior.AllowGet);

            IList<Club> clubs = null;

            clubs = Clubs.GetMyClubs(CurrentUser != null ? CurrentUser.UserID : 0, 40);
            List<int> commendClubIDs = ClubSharedContents.GetSharedClubIDs(contentUrl);

            if (clubs != null && clubs.Count > 0)
                clubs = clubs.Where(n => n.ClubName.Contains(clubNameFilter) && !commendClubIDs.Contains(n.ClubID)).ToList();

            if (clubs == null)
                clubs = new List<Club>();
            return Json(clubs.Select(n => new
            {
                clubID = n.ClubID,
                value = n.ClubName
            }), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 获取当前用户的群组列表
        /// </summary>
        /// <param name="categoryID"></param>
        /// <returns></returns>
        public JsonResult GetMyClubs()
        {
            IList<Club> clubs = Clubs.GetMyClubs(CurrentUser != null ? CurrentUser.UserID : 0, 40);

            return Json(clubs.Select(n => new
            {
                clubID = n.ClubID,
                value = n.ClubName
            }), JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region 群组地区选择
        /// <summary>
        /// 返回某群组地区统计下的所有子地区统计列表,使用json格式传输
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public JsonResult GetArea_StatisticsNavigations(string parentCode)
        {
            List<ClubAreaStatistics> list = null;
            if (!string.IsNullOrEmpty(parentCode))
            {
                ClubAreaStatistics parentArea = ClubAreaStatisticses.GetClubAreaStatistics(parentCode);
                list = parentArea.Childs;
            }
            else
            {
                ClubDomainConfiguration config = ClubDomainConfiguration.Instance();
                if (!string.IsNullOrEmpty(config.RootAreaCode))
                {
                    ClubAreaStatistics rootArea = ClubAreaStatisticses.GetClubAreaStatistics(config.RootAreaCode);
                    list = rootArea.Childs;
                }
                else
                    list = ClubAreaStatisticses.GetRootClubAreaStatisticses();
            }
            if (list == null)
                list = new List<ClubAreaStatistics>();
            List<StatisticsNavigation> navs = new List<StatisticsNavigation>();
            foreach (ClubAreaStatistics statistics in list)
            {
                StatisticsNavigation navigation = new StatisticsNavigation();
                navigation.ID = statistics.AreaCode.ToString();
                navigation.Name = statistics.GetMultilingualAreaName(CurrentUserLanguage);
                navigation.Url = SiteUrls.Instance().ListClubsByArea(statistics.AreaCode);
                navigation.Count = statistics.CumulateTotalCount;
                navs.Add(navigation);
            }
            return Json(navs, JsonRequestBehavior.AllowGet);
        }
        #endregion

        #region Help

        /// <summary>
        /// 获取当前类别的子类别，如果是最后一级，则获取同辈类别
        /// </summary>
        /// <param name="siteCategoryID">当前类别</param>
        private List<SiteCategory> GetChildCategories(int? currentCagoryID, out Stack<SiteCategory> categoryStack)
        {
            categoryStack = new Stack<SiteCategory>();
            SiteCategory currentCategory = null;
            if (currentCagoryID.HasValue && currentCagoryID.Value > 0)
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Club()).Get(currentCagoryID.Value, false);

            List<SiteCategory> siteCategoties = null;

            if (currentCategory == null)
                siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetRoots(false);
            else if (currentCategory.ChildCount > 0)
                siteCategoties = currentCategory.Childs;
            else
            {
                if (currentCategory.ParentID <= 0)
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Club()).GetRoots(false);
                else
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Club()).Get(currentCategory.ParentID, false).Childs;
            }

            while (currentCategory != null)
            {
                if (currentCategory.ChildCount > 0)
                    categoryStack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Club()).Get(currentCategory.ParentID, false);
            }

            return siteCategoties;
        }

        /// <summary>
        /// 获取当前地区的子地区，如果是最后一级，则获取同辈地区
        /// </summary>
        /// <param name="siteCategoryID">当前类别</param>
        private List<ClubAreaStatistics> GetChildAreas(string currentAreaCode, out Stack<ClubAreaStatistics> clubAreaStatisticsStack)
        {
            List<ClubAreaStatistics> clubAreaStatistics = null;
            clubAreaStatisticsStack = new Stack<ClubAreaStatistics>();

            string rootAreaCode = ClubDomainConfiguration.Instance().RootAreaCode;
            ClubAreaStatistics rootAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(rootAreaCode);
            if (!string.IsNullOrEmpty(currentAreaCode))
            {
                int maxDepth = ClubDomainConfiguration.Instance().AreaLevel;
                if (rootAreaStatistics != null)
                    maxDepth += rootAreaStatistics.Depth;

                ClubAreaStatistics currentAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(currentAreaCode);
                if (currentAreaStatistics != null)
                {
                    if (currentAreaStatistics.Depth == maxDepth)
                    {
                        ClubAreaStatistics parentAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(currentAreaStatistics.ParentCode);
                        if (parentAreaStatistics != null)
                            clubAreaStatistics = parentAreaStatistics.Childs;
                        else
                            clubAreaStatistics = ClubAreaStatisticses.GetRootClubAreaStatisticses();

                    }
                    else if (currentAreaStatistics.Depth < maxDepth)
                    {
                        clubAreaStatistics = currentAreaStatistics.Childs;
                    }
                }
                else
                    clubAreaStatistics = ClubAreaStatisticses.GetRootClubAreaStatisticses();

                while (currentAreaStatistics != null)
                {
                    if (currentAreaStatistics.ChildCount > 0)
                        clubAreaStatisticsStack.Push(currentAreaStatistics);
                    currentAreaStatistics = ClubAreaStatisticses.GetClubAreaStatistics(currentAreaStatistics.ParentCode);
                }
            }
            else
            {
                if (rootAreaStatistics != null)
                    clubAreaStatistics = rootAreaStatistics.Childs;
                else
                    clubAreaStatistics = ClubAreaStatisticses.GetRootClubAreaStatisticses();
            }

            return clubAreaStatistics;
        }

        #endregion
    }

    /// <summary>
    /// 总站中群组二级菜单
    /// </summary>
    public enum ChannelClubSubMenu
    {
        /// <summary>
        /// 群组站点首页
        /// </summary>
        Home,

        /// <summary>
        /// 群组排行
        /// </summary>
        Rank,

        /// <summary>
        /// 群组搜索
        /// </summary>
        Search,

        /// <summary>
        /// 我的群组
        /// </summary>
        MyClub,
    }
}
