﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;

using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Common;
using System.Collections;
using SpaceBuilder.Web;
using System.Text.RegularExpressions;
using SpaceBuilder.Utils;
using System.Web.Routing;
using System.Web.UI;
using SpaceBuilder.Common;
using System.Net.Mail;
namespace SpaceBuilder.Club.Controllers
{
    [HttpCompress]
    public class ClubDomainController : ClubDomainControllerBase
    {
        int clubApplicationID = ApplicationIDs.Instance().Club();

        /// <summary>
        /// 群组首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ClubMemberVisitAndClubDomainHitTimeAttribute]
        public ActionResult Home(string clubDomainName)
        {
            SetPageTitle(GetResourceString("Title_ClubHome", clubApplicationID), clubDomainName);

            Club currentClub = GetCurrentClub(clubDomainName);
            try
            {
                ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true));
            }
            catch
            {
                return Redirect(SiteUrls.Instance().Error404("没有浏览当前群组的权限！"));
            }

            #region 设置meta标签
            SetPageMeta(currentClub);
            #endregion

            return View("Pages/Home.aspx", currentClub);
        }

        #region Club

        /// <summary>
        /// 申请加入群组
        /// </summary>
        public ActionResult ApplyClubMember(string clubDomainName, int? inviteUserID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (CurrentUser == null)
            {
                Response.Redirect(SiteUrls.Instance().Login(true));
            }

            if (ClubMembers.IsClubMember(currentClub.ClubID, CurrentUser.UserID))
            {
                ViewData["isAllowApply"] = false;
            }

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)) && inviteUserID <= 0)
                return new EmptyResult();

            SetPageTitle(GetResourceString("Title_ApplyClubMember", clubApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ApplyClubMember", ApplicationIDs.Instance().Club()), string.Empty);

            ViewData["inviteUserID"] = inviteUserID;

            return View("Pages/ApplyClubMember.aspx", "Masters/Club.master", currentClub);
        }

        /// <summary>
        /// 生成群组成员
        /// </summary>
        public ActionResult ApplyClubMemberManage(string clubDomainName, int inviteUserID)
        {
            Club club = GetCurrentClub(clubDomainName);
            if (club == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", clubApplicationID), club.ClubID)));
            }
            if (CurrentUser == null)
            {
                Response.Redirect(SiteUrls.Instance().Login(true));
            }
            else if (ClubMembers.IsClubMember(club.ClubID, CurrentUser.UserID))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageSuccessMessage"] = GetResourceString("Message_SorryApplyFail", clubApplicationID);
                return View("Pages/ApplyClubMember.aspx", "Masters/Club.master", club);
            }

            if (!ClubPermission.AccessCheck(club, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)) && inviteUserID <= 0)
                return new EmptyResult();

            ClubMember clubMember = new ClubMember();
            clubMember.ClubID = club.ClubID;
            clubMember.UserID = CurrentUser.UserID;
            clubMember.InviteUserID = inviteUserID;
            clubMember.DisplayName = CurrentUser.DisplayName;

            User inviteUser = Users.GetUser(inviteUserID, false);
            if (inviteUser != null)
                clubMember.InviteUserDisplayName = inviteUser.DisplayName;
            else
                clubMember.InviteUserDisplayName = string.Empty;

            if (inviteUser != null)
            {
                clubMember.Status = ClubMemberStatuses.IsApproved;
                TempData["StatusMessageSuccessMessage"] = GetResourceString("Message_YouAlreadyBecomeMember", clubApplicationID);
            }
            else if (club.IsPublic)
            {
                if (club.IsMemberNeedAuthorize)
                {
                    clubMember.Status = ClubMemberStatuses.Pending;
                    TempData["StatusMessageSuccessMessage"] = GetResourceString("Message_ApplyAlreadySubmitPleaseWait", clubApplicationID);
                }
                else
                {
                    clubMember.Status = ClubMemberStatuses.IsApproved;
                    TempData["StatusMessageSuccessMessage"] = GetResourceString("Message_YouAlreadyBecomeMember", clubApplicationID);
                }
            }

            clubMember.ApplyRemark = Request.Form.GetString("applyRemark", string.Empty);

            try
            {
                ClubMembers.CreateClubMember(clubMember);
                if (clubMember.Status == ClubMemberStatuses.Pending)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                    TempData["StatusMessageSuccessMessage"] = "申请成功，需要等待群组管理员审核审核通过！";
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                }
            }
            catch (Exception)
            {

                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            }

            return View("Pages/ApplyClubMember.aspx", "Masters/Club.master", club);
        }

        /// <summary>
        /// 群组资料
        /// </summary>
        public ActionResult Control_ClubStatisticsBlock(string clubDomainName)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            return PartialView("Controls/ClubStatisticsBlock.ascx", currentClub);
        }
        /// <summary>
        /// 群主
        /// </summary>
        public ActionResult Control_ClubOwner(string clubDomainName)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            User clubOwner = currentClub.ClubOwner;
            ICollection<ClubMember> clubAssistants = ClubMembers.GetClubAssistants(currentClub.ClubID);
            ViewData["clubAssistants"] = clubAssistants;
            return PartialView("Controls/ClubOwner.ascx", clubOwner);
        }

        /// <summary>
        /// 群组公告
        /// </summary>
        public ActionResult Control_ClubPlacard(string clubDomainName)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            return PartialView("Controls/ClubPlacard.ascx", currentClub);
        }

        /// <summary>
        /// 群组分享模块
        /// </summary>
        public ActionResult Control_ListShareContentsBlock(string clubDomainName, int? displayCount)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            bool isRealTime = ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageShareItemOfClub());
            PagingDataSet<ClubSharedContent> pds = ClubSharedContents.GetClubSharedContents(currentClub.ClubID, null, null, null, null, 1, isRealTime);
            IList<ClubSharedContent> shares = pds.Records;
            if (pds.Records.Count > (displayCount ?? 10))
                shares = pds.Records.Take(displayCount ?? 10).ToList();
            return PartialView("Controls/ListShareContentsBlock.ascx", shares);
        }
        /// <summary>
        ///友情链接 
        /// </summary>
        public ActionResult Control_ClubDomainLinks(string clubDomainName)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            bool isRealTime = CurrentUser != null && ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubLink());

            ClubLinkManager CLinkManage = ClubLinkManager.Instance();
            List<Link> ListLinks = CLinkManage.GetLinks(currentClub.ClubID, null, null, isRealTime);
            ViewData["ListLinks"] = ListLinks;
            List<LinkCategory> LinkCates = CLinkManage.GetLinkCategoriesByOwnerID(currentClub.ClubID, isRealTime);
            LinkCates = (from l in
                             (from lk in ListLinks
                              select new { lk.CategoryID }
                              ).Distinct()
                         from c in LinkCates
                         where l.CategoryID == c.CategoryID
                         select new LinkCategory
                         {
                             CategoryID = c.CategoryID,
                             Description = c.Description,
                             CategoryName = c.CategoryName,
                             DisplayOrder = c.DisplayOrder,
                             OwnerID = c.OwnerID,
                         }).ToList();

            ViewData["ListLInkCategory"] = LinkCates;
            return PartialView("Controls/ClubDomainLinks.ascx");
        }

        /// <summary>
        /// 退出群组
        /// </summary>
        public ActionResult QuitFromClub(string clubDomainName, int userID)
        {
            if (CurrentUser == null || CurrentUser.UserID != userID && !CurrentUser.IsContentAdministrator)
            {
                throw new ArgumentException(GetResourceString("Exception_YouNotHaveLimits", clubApplicationID));
            }
            Club currentClub = GetCurrentClub(clubDomainName);
            ClubMember clubMember = ClubMembers.GetClubMember(currentClub.ClubID, userID, true);
            if (clubMember == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubMemberNotFind", clubApplicationID), clubDomainName)));
            }
            ClubMembers.DeleteClubMember(clubMember.ClubID, userID);

            return RedirectToAction("Home", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
        }

        #endregion

        #region 群组分享

        /// <summary>
        /// 群组分享首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ClubMemberVisitAndClubDomainHitTimeAttribute]
        public ActionResult Share(string clubDomainName, int? applicationID, string tagName)
        {
            SetPageTitle(GetResourceString("Title_ClubMemberShare", clubApplicationID), clubDomainName);
            Club currentClub = Clubs.GetClub(clubDomainName, false);

            //BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SiteUrls.Instance().ClubDomainHome(clubDomainName));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Common_Share"), SPBUrlHelper.Action("Share", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));

            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", clubApplicationID), clubDomainName)));
            }
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();


            ViewData["listNavigation"] = ClubDomainConfiguration.Instance().EnabledSharedApplicationIDs;
            ViewData["SharedApplicationID"] = applicationID.HasValue ? applicationID.Value : -1;

            if (applicationID.HasValue && applicationID.Value > 0)
            {
                BreadCrumb.AddBreadCrumbItem(Applications.GetApplication(applicationID.Value).ApplicationName, SPBUrlHelper.Action("Share", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "applicationID", applicationID.Value }, { "tagName", tagName } }));
            }
            return View("Pages/ShareContents.aspx");
        }

        public ActionResult Control_SubMenus()
        {
            return PartialView("Controls/ShareSubMenus.ascx");
        }
        /// <summary>
        /// 群组分享列表
        /// </summary>
        public ActionResult Control_ListShareContents(string clubDomainName, int? pageIndex, int? applicationID, string tagName)
        {
            Club currentClub = Clubs.GetClub(clubDomainName, false);
            //if (!string.IsNullOrEmpty(tagName))
            //{
            //    ApplicationTag applicationTag = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().ClubSharedContent()).GetApplicationTag(tagName, false);
            //    if (applicationTag != null)
            //        ViewData["TagName"] = applicationTag.TagName;
            //}

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;
            bool isRealTime = CurrentUser != null && ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageShareItemOfClub());
            PagingDataSet<ClubSharedContent> pds = ClubSharedContents.GetClubSharedContents(currentClub.ClubID, null, applicationID, null, tagName, pageIndex.Value, isRealTime);
            if (applicationID.HasValue && applicationID.Value > 0)
            {
                ViewData["application"] = Applications.GetApplication(applicationID.Value);
            }
            return PartialView("Controls/ListClubShareContents.ascx", pds);
        }
        /// <summary>
        /// 我的分享列表
        /// </summary>
        public ActionResult Control_MyShareContents(string clubDomainName, int? applicationID)
        {
            Club currentClub = Clubs.GetClub(clubDomainName, false);
            if (currentClub == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_ClubNotExist", clubApplicationID), clubDomainName)));
            }
            bool isRealTime = CurrentUser != null && ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageShareItemOfClub());
            PagingDataSet<ClubSharedContent> pds = ClubSharedContents.GetClubSharedContents(currentClub.ClubID, CurrentUser != null ? CurrentUser.UserID : 0, applicationID, null, null, 1, isRealTime);
            if (applicationID.HasValue && applicationID.Value > 0)
            {
                ViewData["application"] = Applications.GetApplication(applicationID.Value);
            }
            return PartialView("Controls/MyShareContents.ascx", pds);
        }
        /// <summary>
        /// 分享精华列表
        /// </summary>
        public ActionResult Control_ShareStickyContents(string clubDomainName, int? applicationID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            bool isRealTime = CurrentUser != null && ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageShareItemOfClub());
            PagingDataSet<ClubSharedContent> pds = ClubSharedContents.GetClubSharedContents(currentClub.ClubID, null, applicationID, true, null, 1, isRealTime);
            if (applicationID.HasValue && applicationID.Value > 0)
            {
                ViewData["application"] = Applications.GetApplication(applicationID.Value);
            }
            return PartialView("Controls/ShareStickyContents.ascx", pds);
        }

        /// <summary>
        /// 详细视图显示分享到群组的图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListThumbPhotos(string clubDomainName, int applicationID, int? pageIndex)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;
            bool isRealTime = CurrentUser != null && ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageShareItemOfClub());
            PagingDataSet<ClubSharedContent> pds = ClubSharedContents.GetClubSharedContents(currentClub.ClubID, null, applicationID, null, null, pageIndex.Value, isRealTime);
            return View("Controls/Photos/ListThumbPhotos.ascx", pds);
        }

        /// <summary>
        /// 显示添加修改标签页
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ShareContentTag(string clubDomainName, int shareContentID)
        {
            List<string> applicationTags = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().ClubSharedContent()).GetApplicationTagsOfItem(shareContentID, true);
            if (applicationTags != null && applicationTags.Count > 0)
            {
                string tagNames = string.Empty;
                foreach (var tag in applicationTags)
                {
                    tagNames += tag + ",";
                }
                ViewData["tagNames"] = tagNames;
            }
            return PartialView("Controls/ShareContentTag.ascx");
        }
        /// <summary>
        /// 添加修改标签
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateShareContentTag(string clubDomainName, int shareContentID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);

            List<string> tagList = new List<string>();
            string[] formattedTagNames = Request.Form.GetString("itemTag", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (formattedTagNames != null && formattedTagNames.Length > 0)
            {
                ApplicationTagManager applicationTagManager = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().ClubSharedContent());
                foreach (string formattedTagName in formattedTagNames)
                {
                    if (applicationTagManager.GetApplicationTag(formattedTagName, true) == null)
                    {
                        ApplicationTag tag = new ApplicationTag();
                        tag.TagName = formattedTagName;
                        tag.UserID = CurrentUser.UserID;
                        tag.Author = CurrentUser.DisplayName;
                        tag.OwnerID = currentClub.ClubID;
                        tag.TagName = Formatter.CleanTagName(tag.TagName);
                        tag.TagName = Formatter.FormatSingleLinePlainTextForStorage(tag.TagName);

                        bool isCreated = applicationTagManager.CreateApplicationTag(tag);
                        if (isCreated)
                            tagList.Add(tag.TagName);
                    }
                    else
                    {
                        tagList.Add(ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().ClubSharedContent()).GetApplicationTag(formattedTagName, true).TagName);
                    }
                }
            }
            ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().ClubSharedContent()).UpdateApplicationTagsOfItem(shareContentID, tagList);
            TempData["statusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateTagSuccess", clubApplicationID));
            return RedirectToAction("Control_ShareContentTag", new RouteValueDictionary { { "shareContentID", shareContentID } });
        }
        /// <summary>
        /// 热门标签
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_HotTags(string clubDomainName, int? displayCount)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            Dictionary<string, KeyValuePair<int, ApplicationTag>> applicationTags = ApplicationTagManager.Instance(ApplicationTagItemKeys.Instance().ClubSharedContent()).GetTopApplicationTags(displayCount ?? 20, ApplicationTagsSortBy.ItemCount, currentClub.ClubID);
            return PartialView("Controls/HotTag.ascx", applicationTags);
        }

        #endregion

        #region 群组动态

        /// <summary>
        /// 显示群组动态模块
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListClubActivitiesBlock(string clubDomainName, int? displayCount)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            bool isRealTime = CurrentUser != null && (ClubMembers.IsClubMember(currentClub.ClubID, CurrentUser.UserID) || CurrentUser.IsContentAdministrator);
            IList<ClubActivity> clubActivities = ClubActivities.GetTopClubActivitiesForClub(currentClub.ClubID, null, displayCount ?? 10, isRealTime);
            ViewData["HasClubManagerPermission"] = ClubMembers.HasClubManagerPermission(currentClub.ClubID, CurrentUser != null ? CurrentUser.UserID : 0);
            return PartialView("Controls/ListClubActivitiesBlock.ascx", clubActivities);
        }
        /// <summary>
        /// 显示群组动态用户控件
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListClubActivities(string clubDomainName, int? pageIndex)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            bool isRealTime = CurrentUser != null && (ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageShareItemOfClub()) || CurrentUser.IsContentAdministrator);
            PagingDataSet<ClubActivity> pds = ClubActivities.GetClubActivitiesForClub(currentClub.ClubID, null, pageIndex ?? 1, isRealTime);
            ViewData["HasClubManagerPermission"] = ClubMembers.HasClubManagerPermission(currentClub.ClubID, CurrentUser == null ? 0 : CurrentUser.UserID);
            return PartialView("Controls/ListClubActivities.ascx", pds);
        }
        /// <summary>
        /// 显示群组动态列表
        /// </summary>
        public ActionResult ListClubActivities(string clubDomainName)
        {
            SetPageTitle(GetResourceString("Title_ClubDynamic", clubApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ClubDynamic", clubApplicationID), string.Empty);

            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            IList<ClubMember> lastVisitClubMembers = ClubMembers.GetTopMembers(currentClub.ClubID, ClubMembersSortBy.LastVisitTime, 9);
            ViewData["lastVisitClubMembers"] = lastVisitClubMembers;
            ViewData["applicationID"] = ClubDomainSpecialAppIDs.Instance().Activities();
            return View("Pages/ListClubActivities.aspx", "Masters/Club.Master");
        }

        /// <summary>
        /// 在模块中删除群组动态
        /// </summary>
        public ActionResult DeleteClubActivityInBlock(string clubDomainName, int applicationID, int displayItemCount, int activityID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (ClubMembers.HasClubManagerPermission(currentClub.ClubID, CurrentUser.UserID))
            {
                ClubActivities.DeleteClubActivity(currentClub.ClubID, applicationID, activityID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteClubDynamicSuccess", clubApplicationID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_YouNotHaveDeleteLimits", clubApplicationID));
            }
            return RedirectToAction("Control_ListClubActivitiesBlock", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "displayItemCount", displayItemCount } });
        }

        /// <summary>
        /// 在群组动态列表页删除群组动态
        /// </summary>
        public ActionResult DeleteClubActivity(string clubDomainName, int applicationID, int? pageIndex, int activityID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (ClubMembers.HasClubManagerPermission(currentClub.ClubID, CurrentUser.UserID))
            {
                ClubActivities.DeleteClubActivity(currentClub.ClubID, applicationID, activityID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteClubDynamicSuccess", clubApplicationID));
            }
            else
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_YouNotHaveDeleteLimits", clubApplicationID));
            }
            return RedirectToAction("Control_ListClubActivities", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "pageIndex", pageIndex } });
        }

        #endregion

        #region 邀请朋友

        public ActionResult InvitationJoin()
        {

            string Token = Request.QueryString.GetString("Token", string.Empty).Trim();

            int inviteUserID = 0;
            if (!string.IsNullOrEmpty(Token))
            {
                try
                {
                    Token = Token.Replace(' ', '+');
                    Token = EncryptManager.DecryptTokenForInviteFriend(Token);
                    string[] qs = Token.Split(',');
                    DateTime dateCreated = Convert.ToDateTime(qs[1]);
                    if (dateCreated.CompareTo(DateTime.Now) < 0)
                        return Redirect(SiteUrls.Instance().Error("链接已过期"));
                    inviteUserID = int.Parse(qs[0]);
                }
                catch { }
            }
            User inviteUser = Users.GetUser(inviteUserID, false);

            if (inviteUser != null && inviteUser.UserType == UserTypes.PersonUser)
            {
                string titleForGender = string.Empty;
                if ((inviteUser as PersonUser) != null)
                {
                    switch ((inviteUser as PersonUser).Gender)
                    {
                        case Genders.Male:
                            titleForGender = GetResourceString("Title_Pattern_ShareHisFirendsWeb", applicationID);
                            break;
                        case Genders.Female:
                            titleForGender = GetResourceString("Title_Pattern_ShareHerFirendsWeb", applicationID);
                            break;
                        default:
                            titleForGender = GetResourceString("Title_Pattern_ShareHisFirendsWeb", applicationID);
                            break;
                    }
                }
                SetPageTitle(string.Format(titleForGender, inviteUser.DisplayName));
                //ViewData["GenderName"] = gender;
                ViewData["InviteUser"] = inviteUser;
                if (CurrentUser != null)
                {
                    if (inviteUser.UserID == CurrentUser.UserID)
                    {
                        ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        ViewData["StatusMessageContent"] = GetResourceString("Message_InvitErrorForInvitSelf", applicationID);
                    }
                    else if (Friends.IsFriend(CurrentUser.UserID, inviteUser.UserID))
                    {
                        ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        ViewData["StatusMessageContent"] = GetResourceString("Message_InvitErrorForRepeatInvit", applicationID);
                    }
                }
            }
            else
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_TheUserInviteIsNotFound", applicationID)));
            }
            return View("Pages/InvitationFriendToClub.aspx");
        }

        /// <summary>
        /// 发送群组邀请
        /// </summary>
        public ActionResult NewJoinClubInvitation(string clubDomainName)
        {
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));

            Club currentClub = GetCurrentClub(clubDomainName);
            string clubName = currentClub.ClubName;
            SetPageTitle(GetResourceString("Title_NewClubInvitation", ApplicationIDs.Instance().User()));


            InvitationTemplate template = InvitationTemplates.GetInvitationTemplate(ApplicationIDs.Instance().Club());
            if (template != null)
            {
                //ViewData["subject"] = InvitationTemplates.EmailFormatterForPersonUser(template.Subject, CurrentUser);
                string body = InvitationTemplates.EmailFormatterForClub(template.Body, CurrentUser, clubDomainName, clubName, currentClub.ClubID);

                ViewData["body"] = InvitationTemplates.EmailFormatterForBody(body, GetResourceString("Description_InvitationBodyPlaceHolderInTemplates", applicationID));
            }
            return View("Pages/NewJoinClubInvitation.aspx", "Masters/Club.master", currentClub);
        }

        /// <summary>
        /// 发送email
        /// </summary>
        private bool SendEmail(Invitation invitation, string body)
        {
            //Email类型的邀请不再存储到数据库
            if (invitation.TransportMode == InvitationTransportModes.Email)
            {
                MailMessage mailMessage = ConvertMessageToMailMessage(invitation, body);
                Emails.EnqueuEmail(mailMessage);
                return true;
            }
            return false;

        }
        /// <summary>
        /// 把Invitation转化成MailMessage
        /// </summary>
        private MailMessage ConvertMessageToMailMessage(Invitation invitation, string body)
        {
            if (string.IsNullOrEmpty(invitation.SenderEmail) || string.IsNullOrEmpty(invitation.ReceiverEmail) || string.IsNullOrEmpty(invitation.Body))
                return null;

            MailMessage email = new MailMessage();
            email.From = new MailAddress(invitation.SenderEmail, invitation.Sender);

            email.To.Add(invitation.ReceiverEmail);

            email.Subject = invitation.Subject;
            email.Priority = MailPriority.High;
            email.Body = body;
            //email.Body = "<html><body>" + email.Body + "</body></html>";

            ////HTML Editor会经常把链接转换成基于本地的相对链接，所以需要转换回来
            email.Body = Formatter.ConvertLocalUrls(email.Body, WebUtils.HostPath(Request.Url));

            email.IsBodyHtml = true;

            //email.SubjectEncoding = System.Text.Encoding.UTF8;  //默认UTF8
            //email.BodyEncoding = System.Text.Encoding.UTF8;     //默认UTF8
            email.SubjectEncoding = System.Text.Encoding.Default;
            email.BodyEncoding = System.Text.Encoding.Default;
            return email;
        }

        /// <summary>
        /// 处理群组Email邀请
        /// </summary>

        private int applicationID = ApplicationIDs.Instance().User();
        public ActionResult CreateFriendInvitationByEmail(string clubDomainName)
        {
            string receivers = Request.Form.GetString("receivers", string.Empty);
            string[] emailAddress = receivers.Split(new Char[] { ',', ';', '，', '；', '\n', '\r' });
            List<string> validatedEmailAddress = new List<string>();
            List<string> unvalidatedEmailAddress = new List<string>();
            Regex emailRegex = new Regex(@"^\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$");

            for (int i = 0; i < emailAddress.Length; i++)
            {
                if (emailRegex.IsMatch(emailAddress[i]))
                    validatedEmailAddress.Add(emailAddress[i]);
                else
                    unvalidatedEmailAddress.Add(emailAddress[i]);
            }
            if (validatedEmailAddress.Count == 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_EmailInvitErrorForNoValidatedEmailAddress", applicationID);
                TempData["subject"] = Request.Form.GetString("subject", string.Empty);
                TempData["body"] = Request.Form.GetString("body", string.Empty);
                return RedirectToAction("NewJoinClubInvitation", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
            }

            //发送Email成功的Email集合
            List<string> successEmails = new List<string>();
            //重复发送的Email集合
            List<string> duplicateEmails = new List<string>();
            string subject = "您的好友邀请您加入群组";
            string invitationBody = Request.Form.GetString("invitationBody", string.Empty);
            string body = string.Empty;
            Club currentClub = GetCurrentClub(clubDomainName);
            string clubName = currentClub.ClubName;
            InvitationTemplate template = InvitationTemplates.GetInvitationTemplate(ApplicationIDs.Instance().Club());
            if (template != null)
            {
                // subject = InvitationTemplates.EmailFormatterForPersonUser(template.Subject, CurrentUser);
                body = InvitationTemplates.EmailFormatterForClub(template.Body, CurrentUser, clubDomainName, clubName, currentClub.ClubID);
                body = InvitationTemplates.EmailFormatterForBody(body, invitationBody);
            }
            for (int i = 0; i < validatedEmailAddress.Count; i++)
            {
                Invitation invitation = new Invitation();
                invitation.Subject = subject;
                invitation.SenderUserID = CurrentUser.UserID;
                invitation.Sender = CurrentUser.DisplayName;
                invitation.Body = invitationBody;
                invitation.SenderEmail = string.IsNullOrEmpty(CurrentUser.PublicEmail) ? CurrentUser.PrivateEmail : CurrentUser.PublicEmail;
                invitation.Receiver = (string)validatedEmailAddress[i];
                invitation.ReceiverEmail = (string)validatedEmailAddress[i];
                invitation.ApplicationID = ApplicationIDs.Instance().Friend();
                invitation.TransportMode = InvitationTransportModes.Email;
                SendEmail(invitation, body);
                CreateInvitationStatuses status = Invitations.CreateInvitation(invitation);
                if (status == CreateInvitationStatuses.Duplicate)
                    duplicateEmails.Add((string)validatedEmailAddress[i]);
                else
                    successEmails.Add((string)validatedEmailAddress[i]);
            }
            string strEmails = string.Empty;
            if (successEmails.Count > 0)
            {
                foreach (string email in successEmails)
                {
                    strEmails += "," + email;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitSendSuccess", applicationID), strEmails.Substring(1));

                string strDuplicateEmails = string.Empty;
                if (duplicateEmails.Count > 0)
                {
                    foreach (string dispalyname in duplicateEmails)
                    {
                        strDuplicateEmails += "," + dispalyname;
                    }
                    TempData["StatusMessageContent"] += string.Format(GetResourceString("Message_Pattern_EmailInvitForRepeatInvit", applicationID), strDuplicateEmails.Substring(1));
                }
            }
            else if (duplicateEmails.Count > 0)
            {
                string strDuplicateEmails = string.Empty;

                foreach (string dispalyname in duplicateEmails)
                {
                    strDuplicateEmails += "," + dispalyname;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitForRepeatInvit", applicationID), strDuplicateEmails.Substring(1));
            }
            //提示未通过验证的Email地址
            if (unvalidatedEmailAddress.Count > 0)
            {
                string strUnvalidatedEmailAddress = string.Empty;
                foreach (string address in unvalidatedEmailAddress)
                {
                    strUnvalidatedEmailAddress += "," + address;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitForNoValidatedEmailAddress", applicationID), strUnvalidatedEmailAddress.Substring(1));
                TempData["invitationBody"] = invitationBody;
            }
            return RedirectToAction("NewJoinClubInvitation", new RouteValueDictionary { { "clubDomainName", clubDomainName } });
        }


        /// <summary>
        /// 处理发送群组邀请
        /// </summary>
        public ActionResult CreateJoinClubInvitation(string userDomainName, int relativeItemID)
        {
            int applicationID = ApplicationIDs.Instance().User();
            User currentDomainUser = Users.GetUser(userDomainName, false);
            Club club = Clubs.GetClub(relativeItemID, false);
            string[] receivers;
            string receiverIDs = Request.Form.GetString("receiverIDs", string.Empty);
            if (!string.IsNullOrEmpty(receiverIDs))
            {
                //收件人集合
                receivers = receiverIDs.Split(',');
                //每个消息最多允许100个接收人
                if (receivers.Length > 100)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForBeyondMaxPersonCount", applicationID);
                    return RedirectToAction("NewJoinClubInvitation", new RouteValueDictionary { { "clubDomainName", club.DomainName } });
                }
                //发送成功的接收人集合
                List<string> successReceivers = new List<string>();
                //重复发送的接受人集合
                List<string> duplicateReceivers = new List<string>();
                //已经成为群组成员的接受人集合
                List<string> clubMembers = new List<string>();
                int toUserID = -1;
                //为每个收件人建立私信
                for (int i = 0; i < receivers.Length; i++)
                {
                    //查找收件人
                    User user = null;
                    if (int.TryParse(receivers[i], out toUserID))
                    {
                        user = Users.GetUser(toUserID, false);
                    }
                    if (user != null)
                    {
                        //判断当前空间者是否接受访问者的群组邀请
                        if (Privacys.Validate(user.UserID, currentDomainUser, PrivacyItemKeys.Instance().Action_ReceiveJoinClubInvitation()))
                        {
                            if (ClubMembers.IsClubMember(relativeItemID, user.UserID))
                                clubMembers.Add(user.DisplayName);
                            else
                            {
                                Invitation invitation = new Invitation();
                                invitation.Subject = string.Format(ResourceManager.GetString("Title_Pattern_DefaultClubInvitationSubject", user.Language, applicationID), string.Format("<a href ='{0}' target='_blank'>{1}</a>", SiteUrls.Instance().ClubDomainHome(club.DomainName), club.ClubName));
                                invitation.Body = WebUtils.HtmlEncode(Request.Form.GetString("body", string.Empty));
                                invitation.SenderUserID = CurrentUser.UserID;
                                invitation.Sender = CurrentUser.DisplayName;
                                invitation.ApplicationID = ApplicationIDs.Instance().Club();
                                invitation.RelativeItemID = relativeItemID;
                                invitation.RelativeItemName = club.ClubName;
                                invitation.ReceiverUserID = user.UserID;
                                invitation.Receiver = user.DisplayName;
                                CreateInvitationStatuses status = Invitations.CreateInvitation(invitation);
                                if (status == CreateInvitationStatuses.Duplicate)
                                    duplicateReceivers.Add(user.DisplayName);
                                else
                                    successReceivers.Add(user.DisplayName);
                            }
                        }
                    }
                }
                //如果有成功发送的消息则显示提示信息
                if (successReceivers.Count > 0)
                {
                    string strReceivers = string.Empty;
                    foreach (string displayName in successReceivers)
                    {
                        strReceivers += "," + displayName;
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_ClubInvitSendSuccess", applicationID), strReceivers.Substring(1));
                    string strDuplicateReceivers = string.Empty;
                    if (duplicateReceivers.Count > 0)
                    {
                        foreach (string dispalyname in duplicateReceivers)
                        {
                            strDuplicateReceivers += "," + dispalyname;
                        }
                        TempData["StatusMessageContent"] += string.Format(GetResourceString("Message_Pattern_ClubInvitForRepeatInvit", applicationID), strDuplicateReceivers.Substring(1));
                    }

                    string strClubMembers = string.Empty;
                    if (clubMembers.Count > 0)
                    {
                        foreach (string displayName in clubMembers)
                        {
                            strClubMembers += "," + displayName;
                        }
                        TempData["StatusMessageContent"] += "\n" + string.Format(GetResourceString("Message_Pattern_ClubInvitForPersonIsAlreadyClubMember", applicationID), strClubMembers.Substring(1));
                    }

                }
                else if (duplicateReceivers.Count > 0 || clubMembers.Count > 0)
                {
                    string strDuplicateReceivers = string.Empty;
                    if (duplicateReceivers.Count > 0)
                    {
                        foreach (string dispalyname in duplicateReceivers)
                        {
                            strDuplicateReceivers += "," + dispalyname;
                        }
                        TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_ClubInvitForRepeatInvit", applicationID), strDuplicateReceivers.Substring(1));
                    }
                    string strClubMembers = string.Empty;
                    if (clubMembers.Count > 0)
                    {
                        foreach (string displayName in clubMembers)
                        {
                            strClubMembers += "," + displayName;
                        }
                        TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        TempData["StatusMessageContent"] = "\n" + string.Format(GetResourceString("Message_Pattern_ClubInvitForPersonIsAlreadyClubMember", applicationID), strClubMembers.Substring(1));
                    }
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_ClubInvitErrorForDenied", applicationID);
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_ClubInvitErrorForFriendEmpty", applicationID);
            }
            return RedirectToAction("NewJoinClubInvitation", new RouteValueDictionary { { "clubDomainName", club.DomainName } });
        }


        #endregion

        #region 留言

        #region 页面
        /// <summary>
        /// 显示全部留言页
        /// </summary>
        [ClubMemberVisitAndClubDomainHitTime]
        public ActionResult ShowClubGuestbook(string clubDomainName)
        {
            SetPageTitle(GetResourceString("Title_ClubMessagePage", clubApplicationID), clubDomainName);

            Club currentClub = GetCurrentClub(clubDomainName);
            return View("Pages/ShowClubGuestbook.aspx");
        }
        #endregion





        #endregion

        #region help

        /// <summary>
        /// 设置Meta标签：KeyWords,Description
        /// </summary>
        private void SetPageMeta(Club currentClub)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;

            if (currentClub.ClubSiteCategoy != null)
                metaKeyWords = currentClub.ClubSiteCategoy.CategoryName;

            if (currentClub.Area != null)
                metaKeyWords = string.Format("{0},{1}", metaKeyWords, currentClub.Area.Name);

            if (string.IsNullOrEmpty(metaKeyWords))
                metaKeyWords = currentClub.ClubName;
            else
                metaKeyWords = string.Format("{0},{1}", metaKeyWords, currentClub.ClubName);

            metaDescription = string.Format("{0}：{1}", currentClub.ClubName, HtmlUtils.TrimHtml(currentClub.Brief, 150));

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(metaKeyWords);
        }


        #endregion
    }

    #region Enums

    /// <summary>
    /// 各个应用在群组中对应项
    /// </summary>
    public enum ApplicationIDsInClub
    {

        /// <summary>
        /// 博客
        /// </summary>
        Blog = 111,

        /// <summary>
        /// 文件
        /// </summary>
        File = 113,

        /// <summary>
        /// 网摘
        /// </summary>
        Bookmark = 114,

        /// <summary>
        /// 图片
        /// </summary>
        Gallery = 112,

        /// <summary>
        /// 论坛
        /// </summary>
        Forum = 211,

        /// <summary>
        /// 活动
        /// </summary>
        Event = 221,

    }

    #endregion
}
