﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Club;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using System.Web.Routing;
using System.Web.UI;

namespace SpaceBuilder.Club.Controllers
{
    [HttpCompress]
    public class ClubMemberController : ClubDomainControllerBase
    {
        int clubApplicationID = ApplicationIDs.Instance().Club();

        /// <summary>
        /// 群组成员首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ClubMemberVisitAndClubDomainHitTimeAttribute]
        public ActionResult Home(string clubDomainName, ClubMembersSortBy? sortBy)
        {
            SetPageTitle(GetResourceString("Title_ClubMemberHome", clubApplicationID), clubDomainName);
            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ViewClub(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            string applicationNameInclub = GetResourceString("Label_ClubMemberCount", clubApplicationID);
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ClubMember", ApplicationIDs.Instance().Club()), string.Empty);

            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            if (sortBy.HasValue)
                ViewData["SortBy"] = sortBy.Value;
            else
                ViewData["SortBy"] = userCookie.GetClubMemberListSortBy();

            return this.View("Pages/Members/Home.aspx", "Masters/Member.master");
        }

        #region Show

        /// <summary>
        /// 群组成员列表控件
        /// </summary>
        public ActionResult Control_ListMembers(string clubDomainName, ClubMembersSortBy? sortBy, int? pageIndex)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            userCookie.SaveClubMemberListSortBy(sortBy ?? ClubMembersSortBy.JoinDate);
            bool isRealTime = ClubPermission.Validate(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember());
            PagingDataSet<ClubMember> pds = ClubMembers.GetMembers(currentClub.ClubID, ClubMemberRoles.ClubMember, sortBy ?? ClubMembersSortBy.JoinDate, 12, pageIndex ?? 1, isRealTime);
            ViewData["SortBy"] = sortBy;

            ICollection<ClubMember> clubAssistants = ClubMembers.GetClubAssistants(currentClub.ClubID);
            ViewData["clubAssistants"] = clubAssistants;
            ViewData["currentClub"] = currentClub;

            ClubMember clubOwner = ClubMembers.GetClubOwner(currentClub.ClubID);
            ViewData["clubOwner"] = clubOwner;

            return PartialView("Controls/Members/ListMembers.ascx", pds);
        }

        /// <summary>
        /// 最近访客
        /// </summary>
        public ActionResult Control_LastVisitClubMembers(string clubDomainName, int? displayCount)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            PagingDataSet<ClubMember> lastVisitClubMembers = ClubMembers.GetMembers(currentClub.ClubID, null, ClubMembersSortBy.LastVisitTime, displayCount ?? 9, 1, false);
            return PartialView("Controls/Members/LastVisitClubMembers.ascx", lastVisitClubMembers.Records);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 管理群组成员
        /// </summary>
        public ActionResult ManageClubMembers(string clubDomainName)
        {
            SetPageTitle(GetResourceString("Title_ManagerClubMember", clubApplicationID), clubDomainName);
            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SPBUrlHelper.Action("Home", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Manager", clubApplicationID), SPBUrlHelper.Action("Home", "ClubManage", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ManagerClubMember", clubApplicationID), SPBUrlHelper.Action("ManageClubMembers", "ClubMember", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));

            return View("Pages/Manage/ManageClubMembers.aspx", "Masters/Manage.master");
        }

        /// <summary>
        /// 处理代理准成员
        /// </summary>
        /// <param name="clubDomainName"></param>
        /// <returns></returns>
        public ActionResult ManagePendingClubMembers(string clubDomainName, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_WithoutApprovalMember", clubApplicationID), clubDomainName);

            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            BreadCrumb.AddBreadCrumbItem(currentClub.ClubName, SPBUrlHelper.Action("Home", "ClubDomain", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Manager", clubApplicationID), SPBUrlHelper.Action("Home", "ClubManage", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_WithoutApprovalMember", clubApplicationID), SPBUrlHelper.Action("ManagePendingClubMembers", "ClubMember", new RouteValueDictionary { { "clubDomainName", clubDomainName } }));

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<ClubMember> clubMembers = ClubMembers.GetMembersForAdmin(currentClub.ClubID, ClubMemberStatuses.NotApproved, null, ClubMembersSortBy.JoinDate, 10, pageIndex.Value);

            return View("Pages/Manage/ManagePendingClubMember.aspx", "Masters/Manage.master", clubMembers);
        }

        /// <summary>
        /// 管理群组成员
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageClubMembers(string clubDomainName, ClubMemberRoles? clubMemberRoles, ClubMemberStatuses? clubMemberStatuses, int? pageIndex)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            bool isTopManager = false;
            ClubMember currentClubMember = ClubMembers.GetClubMember(currentClub.ClubID, CurrentUser.UserID, true);
            if ((currentClubMember != null && currentClubMember.ClubMemberRole == ClubMemberRoles.ClubOwner) || CurrentUser.IsContentAdministrator)
                isTopManager = true;
            ViewData["isTopManager"] = isTopManager;

            PagingDataSet<ClubMember> clubMembers = ClubMembers.GetMembersForAdmin(currentClub.ClubID, clubMemberStatuses, clubMemberRoles, userCookie.GetClubMemberListSortBy(), 10, pageIndex.Value);
            ViewData["clubMemberRoles"] = clubMemberRoles;
            ViewData["clubMemberStatuses"] = clubMemberStatuses;
            return PartialView("Controls/Manage/ManageClubMembers.ascx", clubMembers);
        }

        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManagePendingClubMembers(string clubDomainName, int? pageIndex)
        {
            Club currentClub = GetCurrentClub(clubDomainName);

            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<ClubMember> clubMembers = ClubMembers.GetMembersForAdmin(currentClub.ClubID, ClubMemberStatuses.Pending, null, ClubMembersSortBy.JoinDate, 10, pageIndex.Value);
            return PartialView("Controls/Manage/ManagePendingClubMembers.ascx", clubMembers);
        }

        /// <summary>
        /// 处理待批准成员
        /// </summary>
        public ActionResult Control_ManageApplyMembers(string clubDomainName, int? pageIndex)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;

            PagingDataSet<ClubMember> clubMembers = ClubMembers.GetMembersForAdmin(currentClub.ClubID, ClubMemberStatuses.NotApproved, null, ClubMembersSortBy.JoinDate, 10, pageIndex.Value);

            return PartialView("Controls/Members/ManageApplyMembers.ascx", clubMembers);
        }

        /// <summary>
        /// 批准申请成员
        /// </summary>
        public ActionResult SetMembersStatus(string clubDomainName, ClubMemberStatuses updateStatus, ClubMemberStatuses? status, ClubMemberRoles? clubMemberRole, int? pageIndex, bool isPending)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            string memberIDsString = Request.Form.Get<string>("clubMembersCheckBoxGroup", string.Empty);
            //string hhhh = (string)ClubMemberRoles.ClubMember;
            //ClubMemberRoles? clubMemberRole = (ClubMemberRoles)Request.Form.Get<string>("clubMemberRolesDropDownList", string.Empty);
            //ClubMemberRoles? clubMemberRole = null;
            if (!string.IsNullOrEmpty(memberIDsString))
            {
                string[] memberIDStings = memberIDsString.Split(',');
                List<int> ids = new List<int>();
                if (memberIDStings.Length > 0)
                {
                    foreach (string clubIDstring in memberIDStings)
                    {
                        if (!string.IsNullOrEmpty(clubIDstring))
                        {
                            try
                            {
                                if (Convert.ToInt32(clubIDstring) != currentClub.ClubOwnerUserID)
                                    ids.Add(Convert.ToInt32(clubIDstring));
                            }
                            catch (Exception)
                            {
                            }

                        }
                    }
                }
                int[] userIDs = new int[ids.Count];
                ids.CopyTo(userIDs);
                ClubMembers.UpdateStatus(userIDs, currentClub.ClubID, updateStatus);

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessage"] = GetResourceString("Message_MemberUpdateSuccess", clubApplicationID);
            }

            if (isPending)
                return RedirectToAction("Control_ManagePendingClubMembers", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "pageIndex", pageIndex } });
            else
                return RedirectToAction("Control_ManageClubMembers", new RouteValueDictionary { { "clubDomainName", clubDomainName }, { "clubMemberRoles", clubMemberRole }, { "clubMemberStatuses", status }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 删除群组成员
        /// </summary>
        public ActionResult DeleteClubMembers(string clubDomainName, ClubMemberRoles? clubMemberRole, ClubMemberStatuses? clubMemberStatuses, int? pageIndex)
        {
            Club currenClub = GetCurrentClub(clubDomainName);
            string memberIDsString = Request.Form.Get<string>("clubMembersCheckBoxGroup", string.Empty);
            if (!ClubPermission.AccessCheck(currenClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();

            //ClubMemberRoles? clubMemberRole = Request.Form.Get<ClubMemberRoles?>("clubMemberRolesDropDownList", null);
            //ClubMemberStatuses? clubMemberStatuses = Request.Form.Get<ClubMemberStatuses?>("ClubMemberStatuses", null);
            if (!string.IsNullOrEmpty(memberIDsString))
            {
                string[] memberIDStings = memberIDsString.Split(',');
                List<int> ids = new List<int>();
                if (memberIDStings.Length > 0)
                {
                    foreach (string clubIDstring in memberIDStings)
                    {
                        if (!string.IsNullOrEmpty(clubIDstring))
                        {
                            if (currenClub.ClubOwnerUserID != Convert.ToInt32(clubIDstring))
                                ids.Add(Convert.ToInt32(clubIDstring));
                        }
                    }
                }
                int[] userIDs = new int[ids.Count];
                ids.CopyTo(userIDs);
                ClubMembers.DeleteClubMember(currenClub.ClubID, userIDs);

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessage"] = GetResourceString("Message_DeleteSuccess", clubApplicationID);
            }

            return Control_ManageClubMembers(clubDomainName, clubMemberRole, clubMemberStatuses, pageIndex);
        }

        /// <summary>
        /// 设置圈副
        /// </summary>
        public ActionResult SetClubAssistant(string clubDomainName, int userID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            ClubMember currentClubMember = ClubMembers.GetClubMember(currentClub.ClubID, userID, true);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            if (currentClubMember != null && currentClubMember.UserID != currentClub.ClubOwnerUserID)
            {
                currentClubMember.ClubMemberRole = ClubMemberRoles.ClubAssistant;
                ClubMembers.UpdateClubMember(currentClubMember);
            }
            //return Control_ManageClubMembers(clubDomainName, null, null, null);
            return new EmptyResult();
        }

        /// <summary>
        /// 取消圈副
        /// </summary>
        public ActionResult CancelClubAssistant(string clubDomainName, int userID)
        {
            Club currentClub = GetCurrentClub(clubDomainName);
            ClubMember currentClubMember = ClubMembers.GetClubMember(currentClub.ClubID, userID, true);
            if (!ClubPermission.AccessCheck(currentClub, CurrentUser, PermissionItemKeys.Instance().ManageClubMember(), HttpContext, SiteUrls.Instance().Login(true)))
                return new EmptyResult();
            if (currentClubMember != null && currentClubMember.UserID != currentClub.ClubOwnerUserID)
            {
                currentClubMember.ClubMemberRole = ClubMemberRoles.ClubMember;
                ClubMembers.UpdateClubMember(currentClubMember);
            }
            return new EmptyResult();
        }

        #endregion
    }
}