﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using System.Web.Mvc;
using System.IO;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using SpaceBuilder.Common.Handlers;


namespace SpaceBuilder.Club.Handlers
{
    /// <summary>
    /// 显示ClubLogo
    /// </summary>
    public class ClubLogoHandler : DownloadFileHandlerBase
    {
        public override void ProcessRequest(HttpContext context)
        {
            //防盗链检测
            if (SPBConfig.Instance().EnableAntiLeech && !Globals.IsAllowableReferrer(context.Request))
            {
                context.Response.Redirect(Globals.GetFullUrlInResourceSite("~/Themes/Shared/Styles/Images/anti-Leech.gif"), true);
                return;
            }

            int clubID = context.Request.QueryString.GetInt("clubID", -1);
            ClubLogoSizeTypes clubLogoSizeType = (ClubLogoSizeTypes)context.Request.QueryString.GetInt("clubLogoSizeType", (int)ClubLogoSizeTypes.Normal);
            bool enableCaching = context.Request.QueryString.GetBool("enableCaching", true);

            if (clubID <= 0)
            {
                WebUtils.Return404(context);
                return;
            }

            IFile clubLogoFile = ClubLogos.GetClubLogo(clubID, clubLogoSizeType);
            if (clubLogoFile != null)
            {
                DateTime lastModified = clubLogoFile.LastModified.ToUniversalTime();
                if (enableCaching && IsCacheOK(context, lastModified))
                {
                    WebUtils.Return304(context);
                    return;
                }
                else
                {
                    context.Response.ContentType = MimeTypeConfiguration.GetMimeType(clubLogoFile.FileName);

                    FileSystemFile fileSystemFile = clubLogoFile as FileSystemFile;
                    if (fileSystemFile != null && (!fileSystemFile.FullLocalPath.StartsWith(@"\")))
                    {
                        // Send files stored on UNC paths explicitly to avoid a bug with TransmitFile.
                        context.Response.TransmitFile(fileSystemFile.FullLocalPath);
                    }
                    else
                    {
                        context.Response.AddHeader("Content-Length", clubLogoFile.ContentLength.ToString("0"));
                        context.Response.Buffer = false;
                        context.Response.BufferOutput = false;
                        Stream stream = null;
                        try
                        {
                            stream = clubLogoFile.OpenReadStream();
                            if (stream == null)
                            {
                                WebUtils.Return404(context);
                                return;
                            }

                            int bufferLength = clubLogoFile.ContentLength <= DownloadFileHandlerBase.BufferLength ? clubLogoFile.ContentLength : DownloadFileHandlerBase.BufferLength;
                            byte[] buffer = new byte[bufferLength];
                            int readedSize;
                            while ((readedSize = stream.Read(buffer, 0, bufferLength)) > 0)
                            {
                                if (!context.Response.IsClientConnected)
                                    break;

                                context.Response.OutputStream.Write(buffer, 0, readedSize);
                                context.Response.OutputStream.Flush();
                            }

                            context.Response.OutputStream.Flush();
                            context.Response.Flush();
                        }
                        catch (Exception ex)
                        {
                            context.Response.Redirect(SiteUrls.Instance().Error(ex.ToString()), true);
                            return;
                        }
                        finally
                        {
                            if (stream != null)
                                stream.Close();
                        }
                    }

                    if (enableCaching)
                    {
                        // Browser cache settings
                        context.Response.Cache.SetCacheability(HttpCacheability.Private);
                        context.Response.Cache.SetLastModified(lastModified);
                        context.Response.Cache.SetETag(lastModified.Ticks.ToString());
                        context.Response.Cache.SetAllowResponseInBrowserHistory(true);
                        context.Response.Cache.SetValidUntilExpires(true);
                    }
                    else
                    {
                        context.Response.Cache.SetExpires(DateTime.Now.AddMonths(-1));
                    }
                }

            }
            else
            {
                string clubLogoFullPath;
                string clubLogoFileName;
                switch (clubLogoSizeType)
                {
                    case ClubLogoSizeTypes.Small:
                        clubLogoFileName = "~/Themes/Shared/Styles/Images/club_nophoto_small.gif";
                        clubLogoFullPath = context.Server.MapPath(clubLogoFileName);
                        break;

                    case ClubLogoSizeTypes.Normal:
                    default:
                        clubLogoFileName = "~/Themes/Shared/Styles/Images/club_nophoto.gif";
                        clubLogoFullPath = context.Server.MapPath(clubLogoFileName);
                        break;
                }

                if (!System.IO.File.Exists(clubLogoFullPath))
                {
                    WebUtils.Return404(context);
                    return;
                }

                if (enableCaching)
                {
                    context.Response.Cache.SetExpires(DateTime.UtcNow.AddHours(4));
                    context.Response.Cache.SetCacheability(HttpCacheability.Private);
                    context.Response.Cache.SetValidUntilExpires(true);
                }

                context.Response.ContentType = "image/gif";
                context.Response.AddHeader("content-disposition", "inline; filename=" + clubLogoFileName);
                context.Response.TransmitFile(clubLogoFullPath);
            }

            context.Response.Cache.VaryByParams["clubID"] = true;
            context.Response.Cache.VaryByParams["clubLogoSizeType"] = true;
            context.Response.Cache.VaryByParams["enableCaching"] = true;
            context.Response.End();
        }

    }
}