﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using SpaceBuilder.Club;
using SpaceBuilder.Common;
using SpaceBuilder.Common.Mail;

using SpaceBuilder.Web;
using SpaceBuilder.Utils;

namespace SpaceBuilder.Club.Modules
{
    /// <summary>
    /// 处理与群组相关的动态、通知
    /// </summary>
    public class DisposeActivityForClub : IClubModule
    {
        #region IClubModule 成员

        /// <summary>
        /// 注册相关事件处理程序
        /// </summary>
        public void Init(ClubEventManager em, System.Xml.Linq.XElement node)
        {
            em.AfterClubChange += new ClubEventHandler(em_AfterClubChange);

            em.AfterClubMemberChange += new ClubMemberEventHandler(em_AfterClubMemberChange);

            em.AfterClubSharedContentChange += new ClubSharedContentEventHandler(em_AfterClubSharedContentChange);
        }

        #endregion

        void em_AfterClubSharedContentChange(ClubSharedContent clubSharedContent, GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                Application application = Applications.GetApplication(clubSharedContent.ApplicationID);
                if (application == null)
                    return;
                User author = Users.GetUser(clubSharedContent.AuthorUserID, false);
                if (author == null)
                    return;
                ClubActivity clubActivity = new ClubActivity();
                clubActivity.ApplicationID = clubSharedContent.ApplicationID;
                clubActivity.ClubID = clubSharedContent.ClubID;
                clubActivity.LeadingActorUserID = clubSharedContent.SharerUserID;
                clubActivity.LeadingActor = clubSharedContent.Sharer;
                clubActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewClubSharedContent", author.Language, ApplicationIDs.Instance().Club()), SPBHtmlHelper.Instance().UserDisplayNameLink(author,true),
                    application.GetMultilingualApplicationName(author.Language), GetHyperlink(clubSharedContent.DetailsUrl, clubSharedContent.Subject));
                clubActivity.Body = clubSharedContent.Body;
                ClubActivities.CreateClubActivity(clubActivity);
                if (clubActivity.ActivityID > 0)
                {
                    IList<ClubMember> members = ClubMembers.GetTopMembers(clubActivity.ClubID, ClubMembersSortBy.LastVisitTime, ValueHelper.GetSafeSqlInt(int.MaxValue));
                    List<int> memberIDs = new List<int>();
                    foreach (ClubMember member in members)
                    {
                        memberIDs.Add(member.UserID);
                    }
                    ClubActivities.SendToClubMembers(clubActivity.ActivityID, clubActivity.ApplicationID, memberIDs);
                }
            }
        }


        void em_AfterClubChange(Club club, GlobalEventArgs e)
        {
            if (e.State == ObjectState.Create)
            {
                ClubMember clubOwner = ClubMembers.GetClubOwner(club.ClubID);
                User user = Users.GetUser(clubOwner.UserID, false);
                if (user == null )
                    return;

                string itemKey = ActivityItemKeys.Instance().NewClub();
                ActivityItem activityItem = Activities.GetActivityItem(itemKey);

                if (activityItem == null)
                    return;

                #region 处理个人动态
                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;

                userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewClub", user.Language, e.ApplicationID), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetClubNameWithHyperlink(club));

                if (club.IsPublic)
                    userActivity.IsPublic = true;
                else
                    userActivity.IsPublic = false;

                userActivity.RelativeItemID = club.ClubID;
                userActivity.RelativeItemName = club.ClubName;

                Activities.CreateUserActivity(userActivity);
                #endregion

                #region 处理朋友动态
                if (club.IsPublic)
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;
                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.RelativeItemID = club.ClubID;
                        friendActivity.RelativeItemName = club.ClubName;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion
            }
        }

        /// <summary>
        /// 处理群组成员
        /// </summary>
        void em_AfterClubMemberChange(ClubMember clubMember, GlobalEventArgs e)
        {
            if ((e.State == ObjectState.Create || e.State == ObjectState.Delete) && (!(clubMember.ClubMemberRole == ClubMemberRoles.ClubOwner)))
            {

                User user = Users.GetUser(clubMember.UserID, false);
                if (user == null )
                    return;
                Club club = Clubs.GetClub(clubMember.ClubID, false);
                if (club == null)
                    return;
                ActivityItem activityItem = null;
                string itemKey = string.Empty;

                if (e.State == ObjectState.Create)
                    itemKey = ActivityItemKeys.Instance().NewClubMember();

                else if (e.State == ObjectState.Delete)
                    itemKey = ActivityItemKeys.Instance().ExitClub();

                activityItem = Activities.GetActivityItem(itemKey);
                if (activityItem == null)
                    return;

                #region 处理个人动态
                UserActivity userActivity = new UserActivity();
                userActivity.ApplicationID = activityItem.ApplicationID;
                userActivity.ActivityItemID = activityItem.ItemID;
                userActivity.LeadingActorUserID = user.UserID;
                userActivity.LeadingActor = user.DisplayName;

                if (e.State == ObjectState.Create)
                    userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_NewClubMember", user.Language, e.ApplicationID), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetClubNameWithHyperlink(club));
                else if (e.State == ObjectState.Delete)
                    userActivity.Subject = string.Format(ResourceManager.GetString("Title_Pattern_Activity_DeleteClubMember", user.Language, e.ApplicationID), SPBHtmlHelper.Instance().UserDisplayNameLink(user,true), GetClubNameWithHyperlink(club));

                if (club.IsPublic)
                    userActivity.IsPublic = true;
                else
                    userActivity.IsPublic = false;

                userActivity.RelativeItemID = club.ClubID;
                userActivity.RelativeItemName = club.ClubName;

                Activities.CreateUserActivity(userActivity);
                #endregion


                #region 处理朋友动态
                if (club.IsPublic)
                {
                    UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(user.UserID, false);
                    //判断用户是否允许发送给 朋友动态项目
                    if (userActivitySettings.IsAllowedInActivityItemsOfSending(itemKey))
                    {
                        FriendActivity friendActivity = new FriendActivity();
                        friendActivity.ApplicationID = activityItem.ApplicationID;
                        friendActivity.ActivityItemID = activityItem.ItemID;
                        friendActivity.LeadingActorUserID = user.UserID;
                        friendActivity.LeadingActor = user.DisplayName;
                        friendActivity.Subject = userActivity.Subject;
                        friendActivity.RelativeItemID = club.ClubID;
                        friendActivity.RelativeItemName = club.ClubName;

                        List<int> receiverUserIDs = Activities.GetFriendActivityReceiverUserIDsFromFriends(user, itemKey);
                        Activities.CreateFriendActivity(friendActivity, receiverUserIDs);
                    }
                }
                #endregion

            }
        }

        /// <summary>
        /// 获取带链接的html代码
        /// </summary>
        private static string GetHyperlink(string href, string text)
        {
            return string.Format("<a {0} href=\"{1}\">{2}</a>", HyperlinkTargetOfUserActivity, href, text);
        }

        /// <summary>
        /// 获取带链接的群组名称的html代码
        /// </summary>
        private static string GetClubNameWithHyperlink(Club club)
        {
            return string.Format("<a {0} href=\"{1}\">{2}</a>", HyperlinkTargetOfUserActivity, SiteUrls.Instance().ClubDomainHome(club.DomainName), club.ClubName);
        }

        private static readonly string HyperlinkTargetOfUserActivity = SiteSettingsManager.GetSiteSettings().OpenNewWindowInUserActivity ? "target=\"_blank\"" : string.Empty;

    }
}
