﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using System.Reflection;
using System.IO;
using System.Web.Mvc;
using System.Web.UI;

namespace SpaceBuilder.Common.Controllers
{
    /// <summary>
    /// 用于Action的输出缓存
    /// </summary>
    /// <remarks>
    /// 代替asp.net mvc 的OutputCache，解决在Action(例如使用RenderAction时)上设置OutputCache导致整个View都设置OutputCache的问题
    /// </remarks>
    public class ActionOutputCacheAttribute : ActionFilterAttribute
    {
        // This hack is optional;
        private static MethodInfo _switchWriterMethod = typeof(HttpResponse).GetMethod("SwitchWriter", BindingFlags.Instance | BindingFlags.NonPublic);

        ///// <summary>
        ///// 构造函数
        ///// </summary>
        ///// <param name="cacheDuration">缓存时间用秒表示</param>
        //public ActionOutputCacheAttribute(int cacheDuration)
        //{
        //    _cacheDuration = cacheDuration;
        //}

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="cachingExpirationType">缓存过期时间类型</param>
        public ActionOutputCacheAttribute(CachingExpirationType cachingExpirationType)
        {
            switch (cachingExpirationType)
            {
                case CachingExpirationType.Stable:
                    _cacheDuration = 60 * 10;
                    break;
                case CachingExpirationType.RelativelyStable:
                    _cacheDuration = 60 * 5;
                    break;
                case CachingExpirationType.UsualObjectCollection:
                default:
                    _cacheDuration = 60 * 1;
                    break;
            }
        }

        //目前还不能设置为Client缓存，会与OutputCache同样的问题
        private CachePolicy _cachePolicy = CachePolicy.NoCache;
        private int _cacheDuration;
        private TextWriter _originalWriter;
        private string _cacheKey;

        public override void OnActionExecuting(ActionExecutingContext filterContext)
        {
            // Server-side caching?
            if (_cachePolicy == CachePolicy.Server || _cachePolicy == CachePolicy.ClientAndServer)
            {
                _cacheKey = GenerateCacheKey(filterContext);
                CacheContainer cachedOutput = (CacheContainer)filterContext.HttpContext.Cache[_cacheKey];
                if (cachedOutput != null)
                {
                    filterContext.HttpContext.Response.ContentType = cachedOutput.ContentType;
                    filterContext.Result = new ContentResult { Content = cachedOutput.Output };
                }
                else
                {
                    StringWriter stringWriter = new StringWriterWithEncoding(filterContext.HttpContext.Response.ContentEncoding);
                    HtmlTextWriter newWriter = new HtmlTextWriter(stringWriter);
                    _originalWriter = (TextWriter)_switchWriterMethod.Invoke(HttpContext.Current.Response, new object[] { newWriter });
                }
            }
        }

        public override void OnResultExecuted(ResultExecutedContext filterContext)
        {
            // Server-side caching?
            if (_cachePolicy == CachePolicy.Server || _cachePolicy == CachePolicy.ClientAndServer)
            {
                if (_originalWriter != null) // Must complete the caching
                {
                    HtmlTextWriter cacheWriter = (HtmlTextWriter)_switchWriterMethod.Invoke(HttpContext.Current.Response, new object[] { _originalWriter });
                    string textWritten = ((StringWriter)cacheWriter.InnerWriter).ToString();
                    filterContext.HttpContext.Response.Write(textWritten);
                    CacheContainer container = new CacheContainer(textWritten, filterContext.HttpContext.Response.ContentType);
                    filterContext.HttpContext.Cache.Add(_cacheKey, container, null, DateTime.Now.AddSeconds(_cacheDuration), System.Web.Caching.Cache.NoSlidingExpiration, System.Web.Caching.CacheItemPriority.Normal, null);
                }
            }
        }

        /// <summary>
        /// 添加客户端缓存
        /// </summary>
        /// <param name="filterContext"></param>
        public override void OnActionExecuted(ActionExecutedContext filterContext)
        {
            // Client-side caching?
            if (_cachePolicy == CachePolicy.Client || _cachePolicy == CachePolicy.ClientAndServer)
            {
                if (_cacheDuration <= 0)
                    return;

                HttpCachePolicyBase cache = filterContext.HttpContext.Response.Cache;
                TimeSpan cacheDuration = TimeSpan.FromSeconds(_cacheDuration);

                cache.SetCacheability(HttpCacheability.Public);
                cache.SetExpires(DateTime.Now.Add(cacheDuration));
                cache.SetMaxAge(cacheDuration);
                cache.AppendCacheExtension("must-revalidate, proxy-revalidate");
            }
        }


        private string GenerateCacheKey(ActionExecutingContext filterContext)
        {
            StringBuilder cacheKey = new StringBuilder("OutputCacheKey:");

            // Controller + action
            cacheKey.Append(filterContext.Controller.GetType().FullName.GetHashCode());
            if (filterContext.RouteData.Values.ContainsKey("action"))
            {
                cacheKey.Append("_");
                cacheKey.Append(filterContext.RouteData.Values["action"].ToString());
            }

            //当前只取ActionParameters
            //foreach (KeyValuePair<string, object> pair in filterContext.RouteData.Values)
            //{
            //    cacheKey.Append("_");
            //    cacheKey.Append(pair.Key);
            //    cacheKey.Append("=");

            //    if (pair.Value != null)
            //        cacheKey.Append(pair.Value.ToString());
            //    else
            //        cacheKey.Append(string.Empty);
            //}

            foreach (KeyValuePair<string, object> pair in filterContext.ActionParameters)
            {
                cacheKey.Append("_");
                cacheKey.Append(pair.Key);
                cacheKey.Append("=");

                if (pair.Value != null)
                    cacheKey.Append(pair.Value.ToString());
                else
                    cacheKey.Append(string.Empty);
            }

            return cacheKey.ToString();
        }

        private class CacheContainer
        {
            public string Output;
            public string ContentType;

            /// <summary>
            /// Initializes a new instance of the CacheContainer class.
            /// </summary>
            /// <param name="data"></param>
            /// <param name="contentType"></param>
            public CacheContainer(string data, string contentType)
            {
                Output = data;
                ContentType = contentType;
            }
        }


        /// <summary>
        /// 缓存策略
        /// </summary>
        public enum CachePolicy
        {
            NoCache = 0,
            Client = 1,
            Server = 2,
            ClientAndServer = 3
        }

        /// <summary>
        /// 缓存期限类型
        /// </summary>
        public enum CachingExpirationType
        {
            /// <summary>
            /// 稳定数据      
            /// </summary>
            /// <remarks>
            /// 例如： 站点类别、地区统计
            /// </remarks>
            Stable = 1,

            /// <summary>
            /// 相对稳定
            /// </summary>
            /// <remarks>
            /// 例如：信息推荐
            /// </remarks>
            RelativelyStable = 2,

            /// <summary>
            /// 常用的对象集合
            /// </summary>
            /// <remarks>
            ///  例如： 最新资讯
            /// </remarks>
            UsualObjectCollection = 4,
        }

    }


    public class StringWriterWithEncoding : StringWriter
    {
        Encoding encoding;
        public StringWriterWithEncoding(Encoding encoding)
        {
            this.encoding = encoding;
        }
        public override Encoding Encoding
        {
            get { return encoding; }
        }
    }

}
