﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Web;


namespace SpaceBuilder.Common.Controllers
{
    /// <summary>
    /// 用户空间身份验证
    /// </summary>
    [AttributeUsage(AttributeTargets.Class | AttributeTargets.Method, Inherited = true, AllowMultiple = true)]
    public class UserDomainAuthorizeAttribute : FilterAttribute, IAuthorizationFilter
    {

        private bool requireOwnerOrAdministrator = false;
        /// <summary>
        /// 是否需要空间主人或管理员权限
        /// </summary>
        public bool RequireOwnerOrAdministrator
        {
            get { return requireOwnerOrAdministrator; }
            set { requireOwnerOrAdministrator = value; }
        }


        #region IAuthorizationFilter 成员

        public void OnAuthorization(AuthorizationContext filterContext)
        {
            if (filterContext == null)
            {
                throw new ArgumentNullException("filterContext");
            }

            if (!AuthorizeCore(filterContext))
            {
                // auth failed, redirect to login page
                //filterContext.Cancel = true;
                filterContext.Result = new RedirectResult(SiteUrls.Instance().Login(true));
                return;
            }
        }

        #endregion

        private bool AuthorizeCore(AuthorizationContext filterContext)
        {
            User currentUser = Globals.GetCurrentUser();
            //判断该用户是否有访问该空间的权限 
            if (currentUser != null && filterContext.HttpContext.Request.IsAuthenticated)
            {
                string userDomainName = filterContext.GetParameterFromRouteDateOrQueryString("userDomainName");
                if (!string.IsNullOrEmpty(userDomainName) && IsUserDomainOwner(userDomainName) || currentUser.IsContentAdministrator)
                    return true;
                else
                {
                    if (RequireOwnerOrAdministrator)
                    {
                        filterContext.Result = new RedirectResult(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().AccessDenied()));
                    }
                    else
                    {
                        User currentDomainUser = Users.GetUser(userDomainName, false);
                        if (currentDomainUser == null)
                            return false;
                        if (Privacys.IsStopedUser(currentDomainUser.UserID, currentUser.UserID, false))
                            filterContext.Result = new RedirectResult(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().AccessDenied()));
                        else
                            return true;
                    }
                }
            }
            else
            {
                if (!RequireOwnerOrAdministrator)
                    return true;
            }

            return false;
        }

        /// <summary>
        /// 当前用户是否当前空间的所有者
        /// </summary>
        private bool IsUserDomainOwner(string userDomainName)
        {
            return Globals.GetCurrentUser().UserName.Equals(userDomainName, StringComparison.CurrentCultureIgnoreCase);
        }

        /// <summary>
        /// 当前用户是否当前空间的所有者
        /// </summary>
        private bool IsUserDomainOwner(int ownerUserID)
        {
            return Globals.GetCurrentUser().UserID == ownerUserID;
        }
    }
}