﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Web;
using SpaceBuilder.Caching;
using System.Web;
using System.Web.Routing;
using SpaceBuilder.Utils;
using System.Net.Mail;
using System.Web.UI;

namespace SpaceBuilder.Common.Controllers
{
    /// <summary>
    /// 后台礼品管理
    /// </summary>
    [ManageAuthorize]
    [HttpCompress]
    public class AuxiliariesController : ManageControllerBase
    {

        #region 礼品管理

        #region 管理礼品
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult ManageConvertibleGifts(int? pageIndex)
        {
            SetPageTitle("管理礼品");

            ViewData["applicationID"] = ManageSpecialAppIDs.Instance().Gift();

            return View("Pages/Auxiliaries/ManageConvertibleGifts.aspx", "Masters/AuxiliaryModule.Master");
        }

        /// <summary>
        /// 礼品列表
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageConvertibleGifts(int? pageIndex)
        {
            PagingDataSet<ConvertibleGift> pds = ConvertibleGifts.GetGiftsForAdmin(17, pageIndex.HasValue ? pageIndex.Value : 1, "");

            ////如果礼品数量小于等于0，则自动下架
            //foreach (ConvertibleGift gift in pds.Records)
            //{
            //    if (gift.TotalCount <= 0)
            //        ConvertibleGifts.UpdateGiftEnabledStatus(gift.GiftID, false);
            //}

            ViewData["CurrentUserID"] = CurrentUser != null ? CurrentUser.UserID : 0;

            return PartialView("Controls/Auxiliaries/ManageConvertibleGifts.ascx", pds);
        }

        /// <summary>
        /// 显示添加礼品模式框
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_NewConvertibleGift()
        {
            return PartialView("Controls/Auxiliaries/EditConvertibleGift.ascx");
        }

        /// <summary>
        /// 执行添加礼品操作
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateConvertibleGift()
        {
            ConvertibleGift gift = new ConvertibleGift();

            gift.Price = Request.Form.GetInt("price", 0);
            gift.TotalCount = Request.Form.GetInt("TotalCount", 0);
            gift.IsEnabled = Request.Form.GetBool("isenable", false);
            gift.DisplayOrder = Request.Form.GetInt("displayorder", 0);

            gift.Body = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("Body", string.Empty), true);
            gift.Subject = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("Subject", string.Empty));

            System.Web.HttpPostedFileBase picFile = null;
            if (Request.Files != null && Request.Files.Count > 0)
                picFile = Request.Files[0];
            if (picFile == null || picFile.ContentType == null || picFile.ContentType.IndexOf("image") == -1)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择一个图片文件";

                return RedirectToAction("Control_NewConvertibleGift");
            }

            if (string.IsNullOrEmpty(gift.Subject))
                return RedirectToAction("Control_NewConvertibleGift");
            ConvertibleGifts.CreateGift(gift, picFile);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "添加成功";

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_NewConvertibleGift", "Auxiliaries"));
        }

        /// <summary>
        /// 显示编辑礼品模式框
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_EditConvertibleGift(int giftID)
        {
            ConvertibleGift gift = ConvertibleGifts.GetGift(giftID, false);
            if (gift == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("获取ConvertibleGift出错，不存在GiftID为{0}的礼品", giftID)));

            ViewData["gift"] = gift;
            return PartialView("Controls/Auxiliaries/EditConvertibleGift.ascx");
        }

        /// <summary>
        /// 执行修改礼品操作
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateConvertibleGift(int giftID)
        {
            ConvertibleGift gift = ConvertibleGifts.GetGift(giftID, false);
            if (gift == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("获取ConvertibleGift出错，不存在GiftID为{0}的礼品", giftID)));

            gift.Price = Request.Form.GetInt("price", 0);
            gift.TotalCount = Request.Form.GetInt("TotalCount", 0);
            gift.IsEnabled = Request.Form.GetBool("isenable", false);
            gift.DisplayOrder = Request.Form.GetInt("displayorder", 0);

            gift.Body = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("Body", string.Empty), true);
            gift.Subject = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("Subject", string.Empty));


            System.Web.HttpPostedFileBase picFile = null;
            if (Request.Files != null && Request.Files.Count > 0)
                picFile = Request.Files[0];
            if (picFile == null && string.IsNullOrEmpty(picFile.FileName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择一个图片文件";
                return RedirectToAction("Control_EditConvertibleGift", new RouteValueDictionary { { "giftID", giftID } });
            }

            ConvertibleGifts.UpdateGift(gift, picFile);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "修改成功";

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditConvertibleGift", "Auxiliaries", new RouteValueDictionary { { "giftID", giftID } }));

        }

        /// <summary>
        /// 删除单个礼品
        /// </summary>
        /// <param name="GiftID"></param>
        /// <returns></returns>
        public ActionResult DeleteConvertibleGift(int giftID)
        {
            if (giftID <= 0)
                return Redirect(SiteUrls.Instance().Error404(string.Format("不存在GiftID为{0}的礼品", giftID)));

            ConvertibleGifts.DeleteGift(giftID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除礼品操作成功";

            return RedirectToAction("ManageConvertibleGifts");
        }

        /// <summary>
        /// 礼品上/下架
        /// </summary>
        /// <param name="giftID"></param>
        /// <returns></returns>
        public ActionResult UpdateGiftIsEnabledStatus(int giftID, bool isEnabled)
        {
            ConvertibleGift gift = ConvertibleGifts.GetGift(giftID, false);
            if (gift == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("获取ConvertibleGift出错，不存在GiftID为{0}的礼品", giftID)));

            if (gift.TotalCount > 0)
                ConvertibleGifts.UpdateGiftEnabledStatus(giftID, isEnabled);
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "该礼品已没有库存了";
            }

            return RedirectToAction("ManageConvertibleGifts");
        }


        #endregion

        #region 管理礼品兑换申请
        /// <summary>
        /// 显示礼品管理申请
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageGiftExchangeApplies(ConvertibleGiftExchangeApplyStatuses? applyStaus)
        {
            SetPageTitle("管理积分兑换");

            if (applyStaus.HasValue)
                ViewData["stause"] = (ConvertibleGiftExchangeApplyStatuses)applyStaus;

            ViewData["applicationID"] = ManageSpecialAppIDs.Instance().Gift();
            return View("Pages/Auxiliaries/ManageConvertibleGiftsApplies.aspx", "Masters/AuxiliaryModule.Master");
        }

        /// <summary>
        /// 礼品兑换申请列表
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageConvertibleGiftsApplies(ConvertibleGiftExchangeApplyStatuses? applyStause, int? pageIndex)
        {
            if (applyStause == ConvertibleGiftExchangeApplyStatuses.Canceled)
                applyStause = null;
            PagingDataSet<ConvertibleGiftExchangeApply> pds = ConvertibleGiftExchangeApplies.GetAppliesForAdmin(applyStause, pageIndex ?? 1);

            return PartialView("Controls/Auxiliaries/ManageConvertibleGiftsApplies.ascx", pds);
        }

        /// <summary>
        /// 删除单个申请
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteGiftApply(int applyID, ConvertibleGiftExchangeApplyStatuses? applyStause, int? pageIndex)
        {
            if (applyID <= 0)
                return Redirect(SiteUrls.Instance().Error404(string.Format("不存在applyID为{0}的礼品申请", applyID)));

            ConvertibleGiftExchangeApplies.DeleteApply(applyID);

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 更改单个申请状态
        /// </summary>
        /// <returns></returns>
        public ActionResult UpdateGiftApplyStaus(int applyID, ConvertibleGiftExchangeApplyStatuses? applyStause, int? pageIndex)
        {

            ConvertibleGiftExchangeApply apply = ConvertibleGiftExchangeApplies.GetApply(applyID);
            if (apply == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("不存在applyID为{0}的礼品申请", applyID)));

            ConvertibleGift gift = ConvertibleGifts.GetGift(apply.GiftID, false);

            User applyUser = Users.GetUser(apply.UserID, false);
            if (applyUser == null)
                return Redirect(SiteUrls.Instance().Error404(string.Format("不存在UserID为{0}的用户", applyUser.UserID)));
            if (apply.Price > applyUser.TradePoints && applyStause == ConvertibleGiftExchangeApplyStatuses.Accept)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "当前用户的积分不足！";
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            }

            if (apply.TotalCount > gift.TotalCount && applyStause == ConvertibleGiftExchangeApplyStatuses.Accept)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "当前礼品的数量不足！";
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            }


            ConvertibleGiftExchangeApplies.UpdateApplyStatus(applyID, applyStause.HasValue ? applyStause.Value : ConvertibleGiftExchangeApplyStatuses.Unsettled);
            if (applyStause == ConvertibleGiftExchangeApplyStatuses.Accept)
            {
                ConvertibleGiftExchangeRecord giftExchangeRecord = new ConvertibleGiftExchangeRecord();
                ConvertibleGiftExchangeApply giftApply = ConvertibleGiftExchangeApplies.GetApply(applyID);

                giftExchangeRecord.GiftID = giftApply.GiftID;
                giftExchangeRecord.TotalCount = giftApply.TotalCount;
                giftExchangeRecord.Price = giftApply.Price;
                giftExchangeRecord.UserDisplayName = giftApply.UserDisplayName;
                giftExchangeRecord.UserID = giftApply.UserID;

                ConvertibleGiftExchangeRecords.CreateRecord(giftExchangeRecord);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "修改成功！";

            return Redirect(Request.UrlReferrer.AbsoluteUri);

        }

        /// <summary>
        /// 批量删除兑换申请
        /// </summary>
        /// <param name="returnUrl">返回的url</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteGiftApplys(string returnUrl)
        {
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedThreadIDsString))
            {
                string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
                if (selectedThreadIDsStringArray != null && selectedThreadIDsStringArray.Length > 0)
                {
                    int threadID;
                    foreach (string Id in selectedThreadIDsStringArray)
                    {

                        if (int.TryParse(Id, out threadID))
                        {
                            if (threadID > 0)
                            {
                                ConvertibleGiftExchangeApplies.DeleteApply(threadID);
                            }
                        }
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "申请批量删除成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的申请";
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);

        }

        /// <summary>
        /// 批量修改申请状态
        /// </summary>
        /// <param name="returnUrl">返回的url</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateGiftApplyStatuses(string returnUrl, ConvertibleGiftExchangeApplyStatuses applyStatuse)
        {
            string selectedThreadIDsString = Request.Form["CheckBoxGroup"];
            if (string.IsNullOrEmpty(selectedThreadIDsString))
            {
                return RedirectToAction("ManageGiftExchangeApplies");
            }
            string[] selectedThreadIDsStringArray = selectedThreadIDsString.Split(',');
            if (selectedThreadIDsStringArray == null || selectedThreadIDsStringArray.Length <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要修改的申请";

                return Redirect(Request.UrlReferrer.AbsoluteUri);
            }

            int threadID;

            foreach (string Id in selectedThreadIDsStringArray)
            {
                if (int.TryParse(Id, out threadID))
                {
                    if (threadID > 0)
                    {
                        ConvertibleGiftExchangeApply giftApply = ConvertibleGiftExchangeApplies.GetApply(threadID);
                        User user = Users.GetUser(giftApply.UserID, false);
                        ConvertibleGift gift = ConvertibleGifts.GetGift(giftApply.GiftID, false);

                        //判断用户的积分是否充足
                        if (giftApply.Price <= user.TradePoints && giftApply.TotalCount <= gift.TotalCount)
                        {

                            ConvertibleGiftExchangeApplies.UpdateApplyStatus(threadID, applyStatuse);

                            //创建交换记录
                            if (applyStatuse == ConvertibleGiftExchangeApplyStatuses.Accept)
                            {
                                ConvertibleGiftExchangeRecord giftExchangeRecord = new ConvertibleGiftExchangeRecord();

                                giftExchangeRecord.GiftID = giftApply.GiftID;
                                giftExchangeRecord.TotalCount = giftApply.TotalCount;
                                giftExchangeRecord.Price = giftApply.Price;
                                giftExchangeRecord.UserDisplayName = giftApply.UserDisplayName;
                                giftExchangeRecord.UserID = giftApply.UserID;

                                ConvertibleGiftExchangeRecords.CreateRecord(giftExchangeRecord);
                            }

                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                            TempData["StatusMessageContent"] = "申请批量修改成功";
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "有部分审核失败，原因可以有某些用户的积分不足或礼品数量不足！";
                        }
                    }
                }
            }
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 显示申请人的联系方式
        /// </summary>
        /// <returns></returns>
        public ActionResult ShowConsigneeInfo(int applyID)
        {
            ConvertibleGiftExchangeApply giftApply = ConvertibleGiftExchangeApplies.GetApply(applyID);

            ViewData["giftApply"] = giftApply;

            return PartialView("Controls/Auxiliaries/ConsigneeInfo.ascx");
        }

        #endregion
        #endregion

        #region
        #region 客服私信

        #region 页面
        /// <summary>
        /// 显示收件箱
        /// </summary>
        public ActionResult MessageInBox(int? pageIndex)
        {
            SetPageTitle("客服收件箱");
            ViewData["PageIndex"] = pageIndex;

            return View("Pages/Auxiliaries/MessageInBox.aspx", "Masters/AuxiliaryModule.Master");
        }
        /// <summary>
        /// 显示发件箱
        /// </summary>
        public ActionResult MessageOutBox(int? pageIndex)
        {
            SetPageTitle("客服发件箱");
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/Auxiliaries/MessageOutBox.aspx", "Masters/AuxiliaryModule.Master");
        }


        /// <summary>
        /// 显示消息
        /// </summary>
        public ActionResult ShowMessage(int threadID)
        {
            SetPageTitle("显示消息详细内容");
            ViewData["ThreadID"] = threadID;
            return View("Pages/Auxiliaries/ShowMessage.aspx", "Masters/AuxiliaryModule.Master");
        }

        #endregion

        #region 控件
        /// <summary>
        /// 显示收件箱信件列表控件
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_MessageInBox(int? pageIndex)
        {
            ViewData["NoReadCount"] = PrivateMessages.GetNoReadCount((int)PrivateMessageEspecialUserID.Serve);
            PagingDataSet<PrivateMessageInboxItem> pds = PrivateMessages.GetPrivateMessagesFromInbox((int)PrivateMessageEspecialUserID.Serve, null, null, pageIndex ?? 1);
            return PartialView("Controls/Auxiliaries/MessageInBox.ascx", pds);
        }

        /// <summary>
        /// 显示发件箱信件列表控件
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_MessageOutBox(int? pageIndex)
        {
            //获取查询结果
            PagingDataSet<PrivateMessageOutboxItem> pds = PrivateMessages.GetPrivateMessagesFromOutbox((int)PrivateMessageEspecialUserID.Serve, null, pageIndex ?? 1);
            return PartialView("Controls/Auxiliaries/MessageOutBox.ascx", pds);
        }

        /// <summary>
        /// 显示发客服消息控件
        /// </summary>
        public ActionResult Control_SendMessage(string toUserName)
        {
            User toUser = Users.GetUser(toUserName, true);
            if (toUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());
            return View("Controls/Auxiliaries/SendMessage.ascx", toUser);
        }
        /// <summary>
        /// 显示消息控件
        /// </summary>
        public ActionResult Control_ShowMessage(int threadID)
        {
            PrivateMessage message = PrivateMessages.GetPrivateMessage(threadID);
            if (message == null)
                throw new SPBException(ExceptionTypes.Instance().PrivateMessageToSelfNotAllowd());

            //设置消息为已读
            if (message.MessageType == PrivateMessageType.System)
            {
                PrivateMessages.SetReadFlag((int)PrivateMessageEspecialUserID.System, message.ThreadID, true);
            }
            else
            {
                PrivateMessages.SetReadFlag((int)PrivateMessageEspecialUserID.Serve, message.ThreadID, true);
            }
            return View("Controls/Auxiliaries/ShowMessage.ascx", message);
        }

        #endregion

        #region 操作
        /// <summary>
        /// 从收件箱中删除一条私信
        /// </summary>
        public ActionResult DeleteInBoxMessage(int threadID, int? pageIndex)
        {
            PrivateMessages.DeletePrivateMessageFromInbox((int)PrivateMessageEspecialUserID.Serve, threadID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除操作成功";
            return RedirectToAction("Control_MessageInBox", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 从发件箱中删除一条私信
        /// </summary>
        public ActionResult DeleteOutBoxMessage(int threadID, int? pageIndex)
        {
            PrivateMessages.DeletePrivateMessageFromOutbox((int)PrivateMessageEspecialUserID.Serve, threadID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除操作成功";
            return RedirectToAction("Control_MessageOutBox", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 批量删除收件箱中的私信
        /// </summary>
        public ActionResult DeleteInBoxMessages()
        {
            string selectedIDsString = Request.Form["MessageInBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int threadID = int.Parse(IDString);
                        if (threadID > 0)
                            PrivateMessages.DeletePrivateMessageFromInbox((int)PrivateMessageEspecialUserID.Serve, threadID);
                    }
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "批量删除操作成功";
            return RedirectToAction("Control_MessageInBox", new RouteValueDictionary { { "pageIndex", 1 } });
        }
        /// <summary>
        /// 批量删除发件箱中的私信
        /// </summary>
        public ActionResult DeleteOutBoxMessages()
        {
            string selectedIDsString = Request.Form["MessageOutBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int threadID = int.Parse(IDString);
                        if (threadID > 0)
                            PrivateMessages.DeletePrivateMessageFromOutbox((int)PrivateMessageEspecialUserID.Serve, threadID);
                    }
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "批量删除操作成功";
            return RedirectToAction("Control_MessageOutBox", new RouteValueDictionary { { "pageIndex", 1 } });
        }

        /// <summary>
        /// 设置私信的阅读状态
        /// </summary>    
        public ActionResult SetMessageReadStatus(bool isread, int? pageIndex)
        {
            string selectedIDsString = Request.Form["MessageInBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int threadID = int.Parse(IDString);
                        if (threadID > 0)
                            PrivateMessages.SetReadFlag((int)PrivateMessageEspecialUserID.Serve, threadID, isread);
                    }
                }
            }
            return RedirectToAction("Control_MessageInBox", new RouteValueDictionary { { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 处理新私信
        /// </summary>
        public ActionResult CreateMessage()
        {
            int receiverID = Request.Form.GetInt("receiverID", 0);
            User toUser = Users.GetUser(receiverID, true);
            if (toUser == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "抱歉，没有找到收件人";
                return RedirectToAction("Control_SendMessage");
            }
            PrivateMessage pm = new PrivateMessage();
            pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("subject", string.Empty).Trim());
            pm.Body = Formatter.FormatMultiLinePlainTextForStorage(Request.Form["body"], true);

            pm.SenderUserID = (int)PrivateMessageEspecialUserID.Serve;

            pm.ReceiverUserID = receiverID;
            pm.Receiver = toUser.DisplayName;
            pm.MessageType = PrivateMessageType.Serve;

            PrivateMessages.SendMessage(pm);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "新私信已成功发送";
            return RedirectToAction("Control_SendMessage", new RouteValueDictionary { { "toUserName", toUser.UserName } });
        }
        /// <summary>
        /// 回复消息
        /// </summary> 
        public ActionResult ReplyMessage(int threadID)
        {
            PrivateMessage pm = PrivateMessages.GetPrivateMessage(threadID);
            if (pm == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "抱歉，没有找到私信";
                return RedirectToAction("ShowMessage", new RouteValueDictionary { { "threadID", threadID } });
            }
            string body = Request.Form.GetString("body", string.Empty);
            PrivateMessagePost post = new PrivateMessagePost();
            post.ThreadID = threadID;
            post.ParentPostID = pm.PostID;
            post.UserID = (int)PrivateMessageEspecialUserID.Serve;
            post.Author = "客服";
            post.Subject = string.Format("{0}:{1}", GetResourceString("Common_Reply"), pm.Subject);
            post.Body = body;
            PrivateMessages.ReplyMessage(post, pm);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "回复消息操作成功";
            return RedirectToAction("Control_ShowMessage", new RouteValueDictionary { { "threadID", threadID } });
        }

        #endregion

        #endregion

        #region Impeach

        #region show

        /// <summary>
        /// 管理举报页
        /// </summary>
        public ActionResult ManageImpeaches(bool? sortBy, int? pageIndex)
        {
            SetPageTitle("举报管理");
            ViewData["sortBy"] = sortBy;
            return View("Pages/Auxiliaries/ManageImpeaches.aspx", "Masters/AuxiliaryModule.Master");
        }

        ///// <summary>
        ///// 根据排序显示举报
        ///// </summary>
        //public ActionResult ListImpeachSortBy(string sortBy)
        //{
        //    //string sortBy = Request.Form["impeachReportOrderBy"];

        //    return RedirectToAction("Control_ListImpeach", new RouteValueDictionary { { "sortBy", sortBy }, { "pageIndex", 1 } });
        //}

        /// <summary>
        /// 列表显示举报
        /// </summary>
        /// <param name="sortBy"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        public ActionResult Control_ListImpeach(string sortBy, int? pageIndex)
        {
            //if (CurrentUser == null || !CurrentUser.AllowViewControlPanel)
            //    PermissionBase.RedirectOrExcpetion(SBExceptionType.AccessDenied);
            int storeIndex = 1;
            if (pageIndex.HasValue && pageIndex.Value > 1)
            {
                storeIndex = pageIndex.Value;
            }
            int pageSize = 15;



            PagingDataSet<ImpeachReport> pds = new PagingDataSet<ImpeachReport>();
            if (string.IsNullOrEmpty(sortBy))
                sortBy = "false";

            pds = ImpeachReports.GetImpeachReports(Convert.ToBoolean(sortBy), pageSize, storeIndex);
            ViewData["sortBy"] = Convert.ToBoolean(sortBy);


            return PartialView("Controls/Auxiliaries/ListImpeach.ascx", pds);
        }

        /// <summary>
        /// 举报详细页
        /// </summary>
        public ActionResult Control_ShowImpeach(string originalUrl, string remark)
        {
            ViewData["originalUrl"] = originalUrl;
            ViewData["remark"] = remark;
            return PartialView("Controls/Auxiliaries/ShowImpeach.ascx");
        }

        #endregion

        #region Manage

        /// <summary>
        /// 删除举报
        /// </summary>
        public ActionResult DeleteImpeachReport(int reportID, int? pageIndex, string sortBy)
        {
            List<int> list = new List<int>();
            list.Add(reportID);
            ImpeachReports.DeleteImpeachReports(list);
            return RedirectToAction("Control_ListImpeach", new RouteValueDictionary { { "sortBy", sortBy }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 批量删除举报
        /// </summary>
        public ActionResult DeleteImpeachReports(int? pageIndex)
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("impeachReportCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("impeachReportCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        if (Convert.ToInt32(postID) > 0)
                            list.Add(Convert.ToInt32(postID));
                    }
                    ImpeachReports.DeleteImpeachReports(list);
                }
            }
            string sortBy = Request.Form["impeachReportOrderBy"];
            return RedirectToAction("Control_ListImpeach", new RouteValueDictionary { { "pageIndex", pageIndex }, { "sortBy", sortBy } });
        }

        /// <summary>
        /// 更新举报状态
        /// </summary>
        public ActionResult UpdateImpeachReportStatus(int reportID, bool newCompletedStatus, int? pageIndex, string sortBy)
        {
            ImpeachReports.UpdateImpeachReportStatus(reportID, newCompletedStatus);

            return RedirectToAction("Control_ListImpeach", new RouteValueDictionary { { "sortBy", sortBy }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 批量更新举报状态
        /// </summary>
        public ActionResult UpdateImpeachReportStatuses(bool newCompletedStatus, int? pageIndex)
        {

            bool? isCompleted = null;
            bool isSuccessed = false;
            isSuccessed = bool.TryParse(Request.Form["impeachReportOrderBy"] as string, out isSuccessed);
            if (isSuccessed == true)
            {
                isCompleted = Convert.ToBoolean(Request.Form["impeachReportOrderBy"]);
            }
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("impeachReportCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("impeachReportCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    foreach (string postID in deletePostIDs)
                    {
                        if (Convert.ToInt32(postID) > 0)
                            ImpeachReports.UpdateImpeachReportStatus(Convert.ToInt32(postID), newCompletedStatus);
                    }
                }
            }
            string sortBy = Request.Form["impeachReportOrderBy"];
            return RedirectToAction("Control_ListImpeach", new RouteValueDictionary { { "sortBy", sortBy }, { "pageIndex", pageIndex } });
        }

        #endregion

        #endregion

        #region 群发消息
        /// <summary>
        /// 显示群发消息页面
        /// </summary>
        public ActionResult MassSystemMessage()
        {
            SetPageTitle("群发消息");
            ICollection<Role> roles = Roles.GetRolesByApplicationID(ApplicationIDs.Instance().User(), true, true);
            return View("Pages/Auxiliaries/MassSystemMessage.aspx", "Masters/AuxiliaryModule.Master", roles);
        }

        /// <summary>
        /// 处理群发消息
        /// </summary>
        public ActionResult MassSystemMessageAction()
        {
            ReceiversTypes type = (ReceiversTypes)Request.Form.GetInt("type", 0);
            string subject = Request.Form.GetString("subject", string.Empty);
            string body = Request.Form.GetString("body", string.Empty);
            TransportModes mode = (TransportModes)Request.Form.GetInt("transportMode", 0);

            if (string.IsNullOrEmpty(subject) || string.IsNullOrEmpty(body))
            {
                TempData["subject"] = subject;
                TempData["body"] = body;
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "标题和内容都不允许为空");
                return RedirectToAction("MassSystemMessage");
            }
            PagingDataSet<PersonUser> pds = null;
            PersonUserQueryForAdmin query = new PersonUserQueryForAdmin();
            if (type == ReceiversTypes.UserRole)
            {
                int roleID = Request.Form.GetInt("roleID", 5);
                query.RoleID = roleID;
                pds = Users.GetPersonUsersForAdmin(query, ValueHelper.GetSafeSqlInt(int.MaxValue), 1);
                SendMessage(subject, body, pds.Records, mode);
            }
            else if (type == ReceiversTypes.UserRank)
            {
                int minRank = Request.Form.GetInt("minRank", 1);
                int maxRank = Request.Form.GetInt("maxRank", 1);
                if (maxRank < minRank)
                {
                    int tempRank = maxRank;
                    maxRank = minRank;
                    minRank = tempRank;
                }
                for (int rank = minRank; rank <= maxRank; rank++)
                {
                    query.Rank = rank;
                    pds = Users.GetPersonUsersForAdmin(query, ValueHelper.GetSafeSqlInt(int.MaxValue), 1);
                    SendMessage(subject, body, pds.Records, mode);
                }
            }
            else
            {
                pds = Users.GetPersonUsersForAdmin(query, ValueHelper.GetSafeSqlInt(int.MaxValue), 1);
                SendMessage(subject, body, pds.Records, mode);
            }

            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "群发消息成功");
            return RedirectToAction("MassSystemMessage");
        }
        /// <summary>
        /// 批量发送消息
        /// </summary>
        /// <param name="subject">标题</param>
        /// <param name="body">内容</param>
        /// <param name="receivers">接受人集合</param>
        /// <param name="mode">传输方式</param>
        private void SendMessage(string subject, string body, IList<PersonUser> receivers, TransportModes mode)
        {
            foreach (PersonUser receiver in receivers)
            {
                if (mode == TransportModes.PrivateMessage)
                {
                    PrivateMessage pm = new PrivateMessage();
                    pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(subject);
                    pm.Body = Formatter.FormatMultiLinePlainTextForStorage(body, true);
                    pm.SenderUserID = (int)PrivateMessageEspecialUserID.Serve;
                    pm.ReceiverUserID = receiver.UserID;
                    pm.Receiver = receiver.DisplayName;
                    pm.MessageType = PrivateMessageType.Serve;

                    PrivateMessages.SendMessage(pm);
                }
                else if (mode == TransportModes.Email)
                {
                    MailMessage mail = new MailMessage();
                    mail.Subject = subject;
                    mail.Body = body;
                    try
                    {
                        mail.From = new MailAddress(SiteSettingsManager.GetSiteSettings().AdminEmailAddress, Globals.SiteName);
                        mail.To.Add(new MailAddress(receiver.PrivateEmail, receiver.DisplayName));
                    }
                    catch (Exception)
                    {
                        continue;
                    }
                    Emails.EnqueuEmail(mail);
                }
            }
        }

        #endregion
        #endregion

        #region 管理
        #region 友情链接
        /// <summary>
        /// 显示友情连接
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageLinks()
        {
            SetPageTitle("友情链接管理");
            SiteLinkManager SLManager = SiteLinkManager.Instance();
            List<Link> SiteLinks = SLManager.GetLinksForManage(-1);
            ViewData["SiteLinks"] = SiteLinks;
            return View("Pages/Auxiliaries/ManageLinks.aspx", "Masters/AuxiliaryModule.Master");
        }
        /// <summary>
        /// 删除友情链接
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteLinkAction()
        {
            string strLinks = Request.Form.GetString("LinkGroup", string.Empty);
            string[] UserLinks = strLinks.Split(',');
            List<int> LinkIDs = new List<int>();
            foreach (var item in UserLinks)
            {
                try
                {
                    LinkIDs.Add(Convert.ToInt32(item));
                }
                catch { }
            }
            SiteLinkManager.Instance().DeleteLinks(LinkIDs);
            return RedirectToAction("ManageLinks");
        }
        /// <summary>
        /// 显示添加友情连接页
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_AddLink()
        {
            SiteLinkManager slManage = SiteLinkManager.Instance();
            List<LinkCategory> listLinkCate = slManage.GetAllLinkCategories(true);
            if (listLinkCate == null || listLinkCate.Count <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "你还没有链接类别，请先添加！";
                return RedirectToAction("Control_AddLinkCategory");
            }
            ViewData["LinkCategories"] = listLinkCate;

            int statusMessageType = Request.QueryString.GetInt("StatusMessageType", -2);
            string statusMessageContent = WebUtils.UrlDecode(Request.QueryString.GetString("StatusMessageContent", string.Empty));
            if (statusMessageType > -2)
            {
                TempData["StatusMessageType"] = statusMessageType;
            }

            if (!string.IsNullOrEmpty(statusMessageContent))
            {
                TempData["StatusMessageContent"] = statusMessageContent;
            }

            return View("Controls/Auxiliaries/EditLink.ascx");
        }
        /// <summary>
        /// 添加友情链接
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_AddLinkAction()
        {
            int currentUserID = int.Parse(EncryptManager.DecryptTokenForInviteFriend(Request.Form["currentUserID"]));

            // 判断用户是否有上传图片的权限
            User currentUser = Users.GetUser(currentUserID, false);

            Link l = new Link();
            l.Title = Request.Form.GetString("Title", string.Empty);
            l.LinkHref = Request.Form.GetString("linkUrl", string.Empty);
            l.LinkTitle = Request.Form.GetString("linkTitle", string.Empty);
            l.CategoryID = Request.Form.GetInt("linkcate", 0);
            l.LinkType = (LinkTypes)Request.Form.GetInt("SiteLinkTypes", 1);
            l.IsEnabled = Request.Form.GetBool("isenable", false);
            l.DisplayOrder = Request.Form.GetInt("displayorder", 0);

            if (!string.IsNullOrEmpty(l.Title))
                l.Title = Formatter.FormatSingleLinePlainTextForStorage(l.Title);
            if (!string.IsNullOrEmpty(l.LinkTitle))
                l.LinkTitle = Formatter.FormatSingleLinePlainTextForStorage(l.LinkTitle);

            SiteLinkManager slManage = SiteLinkManager.Instance();
            if (l.LinkType == LinkTypes.Image)
            {
                System.Web.HttpPostedFileBase picFile = null;
                if (Request.Files != null && Request.Files.Count > 0)
                    picFile = Request.Files[0];
                if (picFile == null)
                {
                    return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_AddLink", "Auxiliaries", new RouteValueDictionary { { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode("请选择一个图片文件") } }));
                }
                if (picFile.ContentType.IndexOf("image") == -1)
                {
                    return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_AddLink", "Auxiliaries", new RouteValueDictionary { { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode("请选择图片文件进行上传") } }));
                }
                slManage.CreateLink(l, picFile);
            }
            else
            {
                slManage.CreateLink(l, null);
            }

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_AddLink", "Auxiliaries", new RouteValueDictionary { { "StatusMessageType", (int)StatusMessageType.Success }, { "StatusMessageContent", WebUtils.UrlEncode("添加成功！") } }));
        }
        /// <summary>
        /// 显示编辑友情连接
        /// </summary>
        /// <param name="linkID">连接id</param>
        /// <returns></returns>
        public ActionResult Control_EditLink(int linkID)
        {
            SiteLinkManager SlManager = SiteLinkManager.Instance();
            List<LinkCategory> listLinkCate = SlManager.GetAllLinkCategories(true);
            if (listLinkCate == null || listLinkCate.Count <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "你还没有链接类别，请先添加！";
                return RedirectToAction("Control_AddLinkCategory");
            }
            ViewData["LinkCategories"] = listLinkCate;

            Link SiteLink = SlManager.GetLink(linkID, true);
            ViewData["SiteLink"] = SiteLink;

            int statusMessageType = Request.QueryString.GetInt("StatusMessageType", -2);
            string statusMessageContent = WebUtils.UrlDecode(Request.QueryString.GetString("StatusMessageContent", string.Empty));
            if (statusMessageType > -2)
            {
                TempData["StatusMessageType"] = statusMessageType;
            }

            if (!string.IsNullOrEmpty(statusMessageContent))
            {
                TempData["StatusMessageContent"] = statusMessageContent;
            }

            return PartialView("Controls/Auxiliaries/EditLink.ascx");
        }
        /// <summary>
        /// 更新编辑友情连接
        /// </summary>
        /// <param name="linkID">连接id</param>
        /// <returns></returns>
        public ActionResult Control_EditLinkAction(int linkID)
        {
            int currentUserID = int.Parse(EncryptManager.DecryptTokenForInviteFriend(Request.Form["currentUserID"]));

            // 判断用户是否有上传图片的权限
            User currentUser = Users.GetUser(currentUserID, false);

            if (!currentUser.IsContentAdministrator)
                throw new SPBException(ExceptionTypes.Instance().AccessDenied(), "您无权设置广告位");

            SiteLinkManager SLManage = SiteLinkManager.Instance();
            Link l = SLManage.GetLink(linkID, true);
            l.Title = Request.Form.GetString("Title", string.Empty);
            l.LinkHref = Request.Form.GetString("linkUrl", string.Empty);
            l.LinkTitle = Request.Form.GetString("linkTitle", string.Empty);
            l.CategoryID = Request.Form.GetInt("linkcate", 0);

            l.IsEnabled = Request.Form.GetBool("isenable", false);
            l.DisplayOrder = Request.Form.GetInt("displayorder", 0);

            if (!string.IsNullOrEmpty(l.Title))
                l.Title = Formatter.FormatSingleLinePlainTextForStorage(l.Title);
            if (!string.IsNullOrEmpty(l.LinkTitle))
                l.LinkTitle = Formatter.FormatSingleLinePlainTextForStorage(l.LinkTitle);

            if (l.LinkType == LinkTypes.Image)
            {
                l.LinkType = (LinkTypes)Request.Form.GetInt("SiteLinkTypes", 1);
                if (l.LinkType == LinkTypes.Image)
                {
                    System.Web.HttpPostedFileBase picFile = null;
                    if (Request.Files != null && Request.Files.Count > 0)
                        picFile = Request.Files[0];
                    if (picFile != null && !string.IsNullOrEmpty(picFile.FileName))
                    {
                        if (picFile.ContentType.IndexOf("image") == -1)
                        {
                            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditLink", "Auxiliaries", new RouteValueDictionary { { "linkID", linkID }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode("请选择图片文件进行上传") } }));
                        }
                        SLManage.UpdateLink(l, picFile);
                    }
                    else
                    {
                        SLManage.UpdateLink(l, null);
                    }
                }
                else
                {
                    SLManage.UpdateLink(l, null);
                }
            }
            else
            {
                l.LinkType = (LinkTypes)Request.Form.GetInt("SiteLinkTypes", 1);
                if (l.LinkType == LinkTypes.Image)
                {
                    System.Web.HttpPostedFileBase picFile = null;
                    if (Request.Files != null && Request.Files.Count > 0)
                        picFile = Request.Files[0];
                    if (picFile != null)
                    {
                        if (picFile.ContentType.IndexOf("image") == -1)
                        {
                            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditLink", "Auxiliaries", new RouteValueDictionary { { "linkID", linkID }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode("请选择图片文件进行上传") } }));
                        }
                        SLManage.UpdateLink(l, picFile);
                    }
                }
                else
                {
                    SLManage.UpdateLink(l, null);
                }
            }

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditLink", "Auxiliaries", new RouteValueDictionary { { "linkID", linkID }, { "StatusMessageType", (int)StatusMessageType.Success }, { "StatusMessageContent", WebUtils.UrlEncode("修改成功！") } }));
        }
        #endregion

        #region 友情连接类别
        /// <summary>
        /// 显示友情连接类别
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult ManageLinkCategories()
        {
            SetPageTitle("友情链接类别管理");
            ViewData["LinkCategories"] = SiteLinkManager.Instance().GetAllLinkCategories(true);
            return View("Pages/Auxiliaries/ManageLinkCategories.aspx", "Masters/AuxiliaryModule.Master");
        }
        /// <summary>
        /// 显示添加友情类别
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_AddLinkCategory()
        {
            return PartialView("Controls/Auxiliaries/EditLinkCategory.ascx");
        }
        /// <summary>
        /// 添加友情类别
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_AddLinkCategoryAction()
        {
            LinkCategory lc = new LinkCategory();
            lc.CategoryName = Request.Form.GetString("category", string.Empty);
            lc.DisplayOrder = Request.Form.GetInt("displayorder", -1);
            lc.Description = Request.Form.GetString("decript", string.Empty);

            if (!string.IsNullOrEmpty(lc.CategoryName))
                lc.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(lc.CategoryName);
            if (!string.IsNullOrEmpty(lc.Description))
                lc.Description = Formatter.FormatMultiLinePlainTextForStorage(lc.Description, true);


            if (lc.DisplayOrder < 0 || string.IsNullOrEmpty(lc.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "添加失败,注意类别名称不能为空且排序序号必须为非负整数！";
            }
            else
            {
                SiteLinkManager SLManage = SiteLinkManager.Instance();
                SLManage.CreateUserLinkCategory(lc);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "添加成功";
            }
            return RedirectToAction("Control_AddLinkCategory");
        }
        /// <summary>
        ///  显示编辑友情类别页
        /// </summary>
        /// <param name="CateID">类别id</param>
        /// <returns></returns>
        public ActionResult Control_EditLinkCategory(int CateID)
        {
            LinkCategory LinkCate = SiteLinkManager.Instance().GetLinkCategory(CateID, true);
            ViewData["LinkCate"] = LinkCate;
            return PartialView("Controls/Auxiliaries/EditLinkCategory.ascx");
        }
        /// <summary>
        /// 更新友情类别
        /// </summary>
        /// <param name="CateID">类别id</param>
        /// <returns></returns>
        public ActionResult Control_EditLinkCategoryAction(int CateID)
        {
            LinkCategory lc = SiteLinkManager.Instance().GetLinkCategory(CateID, true);
            if (lc == null)
                throw new ArgumentException("连接类别错误");
            lc.CategoryName = Request.Form.GetString("category", string.Empty);
            lc.DisplayOrder = Request.Form.GetInt("displayorder", -1);
            lc.Description = Request.Form.GetString("decript", string.Empty);

            if (!string.IsNullOrEmpty(lc.CategoryName))
                lc.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(lc.CategoryName);
            if (!string.IsNullOrEmpty(lc.Description))
                lc.Description = Formatter.FormatMultiLinePlainTextForStorage(lc.Description, true);

            if (lc.DisplayOrder < 0 || string.IsNullOrEmpty(lc.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "修改失败,注意类别名称不能为空且排序序号必须为非负整数！";
            }
            else
            {
                SiteLinkManager.Instance().UpdateCategory(lc);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "修改成功";
            }
            return RedirectToAction("Control_EditLinkCategory", new RouteValueDictionary { { "CateID", CateID } });
        }
        /// <summary>
        /// 删除友情类别
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteLinkCategoryAction()
        {
            string strCategroies = Request.Form.GetString("CategroyGroup", string.Empty);
            string[] Categroies = strCategroies.Split(',');
            List<int> CateIDs = new List<int>();
            foreach (var item in Categroies)
            {
                try
                {
                    CateIDs.Add(Convert.ToInt32(item));
                }
                catch { }
            }
            SiteLinkManager.Instance().DeleteCategories(CateIDs);
            return RedirectToAction("ManageLinkCategories");
        }

        #endregion

        #region AdvertisingPosition

        #region Show

        /// <summary>
        /// 管理广告位
        /// </summary>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult ManageAdvertisingPositions()
        {
            SetPageTitle("管理广告位");
            List<AdvertisingPosition> position = Advertisings.GetAdvertisingPositions(null);
            return View("Pages/Auxiliaries/ManageAdvertisingPositions.aspx", "Masters/AuxiliaryModule.Master", position);
        }

        /// <summary>
        /// 编辑广告位
        /// </summary>
        /// <param name="positionCode">位置编码</param>
        /// <param name="isCreate">创建为True，更新为False</param>
        /// <returns></returns>
        public ActionResult Control_EditAdvertisingPosition(string positionCode, bool isCreate)
        {
            AdvertisingPosition position = new AdvertisingPosition();

            if (isCreate)
            {
                SetPageTitle("新建广告位");
            }
            else
            {
                SetPageTitle("编辑广告位");
                position = Advertisings.GetAdvertisingPositionForAdmin(positionCode);
                if (position == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
            }
            ViewData["isCreate"] = isCreate;

            int statusMessageType = Request.QueryString.GetInt("StatusMessageType", -2);
            string statusMessageContent = WebUtils.UrlDecode(Request.QueryString.GetString("StatusMessageContent", string.Empty));
            if (statusMessageType > -2)
            {
                TempData["StatusMessageType"] = statusMessageType;
            }

            if (!string.IsNullOrEmpty(statusMessageContent))
            {
                TempData["StatusMessageContent"] = statusMessageContent;
            }

            return PartialView("Controls/Auxiliaries/EditAdvertisingPosition.ascx", position);
        }

        /// <summary>
        /// 显示该广告位下的广告
        /// </summary>
        /// <param name="positionCode"></param>
        /// <returns></returns>
        public ActionResult ListAdvertisingsOfPosition(string positionCode)
        {

            AdvertisingPosition position = Advertisings.GetAdvertisingPositionForAdmin(positionCode);
            SetPageTitle(string.Format("\"{0}\"下的广告", position.Description));
            if (position == null)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
            ViewData["position"] = position;

            IList<Advertising> advertisings = Advertisings.GetAdvertisingsOfPosition(positionCode, null, true, true);
            return View("Pages/Auxiliaries/ListAdvertisingsOfPosition.aspx", "Masters/AuxiliaryModule.Master", advertisings);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新广告位
        /// </summary>
        /// <param name="positionCode"></param>
        /// <param name="isCreate"></param>
        /// <returns></returns>
        public ActionResult CreateUpdateAdvertisingPosition(string positionCode, bool isCreate)
        {
            int currentUserID = int.Parse(EncryptManager.DecryptTokenForInviteFriend(Request.Form["currentUserID"]));

            // 判断用户是否有上传图片的权限
            User currentUser = Users.GetUser(currentUserID, false);

            if (currentUser == null || !currentUser.IsContentAdministrator)
                throw new SPBException(ExceptionTypes.Instance().AccessDenied(), "您无权设置广告位");

            AdvertisingPosition position = new AdvertisingPosition();

            if (isCreate)
            {
                position.PositionCode = Request.Form.GetString("positionCode", string.Empty);
            }
            else
            {
                position = Advertisings.GetAdvertisingPositionForAdmin(positionCode);
                if (position == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
            }

            if (isCreate && Advertisings.GetAdvertisingPositionForAdmin(position.PositionCode) != null)
            {
                return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditAdvertisingPosition", "Auxiliaries", new RouteValueDictionary { { "positionCode", positionCode }, { "isCreate", isCreate }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode("该位置编码已经使用，请换个位置编码") } }));
            }

            position.Description = Request.Form.GetString("description", string.Empty);
            position.AllowSystem = Request.Form.GetBool("allowSystem", true);
            position.PresentAreaID = Request.Form.GetInt("PresentAreaID", 10);
            position.Height = Request.Form.GetInt("height", 100);
            position.Width = Request.Form.GetInt("width", 100);

            position.Maximum = Request.Form.GetInt("maximum", 0);

            System.Web.HttpPostedFileBase hpf = null;
            if ((Request.Files["imageUrl"] as System.Web.HttpPostedFileBase).ContentLength > 0 && !string.IsNullOrEmpty((Request.Files["imageUrl"] as System.Web.HttpPostedFileBase).FileName))
            {
                hpf = Request.Files["imageUrl"] as System.Web.HttpPostedFileBase;

                if (!string.IsNullOrEmpty(hpf.FileName))
                {
                    if (hpf.ContentType.IndexOf("image") < 0)
                    {
                        return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditAdvertisingPosition", "Auxiliaries", new RouteValueDictionary { { "positionCode", positionCode }, { "isCreate", isCreate }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode("只允许上传图片") } }));
                    }
                }
            }

            int statusMessageType = 0;
            string statusMessageContent = string.Empty;

            if (isCreate)
            {
                try
                {
                    if (hpf != null)
                    {
                        Advertisings.CreateAdvertisingPosition(position, hpf.InputStream);
                        hpf.InputStream.Close();
                        hpf.InputStream.Dispose();
                    }

                    else
                        Advertisings.CreateAdvertisingPosition(position, null);

                    statusMessageType = (int)StatusMessageType.Success;
                }
                catch (Exception)
                {
                    statusMessageType = (int)StatusMessageType.Error;
                    statusMessageContent = "创建广告位失败，请重试";
                }
            }
            else
            {
                if (hpf != null)
                    Advertisings.UpdateAdvertisingPosition(position, hpf.InputStream);
                else
                    Advertisings.UpdateAdvertisingPosition(position, null);

                statusMessageType = (int)StatusMessageType.Success;
            }

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditAdvertisingPosition", "Auxiliaries", new RouteValueDictionary { { "positionCode", positionCode }, { "isCreate", isCreate }, { "StatusMessageType", statusMessageType }, { "StatusMessageContent", WebUtils.UrlEncode(statusMessageContent) } }));
        }

        /// <summary>
        /// 删除广告位
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteAdvertisingPositions()
        {
            string[] positionCodes = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("positionCheckBoxGroup", string.Empty)))
            {
                positionCodes = Request.Form.GetString("positionCheckBoxGroup", string.Empty).Split(',');
                if (positionCodes != null && positionCodes.Length > 0)
                {
                    List<string> positionCodeList = positionCodes.ToList<string>();
                    Advertisings.DeleteAdvertisingPositions(positionCodeList);
                }
            }

            return RedirectToAction("ManageAdvertisingPositions");
        }

        #endregion

        #endregion

        #region Advertising

        #region Show

        /// <summary>
        /// 管理广告
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult ManageAdvertisings(int? pageIndex)
        {
            SetPageTitle("管理广告");
            PagingDataSet<Advertising> pds = Advertisings.GetAdvertisings(20, pageIndex ?? 1, PresentAreaIDs.Instance().Channel(), null);
            return View("Pages/Auxiliaries/ManageAdvertisings.aspx", "Masters/AuxiliaryModule.Master", pds);
        }

        /// <summary>
        /// 编辑广告
        /// </summary>
        /// <param name="advertisingID"></param>
        /// <returns></returns>
        public ActionResult EditAdvertising(int? advertisingID)
        {
            Advertising advertising = new Advertising();

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                SetPageTitle("编辑广告");
                advertising = Advertisings.GetAdvertising(advertisingID.Value, true);
                if (advertising == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
            }
            else
            {
                SetPageTitle("发布广告");
                if (TempData["Advertising"] != null)
                {
                    advertising = TempData["Advertising"] as Advertising;
                }
                else
                {
                    advertising.EffectiveDateFrom = DateTime.Now;
                    advertising.EffectiveDateTo = advertising.EffectiveDateFrom.AddMonths(1);
                }
            }

            return View("Pages/Auxiliaries/EditAdvertising.aspx", "Masters/AuxiliaryModule.Master", advertising);
        }

        /// <summary>
        /// 广告投放时显示的广告位列表
        /// </summary>
        /// <param name="selectedPosition"></param>
        /// <param name="width"></param>
        /// <returns></returns>
        public ActionResult Control_ListAdvertisingPositions(int? advertisingID, int? width)
        {
            List<AdvertisingPosition> position = Advertisings.GetAdvertisingPositions(true);

            List<int> widthList = Advertisings.GetAdvertisingPositionWidths();
            ViewData["widthList"] = widthList;

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                Advertising advertising = Advertisings.GetAdvertising(advertisingID.Value, true);
                if (advertising == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }

                ViewData["selectedPosition"] = advertising.UsedAdvertisingPositions;
            }

            if (width.HasValue && width.Value > 0)
            {
                ViewData["currentWidth"] = width.Value;
            }

            return PartialView("Controls/Auxiliaries/ListAdvertisingPositions.ascx", position);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新广告
        /// </summary>
        /// <param name="advertisingID"></param>
        /// <returns></returns>
        public ActionResult CreateUpdateAdvertising(int? advertisingID)
        {
            Advertising advertising = new Advertising();

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                advertising = Advertisings.GetAdvertising(advertisingID.Value, true);
                if (advertising == null)
                {
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                }
            }

            advertising.AdvertisingName = Request.Form.GetString("advertisingName", string.Empty).Trim();
            advertising.AdvertisingType = (AdvertisingTypes)Request.Form.GetInt("advertisingType", 0);
            advertising.IsEnabled = Request.Form.GetBool("isEnabled", true);
            advertising.PresentAreaID = PresentAreaIDs.Instance().Channel();
            advertising.Script = Request.Form.GetString("script", string.Empty).Trim();

            DateTime datetime;
            if (DateTime.TryParse(Request.Form.GetString("effectiveDateFrom", string.Empty).Trim(), out datetime))
            {
                advertising.EffectiveDateFrom = datetime;
            }
            else
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                {
                    TempData["Advertising"] = advertising;
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = "开始时间格式不正确";
                return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "advertisingID", advertisingID } });
            }

            if (DateTime.TryParse(Request.Form.GetString("effectiveDateTo", string.Empty).Trim(), out datetime))
            {
                advertising.EffectiveDateTo = datetime;
            }
            else
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                {
                    TempData["Advertising"] = advertising;
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = "结束时间格式不正确";
                return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "advertisingID", advertisingID } });
            }

            if (advertising.EffectiveDateFrom > advertising.EffectiveDateTo)
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                {
                    TempData["Advertising"] = advertising;
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = "开始时间不能晚于结束时间";
                return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "advertisingID", advertisingID } });
            }

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                Advertisings.UpdateAdvertising(advertising);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            else
            {
                try
                {
                    Advertisings.CreateAdvertising(advertising);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                }
                catch (Exception)
                {
                    TempData["Advertising"] = advertising;
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "创建广告失败，请重试";
                }
            }

            string[] positionCodes = null;
            List<string> positionCodeList = new List<string>();
            bool isError = false;
            if (!string.IsNullOrEmpty(Request.Form.GetString("positionCheckBoxGroup", string.Empty)))
            {
                positionCodes = Request.Form.GetString("positionCheckBoxGroup", string.Empty).Split(',');
                if (positionCodes != null && positionCodes.Length > 0)
                {
                    List<AdvertisingPosition> selectPosition = advertising.UsedAdvertisingPositions;
                    foreach (var positionCode in positionCodes)
                    {
                        AdvertisingPosition position = Advertisings.GetAdvertisingPositionForAdmin(positionCode);
                        if (selectPosition.Contains(position) || position.CanAddAdvertisings() || position.PresentAreaID != PresentAreaIDs.Instance().Channel())
                        {
                            positionCodeList.Add(positionCode);
                        }
                        else
                        {
                            if (isError)
                                continue;
                            isError = true;
                        }
                    }
                }
            }

            if (isError)
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                    TempData["Advertising"] = advertising;

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = "投放的广告数超过广告位最大限制，部分广告会投放失败";
            }

            Advertisings.UpdatePositionsOfAdvertisingForAdmin(advertising.AdvertisingID, positionCodeList);

            return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "advertisingID", advertisingID } });
        }

        /// <summary>
        /// 删除广告
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteAdvertisings()
        {
            string[] advertisingCodes = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("advertisingCheckBoxGroup", string.Empty)))
            {
                advertisingCodes = Request.Form.GetString("advertisingCheckBoxGroup", string.Empty).Split(',');
                if (advertisingCodes != null && advertisingCodes.Length > 0)
                {
                    List<int> advertisingIDs = new List<int>();
                    foreach (var id in advertisingCodes)
                    {
                        if (Convert.ToInt32(id) > 0)
                            advertisingIDs.Add(Convert.ToInt32(id));
                    }
                    Advertisings.DeleteAdvertisings(0, advertisingIDs.ToArray());
                }
            }

            return RedirectToAction("ManageAdvertisings");
        }

        #endregion

        #endregion

        #endregion

        #region 推荐
        #region 信息推荐
        /// <summary>
        /// 管理推荐信息
        /// </summary>
        public ActionResult ManageRecommendedItems(int? commendTypeID, int? pageIndex)
        {
            SetPageTitle("管理推荐信息");
            ViewData["CommendTypeID"] = commendTypeID;
            ViewData["PageIndex"] = pageIndex;
            List<Area> listArea = Areas.GetRootAreas(false);
            return View("Pages/Auxiliaries/ManageRecommendedItems.aspx", "Masters/AuxiliaryModule.Master");
        }

        /// <summary>
        /// 显示推荐信息列表控件
        /// </summary>
        public ActionResult Control_ManageRecommendedItems(int? commendTypeID, int? pageIndex)
        {
            PagingDataSet<RecommendedItem> pds = RecommendedItems.GetRecommendedItems(17, pageIndex ?? 1, null, commendTypeID);

            ViewData["RecommendationTypeDictionary"] = RecommendationTypes.GetRecommendationTypeDictionary(false);
            ViewData["CommendTypeID"] = commendTypeID;
            ViewData["PageIndex"] = pageIndex;
            return PartialView("Controls/Auxiliaries/ManageRecommendedItems.ascx", pds);
        }

        /// <summary>
        /// 显示信息推荐项编辑模式框
        /// </summary>
        public ActionResult Control_EditRecommendedItem(int commendID)
        {
            RecommendedItem item = RecommendedItems.GetRecommendedItem(commendID, true);
            RecommendationType type = RecommendationTypes.GetRecommendationType(item.CommendTypeID, false);
            ViewData["TypeName"] = type.Name;
            return PartialView("Controls/Auxiliaries/EditRecommendedItem.ascx", item);
        }

        /// <summary>
        /// 更新信息推荐项
        /// </summary>
        public ActionResult UpdateRecommendedItem(int commendID)
        {
            RecommendedItem item = RecommendedItems.GetRecommendedItem(commendID, true);
            if (item != null)
            {
                item.DisplayOrder = Request.Form.GetInt("DisplayOrder", 100);
                RecommendedItems.UpdateRecommendedItem(item);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "更新信息推荐项操作成功";
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "更新信息推荐项操作失败，找不到该信息推荐项";
            }
            return RedirectToAction("Control_EditRecommendedItem", new RouteValueDictionary { { "commendID", commendID } });
        }

        /// <summary>
        /// 删除一条信息推荐
        /// </summary>
        public ActionResult DeleteRecommendedItem(int commendID, int? commendTypeID, int? pageIndex)
        {
            RecommendedItem item = RecommendedItems.GetRecommendedItem(commendID, true);
            if (item != null)
            {
                RecommendedItems.DeleteRecommendedItem(commendID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "删除信息推荐操作成功";
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "删除信息推荐操作失败，原因是找不到该推荐项";
            }
            return RedirectToAction("Control_ManageRecommendedItems", new RouteValueDictionary { { "commendTypeID", commendTypeID }, { "pageIndex", pageIndex } });
        }

        #endregion

        #region 信息推荐类别
        /// <summary>
        /// 管理推荐信息类别
        /// </summary>
        public ActionResult ManageRecommendationTypes()
        {
            SetPageTitle("管理推荐信息类别");
            return View("Pages/Auxiliaries/ManageRecommendationTypes.aspx", "Masters/AuxiliaryModule.Master");
        }
        /// <summary>
        /// 显示推荐信息类别列表控件
        /// </summary>
        public ActionResult Control_ManageRecommendationTypes()
        {
            Dictionary<int, RecommendationType> dictionary = RecommendationTypes.GetRecommendationTypeDictionary(true);
            Dictionary<int, Application> applications = Applications.GetApplicationsForAdmin();
            ViewData["Applications"] = applications;
            return PartialView("Controls/Auxiliaries/ManageRecommendationTypes.ascx", dictionary);
        }

        /// <summary>
        /// 编辑信息推荐类别模式框
        /// </summary>
        public ActionResult Control_EditRecommendationType(int? typeID)
        {
            RecommendationType type = null;
            if (typeID.HasValue && typeID.Value > 0)
                type = RecommendationTypes.GetRecommendationType(typeID.Value, true);
            else
                type = new RecommendationType();
            Dictionary<int, Application> applications = Applications.GetApplicationsForAdmin();
            ViewData["Applications"] = applications;
            ViewData["TypeID"] = typeID;
            return PartialView("Controls/Auxiliaries/EditRecommendationType.ascx", type);
        }

        /// <summary>
        /// 更新信息推荐类别
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateRecommendationType(int recommendTypeID)
        {
            RecommendationType type = null;
            if (recommendTypeID > 0)
                type = RecommendationTypes.GetRecommendationType(recommendTypeID, true);
            else
            {
                int typeIDvalue = Request.Form.GetInt("typeID", -1);
                if (RecommendationTypes.IsTypeIDExist(typeIDvalue))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format("已经存在TypeID为{0}的推荐类别，请选择推荐类别列表中尚未使用的TypeID", typeIDvalue);
                    return RedirectToAction("Control_EditRecommendationType");
                }
                type = new RecommendationType();
                type.TypeID = typeIDvalue;
            }
            type.Name = Request.Form.GetString("typeName", string.Empty);
            type.ApplicationID = Request.Form.GetInt("applicationID", -1);
            type.Description = Request.Form.GetString("description", string.Empty);
            RecommendationTypes.UpdateRecommendedItemType(type);

            Dictionary<string, string> recommendationTypeRes = new Dictionary<string, string>();
            foreach (Language language in Languages.GetLanguages(true).Values)
            {
                string recommendName = Request.Form.GetString("RecommendationType_" + language.LanguageCode, string.Empty);
                if (!string.IsNullOrEmpty(recommendName))
                    recommendationTypeRes[language.LanguageCode] = recommendName;
            }


            if (recommendTypeID > 0)
            {
                if (type == null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "更新站点类别操作失败，找不到所要更新的推荐类别";
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "更新推荐类别操作成功";
                    if (recommendationTypeRes.Count > 0)
                        RecommendationTypes.UpdateRecommendationTypeRes(type.TypeID, recommendationTypeRes);
                }
            }
            else
            {
                RecommendationTypes.Create(type);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "创建推荐类别操作成功";
                if (recommendationTypeRes.Count > 0)
                    RecommendationTypes.UpdateRecommendationTypeRes(type.TypeID, recommendationTypeRes);
            }
            return RedirectToAction("Control_EditRecommendationType", new RouteValueDictionary { { "typeID", recommendTypeID } });
        }

        /// <summary>
        /// 删除一个信息推荐类别
        /// </summary>
        public ActionResult DeleteRecommendationType(int typeID)
        {
            RecommendationType type = RecommendationTypes.GetRecommendationType(typeID, true);
            if (type != null)
            {
                RecommendationTypes.DeleteRecommendedItemType(typeID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "删除信息推荐类别操作成功";
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "删除信息推荐类别操作失败，原因是找不到该推荐类别";
            }
            return RedirectToAction("Control_ManageRecommendationTypes");
        }
        #endregion
        #endregion

        #region 公告
        /// <summary>
        /// 编辑公告
        /// </summary>

        public ActionResult EditAnnouncement(int? threadID)
        {

            Announcement announcement = null;
            if (threadID.HasValue && threadID.Value > 0)
            {
                announcement = Announcements.GetAnnouncement(threadID.Value, true);
                SetPageTitle("编辑公告");
            }
            if (TempData["announcement"] != null)
                announcement = TempData.Get<Announcement>("announcement", new Announcement());

            if (announcement == null)
            {
                announcement = new Announcement();
                SetPageTitle("创建公告");
            }
            return View("Pages/Auxiliaries/EditAnnouncement.aspx", "Masters/AuxiliaryModule.Master", announcement);
        }
        /// <summary>
        /// 创建、更新公告
        /// </summary>     
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUpdateAnnouncement(int? threadID)
        {
            Announcement announcement = new Announcement();
            SetPageTitle("创建公告");
            if (threadID.HasValue && threadID.Value > 0)
            {
                announcement = Announcements.GetAnnouncement(threadID.Value, true);
                if (announcement == null)
                    return Redirect(SiteUrls.Instance().Error404(string.Format("找不到ID为{0}的公告", threadID.Value)));
                SetPageTitle("编辑公告");

            }
            else
            {
                announcement = new Announcement();
                SetPageTitle("创建公告");

            }

            string[] selectedPresentAreaIDsStringArray = Request.Form.GetString("PresentAreaIDs", string.Empty).Split(',');
            announcement.PresentAreaIDs.Clear();
            foreach (var strPresentAreaID in selectedPresentAreaIDsStringArray)
            {
                int presentAreaID = 0;
                int.TryParse(strPresentAreaID, out presentAreaID);
                if (presentAreaID > 0)
                    announcement.PresentAreaIDs.Add(presentAreaID);

            }

            announcement.Subject = Request.Form.GetString("title", string.Empty);
            announcement.SubjectStyle = Request.Form.GetString("SubjectStyle", string.Empty);
           // announcement.Body = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("body", string.Empty));
            announcement.Body = HtmlScrubber.Clean(Request.Form.GetString("body",string.Empty),false,true);
            //敏感词过滤
            if (SiteSettingsManager.GetSiteSettings().EnableCensorship)
            {
                announcement.Body = Censors.CensorPost(announcement.Body);
            }
            announcement.DateExpired = DateTime.Parse(Request.Form.GetString("dateExpired", string.Empty));

            bool Immediately = Request.Form.GetBool("Immediately", false);

            if (Immediately)
                announcement.ReleaseDate = DateTime.Now;
            else
                announcement.ReleaseDate = DateTime.Parse(Request.Form.GetString("releaseDate", string.Empty));
            

            announcement.IsHyperLink = Request.Form.GetBool("isHyperLink", false);
            announcement.IsSticky = Request.Form.GetBool("isSticky", false);
            announcement.DisplayOrder = Request.Form.GetInt("displayOrder", 100);
            announcement.UserID = CurrentUser.UserID;
            announcement.HyperLinkUrl = Request.Form.GetString("isHyper", string.Empty);
            if (announcement.IsHyperLink && string.IsNullOrEmpty(announcement.HyperLinkUrl))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "外链地址不能为空！");
                TempData["announcement"] = announcement;
                return RedirectToAction("EditAnnouncement", threadID = threadID.Value);

            }

            if (!announcement.IsHyperLink && string.IsNullOrEmpty(announcement.Body))
            {

                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "公告内容不能为空！");
                TempData["announcement"] = announcement;
                return RedirectToAction("EditAnnouncement", threadID = threadID.Value);
 
            }
            if (announcement.DateExpired <= announcement.ReleaseDate)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "公告过期时间不小于发布时间！");
                TempData["announcement"] = announcement;
                return RedirectToAction("EditAnnouncement", threadID = threadID.Value);
            }

            if (announcement.ReleaseDate.CompareTo(DateTime.Now) < 0)
            {
                announcement.ReleaseDate = DateTime.Now;
 
            }

            if (threadID.HasValue && threadID.Value > 0)
            {
                Announcements.UpdateAnnouncement(announcement);
            }
            else
            {
                Announcements.CreateAnnouncement(announcement);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "您创建的公告五分钟以后在前台显示";


            return RedirectToAction("ManageAnnouncements", new { threadID = threadID });

        }
        /// <summary>
        /// 管理公告
        /// </summary>
        public ActionResult ManageAnnouncements(int? presentAreaID, int? pageIndex, string searchKeywords)
        {
            SetPageTitle("管理公告");
            searchKeywords = Request.QueryString.GetString("searchKeywords", string.Empty);
            if (!pageIndex.HasValue || pageIndex.Value <= 0)
                pageIndex = 1;
            PagingDataSet<Announcement> announcement = Announcements.GetAnnouncementsForAdmin(presentAreaID, null, searchKeywords, 15, pageIndex ?? 1);


            return View("Pages/Auxiliaries/ManageAnnouncements.aspx", "Masters/AuxiliaryModule.Master", announcement);

        }
        /// <summary>
        /// 删除公告
        /// </summary>
        public ActionResult DeleteAnnouncement(int threadID)
        {
            Announcement announcement = Announcements.GetAnnouncement(threadID, true);
            if (announcement == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("不存在ID为{0}的公告", threadID);
                return Redirect(Request.UrlReferrer.AbsoluteUri);
            }
            Announcements.DeleteAnnouncement(threadID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除成功";
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }
        /// <summary>
        /// 批量删除公告
        /// </summary>
        public ActionResult DeleteAnnouncements(string returnUrl)
        {
            string selectPostIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectPostIDsString))
            {
                string[] selectedPostIDsStringArray = selectPostIDsString.Split(',');

                if (selectedPostIDsStringArray != null && selectedPostIDsStringArray.Length > 0)
                {
                    List<int> IDs = new List<int>();
                    foreach (string id in selectedPostIDsStringArray)
                    {
                        int addID = 0;
                        if (int.TryParse(id, out addID) && addID > 0)

                            Announcements.DeleteAnnouncement(addID);
                        //IDs.Add(addID);
                    }
                    // Announcements.DeleteAnnouncement
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "批量删除公告成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要删除的公告";
            }

            return Redirect(returnUrl);
        }
        /// <summary>
        /// 公告置顶显示
        /// </summary>
        //@preview: SetTop-> SetSticky
        public ActionResult SetSticky(int threadID, bool isSticky)
        {
            if (threadID > 0)
            {
                Announcements.SetSticky(threadID, isSticky);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("不存在ID为{0}的公告", threadID);
            }

            return RedirectToAction("ManageAnnouncements");
        }

        /// <summary>
        ///搜索公告
        /// </summary>        
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchAnnouncement(int? presentAresID)
        {
            string searchKeywords = Request.Form.GetString("searchKeywords", string.Empty).Trim();
            return RedirectToAction("ManageAnnouncements", new RouteValueDictionary { { "presentAreaID", presentAresID }, { "searchKeywords", searchKeywords } });
        }
        #endregion

    }

    public enum AuxiliaryModuleMenu
    {
        /// <summary>
        /// 消息反馈
        /// </summary>
        InformationFeedback = 101,

        /// <summary>
        /// 信息推荐
        /// </summary>
        InformationRecommendation = 102,

        /// <summary>
        /// 广告管理
        /// </summary>
        ManageAdvertising = 103,

        /// <summary>
        /// 友情链接
        /// </summary>
        ManageLinks = 104,

        /// <summary>
        /// 积分兑换
        /// </summary>
        ExchangePoints = 105,
        /// <summary>
        /// 公告
        /// </summary>
        Announcements = 106

    }
    public enum AuxiliaryModuleSubMenu
    {
        /// <summary>
        /// 收件箱
        /// </summary>
        MessageInbox = 1001,

        /// <summary>
        /// 发件箱
        /// </summary>
        MessageOutbox = 1002,

        /// <summary>
        /// 用户举报
        /// </summary>
        UserReports = 1003,

        /// <summary>
        /// 群发消息
        /// </summary>
        MassSystemMessage = 1004,

        //// <summary>
        /// 推荐信息管理
        /// </summary>
        ManageReCommendedItems = 1005,

        /// <summary>
        /// 推荐信息类别管理
        /// </summary>
        ManageRecommendationTypes = 1006,

        /// <summary>
        /// 广告管理
        /// </summary>
        ManageAdvertisings = 1007,

        /// <summary>
        /// 管理广告位
        /// </summary>
        ManageAdvertisingPositions = 1008,

        /// <summary>
        /// 友情链接
        /// </summary>
        ManageLinks = 1009,

        /// <summary>
        /// 友情链接类别管理
        /// </summary>
        ManageLinkCategory = 1010,

        /// <summary>
        /// 礼品管理
        /// </summary>
        ManageConvertibleGift = 1011,

        /// <summary>
        /// 管理礼品兑换
        /// </summary>
        ManageGiftExchangeApplie = 1012,

        /// <summary>
        /// 创建公告
        /// </summary>
        EditAnnouncement = 1013,
        /// <summary>
        /// 管理公告
        /// </summary>
        ManageAnnouncements = 1014



    }

    /// <summary>
    /// 消息传输方式
    /// </summary>
    public enum TransportModes
    {
        /// <summary>
        /// 站内私信
        /// </summary>
        PrivateMessage = 0,

        /// <summary>
        /// 电子邮件
        /// </summary>
        Email = 1,

        ///// <summary>
        ///// 手机短信
        ///// </summary>
        //Mobile = 2
    }
    /// <summary>
    /// 群发消息的选择对象类型
    /// </summary>
    public enum ReceiversTypes
    {
        /// <summary>
        /// 选择所有会员
        /// </summary>
        AllUsers = 0,
        /// <summary>
        /// 按会员角色进行选择
        /// </summary>
        UserRole = 1,
        /// <summary>
        /// 按会员级别进行选择
        /// </summary>
        UserRank = 2

    }
}
