﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using System.Security.Principal;
using SpaceBuilder.Common;

using SpaceBuilder.Web;

namespace SpaceBuilder.Common.Controllers
{
    /// <summary>
    /// 空间中Controller的基类
    /// </summary>
    public abstract class UserDomainControllerBase : SPBControllerBase
    {

        public override PresentAreaThemeManager ThemeManager
        {
            get
            {
                User currentDomainUser = GetCurrentDomainUser(ControllerContext.GetParameterFromRouteDateOrQueryString("userDomainName"));
                if (currentDomainUser.UserType == UserTypes.PersonUser)
                    return UserPresentAreaThemeManager.Instance();
                else
                    return CompanyUserPresentAreaThemeManager.Instance();
            }
        }


        #region 空间中共用方法

        /// <summary>
        /// 设置Html文件的title
        /// </summary>
        /// <param name="text">title名称</param>
        protected virtual void SetPageTitle(string userDomainName, string text)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                return;
            if (string.IsNullOrEmpty(text))
                Header.AddTitle(string.Format("{1}{0}{2}", Globals.BrowserTitleSeparator, currentDomainUser.DisplayName, Globals.SiteName));
            else
                Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, text, currentDomainUser.DisplayName, Globals.SiteName));
        }

        /// <summary>
        /// 获取当前空间拥有者
        /// </summary>
        /// <param name="userDomainName">空间拥有者的用户名</param>
        /// <returns>当前空间用户</returns>
        protected User GetCurrentDomainUser(string userDomainName)
        {
            User currentDomainUser;
            if (IsUserDomainOwner(userDomainName))
                currentDomainUser = this.CurrentUser;
            else
                currentDomainUser = Users.GetUser(userDomainName, false);

            if (currentDomainUser == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserNotFound"), userDomainName)));
            }
            return currentDomainUser;
        }

        /// <summary>
        /// 当前用户是否当前空间的所有者
        /// </summary>
        protected bool IsUserDomainOwner(string userDomainName)
        {
            if (CurrentUser == null)
                return false;
            return CurrentUser.UserName.Equals(userDomainName, StringComparison.CurrentCultureIgnoreCase);
        }

        /// <summary>
        /// 验证当前用户是否操作自己的item
        /// </summary>
        /// <param name="itemUserID">内容的拥有者ID</param>
        protected void ValidateItemOwner(int itemUserID)
        {
            if (CurrentUser == null || (CurrentUser != null && CurrentUser.UserID != itemUserID && !CurrentUser.IsContentAdministrator))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
        }

        /// <summary>
        /// 评论删除权限控制
        /// </summary>
        /// <param name="ownerUserID">被评论用户UserID</param>
        /// <param name="userID">评论用户UserID</param>
        protected void ValidateCommentDeleteAuthority(int ownerUserID, int userID)
        {

            if (CurrentUser == null || (CurrentUser != null && CurrentUser.UserID != ownerUserID && CurrentUser.UserID != userID && !CurrentUser.IsContentAdministrator))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
        }

        protected string GetGenderName(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            string genderName = "我";
            if (CurrentUser != null && currentDomainUser.UserID != CurrentUser.UserID)
            {
                PersonUser pu = currentDomainUser as PersonUser;
                if (pu != null)
                {
                    genderName = "他";
                    if (pu.Gender == Genders.Female)
                        genderName = "她";
                }
            }
            return genderName;
        }

        /// <summary>
        /// 是否具有管理空间权限
        /// </summary>
        protected bool HasManagePermission(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            return (CurrentUser != null && (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }
        /// <summary>
        /// 是否具有管理空间权限
        /// </summary>
        protected bool HasManagePermission(User currentDomainUser)
        {
            return (CurrentUser != null && (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }

        #endregion

        #region 辅助方法

        /// <summary>
        /// 设置浏览计数
        /// </summary>
        /// <param name="type">类型</param>
        /// <param name="itemID">浏览次数</param>
        protected void AddWebCount<T>(int itemID) where T : Counter
        {
            CounterFactory.IncreaseCount<T>(IncreasableCountTypes.WebView, itemID);
        }

        #endregion
    }
}
