﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using System.Web.Security;
using SpaceBuilder.Common;
using System.Text.RegularExpressions;
using System.Web.UI;
using SpaceBuilder.Utils;
using System.Web.Routing;
using System.Web;
using SpaceBuilder.Web;
using SpaceBuilder.LuceneSearch;
using System.Configuration;
using SpaceBuilder.Club;
using System.Diagnostics;
using SpaceBuilder.Caching;
using System.Web.Script.Serialization;

namespace SpaceBuilder.Common.Controllers
{
    [HttpCompress]
    public class ChannelController : ChannelControllerBase
    {
        /// <summary>
        /// 总站首页
        /// </summary>
        public ActionResult Home()
        {
            if (!Request.IsAuthenticated && SiteSettingsManager.GetSiteSettings().EnableSimpleHomepageForAnonymous)
            {
                return Redirect(SiteUrls.Instance().SimpleHome());
            }
            Header.AddTitle(Globals.SiteName);
            Header.AddMetaKeywords(SiteSettingsManager.GetSiteSettings().SearchMetaKeywords);
            Header.AddMetaDescription(SiteSettingsManager.GetSiteSettings().SearchMetaDescription);
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.EnableFreeHome)
                return View("Pages/FreeHome.aspx");
            return View("Pages/Home.aspx");
        }

        /// <summary>
        /// 简洁的整站首页
        /// </summary>
        /// <returns></returns>
        public ActionResult SimpleHome()
        {
            if (Request.IsAuthenticated)
                return Redirect(SiteUrls.Instance().UserDomainHome(Globals.GetCurrentUser()));

            Header.AddTitle(Globals.SiteName);
            Header.AddMetaKeywords(SiteSettingsManager.GetSiteSettings().SearchMetaKeywords);
            Header.AddMetaDescription(SiteSettingsManager.GetSiteSettings().SearchMetaDescription);

            IList<UserActivity> userActivities = Activities.GetTopSiteUserActivities(null, 20);

            return View("Pages/SimpleHome.aspx", "Masters/SimpleHeader.master", userActivities);
        }

        /// <summary>
        /// 站点皮肤设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ManageAuthorize(RequireSystemAdministrator = true, CheckCookie = false)]
        public ActionResult ModifyHome()
        {
            SetPageTitle(GetResourceString("Title_ModifySiteHome"));
            return View("Pages/ModifyHome.aspx");
        }

        /// <summary>
        /// 添加关注
        /// </summary>
        /// <param name="favoriteUserID"></param>
        /// <returns></returns>
        public JsonResult CreateFavorite(int favoriteUserID)
        {
            bool isSuccess = false;
            if (CurrentUser != null && CurrentUser.UserID != favoriteUserID)
            {
                User favoriteUser = Users.GetUser(favoriteUserID, false);
                if (favoriteUser != null)
                {
                    Favorite favorite = new Favorite();
                    favorite.UserID = CurrentUser.UserID;
                    favorite.DisplayName = CurrentUser.DisplayName;
                    favorite.ItemID = favoriteUser.UserID;
                    favorite.PrivacyStatus = PrivacyStatuses.Public;

                    FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).CreateFavorite(favorite);
                    isSuccess = true;
                }
            }

            return Json(new { isSuccess = isSuccess }, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 显示错误页面
        /// </summary>
        public ActionResult Error(string message)
        {
            if (!string.IsNullOrEmpty(message))
                ViewData["subject"] = WebUtils.UrlDecode(message);
            return View("Pages/Error.aspx", "Masters/SimpleHeader.master");
        }

        /// <summary>
        /// 未找到文件时显示的错误页面
        /// </summary>
        public ActionResult Error404(string title, string body)
        {
            Response.StatusCode = 404;
            SetPageTitle("FileNotFound");
            if (!string.IsNullOrEmpty(title))
                ViewData["title"] = WebUtils.UrlDecode(title);
            if (!string.IsNullOrEmpty(body))
                ViewData["body"] = WebUtils.UrlDecode(body);
            return View("Pages/Error404.aspx", "Masters/SimpleHeader.master");
        }

        /// <summary>
        /// 显示暂停页面
        /// </summary>
        public ActionResult SiteSuspended()
        {
            SetPageTitle(GetResourceString("Title_SiteSuspended"));
            return View("Pages/SiteSuspended.aspx");
        }

        /// <summary>
        /// 显示抛出的异常信息
        /// </summary>
        public ActionResult ShowExceptionMessage(string exceptionType, string message, int? applicationID)
        {
            SPBException spbException = new SPBException(exceptionType, message);
            SetPageTitle(GetResourceString("Title_ShowExceptionMessage"));
            return View("Pages/ShowExceptionMessage.aspx", "Masters/SimpleHeader.master", spbException);
        }
        /// <summary>
        /// 显示验证提示信息
        /// </summary>
        public ActionResult ValidationMessage(string exceptionType, string message)
        {
            SetPageTitle("验证提示");
            SPBException spbException = new SPBException(exceptionType, message);
            ViewData["message"] = message;
            return View("Pages/ValidationMessage.aspx", "Masters/SimpleHeader.master", spbException);
        }

        #region 信息推荐

        /// <summary>
        /// 推荐信息
        /// </summary>
        public ActionResult Control_RecommendItem(int applicationID, int itemID, string itemName, int domainUserID)
        {
            if (!CurrentUser.IsContentAdministrator)
                throw new SPBException(ExceptionTypes.Instance().AdministrationAccessDenied());

            List<RecommendationType> allRecommendationTypes = RecommendationTypes.GetRecommendationTypes(applicationID, false);

            string excludeTypeIDs = Request.QueryString.GetString("excludeTypeIDs", string.Empty);

            int onlyTypeID = Request.QueryString.GetInt("onlyTypeID", 0);
            if (onlyTypeID > 0)
                allRecommendationTypes.RemoveAll(n => n.TypeID != onlyTypeID);
            else
            {
                if (excludeTypeIDs != null)
                {
                    string[] excludeTypeIDsArray = excludeTypeIDs.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                    if (excludeTypeIDsArray != null && excludeTypeIDsArray.Length > 0)
                    {
                        foreach (string excludeTypeID in excludeTypeIDsArray)
                        {
                            allRecommendationTypes.RemoveAll(n => n.TypeID == Convert.ToInt32(excludeTypeID));
                        }
                    }
                }
            }

            int CommendTypeID = Request.QueryString.GetInt("CommendTypeID", 0);
            int CommendID = Request.QueryString.GetInt("CommendID", 0);
            if (CommendID > 0)
            {
                RecommendedItem item = RecommendedItems.GetRecommendedItem(CommendID, false);
                ViewData["CommendID"] = CommendID;
                ViewData["stopDate"] = item.Deadline;
                ViewData["displyOrder"] = item.DisplayOrder;
            }
            bool noRecommendationType = true;
            if (allRecommendationTypes != null && allRecommendationTypes.Count > 0)
                noRecommendationType = false;
            List<RecommendationType> recommendationTypes = new List<RecommendationType>();
            foreach (RecommendationType recommendationType in allRecommendationTypes)
            {
                recommendationTypes.Add(recommendationType);
            }

            IList<RecommendedItem> recommendedItems = RecommendedItems.GetTopRecommendedItemsOfItem(applicationID, itemID);

            if (recommendedItems != null && allRecommendationTypes != null && recommendedItems.Count > 0 && allRecommendationTypes.Count > 0)
            {
                foreach (RecommendedItem recommendedItem in recommendedItems)
                {
                    foreach (RecommendationType recommendationType in allRecommendationTypes)
                    {
                        if (recommendationType.TypeID == recommendedItem.CommendTypeID && CommendTypeID <= 0)
                            recommendationTypes.Remove(recommendationType);
                    }
                }
            }

            ViewData["CommendTypeID"] = CommendTypeID;

            ViewData["noRecommendationType"] = noRecommendationType;
            ViewData["itemID"] = itemID;
            ViewData["itemName"] = WebUtils.UrlDecode(itemName);
            ViewData["domainUserID"] = domainUserID;
            ViewData["applicationID"] = applicationID;
            ViewData["recommendationTypes"] = recommendationTypes;
            return PartialView("Controls/RecommendItem.ascx", recommendedItems);
        }

        /// <summary>
        /// 推荐项处理
        /// </summary>
        public ActionResult RecommendItemHandler(int applicationID, int domainUserID, int itemID)
        {
            if (CurrentUser == null || (CurrentUser != null && !CurrentUser.IsContentAdministrator && !CurrentUser.IsAdministrator))
                throw new SPBException(ExceptionTypes.Instance().AdministrationAccessDenied());

            StatusMessageData statusMessageData = null;

            User domainUser = Users.GetUser(domainUserID, false);
            if (domainUser == null)
            {
                statusMessageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_RecommendedFailure"));
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID } });
            }

            RecommendedItem recommendedItem = null;

            int CommendTypeID = Request.Form.GetInt("CommendTypeID", 0);
            int CommendID = Request.Form.GetInt("CommendID", 0);

            if (CommendTypeID > 0 && CommendID > 0)
            {
                recommendedItem = RecommendedItems.GetRecommendedItem(CommendID, false);
                recommendedItem.DisplayOrder = Request.Form.GetInt("sortOrder", 0);
                recommendedItem.Deadline = Request.Form.Get<DateTime>("stopTime", DateTime.Now);
                recommendedItem.DateCreated = DateTime.Now;
                recommendedItem.CommendTypeID = CommendTypeID;
                recommendedItem.CommendID = CommendID;
                recommendedItem.ItemName = Request.Form.GetString("commendName", string.Empty);
                RecommendedItems.UpdateRecommendedItem(recommendedItem);
                statusMessageData = new StatusMessageData(StatusMessageType.Success, "修改成功");
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", recommendedItem.ItemName }, { "CommendTypeID", CommendTypeID } });
            }

            recommendedItem = new RecommendedItem();

            string itemName = recommendedItem.ItemName;
            recommendedItem.DisplayOrder = Request.Form.GetInt("sortOrder", 0);
            recommendedItem.Deadline = Request.Form.Get<DateTime>("stopTime", DateTime.Now);
            recommendedItem.ItemName = Censors.CensorPost(WebUtils.HtmlEncode(Request.Form.GetString("commendName", string.Empty).Trim()));
            recommendedItem.Author = domainUser.DisplayName;

            recommendedItem.CommendTypeID = Request.Form.GetInt("recommendationTypesDropDownList", 0);
            recommendedItem.AuthorUserID = domainUser.UserID;

            recommendedItem.CommenderUserID = CurrentUser.UserID;
            recommendedItem.Commender = CurrentUser.DisplayName;
            recommendedItem.ApplicationID = applicationID;
            recommendedItem.ItemID = itemID;

            if (recommendedItem.CommendTypeID > 0)
            {
                RecommendationType recommendationType = RecommendationTypes.GetRecommendationType(recommendedItem.CommendTypeID, true);
                if (recommendationType == null)
                {
                    statusMessageData = new StatusMessageData(StatusMessageType.Error, string.Format(GetResourceString("Recommended FailureTypeNotFound"), recommendedItem.CommendTypeID));
                    TempData["StatusMessageData"] = statusMessageData;
                    return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", itemName } });
                }
            }
            else
            {
                TempData["itemName"] = Censors.CensorPost(WebUtils.HtmlEncode(Request.Form.GetString("commendName", string.Empty).Trim()));
                statusMessageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_RecommendedFailureTypeID"));
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", itemName } });
            }

            try
            {
                RecommendedItems.CreateRecommendedItem(recommendedItem);
                statusMessageData = new StatusMessageData(StatusMessageType.Success, GetResourceString("Exception_RecommendedSucecssCacheShow"));
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", itemName } });
            }
            catch (Exception)
            {
                TempData["itemName"] = Censors.CensorPost(WebUtils.HtmlEncode(Request.Form.GetString("commendName", string.Empty).Trim()));
                statusMessageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Exception_RecommendedFailure"));
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", itemName } });
            }
        }

        /// <summary>
        /// 删除推荐项
        /// </summary>
        public ActionResult DeleteRecommendedItem(int commendID, int applicationID, int domainUserID, int itemID, string itemName)
        {
            if (CurrentUser == null || (CurrentUser != null && !CurrentUser.IsContentAdministrator && !CurrentUser.IsAdministrator))
                throw new SPBException(ExceptionTypes.Instance().AdministrationAccessDenied());
            RecommendedItem recommendedItem = RecommendedItems.GetRecommendedItem(commendID, true);
            if (recommendedItem != null)
            {
                StatusMessageData statusMessageData = null;
                try
                {
                    RecommendedItems.DeleteRecommendedItem(commendID);
                    statusMessageData = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteSuccess"));
                    return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", itemName } });
                }
                catch (Exception)
                {
                    statusMessageData = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteFailed"));
                    return RedirectToAction("Control_RecommendItem", new RouteValueDictionary { { "applicationID", applicationID }, { "domainUserID", domainUserID }, { "itemID", itemID }, { "itemName", itemName } });
                }
            }
            else
                return new EmptyResult();
        }

        #endregion

        #region 会员

        /// <summary>
        /// 会员首页
        /// </summary>
        /// <param name="tab"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult MemberHome()
        {
            SetPageTitle(GetResourceString("Title_MemberHome", ApplicationIDs.Instance().User()));
            IList<PersonUser> members = Users.GetTopPersonUserSortBy(PersonUsersSortBy.StageHitTimes, 16);
            IList<UserActivity> userActivities = Activities.GetTopSiteUserActivities(null, 10);
            ViewData["userActivities"] = userActivities;

            return View("Pages/Members/Home.aspx", "Masters/Member.master", members);
        }

        /// <summary>
        /// 在线会员块
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_OnlineMembersBlock(int displayCount)
        {
            Dictionary<int, OnlineUser> onlineMembers = UserOnlines.GetMembersOnline();
            if (onlineMembers == null)
                onlineMembers = new Dictionary<int, OnlineUser>();
            return PartialView("Controls/Members/OnlineMembersBlock.ascx", onlineMembers.Values);
        }

        /// <summary>
        /// 获取用户卡片
        /// </summary>
        /// <param name="userID"></param>
        /// <returns></returns>
        public ActionResult Control_MemberCard(int? userID)
        {
            if (userID == null)
                return new EmptyResult();
            PersonUser personUser = Users.GetUser(userID ?? 0, false) as PersonUser;
            if (personUser == null)
                return new EmptyResult();
            Random random = new Random();
            string controlID = Request.QueryString.GetString("controlID", random.Next(9999).ToString());
            ViewData["controlID"] = controlID;
            return PartialView("Controls/MemberCard.ascx", personUser);
        }

        /// <summary>
        /// 最近活跃会员列表
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ListActiveMembers(int displayCount)
        {
            IList<PersonUser> list = Users.GetRandomActivePersonUsers(displayCount);

            return PartialView("Controls/Members/ListActiveMembers.ascx", list);
        }

        /// <summary>
        /// 总站首页的推荐用户
        /// </summary>
        public ActionResult Control_RecommendUserHomeBlock(int commendTypeID, int? displayCount)
        {
            RecommendationType type = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            IList<RecommendedItem> pds = null;

            if (type != null)
                pds = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().User(), commendTypeID, displayCount ?? 3);
            else
                pds = new List<RecommendedItem>();

            return PartialView("Controls/Members/RecommendUserHomeBlock.ascx", pds);
        }

        /// <summary>
        /// 用户推荐
        /// </summary>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_RecommendUser(int commendTypeID, int? displayCount)
        {
            RecommendationType type = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            IList<RecommendedItem> pds = null;

            if (type != null)
            {
                ViewData["RecommendactionType"] = type;
                pds = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().User(), commendTypeID, displayCount ?? 3);
            }
            else
                pds = new List<RecommendedItem>();

            return PartialView("Controls/Members/RecommendUser.ascx", pds);
        }

        /// <summary>
        /// 会员列表
        /// </summary>
        /// <param name="pageSize">显示数量</param>
        /// <param name="sortBy">排序方式</param>
        /// <param name="checkDisplayNameLength">截取长度，不截取可以设置为0</param>
        /// <param name="lineSize">每行显示数量</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_ListMembers(int pageSize, PersonUsersSortBy? sortBy)
        {
            IList<PersonUser> pds = Users.GetTopPersonUserSortBy(sortBy ?? PersonUsersSortBy.CreateDate, pageSize);
            ViewData["sortBy"] = (int)(sortBy ?? PersonUsersSortBy.CreateDate);
            return PartialView("Controls/Members/ListMembers.ascx", pds);
        }

        /// <summary>
        /// 会员排行
        /// </summary>
        /// <param name="sortBy">排序方式</param>
        /// <param name="pageIndex">页数</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult MemberRanks(PersonUsersSortBy? sortBy, int? pageIndex)
        {
            int applicationID = ApplicationIDs.Instance().User();
            Header.AddTitle(string.Format("{1}{0}{2}", Globals.BrowserTitleSeparator, GetResourceString("Title_MemberRank", ApplicationIDs.Instance().User()), GetResourceString("Title_Member", ApplicationIDs.Instance().User()), Globals.SiteName));
            PagingDataSet<PersonUser> pds = Users.GetPagingPersonUsersSortBy(sortBy ?? PersonUsersSortBy.StageHitTimes, 10, pageIndex ?? 1);
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_MemberRank", applicationID), string.Empty);

            ViewData["sortBy"] = sortBy;
            return View("Pages/Members/MemberRanks.aspx", "Masters/Member.master", pds);
        }

        /// <summary>
        /// 在线会员
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult OnlineMembers()
        {
            Header.AddTitle(string.Format("{1}{0}{2}", Globals.BrowserTitleSeparator, GetResourceString("Title_OnlineMember", ApplicationIDs.Instance().User()), GetResourceString("Title_Member", ApplicationIDs.Instance().User()), Globals.SiteName));
            Dictionary<int, OnlineUser> onlineMembers = UserOnlines.GetMembersOnline();
            if (onlineMembers == null)
                onlineMembers = new Dictionary<int, OnlineUser>();
            return View("Pages/Members/OnlineMembers.aspx", "Masters/Member.master", onlineMembers.Values);
        }

        #endregion

        #region 用户举报

        /// <summary>
        /// 用户举报
        /// </summary>
        /// <returns></returns>
        public ActionResult NewImpeachReport()
        {
            return PartialView("Controls/Common/NewImpeachReport.ascx");
        }

        /// <summary>
        /// 创建举报
        /// </summary>
        /// <returns></returns>
        public ActionResult CreateImpeachReport()
        {
            ImpeachReport impeachReport = new ImpeachReport();
            impeachReport.Author = Formatter.FormatMultiLinePlainTextForStorage(SPBHtmlHelper.Instance().UserDisplayName(CurrentUser), true);
            impeachReport.Email = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("userEmail", string.Empty).Trim());
            impeachReport.OriginalUrl = Request.UrlReferrer.AbsoluteUri;
            impeachReport.Reason = (ImpeachReason)Request.Form.GetInt("impeachReason", 99);
            impeachReport.Remark = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("remark", string.Empty).Trim(), true);
            impeachReport.UserID = CurrentUser != null ? CurrentUser.UserID : 0;

            ImpeachReports.CreateImpeachReport(impeachReport);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "举报成功！";
            return RedirectToAction("NewImpeachReport");
        }

        #endregion

        #region SiteCategory

        /// <summary>
        /// 站点类别下拉列表
        /// </summary>
        public ActionResult Control_SiteCategoryDropDownList(int applicationID, int? depth, string controlName, int? categoryID, string cssClass)
        {
            List<SiteCategory> siteCategoryList = new List<SiteCategory>();

            SiteCategory currentSiteCategory = new SiteCategory();
            List<SiteCategory> allSiteCategoryList = SiteCategories.Instance(applicationID).GetAll(false);
            if (allSiteCategoryList != null && allSiteCategoryList.Count > 0)
            {
                // 设置传送到页面的站点类别集合，如果配置文件中已设置深度则取指定深度内的站点类别
                if (depth != null && !(depth < 0))
                {
                    foreach (SiteCategory siteCategory in allSiteCategoryList)
                    {
                        if ((siteCategory.Depth < depth) || siteCategory.Depth == depth)
                        {
                            siteCategoryList.Add(siteCategory);
                        }
                    }
                }
                else
                    siteCategoryList = allSiteCategoryList;

                // 获取选中的站点类别
                if (categoryID != null && (int)categoryID > 0 && siteCategoryList != null)
                {
                    currentSiteCategory = SiteCategories.Instance(applicationID).Get(categoryID.Value, false);
                    if (currentSiteCategory != null && currentSiteCategory.Depth > depth)
                    {
                        while (currentSiteCategory != null && currentSiteCategory.Depth > depth)
                            currentSiteCategory = SiteCategories.Instance(applicationID).Get(currentSiteCategory.ParentID, false);
                    }

                    if (currentSiteCategory != null && !siteCategoryList.Contains(currentSiteCategory))
                        currentSiteCategory = null;
                }
                else
                    currentSiteCategory = null;
            }
            else
            {
                siteCategoryList = null;
                currentSiteCategory = null;
            }
            if (!string.IsNullOrEmpty(cssClass))
                ViewData["cssClass"] = cssClass;
            else
                ViewData["cssClass"] = "tn-dropdownlist";

            if (!string.IsNullOrEmpty(controlName))
                ViewData["controlName"] = controlName;
            else
                ViewData["controlName"] = "siteCategoryDropDownList";

            ViewData["currentSiteCategory"] = currentSiteCategory;

            return PartialView("Controls/Common/SiteCategoryDropDownList.ascx", siteCategoryList);
        }

        #endregion

        #region 地区选择器 Area
        /// <summary>
        /// 地区下拉列表
        /// </summary>
        /// <param name="controlName">控件名称（name和id）</param>
        /// <param name="rootID">顶级地区（例如设置山东为顶级地区那么 只显示山东下面的地区）</param>
        /// <param name="areaCode">设置选中的地区ID（编辑时用到）</param>
        /// <param name="depth">地区层次深度</param>
        /// <param name="cssClass">设置DropDownlist样式</param>
        public ActionResult Control_AreasDropDownList(string controlName, string rootAreaCode, string selectedAreaCode, int areaLevel, string cssClass)
        {
            List<Area> rootAreas = null;
            if (!string.IsNullOrEmpty(rootAreaCode))
            {
                Area rootArea = Areas.GetArea(rootAreaCode, false);
                if (rootArea == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format("找不到areaCode为{0}的地区", rootAreaCode)));
                }
                rootAreas = rootArea.Childs;
                ViewData["rootAreaDepth"] = rootArea.Depth + 1;

            }
            else
            {
                rootAreas = Areas.GetRootAreas(false);
                ViewData["rootAreaDepth"] = 0;
            }
            Dictionary<string, string> rootAreasDictionary = rootAreas.ToDictionary<Area, string, string>(a => a.AreaCode, a => a.GetMultilingualAreaName(CurrentUserLanguage));
            ViewData["rootAreas"] = rootAreasDictionary;
            if (!string.IsNullOrEmpty(selectedAreaCode))
            {
                Stack<KeyValuePair<string, Dictionary<string, string>>> stack = new Stack<KeyValuePair<string, Dictionary<string, string>>>();
                Area area = Areas.GetArea(selectedAreaCode, false);
                ViewData["selectedArea"] = area;
                while (area != null && area.ParentCode != rootAreaCode && area.Parent != null)
                {
                    stack.Push(new KeyValuePair<string, Dictionary<string, string>>(area.AreaCode, area.Parent.Childs.ToDictionary<Area, string, string>(a => a.AreaCode, a => a.GetMultilingualAreaName(CurrentUserLanguage))));
                    area = area.Parent;
                }
                if (area != null)
                {
                    stack.Push(new KeyValuePair<string, Dictionary<string, string>>(area.AreaCode, rootAreasDictionary));
                }
                ViewData["stack"] = stack;
            }
            return PartialView("Controls/Common/AreasDropDownList.ascx");
        }
        /// <summary>
        /// 获取子地区
        /// </summary>
        public JsonResult GetChildAreas(string areaCode)
        {
            Area area = Areas.GetArea(areaCode, false);
            if (area == null || area.Childs == null)
                return Json(new object(), JsonRequestBehavior.AllowGet);
            return Json(area.Childs.Select(
            n => new
            {
                id = n.AreaCode,
                name = n.GetMultilingualAreaName(CurrentUserLanguage)
            }), JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region 用户内容选择器

        /// <summary>
        /// 用户内容选择器
        /// </summary>
        public ActionResult UserContentSelector(string manageAttachmentsUrl, bool isImage)
        {
            return PartialView("Controls/UserContentSelector.ascx");
        }

        #endregion

        #region 用户选择器
        /// <summary>
        /// 用户选择器
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult UserSelector()
        {
            Dictionary<int, string> searchRangeDict = new Dictionary<int, string>();
            searchRangeDict[1] = GetResourceString("Description_MyFristLevelFriend", ApplicationIDs.Instance().Friend());
            searchRangeDict[2] = GetResourceString("Description_SecondLevelFriend", ApplicationIDs.Instance().Friend());
            SelectList searchRangeSelectList;

            User currentUser = Globals.GetCurrentUser();
            IList<PersonUser> personUsers = null;
            if (currentUser != null && currentUser != null)
            {
                PersonUserFullTextQuery query = new PersonUserFullTextQuery();
                query.PageSize = 300;

                if (TempData.ContainsKey("nameKeyword"))
                    query.NameKeyword = TempData.GetString("nameKeyword", string.Empty);

                query.SearchRange = TempData.GetInt("searchRange", 1);
                searchRangeSelectList = new SelectList(searchRangeDict, "key", "value", query.SearchRange);

                SearchResultDataSet<PersonUser> pds = PersonUserSearchManager.Instance().Search(Globals.GetCurrentUser(), query);

                if (pds.Records != null && pds.Records.Count > 0)
                    personUsers = pds.Records;
            }
            else
            {
                searchRangeSelectList = new SelectList(searchRangeDict, "key", "value");
            }

            ViewData["SearchRangeSelectList"] = searchRangeSelectList;
            return PartialView("Controls/User/UserSelector.ascx", personUsers);
        }

        /// <summary>
        /// 处理用户选择器的搜索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UserSelectorSearchAction()
        {
            string userSelectorNameKeyword = Request.Form.GetString("userSelectorNameKeyword", string.Empty).Trim();
            if (!string.IsNullOrEmpty(userSelectorNameKeyword))
                TempData["nameKeyword"] = userSelectorNameKeyword;

            int userSelectorSearchRange = Request.Form.GetInt("userSelectorSearchRange", 1);
            if (userSelectorSearchRange > 1)
                TempData["searchRange"] = userSelectorSearchRange;

            return UserSelector();
        }

        /// <summary>
        /// 获取朋友列表
        /// </summary>
        public ActionResult GetFriends(string keyWord)
        {
            PersonUserFullTextQuery query = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(keyWord))
                query.Keyword = keyWord;
            query.SearchRange = 1;
            query.PageSize = 1000;
            SearchResultDataSet<PersonUser> pds = PersonUserSearchManager.Instance().Search(CurrentUser, query);
            return Json(pds.Records.Select(p => new { userid = p.UserID, username = p.UserName, commonname = p.CommonName, displayname = p.DisplayName }), JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region 多文件上传

        /// <summary>
        /// 多文件上传
        /// </summary>
        /// <param name="uploadUrl">处理上传文件的Url</param>
        /// <param name="AssociateID">附件相关联对象的ID</param>
        /// <param name="fileTypes">能够上传的文件类型，例如："*.jpg;*.gif" 用英文分号隔开</param>
        /// <param name="attributes">创建附件附带的属性对应控件的name，例如："categoryID;tagName"，使用分号分开</param>
        /// <param name="fileQueryLimit">允许排队的文件总数，0为不限制</param>
        /// <param name="fileSizeLimit">单个文件大小的限制 参数接收一个单位，有效的单位有B、KB、MB、GB，默认单位是KB。例如: 2147483648 B, 2097152, 2097152KB, 2048 MB, 2 GB</param>
        public ActionResult Control_UploadAttachment(string uploadUrl, string flashUrl, int AssociateID, string fileTypes, string attributes, int fileQueryLimit, string fileSizeLimit)
        {
            ViewData["uploadUrl"] = uploadUrl;
            ViewData["flashUrl"] = flashUrl;
            ViewData["fileTypes"] = fileTypes;
            ViewData["attributes"] = attributes;
            ViewData["fileQueryLimit"] = fileQueryLimit;
            ViewData["fileSizeLimit"] = fileSizeLimit;
            ViewData["associateID"] = AssociateID;
            return PartialView("Controls/UploadAttachment.ascx");
        }

        /// <summary>
        /// 用于头像 企业标识 群组logo上传
        /// </summary>
        /// <param name="uploadUrl">处理上传文件的URL</param>
        /// <param name="flashUrl">处理上传文件的Flash URL</param>
        /// <param name="AssociateID">附件相关联对象的ID</param>
        /// <param name="fileTypes">能够上传的文件类型，例如："*.jpg;*.gif" 用英文分号隔开</param>
        /// <param name="fileSizeLimit">单个文件大小的限制 参数接收一个单位，有效的单位有B、KB、MB、GB，默认单位是KB。例如: 2147483648 B, 2097152, 2097152KB, 2048 MB, 2 GB</param>
        /// <returns></returns>
        public ActionResult Control_UploadAvatarAttachment(string userDomainName, string uploadUrl, string flashUrl, int AssociateID, string fileTypes, string fileSizeLimit, bool canCropAvatar, string cropAvatarUrl)
        {
            ViewData["uploadUrl"] = uploadUrl;
            ViewData["flashUrl"] = flashUrl;
            ViewData["fileTypes"] = fileTypes;
            ViewData["fileSizeLimit"] = fileSizeLimit;
            ViewData["associateID"] = AssociateID;
            ViewData["cropAvatarUrl"] = cropAvatarUrl;
            ViewData["canCropAvatar"] = canCropAvatar;
            return PartialView("Controls/UploadAvatarAttachment.ascx");
        }

        #endregion

        #region 日期选择器
        /// <summary>
        /// 日期选择器
        /// </summary>
        /// <param name="minYear">最小年份</param>
        /// <param name="maxYear">最大年份</param>
        /// <param name="dateTime">需要显示的日期时间</param>
        /// <param name="showTime">是否需要显示时刻</param>
        public ActionResult Control_Datepicker(string controlName, int? minYear, int? maxYear, DateTime? dateTime, bool showTime, string cssClass)
        {

            ViewData["ControlName"] = controlName;
            ViewData["ShowTime"] = showTime;
            ViewData["CssClass"] = cssClass;
            if (dateTime.HasValue)
            {
                ViewData["DateTime"] = dateTime;
            }
            if (minYear.HasValue && maxYear.HasValue)
            {
                if (minYear.Value < maxYear.Value)
                {
                    ViewData["minYear"] = minYear;
                    ViewData["maxYear"] = maxYear;
                }
            }
            return PartialView("Controls/Common/Datepicker.ascx");
        }
        #endregion

        #region 学校选择模式框
        /// <summary>
        /// 显示学校选择模式框
        /// 它返回学校名和学校ID
        /// </summary>
        /// <param name="textBoxID">返回的学校名要写到那个文本框中</param>
        /// <param name="hiddenID">返回的学校ID要写到哪个隐藏域中</param>
        public ActionResult Control_SchoolSelector(string textBoxID, string hiddenID)
        {
            ViewData["TextBoxID"] = textBoxID;
            ViewData["HiddenID"] = hiddenID;
            return PartialView("Controls/Common/SchoolSelector.ascx");
        }

        /// <summary>
        /// 获取所有根级地区
        /// </summary>
        public JsonResult GetRootAreas()
        {
            Area area = Areas.GetArea(UserDomainConfiguration.Instance().RootAreaCode, false);

            List<Area> areas = null;

            if (area != null)
                areas = area.Childs;
            else
                areas = Areas.GetRootAreas(true);

            return Json(areas.Select(
            n => new
            {
                id = n.AreaCode,
                name = n.GetMultilingualAreaName(CurrentUserLanguage)
            }), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 返回某地区下的所有学校列表
        /// </summary>
        public JsonResult GetSchools(string areaCode)
        {
            PagingDataSet<School> pds = Schools.GetSchools(1000, Request.QueryString.GetInt("pageIndex", 1), areaCode, null, false);
            IList<School> schools = pds.Records;
            return Json(schools.Select(n => new
            {
                id = n.SchoolID.ToString(),
                name = n.SchoolName
            }), JsonRequestBehavior.AllowGet);
        }


        #endregion

        #region 表情选择器
        /// <summary>
        /// 表情图片选择器
        /// </summary>
        public ActionResult Control_SmileySelector(string controlName, string textAreaID)
        {
            List<EmotionCategory> categories = EmotionManager.Instance().Categories;
            if (categories == null || categories.Count == 0)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_NoSmileyCategory"));
                return new EmptyResult();
            }
            ViewData["Categories"] = categories;

            return PartialView("Controls/Common/SmileySelector.ascx");
        }
        /// <summary>
        /// 获取某类别下所有表情
        /// </summary>
        /// <param name="categoryID">表情类别ID</param>
        public JsonResult GetSmilies(int categoryID)
        {
            EmotionCategory emotionCategory = EmotionManager.Instance().GetEmotionCategory(categoryID);
            return Json(emotionCategory.Emotions.Select(n => new
            {
                CategoryID = emotionCategory.CategoryID,
                Code = n.Code,
                SmileyImgUrl = n.ImageUrl,
                Description = n.Description
            }), JsonRequestBehavior.AllowGet);
        }




        #endregion

        #region 统计导航器
        /// <summary>
        /// 统计导航器
        /// </summary>
        /// <param name="getStatisticssUrl">获取下一级统计集合的链接,请不要带参数</param>
        /// <param name="maxDepth">需要显示的最大深度</param>
        public ActionResult Control_StatisticsNavigationSelector(string getStatisticssUrl, int maxDepth)
        {
            if (string.IsNullOrEmpty(getStatisticssUrl) || maxDepth < 0)
            {
                string title = string.Format(GetResourceString("Exception_Pattern_StatisticsNavigationSelectorArgumentError"), getStatisticssUrl, maxDepth);
                return Redirect(SiteUrls.Instance().Error404(title));
            }
            return PartialView("Controls/Common/StatisticsNavigationSelector.ascx");
        }

        #endregion

        #region 分享内容给好友
        /// <summary>
        /// 分享内容给好友
        /// </summary>
        /// <param name="applicationID">应用ID</param>
        /// <param name="contentUrl">内容Url</param>
        public ActionResult ShareToMyFriends(int applicationID, string contentUrl)
        {

            ViewData["applicationID"] = applicationID;
            ViewData["contentSubject"] = Request.QueryString.Get<string>("contentSubject", string.Empty);
            ViewData["contentBody"] = Request.QueryString.Get<string>("contentBody", string.Empty);

            ViewData["contentImgUrl"] = Request.QueryString.Get<string>("contentImgUrl", string.Empty);
            ViewData["contentUrl"] = contentUrl;

            ViewData["contentAuthor"] = Request.QueryString.Get<string>("contentAuthor", string.Empty);
            ViewData["contentUserID"] = Request.QueryString.Get<int>("contentUserID", 0);

            return PartialView("Controls/Common/ShareToMyFriends.ascx");
        }
        /// <summary>
        /// 处理分享内容给我的好友
        /// </summary>
        public ActionResult ShareToMyFriendsAction(int applicationID, string contentUrl, int authorUserID)
        {

            string shareClubs = Request.Form.Get<string>("shareToMyClubsControl", string.Empty);
            string contentSubject = Request.Form.Get<string>("contentSubject", string.Empty);
            string contentBody = Request.Form.Get<string>("contentBody", string.Empty);
            string contentAuthor = Request.Form.Get<string>("contentAuthor", string.Empty);
            string contentImgUrl = Request.Form.Get<string>("contentImgUrl", string.Empty);

            return RedirectToAction("ShareToMyClubs", "Channel", new RouteValueDictionary { { "applicationID", applicationID }, { "contentUrl", contentUrl } });
        }

        #endregion

        #region 收藏
        /// <summary>
        /// 显示所有公开收藏该活动的用户
        /// </summary>
        public ActionResult Control_ListFavoriteUsersBlock(int itemID, string favoriteItemKey, string moduleTitle, int? displayItemCount)
        {
            IList<Favorite> favorites = FavoriteManager.Instance(favoriteItemKey).GetTopLatestFavoritesOfItem(itemID, displayItemCount ?? 12);
            ViewData["moduleTitle"] = moduleTitle;
            return PartialView("Controls/Common/ListFavoriteUsersBlock.ascx", favorites);
        }

        public ActionResult Control_NewFavorite(string favoriteItemName, string createFavoriteItemUrl)
        {
            ViewData["favoriteItemName"] = WebUtils.UrlDecode(favoriteItemName);
            ViewData["createFavoriteItemUrl"] = createFavoriteItemUrl;
            return PartialView("Controls/Common/NewFavorite.ascx");
        }


        #endregion

        #region 站点友情连接
        /// <summary>
        /// 显示站点的友情链接
        /// </summary>
        /// <param name="CategoryID">链接的类别ID</param>
        /// <returns></returns>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.Stable)]
        public ActionResult Control_ShowLinks(int? categoryID)
        {
            bool isRealTime = CurrentUser != null && CurrentUser.IsContentAdministrator;
            SiteLinkManager SLinkManage = SiteLinkManager.Instance();
            List<Link> ListLinks = null;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                ListLinks = SLinkManage.GetLinks(null, categoryID.Value, null, isRealTime);
                ViewData["Category"] = SLinkManage.GetLinkCategory(categoryID.Value, false);
            }
            else
            {
                ListLinks = SLinkManage.GetLinks(null, null, null, isRealTime);
                ViewData["Categories"] = SLinkManage.GetAllLinkCategories(false);
            }

            ViewData["ListLinks"] = ListLinks;
            return PartialView("Controls/SiteLinks.ascx");
        }

        /// <summary>
        /// 友情链接更多页面
        /// </summary>
        public ActionResult ShowAllLinks()
        {
            SiteLinkManager SLinkManage = SiteLinkManager.Instance();
            List<Link> ListLinks = null;
            ViewData["Categories"] = SLinkManage.GetAllLinkCategories(false);
            bool isRealTime = false;
            if (CurrentUser != null && CurrentUser.IsContentAdministrator)
                isRealTime = true;
            ListLinks = SLinkManage.GetLinks(null, null, null, isRealTime);
            ViewData["ListLinks"] = ListLinks;
            return View("Pages/SiteLinks.aspx", "Masters/Simple.master");
        }
        #endregion


        #region Widget操作

        /// <summary>
        /// 移动WidgetInstance
        /// </summary>
        /// <param name="instanceID"></param>
        /// <param name="toZoneID"></param>
        /// <param name="toBeforeInstanceID"></param>
        /// <returns></returns>
        public ActionResult MoveWidgetInstance(int instanceID, int toZoneID, int toBeforeInstanceID)
        {
            if (!HasManageWidgetInstancePermission(instanceID))
            {
                string title = string.Format(GetResourceString("Exception_Pattern_ManageWidgetInstancePermissionRefused"), instanceID);
                return Redirect(SiteUrls.Instance().Error404(title));
            }

            WidgetInstances.MoveWidgetInstance(instanceID, toZoneID, toBeforeInstanceID);

            return new EmptyResult();
        }

        /// <summary>
        /// 删除WidgetInstance
        /// </summary>
        /// <param name="instanceID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteWidgetInstance(int instanceID)
        {
            if (!HasManageWidgetInstancePermission(instanceID))
            {
                string title = string.Format(GetResourceString("Exception_Pattern_ManageWidgetInstancePermissionRefused"), instanceID);
                return Redirect(SiteUrls.Instance().Error404(title));
            }

            WidgetInstances.DeleteWidgetInstance(instanceID);

            return new EmptyResult();
        }

        /// <summary>
        /// 更新WidgetInstance设置
        /// </summary>
        /// <param name="instanceID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateWidgetInstanceSettings(int instanceID)
        {
            if (!HasManageWidgetInstancePermission(instanceID))
            {
                string title = string.Format(GetResourceString("Exception_Pattern_ManageWidgetInstancePermissionRefused"), instanceID);
                return Redirect(SiteUrls.Instance().Error404(title));
            }

            WidgetInstance widgetInstance = WidgetInstances.GetWidgetInstance(instanceID, true);
            if (widgetInstance != null)
            {
                foreach (var formItemName in Request.Form.AllKeys)
                {
                    if (widgetInstance.UserPrefDictionary.ContainsKey(formItemName))
                    {
                        string defaultValue = Request.Form[formItemName];
                        if (widgetInstance.UserPrefDictionary[formItemName].Datatype == WidgetUserPrefDataTypes.String)
                            defaultValue = Formatter.FormatSingleLinePlainTextForStorage(defaultValue);
                        widgetInstance.UserPrefDictionary[formItemName].DefaultValue = defaultValue;
                    }
                }

                if (widgetInstance.WidgetType == WidgetTypes.Html)
                {
                    //只允许管理员录入js脚本
                    if (CurrentUser.IsContentAdministrator)
                        widgetInstance.Content = Request.Form.GetString("WidgetContent", string.Empty);
                    else
                        widgetInstance.Content = HtmlScrubber.Clean(Request.Form.GetString("WidgetContent", string.Empty), false, true);
                }

                WidgetInstances.UpdateWidgetInstance(widgetInstance);
            }

            return new EmptyResult();
        }

        /// <summary>
        /// 添加WidgetInstance
        /// </summary>
        /// <param name="instanceID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateWidgetInstance(int presentAreaID, int ownerID, string widgetID, string theme, string pageCode)
        {
            PresentAreaThemeManager themeManager = GetPresentAreaThemeManager(presentAreaID);
            if (!themeManager.HasThemeManagerPermission(ownerID, CurrentUser))
            {
                string title = string.Format(GetResourceString("Exception_Pattern_CreateWidgetInstancePermissionRefused"), ownerID);
                return Redirect(SiteUrls.Instance().Error404(title));
            }

            WidgetInstances.CreateWidgetInstance(widgetID, ownerID, theme, pageCode);
            return new EmptyResult();
        }

        /// <summary>
        /// 检查currentUser对widgetInstance是否有管理权限
        /// </summary>
        /// <param name="instanceID"></param>
        /// <returns></returns>
        private bool HasManageWidgetInstancePermission(int instanceID)
        {

            WidgetInstance widgetInstance = WidgetInstances.GetWidgetInstance(instanceID, true);
            if (widgetInstance == null)
                return false;
            PresentAreaThemeManager themeManager = GetPresentAreaThemeManager(widgetInstance.PresentAreaID);

            return themeManager.HasThemeManagerPermission(widgetInstance.OwnerID, CurrentUser);
        }

        #endregion


        #region ThemeSettings

        /// <summary>
        /// 删除WidgetInstance
        /// </summary>
        /// <param name="instanceID"></param>
        /// <returns></returns>
        public ActionResult Control_ThemeSettings(int presentAreaID, int ownerID, string pageCode, string exitUrl)
        {
            PresentAreaThemeManager themeManager = GetPresentAreaThemeManager(presentAreaID);

            IList<ThemeAppearance> themeAppearances = Themes.GetThemeAppearancesOfEnabled(presentAreaID, true);
            KeyValuePair<string, string> selectedThemeAppearance = new KeyValuePair<string, string>(themeManager.GetSelectedThemeAppearanceName(ControllerContext), themeManager.GetSelectedThemeName(ControllerContext));

            IList<Widget> initialWidgets = Widgets.GetInitialWidgets(presentAreaID, true);

            IList<string> layoutModes = themeManager.LayoutModes;
            string currentLayoutMode = Themes.GetOwnerLayout(ownerID, selectedThemeAppearance.Value, pageCode, true);

            ViewData.Add("presentAreaID", presentAreaID);
            ViewData.Add("ownerID", ownerID);
            ViewData.Add("pageCode", pageCode);

            ViewData.Add("themeAppearances", themeAppearances);
            ViewData.Add("selectedThemeAppearance", selectedThemeAppearance);

            Dictionary<string, int> usedWidgetIDsOfPage = WidgetInstances.GetUsedWidgetIDs(ownerID, pageCode, selectedThemeAppearance.Value);
            ViewData.Add("initialWidgets", initialWidgets);
            ViewData.Add("usedWidgetIDsOfPage", usedWidgetIDsOfPage);

            ViewData.Add("layoutModes", layoutModes);
            ViewData.Add("currentLayoutMode", currentLayoutMode);

            ViewData.Add("themeLocation", themeManager.PresentArea.ThemeLocation);
            ViewData.Add("exitUrl", exitUrl);

            PresentArea presentArea = PresentAreas.GetPresentArea(presentAreaID, true);
            if (presentArea == null)
                throw new SPBException(ExceptionTypes.Instance().UnknownError());
            ViewData["presentArea"] = presentArea;

            return PartialView("Controls/Common/ThemeSettings.ascx");
        }


        /// <summary>
        /// 更新风格
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_UpdateThemeAppearance(int presentAreaID)
        {
            PresentAreaThemeManager themeManager = GetPresentAreaThemeManager(presentAreaID);
            string selectedTheme = Request.Form["selectedTheme"];
            string selectedAppearance = Request.Form["selectedAppearance"];
            int ownerID = Request.Form.Get<int>("ownerID", -1);

            if (!string.IsNullOrEmpty(selectedTheme) && !string.IsNullOrEmpty(selectedAppearance))
            {
                if (themeManager.HasThemeManagerPermission(ownerID, CurrentUser))
                    themeManager.SavaThemeAndAppearance(ControllerContext, selectedTheme, selectedAppearance);
            }
            return new EmptyResult();
        }


        /// <summary>
        /// 更新风格
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_UpdatePageLayout(int presentAreaID, string pageCode, string theme)
        {
            PresentAreaThemeManager themeManager = GetPresentAreaThemeManager(presentAreaID);
            int ownerID = Request.Form.Get<int>("ownerID", -1);
            int layoutMode = Request.Form.Get<int>("layoutMode", -1);
            User currentUser = Globals.GetCurrentUser();

            if (layoutMode > 0)
            {
                if (themeManager.HasThemeManagerPermission(ownerID, CurrentUser))
                {
                    Themes.SaveOwnerLayoutPage(ownerID, theme, pageCode, layoutMode);
                }
                else
                {
                    string title = GetResourceString("Exception_UpdatePageLayoutPermissionRefused");
                    return Redirect(SiteUrls.Instance().Error404(title));
                }
            }
            return new EmptyResult();
        }

        /// <summary>
        /// 还原默认装扮
        /// </summary>
        public JsonResult RestoreDefaultSettings(int presentAreaID, int ownerID, string pageCode, string theme)
        {
            PresentAreaThemeManager themeManager = GetPresentAreaThemeManager(presentAreaID);
            if (themeManager.HasThemeManagerPermission(ownerID, CurrentUser))
            {
                WidgetInstances.RestoreDefaultSettings(ownerID, theme, pageCode);
            }
            else
            {
                string title = GetResourceString("Exception_UpdatePageLayoutPermissionRefused");
                return Json(new { isSuccess = true, message = title }, JsonRequestBehavior.AllowGet);
            }
            return Json(new { isSuccess = true }, JsonRequestBehavior.AllowGet);
        }


        private PresentAreaThemeManager GetPresentAreaThemeManager(int presentAreaID)
        {
            if (presentAreaID == PresentAreaIDs.Instance().Channel())
                return ChannelPresentAreaThemeManager.Instance();
            else if (presentAreaID == PresentAreaIDs.Instance().UserDomain())
                return UserPresentAreaThemeManager.Instance();
            else if (presentAreaID == PresentAreaIDs.Instance().CompanyUserDomain())
                return CompanyUserPresentAreaThemeManager.Instance();
            else if (presentAreaID == PresentAreaIDs.Instance().ClubDomain())
                return ClubPresentAreaThemeManager.Instance();
            else if (presentAreaID == PresentAreaIDs.Instance().Manage())
                return ManagePresentAreaThemeManager.Instance();
            else if (presentAreaID == PresentAreaIDs.Instance().MicroBlogDomain())
                return MicroBlogPresentAreaThemeManager.Instance();
            return UserPresentAreaThemeManager.Instance();
        }

        #endregion


        ///// <summary>
        ///// 生成验证码图片
        ///// </summary>
        //public ActionResult ShowVerificationCodeImage(VerificationCodeDifficultyLevel level, RouteValueDictionary htmlValue, bool IsUseLineNoise, string tagID)
        //{

        //    TagBuilder img = new TagBuilder("img");

        //    string altMessage = ResourceManager.GetString("Description_ChangeVerificationCodeImage", Globals.GetCurrentUserLanguage());
        //    string strQuerying = Globals.GetFullUrlInMainSite(string.Format("~/Services/VerifyCodeImage.ashx?level={0}&r={1}&lineNoise={2}", ((int)level).ToString(), new Random().Next(), IsUseLineNoise));

        //    img.MergeAttribute("id", "codeImage_" + tagID);
        //    img.MergeAttribute("src", strQuerying);
        //    img.MergeAttribute("alt", altMessage);
        //    img.MergeAttribute("title", altMessage);
        //    string onclickString = "$('#codeImage_" + tagID + "').attr('src','" + strQuerying + "'+'&isremove=true&jsr='+Math.round(Math.random()*10000))";
        //    img.MergeAttribute("onclick", onclickString);
        //    img.MergeAttributes(htmlValue);

        //    string spanString = string.Format("<span class='tn-action'><a class='tn-action-text' href=\"javascript:;\" onclick=\"{1}\">{0}</a></span>", GetResourceString("Description_ChangeVerificationCodeImage"), onclickString);
        //    return Content(img.ToString(TagRenderMode.SelfClosing) + spanString);
        //}

        /// <summary>
        /// 设置用户系统采用什么语言浏览
        /// </summary>
        /// <param name="languageCode"></param>
        public ActionResult setUserLanguage(string languageCode, string returnUrl)
        {
            if (CurrentUser == null)
            {
                UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
                userCookie.SaveLanguageCode(languageCode);
            }
            else
            {
                CurrentUser.Language = languageCode;
                User _cUser = Globals.GetCurrentUser();
                _cUser.Language = languageCode;

                Users.UpdateUserLanguage(CurrentUser);
            }
            if (string.IsNullOrEmpty(returnUrl))
                returnUrl = SiteUrls.Instance().Home();
            return Redirect(returnUrl);
        }

        #region 标签录入插件
        /// <summary>
        /// 获取站点签并返回josn
        /// </summary>
        public JsonResult GetSuggestTagsJosn(string userTagItemKey)
        {
            //标签名称过滤条件
            string tagNameFilter = Request.QueryString.GetString("q", string.Empty);


            if (string.IsNullOrEmpty(tagNameFilter))
                return Json(new { }, JsonRequestBehavior.AllowGet);

            tagNameFilter = WebUtils.UrlDecode(tagNameFilter);

            //最大返回记录数
            int maxRecords = 10;


            List<string> tagNames = null;
            ICollection<UserTag> userTags = UserTagManager.Instance(userTagItemKey).GetUserTags(CurrentUser.UserID, false);
            if (userTags != null && userTags.Count > 0)
                tagNames = userTags.Select(n => n.TagName).Where(tagName => tagName.Contains(tagNameFilter)).ToList();
            if (tagNames == null)
                tagNames = new List<string>();
            if (tagNames.Count < maxRecords)
            {
                maxRecords = maxRecords - tagNames.Count;
                IList<SiteTag> tags = UserTagManager.Instance(userTagItemKey).GetSiteTagsFilterByTagName(tagNameFilter, maxRecords);
                if (tags != null && tags.Count > 0)
                {
                    tagNames = tagNames.Union(tags.Select(n => n.TagName).ToList()).ToList();
                }
            }
            return Json(tagNames.Select(n => new
            {
                value = n
            }), JsonRequestBehavior.AllowGet);
        }

        public JsonResult GetAppTagsJosn(string applicationTagItemKey)
        {
            //标签名称过滤条件
            string tagNameFilter = Request.QueryString.GetString("q", string.Empty);

            if (string.IsNullOrEmpty(tagNameFilter))
                return Json(new { }, JsonRequestBehavior.AllowGet);

            tagNameFilter = WebUtils.UrlDecode(tagNameFilter);

            //最大返回记录数
            int maxRecords = 20;

            List<string> tagNames = null;

            if (!string.IsNullOrEmpty(tagNameFilter))
            {
                IList<ApplicationTag> tags = ApplicationTagManager.Instance(applicationTagItemKey).GetApplicationTagsFilterByTagName(tagNameFilter, maxRecords);

                tagNames = tags.Select(n => n.TagName).ToList();
            }

            if (tagNames == null)
                tagNames = new List<string>();
            return Json(tagNames.Select(n => new
            {
                value = n
            }), JsonRequestBehavior.AllowGet);
        }
        #endregion

        #region 好友选择器
        /// <summary>
        /// 显示好友选择器
        /// </summary>
        /// <param name="controlName">控件Name</param>
        /// <param name="selectedUserIDs">初始选中值</param>
        /// <param name="selectionLimit">最大可选好友数</param>
        public ActionResult Control_FriendSelector(string controlName, string selectedUserIDs, int? selectionLimit)
        {
            //List<SimpleFriend> simpleFriends = new List<SimpleFriend>();
            //if (!string.IsNullOrEmpty(selectedUserIDs))
            //{
            //    string[] strUserIDs = selectedUserIDs.Split(UserTagManager.SplitCharacters);
            //    foreach (string strUserID in strUserIDs)
            //    {
            //        int userID = 0;
            //        int.TryParse(strUserID, out userID);
            //如果不是朋友，那么就获取不到朋友，但是应该可以发私信
            //        Friend friend = Friends.GetFriendByUserID(CurrentUser.UserID, userID);
            //        SimpleFriend simpleFriend = null;
            //        if (friend != null)
            //            simpleFriend = new SimpleFriend(friend);
            //        if (simpleFriend != null)
            //            simpleFriends.Add(simpleFriend);
            //    }
            //}

            IList<UserIdentifier> identifiers = new List<UserIdentifier>();
            if (!string.IsNullOrEmpty(selectedUserIDs))
            {
                string[] strUserIDs = selectedUserIDs.Split(UserTagManager.SplitCharacters);
                foreach (string strUserID in strUserIDs)
                {
                    int userID = 0;
                    int.TryParse(strUserID, out userID);
                    UserIdentifier userIdentifier = Users.GetUserIdentifier(userID);
                    if (userIdentifier != null)
                        identifiers.Add(userIdentifier);
                }
            }
            return PartialView("Controls/Common/FriendSelector.ascx", identifiers);
        }

        /// <summary>
        /// 获取当前用户的朋友列表
        /// </summary>
        /// <param name="categoryID"></param>
        /// <returns></returns>
        public JsonResult GetMyFriends(int? categoryID)
        {

            List<SimpleFriend> simpleFriends = Friends.GetAllSimpleFriends(CurrentUser != null ? CurrentUser.UserID : 0, categoryID, true);

            return Json(
           simpleFriends
           .Select(n => new
           {
               userID = n.FriendUserID,
               displayName = string.IsNullOrEmpty(n.NoteName) ? n.DisplayName : n.NoteName,
               commonName = n.CommonName,
               noteName = n.NoteName,
               nickName = n.NickName,
               userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.UserID, AvatarSizeTypes.Small)
           }), JsonRequestBehavior.AllowGet);

            //IList<UserIdentifier> userIdentifiers = Friends.GetMyFriends(CurrentUser.UserID, categoryID);
            //return Json(
            //userIdentifiers
            //.Select(n => new
            //{
            //    userID = n.UserID,
            //    displayName = n.DisplayName,
            //    commonName = n.CommonName,
            //    nickName = n.NickName,
            //    userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.UserID, AvatarSizeTypes.Small)
            //}), JsonRequestBehavior.AllowGet);
        }
        /// <summary>
        /// 搜索用户
        /// </summary>
        public JsonResult SearchFriends()
        {
            string term = Request.QueryString.GetString("q", string.Empty);
            if (string.IsNullOrEmpty(term))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            term = WebUtils.UrlDecode(term);

            List<SimpleFriend> simpleFriends = Friends.GetAllSimpleFriends(CurrentUser != null ? CurrentUser.UserID : 0, null, true);

            var selectList = simpleFriends
            .Where(
            n => (
            n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.NoteName.Contains(term)
            || n.UserName.Contains(term))))
            .Select(n => new
            {
                userID = n.FriendUserID,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName,
                noteName = n.NoteName,
                userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.FriendUserID, AvatarSizeTypes.Small)
            });

            return Json(selectList, JsonRequestBehavior.AllowGet);

            //IList<UserIdentifier> userIdentifiers = Friends.GetMyFriends(CurrentUser.UserID, null);
            //return Json(userIdentifiers
            //.Where(
            //n => (
            //n != null
            //&& (n.CommonName.Contains(term)
            //|| n.NickName.Contains(term)
            //|| n.UserName.Contains(term))))
            //.Select(n => new
            //{
            //    userID = n.UserID,
            //    displayName = n.DisplayName,
            //    commonName = n.CommonName,
            //    nickName = n.NickName,
            //    userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.UserID, AvatarSizeTypes.Small)
            //}), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 搜索用户
        /// </summary>
        public JsonResult SearchUsers()
        {
            string term = Request.QueryString.GetString("q", string.Empty);
            if (string.IsNullOrEmpty(term))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            term = WebUtils.UrlDecode(term);


            List<SimpleFriend> simpleFriends = Friends.GetAllSimpleFriends(CurrentUser != null ? CurrentUser.UserID : 0, null, true);

            var selectList = simpleFriends
            .Where(
            n => (
            n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.NoteName.Contains(term)
            || n.UserName.Contains(term))))
            .Select(n => new
            {
                userID = n.FriendUserID,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName,
                noteName = n.NoteName,
                userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.FriendUserID, AvatarSizeTypes.Small)
            });

            IList<MicroBlog.MicroBlogSimpleFollow> follows = MicroBlog.MicroBlogFollows.GetAllSimpleFollows(CurrentUser.UserID, null, true);

            if (follows != null)
            {
                selectList = selectList.Concat(follows.Where(n => n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.UserName.Contains(term))).Select(n => new
            {
                userID = n.ItemID,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName,
                noteName = "",
                userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.ItemID, AvatarSizeTypes.Small)
            }));
            }
            return Json(selectList.Distinct(), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 获取我朋友的类别
        /// </summary>
        public JsonResult GetMyFriendCategories()
        {
            List<UserCategory> categories = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategories(CurrentUser.UserID, false);
            return Json(
            categories.Select(n => new
            {
                value = n.CategoryID,
                name = n.CategoryName
            }), JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region 顶评价
        /// <summary>
        /// 获取
        /// </summary>
        public JsonResult Digg(int itemID, string typeName)
        {
            if (itemID <= 0 || CurrentUser.UserID <= 0)
                return Json(new { isSuccess = false }, JsonRequestBehavior.AllowGet);

            Type type = Type.GetType(typeName);
            if (type != null)
            {
                IDiggable dig = Activator.CreateInstance(type) as IDiggable;
                if (dig == null)
                    return Json(new { isSuccess = false }, JsonRequestBehavior.AllowGet);
                Digger digger = new Digger();
                digger.ItemID = itemID;
                digger.UserID = CurrentUser.UserID;
                digger.Author = CurrentUser.DisplayName;
                dig.Digg(digger);
            }
            return Json(new { isSuccess = true }, JsonRequestBehavior.AllowGet);
        }
        #endregion

        #region 公告
        /// <summary>
        /// 公告列表
        /// </summary>

        public ActionResult ListAnnouncements(int? presentAreaID)
        {
            SetPageTitle("站点公告");
            int pageindex = Request.QueryString.GetInt("pageindex", 1);
            PagingDataSet<Announcement> announcements = Announcements.GetAnnouncements(presentAreaID, pageindex);
            BreadCrumb.AddBreadCrumbItem("站点公告", string.Empty);
            return View("Pages/Announcements/ListAnnouncements.aspx", "Masters/Announcement.Master", announcements);
        }

        /// <summary>
        /// 用于显示在频道和群组页的公告条
        /// </summary>

        public ActionResult Control_AnnouncementsBlockForLong(int? presentAreaID, int topNumber)
        {
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            IList<int> threadIDs = userCookie.GetReadedAnnouncementIDs();
            IList<Announcement> announcements = Announcements.GetTopThreads(presentAreaID, threadIDs, topNumber);
            return PartialView("Controls/Announcements/AnnouncementsBlockForLong.ascx", announcements);

        }
        /// <summary>
        /// 用于显示在个人中心和微博的公告条
        ///</summary>

        public ActionResult Control_AnnouncementsBlockForShort(int? presentAreaID, int topNumber)
        {
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            IList<int> threadIDs = userCookie.GetReadedAnnouncementIDs();

            IList<Announcement> announcements = Announcements.GetTopThreads(presentAreaID, threadIDs, topNumber);

            return PartialView("Controls/Announcements/AnnouncementsBlockForShort.ascx", announcements);
        }


        /// <summary>
        /// 公告详细显示页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserVisitAndUserDomainHitTime]
        public ActionResult ShowAnnouncement(int threadID)
        {

            Announcement announcement = Announcements.GetAnnouncement(threadID, false);
            if (announcement == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("ID为{0}的公告不存在", threadID)));
            }

            #region 保存阅读状态
            UserCookie userCookie = new UserCookie(CurrentUser, HttpContext);
            userCookie.SaveReadedAnnouncementIDs(threadID);
            #endregion

            #region 浏览计数
            CounterFactory.IncreaseCount<AnnouncementCounter>(IncreasableCountTypes.WebView, threadID);
            #endregion

            if (announcement.IsHyperLink)
                return Redirect(announcement.HyperLinkUrl);
            BreadCrumb.AddBreadCrumbItem("站点公告", SPBUrlHelper.Action("ListAnnouncements", "Channel"));
            BreadCrumb.AddBreadCrumbItem(announcement.Subject, string.Empty);
            SetPageTitle(announcement.Subject);

            return View("Pages/Announcements/ShowAnnouncement.aspx", "Masters/Announcement.Master", announcement);

        }
        /// <summary>
        /// 设置置顶
        /// </summary>
        //@preview: SetTop-> SetSticky
        public ActionResult SetSticky(int threadID, bool isSticky)
        {

            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().AdministrationAccessDenied()));
            Announcements.SetSticky(threadID, isSticky);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            if (isSticky)
                TempData["StatusMessageContent"] = "置顶成功";
            else
                TempData["StatusMessageContent"] = "取消置顶成功";

            return RedirectToAction("ShowAnnouncement", new { threadID = threadID });
        }
        /// <summary>
        /// 删除公告
        /// </summary>
        public ActionResult DeleteAnnouncement(int threadID)
        {

            if (CurrentUser == null || !CurrentUser.IsContentAdministrator)
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().AdministrationAccessDenied()));
            Announcements.DeleteAnnouncement(threadID);
            return RedirectToAction("ListAnnouncements");
        }
        #endregion
    }

    public enum ThemeSettingSubMenus
    {
        /// <summary>
        /// 风格
        /// </summary>
        ThemeAppearanceList = 1,
        /// <summary>
        /// 布局
        /// </summary>
        LayoutModeList = 2,
        /// <summary>
        /// 模块
        /// </summary>
        WidgetList = 3
    }

}
