﻿//------------------------------------------------------------------------------
//<copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
//</copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using System.Web.Routing;
using SpaceBuilder.Web;
using SpaceBuilder.Common;
using System.Web.Security;
using System.Web;
using SpaceBuilder.LuceneSearch;
using SpaceBuilder.Utils;
using System.Configuration;
using Jayrock.Json.Conversion;
using QzoneSDK.Models;
using System.IO;
using System.Net;

namespace SpaceBuilder.Common.Controllers
{
    [HttpCompress]
    public class ChannelUserController : ChannelControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().User();
        /// <summary>
        /// 登录显示页面
        /// </summary>
        public ActionResult Login(string token, string returnUrl)
        {
            if (!string.IsNullOrEmpty(returnUrl))
                ViewData["returnUrl"] = returnUrl;
            SetPageTitle(GetResourceString("Action_Login"));
            return View("Pages/User/Login.aspx", "Masters/SimpleHeader.master");
        }

        /// <summary>
        /// 登录显示页面
        /// </summary>
        public ActionResult LoginInModal(string returnUrl)
        {
            return View("Controls/User/LoginInModal.ascx");
        }

        /// <summary>
        /// 搜索用户
        /// </summary>
        public JsonResult SearchUsers()
        {
            string term = Request.QueryString.GetString("term", string.Empty);
            if (string.IsNullOrEmpty(term))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            term = WebUtils.UrlDecode(term);
            PersonUserFullTextQuery personUserQuery = new PersonUserFullTextQuery();
            personUserQuery.Keyword = term;
            IList<PersonUser> users = PersonUserSearchManager.Instance().Search(CurrentUser, personUserQuery).Records;
            return Json(users.Select(n => n)
            .Where(
            n => (
            n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.UserName.Contains(term))))
            .Select(n => new
            {
                userID = n.UserID,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName,
                userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.UserID, AvatarSizeTypes.Small)
            }), JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 处理用户登录
        /// </summary>
        /// <param name="loginInModal">是否在模式窗口登录</param>
        /// <param name="returnUrl">登录成功后要跳转的url</param>
        /// <param name="token">邀请码</param>
        public ActionResult LoginAction(bool loginInModal, string returnUrl, string token, int? accountType, string openID, string OauthToken, string OauthTokenSecret)
        {
            string userName;
            string password;
            bool autoLogin;
            if (loginInModal)
            {
                userName = Request.Form.GetString("loginInModalUserName", string.Empty).Trim();
                password = Request.Form.GetString("loginInModalPassword", string.Empty).Trim();
                autoLogin = Request.Form.GetBool("loginInModalAutoLogin", false);
            }
            else
            {
                userName = Request.Form.GetString("userName", string.Empty).Trim();
                password = Request.Form.GetString("password", string.Empty).Trim();
                autoLogin = Request.Form.GetBool("autoLogin", false);
            }

            //模式窗口登录 不需要录入验证码
            if (!loginInModal)
            {
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (verifyCode != null)
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    TempData["UserName"] = userName;
                    TempData["Password"] = password;
                    return RedirectToAction("login", new RouteValueDictionary { { "returnUrl", returnUrl }, { "token", token } });
                }
            }

            User userToLogin = new User();
            userToLogin.UserName = userName;
            userToLogin.Password = password;

            LoginUserStatuses loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);

            //如果作为用户名登录失败，则作为Email重试
            if (loginStatus == LoginUserStatuses.InvalidCredentials)
            {
                User userByEmail = Users.FindUserByEmail(userName);
                if (userByEmail != null)
                {
                    if (!userByEmail.IsEmailVerified)
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        string emailAddress = EmailAddressConfiguration.GetEmailAddress(userName);
                        string emailVerifiedMessage = string.Empty;
                        if (!string.IsNullOrEmpty(emailAddress))
                            emailVerifiedMessage = string.Format(GetResourceString("Message_Pattern_UserLoginFailedForEmailVerified", applicationID), emailAddress);
                        TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForUserName", applicationID) + emailVerifiedMessage;
                        TempData["UserName"] = userName;
                        TempData["Password"] = password;

                        return RedirectToAction("login", new RouteValueDictionary { { "returnUrl", returnUrl }, { "token", token } });
                    }


                    userToLogin.UserName = userByEmail.UserName;
                    loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
                }
            }

            //是否允许未批准用户登录
            bool enablePendingUsersToLogin = SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin;

            // Change to let banned users in
            //
            if (loginStatus == LoginUserStatuses.Success || (enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending))
            {
                /** 
                 // TODO 不允许登录设置
                if (!SiteSettingsManager.GetSiteSettings().AllowLogin && !userToLogin.IsAdministrator) {
                    throw new SBException(SBExceptionType.UserLoginDisabled);
                }
                **/

                if (autoLogin)
                {
                    //HttpCookie formsAuthCookie;
                    //formsAuthCookie = FormsAuthentication.GetAuthCookie(userToLogin.UserName, true);
                    //formsAuthCookie.Expires = DateTime.Now.AddDays(30);
                    //this.Context.Response.Cookies.Add(formsAuthCookie);

                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, true);
                }
                else
                {
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, false);
                }

                userToLogin = Users.GetUser(userName, false);

                #region 账号绑定
                if (!string.IsNullOrEmpty(openID) && userToLogin != null
                    && AccountBindings.GetUser(accountType ?? AccountTypes.Instance().QQ(), openID) == null)
                {

                    AccountBinding account = new AccountBinding();
                    account.AccountType = accountType ?? AccountTypes.Instance().QQ();
                    account.Identification = openID;
                    account.UserID = userToLogin.UserID;
                    account.OauthToken = OauthToken;
                    account.OauthTokenSecret = OauthTokenSecret;
                    AccountBindings.CreateAccountBinding(account);
                }
                #endregion


                //模式窗口登录 不处理邀请登录
                if (!loginInModal)
                {
                    int inviteUserID = 0;
                    int inviteClubID = 0;
                    if (!string.IsNullOrEmpty(token))
                    {
                        try
                        {
                            string detoken = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
                            string[] qs = detoken.Split(',');
                            DateTime dateCreated = DateTime.Parse(qs[1]);
                            if (dateCreated.CompareTo(DateTime.Now) < 0)
                                return Redirect(SiteUrls.Instance().Error("链接已过期"));
                            inviteUserID = int.Parse(qs[0]);
                            if (qs.Count() > 2)
                                inviteClubID = int.Parse(qs[2]);
                        }
                        catch { }

                        //处理朋友邀请
                        if (inviteUserID > 0 && userToLogin.UserID != inviteUserID && inviteClubID <= 0)
                        {
                            if (!Friends.IsFriend(userToLogin.UserID, inviteUserID))
                            {
                                User inviteUser = Users.GetUser(inviteUserID, false);
                                if (inviteUser != null && inviteUser.UserType == UserTypes.PersonUser)
                                    SpaceBuilder.Common.Friends.CreateFriend(userToLogin.UserID, inviteUser.UserID);
                            }
                        }

                        if (inviteClubID > 0)
                        {

                            if (!Club.ClubMembers.IsClubMember(inviteClubID, userToLogin.UserID))
                            {
                                Club.ClubMember member = new SpaceBuilder.Club.ClubMember();
                                member.InviteUserID = inviteUserID;
                                member.UserID = userToLogin.UserID;
                                member.ClubID = inviteClubID;
                                member.DisplayName = userToLogin.DisplayName;
                                member.Status = Club.ClubMemberStatuses.IsApproved;
                                member.ClubMemberRole = SpaceBuilder.Club.ClubMemberRoles.ClubMember;
                                Club.ClubMembers.CreateClubMember(member);
                            }
                        }
                    }
                }


                string redirectUrl = null;


                // Get the link from the context
                if (!string.IsNullOrEmpty(Request.QueryString["returnUrl"]))
                    redirectUrl = Request.QueryString.GetString("returnUrl", string.Empty);

                if (string.IsNullOrEmpty(redirectUrl) || (redirectUrl.IndexOf("ShowExceptionMessage") != -1)
                    || (redirectUrl.IndexOf("ChangePassword") != -1)
                    || (redirectUrl.IndexOf("EmailForgottenPassword") != -1))
                {
                    redirectUrl = SiteUrls.Instance().UserDomainHome(userToLogin);
                }

                return new RedirectResult(redirectUrl);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (loginStatus == LoginUserStatuses.InvalidCredentials)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForUserNameOrPassword", applicationID);
                }
                else if (!enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountPending", applicationID);
                }
                else if (loginStatus == LoginUserStatuses.AccountDisapproved)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountDisapproved", applicationID);
                }
                else if (loginStatus == LoginUserStatuses.UnknownError)
                {
                    throw new SPBException(ExceptionTypes.Instance().UserUnknownLoginError());
                }
                else if (loginStatus == LoginUserStatuses.AccountBanned)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountBanned", applicationID);
                }
                TempData["UserName"] = userName;
                TempData["Password"] = password;

                return RedirectToAction("login", new RouteValueDictionary { { "returnUrl", returnUrl }, { "token", token }, { "accountType", accountType }, { "openID", openID }, { "OauthToken", OauthToken }, { "OauthTokenSecret", OauthTokenSecret } });
            }
        }

        /// <summary>
        /// 显示简洁首页登陆框
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_LoginInSample()
        {
            return PartialView("Controls/User/LoginInSample.ascx");
        }
        /// <summary>
        /// 简洁首页登陆
        /// </summary>
        /// <returns></returns>
        public ActionResult LoginInSampleAction()
        {
            string userName = Request.Form.GetString("LoginUserName", string.Empty).Trim();
            string password = Request.Form.GetString("LoginPassword", string.Empty).Trim();
            bool autoLogin = Request.Form.GetBool("AutoLogin", false);

            User userToLogin = new User();
            userToLogin.UserName = userName;
            userToLogin.Password = password;

            LoginUserStatuses loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
            //如果作为用户名登录失败，则作为Email重试
            if (loginStatus == LoginUserStatuses.InvalidCredentials)
            {
                User userByEmail = Users.FindUserByEmail(userName);
                if (userByEmail != null)
                {
                    userToLogin.UserName = userByEmail.UserName;
                    loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
                }
            }

            //是否允许未批准用户登录
            bool enablePendingUsersToLogin = SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin;

            if (loginStatus == LoginUserStatuses.Success || (enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending))
            {
                if (autoLogin)
                {
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, true);
                }
                else
                {
                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, false);
                }
                return Redirect(SiteUrls.Instance().UserDomainHome(userToLogin.UserName));
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (loginStatus == LoginUserStatuses.InvalidCredentials)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForUserNameOrPassword", applicationID);
                }
                else if (!enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountPending", applicationID);
                }
                else if (loginStatus == LoginUserStatuses.AccountDisapproved)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountDisapproved", applicationID);
                }
                else if (loginStatus == LoginUserStatuses.UnknownError)
                {
                    throw new SPBException(ExceptionTypes.Instance().UserUnknownLoginError());
                }
                else if (loginStatus == LoginUserStatuses.AccountBanned)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UserLoginFailedForAccountBanned", applicationID);
                }
                return RedirectToAction("Login");
            }
        }

        /// <summary>
        /// 注销
        /// </summary>
        //TODO: 暂时方法
        public ActionResult Logout()
        {
            ///移除在线用户
            if (CurrentUser != null)
                UserOnlines.DeleteOnlineUser(CurrentUser.UserID);

            FormsAuthentication.SignOut();

            string returnUrl;
            if (Request.UrlReferrer != null)
                returnUrl = Request.UrlReferrer.AbsoluteUri;
            else
                returnUrl = SiteUrls.Instance().Home();

            return new RedirectResult(returnUrl);
        }


        /// <summary>
        /// 注册用户
        /// </summary>
        public ActionResult Register(string token)
        {
            SetPageTitle(GetResourceString("Action_Register"));

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.RegistrationMode == RegistrationModes.DisableAll)
            {
                return Redirect(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().UserAccountRegistrationDisabled()));
            }
            if (settings.RegistrationMode == RegistrationModes.OnlyByInvitation)
            {
                if (string.IsNullOrEmpty(token))
                    token = TempData.GetString("Token", string.Empty);

                int inviteUserID = 0;
                string detoken = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
                string[] qs = detoken.Split(',');
                DateTime dateCreated = DateTime.Parse(qs[1]);
                if (dateCreated.CompareTo(DateTime.Now) < 0)
                    return Redirect(SiteUrls.Instance().Error("链接已过期"));
                inviteUserID = int.Parse(qs[0]);

                User user = Users.GetUser(inviteUserID, false);
                if (user == null)
                    return Redirect(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().UserAccountRegistrationDisabled()));
            }

            ViewData["UserNameMinLength"] = settings.UserNameMinLength;
            ViewData["UserNameMaxLength"] = settings.UserNameMaxLength;
            ViewData["UserNameRegex"] = settings.UserNameRegex;
            ViewData["PasswordRegex"] = settings.PasswordRegex;

            return View("Pages/User/Register.aspx", "Masters/SimpleHeader.master");
        }

        /// <summary>
        /// 注册企业用户
        /// </summary>
        public ActionResult CompanyRegister()
        {
            SetPageTitle(GetResourceString("Title_CompanyUserRegister", applicationID));
            ViewData["CompanyPropertys"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().CompanyProperty());
            ViewData["Trades"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().Trade());
            ViewData["AreaDepth"] = 1;
            ViewData["Funds"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().Fund());
            ViewData["EmployeeCounts"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().EmployeeCount());
            if (SiteSettingsManager.GetSiteSettings().IsCloseCompanyUser)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_CompanyUserIsClosed", applicationID);
            }
            return View("Pages/User/CompanyRegister.aspx", "Masters/SimpleHeader.master");
        }
        /// <summary>
        /// 注册企业用户
        /// </summary>
        public ActionResult CompanyRegisterAction()
        {
            if (SiteSettingsManager.GetSiteSettings().IsCloseCompanyUser)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_CompanyUserIsClosed", applicationID);
                return RedirectToAction("CompanyRegister");
            }
            string userName = Request.Form.GetString("userName", string.Empty).Trim();

            string password = Request.Form.GetString("password", string.Empty).Trim();
            string email = Request.Form.GetString("userEmail", string.Empty).Trim();
            string Licence = Request.Form.GetString("Licence", string.Empty).Trim();
            int Trades = Request.Form.GetInt("Trades", -1);
            string areaCode = Request.Form.GetString("areaCode", string.Empty);
            int CompanyPropertys = Request.Form.GetInt("CompanyPropertys", -1);
            int Funds = Request.Form.GetInt("Funds", -1);
            int EmployeeCounts = Request.Form.GetInt("EmployeeCounts", -1);
            string Linkman = Request.Form.GetString("Linkman", string.Empty).Trim();
            string LinkmanDuty = Request.Form.GetString("LinkmanDuty", string.Empty).Trim();
            string LinkTel = Request.Form.GetString("LinkTel", string.Empty).Trim();
            string Fax = Request.Form.GetString("Fax", string.Empty).Trim();
            string Address = Request.Form.GetString("Address", string.Empty).Trim();
            string Intro = Request.Form.GetString("Intro", string.Empty).Trim();
            string CommonName = Request.Form.GetString("CommonName", string.Empty).Trim();

            CompanyUser u = new CompanyUser();
            u.UserName = userName;
            u.Password = password;
            u.PublicEmail = email;
            u.PrivateEmail = email;
            u.Licence = Licence;
            u.Trade = Trades;
            u.AreaCode = areaCode;
            u.CompanyProperty = CompanyPropertys;
            u.CommonName = CommonName;
            u.Fund = Funds;
            u.EmployeeNum = EmployeeCounts;
            u.Linkman = Linkman;
            u.LinkmanDuty = LinkmanDuty;
            u.LinkTel = LinkTel;
            u.Fax = Fax;
            u.Address = Address;
            u.Intro = Intro;
            u.UserType = UserTypes.CompanyUser;
            u.AccountStatus = SiteSettingsManager.GetSiteSettings().IsAuditCompanyUser ? UserAccountStatuses.ApprovalPending : UserAccountStatuses.Approved;
            u.IsModerated = SiteSettingsManager.GetSiteSettings().DefaultModeratedStatus;
            string acceptArgeement = Request.Form.GetString("acceptArgeement", string.Empty).Trim();
            if (string.IsNullOrEmpty(acceptArgeement))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForNotAgreeWithAgreement", applicationID);
                TempData["companyUser"] = u;
                return RedirectToAction("CompanyRegister");
            }
            //检查验证码是否录入正确
            string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
            string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
            if (verifyCode != null)
                verifyCode = verifyCode.Trim();
            if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                TempData["companyUser"] = u;
                return RedirectToAction("CompanyRegister");
            }
            string errorMessage = string.Empty;
            if (!Globals.ValidateUserName(userName, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                TempData["companyUser"] = u;
                return RedirectToAction("CompanyRegister");
            }
            CreateUpdateUserStatuses status = Users.Create(u, true);
            switch (status)
            {
                // Username already exists!
                case CreateUpdateUserStatuses.DuplicateUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForDuplicateUsername", applicationID);
                    TempData["companyUser"] = u;
                    break;
                case CreateUpdateUserStatuses.BeyondLicensedCount:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForBeyondLicensedCount", applicationID);
                    TempData["companyUser"] = u;
                    break;

                // Email already exists!
                case CreateUpdateUserStatuses.DuplicateEmailAddress:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_RegisterFailedForDuplicateEmailAddress", applicationID), SiteUrls.Instance().ForgotPassword());
                    TempData["companyUser"] = u;
                    break;

                // Unknown failure has occurred!
                case CreateUpdateUserStatuses.UnknownFailure:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForUnknownFailure", applicationID);
                    TempData["companyUser"] = u;
                    break;

                // Username is disallowed
                case CreateUpdateUserStatuses.DisallowedUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForDisallowedUsername", applicationID);
                    TempData["companyUser"] = u;
                    break;

                case CreateUpdateUserStatuses.InvalidQuestionAnswer:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForInvalidQuestionAnswer", applicationID);
                    TempData["companyUser"] = u;
                    break;

                case CreateUpdateUserStatuses.InvalidPassword:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForInvalidPassword", applicationID);
                    TempData["companyUser"] = u;
                    break;

                // Everything went off fine, good
                case CreateUpdateUserStatuses.Created:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    if (SiteSettingsManager.GetSiteSettings().IsAuditCompanyUser && !SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin)
                        TempData["StatusMessageContent"] = GetResourceString("Message_RegisterSucceed", applicationID);
                    if (SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin)
                    {//TempData["StatusMessageContent"] = "注册成功，你可以立刻登陆！";
                        FormsAuthentication.SetAuthCookie(u.UserName, false);
                        LeaveSecureConnection(SiteUrls.Instance().UserDomainHome(u));
                    }
                    break;

            }
            return RedirectToAction("CompanyRegister");
        }

        /// <summary>
        /// 注册用户
        /// </summary>
        /// <param name="token">邀请码</param>
        public ActionResult RegisterAction(string token)
        {
            PersonUser personUser = new PersonUser();
            personUser.Language = Globals.GetCurrentUserLanguage();
            personUser.UserName = Request.Form.GetString("userName", string.Empty).Trim();
            personUser.Password = Request.Form.GetString("password", string.Empty).Trim();
            personUser.PrivateEmail = Request.Form.GetString("userEmail", string.Empty).Trim();
            personUser.CommonName = Request.Form.GetString("commonName", string.Empty).Trim();
            personUser.NickName = Request.Form.GetString("nickName", personUser.CommonName).Trim();
            personUser.Gender = (Genders)Request.Form.GetInt("gender", 1);
            personUser.IsModerated = SiteSettingsManager.GetSiteSettings().DefaultModeratedStatus;

            //检查验证码是否录入正确
            string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
            string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
            if (verifyCode != null)
                verifyCode = verifyCode.Trim();

            if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                TempData["personUser"] = personUser;
                return RedirectToAction("Register", new RouteValueDictionary { { "token", token } });
            }
            string errorMessage = "";

            if (!Globals.ValidateUserName(personUser.UserName, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                TempData["personUser"] = personUser;

                return RedirectToAction("Register", new RouteValueDictionary { { "token", token } });
            }
            if (!Globals.ValidatePassword(personUser.Password, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                TempData["personUser"] = personUser;

                return RedirectToAction("Register", new RouteValueDictionary { { "token", token } });
            }

            CreateUpdateUserStatuses status = CreateUpdateUserStatuses.UnknownFailure;
            AccountActivations activation = SiteSettingsManager.GetSiteSettings().AccountActivation;
            // Does the user require approval?
            if (activation == AccountActivations.Automatic)
                personUser.AccountStatus = UserAccountStatuses.Approved;
            else
                personUser.AccountStatus = UserAccountStatuses.ApprovalPending;
            User inviteUser = null;
            int inviteUserID = 0;
            int inviteClubID = 0;
            if (!string.IsNullOrEmpty(token))
            {
                try
                {
                    string detoken = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
                    string[] qs = detoken.Split(',');
                    DateTime dateCreated = DateTime.Parse(qs[1]);
                    if (dateCreated.CompareTo(DateTime.Now) < 0)
                        return Redirect(SiteUrls.Instance().Error("链接已过期"));
                    inviteUserID = int.Parse(qs[0]);
                    if (qs.Count() > 2)
                        inviteClubID = int.Parse(qs[2]);
                }
                catch { }

                if (inviteUserID > 0 && inviteClubID <= 0)
                {
                    inviteUser = Users.GetUser(inviteUserID, false);
                    if (inviteUser != null)
                        personUser.InviteUserID = inviteUser.UserID;
                }

            }

            if (personUser.UserName.Equals("Anonymous", StringComparison.CurrentCultureIgnoreCase))
            {
                status = CreateUpdateUserStatuses.DuplicateUsername;
            }
            else
            {
                try
                {
                    personUser = ConvertUserForStorage(personUser);
                    status = Users.Create(personUser, true);
                }
                catch
                {
                    //status = exception.;
                }
            }

            // Determine if the account was created successfully
            switch (status)
            {
                // Username already exists!
                case CreateUpdateUserStatuses.DuplicateUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForDuplicateUsername", applicationID);
                    break;
                case CreateUpdateUserStatuses.BeyondLicensedCount:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForBeyondLicensedCount", applicationID);
                    break;

                // Email already exists!
                case CreateUpdateUserStatuses.DuplicateEmailAddress:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_RegisterFailedForDuplicateEmailAddress", applicationID), SiteUrls.Instance().ForgotPassword());
                    break;

                // Unknown failure has occurred!
                case CreateUpdateUserStatuses.UnknownFailure:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForUnknownFailure", applicationID);
                    break;

                // Username is disallowed
                case CreateUpdateUserStatuses.DisallowedUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForDisallowedUsername", applicationID);
                    break;

                case CreateUpdateUserStatuses.InvalidQuestionAnswer:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForInvalidQuestionAnswer", applicationID);
                    break;

                case CreateUpdateUserStatuses.InvalidPassword:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForInvalidPassword", applicationID);
                    break;

                // Everything went off fine, good
                case CreateUpdateUserStatuses.Created:
                    //建立朋友连接
                    if (inviteUser != null && inviteClubID <= 0 && inviteUser.UserType == UserTypes.PersonUser)
                    {
                        Friends.CreateFriend(inviteUser.UserID, personUser.UserID);

                        #region 触发 成功邀请站外朋友事件
                        Friend newFriend = new Friend();
                        newFriend.UserID = inviteUser.UserID;
                        newFriend.FriendUserID = personUser.UserID;
                        FriendEvents.SuccessInviteFriend(newFriend);
                        #endregion

                        #region 更新邀请者 邀请的朋友数
                        inviteUser.InviteFriendCount += 1;
                        Users.UpdateUserStatistics(inviteUser);
                        #endregion
                    }

                    if (inviteClubID > 0)
                    {
                        if (!Club.ClubMembers.IsClubMember(inviteClubID, personUser.UserID))
                        {
                            Club.ClubMember member = new SpaceBuilder.Club.ClubMember();
                            member.InviteUserID = inviteUserID;
                            member.UserID = personUser.UserID;
                            member.ClubID = inviteClubID;
                            member.DisplayName = personUser.DisplayName;
                            member.Status = Club.ClubMemberStatuses.IsApproved;
                            member.ClubMemberRole = SpaceBuilder.Club.ClubMemberRoles.ClubMember;
                            Club.ClubMembers.CreateClubMember(member);
                        }
                    }
                    switch (activation)
                    {
                        case AccountActivations.AdminApproval:
                            Response.Redirect(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().UserAccountPending()));
                            break;

                        case AccountActivations.Email:
                            LeaveSecureConnection(SPBUrlHelper.Action("UserAccountCreatedValidateEmail", "ChannelUser", new RouteValueDictionary { { "userName", WebUtils.UrlEncode(personUser.UserName) } }));
                            break;

                        case AccountActivations.Automatic:
                            if (!Request.IsAuthenticated)
                                FormsAuthentication.SetAuthCookie(personUser.UserName, false);
                            LeaveSecureConnection(SiteUrls.Instance().UserDomainHome(personUser));
                            break;
                    }
                    return new EmptyResult();
            }
            TempData["personUser"] = personUser;
            return RedirectToAction("Register", new RouteValueDictionary { { "token", token } });
        }


        /// <summary>
        /// 显示上传头像 进入向导模式
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UploadAvatar(string userDomainName)
        {
            SetPageTitle("上传头像");
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            if (currentDomainUser.HasAvatar)
                ViewData["canCropAvatar"] = true;

            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/User/UploadAvatar.aspx", "Masters/SimpleHeader.master", currentDomainUser);
        }

        /// <summary>
        /// 裁剪头像
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CropAvatar(string userDomainName)
        {
            SetPageTitle("裁剪头像");
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            StatusMessageData statusMessageData = null;
            if (!currentDomainUser.HasAvatar)
            {
                statusMessageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotEditForNotUploadFile", applicationID));
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("UploadAvatar", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/User/CropAvatar.aspx", "Masters/SimpleHeader.master");
        }


        /// <summary>
        /// 更新个人资料
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateProfile(string userDomainName)
        {
            User domainUser = Users.GetUser(userDomainName, true);
            PersonUser userToEdit = domainUser as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            userToEdit.CommonName = Request.Form.GetString("commonName", userToEdit.CommonName);
            if (SiteSettingsManager.GetSiteSettings().EnableNickName)
                userToEdit.NickName = Request.Form.GetString("nickName", userToEdit.NickName);
            userToEdit.Gender = (Genders)Request.Form.GetInt("gender", (int)userToEdit.Gender);
            try
            {
                userToEdit.Birthday = DateTime.Parse(Request.Form.GetString("birthday", userToEdit.Birthday.ToString()));
            }
            catch
            {
            }
            userToEdit.NowAreaCode = Request.Form.GetString("nowAreaCode", userToEdit.NowAreaCode);
            userToEdit.HomeAreaCode = Request.Form.GetString("homeAreaCode", userToEdit.HomeAreaCode); ;
            userToEdit.School = Request.Form.GetString("school", userToEdit.School);
            userToEdit.SchoolID = Request.Form.GetInt("schoolID", Convert.ToInt32(userToEdit.SchoolID));


            userToEdit.JobSort = Request.Form.GetString("jobSort", userToEdit.JobSort);
            userToEdit.CompanyName = Request.Form.GetString("companyName", userToEdit.CompanyName);
            userToEdit.CompanyTrade = Request.Form.GetInt("companyTrade", userToEdit.CompanyTrade);


            #region 编码

            userToEdit.CommonName = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.CommonName);
            userToEdit.NickName = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.NickName);

            userToEdit.AvatarUrl = WebUtils.HtmlEncode(userToEdit.AvatarUrl);

            #endregion

            #region 处理标签
            string selectedTagNames = Request.Form.GetString("userTags", string.Empty);
            List<UserTag> selectedTags = null;
            if (!string.IsNullOrEmpty(selectedTagNames))
            {
                string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (tagNames != null && tagNames.Length > 0)
                {
                    selectedTags = new List<UserTag>();
                    foreach (string tagName in tagNames)
                    {
                        if (!string.IsNullOrEmpty(tagName.Trim()))
                        {
                            UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTag(tagName, domainUser.UserID, true);
                            if (tag != null)
                            {
                                selectedTags.Add(tag);
                            }
                            else
                            {
                                UserTag userTag = new UserTag();
                                userTag.UserID = domainUser.UserID;
                                userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).CreateUserTag(userTag);
                                if (isSuccess)
                                    selectedTags.Add(userTag);
                            }
                        }
                    }
                }
            }

            if (selectedTags != null && selectedTags.Count > 0)
            {
                UserTagManager.Instance(UserTagItemKeys.Instance().Person()).UpdateUserTagsOfItem(domainUser.UserID, domainUser.UserID, selectedTags.Select(n => n.TagName).ToList());
            }
            #endregion

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.EnableSelectUserRankType)
                userToEdit.RankType = (UserRankTypes)Request.Form.GetInt("rankType", (int)userToEdit.RankType);

            // Update the user
            Users.UpdateUser(userToEdit);
            return RedirectToAction("AddRecommendUser", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult EditPersonInfo(string userDomainName)
        {
            SetPageTitle("编辑资料");
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            return View("Pages/User/EditPersonInfo.aspx", "Masters/SimpleHeader.master", currentDomainUser);
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult AddRecommendUser(string userDomainName)
        {
            SetPageTitle("添加好友");
            if (CurrentUser == null)
                return Redirect(SiteUrls.Instance().Login(true));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            return View("Pages/User/AddRecommendUser.aspx", "Masters/SimpleHeader.master", currentDomainUser);
        }

        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListTopPersonInWizardMode(string userDomainName, int? pageindex)
        {
            PagingDataSet<PersonUser> pds = Users.GetPagingPersonUsersSortBy(PersonUsersSortBy.StageHitTimes, 12, pageindex ?? 1);
            return this.View("Controls/User/ListPersonInWizardMode.ascx", pds);
        }


        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListPersonInWizardMode(string userDomainName, int? pageindex)
        {
            PersonUser pu = Users.GetUser(userDomainName, false) as PersonUser;
            PagingDataSet<PersonUser> pds = new PagingDataSet<PersonUser>();
            SearchResultDataSet<PersonUser> results = null;
            PersonUserFullTextQuery query = new PersonUserFullTextQuery();

            query.TagName = Request.QueryString.GetString("TagName", string.Empty);
            query.CompanyName = Request.QueryString.GetString("CompanyName", string.Empty);
            query.NowAreaCode = Request.QueryString.GetString("NowAreaCode", string.Empty);
            query.HomeAreaCode = Request.QueryString.GetString("HomeAreaCode", string.Empty);
            query.CompanyTrade = Request.QueryString.GetInt("CompanyTrade", 0);
            query.School = Request.QueryString.GetString("School", string.Empty);
            query.PageSize = 12;
            query.PageIndex = pageindex ?? 1;
            if (query.IsValid())
            {
                results = PersonUserSearchManager.Instance().Search(pu, query);
                pds.PageIndex = results.PageIndex;
                pds.PageSize = results.PageSize;
                pds.Records = results.Records;
                pds.TotalRecords = results.TotalRecords;
            }
            if (pds.Records.Contains(pu))
                pds.Records.Remove(pu);
            RouteValueDictionary dictionary = new RouteValueDictionary();
            foreach (string q in Request.QueryString.Keys)
            {
                if (q.Equals(userDomainName, StringComparison.CurrentCultureIgnoreCase)
                    || !q.Equals(pageindex.ToString(), StringComparison.CurrentCultureIgnoreCase))
                    continue;
                dictionary[q] = Request.QueryString[q];
            }
            ViewData["CurrentRouteValueDictionary"] = dictionary;
            return this.View("Controls/User/ListPersonInWizardMode.ascx", pds);
        }

        public ActionResult Control_SetShowWizardMode(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());
            }

            return PartialView("Controls/User/SetShowWizardMode.ascx");
        }
        /// <summary>
        /// 设置是否下次还需要进入向导
        /// </summary>
        public ActionResult SetShowWizardMode()
        {
            string userDomainName = WebUtils.UrlDecode(Request.QueryString.GetString("userDomainName", string.Empty));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());
            }
            bool isEnterWizardForever = !Request.QueryString.GetBool("notEnterWizardMode", false);
            if (!isEnterWizardForever)
            {
                //记住以后是否还需要进入向导
                SaveIsEnterWizardForever(currentDomainUser.UserID, false);
            }
            else
            {
                //记住今天不再进入向导
                SaveIsEnterWizardToday(currentDomainUser.UserID, false);
            }
            return Redirect(SiteUrls.Instance().UserDomainHome(userDomainName));
        }

        /// <summary>
        /// 将以后是否还需要进入向导保存到cookie中
        /// </summary>
        private void SaveIsEnterWizardForever(int userID, bool isEnterWizardForever)
        {
            string cookieName = "IsEnterWizardForeverCookie_" + userID + SiteSettingsManager.GetSiteSettings().EnterWizardModeValue;
            HttpCookie cookie = new HttpCookie(cookieName, isEnterWizardForever.ToString());
            cookie.Expires = DateTime.Now.AddYears(1);
            Response.Cookies.Add(cookie);
        }
        /// <summary>
        /// 将今天是否再次进入向导保存到cookie中
        /// </summary>
        private void SaveIsEnterWizardToday(int userID, bool isEnterWizardToday)
        {
            string cookieName = "IsEnterWizardTodayCookie_" + userID;
            HttpCookie cookie = new HttpCookie(cookieName, isEnterWizardToday.ToString());
            cookie.Expires = DateTime.Now.AddDays(1);
            Response.Cookies.Add(cookie);
        }

        /// <summary>
        /// 添加好友
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_BatchCreateFriendInvitations(string userDomainName)
        {
            userDomainName = WebUtils.UrlDecode(userDomainName);
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                throw new SPBException(ExceptionTypes.Instance().UserNotFound());
            }

            return PartialView("Controls/User/CreateFriendInvitations.ascx");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult BatchCreateFriendInvitations(string userDomainName)
        {
            string[] userIDs = Request.Form.GetString("userIDs", string.Empty).Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);

            if (userIDs != null && userIDs.Length > 0)
            {
                CreateInvitationStatuses status = CreateInvitationStatuses.Success;
                foreach (string userID in userIDs)
                {
                    int id = int.Parse(userID);
                    User user = Users.GetUser(id, false);
                    if (user == null)
                        continue;
                    if (Friends.IsFriend(CurrentUser.UserID, id) || !Privacys.Validate(id, CurrentUser, PrivacyItemKeys.Instance().Action_ReceiveFriendInvitation()))
                        continue;
                    Invitation invitation = new Invitation();
                    invitation.Subject = ResourceManager.GetString("Title_DefaultFriendInvitationSubject", user.Language, applicationID);
                    invitation.Body = WebUtils.HtmlEncode(Request.Form.GetString("body", string.Empty));
                    invitation.SenderUserID = CurrentUser.UserID;
                    invitation.Sender = CurrentUser.DisplayName;
                    invitation.Receiver = user.DisplayName;
                    invitation.ReceiverUserID = user.UserID;
                    invitation.ApplicationID = ApplicationIDs.Instance().Friend();

                    status = Invitations.CreateInvitation(invitation);
                }

                if (status == CreateInvitationStatuses.Duplicate)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_InvitErrorForAlreadySendApply", applicationID));
                }
                else if (status == CreateInvitationStatuses.Success)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_InvitFriendSuccess", applicationID));
                }
            }
            return RedirectToAction("Control_BatchCreateFriendInvitations", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        ///  注册完成，提示查看确认邮件
        /// </summary>
        public ActionResult UserAccountCreatedValidateEmail(string userName)
        {
            User user = Users.GetUser(WebUtils.UrlDecode(userName), false);
            if (user == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            return View("Pages/User/UserAccountCreatedValidateEmail.aspx", "Masters/SimpleHeader.master", user);
        }

        /// <summary>
        /// 重发验证码
        /// </summary>
        public ActionResult ResendVerificationCode(string userName)
        {
            SetPageTitle(GetResourceString("Title_ResendVerificationCode", applicationID));

            User user = Users.GetUser(WebUtils.UrlDecode(userName), false);
            if (user == null)
            {
                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotFindUser", applicationID));
            }
            UserEmailsProvider.Instance().UserCreateValidateEmail(user);
            return RedirectToAction("UserAccountCreatedValidateEmail", new RouteValueDictionary { { "userName", WebUtils.UrlEncode(userName) } });
        }

        /// <summary>
        /// 注册条款
        /// </summary>
        public ActionResult Control_Agreement()
        {
            SetPageTitle(GetResourceString("Title_Agreement", applicationID));
            return View("Controls/User/Agreement.ascx");
        }
        /// <summary>
        /// 显示重新获取密码页
        /// </summary>
        public ActionResult ForgotPassword()
        {
            SetPageTitle(GetResourceString("Title_GetPasswordBack", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_GetPasswordBack", applicationID), string.Empty);
            return View("Pages/User/ForgotPassword.aspx", "Masters/SimpleHeader.master");
        }
        /// <summary>
        /// 处理获取密码
        /// </summary>
        public ActionResult RecoverPassword()
        {
            string email = Request.Form.GetString("email", string.Empty).Trim();
            User user = Users.FindUserByEmail(email.Trim());
            if (user == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_CannotEmailCheckAgain", applicationID);
                return RedirectToAction("ForgotPassword");
            }

            if (user.SendRecoverPasswordSecureLink())
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_CannotFindUser", applicationID);
            }
            return RedirectToAction("ForgotPassword");
        }

        /// <summary>
        /// 邀请朋友
        /// </summary>
        public ActionResult InvitationFriend()
        {

            string Token = Request.QueryString.GetString("Token", string.Empty).Trim();

            int inviteUserID = 0;
            if (!string.IsNullOrEmpty(Token))
            {
                try
                {
                    Token = Token.Replace(' ', '+');
                    Token = EncryptManager.DecryptTokenForInviteFriend(Token);
                    string[] qs = Token.Split(',');
                    DateTime dateCreated = Convert.ToDateTime(qs[1]);
                    if (dateCreated.CompareTo(DateTime.Now) < 0)
                        return Redirect(SiteUrls.Instance().Error("链接已过期"));
                    inviteUserID = int.Parse(qs[0]);
                }
                catch { }
            }
            User inviteUser = Users.GetUser(inviteUserID, false);

            if (inviteUser != null && inviteUser.UserType == UserTypes.PersonUser)
            {
                string titleForGender = string.Empty;
                if ((inviteUser as PersonUser) != null)
                {
                    switch ((inviteUser as PersonUser).Gender)
                    {
                        case Genders.Male:
                            titleForGender = GetResourceString("Title_Pattern_ShareHisFirendsWeb", applicationID);
                            break;
                        case Genders.Female:
                            titleForGender = GetResourceString("Title_Pattern_ShareHerFirendsWeb", applicationID);
                            break;
                        default:
                            titleForGender = GetResourceString("Title_Pattern_ShareHisFirendsWeb", applicationID);
                            break;
                    }
                }
                SetPageTitle(string.Format(titleForGender, inviteUser.DisplayName));
                //ViewData["GenderName"] = gender;
                ViewData["InviteUser"] = inviteUser;
                if (CurrentUser != null)
                {
                    if (inviteUser.UserID == CurrentUser.UserID)
                    {
                        ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        ViewData["StatusMessageContent"] = GetResourceString("Message_InvitErrorForInvitSelf", applicationID);
                    }
                    else if (Friends.IsFriend(CurrentUser.UserID, inviteUser.UserID))
                    {
                        ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        ViewData["StatusMessageContent"] = GetResourceString("Message_InvitErrorForRepeatInvit", applicationID);
                    }
                }
            }
            else
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_TheUserInviteIsNotFound", applicationID)));
            }
            return View("Pages/User/InvitationFriend.aspx", "Masters/Simple.master");
        }

        /// <summary>
        /// 邀请好友加入群组
        /// </summary>
        public ActionResult InvitationJoinClub()
        {

            string Token = Request.QueryString.GetString("Token", string.Empty).Trim();

            int inviteUserID = 0;
            int inviteClubID = 0;
            if (!string.IsNullOrEmpty(Token))
            {
                try
                {
                    Token = Token.Replace(' ', '+');
                    Token = EncryptManager.DecryptTokenForInviteFriend(Token);
                    string[] qs = Token.Split(',');
                    DateTime dateCreated = Convert.ToDateTime(qs[1]);
                    if (dateCreated.CompareTo(DateTime.Now) < 0)
                        return Redirect(SiteUrls.Instance().Error("链接已过期"));
                    inviteUserID = int.Parse(qs[0]);
                    inviteClubID = int.Parse(qs[2]);
                }
                catch { }
            }
            User inviteUser = Users.GetUser(inviteUserID, false);
            Club.Club inviteClub = Club.Clubs.GetClub(inviteClubID, false);

            if (inviteUser != null && inviteUser.UserType == UserTypes.PersonUser && inviteClub != null)
            {

                SetPageTitle(string.Format("{0},邀请你加入TA的群组", inviteUser.DisplayName));
                //ViewData["GenderName"] = gender;
                ViewData["InviteUser"] = inviteUser;
                if (CurrentUser != null)
                {
                    if (inviteUser.UserID == CurrentUser.UserID)
                    {
                        ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        ViewData["StatusMessageContent"] = "自己不可以邀请自己哦！";
                    }
                    else if (Club.ClubMembers.IsClubMember(inviteClubID, CurrentUser.UserID))
                    {
                        ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                        ViewData["StatusMessageContent"] = "你已经接受过邀请了!";
                    }
                }
            }
            else
            {
                return Redirect(SiteUrls.Instance().Error404("邀请您的用户不存在或已被删除！"));
            }
            return View("Pages/User/InvitationJoinClub.aspx", "Masters/Simple.master");
        }

        /// <summary>
        /// 接受朋友的邀请
        /// </summary>
        public ActionResult AcceptInviteAction(string token)
        {
            int inviteUserID = 0;
            int inviteClubID = 0;
            if (!string.IsNullOrEmpty(token))
            {
                try
                {
                    string detoken = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
                    string[] qs = detoken.Split(',');
                    DateTime dateCreated = DateTime.Parse(qs[1]);
                    if (dateCreated.CompareTo(DateTime.Now) < 0)
                        return Redirect(SiteUrls.Instance().Error("链接已过期"));
                    inviteUserID = int.Parse(qs[0]);
                    if (qs.Count() > 2)
                        inviteClubID = int.Parse(qs[2]);
                }
                catch { }
            }

            User inviteUser = Users.GetUser(inviteUserID, false);
            User currentUser = Globals.GetCurrentUser();
            if (inviteUser != null && inviteUser.UserType == UserTypes.PersonUser)
            {
                if (currentUser != null && currentUser.UserType == UserTypes.PersonUser && inviteClubID <= 0)
                {
                    if (!Friends.IsFriend(currentUser.UserID, inviteUser.UserID))
                    {
                        Friends.CreateFriend(currentUser.UserID, inviteUser.UserID);
                        return Redirect(SiteUrls.Instance().UserFriendHome(currentUser.UserName));
                    }
                    else
                    {
                        return RedirectToAction("InvitationFriendAction");
                    }
                }
                if (inviteClubID > 0)
                {
                    if (!Club.ClubMembers.IsClubMember(inviteClubID, currentUser.UserID))
                    {
                        Club.ClubMember member = new SpaceBuilder.Club.ClubMember();
                        member.InviteUserID = inviteUser.UserID;
                        member.UserID = currentUser.UserID;
                        member.ClubID = inviteClubID;
                        member.DisplayName = currentUser.DisplayName;
                        member.ClubMemberRole = SpaceBuilder.Club.ClubMemberRoles.ClubMember;
                        Club.ClubMembers.CreateClubMember(member);
                    }
                    return Redirect(SiteUrls.Instance().UserDomainHome(currentUser.UserID));
                }


            }
            return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_TheUserInviteIsNotFound", applicationID)));
        }

        /// <summary>
        /// 显示重设密码页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ResetPassword(string userName, string token)
        {
            SetPageTitle(GetResourceString("Title_ReSetPassword", applicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ChangePassword", applicationID), string.Empty);
            token = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
            string[] qs = token.Split(',');
            DateTime dateCreated = Convert.ToDateTime(qs[1]);
            if (dateCreated.CompareTo(DateTime.Now) < 0)
            {
                return Redirect(SiteUrls.Instance().Error("链接已过期"));
            }
            string membershipID = qs[0];
            if (string.IsNullOrEmpty(userName))
            {
                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotFindUser", applicationID));
            }
            else
            {
                User user = Users.GetUser(userName, false);
                if (user == null)
                {
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotFindUser", applicationID));
                }
                else
                {
                    try
                    {
                        if ((Guid)user.MembershipID != new Guid(membershipID))
                        {
                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TokenForPasswordIsWorng", applicationID));
                        }
                    }
                    catch
                    {
                        ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TokenForPasswordIsWorng", applicationID));
                    }
                }
            }
            return View("Pages/User/ResetPassword.aspx", "Masters/SimpleHeader.master");
        }

        /// <summary>
        /// 处理重设密码
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DealResetPassword(string userName, string token)
        {
            if (!string.IsNullOrEmpty(userName))
            {
                User user = Users.GetUser(userName, false);
                if (user != null)
                {
                    token = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
                    string[] qs = token.Split(',');
                    DateTime dateCreated = Convert.ToDateTime(qs[1]);
                    if (dateCreated.CompareTo(DateTime.Now) < 0)
                    {
                        return Redirect(SiteUrls.Instance().Error("链接已过期！"));
                    }
                    string membershipID = qs[0];
                    try
                    {
                        if ((Guid)user.MembershipID == new Guid(membershipID))
                        {
                            string password = Request.Form.GetString("password", string.Empty).Trim();
                            string messageContent = string.Empty;
                            bool status = false;
                            if (Globals.ValidatePassword(password, out messageContent))
                            {
                                status = user.ChangePassword(string.Empty, password, false);
                                if (status)
                                    LeaveSecureConnection(SiteUrls.Instance().Login());
                                else
                                {
                                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_UpdatePasswordFailed", applicationID));
                                }
                            }
                            else
                            {
                                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, messageContent);
                            }
                        }
                    }
                    catch (Exception)
                    {
                    }
                }
            }
            return RedirectToAction("ResetPassword", new RouteValueDictionary { { "userName", userName }, { "token", token } });
        }

        /// <summary>
        /// 显示验证Email页
        /// </summary>
        public ActionResult ValidateUserEmail(string userName, string token)
        {
            SetPageTitle(GetResourceString("Title_ValidateUserEmail", applicationID));
            if (string.IsNullOrEmpty(userName))
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotFindUser", applicationID));
            }
            else
            {
                User user = Users.GetUser(WebUtils.UrlDecode(userName), false);
                if (user == null)
                {
                    TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotFindUser", applicationID));
                }
                else
                {
                    try
                    {

                        token = EncryptManager.DecryptTokenForInviteFriend(token.Replace(' ', '+'));
                        string[] qs = token.Split(',');
                        DateTime dateCreated = Convert.ToDateTime(qs[1]);
                        if (dateCreated.CompareTo(DateTime.Now) < 0)
                        {
                            return Redirect(SiteUrls.Instance().Error("链接已过期"));
                        }
                        string membershipID = qs[0];
                        if ((Guid)user.MembershipID != new Guid(membershipID))
                        {
                            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TokenForPasswordIsWorng", applicationID));
                        }
                        else
                        {
                            user.IsEmailVerified = true;
                            user.AccountStatus = UserAccountStatuses.Approved;
                            Users.UpdateUserAccountStatus(user);
                            FormsAuthentication.SetAuthCookie(user.UserName, false);
                        }
                    }
                    catch
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_TokenForPasswordIsWorng", applicationID));
                    }
                }
            }
            return Redirect(SiteUrls.Instance().UserDomainHome(userName));
            //return View("Pages/User/ValidateUserEmail.aspx", "Masters/Simple.master");
        }

        #region Help Methods
        /// <summary>
        /// 用户档案存储前进行转化
        /// </summary>
        private static PersonUser ConvertUserForStorage(PersonUser pu)
        {
            pu.CommonName = Formatter.FormatSingleLinePlainTextForStorage(pu.CommonName);
            pu.NickName = Formatter.FormatSingleLinePlainTextForStorage(pu.NickName);
            pu.PublicEmail = Formatter.FormatSingleLinePlainTextForStorage(pu.PublicEmail);
            pu.MsnIM = Formatter.FormatSingleLinePlainTextForStorage(pu.MsnIM);
            pu.QQIM = Formatter.FormatSingleLinePlainTextForStorage(pu.QQIM);
            pu.YahooIM = Formatter.FormatSingleLinePlainTextForStorage(pu.YahooIM);
            pu.SkypeIM = Formatter.FormatSingleLinePlainTextForStorage(pu.SkypeIM);
            pu.PersonalHomepage = Formatter.FormatSingleLinePlainTextForStorage(pu.PersonalHomepage);
            return pu;
        }

        private void LeaveSecureConnection(string url)
        {
            if (Request.IsSecureConnection)
                url = string.Format("https://{0}{1}", Request.Url.Host, url);
            Response.Redirect(url, true);
        }

        #endregion

        /// <summary>
        /// 处理验证邮箱
        /// </summary>
        public JsonResult ValidateEmail(string userEmail)
        {
            bool isSuccess = false;
            string messageContent = string.Empty;
            if (Globals.ValidateEmail(userEmail, out messageContent))
            {
                isSuccess = true;
                messageContent = GetResourceString("Validate_UserEmail_Sucess");
            }
            return Json(new { isSuccess = isSuccess, messageContent = messageContent }, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 处理验证用户名
        /// </summary>
        public JsonResult ValidateUserName(string userName)
        {
            bool isSuccess = false;
            string messageContent = string.Empty;
            if (Globals.ValidateUserName(userName, out messageContent))
            {
                isSuccess = true;
                messageContent = GetResourceString("Message_ValidateUserNameSucceed", applicationID);
            }
            return Json(new { isSuccess = isSuccess, messageContent = messageContent }, JsonRequestBehavior.AllowGet);
        }
        /// <summary>
        /// 处理验证密码
        /// </summary>
        public JsonResult ValidatePassword(string password)
        {
            bool isSuccess = false;
            string messageContent = string.Empty;
            if (Globals.ValidatePassword(password, out messageContent))
            {
                isSuccess = true;
                messageContent = GetResourceString("Message_ValidatePasswordSucceed", applicationID);
            }
            return Json(new { isSuccess = isSuccess, messageContent = messageContent }, JsonRequestBehavior.AllowGet);
        }

        /// <summary>
        /// 处理验证验证码
        /// </summary>
        public JsonResult ValidateVerifyCode(string verifyCode)
        {
            bool isSuccess = false;
            string messageContent = string.Empty;

            //检查验证码是否录入正确
            string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
            if (verifyCode != null)
                verifyCode = verifyCode.Trim();

            if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                messageContent = GetResourceString("Message_IdentifyingCodeError");
            else
                isSuccess = true;

            return Json(new { isSuccess = isSuccess, messageContent = messageContent }, JsonRequestBehavior.AllowGet);
        }

        #region PersonUser FullTextSearch

        /// <summary>
        /// 会员全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult UserSearch(int? pageIndex, bool? isAccessUserDomainWhenOne)
        {
            SetPageTitle(GetResourceString("Title_UserSearch", applicationID));

            PersonUserFullTextQuery query = new PersonUserFullTextQuery();

            query.ConvertFromQueryString(Request);

            //用于群组搜索成员
            if (string.IsNullOrEmpty(query.Keyword))
            {
                query.Keyword = Request.QueryString["Keyword"];
            }

            query.PageSize = 10;

            if (pageIndex.HasValue)
                query.PageIndex = pageIndex.Value;

            SearchResultDataSet<PersonUser> results = null;
            if (query.IsValid())
            {
                results = PersonUserSearchManager.Instance().Search(CurrentUser, query);

                for (int i = 0; i < results.Records.Count; i++)
                {
                    results.Records[i] = Users.GetUser(results.Records[i].UserID, false) as PersonUser;
                }
            }

            if (isAccessUserDomainWhenOne.HasValue && isAccessUserDomainWhenOne.Value && results != null && results.Records != null && results.TotalRecords == 1)
                return Redirect(SiteUrls.Instance().UserDomainHome(results.Records.First().UserID));

            ViewData["Query"] = query;
            return this.View("Pages/Members/Search.aspx", "Masters/Member.master", results);
        }

        /// 会员全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UserSearch()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();
            parameters["NameKeyword"] = Request.Form.GetString("Keyword", string.Empty).Trim();

            return RedirectToAction("UserSearch", parameters);
        }

        /// 会员全文检索
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UserAdvancedSearch()
        {
            RouteValueDictionary parameters = new RouteValueDictionary();

            int minAge = Request.Form.GetInt("MinAge", -1);
            int maxAge = Request.Form.GetInt("MaxAge", -1);
            if (minAge > 0 && maxAge > 0)
            {
                if (minAge > maxAge)
                {
                    parameters["MinAge"] = maxAge;
                    parameters["MaxAge"] = minAge;
                }
                else
                {
                    parameters["MinAge"] = minAge;
                    parameters["MaxAge"] = maxAge;
                }
            }

            if (!string.IsNullOrEmpty(Request.Form.GetString("HomeAreaCode", string.Empty)))
                parameters["HomeAreaCode"] = Request.Form.GetString("HomeAreaCode", string.Empty);

            if (!string.IsNullOrEmpty(Request.Form.GetString("NowAreaCode", string.Empty)))
                parameters["NowAreaCode"] = Request.Form.GetString("NowAreaCode", string.Empty);

            if (!string.IsNullOrEmpty(Request.Form.GetString("JobSort", string.Empty)))
                parameters["JobSort"] = Request.Form.GetString("JobSort", string.Empty);

            if (!string.IsNullOrEmpty(Request.Form.GetString("CompanyName", string.Empty)))
                parameters["CompanyName"] = Request.Form.GetString("CompanyName", string.Empty);

            if (!string.IsNullOrEmpty(Request.Form.GetString("School", string.Empty)))
                parameters["School"] = Request.Form.GetString("School", string.Empty);

            if (!string.IsNullOrEmpty(Request.Form.GetString("TagName", string.Empty)))
                parameters["TagName"] = Request.Form.GetString("TagName", string.Empty);

            if (Request.Form.GetInt("Gender", -1) > -1)
                parameters["Gender"] = Request.Form.GetInt("Gender", (int)Genders.All);

            if (!string.IsNullOrEmpty(Request.Form.GetString("Degree", string.Empty)))
                parameters["Degree"] = Request.Form.GetInt("Degree", -1);

            if (!string.IsNullOrEmpty(Request.Form.GetString("AgeScope", string.Empty)))
                parameters["AgeScope"] = Request.Form.GetInt("AgeScope", -1);

            if (Request.Form.GetString("Keyword", string.Empty) != string.Empty)
                parameters["NameKeyword"] = Request.Form.GetString("Keyword", string.Empty).Trim();

            return RedirectToAction("UserSearch", parameters);
        }

        #endregion


        #region 标签

        /// <summary>
        /// 会员搜索页面标签控件
        /// </summary>
        public ActionResult Control_ListUserSiteTags()
        {
            //取用户站点标签
            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetTopSiteTagsSortBy(SiteTagsSortBy.UserCount, 15);
            return PartialView("Controls/Members/ListUserSiteTags.ascx", siteTags);
        }


        /// <summary>
        /// 标签地图
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult MemberTagMap()
        {
            Header.AddTitle(string.Format("{1}{0}{2}{0}{3}", Globals.BrowserTitleSeparator, GetResourceString("Title_TagMap"), GetResourceString("Title_Member", ApplicationIDs.Instance().User()), Globals.SiteName));

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_TagMap"), string.Empty);

            Dictionary<string, KeyValuePair<int, SiteTag>> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetTopSiteTagsSortBy(SiteTagsSortBy.ItemCount, 300);

            return View("Pages/Members/MemberTagMap.aspx", "Masters/Member.master", siteTags);
        }

        #endregion

        #region 第三方账号
        /// <summary>
        /// 请求QQ授权页
        /// </summary>
        public ActionResult LoginToQQ()
        {

            string key = ConfigurationManager.AppSettings["QQ_AppKey"];
            string secret = ConfigurationManager.AppSettings["QQ_AppSecret"];

            var context = new QzoneSDK.Context.QzoneContext(key, secret);
            //Get a Request Token
            var callbackUrl = WebUtils.FullPath(SPBUrlHelper.Action("QQCallback", "ChannelUser")); //"/qzone/account/QQCallback.aspx";
            var requestToken = context.GetRequestToken(callbackUrl);
            //保存 request token, request token secret
            OAuthTokenKey2Secret.AddKey2Secret(requestToken.TokenKey, requestToken.TokenSecret);

            //Get the Qzone authentication page for the user to go to in order to authorize the Request Token.
            var authenticationUrl = context.GetAuthorizationUrl(requestToken, callbackUrl);

            return Redirect(authenticationUrl);
        }

        /// <summary>
        /// 请求Sina授权页
        /// </summary>
        public ActionResult LoginToSina()
        {
            oAuthSina oauth = new oAuthSina();
            oauth.RequestTokenGet();
            string url = oauth.AuthorizationGet();
            OAuthTokenKey2Secret.AddKey2Secret(oauth.token, oauth.tokenSecret);
            var callbackUrl = WebUtils.FullPath(SPBUrlHelper.Action("SinaCallback", "ChannelUser"));
            return Redirect(url + "&oauth_callback=" + callbackUrl);
        }

        /// <summary>
        /// QQ 回调函数
        /// </summary>
        public ActionResult QQCallback()
        {
            string returnUrl = string.Empty;
            if (Request.QueryString["oauth_vericode"] != null)
            {
                var requestTokenKey = Request.QueryString["oauth_token"];
                var requestTokenSecret = OAuthTokenKey2Secret.GetSecret(requestTokenKey);
                var verifier = Request.QueryString["oauth_vericode"];
                string key = ConfigurationManager.AppSettings["QQ_AppKey"];
                string secret = ConfigurationManager.AppSettings["QQ_AppSecret"];

                QzoneSDK.Qzone qzone = new QzoneSDK.Qzone(key, secret, requestTokenKey, requestTokenSecret, verifier, false, string.Empty);

                QzoneSDK.Qzone qzone2 = new QzoneSDK.Qzone(key, secret, qzone.OAuthTokenKey, qzone.OAuthTokenSecret, string.Empty, true, qzone.OpenID);

                string currentUser = qzone2.GetCurrentUser();

                BasicProfile user = (BasicProfile)JsonConvert.Import(typeof(BasicProfile), currentUser);
                if (user != null)
                {
                    if (CurrentUser != null)
                    {
                        if (AccountBindings.GetUser(AccountTypes.Instance().QQ(), qzone2.OpenID) == null)
                        {
                            AccountBinding account = new AccountBinding();
                            account.AccountType = AccountTypes.Instance().QQ();
                            account.Identification = qzone2.OpenID;
                            account.UserID = CurrentUser.UserID;
                            account.OauthToken = qzone2.OAuthTokenKey;
                            account.OauthTokenSecret = qzone2.OAuthTokenSecret;
                            AccountBindings.CreateAccountBinding(account);

                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "绑定成功");
                        }
                        else
                        {
                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "此账号已在网站中绑定过，不可再绑定其他网站账号");
                        }
                    }
                    else
                    {
                        User systemUser = AccountBindings.GetUser(AccountTypes.Instance().QQ(), qzone2.OpenID);
                        if (systemUser != null)
                        {
                            FormsAuthentication.SetAuthCookie(systemUser.UserName, true);
                            returnUrl = SiteUrls.Instance().UserDomainHome(systemUser);
                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "登录成功");
                        }
                        else
                        {
                            if (SiteSettingsManager.GetSiteSettings().RegistrationMode == RegistrationModes.DisableAll)
                            {
                                returnUrl = SiteUrls.Instance().UserRegister();
                                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "管理员限制了新帐号的创建，请过后再试.");
                            }
                            else
                            {
                                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, string.Empty);
                                ViewData["UserName"] = WebUtils.UrlEncode(user.Nickname);
                                string nickName = WebUtils.UrlEncode(user.Nickname);
                                string figureurl = user.Figureurl.Remove(user.Figureurl.LastIndexOf('/') + 1) + "100";
                                ViewData["FirstLogin"] = true;
                                returnUrl = SPBUrlHelper.Action("ThirdRegister", "ChannelUser", new RouteValueDictionary { { "accountType", AccountTypes.Instance().QQ() }, { "openID", qzone2.OpenID }, { "OauthToken", qzone2.OAuthTokenKey }, { "OauthTokenSecret", qzone2.OAuthTokenSecret }, { "nickName", nickName }, { "figureurl", figureurl } });
                            }
                        }
                    }
                }
                else
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "授权失败！");
            }
            else
                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "参数错误，授权失败！");

            ViewData["returnUrl"] = returnUrl;
            return View("Pages/User/LoginCallback.aspx");
        }

        /// <summary>
        /// 新浪回调函数
        /// </summary>
        public ActionResult SinaCallback()
        {
            string returnUrl = string.Empty;
            oAuthSina oauth = new oAuthSina();
            if (Request.QueryString["oauth_verifier"] != null)
            {
                oauth.token = Request.QueryString["oauth_token"];
                oauth.tokenSecret = OAuthTokenKey2Secret.GetSecret(oauth.token);
                oauth.Verifier = Request["oauth_verifier"].ToString();
                oauth.AccessTokenGet();

                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "登录成功");
                SinaUser sinaUser = oauth.GetCurrentUser();

                if (sinaUser != null)
                {
                    //登录用户直接绑定账号
                    if (CurrentUser != null)
                    {
                        if (AccountBindings.GetUser(AccountTypes.Instance().SinaWeibo(), sinaUser.ID.ToString()) == null)
                        {
                            AccountBinding account = new AccountBinding();
                            account.AccountType = AccountTypes.Instance().SinaWeibo();
                            account.Identification = sinaUser.ID;
                            account.UserID = CurrentUser.UserID;
                            account.OauthToken = oauth.token;
                            account.OauthTokenSecret = oauth.tokenSecret;
                            AccountBindings.CreateAccountBinding(account);
                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "绑定成功");
                        }
                        else
                        {
                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "此账号已在网站中绑定过，不可再绑定其他网站账号");
                        }
                    }
                    else
                    {
                        User systemUser = AccountBindings.GetUser(AccountTypes.Instance().SinaWeibo(), sinaUser.ID.ToString());
                        //已经绑定过，直接登录
                        if (systemUser != null)
                        {
                            FormsAuthentication.SetAuthCookie(systemUser.UserName, true);
                            returnUrl = SiteUrls.Instance().UserDomainHome(systemUser);
                            ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "登录成功");
                        }
                        else
                        {
                            if (SiteSettingsManager.GetSiteSettings().RegistrationMode == RegistrationModes.DisableAll)
                            {
                                returnUrl = SiteUrls.Instance().UserRegister();
                                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, "管理员限制了新帐号的创建，请过后再试.");
                            }
                            else
                            {
                                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, string.Empty);
                                string nickName = WebUtils.UrlEncode(sinaUser.Name);
                                string figureurl = sinaUser.Profile_image_url.Replace("/50/", "/180/");
                                ViewData["UserName"] = nickName;
                                ViewData["FirstLogin"] = true;
                                returnUrl = SPBUrlHelper.Action("ThirdRegister", "ChannelUser", new RouteValueDictionary { { "accountType", AccountTypes.Instance().SinaWeibo() }, { "openID", sinaUser.ID }, { "OauthToken", oauth.token }, { "OauthTokenSecret", oauth.tokenSecret }, { "nickName", nickName }, { "figureurl", figureurl } });
                            }
                        }
                    }
                }
                else
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "授权失败！");

            }
            else
                ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "参数错误，授权失败！");


            ViewData["returnUrl"] = returnUrl;
            return View("Pages/User/LoginCallback.aspx");
        }

        /// <summary>
        /// 首次登录完善账号资料
        /// </summary>
        [HttpGet]
        public ActionResult ThirdRegister(int? accountType, string openID, string OauthToken, string OauthTokenSecret, string nickname, string figureurl)
        {
            ViewData["Nickname"] = nickname;
            ViewData["Figureurl"] = figureurl;

            return View("Pages/User/ThirdRegister.aspx", "Masters/SimpleHeader.master");
        }
        /// <summary>
        /// 首次登录完善账号资料
        /// </summary>
        [HttpPost]
        public ActionResult ThirdRegisterAction(int? accountType, string openID, string OauthToken, string OauthTokenSecret)
        {
            if (string.IsNullOrEmpty(openID))
                return Redirect(SiteUrls.Instance().Login());
            accountType = accountType ?? AccountTypes.Instance().QQ();

            string figureurl = Request.Form.GetString("figureurl", string.Empty);

            PersonUser personUser = new PersonUser();
            personUser.Language = Globals.GetCurrentUserLanguage();
            personUser.UserName = Request.Form.GetString("userName", string.Empty).Trim();
            //随机密码
            string tmpstr = string.Empty;
            string pwdchars = "abcdefghijklmnopqrstuvwxyz0123456789";
            int iRandNum;
            Random rnd = new Random();
            for (int i = 0; i < 6; i++)
            {
                iRandNum = rnd.Next(pwdchars.Length);
                tmpstr += pwdchars[iRandNum];
            }

            personUser.Password = tmpstr;
            personUser.PrivateEmail = Request.Form.GetString("userEmail", string.Empty).Trim();
            personUser.Gender = (Genders)Request.Form.GetInt("gender", 1);
            personUser.IsModerated = SiteSettingsManager.GetSiteSettings().DefaultModeratedStatus;

            string errorMessage = "";

            if (!Globals.ValidateUserName(personUser.UserName, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                TempData["personUser"] = personUser;

                return RedirectToAction("ThirdRegister", new RouteValueDictionary { { "accountType", accountType }, { "openID", openID }, { "OauthToken", OauthToken }, { "OauthTokenSecret", OauthTokenSecret }, { "figureurl", figureurl } });
            }
            if (!Globals.ValidateEmail(personUser.PrivateEmail, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                TempData["personUser"] = personUser;

                return RedirectToAction("ThirdRegister", new RouteValueDictionary { { "accountType", accountType }, { "openID", openID }, { "OauthToken", OauthToken }, { "OauthTokenSecret", OauthTokenSecret }, { "figureurl", figureurl } });
            }


            CreateUpdateUserStatuses status = CreateUpdateUserStatuses.UnknownFailure;
            AccountActivations activation = SiteSettingsManager.GetSiteSettings().AccountActivation;
            // Does the user require approval?
            if (activation == AccountActivations.Automatic)
                personUser.AccountStatus = UserAccountStatuses.Approved;
            else
                personUser.AccountStatus = UserAccountStatuses.ApprovalPending;

            if (personUser.UserName.Equals("Anonymous", StringComparison.CurrentCultureIgnoreCase))
            {
                status = CreateUpdateUserStatuses.DuplicateUsername;
            }
            else
            {
                try
                {
                    personUser = ConvertUserForStorage(personUser);
                    status = Users.Create(personUser, true);
                }
                catch
                {
                    //status = exception.;
                }
            }

            // Determine if the account was created successfully
            switch (status)
            {
                // Username already exists!
                case CreateUpdateUserStatuses.DuplicateUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForDuplicateUsername", applicationID);
                    break;
                case CreateUpdateUserStatuses.BeyondLicensedCount:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForBeyondLicensedCount", applicationID);
                    break;

                // Email already exists!
                case CreateUpdateUserStatuses.DuplicateEmailAddress:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_RegisterFailedForDuplicateEmailAddress", applicationID), SiteUrls.Instance().ForgotPassword());
                    break;

                // Unknown failure has occurred!
                case CreateUpdateUserStatuses.UnknownFailure:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForUnknownFailure", applicationID);
                    break;

                // Username is disallowed
                case CreateUpdateUserStatuses.DisallowedUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForDisallowedUsername", applicationID);
                    break;

                case CreateUpdateUserStatuses.InvalidQuestionAnswer:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForInvalidQuestionAnswer", applicationID);
                    break;

                case CreateUpdateUserStatuses.InvalidPassword:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_RegisterFailedForInvalidPassword", applicationID);
                    break;

                // Everything went off fine, good
                case CreateUpdateUserStatuses.Created:

                    #region 账号绑定
                    if (!string.IsNullOrEmpty(openID))
                    {
                        AccountBinding account = new AccountBinding();
                        account.AccountType = accountType ?? AccountTypes.Instance().QQ();
                        account.Identification = openID;
                        account.UserID = personUser.UserID;
                        account.OauthToken = OauthToken;
                        account.OauthTokenSecret = OauthTokenSecret;
                        AccountBindings.CreateAccountBinding(account);
                    }
                    #endregion

                    #region 处理头像
                    bool useAvatar = Request.Form.GetBool("useAvatar", false);
                    if (useAvatar)
                    {
                        if (!string.IsNullOrEmpty(figureurl))
                        {
                            try
                            {

                                WebRequest webRequest = WebRequest.Create(figureurl);
                                HttpWebResponse httpWebResponse = (HttpWebResponse)webRequest.GetResponse();
                                Stream stream = httpWebResponse.GetResponseStream();

                                bool isImage = httpWebResponse.ContentType.IndexOf("image") > -1;
                                if (isImage && stream != null && stream.CanRead)
                                {
                                    MemoryStream memoryStream = new MemoryStream();
                                    const int bufferLength = 1024;
                                    int actual;
                                    byte[] buffer = new byte[bufferLength];

                                    while ((actual = stream.Read(buffer, 0, bufferLength)) > 0)
                                    {
                                        memoryStream.Write(buffer, 0, actual);
                                    }

                                    Avatars.UploadOriginalAvatar(personUser.UserID, memoryStream);
                                }
                            }
                            catch (Exception e)
                            {
                                Logger logger = LoggerFactory.GetLogger();
                                logger.Error(e.Message, e);
                            }
                        }
                    }
                    #endregion

                    switch (activation)
                    {
                        case AccountActivations.AdminApproval:
                            Response.Redirect(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().UserAccountPending()));
                            break;

                        case AccountActivations.Email:
                            LeaveSecureConnection(SPBUrlHelper.Action("UserAccountCreatedValidateEmail", "ChannelUser", new RouteValueDictionary { { "userName", WebUtils.UrlEncode(personUser.UserName) } }));
                            break;

                        case AccountActivations.Automatic:
                            if (!Request.IsAuthenticated)
                                FormsAuthentication.SetAuthCookie(personUser.UserName, false);
                            LeaveSecureConnection(SiteUrls.Instance().UserDomainHome(personUser));
                            break;
                    }
                    return new EmptyResult();
            }
            TempData["personUser"] = personUser;
            return RedirectToAction("ThirdRegister", new RouteValueDictionary { { "accountType", accountType }, { "openID", openID }, { "OauthToken", OauthToken }, { "OauthTokenSecret", OauthTokenSecret }, { "figureurl", figureurl } });
        }
        #endregion
    }
}
