﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------ 

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;

using System.Web;

using System.Web.Routing;
using SpaceBuilder.Web;
using SpaceBuilder.Utils;
using SpaceBuilder.LuceneSearch;

namespace SpaceBuilder.Common.Controllers
{
    [HttpCompress]
    public class FriendController : UserDomainControllerBase
    {
        private int friendApplicationID = ApplicationIDs.Instance().Friend();

        /// <summary>
        /// 空间朋友主页
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult Home(string userDomainName, int? groupID, int? pageIndex, Friendliness? sortBy)
        {
            #region 标题
            SetPageTitle(userDomainName, string.Format(GetResourceString("Title_Pattern_WhoseFriend", friendApplicationID), GetCurrentDomainUser(userDomainName).DisplayName));
            #endregion

            bool isManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManager = true;
            ViewData["isManager"] = isManager;

            ViewData["friendliness"] = sortBy;

            if (isManager)
                return View("Pages/Friends/Home.aspx", "Masters/Friend.master");
            else
                return View("Pages/Friends/OthersFriend.aspx", "Masters/Friend.master");
        }

        #region Friend

        /// <summary>
        /// 可能认识的人
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMightKnowUsers(string userDomainName, int? pageIndex, int? displayCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            PersonUser personUser = currentDomainUser as PersonUser;

            #region 同事
            SearchResultDataSet<PersonUser> resultsCompany = null;
            PersonUserFullTextQuery queryCompany = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.CompanyName))
            {
                queryCompany.CompanyName = personUser.CompanyName;
                queryCompany.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCompany.SortOrder = SortOrder.Descending;
                resultsCompany = PersonUserSearchManager.Instance().Search(currentDomainUser, queryCompany);
                ViewData["resultsCompany"] = MayKnowExceptMeAndFirend(currentDomainUser.UserID, resultsCompany.Records, displayCount ?? 50);
            }
            #endregion

            #region 二层朋友
            SearchResultDataSet<PersonUser> resultsSecondFriends = null;
            PersonUserFullTextQuery querySecondFriends = new PersonUserFullTextQuery();
            querySecondFriends.SearchRange = 2;
            querySecondFriends.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
            querySecondFriends.SortOrder = SortOrder.Descending;
            resultsSecondFriends = PersonUserSearchManager.Instance().Search(currentDomainUser, querySecondFriends);
            ViewData["resultsSecondFriends"] = MayKnowExceptMeAndFirend(currentDomainUser.UserID, resultsSecondFriends.Records, displayCount ?? 50);
            #endregion

            #region 同行
            SearchResultDataSet<PersonUser> resultsCompanyTrade = null;
            PersonUserFullTextQuery queryCompanyTrade = new PersonUserFullTextQuery();
            if (personUser.CompanyTrade > 0)
            {
                queryCompanyTrade.CompanyTrade = personUser.CompanyTrade;
                queryCompanyTrade.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCompanyTrade.SortOrder = SortOrder.Descending;
                resultsCompanyTrade = PersonUserSearchManager.Instance().Search(currentDomainUser, queryCompanyTrade);
                if (resultsCompanyTrade.Records.Count > 0)
                    ViewData["resultsCompanyTrade"] = MayKnowExceptMeAndFirend(currentDomainUser.UserID, resultsCompanyTrade.Records, displayCount ?? 50);
                else
                    ViewData["resultsCompanyTrade"] = null;
            }
            #endregion

            #region 校友
            SearchResultDataSet<PersonUser> resultsSchool = null;
            PersonUserFullTextQuery querySchool = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.School))
            {
                querySchool.School = personUser.School;
                querySchool.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                querySchool.SortOrder = SortOrder.Descending;
                resultsSchool = PersonUserSearchManager.Instance().Search(currentDomainUser, querySchool);
                ViewData["resultsSchool"] = MayKnowExceptMeAndFirend(currentDomainUser.UserID, resultsSchool.Records, displayCount ?? 50);
            }
            #endregion

            #region 同城
            SearchResultDataSet<PersonUser> resultsCity = null;
            PersonUserFullTextQuery queryCity = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.NowAreaCode))
            {
                queryCity.NowAreaCode = personUser.NowAreaCode;
                queryCity.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCity.SortOrder = SortOrder.Descending;
                resultsCity = PersonUserSearchManager.Instance().Search(currentDomainUser, queryCity);
                ViewData["resultsCity"] = MayKnowExceptMeAndFirend(currentDomainUser.UserID, resultsCity.Records, displayCount ?? 50);
            }
            #endregion

            #region 同乡
            SearchResultDataSet<PersonUser> resultsHome = null;
            PersonUserFullTextQuery queryHome = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.HomeAreaCode))
            {
                queryHome.HomeAreaCode = personUser.HomeAreaCode;
                queryHome.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryHome.SortOrder = SortOrder.Descending;
                resultsHome = PersonUserSearchManager.Instance().Search(currentDomainUser, queryHome);
                ViewData["resultsHome"] = MayKnowExceptMeAndFirend(currentDomainUser.UserID, resultsHome.Records, displayCount ?? 50);
            }
            #endregion

            ViewData["currentDomainUserID"] = currentDomainUser.UserID;
            return View("Pages/Friends/ListMightKnowUsers.aspx", "Masters/Friend.master");
        }

        /// <summary>
        /// 判断我认识的人包不包括我的好友，和自己，包括就移除，并返回出去我认识的人
        /// </summary>
        /// <param name="displayCount"></param>
        /// <param name="CurrentID"></param>
        /// <param name="result"></param>
        /// <param name="listUserID"></param>
        /// <returns></returns>
        private IList<PersonUser> MayKnowExceptMeAndFirend(int CurrentID, IList<PersonUser> result, int displayCount)
        {
            IList<PersonUser> personList = new List<PersonUser>();
            IList<PersonUser> personListResult = new List<PersonUser>();
            for (int i = 0; i < result.Count; i++)
            {
                int userID = result[i].UserID;
                if (!Friends.IsFriend(CurrentID, userID))
                {
                    if (userID != CurrentID)
                    {
                        personList.Add(result[i]);
                    }
                }
            }
            if (personList.Count > displayCount)
            {
                for (int i = 0; i < displayCount; i++)
                {
                    personListResult.Add(personList[i]);
                }
            }
            else
                personListResult = personList;
            return personListResult;
        }

        public ActionResult Control_MayKnowUsersBlock(string userDomainName, int? displayCount)
        {
            List<int> userlist = MayKnowUserID(userDomainName, displayCount ?? 9);
            IList<User> mayUserList = new List<User>();
            foreach (int userID in userlist)
            {
                User mayUser = Users.GetUser(userID, false);
                if (mayUser != null)
                {
                    mayUserList.Add(mayUser);
                }
            }
            ViewData["MayUser"] = mayUserList;
            return PartialView("Controls/PersonUsers/MayKnowUsersBlock.ascx");

        }

        /// <summary>
        /// 用户空间首页 获取我可能认识的人
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        private List<int> MayKnowUserID(string userDomainName, int displayCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            PersonUser personUser = currentDomainUser as PersonUser;
            ViewData["currentDomainUserID"] = currentDomainUser.UserID;
            List<int> listUserID = new List<int>();

            #region 同事
            SearchResultDataSet<PersonUser> resultsCompany = null;
            PersonUserFullTextQuery queryCompany = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.CompanyName))
            {
                queryCompany.CompanyName = personUser.CompanyName;
                queryCompany.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCompany.SortOrder = SortOrder.Descending;
                resultsCompany = PersonUserSearchManager.Instance().Search(currentDomainUser, queryCompany);

                for (int i = 0; i < (resultsCompany.Records.Count > displayCount ? displayCount : resultsCompany.Records.Count); i++)
                {
                    int userID = resultsCompany.Records[i].UserID;
                    if (!Friends.IsFriend(currentDomainUser.UserID, userID))
                    {
                        if (userID != currentDomainUser.UserID)
                        {
                            listUserID.Add(userID);
                        }
                    }
                }
            }

            if (listUserID.Count > displayCount)
            {
                return listUserID.GetRange(0, displayCount);
            }

            #endregion

            #region 二层朋友
            SearchResultDataSet<PersonUser> resultsSecondFriends = null;
            PersonUserFullTextQuery querySecondFriends = new PersonUserFullTextQuery();
            querySecondFriends.SearchRange = 2;
            querySecondFriends.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
            querySecondFriends.SortOrder = SortOrder.Descending;
            resultsSecondFriends = PersonUserSearchManager.Instance().Search(currentDomainUser, querySecondFriends);

            if (resultsSecondFriends.Records.Count > (displayCount - listUserID.Count))
                listUserID = listUser((displayCount - listUserID.Count), currentDomainUser.UserID, resultsSecondFriends, ref listUserID);
            else
                listUserID = listUser(resultsSecondFriends.Records.Count, currentDomainUser.UserID, resultsSecondFriends, ref listUserID);


            if (listUserID.Count > displayCount)
            {
                return listUserID.GetRange(0, displayCount);
            }
            #endregion

            #region 同行
            SearchResultDataSet<PersonUser> resultsCompanyTrade = null;
            PersonUserFullTextQuery queryCompanyTrade = new PersonUserFullTextQuery();
            if (personUser.CompanyTrade > 0)
            {
                queryCompanyTrade.CompanyTrade = personUser.CompanyTrade;
                queryCompanyTrade.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCompanyTrade.SortOrder = SortOrder.Descending;
                resultsCompanyTrade = PersonUserSearchManager.Instance().Search(currentDomainUser, queryCompanyTrade);

                if (resultsCompanyTrade.Records.Count > (displayCount - listUserID.Count))
                    listUserID = listUser((displayCount - listUserID.Count), currentDomainUser.UserID, resultsCompanyTrade, ref listUserID);
                else
                    listUserID = listUser(resultsCompanyTrade.Records.Count, currentDomainUser.UserID, resultsCompanyTrade, ref listUserID);

                if (listUserID.Count > displayCount)
                {
                    return listUserID.GetRange(0, displayCount);
                }
            }
            #endregion

            #region 校友
            SearchResultDataSet<PersonUser> resultsSchool = null;
            PersonUserFullTextQuery querySchool = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.School))
            {
                querySchool.School = personUser.School;
                querySchool.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                querySchool.SortOrder = SortOrder.Descending;
                resultsSchool = PersonUserSearchManager.Instance().Search(currentDomainUser, querySchool);

                if (resultsSchool.Records.Count > (displayCount - listUserID.Count))
                    listUserID = listUser((displayCount - listUserID.Count), currentDomainUser.UserID, resultsSchool, ref listUserID);
                else
                    listUserID = listUser(resultsSchool.Records.Count, currentDomainUser.UserID, resultsSchool, ref listUserID);

                if (listUserID.Count > displayCount)
                {
                    return listUserID.GetRange(0, displayCount);
                }
            }
            #endregion

            #region 同城
            SearchResultDataSet<PersonUser> resultsCity = null;
            PersonUserFullTextQuery queryCity = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.NowAreaCode))
            {
                queryCity.NowAreaCode = personUser.NowAreaCode;
                queryCity.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryCity.SortOrder = SortOrder.Descending;
                resultsCity = PersonUserSearchManager.Instance().Search(currentDomainUser, queryCity);

                if (resultsCity.Records.Count > (displayCount - listUserID.Count))
                    listUserID = listUser((displayCount - listUserID.Count), currentDomainUser.UserID, resultsCity, ref listUserID);
                else
                    listUserID = listUser(resultsCity.Records.Count, currentDomainUser.UserID, resultsCity, ref listUserID);

                if (listUserID.Count > displayCount)
                {
                    return listUserID.GetRange(0, displayCount);
                }
            }
            #endregion

            #region 同乡
            SearchResultDataSet<PersonUser> resultsHome = null;
            PersonUserFullTextQuery queryHome = new PersonUserFullTextQuery();
            if (!string.IsNullOrEmpty(personUser.HomeAreaCode))
            {
                queryHome.HomeAreaCode = personUser.HomeAreaCode;
                queryHome.SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;
                queryHome.SortOrder = SortOrder.Descending;
                resultsHome = PersonUserSearchManager.Instance().Search(currentDomainUser, queryHome);

                if (resultsHome.Records.Count > (displayCount - listUserID.Count))
                    listUserID = listUser((displayCount - listUserID.Count), currentDomainUser.UserID, resultsHome, ref listUserID);
                else
                    listUserID = listUser(resultsHome.Records.Count, currentDomainUser.UserID, resultsHome, ref listUserID);

                if (listUserID.Count > displayCount)
                {
                    return listUserID.GetRange(0, displayCount);
                }
            }
            #endregion

            return listUserID;
        }

        /// <summary>
        /// 判断我认识的人包不包括我的好友，和自己，不包括就把用户的ID加到listUserID里面，并返回出去
        /// </summary>
        /// <param name="displayCount"></param>
        /// <param name="CurrentID"></param>
        /// <param name="result"></param>
        /// <param name="listUserID"></param>
        /// <returns></returns>
        private List<int> listUser(int displayCount, int CurrentID, SearchResultDataSet<PersonUser> result, ref List<int> listUserID)
        {

            for (int i = 0; i < displayCount; i++)
            {
                int userID = result.Records[i].UserID;
                if (!Friends.IsFriend(CurrentID, userID))
                {
                    if (userID != CurrentID)
                    {
                        if (!listUserID.Contains(userID))
                        {
                            listUserID.Add(userID);
                        }
                    }
                }
            }
            return listUserID;
        }

        /// <summary>
        /// 在线朋友
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListOnlineFriends(string userDomainName)
        {
            return View("Pages/Friends/ListOnlineFriends.aspx", "Masters/Friend.master");
        }

        /// <summary>
        /// 在线好友列表
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult Control_ListOnlineFriends(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            IList<PersonUser> onLineFriends = Friends.GetOnlineFriends(currentDomainUser.UserID);
            ViewData["domainUser"] = currentDomainUser;
            return PartialView("Controls/Friends/ListOnlineFriends.ascx", onLineFriends);
        }

        /// <summary>
        /// 共同的朋友
        /// </summary>
        /// <returns></returns>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult MutualFriends(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Navigation_BothFriend", friendApplicationID));
            return View("Pages/Friends/MutualFriends.aspx", "Masters/Friend.master");
        }

        /// <summary>
        /// 共同朋友列表
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult Control_MutualFriends(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            if (CurrentUser == null)
            {
                return Redirect(SiteUrls.Instance().Login(true));
            }

            IList<PersonUser> mutualFriends = Friends.GetMutualFriends(currentDomainUser.UserID, CurrentUser.UserID);
            ViewData["domainUser"] = currentDomainUser;
            return PartialView("Controls/Friends/ListMutualFriends.ascx", mutualFriends);
        }

        /// <summary>
        /// 人脉网络
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListConnectPaths(string userDomainName, int? tier, int? toUserID, int? pageIndex)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_TheseNetwork", friendApplicationID));
            #endregion

            User domainUser = GetCurrentDomainUser(userDomainName);
            if (CurrentUser == null)
            {
                return Redirect(SiteUrls.Instance().Login(true));
            }

            int newPageIndex = 1;
            if (pageIndex.HasValue && pageIndex.Value > 1)
            {
                newPageIndex = pageIndex.Value;
            }

            int pageSize = 50;
            ViewData["pageSize"] = pageSize;

            int oneTierCount = 0;
            List<FriendConnectPath> oneTierList = Friends.GetConnectPaths(domainUser.UserID, 1, true, newPageIndex, pageSize, out oneTierCount);
            ViewData["oneTierCount"] = oneTierCount;

            int twoTierCount = 0;
            List<FriendConnectPath> twoTierList = Friends.GetConnectPaths(domainUser.UserID, 2, true, newPageIndex, pageSize, out twoTierCount);
            ViewData["twoTierCount"] = twoTierCount;

            //int threeTierCount = 0;
            //List<FriendConnectPath> threeTierList = Friends.GetConnectPaths(domainUser.UserID, 3, true, newPageIndex, pageSize, out threeTierCount);
            //ViewData["threeTierCount"] = threeTierCount;

            Dictionary<FriendConnectPath, List<User>> dictionary = new Dictionary<FriendConnectPath, List<User>>();
            List<FriendConnectPath> pathList = new List<FriendConnectPath>();
            List<User> userList = new List<User>();
            int totalRecord = 0;

            if (toUserID.HasValue && toUserID.Value > 0)
            {
                pathList = Friends.GetConnectPaths(domainUser.UserID, 2, toUserID.Value);
                User toUser = Users.GetUser(toUserID.Value, false);
                ViewData["displayName"] = SPBHtmlHelper.Instance().UserDisplayName(toUser);
            }
            else
            {
                if (tier.HasValue && (int)tier > 1)
                {
                    if (tier.Value == 2)
                    {
                        pathList = twoTierList;
                        totalRecord = twoTierCount;
                    }
                    //else if (tier.Value == 3)
                    //{
                    //    pathList = threeTierList;
                    //    totalRecord = threeTierCount;
                    //}
                }
                else
                {
                    pathList = oneTierList;
                    totalRecord = oneTierCount;
                }
                ViewData["totalRecord"] = totalRecord;
            }


            foreach (FriendConnectPath path in pathList)
            {
                userList = new List<User>();
                foreach (int userID in path.PathUserIDs)
                {
                    User user = Users.GetUser(userID, false);
                    if (user != null)
                        userList.Add(user);
                }
                dictionary.Add(path, userList);
            }
            bool isOwner = false;
            if (IsUserDomainOwner(userDomainName))
                isOwner = true;
            ViewData["isOwner"] = isOwner;
            if (((PersonUser)domainUser) != null)
                ViewData["gender"] = ((PersonUser)domainUser).Gender;
            return View("Pages/Friends/ListConnectPaths.aspx", "Masters/Friend.master", dictionary);
        }

        /// <summary>
        /// 空间主页朋友模块
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ListFriendsBlock(string userDomainName, int? displayCount)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            IList<Friend> friends;
            if (CurrentUser != null && domainUser.UserID == CurrentUser.UserID)
            {
                friends = Friends.GetTopFriends(domainUser.UserID, true, displayCount ?? 9);
                ViewData["isOwner"] = true;
            }
            else
            {
                friends = Friends.GetTopFriends(domainUser.UserID, false, displayCount ?? 9);
            }

            return PartialView("Controls/Friends/ListFriendsBlock.ascx", friends);
        }
        /// <summary>
        /// 空间主页共同好友模块
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult Control_BothFriendsBlock(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            if (CurrentUser == null)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            IList<PersonUser> mutualFriends = Friends.GetMutualFriends(currentDomainUser.UserID, CurrentUser.UserID);
            ViewData["domainUser"] = currentDomainUser;

            return PartialView("Controls/Friends/BothFriendsBlock.ascx", mutualFriends);
        }

        /// <summary>
        /// 空间主页共同好友模块
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="displayCount"></param>
        /// <returns></returns>
        public ActionResult ListBothFriends(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            if (CurrentUser == null)
            {
                return Redirect(SiteUrls.Instance().Login(true));
            }

            IList<PersonUser> mutualFriends = Friends.GetMutualFriends(currentDomainUser.UserID, CurrentUser.UserID);
            ViewData["domainUser"] = currentDomainUser;

            return View("Pages/Friends/ListBothFriends.aspx", "Masters/Friend.master", mutualFriends);
        }


        public ActionResult Control_FriendCard(string userDomainName, string id)
        {
            if (string.IsNullOrEmpty(id))
                return new EmptyResult();

            string[] ids = id.Split(',');
            if (ids != null && ids.Length > 0)
            {
                Friend friend = Friends.GetFriend(Convert.ToInt32(ids[1]), false);
                ViewData["friend"] = friend;
                PersonUser personUser = Users.GetUser(Convert.ToInt32(ids[0]), false) as PersonUser;

                bool isManager = false;
                if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                    isManager = true;
                ViewData["isManager"] = isManager;

                return PartialView("Controls/Friends/FriendCard.ascx", personUser);
            }

            return new EmptyResult();
        }

        /// <summary>
        /// 收藏的朋友
        /// </summary>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserVisitAndUserDomainHitTime]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFavoriteUsers(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Navigation_FavoriteUsers", friendApplicationID));
            #endregion
            User domainUser = GetCurrentDomainUser(userDomainName);
            return View("Pages/Friends/ListFavoriteUsers.aspx", "Masters/Friend.master");
        }



        /// <summary>
        /// 收藏朋友列表
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="isFriend"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListFavoriteUsers(string userDomainName, bool? isFriend)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (CurrentUser == null || (CurrentUser != null && domainUser.UserID != CurrentUser.UserID && !CurrentUser.IsContentAdministrator))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            IList<Favorite> favoriteUsers = FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).GetFavoritesOfUser(domainUser.UserID);
            return PartialView("Controls/Friends/ListFavoriteUsers.ascx", favoriteUsers);
        }

        /// <summary>
        /// 两人最佳连接路径
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult Control_BestConnectPath(string userDomainName)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            ViewData["domainUser"] = domainUser;

            bool isOwner = IsUserDomainOwner(userDomainName);
            FriendConnectPath bestPath = null;

            if (isOwner || CurrentUser == null)
            {
                ViewData["isShow"] = false;
            }
            else
            {
                ViewData["isShow"] = true;

                bestPath = Friends.GetConnectPath(CurrentUser.UserID, domainUser.UserID);
            }

            return PartialView("Controls/Friends/BestConnectPath.ascx", bestPath);
        }


        /// <summary>
        /// 编辑朋友
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="friendID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_EditFriend(string userDomainName, int friendID)
        {
            Friend friend = Friends.GetFriend(friendID, true);
            IsFriends(friend);
            ViewData["Friendliness"] = (int)friend.Friendliness;
            return PartialView("Controls/Friends/EditFriend.ascx", friend);
        }

        ///// <summary>
        ///// 朋友菜单
        ///// </summary>
        ///// <param name="userDomainName"></param>
        ///// <param name="menuType"></param>
        ///// <returns></returns>
        //[AcceptVerbs(HttpVerbs.Get)]
        //[UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        //public ActionResult Control_FriendMenu(string userDomainName, FriendMenuTypes menuType)
        //{
        //    bool isManager = false;
        //    if (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator)
        //        isManager = true;
        //    ViewData["isManager"] = isManager;
        //    User domainUser = GetCurrentDomainUser(userDomainName);
        //    ViewData["menuType"] = (int)menuType;
        //    ViewData["userID"] = domainUser.UserID;
        //    return PartialView("Controls/Friends/FriendMenu.ascx");
        //}

        /// <summary>
        /// 列表显示分组
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListGroups(string userDomainName)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            List<UserCategory> list = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategories(domainUser.UserID, true);

            ViewData["myFriendCounts"] = domainUser.FriendCount;

            bool isManager = false;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                isManager = true;
            ViewData["isManager"] = isManager;

            return PartialView("Controls/Friends/ListGroups.ascx", list);
        }

        /// <summary>
        /// 朋友列表
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = false)]
        public ActionResult Control_ListFriends(string userDomainName, string keyWords, int? groupID, int? pageIndex, Friendliness? friendliness, bool isManager)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            List<UserCategory> list = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategories(domainUser.UserID, true);
            ViewData["UserCategories"] = list;

            ViewData["userID"] = domainUser.UserID;

            PagingDataSet<Friend> pds;
            if (isManager)
                pds = Friends.GetMyFriends(keyWords, domainUser.UserID, groupID, friendliness, pageIndex ?? 1, 15);
            else
                pds = Friends.GetFriends(domainUser.UserID, pageIndex ?? 1);

            if (friendliness.HasValue)
                ViewData["friendliness"] = friendliness.Value;

            if (isManager)
            {
                StringBuilder sb = new StringBuilder();
                PagingDataSet<Friend> pds2 = Friends.GetMyFriends(string.Empty, domainUser.UserID, 0, null, 1, ValueHelper.GetSqlMaxInt() / 100);
                foreach (Friend friend in pds2.Records)
                {
                    if (friend.FriendUser == null)
                        continue;
                    sb.AppendFormat(",'{0}'", friend.FriendUser.DisplayName);
                }

                if (sb.Length > 0)
                    sb.Remove(0, 1);

                ViewData["friendDisplayName"] = sb.ToString();
                ViewData["keyWords"] = keyWords;
                return PartialView("Controls/Friends/ListFriends.ascx", pds);
            }
            else
            {
                ViewData["domainUser"] = domainUser;
                return PartialView("Controls/Friends/ListFriendsForOther.ascx", pds);
            }
        }



        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UserSearch(string userDomainName)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            int categoryID = Request.Form.GetInt("groupID", 0);
            string keyWords = Request.Form.GetString("keyword", string.Empty);

            Friendliness? friendliness;
            if (string.IsNullOrEmpty(Request.Form.GetString("sortbyselect", string.Empty)))
                friendliness = null;
            else
                friendliness = (Friendliness)Request.Form.GetInt("sortbyselect", 0);


            return RedirectToAction("Control_ListFriends", new RouteValueDictionary { { "userDomainName", userDomainName }, { "keyWords", keyWords }, { "groupID", categoryID }, { "pageIndex", 1 }, { "friendliness", friendliness }, { "isManager", true } });

            //return PartialView("Controls/Friends/ListFriends.ascx", pds);
        }

        /// <summary>
        /// 导出朋友列表到CSV文件
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_ExportFriendToCSV(string userDomainName)
        {
            return PartialView("Controls/Friends/ExportFriendToCSV.ascx");
        }

        /// <summary>
        /// 将朋友添加到分组
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="friendID"></param>
        /// <returns></returns>
        public ActionResult Control_EditFriendToGroup(string userDomainName, int friendID)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            if (CurrentUser == null || (CurrentUser != null && domainUser.UserID != CurrentUser.UserID && !CurrentUser.IsContentAdministrator))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            List<UserCategory> categoryList = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategories(CurrentUser.UserID, true);
            //List<UserCategory> selectedCategory = null;

            if (categoryList == null || categoryList.Count == 0)
            {
                //TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                //TempData["StatusMessageContent"] = GetResourceString("Message_EmptyFriendGroup", friendApplicationID);
                //return RedirectToAction("Control_EditGroup", "UserDomain", new RouteValueDictionary { { "userDomainName", userDomainName }, { "multiUserCategoryItemKey", MultiUserCategoryItemKeys.Instance().Friend() } });

                UserCategory userCategory = new UserCategory();

                userCategory.CategoryName = ResourceManager.GetString("Title_DefaultGroup", Globals.GetCurrentUserLanguage(), ApplicationIDs.Instance().Friend());
                userCategory.PrivacyStatus = PrivacyStatuses.Public;
                userCategory.UserID = domainUser.UserID;

                MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).CreateUserCategory(userCategory);
                categoryList = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategories(CurrentUser.UserID, true);
            }
            //else
            //    selectedCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategoriesOfItem(friendID, true);

            Friend friend = Friends.GetFriend(friendID, false);
            ViewData["friend"] = friend;
            //ViewData["selectedCategory"] = selectedCategory;

            return PartialView("Controls/Friends/EditFriendToGroup.ascx", categoryList);
        }

        ///// <summary>
        ///// 根据用户id，编辑朋友分组
        ///// </summary>
        ///// <param name="userDomainName"></param>
        ///// <param name="userID"></param>
        ///// <param name="friendUserID"></param>
        ///// <returns></returns>
        //[UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        //public ActionResult AcceptFriendInvitation(string userDomainName, int invitationID)
        //{
        //    Invitation invitation = Invitations.GetInvitation(invitationID);
        //    if (invitation == null || invitation.ReceiverUserID != CurrentUser.UserID)
        //        return new EmptyResult();

        //    Invitations.SetStatus(invitationID, InvitationStatuses.Accept);            

        //    Friend friend = Friends.GetFriendByUserID(invitation.ReceiverUserID, invitation.SenderUserID);

        //    if (friend == null)
        //    {
        //        Friends.CreateFriend(invitation.ReceiverUserID, invitation.SenderUserID);
        //        friend = Friends.GetFriendByUserID(invitation.ReceiverUserID, invitation.SenderUserID);
        //        if (friend == null)
        //            return new EmptyResult();
        //    }
        //    return RedirectToAction("Control_EditFriendToGroup", new RouteValueDictionary { { "userDomainName", userDomainName }, { "friendID", friend.FriendID } });
        //}

        #region PersonUser FullTextSearch






        #endregion

        #endregion

        #region Manage

        /// <summary>
        /// 更新朋友所属分组
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="friendID"></param>
        /// <returns></returns>
        public ActionResult UpdateFriendToGroup(string userDomainName, int friendID)
        {
            string[] categoryIDs = null;
            List<int> categoryList = new List<int>();

            if (!string.IsNullOrEmpty(Request.Form.GetString("categoryCheckBoxGroup", string.Empty)))
            {
                categoryIDs = Request.Form.GetString("categoryCheckBoxGroup", string.Empty).Split(',');
                if (categoryIDs != null && categoryIDs.Length > 0)
                {
                    foreach (var category in categoryIDs)
                    {
                        int int_categoryID = 0;
                        int.TryParse(category, out int_categoryID);
                        if (int_categoryID > 0)
                            categoryList.Add(int_categoryID);
                    }
                }
            }

            Friends.UpdateFriendToGroup(friendID, categoryList);
            MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).UpdateUserCategoriesOfItem(friendID, categoryList);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_EditFriendToGroup", new RouteValueDictionary { { "userDomainName", userDomainName }, { "friendID", friendID } });
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetFriendToGroup(string userDomainName, string keyWords, int friendID, int categoryID, int? groupID, int? pageIndex, Friendliness? sortBy)
        {
            List<UserCategory> selectedCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategoriesOfItem(friendID, true);
            List<int> tempCategoryIDs = new List<int>(selectedCategory.Select(n => n.CategoryID));

            if (tempCategoryIDs.Contains(categoryID))
                tempCategoryIDs.Remove(categoryID);
            else
                tempCategoryIDs.Add(categoryID);
            

            Friends.UpdateFriendToGroup(friendID, tempCategoryIDs);
            MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).UpdateUserCategoriesOfItem(friendID, tempCategoryIDs);

            return RedirectToAction("Control_ListFriends", new RouteValueDictionary { { "userDomainName", userDomainName }, { "keyWords", keyWords }, { "groupID", groupID }, { "pageIndex", pageIndex }, { "friendliness", sortBy }, { "isManager", true } });
        }

        /// <summary>
        /// 批量设置好友分组
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="groupID"></param>
        /// <param name="pageIndex"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetFriendsToGroup(string userDomainName, string keyWords, int categoryID, int? groupID, int? pageIndex, Friendliness? sortBy)
        {
            string[] friendIDs = null;
            List<int> tempCategoryIDs = null;

            if (!string.IsNullOrEmpty(Request.Form.GetString("friendCheckBoxGroup", string.Empty)))
            {
                friendIDs = Request.Form.GetString("friendCheckBoxGroup", string.Empty).Split(',');
                if (friendIDs != null && friendIDs.Length > 0)
                {
                    foreach (string friendID in friendIDs)
                    {
                        int fid = 0;
                        int.TryParse(friendID, out fid);

                        List<UserCategory> selectedCategory = MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).GetUserCategoriesOfItem(fid, true);
                        tempCategoryIDs = new List<int>(selectedCategory.Select(n => n.CategoryID));

                        if (categoryID > 0 && !tempCategoryIDs.Contains(categoryID))
                            tempCategoryIDs.Add(categoryID);
                        if (fid > 0)
                        {
                            Friends.UpdateFriendToGroup(fid, tempCategoryIDs);
                            MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).UpdateUserCategoriesOfItem(fid, tempCategoryIDs);
                        }
                    }
                }
            }

            return RedirectToAction("Control_ListFriends", new RouteValueDictionary { { "userDomainName", userDomainName }, { "keyWords", keyWords }, { "groupID", groupID }, { "pageIndex", pageIndex }, { "friendliness", sortBy }, { "isManager", true } });
        }

        /// <summary>
        /// 创建收藏用户
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="fu"></param>
        /// <returns></returns>
        public ActionResult CreateFavoriteUser(string userDomainName, int favoriteUserID)
        {
            Favorite favorite = new Favorite();
            favorite.UserID = CurrentUser.UserID;
            favorite.DisplayName = CurrentUser.DisplayName;
            favorite.ItemID = favoriteUserID;
            favorite.PrivacyStatus = PrivacyStatuses.Public;

            FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).CreateFavorite(favorite);

            return RedirectToAction("Home", "UserDomain", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 管理分组页面
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ManageGroups(string userDomainName)
        {
            #region 标题
            SetPageTitle(userDomainName, GetResourceString("Title_ManageGroup", friendApplicationID));
            #endregion

            return View("Pages/Friends/ManageGroups.aspx", "Masters/Friend.master");
        }



        /// <summary>
        /// 更新朋友
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="friendID"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateFriend(string userDomainName, int friendID)
        {
            Friend friend = Friends.GetFriend(friendID, true);
            IsFriends(friend);
            Friendliness oldFriendliness = friend.Friendliness;
            friend.Friendliness = (Friendliness)Convert.ToInt32(Request.Form.GetString("friendlinessSelect", "99"));
            friend.IsHidden = Request.Form.GetBool("isHidden", false);
            friend.NoteName = Request.Form.GetString("noteName", string.Empty);
            friend.Remark = Request.Form.GetString("remark", string.Empty).Trim();

            Friends.UpdateFriend(friend, oldFriendliness);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            return RedirectToAction("Control_EditFriend", new RouteValueDictionary { { "userDomainName", userDomainName }, { "friendID", friendID } });
        }



        /// <summary>
        /// 更新亲密程度
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="groupID"></param>
        /// <param name="pageIndex"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult UpdateFriendliness(string userDomainName, string keyWords, int friendliness, int? groupID, int? pageIndex, Friendliness? sortBy)
        {
            string[] friendIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("friendCheckBoxGroup", string.Empty)))
            {
                friendIDs = Request.Form.GetString("friendCheckBoxGroup", string.Empty).Split(',');
                if (friendIDs != null && friendIDs.Length > 0)
                {
                    foreach (string friendID in friendIDs)
                    {
                        int fid = 0;
                        int.TryParse(friendID, out fid);
                        if (fid > 0)
                        {
                            Friend friend = Friends.GetFriend(fid, true);
                            Friendliness oldFriendliness = friend.Friendliness;
                            friend.Friendliness = (Friendliness)friendliness;

                            Friends.UpdateFriend(friend, oldFriendliness);
                        }
                    }
                }
            }

            return RedirectToAction("Control_ListFriends", new RouteValueDictionary { { "userDomainName", userDomainName }, { "keyWords", keyWords }, { "groupID", groupID }, { "pageIndex", pageIndex }, { "friendliness", sortBy }, { "isManager", true } });
        }

        /// <summary>
        /// 删除好友
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="friendID"></param>
        /// <param name="groupID"></param>
        /// <param name="pageIndex"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        public ActionResult DeleteFriend(string userDomainName, string keyWords, int friendID, int? groupID, int? pageIndex, Friendliness? sortBy)
        {
            Friend friend = Friends.GetFriend(friendID, true);
            if (friend != null)
            {
                if (friend.FriendUserID == CurrentUser.UserID || friend.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator)
                {
                    Friends.DeleteFriend(friendID);
                }
            }

            return RedirectToAction("Home", new RouteValueDictionary { { "userDomainName", userDomainName }, { "keyWords", keyWords }, { "groupID", groupID }, { "pageIndex", pageIndex }, { "sortBy", sortBy } });
        }

        /// <summary>
        /// 删除收藏的用户
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="favoriteID"></param>
        /// <param name="isFriend"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFavoriteUser(string userDomainName, int favoriteID, bool? isFriend)
        {
            Favorite favorite = FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).GetFavorite(favoriteID, true);
            if (favorite != null)
            {
                FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).DeleteFavorite(favoriteID);
                //Activities.RemoveFavoriteActivitiesFromCache(favorite.UserID, favorite.ItemID);
            }
            return RedirectToAction("Control_ListFavoriteUsers", new RouteValueDictionary { { "userDomainName", userDomainName }, { "isFriend", isFriend } });
        }

        /// <summary>
        /// 批量删除收藏用户
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="isFriend"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFavoriteUsers(string userDomainName, bool? isFriend)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("favoriteCheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("favoriteCheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    foreach (string postID in deletePostIDs)
                    {
                        Favorite favorite = FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).GetFavorite(Convert.ToInt32(postID), true);
                        if (favorite != null)
                        {
                            FavoriteManager.Instance(FavoriteItemKeys.Instance().Users()).DeleteFavorite(favorite.FavoriteID);
                            //Activities.RemoveFavoriteActivitiesFromCache(favorite.UserID, favorite.ItemID);
                        }
                    }
                }
            }
            return RedirectToAction("Control_ListFavoriteUsers", new RouteValueDictionary { { "userDomainName", userDomainName }, { "isFriend", isFriend } });
        }


        /// <summary>
        /// 导出朋友列表到CSV文件
        /// </summary>
        /// <returns></returns>
        public ActionResult Control_Export(string userDomainName)
        {
            Response.Clear();
            Response.ContentType = "text/csv";
            string filename = HttpUtility.UrlEncode(userDomainName + string.Format("(From {0})", SiteSettingsManager.GetSiteSettings().SiteName) + ".csv");
            Response.AddHeader("Content-Disposition", "attachment;filename=" + filename);

            User domainUser = GetCurrentDomainUser(userDomainName);

            IList<Friend> friends = Friends.GetTopFriends(domainUser.UserID, true, ValueHelper.GetSafeSqlInt(int.MaxValue));
            List<Contact> list = new List<Contact>();
            foreach (Friend friend in friends)
            {
                list.Add(Contacts.ConvertUserToContact(domainUser, Users.GetUser(friend.FriendUserID, true)));
            }

            CSVHelper.ExprotToCSV(Response.OutputStream, list.ToArray(), (CSVTypes)Request.Form.GetInt("CSVTypes", 21));
            Response.End();
            TempData["StatusMessageType"] = StatusMessageType.Success;
            return new EmptyResult();
        }

        #endregion

        #region 标签

        /// <summary>
        /// 用户标签
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult Control_PersonUserTags(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            return PartialView("Controls/PersonUsers/ListUserSiteTags.ascx", userTags);
        }


        #endregion

        #region Help

        /// <summary>
        /// 验证当前用户是否操作自己的朋友
        /// </summary>
        private void IsFriends(Friend friend)
        {
            if (friend.FriendUserID != CurrentUser.UserID && friend.UserID != CurrentUser.UserID && !CurrentUser.IsContentAdministrator)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }
        }

        #endregion
    }

    /// <summary>
    /// 朋友菜单
    /// </summary>
    public enum FriendUserDomainSubMenus
    {
        /// <summary>
        /// 朋友列表
        /// </summary>
        MyFriends,

        /// <summary>
        /// 收藏的朋友
        /// </summary>
        FavoriteUsers,

        /// <summary>
        /// 在线朋友
        /// </summary>
        OnlineFriends,

        /// <summary>
        /// 可能认识的人
        /// </summary>
        MightKnows,

        ///// <summary>
        ///// 联系人
        ///// </summary>
        //ContactPersons,

        /// <summary>
        /// 人脉网络
        /// </summary>
        ConnectPaths,

        /// <summary>
        /// 所有朋友
        /// </summary>
        AllFriends,

        /// <summary>
        /// 共同朋友
        /// </summary>
        BothFriend
    }
}
