﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Web;

using System.Web.Security;
using System.Web;
using System.Web.Routing;
using System.Web.UI;


namespace SpaceBuilder.Common.Controllers
{
    [HttpCompress]
    public class ManageController : ManageControllerBase
    {
        /// <summary>
        /// 后台首页
        /// </summary>
        [ManageAuthorize]
        public ActionResult Home()
        {
            Dictionary<int, Application> DicApp = Applications.GetApplicationsForAdmin();

            ViewData["DicApp"] = DicApp;
            return View("Pages/Home.aspx");
        }

        /// <summary>
        /// 获取平台统计数据
        /// </summary>

        public JsonResult GetManageableCounts()
        {
            Dictionary<ServiceManageableCounts, int> dictionary = ServiceManageableCountManager.Instance().GetManageableCounts();
            List<ManageableCount> list = new List<ManageableCount>();
            if (dictionary.ContainsKey(ServiceManageableCounts.PendingPersonUserCount))
                list.Add(new ManageableCount { Name = "位用户等待审核", Value = dictionary[ServiceManageableCounts.PendingPersonUserCount], Url = SPBUrlHelper.Action("ListUsers", "ManageUser", new RouteValueDictionary { { "UserStatus", UserAccountStatuses.ApprovalPending } }) });
            if (dictionary.ContainsKey(ServiceManageableCounts.PendingCompanyUserCount))
                list.Add(new ManageableCount { Name = "位企业用户等待审核", Value = dictionary[ServiceManageableCounts.PendingCompanyUserCount], Url = SPBUrlHelper.Action("ListCompanyUsers", "ManageUser", new RouteValueDictionary { { "UserStatus", UserAccountStatuses.ApprovalPending } }) });
            if (dictionary.ContainsKey(ServiceManageableCounts.NoReadMessageCount))
                list.Add(new ManageableCount { Name = "条未读消息", Value = dictionary[ServiceManageableCounts.NoReadMessageCount], Url = SPBUrlHelper.Action("MessageInBox", "Auxiliaries") });
            if (dictionary.ContainsKey(ServiceManageableCounts.ImpeachMessageCount))
                list.Add(new ManageableCount { Name = "个用户举报", Value = dictionary[ServiceManageableCounts.ImpeachMessageCount], Url = SPBUrlHelper.Action("ManageImpeaches", "Auxiliaries") });
            if (dictionary.ContainsKey(ServiceManageableCounts.PendingConvertibleGiftExchangeApplyCount))
                list.Add(new ManageableCount { Name = "条礼品兑换申请", Value = dictionary[ServiceManageableCounts.PendingConvertibleGiftExchangeApplyCount], Url = SPBUrlHelper.Action("ManageGiftExchangeApplies", "Auxiliaries", new RouteValueDictionary { { "applyStaus", ConvertibleGiftExchangeApplyStatuses.Unsettled } }) });
            if (dictionary.ContainsKey(ServiceManageableCounts.UserGuestbookCount))
                list.Add(new ManageableCount { Name = "条空间留言等待审核", Value = dictionary[ServiceManageableCounts.UserGuestbookCount], Url = SPBUrlHelper.Action("ListGuestBook", "ManageUser", new RouteValueDictionary { { "status", AuditingStatusesForDisplay.Pending } }) });
            return Json(list, JsonRequestBehavior.AllowGet);

        }


        /// <summary>
        /// 后台管理的登录页面
        /// </summary>
        public ActionResult ManageLogin(string returnUrl)
        {
            if (!string.IsNullOrEmpty(returnUrl))
                ViewData["returnUrl"] = returnUrl;

            return View("Pages/ManageLogin.aspx");
        }

        /// <summary>
        /// 处理用户登录
        /// </summary>
        /// <param name="loginInModal">是否在模式窗口登录</param>
        /// <param name="returnUrl">登录成功后要跳转的url</param>
        public ActionResult ManageLoginAction(string returnUrl)
        {
            string userName = Request.Form.GetString("userName", string.Empty).Trim();
            string password = Request.Form.GetString("password", string.Empty);

            string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
            string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
            if (verifyCode != null)
                verifyCode = verifyCode.Trim();

            if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                TempData["UserName"] = userName;
                TempData["Password"] = password;
                return RedirectToAction("ManageLogin", new RouteValueDictionary { { "returnUrl", returnUrl } });
            }
            User userToLogin = new User();
            userToLogin.UserName = userName;
            userToLogin.Password = password;

            LoginUserStatuses loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);

            //如果作为用户名登录失败，则作为Email重试
            if (loginStatus == LoginUserStatuses.InvalidCredentials)
            {
                User userByEmail = Users.FindUserByEmail(userName);
                if (userByEmail != null)
                {
                    userToLogin.UserName = userByEmail.UserName;
                    loginStatus = Users.ValidUser(userToLogin.UserName, userToLogin.Password);
                }
            }

            //是否允许未批准用户登录
            bool enablePendingUsersToLogin = SiteSettingsManager.GetSiteSettings().EnablePendingUsersToLogin;

            // Change to let banned users in
            //
            if (loginStatus == LoginUserStatuses.Success || (enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending))
            {
                User user = Users.GetUser(userToLogin.UserName, true);
                if (user.IsInRoles(SiteSettingsManager.GetSiteSettings().RoleNamesForEnterControlPanel))
                {
                    HttpCookie adminCookie = new HttpCookie("SpaceBuilderAdminCookie");
                    adminCookie.Value = EncryptManager.EncryptTokenForVerifyCode(true.ToString());
                    if (!string.IsNullOrEmpty(FormsAuthentication.CookieDomain))
                        adminCookie.Domain = FormsAuthentication.CookieDomain;
                    adminCookie.HttpOnly = true;
                    Response.Cookies.Add(adminCookie);


                    FormsAuthentication.SetAuthCookie(userToLogin.UserName, false);
                    string redirectUrl = null;
                    if (!string.IsNullOrEmpty(Request.QueryString["returnUrl"]))
                        redirectUrl = Request.QueryString.GetString("returnUrl", string.Empty);
                    else
                        redirectUrl = SiteUrls.Instance().ManageHome();
                    return new RedirectResult(redirectUrl);
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "请以管理员帐号登录";
                    return RedirectToAction("ManageLogin", new RouteValueDictionary { { "returnUrl", returnUrl } });
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (loginStatus == LoginUserStatuses.InvalidCredentials)
                {
                    TempData["StatusMessageContent"] = "用户名或密码出错";
                }
                else if (!enablePendingUsersToLogin && loginStatus == LoginUserStatuses.AccountPending)
                {
                    TempData["StatusMessageContent"] = "超级管理员正在审核您的帐号，审核通过后您将收到邮件通知";
                }
                else if (loginStatus == LoginUserStatuses.AccountDisapproved)
                {
                    TempData["StatusMessageContent"] = "您的帐号未通过审核，不能进行登录";
                }
                else if (loginStatus == LoginUserStatuses.UnknownError)
                {
                    throw new SPBException(ExceptionTypes.Instance().UserUnknownLoginError());
                }
                else if (loginStatus == LoginUserStatuses.AccountBanned)
                {
                    TempData["StatusMessageContent"] = "您的帐号处于封禁状态，不能登录系统";
                }

                TempData["UserName"] = userName;
                TempData["Password"] = password;
                return RedirectToAction("ManageLogin", new RouteValueDictionary { { "returnUrl", returnUrl } });
            }
        }
        /// <summary>
        /// 后台退出
        /// </summary>
        //TODO: 暂时方法
        public ActionResult ManageLogout()
        {
            ///移除在线用户
            if (CurrentUser != null)
                UserOnlines.DeleteOnlineUser(CurrentUser.UserID);

            FormsAuthentication.SignOut();

            string returnUrl = null;
            if (Request.UrlReferrer != null)
                returnUrl = Request.UrlReferrer.AbsoluteUri;

            return RedirectToAction("ManageLogin", new RouteValueDictionary { { "returnUrl", returnUrl } });
        }

        #region 站点标签
        /// <summary>
        /// 站点标签
        /// </summary>
        /// <param name="userTagIteamkey">应用标志</param>      
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageSiteTags(string userTagIteamkey, int? pageIndex)
        {
            string keywords = Request.QueryString.GetString("KeyWords", string.Empty);

            PagingDataSet<SiteTag> pds = UserTagManager.Instance(userTagIteamkey).GetSiteTagsForAdmin(keywords, SiteTagsSortBy.ItemCount, pageIndex ?? 1);

            ViewData["userTagIteamkey"] = userTagIteamkey;
            TempData["keyWords"] = keywords;
            return PartialView("Controls/ManageSiteTags.ascx", pds);
        }
        /// <summary>
        /// 搜索站点标签
        /// </summary>
        /// <param name="userTagIteamkey">应用标志</param>       
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearcheSiteTags(string userTagIteamkey)
        {
            string KeyWords = Request.Form.GetString("KeyWords", string.Empty);
            return RedirectToAction("Control_ManageSiteTags", new RouteValueDictionary { { "userTagIteamkey", userTagIteamkey }, { "KeyWords", KeyWords } });
        }

        /// <summary>
        /// 删除站点标签
        /// </summary>
        /// <param name="userTagIteamkey">应用标志</param>
        /// <param name="tageName">标签名</param>  
        public ActionResult DeleteSiteTag(string userTagIteamkey, string tageName)
        {
            string keyWords = Request.QueryString.GetString("keyWords", string.Empty);
            UserTagManager.Instance(userTagIteamkey).DeleteSiteTag(tageName);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除站点标签操作已成功,需要清除缓存，前台页面才会不显示";
            return RedirectToAction("Control_ManageSiteTags", new RouteValueDictionary { { "userTagIteamkey", userTagIteamkey }, { "KeyWords", keyWords } });
        }

        public ActionResult DeleteSiteTags(string userTagIteamkey)
        {
            string selectedTagsString = Request.Form["CheckBoxGroup"];
            string keyWords = Request.QueryString.GetString("keyWords", string.Empty);
            if (!string.IsNullOrEmpty(selectedTagsString))
            {
                string[] selectedTagsStringArray = selectedTagsString.Split(',');
                foreach (string tageName in selectedTagsStringArray)
                {
                    UserTagManager.Instance(userTagIteamkey).DeleteSiteTag(tageName);
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "批量删除站点标签操作已成功,需要清除缓存，前台页面才会不显示";

            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选择要删除的站点标签";
            }
            return RedirectToAction("Control_ManageSiteTags", new RouteValueDictionary { { "userTagIteamkey", userTagIteamkey }, { "KeyWords", keyWords } });
        }

        #endregion

        #region 站点类别
        /// <summary>
        /// 显示管理类别控件
        /// </summary>
        [ManageAuthorize]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageSiteCategories(int applicationID, int? depth)
        {
            List<SiteCategory> siteCategories = SiteCategories.Instance(applicationID).GetAll(true);
            ViewData["Depth"] = depth;
            ViewData["ApplicationID"] = applicationID;
            return PartialView("Controls/ManageSiteCategories.ascx", siteCategories);
        }
        /// <summary>
        /// 显示编辑类别控件
        /// </summary>
        [ManageAuthorize]
        public ActionResult Control_EditSiteCategory(int applicationID, int? categoryID, int? parentID)
        {
            SiteCategory category;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = SiteCategories.Instance(applicationID).Get(categoryID.Value, true);
            }
            else
            {
                if (parentID.HasValue)
                {
                    SiteCategory parentCategory = SiteCategories.Instance(applicationID).Get(parentID.Value, true);
                    if (parentCategory != null)
                        ViewData["ParentCategoryName"] = parentCategory.CategoryName;
                }
                category = new SiteCategory();
            }
            ViewData["CategoryID"] = categoryID;
            ViewData["ParentID"] = parentID;
            ViewData["ApplicationID"] = applicationID;
            return PartialView("Controls/EditSiteCategory.ascx", category);
        }
        /// <summary>
        /// 更新站点类别
        /// </summary>
        [ManageAuthorize]
        public ActionResult UpdateSiteCategory(int applicationID, int? categoryID, int? parentID)
        {
            SiteCategory category;
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                category = SiteCategories.Instance(applicationID).Get(categoryID.Value, true);
            }
            else
            {
                category = new SiteCategory();
            }
            category.CategoryName = Request.Form.GetString("CategoryName", string.Empty).Trim();
            category.DisplayOrder = Request.Form.GetInt("DisplayOrder", 100);
            if (string.IsNullOrEmpty(category.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "站点类别名不能为空";
                return RedirectToAction("Control_EditSiteCategory", new RouteValueDictionary { { "applicationID", applicationID }, { "categoryID", categoryID }, { "parentID", parentID } });
            }
            if (categoryID.HasValue && categoryID.Value > 0)
            {
                SiteCategories.Instance(applicationID).Update(category);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "更新站点类别操作成功";
            }
            else
            {
                category.ApplicationID = applicationID;
                if (parentID.HasValue)
                {
                    category.ParentID = parentID.Value;
                }
                SiteCategories.Instance(applicationID).Create(category);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "创建站点类别操作成功";
            }

            //添加站点类别名称语言包
            Dictionary<string, string> categoryNameRes = new Dictionary<string, string>();
            foreach (Language language in Languages.GetLanguages(true).Values)
            {
                string languageName = Request.Form.GetString("CategoryName_" + language.LanguageCode, string.Empty);
                if (!string.IsNullOrEmpty(languageName))
                    categoryNameRes[language.LanguageCode] = languageName;
            }

            if (categoryNameRes.Count > 0)
                SiteCategories.UpdateSiteCategorieNameRes(category.CategoryID, categoryNameRes);


            return RedirectToAction("Control_EditSiteCategory", new RouteValueDictionary { { "applicationID", applicationID }, { "categoryID", categoryID }, { "parentID", parentID } });
        }


        /// <summary>
        /// 合并或移动类别
        /// </summary>
        /// <param name="isMerge">是合并还是移动，true为合并，false为移动</param>
        /// <param name="FilterID">需要排除当前类别及所有子类别</param>
        [ManageAuthorize]
        public ActionResult Control_MergeOrMoveSiteCategory(int applicationID, bool isMerge, int filterID)
        {
            //源类别
            SiteCategory fromCategory = SiteCategories.Instance(applicationID).Get(filterID, true);
            //所有活动类别
            List<SiteCategory> allCategories = SiteCategories.Instance(applicationID).GetAll(true);
            //所有需要排除的类别，包括源类别及其所有子类别
            List<SiteCategory> removedCategories = SiteCategories.Instance(applicationID).GetAllChilds(fromCategory.CategoryID, true);
            removedCategories.Add(fromCategory);
            //排除后的类别集合
            List<SiteCategory> filteredCategories = new List<SiteCategory>();

            foreach (var category in allCategories)
            {
                bool isMeet = false;
                foreach (var removedCategory in removedCategories)
                {
                    if (category.CategoryID == removedCategory.CategoryID)
                    {
                        isMeet = true;
                        break;
                    }
                }
                if (isMeet)
                {
                    continue;
                }
                filteredCategories.Add(category);
            }

            ViewData["isMerge"] = isMerge;
            ViewData["fromCategory"] = fromCategory;

            return PartialView("Controls/MergeOrMoveCategory.ascx", filteredCategories);
        }

        /// <summary>
        /// 合并或移动站点类别
        /// </summary>
        [ManageAuthorize]
        public ActionResult MergeOrMoveCategory(int applicationID, bool isMerge, int fromCategoryID)
        {
            int toCategoryID = Request.Form.GetInt("toCategoryID", 0);
            if (isMerge)
            {
                SiteCategories.Instance(applicationID).Merge(fromCategoryID, toCategoryID);
            }
            else
            {
                SiteCategories.Instance(applicationID).Move(fromCategoryID, toCategoryID);
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            if (isMerge)
            {
                TempData["StatusMessageContent"] = "合并站点类别操作已成功";
                fromCategoryID = toCategoryID;
            }
            else
                TempData["StatusMessageContent"] = "移动站点类别操作已成功";
            return RedirectToAction("Control_MergeOrMoveSiteCategory", new RouteValueDictionary { { "applicationID", applicationID }, { "isMerge", isMerge }, { "filterID", fromCategoryID } });
        }

        /// <summary>
        /// 删除站点类别
        /// </summary>
        [ManageAuthorize]
        public ActionResult DeleteSiteCategory(int applicationID, int categoryID)
        {
            SiteCategories.Instance(applicationID).Delete(categoryID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除站点类别操作已成功";
            int depth = Request.QueryString.GetInt("depth", 1);
            return RedirectToAction("Control_ManageSiteCategories", new RouteValueDictionary { { "applicationID", applicationID }, { "depth", depth } });
        }

        #endregion

        #region 快捷菜单
        [ManageAuthorize]
        public ActionResult Control_SiteMap()
        {
            return PartialView("Controls/ManageSiteMap.ascx");
        }

        #endregion


    }
    #region ManageSubMenu
    /// <summary>
    /// ManageSubMenu
    /// </summary>
    public enum ManageSubMenu
    {
        /// <summary>
        /// 后台首页
        /// </summary>
        ManageHome = 8881,

        /// <summary>
        /// 应用管理
        /// </summary>
        ManageApplicationsModule = 8882,

        /// <summary>
        /// 辅助模块
        /// </summary>
        ManageAuxiliaryModule = 8883,

        /// <summary>
        /// 用户管理
        /// </summary>
        ManageUsers = 8884,
        /// <summary>
        /// 系统设置
        /// </summary>
        SystemSettings = 8885,
    }
    #endregion
}
