﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Common.Controllers;
using SpaceBuilder.Web;
using SpaceBuilder.Common;
using SpaceBuilder.Club;
using System.Web.Routing;
using System.Web;

namespace SpaceBuilder.Common.Controllers
{

    [ManageAuthorize(RequireSystemAdministrator = true)]
    [HttpCompress]
    public class ManageUserController : ManageControllerBase
    {
        #region 企业用户管理

        /// <summary>
        /// 企业用户列表
        /// </summary>
        public ActionResult ListCompanyUsers(UserAccountStatuses? UserStatus, int? pageIndex)
        {
            SetPageTitle("企业用户管理");
            int pageSize = 17;
            ViewData["UserStatus"] =UserAccountStatuses.NotActive;

            if (UserStatus.HasValue)
                ViewData["UserStatus"] = UserStatus;
            CompanyUserQueryForAdmin cuq = new CompanyUserQueryForAdmin();
            if (UserStatus == UserAccountStatuses.NotActive)
            {
                cuq.AccountStatus = null;
            }
            else
            {
                cuq.AccountStatus = UserStatus ?? UserAccountStatuses.Approved;
            }
            cuq.IncludeHiddenUsers = true;
            cuq.KeyWord = Request.QueryString.GetString("keyWord", string.Empty).Trim();


            cuq.EmailFilter = Request.QueryString.GetString("UserEmail", string.Empty).Trim();
            PagingDataSet<CompanyUser> pds = Users.GetCompanyUsersForAdmin(cuq, pageSize, pageIndex ?? 1);
            return View("Pages/Member/ManageCompanyUser.aspx", "Masters/User.master", pds);
        }

        public ActionResult UpdateUserAccountStatuses(UserAccountStatuses UserStatus, string returnUrl)
        {
            string selectedUserIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedUserIDsString))
            {
                string[] selectedUserIDsStringArray = selectedUserIDsString.Split(',');
                if (selectedUserIDsStringArray != null && selectedUserIDsStringArray.Length > 0)
                {
                    int userID;
                    User u;
                    foreach (string Id in selectedUserIDsStringArray)
                    {
                        if (int.TryParse(Id, out userID))
                        {
                            if (userID > 0)
                            {
                                try
                                {
                                    u = Users.GetUser(userID, true);
                                    if (u != null)
                                    {
                                        u.AccountStatus = UserStatus;
                                        Users.UpdateUserAccountStatus(u);
                                    }
                                }
                                catch { }
                            }
                        }
                    }
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的用户";
            }
            return Redirect(returnUrl);
        }
        /// <summary>
        /// 删除企业用户
        /// </summary>
        /// <param name="userID">用户ID</param>
        public ActionResult DeleteCompanyUser(int userID, string returnUrl)
        {
            DeleteUserStatuses DelStatus = Users.DeleteUser(userID, Globals.GetCurrentUser().UserName);
            switch (DelStatus)
            {
                case DeleteUserStatuses.Deleted:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "删除用户成功";
                    break;
                case DeleteUserStatuses.DeletingUserNotFound:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "要删除的用户不存在";
                    break;
                default:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "删除失败，未知错误";
                    break;
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 注册企业用户
        /// </summary>
        public ActionResult RegCompanyUsers()
        {
            SetPageTitle("添加企业用户");
            ViewData["CompanyPropertys"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().CompanyProperty());
            ViewData["Trades"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().Trade());
            ViewData["AreaDepth"] = 1;
            ViewData["Funds"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().Fund());
            ViewData["EmployeeCounts"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().EmployeeCount());
            return View("Pages/Member/EditCompanyUser.aspx", "Masters/User.master");
        }

        /// <summary>
        /// 添加企业用户
        /// </summary>
        public ActionResult AddCompanyUserAction()
        {
            string userName = Request.Form.GetString("userName", string.Empty).Trim();
            string password = Request.Form.GetString("password", string.Empty).Trim();
            string email = Request.Form.GetString("email", string.Empty).Trim();
            string Licence = Request.Form.GetString("Licence", string.Empty).Trim();
            int Trades = Request.Form.GetInt("Trades", -1);
            string areaCode = Request.Form.GetString("areaCode", string.Empty);
            int CompanyPropertys = Request.Form.GetInt("CompanyPropertys", -1);
            int Funds = Request.Form.GetInt("Funds", -1);
            int EmployeeCounts = Request.Form.GetInt("EmployeeCounts", -1);
            string Linkman = Request.Form.GetString("Linkman", string.Empty).Trim();
            string LinkmanDuty = Request.Form.GetString("LinkmanDuty", string.Empty).Trim();
            string LinkTel = Request.Form.GetString("LinkTel", string.Empty).Trim();
            string Fax = Request.Form.GetString("Fax", string.Empty).Trim();
            string Address = Request.Form.GetString("Address", string.Empty).Trim();
            string Intro = Request.Form.GetString("Intro", string.Empty).Trim();
            string CommonName = Request.Form.GetString("CommonName", string.Empty).Trim();

            CompanyUser u = new CompanyUser();

            u.IsModerated = false;

            u.UserName = userName;
            u.Password = password;
            u.PublicEmail = email;
            u.PrivateEmail = email;
            u.Licence = Licence;
            u.Trade = Trades;
            u.AreaCode = areaCode;
            u.CompanyProperty = CompanyPropertys;
            u.CommonName = CommonName;
            u.Fund = Funds;
            u.EmployeeNum = EmployeeCounts;
            u.Linkman = Linkman;
            u.LinkmanDuty = LinkmanDuty;
            u.LinkTel = LinkTel;
            u.Fax = Fax;
            u.Address = Address;
            u.Intro = Intro;
            u.UserType = UserTypes.CompanyUser;
            u.AccountStatus = UserAccountStatuses.ApprovalPending;
            CreateUpdateUserStatuses status = Users.Create(u, true);
            switch (status)
            {
                // Username already exists!
                case CreateUpdateUserStatuses.DuplicateUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，您输入的账号已被注册，请输入其他名称";
                    break;
                case CreateUpdateUserStatuses.BeyondLicensedCount:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，网站的注册人数超过了授权数量";
                    break;

                // Email already exists!
                case CreateUpdateUserStatuses.DuplicateEmailAddress:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format("对不起，您输入的电子邮箱地址已经被使用。如果您曾经注册，则可以通过<a href=\"{0}\">[获取密码]</a>重新得到您的账号、密码", SiteUrls.Instance().ForgotPassword());
                    break;

                // Unknown failure has occurred!
                case CreateUpdateUserStatuses.UnknownFailure:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，注册新用户的时候产生了一个未知错误";
                    break;

                // Username is disallowed
                case CreateUpdateUserStatuses.DisallowedUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，您输入的账号禁止使用，请输入其他名称";
                    break;

                case CreateUpdateUserStatuses.InvalidQuestionAnswer:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "不正确的密码问题和答案";
                    break;

                case CreateUpdateUserStatuses.InvalidPassword:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "您的密码长度少于本站要求的最小密码长度，请重新输入";
                    break;

                // Everything went off fine, good
                case CreateUpdateUserStatuses.Created:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "注册成功，请等待管理员的批准！";
                    break;
            }
            return RedirectToAction("RegCompanyUsers");
        }

        #endregion

        #region 个人用户管理

        /// <summary>
        /// 用户列表
        /// </summary>
        public ActionResult ListUsers(UserAccountStatuses? UserStatus, int? pageIndex)
        {
            SetPageTitle("个人用户管理");
            int pageSize = 17;

            ViewData["UserStatus"] = UserAccountStatuses.NotActive;
            if (UserStatus.HasValue)
                ViewData["UserStatus"] = UserStatus;

            PersonUserQueryForAdmin puq = new PersonUserQueryForAdmin();

            if (UserStatus == UserAccountStatuses.NotActive)
            {
                puq.AccountStatus = null;
            }
            else
            {
                puq.AccountStatus = UserStatus ?? UserAccountStatuses.Approved;
            }

            puq.KeyWord = Request.QueryString.GetString("keyWord", string.Empty).Trim();

            puq.EmailFilter = Request.QueryString.GetString("UserEmail", string.Empty).Trim();
            puq.RoleID = Request.QueryString.GetInt("UserRole", -1);
            if (!string.IsNullOrEmpty(Request.QueryString["IsModerated"]))
            {
                puq.IsModerated = Request.QueryString.GetBool("IsModerated", false);
                ViewData["IsModerated"] = puq.IsModerated;
            }

            PagingDataSet<PersonUser> pds = Users.GetPersonUsersForAdmin(puq, pageSize, pageIndex ?? 1);

            ICollection<Role> LRoles = SpaceBuilder.Common.Roles.GetRolesCanConnectToUser(true);
            ViewData["LRoles"] = LRoles;
            return View("Pages/Member/ManageUsers.aspx", "Masters/User.master", pds);
        }

        /// <summary>
        /// 删除用户
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_DeleteUser(int UserId)
        {
            User u = Users.GetUser(UserId, true);
            return PartialView("Controls/Member/DeleteUser.ascx", u);
        }

        /// <summary>
        /// 删除用户
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteUser(int UserId)
        {
            DeleteUserStatuses DelStatus;
            bool isReassign = Request.Form.GetBool("isReassign", true);
            if (!isReassign)
            {
                DelStatus = Users.DeleteUser(UserId, Globals.GetCurrentUser().UserName);
            }
            else
            {
                string reassignUserName = Request.Form.GetString("reassignUserName", string.Empty).Trim();
                DelStatus = Users.DeleteUserAndReassignContent(UserId, reassignUserName);
            }
            switch (DelStatus)
            {
                case DeleteUserStatuses.Deleted:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "删除用户成功";
                    //return PartialView("Controls/Member/DeleteUser.ascx");
                    break;
                case DeleteUserStatuses.DeletingUserNotFound:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "要删除的用户不存在";
                    break;
                case DeleteUserStatuses.InvalidReassignUserName:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "删除失败，接收人不存在";
                    break;
                default:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "删除失败，未知错误";
                    break;
            }
            return RedirectToAction("Control_DeleteUser", new RouteValueDictionary { { "UserId", UserId } });
        }

        /// <summary>
        /// 奖励或处罚
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_PrizeUser(int UserID)
        {
            ViewData["UserID"] = UserID;
            return PartialView("Controls/Member/UserPrize.ascx");
        }

        /// <summary>
        /// 更新积分
        /// </summary>
        public ActionResult Control_UpdateUserPoints(int UserID)
        {
            List<UserPointRecord> Records = new List<UserPointRecord>();
            string descript = Request.Form.GetString("descript", string.Empty).Trim();
            string itemName = string.Empty;
            bool IsPunishment = false;
            int caozuo = Request.Form.GetInt("caozuo", -1);
            if (caozuo < 0)
                throw new ArgumentException("请选择一个操作类别！");
            UserPointItemRole role = Points.GetUserPointItemRole(UserPointItemKeys.Instance().Hortation());
            if (caozuo > 0)
            {
                IsPunishment = true;
                itemName = "处罚";
                role = Points.GetUserPointItemRole(UserPointItemKeys.Instance().Punishment());
            }
            else
                itemName = "奖励";

            int BasicPoints = Request.Form.GetInt("BasicPoints", 0);
            if (BasicPoints > 0)
            {
                if (IsPunishment)
                    BasicPoints = -BasicPoints;
                UserPointRecord u = new UserPointRecord();

                u.ItemID = role.ItemID;
                u.Description = descript;
                u.PointCategory = UserPointCategory.Basic;
                u.UserID = UserID;
                u.Points = BasicPoints;
                u.PointItemName = itemName;
                Records.Add(u);
            }
            int ReportPoints = Request.Form.GetInt("ReportPoints", 0);
            if (ReportPoints > 0)
            {
                if (IsPunishment)
                    ReportPoints = -ReportPoints;
                UserPointRecord u = new UserPointRecord();
                u.ItemID = role.ItemID;
                u.Description = descript;
                u.PointCategory = UserPointCategory.Report;
                u.UserID = UserID;
                u.Points = ReportPoints;
                u.PointItemName = itemName;
                Records.Add(u);
            }
            int CreditPoints = Request.Form.GetInt("CreditPoints", 0);
            if (CreditPoints > 0)
            {
                if (IsPunishment)
                    CreditPoints = -CreditPoints;
                UserPointRecord u = new UserPointRecord();
                u.ItemID = role.ItemID;
                u.Description = descript;
                u.PointCategory = UserPointCategory.Credit;
                u.UserID = UserID;
                u.Points = CreditPoints;
                u.PointItemName = itemName;
                Records.Add(u);
            }
            int TradePoints = Request.Form.GetInt("TradePoints", 0);
            if (TradePoints > 0)
            {
                if (IsPunishment)
                    TradePoints = -TradePoints;
                UserPointRecord u = new UserPointRecord();
                u.ItemID = role.ItemID;
                u.Description = descript;
                u.PointCategory = UserPointCategory.Trade;
                u.UserID = UserID;
                u.Points = TradePoints;
                u.PointItemName = itemName;
                Records.Add(u);
            }
            try
            {
                Points.PrizeUser(Records);
                ViewData["StatusMessageType"] = (int)StatusMessageType.Success;
                ViewData["StatusMessageContent"] = "操作成功";
            }
            catch
            {
                ViewData["StatusMessageType"] = (int)StatusMessageType.Error;
                ViewData["StatusMessageContent"] = "操作失败";
            }
            return PartialView("Controls/Member/UserPrize.ascx");
        }

        /// <summary>
        ///修改用户信息 
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EditUserInfo(int UserID)
        {

            PersonUser u = Users.GetUser(UserID, true) as PersonUser;
            if (u == null)
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().UserNotFound()));
            return PartialView("Controls/Member/EditUser.ascx", u);
        }

        /// <summary>
        /// 更新用户信息
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_UpdateUserInfo(int UserID)
        {
            PersonUser u = Users.GetUser(UserID, true) as PersonUser;
            if (u != null)
            {
                u.EnableAvatar = Request.Form.GetBool("EnableAvatar", true);
                u.IsAvatarApproved = Request.Form.GetBool("IsAvatarApproved", true);
                u.EnableDisplayInMemberList = Request.Form.GetBool("InMember", true);
                u.IsModerated = Request.Form.GetBool("IsModerated", true);
                u.IsForceModerated = u.IsModerated;
                u.PublicEmail = Request.Form.GetString("publicEmail", u.PublicEmail);
                int i = Request.Form.GetInt("UserStatus", -1);
                if (i >= 0)
                {
                    UserAccountStatuses ua = (UserAccountStatuses)i;
                    if (ua == UserAccountStatuses.Banned)
                    {
                        int d = Request.Form.GetInt("UserBanTime", 0);
                        if (d > 0)
                        {
                            u.BannedUntil = DateTime.Now.AddDays(d);
                            u.BanReason = (UserBanReasons)Request.Form.GetInt("UserBanReasons", 0);
                        }
                        else
                        {
                            if (u.AccountStatus != UserAccountStatuses.Banned)
                            {
                                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                                TempData["StatusMessageContent"] = "操作失败，请选择一个封禁天数";
                                return RedirectToAction("Control_EditUserInfo", "ManageUser", new RouteValueDictionary { { "UserID", UserID } });
                            }
                        }
                    }
                    u.AccountStatus = ua;
                }
                try
                {
                    Users.UpdateUser(u);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "操作成功";
                }
                catch
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "操作失败";
                }
            }

            return RedirectToAction("Control_EditUserInfo", "ManageUser", new RouteValueDictionary { { "UserID", UserID } });
        }

        /// <summary>
        /// 修改密码
        /// </summary>
        public ActionResult Control_ResetPassword(int UserID)
        {
            User u = Users.GetUser(UserID, true);
            if (u == null)
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().UserNotFound()));
            ViewData["User"] = u;
            return PartialView("Controls/Member/ResetPassword.ascx");
        }

        /// <summary>
        /// 更新密码
        /// </summary>
        public ActionResult Control_UpdatePassword(int UserID)
        {
            User u = Users.GetUser(UserID, true);
            if (u == null)
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().UserNotFound()));
            string pw1 = Request.Form.GetString("pw1", string.Empty);
            string pw2 = Request.Form.GetString("pw2", string.Empty);
            if (pw1 == pw2)
            {
                if (!string.IsNullOrEmpty(pw1))
                {
                    try
                    {
                        //u.Password = string.Empty;
                        bool o = u.ChangePassword(string.Empty, pw1, true);
                        //bool o = u.ChangePassword(u.Password, pw1);
                        if (o)
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                            TempData["StatusMessageContent"] = "密码修改成功";
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "密码修改失败,请确认你有权限修改密码";
                        }
                    }
                    catch
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = "密码修改失败";
                    }
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "密码不能为空";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "两次密码不相同";
            }
            return RedirectToAction("Control_ResetPassword", "ManageUser", new RouteValueDictionary { { "UserID", UserID } });
        }

        /// <summary>
        /// 设置用户角色
        /// </summary>
        public ActionResult Control_SetUserRoles(int UserID)
        {
            ICollection<Role> AllRoles = Roles.GetAllRoles(true);

            AllRoles = (from Rl in AllRoles
                        where Rl.ConnectToUser == true
                        select new Role
                        {
                            ApplicationID = Rl.ApplicationID,
                            ConnectToUser = Rl.ConnectToUser,
                            Description = Rl.Description,
                            Enabled = Rl.Enabled,
                            FriendlyRoleName = Rl.FriendlyRoleName,
                            IsBuiltIn = Rl.IsBuiltIn,
                            RoleID = Rl.RoleID,
                            RoleName = Rl.RoleName
                        }).ToList<Role>();

            ICollection<Role> UserRoles = Roles.GetRoles(UserID, true);
            ViewData["AllRoles"] = AllRoles;
            ViewData["UserRoles"] = UserRoles;
            ViewData["UserID"] = UserID;
            return PartialView("Controls/Member/UserRoles.ascx");
        }

        /// <summary>
        /// 更新用户角色
        /// </summary>
        public ActionResult Control_UpdateUserRoles(int UserID)
        {
            string strRoles = Request.Form.GetString("RoleName", string.Empty);
            //if (!string.IsNullOrEmpty(strRoles))
            //{
            string[] strRolesArry = strRoles.Split(',');
            Roles.RemoveUserFromAllRoles(UserID);
            Roles.AddUserToRoles(UserID, strRolesArry);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "角色更新成功";
            //}
            //else
            //{
            //    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            //    TempData["StatusMessageContent"] = "角色更新失败";
            //}

            return RedirectToAction("Control_SetUserRoles", "ManageUser", new RouteValueDictionary { { "UserID", UserID } });
        }

        #endregion

        #region 添加新用户
        /// <summary>
        /// 显示添加新用户页面
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult NewUser()
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            ViewData["UserNameMinLength"] = settings.UserNameMinLength;
            ViewData["UserNameMaxLength"] = settings.UserNameMaxLength;
            ViewData["UserNameRegex"] = settings.UserNameRegex;
            ViewData["PasswordRegex"] = settings.PasswordRegex;
            return View("Pages/Member/EditUser.aspx", "Masters/User.master");
        }
        /// <summary>
        /// 处理添加新用户
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateUser()
        {
            string errorMessage = "";
            string userName = Request.Form.GetString("userName", string.Empty).Trim();
            string password = Request.Form.GetString("password", string.Empty);
            if (!Globals.ValidateUserName(userName, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                return RedirectToAction("NewUser");
            }
            if (!Globals.ValidatePassword(password, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                return RedirectToAction("NewUser");
            }
            string email = Request.Form.GetString("email", string.Empty);
            string commonName = Request.Form.GetString("commonName", string.Empty).Trim();
            PersonUser user = new PersonUser();

            user.UserName = userName;
            user.Password = password;
            user.IsModerated = Request.Form.GetBool("IsModerated", false);
            user.PrivateEmail = email;
            user.CommonName = commonName;
            user.PublicEmail = user.PrivateEmail;
            CreateUpdateUserStatuses status = CreateUpdateUserStatuses.UnknownFailure;
            AccountActivations activation = SiteSettingsManager.GetSiteSettings().AccountActivation;
            // Does the user require approval?
            if (activation == AccountActivations.AdminApproval)
                user.AccountStatus = UserAccountStatuses.ApprovalPending;
            else
                user.AccountStatus = UserAccountStatuses.Approved;

            if (user.UserName.Equals("Anonymous", StringComparison.CurrentCultureIgnoreCase))
            {
                status = CreateUpdateUserStatuses.DuplicateUsername;
            }
            else
            {
                try
                {
                    status = Users.Create(user, true);
                }
                catch
                {
                    //status = exception.;
                }
            }

            // Determine if the account was created successfully
            switch (status)
            {
                // Username already exists!
                case CreateUpdateUserStatuses.DuplicateUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，您输入的账号已被注册，请输入其他名称";
                    break;
                case CreateUpdateUserStatuses.BeyondLicensedCount:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，网站的注册人数超过了授权数量";
                    break;

                // Email already exists!
                case CreateUpdateUserStatuses.DuplicateEmailAddress:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，您输入的电子邮箱地址已经被使用。";
                    break;

                // Unknown failure has occurred!
                case CreateUpdateUserStatuses.UnknownFailure:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，注册新用户的时候产生了一个未知错误";
                    break;

                // Username is disallowed
                case CreateUpdateUserStatuses.DisallowedUsername:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "对不起，您输入的账号禁止使用，请输入其他名称";
                    break;

                case CreateUpdateUserStatuses.InvalidQuestionAnswer:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "不正确的密码问题和答案";
                    break;

                case CreateUpdateUserStatuses.InvalidPassword:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "您的密码长度少于本站要求的最小密码长度，请重新输入";
                    break;

                // Everything went off fine, good
                case CreateUpdateUserStatuses.Created:
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "添加新用户操作已成功";
                    break;
            }

            TempData["userName"] = userName;
            return RedirectToAction("NewUser");
        }

        #endregion

        #region 等级管理

        /// <summary>
        /// 等级列表
        /// </summary>
        public ActionResult ListUserRanks()
        {
            SetPageTitle("用户等级管理");
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            ViewData["settings"] = settings;
            SortedList<int, UserRank> sortlistUserRank = UserRanks.GetAllUserRanks(true);
            ViewData["sortlistUserRank"] = sortlistUserRank;
            return View("Pages/Member/ManageUserRanks.aspx", "Masters/User.master");
        }

        /// <summary>
        /// 重设等级
        /// </summary>
        public ActionResult ResetUserRanks()
        {
            UserRanks.ResetRankOfAllUser();
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "重设所有用户级别成功";
            return RedirectToAction("ListUserRanks", "ManageUser");
        }

        /// <summary>
        /// 删除等级
        /// </summary>
        public ActionResult DeleteUserRanks(int Rank)
        {
            UserRanks.DeleteUserRank(Rank);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除等级成功";
            return RedirectToAction("ListUserRanks", "ManageUser");
        }

        /// <summary>
        /// 添加一个等级
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_AddUserRank()
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            ViewData["settings"] = settings;
            ViewData["action"] = "add";
            return PartialView("Controls/Member/EditOrAddUserRank.ascx");
        }

        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_AddUserRankAction()
        {
            int Rank = Request.Form.GetInt("Rank", -1);
            int PointLower = Request.Form.GetInt("PointLower", -1);
            int PointUpper = Request.Form.GetInt("PointUpper", -1);
            if (Rank < 0 || PointLower < 0 || PointUpper < 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "等级级数、等级下限、等级上限必须为正整数";
                return RedirectToAction("Control_AddUserRank", "ManageUser");
            }
            if (PointUpper <= PointLower)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "等级上限必须大于等级下限";
                return RedirectToAction("Control_AddUserRank", "ManageUser");
            }
            UserRank ur = new UserRank();
            ur.Rank = Rank;
            ur.PointLower = PointLower;
            ur.PointUpper = PointUpper;
            ur.TitleOfUserRankType1 = Request.Form.GetString("TitleOfUserRankType1", string.Empty).Trim();
            ur.TitleOfUserRankType2 = Request.Form.GetString("TitleOfUserRankType2", string.Empty).Trim();
            ur.TitleOfUserRankType3 = Request.Form.GetString("TitleOfUserRankType3", string.Empty).Trim();
            ur.TitleOfUserRankType4 = Request.Form.GetString("TitleOfUserRankType4", string.Empty).Trim();
            ur.TitleOfUserRankType5 = Request.Form.GetString("TitleOfUserRankType5", string.Empty).Trim();
            ur.TitleOfUserRankType6 = Request.Form.GetString("TitleOfUserRankType6", string.Empty).Trim();
            ur.TitleOfUserRankType7 = Request.Form.GetString("TitleOfUserRankType7", string.Empty).Trim();
            ur.TitleOfUserRankType8 = Request.Form.GetString("TitleOfUserRankType8", string.Empty).Trim();
            ur.TitleOfUserRankType9 = Request.Form.GetString("TitleOfUserRankType9", string.Empty).Trim();
            ur.TitleOfUserRankType10 = Request.Form.GetString("TitleOfUserRankType10", string.Empty).Trim();
            try
            {
                int status = UserRanks.CreateUserRank(ur);
                if (status < 0)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "不能重复添加等级,请修改等级级数";
                    return RedirectToAction("Control_AddUserRank", "ManageUser");
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "添加等级成功";
            }
            catch
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "添加等级失败";
            }

            return RedirectToAction("Control_AddUserRank", "ManageUser");
        }

        /// <summary>
        /// 修改用户等级
        /// </summary>

        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EditUserRank(int ranks)
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            ViewData["settings"] = settings;
            ViewData["action"] = "edit";
            UserRank ur = UserRanks.GetUserRank(ranks, true);
            ViewData["userRank"] = ur;
            ViewData["rank"] = ranks;
            return PartialView("Controls/Member/EditOrAddUserRank.ascx");
        }
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Control_EditUserRankAction(int ranks)
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            ViewData["settings"] = settings;
            ViewData["action"] = "edit";
            UserRank ur = UserRanks.GetUserRank(ranks, true);
            int Rank = Request.Form.GetInt("Rank", -1);
            int PointLower = Request.Form.GetInt("PointLower", -1);
            int PointUpper = Request.Form.GetInt("PointUpper", -1);
            if (Rank < 0 || PointLower < 0 || PointUpper < 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "等级级数、等级下限、等级上限必须为正整数";
                return RedirectToAction("Control_EditUserRank", "ManageUser", new RouteValueDictionary { { "ranks", ranks } });
            }
            if (PointUpper <= PointLower)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "等级上限必须大于等级下限";
                return RedirectToAction("Control_EditUserRank", "ManageUser", new RouteValueDictionary { { "ranks", ranks } });
            }
            ur.Rank = Rank;
            ur.PointLower = PointLower;
            ur.PointUpper = PointUpper;
            ur.TitleOfUserRankType1 = Request.Form.GetString("TitleOfUserRankType1", string.Empty).Trim();
            ur.TitleOfUserRankType2 = Request.Form.GetString("TitleOfUserRankType2", string.Empty).Trim();
            ur.TitleOfUserRankType3 = Request.Form.GetString("TitleOfUserRankType3", string.Empty).Trim();
            ur.TitleOfUserRankType4 = Request.Form.GetString("TitleOfUserRankType4", string.Empty).Trim();
            ur.TitleOfUserRankType5 = Request.Form.GetString("TitleOfUserRankType5", string.Empty).Trim();
            ur.TitleOfUserRankType6 = Request.Form.GetString("TitleOfUserRankType6", string.Empty).Trim();
            ur.TitleOfUserRankType7 = Request.Form.GetString("TitleOfUserRankType7", string.Empty).Trim();
            ur.TitleOfUserRankType8 = Request.Form.GetString("TitleOfUserRankType8", string.Empty).Trim();
            ur.TitleOfUserRankType9 = Request.Form.GetString("TitleOfUserRankType9", string.Empty).Trim();
            ur.TitleOfUserRankType10 = Request.Form.GetString("TitleOfUserRankType10", string.Empty).Trim();
            try
            {
                UserRanks.UpdateUserRank(ur);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "修改等级成功";
            }
            catch
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "修改等级失败";
            }

            return RedirectToAction("Control_EditUserRank", "ManageUser", new RouteValueDictionary { { "ranks", ranks } });
        }

        #endregion

        #region 等级设置

        public ActionResult UserRankTypes()
        {
            SetPageTitle("用户等级设置");
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            ViewData["settings"] = settings;
            return View("Pages/Member/ManageUserRankTypes.aspx", "Masters/User.master");
        }

        public ActionResult UpdateUserRankTypes()
        {
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            settings.NameOfUserRankTypes[0] = Request.Form.GetString("UserRankType1", string.Empty).Trim();
            settings.NameOfUserRankTypes[1] = Request.Form.GetString("UserRankType2", string.Empty).Trim();
            settings.NameOfUserRankTypes[2] = Request.Form.GetString("UserRankType3", string.Empty).Trim();
            settings.NameOfUserRankTypes[3] = Request.Form.GetString("UserRankType4", string.Empty).Trim();
            settings.NameOfUserRankTypes[4] = Request.Form.GetString("UserRankType5", string.Empty).Trim();
            settings.NameOfUserRankTypes[5] = Request.Form.GetString("UserRankType6", string.Empty).Trim();
            settings.NameOfUserRankTypes[6] = Request.Form.GetString("UserRankType7", string.Empty).Trim();
            settings.NameOfUserRankTypes[7] = Request.Form.GetString("UserRankType8", string.Empty).Trim();
            settings.NameOfUserRankTypes[8] = Request.Form.GetString("UserRankType9", string.Empty).Trim();
            settings.NameOfUserRankTypes[9] = Request.Form.GetString("UserRankType10", string.Empty).Trim();
            settings.EnableSelectUserRankType = Request.Form.GetBool("EnableSelectUserRankType", settings.EnableSelectUserRankType);
            if (settings.EnableSelectUserRankType)
            {
                settings.UserRankTypesForUserSelect = Request.Form.GetString("userSelecteUserRanks2", string.Empty);
            }
            else
            {
                int ui = Request.Form.GetInt("userSelecteUserRanks", 1);
                settings.DefaultUserRankType = (SpaceBuilder.Common.UserRankTypes)ui;
            }
            SiteSettingsManager.Save(settings);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "等级设置成功";
            return RedirectToAction("UserRankTypes", "ManageUser");
        }
        #endregion

        #region 留言管理
        /// <summary>
        /// 显示留言信息
        /// </summary>
        /// <param name="status"></param>
        /// <param name="pageIndex"></param>

        public ActionResult ListGuestBook(AuditingStatusesForDisplay? status, int? pageIndex)
        {
            SetPageTitle("空间留言管理");
            int authorUserID = Request.QueryString.GetInt("PostUserID", -1);
            PagingDataSet<GuestbookPost> pdsBookPost = UserGuestbookManager.Instance().GetGuestbookPostsForAdmin(17, pageIndex ?? 1, status ?? AuditingStatusesForDisplay.Pending, authorUserID);
            ViewData["pdsBookPost"] = pdsBookPost;
            ViewData["status"] = status;
            return View("Pages/Member/ManageUserGuestBook.aspx", "Masters/User.master");
        }
        /// <summary>
        /// 更新留言状态
        /// </summary>
        public ActionResult SetAuditingStatuses(string returnUrl, AuditingStatuses statues)
        {
            string selectedPostIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedPostIDsString))
            {
                string[] selectedPostIDsStringArray = selectedPostIDsString.Split(',');
                if (selectedPostIDsStringArray != null && selectedPostIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListPostID = new List<int>();
                    foreach (string Id in selectedPostIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListPostID.Add(threadID);
                        }
                        catch { }
                    }

                    try
                    {
                        UserGuestbookManager.Instance().UpdatePostAuditingStatus(ListPostID, statues);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = "留言批量更新成功";
                    }
                    catch
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = "留言批量更新失败";
                    }
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的留言";
            }
            return Redirect(returnUrl);
        }

        public ActionResult DeleteGuestBookPost(string returnUrl)
        {
            string selectedPostIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedPostIDsString))
            {
                string[] selectedPostIDsStringArray = selectedPostIDsString.Split(',');
                if (selectedPostIDsStringArray != null && selectedPostIDsStringArray.Length > 0)
                {
                    int threadID;
                    List<int> ListPostID = new List<int>();
                    foreach (string Id in selectedPostIDsStringArray)
                    {
                        try
                        {
                            threadID = int.Parse(Id);
                            ListPostID.Add(threadID);
                        }
                        catch { }
                    }

                    try
                    {
                        UserGuestbookManager.Instance().Delete(ListPostID);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = "留言批量删除成功";
                    }
                    catch
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = "留言批量删除失败";
                    }
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的留言";
            }
            return Redirect(returnUrl);
        }
        #endregion

        #region 角色管理
        /// <summary>
        /// 管理用户角色
        /// </summary>
        public ActionResult ManageUserRoles()
        {
            SetPageTitle("用户角色管理");

            ICollection<Role> AllRoles = Roles.GetAllRoles(true);
            return View("Pages/Member/ManageUserRoles.aspx", "Masters/User.master", AllRoles);
        }
        /// <summary>
        /// 编辑用户角色
        /// </summary>
        public ActionResult Control_EditUserRole(int? roleID)
        {
            Role role = null;
            if (roleID.HasValue && roleID.Value > 0)
            {
                role = Roles.GetRole(roleID.Value, true);
            }
            else
            {
                if (TempData["role"] != null)
                    role = (Role)TempData["role"];
            }
            if (role == null)
                role = new Role();
            return PartialView("Controls/Member/EditUserRole.ascx", role);
        }
        /// <summary>
        /// 创建用户角色
        /// </summary>
        public ActionResult CreateUserRole()
        {
            Role role = new Role();
            role.RoleName = Request.Form.GetString("roleName", string.Empty);
            role.FriendlyRoleName = Request.Form.GetString("friendlyRoleName", string.Empty);
            role.Description = Request.Form.GetString("description", string.Empty);
            role.IsBuiltIn = false;
            role.Enabled = Request.Form.GetBool("enabled", false);
            role.IsPublic = Request.Form.GetBool("isPublic", false);

            role.ConnectToUser = true;
            if (string.IsNullOrEmpty(role.RoleName))
            {
                TempData["role"] = role;
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "创建用户角色失败，角色名称不能为空";
                return RedirectToAction("Control_EditUserRole");
            }
            if (Roles.GetRole(role.RoleName, true) != null)
            {
                TempData["role"] = role;
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "创建用户角色失败，角色名称已经被使用，请换另外一个再试";
                return RedirectToAction("Control_EditUserRole");
            }
            Roles.CreateRole(role);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "创建用户角色成功";
            return RedirectToAction("Control_EditUserRole", new RouteValueDictionary { { "roleID", role.RoleID } });
        }
        /// <summary>
        /// 更新用户角色
        /// </summary>
        public ActionResult UpdateUserRole(int roleID)
        {
            Role role = Roles.GetRole(roleID, true);
            if (role == null)
                throw new ArgumentException("没有找到你要更新的角色");
            role.FriendlyRoleName = Request.Form.GetString("friendlyRoleName", string.Empty);
            role.Description = Request.Form.GetString("description", string.Empty);
            role.Enabled = Request.Form.GetBool("enabled", false);
            role.IsPublic = Request.Form.GetBool("isPublic", false);
            Roles.UpdateRole(role);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "更新用户角色成功";
            return RedirectToAction("Control_EditUserRole", new RouteValueDictionary { { "roleID", roleID } });
        }
        /// <summary>
        /// 删除用户角色
        /// </summary>
        public ActionResult DeleteUserRole(int roleID)
        {
            Role role = Roles.GetRole(roleID, true);
            if (role == null)
                throw new ArgumentException("没有找到你要删除的角色");
            Roles.DeleteRole(roleID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除用户角色成功";
            return RedirectToAction("ManageUserRoles");
        }

        #endregion

        #region 认证标识管理
        /// <summary>
        /// 获取全部身份标识
        /// </summary>

        public ActionResult ManageIdentities()
        {
            SetPageTitle("认证标识管理");
            Dictionary<int, Identity> identities = Identities.GetAllIdentities();
            return View("Pages/Member/ManageIdentities.aspx", "Masters/User.master", identities);
        }

        /// <summary>
        /// 编辑认证标识
        /// </summary>        
        public ActionResult Control_EditIdentity(int? identityID)
        {
            Identity identity = new Identity();
            if (identityID.HasValue && identityID.Value > 0)
            {
                identity = Identities.GetIdentity(identityID.Value);
            }
            if (TempData["identity"] != null)
                identity = TempData.Get<Identity>("identity", new Identity());
            return PartialView("Controls/Member/EditIdentity.ascx", identity);
        }

        /// <summary>
        /// 创建、更新认证标识
        /// </summary>        
        [HttpPost]
        public ActionResult CreateUpdateIdentity(int? identityID)
        {
            Identity identity = null;
            if (identityID.HasValue && identityID.Value > 0)
            {
                identity = Identities.GetIdentity(identityID.Value);
            }
            else
            {

                identity = new Identity();
            }
            identity.Name = Formatter.FormatSingleLinePlainTextForStorage(Request.Form.GetString("IdentityName", string.Empty));
            identity.Description = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("Description", string.Empty), true);

            HttpPostedFileBase bigIconFile = null, smallIconFile = null;

            #region 验证所选文件
            if (Request.Files != null && Request.Files.Count > 0)
            {
                smallIconFile = Request.Files[0];
                if (!identityID.HasValue || identityID.Value <= 0)
                {
                    if (smallIconFile == null || smallIconFile.ContentLength <= 0)
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "必须同时选择大小标识图片");
                        TempData["identity"] = identity;
                        return RedirectToAction("Control_EditIdentity", new { identityID = identityID });
                    }
                }
                if (!identityID.HasValue || identityID.Value <= 0)
                {
                    if (smallIconFile.ContentType.IndexOf("image") == -1)
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请选择图片格式的文件");
                        TempData["identity"] = identity;
                        return RedirectToAction("Control_EditIdentity", new { identityID = identityID });
                    }
                }
                if (Request.Files.Count > 1)
                {
                    bigIconFile = Request.Files[1];
                    if (!identityID.HasValue || identityID.Value <= 0)
                    {

                        if (bigIconFile == null || bigIconFile.ContentLength <= 0)
                        {
                            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "必须同时选择大小标识图片");
                            TempData["identity"] = identity;
                            return RedirectToAction("Control_EditIdentity", new { identityID = identityID });
                        }
                    }
                    if (!identityID.HasValue || identityID.Value <= 0)
                    {
                        if (bigIconFile.ContentType.IndexOf("image") == -1)
                        {
                            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "请选择图片格式的文件");
                            TempData["identity"] = identity;
                            return RedirectToAction("Control_EditIdentity", new { identityID = identityID });
                        }
                    }
                }
            }
            #endregion


            if (identityID.HasValue && identityID.Value > 0)
            {
                Identities.UpdateIdentity(identity, bigIconFile, smallIconFile);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "修改标识认证成功");
            }
            else
            {
                Identities.CreateIdentity(identity, bigIconFile, smallIconFile);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "创建标识认证成功");
            }
            return RedirectToAction("Control_EditIdentity", new { identityID = identityID });
        }
        /// <summary>
        /// 删除认证
        /// </summary>        
        public ActionResult DeleteIdentity(int identityID)
        {
            Identity identity = Identities.GetIdentity(identityID);
            if (identity == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("不存在ID为{0}的认证标识", identityID);
                return Redirect(Request.UrlReferrer.AbsoluteUri);

            }
            Identities.DeleteIdentity(identityID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除认证标识成功";
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }

        /// <summary>
        /// 编辑用户认证标识
        /// </summary>       
        [HttpGet]
        public ActionResult Control_EditUserIdentity(int UserID)
        {
            UserIdentity userIdentity = null;
            Dictionary<int, Identity> AllIdentities = Identities.GetAllIdentities();
            if (UserID > 0)
            {
                userIdentity = Identities.GetUserIdentity(UserID, true);
            }
            if (TempData["userIdentity"] != null)
                userIdentity = TempData.Get<UserIdentity>("userIdentity", new UserIdentity());
            if (userIdentity == null)
                userIdentity = new UserIdentity();
            ViewData["AllIdentities"] = AllIdentities;
            return PartialView("Controls/Member/EditUserIdentity.ascx", userIdentity);
        }
        /// <summary>
        /// 创建、更新用户认证标识
        /// </summary>
        [HttpPost]
        public ActionResult CreateUpdateUserIdentity(int UserID)
        {
            UserIdentity userIdentity = Identities.GetUserIdentity(UserID, true);
            if (userIdentity == null)
            {
                userIdentity = new UserIdentity();
                userIdentity.UserID = UserID;
                userIdentity.IdentityID = Request.Form.GetInt("identityID", 0);
                userIdentity.IdentityDescription = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("Description", string.Empty), true);
                Identities.CreateUserIdentity(userIdentity);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "添加认证标识成功");
            }
            else
            {
                userIdentity.IdentityID = Request.Form.GetInt("identityID", 0);
                userIdentity.IdentityDescription = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("Description", string.Empty), true);
                Identities.UpdateUserIdentity(userIdentity);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "编辑认证标识成功");
            }
            return RedirectToAction("Control_EditUserIdentity", new RouteValueDictionary { { "UserID", UserID } });

        }
        /// <summary>
        /// 取消用户认证
        /// </summary>        
        public ActionResult DeleteUserIdentity(int userID)
        {
            UserIdentity identity = Identities.GetUserIdentity(userID, true);
            if (identity == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = string.Format("ID为{0}的用户没有设置为认证", userID);
                return Redirect(Request.UrlReferrer.AbsoluteUri);

            }
            Identities.DeleteUserIdentity(identity);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "成功取消用户认证";
            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }
        #endregion


    }

    #region UserSubMenu

    public enum UserSubMenu
    {
        /// <summary>
        /// 个人用户管理
        /// </summary>
        ManagePersonUsers = 1601,

        /// <summary>
        /// 企业用户管理
        /// </summary>
        ManageCompanyUsers = 1603,

        /// <summary>
        /// 添加用户
        /// </summary>
        EditUser = 1602,

        /// <summary>
        /// 等级管理
        /// </summary>
        ManageUserRanks = 1611,

        ///// <summary>
        ///// 等级设置
        ///// </summary>
        //UserRankSettings = 1612,

        /// <summary>
        /// 积分设置
        /// </summary>
        UserPointSettings = 1616,

        /// <summary>
        /// 积分规则
        /// </summary>
        UserPointRules = 1617,

        /// <summary>
        /// 用户留言
        /// </summary>
        ManageGuestBook = 1619,

        /// <summary>
        /// 管理用户角色
        /// </summary>
        ManageUserRoles = 1620,
        /// <summary>
        /// 管理身份标识
        /// </summary>
        ManageIdentities = 1621
    }

    #endregion
}
