﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------ 
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Web;
using SpaceBuilder.Club;
using SpaceBuilder.Utils;
using System.Text.RegularExpressions;
using System.Web.Routing;
using System.Net.Mail;
using System.Web.UI;
using MSNPSharp;

namespace SpaceBuilder.Common.Controllers
{
    /// <summary>
    /// 用户空间中的信息中心的Controller
    /// </summary>
    [HttpCompress]
    public class MessageCenterController : UserDomainControllerBase
    {

        private int applicationID = ApplicationIDs.Instance().User();
        #region 私信
        #region 页面
        /// <summary>
        /// 收件箱
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMessageInBox(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_MessageInBox", applicationID));
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/MessageCenters/ListMessageInBox.aspx", "Masters/MessageCenter.master");
        }
        /// <summary>
        /// 发件箱
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMessageOutBox(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_MessageOutBox", applicationID));
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/MessageCenters/ListMessageOutBox.aspx", "Masters/MessageCenter.master");
        }

        /// <summary>
        /// 
        /// 显示新私信页
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult NewMessage(string userDomainName, int? toUserID)
        {
            //string privateMessageTypeTitle = ResourceManager.GetString(string.Format("PrivateMessageType:{0}", PrivateMessageType.Common.ToString()));
            SetPageTitle(userDomainName, GetResourceString("Title_SendMessage", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            if (toUserID.HasValue && toUserID.Value == (int)PrivateMessageEspecialUserID.Serve)
            {
                ViewData["sendToServe"] = true;
                return View("Pages/MessageCenters/NewMessage.aspx", "Masters/MessageCenter.master");
            }



            //给某会员发普通消息
            if (toUserID.HasValue && toUserID.Value > 0)
            {
                User user = Users.GetUser(toUserID.Value, false);

                if (user != null)
                {
                    //判断该会员是否允许当前用户向其发私信
                    if (Privacys.Validate(user.UserID, CurrentUser, PrivacyItemKeys.Instance().Action_ReceivePrivateMessage()))
                    {
                        ViewData["IsAllowed"] = true;
                        ViewData["ToUser"] = user;
                    }
                    else
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Message_StopSendMessageForPrivacy", applicationID);
                    }
                }

                ViewData["IsMultiple"] = false;
            }
            else
            {
                if (currentDomainUser.UserType == UserTypes.CompanyUser)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_StopForCompanySendToMorePerson", applicationID);
                }

                ViewData["IsMultiple"] = true;
            }

            return View("Pages/MessageCenters/NewMessage.aspx", "Masters/MessageCenter.master");
        }
        /// <summary>
        /// 显示私信详细页
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowMessage(string userDomainName, int threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_ShowMessage", applicationID));
            return View("Pages/MessageCenters/ShowMessage.aspx", "Masters/MessageCenter.master");
        }

        #endregion

        #region 控件
        /// <summary>
        /// 私信菜单列表控件
        /// </summary>
        public ActionResult Control_PrivateMessageMenu(string userDomainName, PrivateMessageMenuType menuType)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());
            ViewData["NoReadCount"] = PrivateMessages.GetNoReadCount(currentDomainUser.UserID);
            ViewData["PrivateMessageMenuType"] = menuType;
            return PartialView("Controls/MessageCenters/PrivateMessageMenu.ascx");
        }
        /// <summary>
        /// 收件箱信件列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListMessageInBox(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ViewData["NoReadCount"] = PrivateMessages.GetNoReadCount(currentDomainUser.UserID);
            PagingDataSet<PrivateMessageInboxItem> pds = PrivateMessages.GetPrivateMessagesFromInbox(currentDomainUser.UserID, null, null, pageIndex ?? 1);
            return PartialView("Controls/MessageCenters/ListMessageInBox.ascx", pds);
        }
        /// <summary>
        /// 发件箱信件列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListMessageOutBox(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<PrivateMessageOutboxItem> pds = PrivateMessages.GetPrivateMessagesFromOutbox(currentDomainUser.UserID, null, pageIndex ?? 1);
            return PartialView("Controls/MessageCenters/ListMessageOutBox.ascx", pds);
        }
        /// <summary>
        /// 显示私信内容控件
        /// </summary>   
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ShowMessage(string userDomainName, int threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PrivateMessage message = PrivateMessages.GetPrivateMessage(threadID);
            if (CurrentUser != null && !PrivateMessages.HasOperationPermission(message, CurrentUser))
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied(), GetResourceString("Exception_ViewMessageAccessDenied", applicationID));
            }
            //设置私信为已读
            PrivateMessages.SetReadFlag(CurrentUser != null ? CurrentUser.UserID : 0, message.ThreadID, true);
            return PartialView("Controls/MessageCenters/ShowMessage.ascx", message);
        }
        #endregion

        #region 操作
        /// <summary>
        /// 从收件箱中删除一条私信
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteInBoxMessage(string userDomainName, int threadID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PrivateMessages.DeletePrivateMessageFromInbox(currentDomainUser.UserID, threadID);
            return RedirectToAction("Control_ListMessageInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 从发件箱中删除一条私信
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteOutBoxMessage(string userDomainName, int threadID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PrivateMessages.DeletePrivateMessageFromOutbox(currentDomainUser.UserID, threadID);
            return RedirectToAction("Control_ListMessageOutBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 批量删除收件箱中的私信
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteInBoxMessages(string userDomainName)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string selectedIDsString = Request.Form["MessageInBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int threadID = int.Parse(IDString);
                        if (threadID > 0)
                            PrivateMessages.DeletePrivateMessageFromInbox(currentDomainUser.UserID, threadID);
                    }
                }
            }
            return RedirectToAction("Control_ListMessageInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", 1 } });
        }
        /// <summary>
        /// 批量删除发件箱中的私信
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteOutBoxMessages(string userDomainName)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string selectedIDsString = Request.Form["MessageOutBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int threadID = int.Parse(IDString);
                        if (threadID > 0)
                            PrivateMessages.DeletePrivateMessageFromOutbox(currentDomainUser.UserID, threadID);
                    }
                }
            }
            return RedirectToAction("Control_ListMessageOutBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", 1 } });
        }

        /// <summary>
        /// 设置私信的阅读状态
        /// </summary>    
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetMessageReadStatus(string userDomainName, bool isread, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string selectedIDsString = Request.Form["MessageInBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    List<int> threadIDs = new List<int>();
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int threadID = int.Parse(IDString);
                        if (threadID > 0)
                            threadIDs.Add(threadID);
                    }
                    PrivateMessages.SetReadFlag(currentDomainUser.UserID, threadIDs, isread);
                }
            }
            return RedirectToAction("Control_ListMessageInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 处理新私信
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateMessage(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string[] receivers;
            string receiverIDs = Request.Form["receiverIDs"];
            string subject = Request.Form.GetString("subject", string.Empty);
            string body = Request.Form.GetString("body", string.Empty);
            if (string.IsNullOrEmpty(receiverIDs) || string.IsNullOrEmpty(subject) || string.IsNullOrEmpty(body))
            {
                TempData["subject"] = subject;
                TempData["body"] = body;
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (string.IsNullOrEmpty(receiverIDs))
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForReceiverEmpty", applicationID);
                else if (string.IsNullOrEmpty(subject))
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForSubjectEmpty", applicationID);
                else
                    TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForBodyEmpty", applicationID);
                return RedirectToAction("NewMessage", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }

            //收件人集合
            receivers = receiverIDs.Trim(',').Split(',');

            //每个消息最多允许100个接收人
            if (receivers.Length > 100)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForBeyondMaxPersonCount", applicationID);
                return RedirectToAction("NewMessage");
            }
            //发送私信成功的接收人集合
            List<string> successReceivers = new List<string>();
            int toUserID = -1;

            if (int.TryParse(receiverIDs, out toUserID) && toUserID == (int)PrivateMessageEspecialUserID.Serve)
            {
                PrivateMessage pm = new PrivateMessage();
                pm.Subject = subject;
                pm.Body = body;
                pm.SenderUserID = CurrentUser.UserID;
                pm.Sender = CurrentUser.DisplayName;
                pm.MessageType = PrivateMessageType.Common;

                pm.ReceiverUserID = toUserID;
                pm.Receiver = "客服";

                #region 格式化处理
                pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(pm.Subject);
                pm.Body = Formatter.FormatMultiLinePlainTextForStorage(pm.Body, true);

                if (SPBConfig.Instance().EnableEmoticon)
                    pm.Body = EmotionManager.Instance().EmoticonTransforms(pm.Body);
                #endregion

                PrivateMessages.SendMessage(pm);
                successReceivers.Add(pm.Receiver);
            }



            //为每个收件人建立私信
            for (int i = 0; i < receivers.Length; i++)
            {
                //查找收件人
                User user = null;

                if (int.TryParse(receivers[i], out toUserID))
                {
                    user = Users.GetUser(toUserID, false);
                }

                if (user != null)
                {
                    //判断当前空间者是否接受访问者的私信
                    if (Privacys.Validate(user.UserID, CurrentUser, PrivacyItemKeys.Instance().Action_ReceivePrivateMessage()))
                    {
                        PrivateMessage pm = new PrivateMessage();
                        pm.Subject = subject;
                        pm.Body = body;
                        pm.SenderUserID = CurrentUser.UserID;
                        pm.Sender = CurrentUser.DisplayName;
                        pm.MessageType = PrivateMessageType.Common;

                        pm.ReceiverUserID = user.UserID;
                        pm.Receiver = user.DisplayName;

                        #region 格式化处理
                        pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(pm.Subject);
                        pm.Body = Formatter.FormatMultiLinePlainTextForStorage(pm.Body, true);

                        if (SPBConfig.Instance().EnableEmoticon)
                            pm.Body = EmotionManager.Instance().EmoticonTransforms(pm.Body);
                        #endregion

                        PrivateMessages.SendMessage(pm);
                        successReceivers.Add(user.DisplayName);
                        toUserID = user.UserID;
                    }
                }
            }
            //如果有成功发送的消息则显示提示信息
            if (successReceivers.Count > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                string strReceivers = string.Empty;
                foreach (string displayName in successReceivers)
                {
                    strReceivers += "," + displayName;
                }
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_SendSuccess", applicationID), strReceivers.Substring(1));
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_SendErrorForOther", applicationID);
            }
            //if (toUserID > 0 || toUserID == (int)PrivateMessageEspecialUserID.Serve)
            return RedirectToAction("NewMessage", new RouteValueDictionary { { "userDomainName", userDomainName }, { "toUserID", toUserID } });
            //else
            //    return RedirectToAction("NewMessage", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 回复消息
        /// </summary> 
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ReplyMessage(string userDomainName, int threadID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (threadID > 0)
            {
                PrivateMessage pm = PrivateMessages.GetPrivateMessage(threadID);
                string body = Request.Form["body"];
                if (!string.IsNullOrEmpty(body))
                {
                    PrivateMessagePost post = new PrivateMessagePost();
                    post.ThreadID = threadID;
                    post.ParentPostID = pm.PostID;
                    post.UserID = CurrentUser.UserID;
                    post.Author = CurrentUser.DisplayName;
                    post.Subject = string.Format("{0}:{1}", GetResourceString("Common_Reply"), pm.Subject);
                    post.Body = Formatter.FormatPlainTextComment(body);
                    if (SPBConfig.Instance().EnableEmoticon)
                        post.Body = EmotionManager.Instance().EmoticonTransforms(post.Body);

                    PrivateMessages.ReplyMessage(post, pm);
                }
            }
            return RedirectToAction("Control_ShowMessage", new RouteValueDictionary { { "userDomainName", userDomainName }, { "threadID", threadID } });
        }

        #endregion

        #endregion

        #region 通知
        /// <summary>
        /// 通知列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListNoticeRecords(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Action_Notice"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            return View("Pages/MessageCenters/ListNoticeRecords.aspx", "Masters/MessageCenter.master");
        }
        /// <summary>
        /// 显示通知列表控件
        /// </summary>
        public ActionResult Control_ListNoticeRecords(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<InstationNoticeRecord> pds = InstationNotices.GetInstationNoticeRecords(pageIndex ?? 1, currentDomainUser.UserID);
            InstationNotices.BatchSetIsRead(currentDomainUser.UserID, pageIndex ?? 1, pds.Records);
            return View("Controls/MessageCenters/ListNoticeRecords.ascx", pds);
        }

        #endregion

        #region 邀请
        #region 页面
        /// <summary>
        /// 收到的邀请显示页
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListInvitationInBox(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_InvitationInBox", applicationID));
            ViewData["PageIndex"] = pageIndex;
            ViewData["isDeal"] = Request.QueryString.GetBool("isDeal", false);
            return View("Pages/MessageCenters/PersonUsers/ListInvitationInBox.aspx", "Masters/MessageCenter.master", string.Empty);
        }
        /// <summary>
        /// 发出的邀请显示页
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListInvitationOutBox(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_InvitationOutBox", applicationID));
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/MessageCenters/PersonUsers/ListInvitationOutBox.aspx", "Masters/MessageCenter.master", string.Empty);
        }
        /// <summary>
        /// 加为好友
        /// </summary>       
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult NewFriendInvitation(string userDomainName, int toUserID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_NewFriendInvitation", applicationID));
            User user = Users.GetUser(toUserID, false);
            if (user != null)
            {
                if (user.UserID == currentDomainUser.UserID)
                {
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_InvitErrorForInvitSelf", applicationID));
                }
                else if (Friends.IsFriend(CurrentUser.UserID, toUserID))
                {
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_InvitErrorForRepeatInvit", applicationID));
                }
                else if (!Privacys.Validate(user.UserID, currentDomainUser, PrivacyItemKeys.Instance().Action_ReceiveFriendInvitation()))
                {
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_InvitErrorForPrivacy", applicationID));
                }
                else if (currentDomainUser.UserType == UserTypes.CompanyUser)
                {
                    ViewData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_InvitErrorForCompanyUser", applicationID));
                }
            }
            else
                return Redirect(SiteUrls.Instance().ShowExceptionMessage(ExceptionTypes.Instance().UserNotFound(), string.Format(GetResourceString("Exception_Pattern_UserNotFound"), toUserID)));
            return View("Controls/MessageCenters/PersonUsers/NewFriendInvitation.ascx", user);
        }

        /// <summary>
        /// 处理加为好友
        /// </summary>
        public ActionResult CreateFriendInvitation(string userDomainName, int toUserID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            User user = Users.GetUser(toUserID, false);
            if (user == null)
                return RedirectToAction("NewFriendInvitation", new RouteValueDictionary { { "userDomainName", userDomainName }, { "ToUserID", toUserID } });

            Invitation invitation = new Invitation();
            invitation.Subject = ResourceManager.GetString("Title_DefaultFriendInvitationSubject", user.Language, applicationID);
            invitation.Body = WebUtils.HtmlEncode(Request.Form.GetString("body", string.Empty));
            invitation.SenderUserID = CurrentUser.UserID;
            invitation.Sender = CurrentUser.DisplayName;
            invitation.Receiver = user.DisplayName;
            invitation.ReceiverUserID = user.UserID;
            invitation.ApplicationID = ApplicationIDs.Instance().Friend();

            CreateInvitationStatuses status = Invitations.CreateInvitation(invitation);
            if (status == CreateInvitationStatuses.Duplicate)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_InvitErrorForAlreadySendApply", applicationID));
            }
            else if (status == CreateInvitationStatuses.Success)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_InvitFriendSuccess", applicationID));
            }
            return RedirectToAction("NewFriendInvitation", new RouteValueDictionary { { "userDomainName", userDomainName }, { "ToUserID", toUserID } });
        }



        /// <summary>
        /// 通过Email向好友发邀请,可以向多个好友发邀请,同时可以接受默认值
        /// </summary>
        /// <param name="Tomail">向某个好友发Email邀请</param>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult NewFriendInvitationByEmail(string userDomainName, string Tomail)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser.UserType == UserTypes.CompanyUser)
                throw new SPBException(ExceptionTypes.Instance().AccessDenied(), GetResourceString("Exception_EmailinvitAccessDenied", applicationID));
            SetPageTitle(userDomainName, GetResourceString("Title_EmailNewInvitation", applicationID));
            ViewData["SiteName"] = SiteSettingsManager.GetSiteSettings().SiteName;
            InvitationTemplate template = InvitationTemplates.GetInvitationTemplate(ApplicationIDs.Instance().Friend());
            if (template != null)
            {
                ViewData["subject"] = InvitationTemplates.EmailFormatterForPersonUser(template.Subject, CurrentUser);
                string body = InvitationTemplates.EmailFormatterForPersonUser(template.Body, CurrentUser);
                ViewData["body"] = InvitationTemplates.EmailFormatterForBody(body, GetResourceString("Description_InvitationBodyPlaceHolderInTemplates", applicationID));
            }
            return View("Pages/MessageCenters/PersonUsers/NewFriendInvitationByEmail.aspx", "Masters/MessageCenter.master", string.Empty);
        }
        /// <summary>
        /// 处理Email邀请
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateFriendInvitationByEmail(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser.UserType == UserTypes.CompanyUser)
                throw new SPBException(ExceptionTypes.Instance().AccessDenied(), GetResourceString("Exception_EmailinvitAccessDenied", applicationID));
            string receivers = Request.Form.GetString("receivers", string.Empty);
            string[] emailAddress = receivers.Split(new Char[] { ',', ';', '，', '；', '\n', '\r' });
            List<string> validatedEmailAddress = new List<string>();
            List<string> unvalidatedEmailAddress = new List<string>();
            Regex emailRegex = new Regex(@"^\w+([-+.']\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$");

            for (int i = 0; i < emailAddress.Length; i++)
            {
                if (emailRegex.IsMatch(emailAddress[i]))
                    validatedEmailAddress.Add(emailAddress[i]);
                else
                    unvalidatedEmailAddress.Add(emailAddress[i]);
            }
            if (validatedEmailAddress.Count == 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_EmailInvitErrorForNoValidatedEmailAddress", applicationID);
                TempData["subject"] = Request.Form.GetString("subject", string.Empty);
                TempData["body"] = Request.Form.GetString("body", string.Empty);
                return RedirectToAction("NewFriendInvitationByEmail", new RouteValueDictionary { { "userDomainName", userDomainName }, { "Tomail", string.Empty } });
            }

            //发送Email成功的Email集合
            List<string> successEmails = new List<string>();
            //重复发送的Email集合
            List<string> duplicateEmails = new List<string>();
            string subject = string.Empty;
            string invitationBody = Request.Form.GetString("invitationBody", string.Empty);
            string body = string.Empty;
            InvitationTemplate template = InvitationTemplates.GetInvitationTemplate(ApplicationIDs.Instance().Friend());
            if (template != null)
            {
                subject = InvitationTemplates.EmailFormatterForPersonUser(template.Subject, CurrentUser);
                body = InvitationTemplates.EmailFormatterForPersonUser(template.Body, CurrentUser);
                body = InvitationTemplates.EmailFormatterForBody(body, invitationBody);
            }
            for (int i = 0; i < validatedEmailAddress.Count; i++)
            {
                Invitation invitation = new Invitation();
                invitation.Subject = subject;
                invitation.SenderUserID = CurrentUser.UserID;
                invitation.Sender = CurrentUser.DisplayName;
                invitation.Body = invitationBody;
                invitation.SenderEmail = string.IsNullOrEmpty(CurrentUser.PublicEmail) ? CurrentUser.PrivateEmail : CurrentUser.PublicEmail;
                invitation.Receiver = (string)validatedEmailAddress[i];
                invitation.ReceiverEmail = (string)validatedEmailAddress[i];
                invitation.ApplicationID = ApplicationIDs.Instance().Friend();
                invitation.TransportMode = InvitationTransportModes.Email;
                SendEmail(invitation, body);
                CreateInvitationStatuses status = Invitations.CreateInvitation(invitation);
                if (status == CreateInvitationStatuses.Duplicate)
                    duplicateEmails.Add((string)validatedEmailAddress[i]);
                else
                    successEmails.Add((string)validatedEmailAddress[i]);
            }
            string strEmails = string.Empty;
            if (successEmails.Count > 0)
            {
                foreach (string email in successEmails)
                {
                    strEmails += "," + email;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitSendSuccess", applicationID), strEmails.Substring(1));

                string strDuplicateEmails = string.Empty;
                if (duplicateEmails.Count > 0)
                {
                    foreach (string dispalyname in duplicateEmails)
                    {
                        strDuplicateEmails += "," + dispalyname;
                    }
                    TempData["StatusMessageContent"] += string.Format(GetResourceString("Message_Pattern_EmailInvitForRepeatInvit", applicationID), strDuplicateEmails.Substring(1));
                }
            }
            else if (duplicateEmails.Count > 0)
            {
                string strDuplicateEmails = string.Empty;

                foreach (string dispalyname in duplicateEmails)
                {
                    strDuplicateEmails += "," + dispalyname;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitForRepeatInvit", applicationID), strDuplicateEmails.Substring(1));
            }
            //提示未通过验证的Email地址
            if (unvalidatedEmailAddress.Count > 0)
            {
                string strUnvalidatedEmailAddress = string.Empty;
                foreach (string address in unvalidatedEmailAddress)
                {
                    strUnvalidatedEmailAddress += "," + address;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitForNoValidatedEmailAddress", applicationID), strUnvalidatedEmailAddress.Substring(1));
                TempData["invitationBody"] = invitationBody;
            }
            return RedirectToAction("NewFriendInvitationByEmail", new RouteValueDictionary { { "userDomainName", userDomainName }, { "Tomail", string.Empty } });
        }


        #region MSN邀请好友

        /// <summary>
        /// 处理邀请MSN好友
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult InvitationMSNFriends(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string msnUserName = Request.Form.GetString("msnUserName", string.Empty);
            string msnPassword = Request.Form.GetString("msnPassword", string.Empty);
            if (string.IsNullOrEmpty(msnUserName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_MsnInvitErrorForAccountEmpty", applicationID);
                TempData["msnUserName"] = msnUserName;
                TempData["msnPassword"] = msnPassword;
                return RedirectToAction("NewFriendInvitationByEmail", new RouteValueDictionary() { { "userDomainName", userDomainName } });
            }
            if (string.IsNullOrEmpty(msnPassword))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_MsnInvitErrorForPasswordEmpty", applicationID);
                TempData["msnUserName"] = msnUserName;
                TempData["msnPassword"] = msnPassword;
                return RedirectToAction("NewFriendInvitationByEmail", new RouteValueDictionary() { { "userDomainName", userDomainName } });
            }

            SetPageTitle(userDomainName, GetResourceString("Title_MsnInvitation", applicationID));

            GetAddressByMsn msnAddress = new GetAddressByMsn();
            List<MsnFriend> friends = msnAddress.GetFriends(msnUserName, msnPassword, true);
            if (friends.Count == 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_MsnInvitErrorForGetFriendFailed", applicationID);
                TempData["msnUserName"] = msnUserName;
                TempData["msnPassword"] = msnPassword;
                friends = new List<MsnFriend>();
                return RedirectToAction("NewFriendInvitationByEmail", new RouteValueDictionary() { { "userDomainName", userDomainName } });
            }
            return View("Pages/MessageCenters/PersonUsers/InvitationMSNFriendsResult.aspx", "Masters/MessageCenter.master", friends);
        }

        /// <summary>
        /// 发送邮件邀请MSN好友
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult InvitationMSNFriendsAction(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string userSelectors = Request.Form.GetString("userSelectorCheckBox", string.Empty);
            string[] receivers = null;
            if (string.IsNullOrEmpty(userSelectors))
                return RedirectToAction("InvitationMSNFriendsResult", new RouteValueDictionary() { { "userDomainName", userDomainName } });
            receivers = userSelectors.Split(',');
            //发送Email成功的Email集合
            List<string> successEmails = new List<string>();
            //重复发送的Email集合
            List<string> duplicateEmails = new List<string>();
            string subject = string.Empty;
            string body = string.Empty;
            string invitationBody = Request.Form.GetString("invitationBody", string.Empty);
            InvitationTemplate template = InvitationTemplates.GetInvitationTemplate(ApplicationIDs.Instance().Friend());
            if (template != null)
            {
                subject = InvitationTemplates.EmailFormatterForPersonUser(template.Subject, CurrentUser);
                body = InvitationTemplates.EmailFormatterForPersonUser(template.Body, CurrentUser);
                body = InvitationTemplates.EmailFormatterForBody(body, invitationBody);
            }

            for (int i = 0; i < receivers.Length; i++)
            {
                Invitation invitation = new Invitation();
                invitation.Subject = string.Format(GetResourceString("Title_Pattern_DefaultMsnInvitationSubject", applicationID), CurrentUser.CommonName, Globals.SiteName);
                invitation.SenderUserID = CurrentUser.UserID;
                invitation.Sender = CurrentUser.DisplayName;
                invitation.Body = invitationBody;
                invitation.SenderEmail = string.IsNullOrEmpty(CurrentUser.PublicEmail) ? CurrentUser.PrivateEmail : CurrentUser.PublicEmail;
                invitation.Receiver = receivers[i];
                invitation.ReceiverEmail = (string)receivers[i];
                invitation.ApplicationID = ApplicationIDs.Instance().Friend();
                invitation.TransportMode = InvitationTransportModes.Email;
                SendEmail(invitation, body);
                CreateInvitationStatuses status = Invitations.CreateInvitation(invitation);
                if (status == CreateInvitationStatuses.Duplicate)
                    duplicateEmails.Add(receivers[i]);
                else
                    successEmails.Add(receivers[i]);
            }
            string strEmails = string.Empty;
            if (successEmails.Count > 0)
            {
                foreach (string email in successEmails)
                {
                    strEmails += "," + email;
                }
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitSendSuccess", applicationID), strEmails.Substring(1));

                string strDuplicateEmails = string.Empty;
                if (duplicateEmails.Count > 0)
                {
                    foreach (string dispalyname in duplicateEmails)
                    {
                        strDuplicateEmails += "," + dispalyname;
                    }
                    TempData["StatusMessageContent"] += string.Format(GetResourceString("Message_Pattern_EmailInvitForRepeatInvit", applicationID), strDuplicateEmails.Substring(1));
                }
            }
            else if (duplicateEmails.Count > 0)
            {
                string strDuplicateEmails = string.Empty;

                foreach (string dispalyname in duplicateEmails)
                {
                    strDuplicateEmails += "," + dispalyname;
                }
                TempData["StatusMessageType"] = StatusMessageType.Hint;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EmailInvitForRepeatInvit", applicationID), strDuplicateEmails.Substring(1));
            }
            return RedirectToAction("NewFriendInvitationByEmail", new RouteValueDictionary() { { "userDomainName", userDomainName } });
        }
        #endregion

        /// <summary>
        /// 发送email
        /// </summary>
        private bool SendEmail(Invitation invitation, string body)
        {
            //Email类型的邀请不再存储到数据库
            if (invitation.TransportMode == InvitationTransportModes.Email)
            {
                MailMessage mailMessage = ConvertMessageToMailMessage(invitation, body);
                Emails.EnqueuEmail(mailMessage);
                return true;
            }
            return false;

        }
        /// <summary>
        /// 把Invitation转化成MailMessage
        /// </summary>
        private MailMessage ConvertMessageToMailMessage(Invitation invitation, string body)
        {
            if (string.IsNullOrEmpty(invitation.SenderEmail) || string.IsNullOrEmpty(invitation.ReceiverEmail) || string.IsNullOrEmpty(invitation.Body))
                return null;

            MailMessage email = new MailMessage();
            email.From = new MailAddress(invitation.SenderEmail, invitation.Sender);

            email.To.Add(invitation.ReceiverEmail);

            email.Subject = invitation.Subject;
            email.Priority = MailPriority.High;
            email.Body = body;
            //email.Body = "<html><body>" + email.Body + "</body></html>";

            ////HTML Editor会经常把链接转换成基于本地的相对链接，所以需要转换回来
            email.Body = Formatter.ConvertLocalUrls(email.Body, WebUtils.HostPath(Request.Url));

            email.IsBodyHtml = true;

            //email.SubjectEncoding = System.Text.Encoding.UTF8;  //默认UTF8
            //email.BodyEncoding = System.Text.Encoding.UTF8;     //默认UTF8
            email.SubjectEncoding = System.Text.Encoding.Default;
            email.BodyEncoding = System.Text.Encoding.Default;
            return email;
        }

        #endregion

        #region 控件

        /// <summary>
        /// 邀请菜单列表控件
        /// </summary>
        public ActionResult Control_InvitationMenu(string userDomainName, InvitationMenuType menuType)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Dictionary<int, int> dictionary = Invitations.GetUnsettledCounts(currentDomainUser.UserID);
            int unsettledTotalCount = 0;
            foreach (int itemCount in dictionary.Values)
            {
                unsettledTotalCount += itemCount;
            }
            ViewData["unsettledTotalCount"] = unsettledTotalCount;
            ViewData["InvitationMenuType"] = menuType;
            return PartialView("Controls/MessageCenters/PersonUsers/InvitationMenu.ascx");
        }

        /// <summary>
        /// 收到的邀请列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListInvitationInBox(string userDomainName, int? applicationID, int? pageIndex, bool? isDeal)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<Invitation> pds = Invitations.GetInvitationsFromInbox(pageIndex ?? 1, currentDomainUser.UserID, applicationID, isDeal ?? false);
            ViewData["isDeal"] = isDeal ?? false;
            ViewData["PageIndex"] = pageIndex;
            ViewData["applicationID"] = applicationID;
            return PartialView("Controls/MessageCenters/PersonUsers/ListInvitationInBox.ascx", pds);
        }
        /// <summary>
        /// 发出的邀请列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListInvitationOutBox(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<Invitation> pds = Invitations.GetInvitationsFromOutbox(pageIndex ?? 1, currentDomainUser.UserID, null);
            return PartialView("Controls/MessageCenters/PersonUsers/ListInvitationOutBox.ascx", pds);
        }
        #endregion

        #region 操作
        /// <summary>
        /// 从收到邀请中删除一条邀请信息
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteInBoxInvitation(string userDomainName, int invitationID, int? applicationID, int? pageIndex, bool? isDeal)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Invitations.DeleteInvitationFromInbox(currentDomainUser.UserID, invitationID);
            return RedirectToAction("Control_ListInvitationInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "applicationID", applicationID }, { "pageIndex", pageIndex }, { "isDeal", isDeal } });
        }
        /// <summary>
        /// 从发出的邀请中删除一条邀请信息
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteOutBoxInvitation(string userDomainName, int invitationID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Invitations.DeleteInvitationFromOutbox(currentDomainUser.UserID, invitationID);
            return RedirectToAction("Control_ListInvitationOutBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 批量删除收到的邀请
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteInBoxInvitations(string userDomainName, int? applicationID, bool isDeal)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string selectedIDsString = Request.Form["InvitationInBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int invitationID = int.Parse(IDString);
                        if (invitationID > 0)
                            Invitations.DeleteInvitationFromInbox(currentDomainUser.UserID, invitationID);
                    }
                }
            }
            return RedirectToAction("Control_ListInvitationInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "applicationID", applicationID }, { "pageIndex", 1 }, { "isDeal", isDeal } });

        }
        /// <summary>
        /// 批量删除发出的邀请
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteOutBoxInvitations(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            string selectedIDsString = Request.Form["InvitationOutBoxCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int invitationID = int.Parse(IDString);
                        if (invitationID > 0)
                            Invitations.DeleteInvitationFromOutbox(currentDomainUser.UserID, invitationID);
                    }
                }
            }
            return RedirectToAction("Control_ListInvitationOutBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", 1 } });
        }

        /// <summary>
        /// 设置邀请的接受状态
        /// </summary>    
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetInvitationAcceptStatus(string userDomainName, int invitationID, int? applicationID, bool isAccept, int? pageIndex, bool isDeal)
        {
            //User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Invitation inv = Invitations.GetInvitation(invitationID);

            if (invitationID > 0 && inv.Status == InvitationStatuses.Unsettled)
            {
                if (isAccept)
                {
                    //设置状态为已接受
                    Invitations.SetStatus(invitationID, InvitationStatuses.Accept);
                }
                else
                {
                    Invitations.SetStatus(invitationID, InvitationStatuses.Refuse);
                }


            }
            return RedirectToAction("Control_ListInvitationInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "applicationID", applicationID }, { "pageIndex", pageIndex }, { "isDeal", isDeal } });
        }

        /// <summary>
        /// 设置好友信息 
        /// </summary>    
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_SetFriendInfo(string userDomainName, int invitationID, int? pageIndex)
        {
            Invitation inv = Invitations.GetInvitation(invitationID);
            ViewData["pageIndex"] = pageIndex;
            return PartialView("Controls/MessageCenters/PersonUsers/SetFriendInfo.ascx", inv);
        }

        /// <summary>
        /// 设置好友信息 
        /// </summary>    
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetFriendInfo(string userDomainName)
        {
            int invitationID = Request.QueryString.GetInt("invitationID", 0);
            int pageIndex = Request.QueryString.GetInt("pageIndex", 1);
            Invitation inv = Invitations.GetInvitation(invitationID);
            if (invitationID > 0 && inv.Status == InvitationStatuses.Unsettled)
            {
                //设置状态为已接受
                Invitations.SetStatus(invitationID, InvitationStatuses.Accept);
            }

            Friend friend = Friends.GetFriend(inv.ReceiverUserID, inv.SenderUserID);
            if (friend != null && friend.FriendID > 0)
            {
                int categoryID = Request.QueryString.GetInt("categoryID", 0);
                List<int> tempCategoryIDs = new List<int>();

                friend.NoteName = Request.QueryString.GetString("noteName", string.Empty);

                Friends.UpdateFriend(friend, Friendliness.All);
                if (categoryID > 0)
                {
                    tempCategoryIDs.Add(categoryID);
                }
                else if (!string.IsNullOrEmpty(Request.QueryString.GetString("groupName", string.Empty).Trim()))
                {
                    UserCategory userCategory = new UserCategory();
                    userCategory.UserID = inv.ReceiverUserID;

                    userCategory.CategoryName = Request.QueryString.GetString("groupName", string.Empty).Trim();

                    MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).CreateUserCategory(userCategory);
                    tempCategoryIDs.Add(userCategory.CategoryID);
                }
                MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).UpdateUserCategoriesOfItem(friend.FriendID, tempCategoryIDs);
                Friends.UpdateFriendToGroup(friend.FriendID, tempCategoryIDs);

            }
            return RedirectToAction("ListInvitationInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 设置好友信息 
        /// </summary>    
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_BatchSetFriendGroup(string userDomainName, int? pageIndex)
        {
            ViewData["pageIndex"] = pageIndex;
            return PartialView("Controls/MessageCenters/PersonUsers/BatchSetFriendGroup.ascx");
        }

        /// <summary>
        /// 设置好友信息 
        /// </summary>    
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult BatchSetFriendGroup(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            int categoryID = Request.QueryString.GetInt("categoryID", 0);
            //if (categoryID <= 0 && string.IsNullOrEmpty(groupName))
            //    return RedirectToAction("ListInvitationInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
            if (categoryID <= 0)
            {
                string groupName = Request.QueryString.GetString("groupName", string.Empty).Trim();
                if (!string.IsNullOrEmpty(groupName))
                {
                    UserCategory userCategory = new UserCategory();
                    userCategory.UserID = currentDomainUser.UserID;
                    userCategory.CategoryName = Request.QueryString.GetString("groupName", string.Empty).Trim();
                    MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).CreateUserCategory(userCategory);

                    categoryID = userCategory.CategoryID;
                }
            }


            Invitation inv = null;
            string selectedIDsString = Request.QueryString["invitationIDs"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int id;
                        int.TryParse(IDString, out id);
                        inv = Invitations.GetInvitation(id);
                        if (inv == null || inv.ApplicationID != ApplicationIDs.Instance().Friend())
                            continue;

                        if (inv.Status == InvitationStatuses.Unsettled)
                        {
                            //设置状态为已接受
                            Invitations.SetStatus(id, InvitationStatuses.Accept);
                        }
                        //更新朋友分组
                        Friend friend = Friends.GetFriend(inv.ReceiverUserID, inv.SenderUserID);
                        if (friend != null && friend.FriendID > 0 && categoryID > 0)
                        {
                            List<int> tempCategoryIDs = new List<int>();
                            tempCategoryIDs.Add(categoryID);
                            Friends.UpdateFriendToGroup(friend.FriendID, tempCategoryIDs);
                            MultiUserCategoryManager.Instance(MultiUserCategoryItemKeys.Instance().Friend()).UpdateUserCategoriesOfItem(friend.FriendID, tempCategoryIDs);
                        }
                    }
                }
            }
            ViewData["pageIndex"] = pageIndex;
            return RedirectToAction("ListInvitationInBox", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }



        #endregion

        #endregion

        #region 积分

        #region 礼品兑换

        /// <summary>
        /// 显示我的兑换记录
        /// </summary>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ListGiftExchangeRocords(string userDomainName, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Label_MyGiftExchangeForRecord", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<ConvertibleGiftExchangeRecord> pds = ConvertibleGiftExchangeRecords.GetRecords(null, currentDomainUser.UserID, pageIndex ?? 1);

            return View("Pages/MessageCenters/PersonUsers/ListGiftExchangeRocords.aspx", "Masters/MessageCenter.master", pds);
        }

        /// <summary>
        /// 管理礼品兑换申请
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ManageGiftExchangeApplies(string userDomainName, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Label_ManageGiftsExchange", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<ConvertibleGiftExchangeApply> pds = ConvertibleGiftExchangeApplies.GetMyApplies(currentDomainUser.UserID, pageIndex ?? 1);

            return View("Pages/MessageCenters/PersonUsers/ListGiftExchangeApplies.aspx", "Masters/MessageCenter.master", pds);
        }

        /// <summary>
        /// 取消申请
        /// </summary>
        /// <returns></returns>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CancelGiftExchangeApply(string userDomainName, int applyID)
        {
            if (applyID > 0)
                ConvertibleGiftExchangeApplies.CancelApply(applyID);

            return Redirect(Request.UrlReferrer.AbsoluteUri);
        }


        #endregion


        #region 页面

        /// <summary>
        /// 显示积分记录列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListPointRecords(string userDomainName, UserPointCategory? pointCategory, UserPointDirection? pointDirection, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_PointRecords", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<UserPointRecord> pds = Points.GetUserPointRecords(currentDomainUser.UserID, pointCategory.HasValue ? (UserPointCategory)pointCategory : UserPointCategory.Basic, pointDirection, 15, pageIndex ?? 1);
            ViewData["PointCategory"] = pointCategory;
            ViewData["PointDirection"] = pointDirection;
            return View("Pages/MessageCenters/PersonUsers/ListPointRecords.aspx", "Masters/MessageCenter.master", pds);
        }

        /// <summary>
        /// 显示总积分页
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult TotalUserPoints(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_PointTotal", applicationID));

            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PersonUser pu = currentDomainUser as PersonUser;
            IntegratedPointRule basicPointRule = Points.GetIntegratedPointRule(IntegratedPointItemIDs.Instance().User_BasicPoint());
            if (basicPointRule != null)
                ViewData["BasicPointRate"] = basicPointRule.ScoringRate;

            IntegratedPointRule reportPointRule = Points.GetIntegratedPointRule(IntegratedPointItemIDs.Instance().User_ReportPoint());
            if (reportPointRule != null)
                ViewData["ReportPointRate"] = reportPointRule.ScoringRate;
            IntegratedPointRule creditPointRule = Points.GetIntegratedPointRule(IntegratedPointItemIDs.Instance().User_ReportPoint());
            if (creditPointRule != null)
                ViewData["CreditPointRule"] = creditPointRule.ScoringRate;
            if (pu != null)
            {
                ViewData["TotalPoints"] = pu.BasicPoints * basicPointRule.ScoringRate + pu.ReportPoints * reportPointRule.ScoringRate;
                ViewData["BasicPoints"] = pu.BasicPoints;
                ViewData["ReportPoints"] = pu.ReportPoints;
                ViewData["CreditPoints"] = pu.CreditPoints;
                ViewData["TradePoints"] = pu.TradePoints;
            }

            return View("Pages/MessageCenters/PersonUsers/TotalUserPoints.aspx", "Masters/MessageCenter.master", string.Empty);
        }

        #endregion
        #endregion

        #region 访客
        #region 页面
        /// <summary>
        /// 谁来看过我
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListVisitor(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_ListVisitor", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<UserVisit> pds = UserVisits.GetVisitsTo(currentDomainUser.UserID, null, pageIndex ?? 1);
            Dictionary<int, User> visitor = new Dictionary<int, User>();
            foreach (UserVisit item in pds.Records)
            {
                visitor.Add(item.VisitID, Users.GetUser(item.FromUserID, false));
            }
            ViewData["Visitor"] = visitor;
            if (CurrentUser != null && (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator))
            {
                ViewData["IsUserDomainOwnerOrAdministrator"] = true;
            }

            return View("Pages/MessageCenters/ListVisitor.aspx", "Masters/MessageCenter.master", pds);
        }
        /// <summary>
        /// 我去看过谁
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListVisited(string userDomainName, int? pageIndex)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_ListVisited", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            return View("Pages/MessageCenters/ListVisited.aspx", "Masters/MessageCenter.master", string.Empty);
        }
        #endregion

        #region 控件
        /// <summary>
        /// 显示我去看过的访客列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListVisited(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            PagingDataSet<UserVisit> pds = UserVisits.GetVisitsFrom(currentDomainUser.UserID, pageIndex.HasValue ? pageIndex.Value : 1);
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["IsOwnerOrAdministrator"] = true;

            return PartialView("Controls/MessageCenters/ListVisited.ascx", pds);
        }
        /// <summary>
        /// 显示访客列表模块
        /// </summary>
        public ActionResult Control_ListVisitBlock(string userDomainName, bool? isVisitor)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new SPBException(ExceptionTypes.Instance().UserDomainNotFond());

            IList<UserVisit> userVisits;
            if (isVisitor.HasValue && isVisitor.Value)
                userVisits = UserVisits.GetTopVisitsTo(currentDomainUser.UserID, 9); //谁来看过我
            else
                userVisits = UserVisits.GetTopVisitsFrom(currentDomainUser.UserID, 9); //我去看过谁

            ViewData["isVisitor"] = isVisitor.HasValue ? isVisitor.Value : true;
            return PartialView("Controls/MessageCenters/ListVisitBlock.ascx", userVisits);
        }
        /// <summary>
        /// 显示最近访客列表模块
        /// </summary>
        public ActionResult Control_ListVisitorBlock(string userDomainName, int? displayCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            IList<UserVisit> userVisits = UserVisits.GetTopVisitsTo(currentDomainUser.UserID, displayCount ?? 9);
            return PartialView("Controls/MessageCenters/ListVisitorBlock.ascx", userVisits);
        }

        #endregion

        #region 操作
        /// <summary>
        /// 设置访问记录的状态
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult SetVisitIsVisible(string userDomainName, int visitID, bool isVisible, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserVisits.SetVisitVisible(currentDomainUser.UserID, visitID, isVisible);
            return RedirectToAction("Control_ListVisited", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 删除一条访问记录
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserVisit(string userDomainName, int visitID, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserVisits.DeleteVisit(currentDomainUser.UserID, visitID);
            return RedirectToAction("Control_ListVisited", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }
        #endregion

        #endregion

        #region 动态

        #region 页面

        /// <summary>
        /// 用户动态列表
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListUserActivities(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_ListUserActivities", applicationID));
            ViewData["PageIndex"] = pageIndex;
            //if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
            //    ViewData["IsOwnerOrAdministrator"] = true;
            return View("Pages/MessageCenters/ListUserActivities.aspx", "Masters/MessageCenter.master");
        }

        /// <summary>
        /// 好友动态列表
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFriendActivities(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_FriendActivity"));
            ViewData["PageIndex"] = pageIndex;
            return View("Pages/MessageCenters/PersonUsers/ListFriendActivities.aspx", "Masters/MessageCenter.master");
        }

        /// <summary>
        /// 站点动态列表
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListSiteActivities(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, "动态");
            ViewData["PageIndex"] = pageIndex;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["IsOwnerOrAdministrator"] = true;
            return View("Pages/MessageCenters/PersonUsers/ListSiteActivities.aspx", "Masters/MessageCenter.master");
        }

        /// <summary>
        /// 企业动态列表
        /// </summary>
        //[UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListCompanyActivities(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_ListSiteActivities", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            return View("Pages/MessageCenters/ListCompanyActivities.aspx", "Masters/MessageCenter.master");
        }
        #endregion

        #region 控件

        /// <summary>
        /// 用户动态列表（空间主页中显示用）
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        public ActionResult Control_ListUserActivitiesBlock(string userDomainName, int? displayCount)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            IList<UserActivity> userActivities;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                userActivities = Activities.GetTopUserActivities(currentDomainUser.UserID, null, null, displayCount ?? 10, true);
            else
                userActivities = Activities.GetTopUserActivities(currentDomainUser.UserID, null, true, displayCount ?? 10, false);

            return PartialView("Controls/MessageCenters/ListUserActivitiesBlock.ascx", userActivities);
        }

        /// <summary>
        /// 显示群组动态用户控件
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ListReceivedClubActivities(string userDomainName, int? displayItemCount, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<ClubActivity> pds = ClubActivities.GetClubActivitiesForUser(currentDomainUser.UserID, null, pageIndex ?? 1, HasManagePermission(currentDomainUser));
            return PartialView("Controls/MessageCenters/PersonUsers/ListReceivedClubActivities.ascx", pds);
        }
        /// <summary>
        /// 显示群组动态列表
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListReceivedClubActivities(string userDomainName)
        {
            SetPageTitle(GetResourceString("Title_ClubDynamic", ApplicationIDs.Instance().Club()));
            return View("Pages/MessageCenters/PersonUsers/ListReceivedClubActivities.aspx", "Masters/MessageCenter.master");
        }

        /// <summary>
        /// 在群组动态列表页删除群组动态
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteClubActivity(string userDomainName, int applicationID, int? pageIndex, int activityID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ClubActivities.DeleteClubAcitivityForClubMember(currentDomainUser.UserID, applicationID, activityID);
            return RedirectToAction("Control_ListReceivedClubActivities", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }


        /// <summary>
        /// 个人动态列表
        /// </summary>
        public ActionResult Control_ListUserActivities(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PagingDataSet<UserActivity> userActivities;
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
            {
                ViewData["IsOwnerOrAdministrator"] = true;
                userActivities = Activities.GetUserActivities(currentDomainUser.UserID, null, null, pageIndex ?? 1, true);
            }
            else
                userActivities = Activities.GetUserActivities(currentDomainUser.UserID, null, true, pageIndex ?? 1, false);


            return PartialView("Controls/MessageCenters/ListUserActivities.ascx", userActivities);
        }



        /// <summary>
        /// 好友动态模块显示
        /// </summary>
        public ActionResult Control_ListFriendActivitiesBlock(string userDomainName, int? topNumber, int? applicationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            IList<FriendActivity> friendActivities = Activities.GetTopFriendActivities(currentDomainUser.UserID, applicationID, topNumber ?? 10, HasManagePermission(currentDomainUser));

            int totalRecords = Activities.GetFriendActivities(currentDomainUser.UserID, applicationID, 0, false).TotalRecords;

            ViewData["topNumber"] = topNumber ?? 10;
            ViewData["applicationID"] = applicationID;
            ViewData["totalRecords"] = totalRecords;
            return PartialView("Controls/MessageCenters/PersonUsers/ListFriendActivitiesBlock.ascx", friendActivities);
        }

        /// <summary>
        /// 某个应用中好友的动态
        /// </summary>
        public ActionResult Control_ListApplicationFriendActivitiesBlock(string userDomainName, int applicationID, int count)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            IList<FriendActivity> friendActivities = Activities.GetTopFriendActivities(currentDomainUser.UserID, applicationID, count, HasManagePermission(currentDomainUser));
            return PartialView("Controls/MessageCenters/ListApplicationFriendActivitiesBlock.ascx", friendActivities);
        }

        /// <summary>
        /// 我的首页tab块动态
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="isMyActivity"></param>
        /// <returns></returns>
        public ActionResult Control_ListUserActivitiesForTab(string userDomainName, int? topNumber, bool isMyActivity, int? applicationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            IList<UserActivity> userActivities;
            int totalRecords;
            if (isMyActivity)
            {
                if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                    userActivities = Activities.GetTopUserActivities(currentDomainUser.UserID, applicationID, null, topNumber ?? 10, true);
                else
                    userActivities = Activities.GetTopUserActivities(currentDomainUser.UserID, applicationID, true, topNumber ?? 10, false);
                totalRecords = Activities.GetUserActivities(currentDomainUser.UserID, applicationID, null, 0, false).TotalRecords;

            }
            else
            {
                userActivities = Activities.GetTopSiteUserActivities(applicationID, topNumber ?? 10);
                totalRecords = Activities.GetSiteUserActivities(applicationID, 0).TotalRecords;
            }

            ViewData["topNumber"] = topNumber ?? 10;
            ViewData["applicationID"] = applicationID;
            ViewData["totalRecords"] = totalRecords;
            ViewData["isMyActivity"] = isMyActivity;
            ViewData["isUserDomainOwner"] = IsUserDomainOwner(userDomainName);
            return View("Controls/MessageCenters/PersonUsers/ListUserActivities.ascx", userActivities);
        }

        ///// <summary>
        ///// 关注好友动态     已无该功能
        ///// </summary>
        ///// <param name="userDomainName"></param>
        ///// <returns></returns>
        //public ActionResult Control_ListFavoriteActivities(string userDomainName, int? applicationID, int? topNumber)
        //{
        //    User domainUser = GetCurrentDomainUser(userDomainName);
        //    IList<UserActivity> activities = Activities.GetTopFavoriteActivities(domainUser.UserID, applicationID, topNumber ?? 10);
        //    int totalRecords = Activities.GetTopFavoriteActivities(domainUser.UserID, applicationID, ValueHelper.GetSqlMaxInt()).Count;
        //    ViewData["totalRecords"] = totalRecords; ;
        //    ViewData["topNumber"] = topNumber;
        //    ViewData["applicationID"] = applicationID;
        //    return PartialView("Controls/MessageCenters/PersonUsers/ListFavoriteActivities.ascx", activities);
        //}
        ///// <summary>
        ///// 删除关注好友动态    已无该功能
        ///// </summary>
        ///// <param name="userDomainName"></param>
        ///// <returns></returns>
        //public ActionResult DleteFavoriteActivitie(string userDomainName, int activityID, int? applicationID, int? topNumber)
        //{
        //    User currentDomainUser = GetCurrentDomainUser(userDomainName);
        //    Activities.DeleteUserActivity(currentDomainUser.UserID, activityID);
        //    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
        //    TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");
        //    return Control_ListFavoriteActivities(userDomainName, applicationID, topNumber);

        //}
        /// <summary>
        /// 好友动态列表
        /// </summary>
        public ActionResult Control_ListFriendActivities(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<FriendActivity> friendActivities = Activities.GetFriendActivities(currentDomainUser.UserID, null, pageIndex ?? 1, HasManagePermission(currentDomainUser));
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["IsOwnerOrAdministrator"] = true;

            return PartialView("Controls/MessageCenters/PersonUsers/ListFriendActivities.ascx", friendActivities);
        }

        /// <summary>
        /// 站点动态列表
        /// </summary>
        public ActionResult Control_ListSiteActivities(string userDomainName, int? pageIndex)
        {
            PagingDataSet<UserActivity> userActivity = Activities.GetSiteUserActivities(null, pageIndex ?? 1);
            if (CurrentUser != null && (IsUserDomainOwner(userDomainName) || CurrentUser.IsContentAdministrator))
                ViewData["IsOwnerOrAdministrator"] = true;
            return PartialView("Controls/MessageCenters/PersonUsers/ListSiteUserActivities.ascx", userActivity);
        }

        #endregion

        #region 操作
        /// <summary>
        /// 删除用户动态
        /// </summary>
        /// <param name="userDomainName">用户动态所有者用户名</param>
        /// <param name="activityID">要删除的动态ID</param>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserActivity(string userDomainName, int activityID, int pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Activities.DeleteUserActivity(currentDomainUser.UserID, activityID);
            return Control_ListUserActivities(userDomainName, pageIndex);
        }

        /// <summary>
        /// 删除好友动态
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteFriendActivity(string userDomainName, int activityID, int pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Activities.DeleteFromFriendActivityInbox(currentDomainUser.UserID, activityID);
            return Control_ListFriendActivities(userDomainName, pageIndex);
        }

        /// <summary>
        /// 删除好友动态 在首页模块中
        /// </summary>
        public ActionResult DeleteFirendActivityInBlock(string userDomainName, int topNumber, int activityID, int? applicationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Activities.DeleteFromFriendActivityInbox(currentDomainUser.UserID, activityID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");

            return Control_ListFriendActivitiesBlock(userDomainName, topNumber, applicationID);
        }

        /// <summary>
        /// 删除用户动态 在首页模块中
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserActivityInBlock(string userDomainName, int activityID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Activities.DeleteUserActivity(currentDomainUser.UserID, activityID);
            return Control_ListUserActivitiesBlock(userDomainName, Request.QueryString.GetInt("displayItemCount", 10));
        }


        /// <summary>
        /// 删除我的动态 在个人中心中
        /// </summary>
        public ActionResult DeleteUserActivityForTab(string userDomainName, int? topNumber, int activityID, bool isMyActivity)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            int? userID = currentDomainUser.UserID;
            if (CurrentUser.IsContentAdministrator && !isMyActivity)
                userID = null;
            Activities.DeleteUserActivity(userID, activityID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSuccess");
            return Control_ListUserActivitiesForTab(userDomainName, topNumber, isMyActivity, null);
            //return RedirectToAction("Control_ListUserActivitiesForTab", new RouteValueDictionary { { "userDomainName", userDomainName }, { "topNumber", topNumber }, { "isMyActivity", isMyActivity } });
        }


        /// <summary>
        /// 隐藏某用户的动态
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult UpdateHideUserActivitySetting(string userDomainName, bool? isHome, int leadingActorUserID, int? pageIndex, int? applicationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Activities.DeteteFriendActivitiesByUser(currentDomainUser.UserID, leadingActorUserID);
            UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings((currentDomainUser.UserID), true);

            User leadingActorUser = Users.GetUser(leadingActorUserID, false);
            if (leadingActorUser != null)
            {
                userActivitySettings.IgnoredFriendUsersForReceiveActivities[leadingActorUserID] = leadingActorUser.DisplayName;
                Activities.UpdateUserActivitySettings(userActivitySettings);
            }
            if (isHome == null)
                return Control_ListFriendActivities(userDomainName, pageIndex);
            else
                return
                Control_ListFriendActivitiesBlock(userDomainName, null, applicationID);
        }

        /// <summary>
        /// 隐藏某类消息
        /// </summary>
        /// <returns></returns>
        public ActionResult UpdateHideTypeActivitySetting(string userDomainName, int activityItemID, bool? isHome, int? pageIndex, int? applicationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Activities.DeteteFriendActivitiesByActivityItemID(currentDomainUser.UserID, activityItemID);
            UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings((currentDomainUser.UserID), true);
            ActivityItem item = null;
            try
            {
                item = Activities.GetActivityItem(activityItemID);
            }
            catch
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format("找不到{0}的消息类", activityItemID)));
            }
            ActivityItemSettings setting = new ActivityItemSettings(item);
            setting.IsAllowReceiving = false;
            setting.IsAllowSending = false;
            userActivitySettings.ActivityItemSettingss[item.ItemKey] = setting;
            Activities.UpdateUserActivitySettings(userActivitySettings);

            if (isHome == null)
                return Control_ListFriendActivities(userDomainName, pageIndex);
            else
                return Control_ListFriendActivitiesBlock(userDomainName, null, applicationID);
        }


        #endregion

        #endregion

        #region 留言

        #region 页面
        /// <summary>
        /// 显示全部留言页
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ShowGuestBook(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_ShowGuestBook", applicationID));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            return View("Pages/MessageCenters/ShowGuestBook.aspx", "Masters/MessageCenter.master");
        }
        #endregion

        #region 控件
        /// <summary>
        /// 显示全部留言
        /// </summary>     
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ShowGuestBook(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool hasManagePermission = HasManagePermission(userDomainName);
            ApprovalStatuses? status = ApprovalStatuses.IsApproved;
            if (hasManagePermission)
                status = null;
            PagingDataSet<GuestbookPost> pds = UserGuestbookManager.Instance().GetGuestbookPosts(pageIndex ?? 1, currentDomainUser.UserID, status, hasManagePermission);
            Dictionary<int, User> guestBookUsers = new Dictionary<int, User>();
            foreach (GuestbookPost post in pds.Records)
            {
                User user = Users.GetUser(post.UserID, false);
                if (user != null)
                    guestBookUsers.Add(post.PostID, user);
            }
            ViewData["guestBookUsers"] = guestBookUsers;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || currentDomainUser.UserID == CurrentUser.UserID))
            {
                ViewData["IsAllowed"] = true;
            }

            return PartialView("Controls/MessageCenters/ShowGuestBook.ascx", pds);
        }
        /// <summary>
        /// 在空间主页显示我的留言板
        /// </summary>
        public ActionResult Control_MyGuestBookBlock(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            bool hasManagePermission = HasManagePermission(userDomainName);
            ApprovalStatuses? status = ApprovalStatuses.IsApproved;
            if (hasManagePermission)
                status = null;
            PagingDataSet<GuestbookPost> pds = UserGuestbookManager.Instance().GetGuestbookPosts(1, currentDomainUser.UserID, status, hasManagePermission);
            Dictionary<int, User> guestBookUsers = new Dictionary<int, User>();
            foreach (GuestbookPost post in pds.Records)
            {
                User user = Users.GetUser(post.UserID, false);
                if (user != null)
                    guestBookUsers.Add(post.PostID, user);
            }
            ViewData["guestBookUsers"] = guestBookUsers;
            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || currentDomainUser.UserID == CurrentUser.UserID))
            {
                ViewData["IsAllowed"] = true;
            }
            return PartialView("Controls/MessageCenters/MyGuestBookBlock.ascx", pds);
        }
        #endregion

        #region 操作
        /// <summary>
        /// 从空间主页留言板添加一条留言
        /// </summary>
        public ActionResult CreateGuestBookPostInBlock(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (!settings.EnableAnonymousPosting && CurrentUser == null)
                throw new Exception(GetResourceString("Exception_CreateGuestBookPostErrorForIsAnonymous", applicationID));
            string body = Request.Form.GetString("body", string.Empty);
            string author = Request.Form.GetString("author", string.Empty);
            int parentID = Request.Form.GetInt("parentID", 0);
            bool isPrivate = Request.Form.GetBool("isPrivate", false);

            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                if (CurrentUser == null)
                {
                    string treatedAuthor = Formatter.FormatSingleLinePlainTextForStorage(author);
                    if (string.IsNullOrEmpty(author) || treatedAuthor.Length > 30)
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        if (string.IsNullOrEmpty(author))
                            TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostErrorForAuthorEmpty", applicationID);
                        if (treatedAuthor.Length > 30)
                            TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostErrorForAuthorLong", applicationID);
                        TempData["Author"] = author;
                        TempData["Body"] = body;
                        TempData["ParentID"] = parentID;
                        TempData["isPrivate"] = isPrivate;
                        return RedirectToAction("Control_MyGuestBookBlock", new RouteValueDictionary { { "userDomainName", userDomainName } });
                    }
                }
                if (verifyCode != null)
                {
                    verifyCode = verifyCode.Trim();
                    if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                        TempData["Author"] = author;
                        TempData["Body"] = body;
                        TempData["ParentID"] = parentID;
                        TempData["isPrivate"] = isPrivate;
                        return RedirectToAction("Control_MyGuestBookBlock", new RouteValueDictionary { { "userDomainName", userDomainName } });
                    }
                }
            }

            GuestbookPost post = new GuestbookPost();

            if (CurrentUser != null)
            {
                post.Author = CurrentUser.DisplayName;
                post.UserID = CurrentUser.UserID;
            }
            else
            {
                post.Author = author;
                post.UserID = 0;
            }

            post.Body = body;
            post.GuestbookID = currentDomainUser.UserID;
            post.ParentID = parentID;
            post.IsPrivate = isPrivate;
            if (post.ParentID > 0)
            {
                GuestbookPost parentPost = UserGuestbookManager.Instance().GetGuestbookPost(post.ParentID, false);
                if (parentPost != null)
                {
                    post.ParentAuthor = parentPost.Author;
                    post.ParentBody = parentPost.Body;
                    post.ParentUserID = parentPost.UserID;
                    post.ParentPostDate = parentPost.PostDate;
                    if (parentPost.IsPrivate)
                        post.IsPrivate = true;
                    if (post.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (post.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author).Length)
                        {
                            post.Body = post.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostErrorForBodyEmpty", applicationID);
                            TempData["Body"] = body;
                            TempData["ParentID"] = parentID;
                            TempData["isPrivate"] = isPrivate;

                            return RedirectToAction("Control_MyGuestBookBlock", new RouteValueDictionary { { "userDomainName", userDomainName } });
                        }
                    }
                }
            }

            try
            {
                UserGuestbookManager.Instance().Create(post);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, null);
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_MyGuestBookBlock", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            if ((int)post.AuditingStatus >= (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
                TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostSuccess", applicationID);
            else
                TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostSuccessWithAuditing", applicationID);

            return RedirectToAction("Control_MyGuestBookBlock", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }
        /// <summary>
        /// 从信息中心的留言列表中添加一条留言
        /// </summary>
        public ActionResult CreateGuestBookPost(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (!settings.EnableAnonymousPosting && CurrentUser == null)
                throw new Exception(GetResourceString("Exception_CreateGuestBookPostErrorForIsAnonymous", applicationID));

            string body = Request.Form.GetString("body", string.Empty);
            string author = Request.Form.GetString("author", string.Empty);
            int parentID = Request.Form.GetInt("parentID", 0);
            bool isPrivate = Request.Form.GetBool("isPrivate", false);
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                if (CurrentUser == null)
                {
                    string treatedAuthor = Formatter.FormatSingleLinePlainTextForStorage(author);
                    if (string.IsNullOrEmpty(author) || treatedAuthor.Length > 30)
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        if (string.IsNullOrEmpty(author))
                            TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostErrorForAuthorEmpty", applicationID);
                        if (treatedAuthor.Length > 30)
                            TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostErrorForAuthorLong", applicationID);

                        TempData["Author"] = author;
                        TempData["Body"] = body;
                        TempData["ParentID"] = parentID;
                        TempData["isPrivate"] = isPrivate;
                        return RedirectToAction("Control_ShowGuestBook", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });

                    }
                }
                if (verifyCode != null)
                {
                    verifyCode = verifyCode.Trim();
                    if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                        TempData["Author"] = author;
                        TempData["Body"] = body;
                        TempData["ParentID"] = parentID;
                        TempData["isPrivate"] = isPrivate;
                        return RedirectToAction("Control_ShowGuestBook", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
                    }
                }
            }

            GuestbookPost post = new GuestbookPost();

            if (CurrentUser != null)
            {
                post.Author = CurrentUser.DisplayName;
                post.UserID = CurrentUser.UserID;
            }
            else
            {
                post.Author = author;
                post.UserID = 0;
            }

            post.Body = body;
            post.GuestbookID = currentDomainUser.UserID;
            post.ParentID = parentID;
            post.IsPrivate = isPrivate;

            if (post.ParentID > 0)
            {
                GuestbookPost parentPost = UserGuestbookManager.Instance().GetGuestbookPost(post.ParentID, false);
                if (parentPost != null)
                {
                    post.ParentAuthor = parentPost.Author;
                    post.ParentBody = parentPost.Body;
                    post.ParentUserID = parentPost.UserID;
                    post.ParentPostDate = parentPost.PostDate;
                    if (parentPost.IsPrivate)
                        post.IsPrivate = true;

                    if (post.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (post.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author).Length)
                        {
                            post.Body = post.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostErrorForBodyEmpty", applicationID);
                            TempData["Body"] = body;
                            TempData["ParentID"] = parentID;
                            return RedirectToAction("Control_ShowGuestBook", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
                        }
                    }
                }
            }

            try
            {
                UserGuestbookManager.Instance().Create(post);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, null);
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ShowGuestBook", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            if ((int)post.AuditingStatus >= (int)SiteSettingsManager.GetSiteSettings().AuditingStatusesForPublicDisplay)
                TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostSuccess", applicationID);
            else
                TempData["StatusMessageContent"] = GetResourceString("Message_CreateGuestBookPostSuccessWithAuditing", applicationID);

            return RedirectToAction("Control_ShowGuestBook", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 从空间主页中的留言板删除一条留言
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteGuestBookInBlock(string userDomainName, int postID)
        {
            UserGuestbookManager.Instance().Delete(postID);
            return RedirectToAction("Control_MyGuestBookBlock", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }
        /// <summary>
        /// 从信息中心的留言列表中删除一条留言
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteGuestBook(string userDomainName, int postID, int? pageIndex)
        {
            UserGuestbookManager.Instance().Delete(postID);
            return RedirectToAction("Control_ShowGuestBook", new RouteValueDictionary { { "userDomainName", userDomainName }, { "pageIndex", pageIndex } });
        }
        #endregion
        #endregion

        #region 权限

        /// <summary>
        /// 显示所有权限设置
        /// </summary>
        public ActionResult ListPermission(string userDomainName)
        {
            SetPageTitle(userDomainName, "权限");
            return View("Pages/MessageCenters/ListPermission.aspx", "Masters/MessageCenter.master");
        }

        /// <summary>
        /// 显示所有权限设置
        /// </summary>
        public ActionResult Control_ListPermission(string userDomainName, int appID, int type)
        {
            int typeInt = type;
            int applicationID = appID;

            if (typeInt == 0)
            {
                List<PermissionItem> ListPItem = Permissions.GetPermissionItemsCanAssociateWithUserRole(applicationID);
                ViewData["ListPItem"] = ListPItem;
                ICollection<Role> ListRole = Roles.GetRolesByApplicationID(applicationID, true, true);
                Dictionary<int, PermissionCollection> DicPerCollection = new Dictionary<int, PermissionCollection>();
                foreach (var item in ListRole)
                {
                    PermissionCollection PerCollection = Permissions.GetPermissionItemRolesInUserRoleForAdmin(item.RoleID, applicationID);
                    DicPerCollection.Add(item.RoleID, PerCollection);
                }
                ViewData["ListRole"] = ListRole;
                ViewData["DicPerCollection"] = DicPerCollection;
            }
            else
            {
                List<PermissionItem> ListPItem = Permissions.GetPermissionItemsCanAssociateWithUserRank(applicationID);
                ViewData["ListPItem"] = ListPItem;
                List<UserRank> ListRank = UserRanks.GetAllUserRanks(true).Values.ToList();
                Dictionary<int, PermissionCollection> DicPerCollection = new Dictionary<int, PermissionCollection>();
                foreach (var item in ListRank)
                {
                    PermissionCollection PerCollection = Permissions.GetPermissionItemRolesInUserRank(item.Rank, applicationID);
                    DicPerCollection.Add(item.Rank, PerCollection);
                }
                ViewData["ListRank"] = ListRank;
                ViewData["DicPerCollection"] = DicPerCollection;
            }

            Dictionary<int, Application> dic = Applications.GetApplicationsForAdmin();
            Dictionary<int, Application> dic2 = new Dictionary<int, Application>(dic);
            foreach (var item in dic.Keys)
            {
                List<PermissionItem> l = Permissions.GetPermissionItemsCanAssociateWithUserRole(item);
                if (l == null || l.Count <= 0)
                {
                    dic2.Remove(item);
                }
            }
            ViewData["DicApplication"] = dic2;
            ViewData["appID"] = appID;
            ViewData["type"] = type;
            return PartialView("Controls/MessageCenters/ListPermission.ascx");
        }

        #endregion

        #region 清除过期数据
        /// <summary>
        /// 显示清除过期数据模式框
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_DeleteUserExpiredData(string userDomainName, UserExpiredDataType userExpiredDataType)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ViewData["UserExpiredDataType"] = userExpiredDataType;
            return PartialView("Controls/MessageCenters/DeleteUserExpiredData.ascx");
        }
        /// <summary>
        /// 执行清除过期数据操作
        /// </summary>
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult ExecDeleteUserExpiredData(string userDomainName, UserExpiredDataType userExpiredDataType)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            int beforeDays = Request.Form.GetInt("beforeDays", -1);
            if (beforeDays <= 0)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_DeleteUserExpiredDataErrorForDataEmpty", applicationID));
                return RedirectToAction("Control_DeleteUserExpiredData", new RouteValueDictionary { { "userDomainName", userDomainName }, { "userExpiredDataType", userExpiredDataType } });
            }
            int numberOfDelete = 0;
            switch (userExpiredDataType)
            {
                case UserExpiredDataType.UserVisit:
                    numberOfDelete = UserVisits.BatchDeleteVisits(currentDomainUser.UserID, beforeDays);
                    if (numberOfDelete > 0)
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, string.Format(GetResourceString("Message_DeleteVistorSuccess", applicationID), numberOfDelete));
                    else
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_DeleteVistorErrorForEmpty", applicationID), beforeDays));
                    break;
                case UserExpiredDataType.UserActivity:
                    numberOfDelete = Activities.BatchDeleteUserActivities(currentDomainUser.UserID, beforeDays);
                    if (numberOfDelete > 0)
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, string.Format(GetResourceString("Message_DeleteUserActivitySuccess", applicationID), numberOfDelete));
                    else
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_DeleteUserActivityErrorForEmpty", applicationID), beforeDays));
                    break;
                case UserExpiredDataType.FriendActivity:
                    numberOfDelete = Activities.BatchDeleteFromFriendActivityInbox(currentDomainUser.UserID, beforeDays);
                    if (numberOfDelete > 0)
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, string.Format(GetResourceString("Message_DeleteFriendActivitySuccess", applicationID), numberOfDelete));
                    else
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, string.Format(GetResourceString("Message_DeleteFriendActivityErrorForEmpty", applicationID), beforeDays));
                    break;
                default:
                    break;
            }
            return RedirectToAction("Control_DeleteUserExpiredData", new RouteValueDictionary { { "userDomainName", userDomainName }, { "userExpiredDataType", userExpiredDataType } });
        }

        #endregion
    }

    #region Enums

    /// <summary>
    /// 信息中心列表类型
    /// </summary>
    public enum MessageCenterMenuType
    {
        /// <summary>
        /// 消息
        /// </summary>
        Message,
        /// <summary>
        /// 通知
        /// </summary>
        NoticeRecord,
        /// <summary>
        /// 邀请
        /// </summary>
        Invitation,
        /// <summary>
        /// 积分
        /// </summary>
        Point,
        /// <summary>
        /// 访客
        /// </summary>
        Visit,
        /// <summary>
        /// 动态
        /// </summary>
        Activity,
        /// <summary>
        /// 留言
        /// </summary>
        GuestBook,
        /// <summary>
        /// 权限
        /// </summary>
        Permission

    }
    /// <summary>
    /// 私信列表类型
    /// </summary>
    public enum PrivateMessageMenuType
    {
        /// <summary>
        /// 私信收件箱
        /// </summary>
        MessageInBox,
        /// <summary>
        /// 私信发件箱
        /// </summary>
        MessageOutBox,
        /// <summary>
        /// 写新私信
        /// </summary>
        CreateMessage,
        /// <summary>
        /// 联系客服
        /// </summary>
        MessageToService,

        /// <summary>
        /// 未设置
        /// </summary>
        None
    }

    /// <summary>
    /// 邀请列表类型
    /// </summary>
    public enum InvitationMenuType
    {
        /// <summary>
        /// 邀请收件箱
        /// </summary>
        InvitationInBox,
        /// <summary>
        /// 邀请发件箱
        /// </summary>
        InvitationOutBox,
        /// <summary>
        /// 发新邀请
        /// </summary>
        CreateInvitation
    }

    /// <summary>
    /// 用户过期数据类型
    /// </summary>
    public enum UserExpiredDataType
    {
        /// <summary>
        /// 访客数据
        /// </summary>
        UserVisit,
        /// <summary>
        /// 个人动态数据
        /// </summary>
        UserActivity,
        /// <summary>
        /// 朋友动态数据
        /// </summary>
        FriendActivity
    }

    #endregion
}
