﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------ 
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using System.Collections;
using SpaceBuilder.Utils;
using SpaceBuilder.Common;
using SpaceBuilder.Web;
using SpaceBuilder.Club;
using System.Drawing;
using System.Web.Routing;
using SpaceBuilder.LuceneSearch;
using SpaceBuilder.Forum;
using System.Web.UI;

namespace SpaceBuilder.Common.Controllers
{
    /// <summary>
    /// 用户空间设置频道的Controller
    /// </summary>
    [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
    [HttpCompress]
    public class UserDomainSettingsController : UserDomainControllerBase
    {
        private int applicationID = ApplicationIDs.Instance().User();

        #region 账号
        /// <summary>
        /// 编辑用户资料
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditProfile(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            PersonProfileType type = Request.QueryString.Get<PersonProfileType>("type", PersonProfileType.AccountInfo);
            if (currentDomainUser.UserType == UserTypes.PersonUser)
            {
                return EditPersonUserProfile(currentDomainUser, type.ToString());
            }
            else
                return EditCompanyUserProfile(currentDomainUser);
        }

        //编辑个人用户资料

        private ActionResult EditPersonUserProfile(User currentDomainUser, string pageName)
        {
            #region 标题
            SetPageTitle(currentDomainUser.UserName, GetResourceString("Title_EditPersonUserProfile", applicationID));
            #endregion

            string page = Request.QueryString.GetString("type", PersonProfileType.AccountInfo.ToString());

            PersonUser userToEdit = currentDomainUser as PersonUser;
            userToEdit = userToEdit as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            //只有超级管理员才有权限更新自己的设置
            if (userToEdit.IsContentAdministrator && !CurrentUser.IsContentAdministrator)
                throw new SPBException(ExceptionTypes.Instance().ModerateAccessDenied());
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            #region 为用户等级下拉列表绑定数据
            if (settings.EnableSelectUserRankType)
            {
                string selectUserRanks = settings.UserRankTypesForUserSelect;
                string[] array = null;
                if (!string.IsNullOrEmpty(selectUserRanks))
                    array = selectUserRanks.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);

                SortedList<int, string> userRankDataSoure = new SortedList<int, string>();
                if (array != null)
                {
                    foreach (string s in array)
                    {
                        int key = int.Parse(s);
                        string value = string.Empty;
                        if (settings.EnableMultilingual)
                        {
                            CodeItem item = CodeItems.GetCodeItem(CodeSetCodes.Instance().UserRankType(), key);
                            if (item != null)
                                value = item.GetMultilingualName(CurrentUserLanguage);
                        }
                        if (string.IsNullOrEmpty(value))
                            value = settings.NameOfUserRankTypes[key - 1];
                        userRankDataSoure.Add(key, value);
                    }

                    ViewData["UserRankDataSoure"] = userRankDataSoure;
                }
            }
            #endregion

            Dictionary<int, CodeItem> TimeZoneList = CodeItems.GetCodeItems(CodeSetCodes.Instance().TimeZone());
            ViewData["TimeZone"] = TimeZoneList;

            SiteSettings setting = SiteSettingsManager.GetSiteSettings();
            ViewData["SiteSetting"] = setting;

            Dictionary<string, Language> dictionary = Languages.GetEnabledLanguages();
            Dictionary<string, string> languages = dictionary.ToDictionary(n => n.Key, n => n.Value.NativeName);
            ViewData["languages"] = languages;


            Dictionary<string, PrivacyItemSettings> privacyItemSettingss = Privacys.GetUserPrivacyItemSettingss(currentDomainUser.UserID, true);
            ViewData["PrivacyItemSettingss"] = privacyItemSettingss;

            return View("Pages/Settings/PersonUsers/" + page + ".aspx", "Masters/Settings.master", userToEdit);
        }


        /// <summary>
        /// 论坛信息
        /// </summary>     
        public ActionResult ForumSignature(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(currentDomainUser.UserName, GetResourceString("Title_PersonSignatureForum", applicationID));
            ForumUserData userData = ForumUserDatas.GetUserData(currentDomainUser.UserID, HasManagePermission(userDomainName));
            return View("Pages/Settings/ForumSignature.aspx", "Masters/Settings.master", userData);

        }
        /// <summary>
        /// 更新论坛信息
        /// </summary>       
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateForumSignature(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            ForumUserData userData = ForumUserDatas.GetUserData(currentDomainUser.UserID, HasManagePermission(userDomainName));
            if (userData != null)
            {
                userData.Signature = Request.Form.GetString("forumBody", userData.Signature);
            }
            else
            {
                userData = new ForumUserData();
                userData.Signature = Request.Form.GetString("forumBody", string.Empty);
                userData.UserID = currentDomainUser.UserID;
            }
            if (string.IsNullOrEmpty(userData.Signature))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileError", applicationID);
                return RedirectToAction("ForumSignature", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            if (userData.Signature.Length > 1000)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileErrorNum", applicationID);
                return RedirectToAction("ForumSignature", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            ForumUserDatas.UpdateUserData(userData);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileSucceed", applicationID);
            return RedirectToAction("ForumSignature", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 更新个人资料
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateProfile(string userDomainName, UserAccountStatuses? prevAccountStatus)
        {
            User domainUser = Users.GetUser(userDomainName, true);
            PersonUser userToEdit = domainUser as PersonUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            int type = Request.Form.GetInt("type", 0);

            string privateEmail = Request.Form.GetString("privateEmail", userToEdit.PrivateEmail);
            if (!userToEdit.PrivateEmail.Equals(privateEmail, StringComparison.CurrentCultureIgnoreCase))
            {
                string message = string.Empty;
                if (Globals.ValidateEmail(privateEmail, out message))
                {

                    userToEdit.IsEmailVerified = false;
                    userToEdit.PrivateEmail = privateEmail;
                    UserEmailsProvider.Instance().UserCreateValidateEmail(userToEdit);
                    //TODO：提示信息 
                }
                //TODO：提示信息 message
            }
            userToEdit.Timezone = Request.Form.GetInt("TimeZone", (int)userToEdit.Timezone);
            userToEdit.Language = Request.Form.GetString("languages", userToEdit.Language);
            userToEdit.EnableAvatar = Request.Form.GetBool("enableAvatar", userToEdit.EnableAvatar);
            userToEdit.EnableDisplayInMemberList = Request.Form.GetBool("enableDisplayInMemberList", userToEdit.EnableDisplayInMemberList);
            userToEdit.CommonName = Request.Form.GetString("commonName", userToEdit.CommonName);
            if (SiteSettingsManager.GetSiteSettings().EnableNickName)
                userToEdit.NickName = Request.Form.GetString("nickName", userToEdit.NickName);
            userToEdit.Gender = (Genders)Request.Form.GetInt("gender", (int)userToEdit.Gender);
            try
            {
                userToEdit.Birthday = DateTime.Parse(Request.Form.GetString("birthday", userToEdit.Birthday.ToString()));
            }
            catch
            {
            }
            userToEdit.NowAreaCode = Request.Form.GetString("nowAreaCode", userToEdit.NowAreaCode);
            userToEdit.HomeAreaCode = Request.Form.GetString("homeAreaCode", userToEdit.HomeAreaCode); ;
            userToEdit.Degree = Request.Form.GetInt("degree", Convert.ToInt32(userToEdit.Degree));
            userToEdit.School = Request.Form.GetString("school", userToEdit.School);
            userToEdit.SchoolID = Request.Form.GetInt("schoolID", Convert.ToInt32(userToEdit.SchoolID));
            userToEdit.SelfAppraisal = Request.Form.GetString("selfAppraisal", userToEdit.SelfAppraisal);

            userToEdit.Interest = Request.Form.GetString("interest", userToEdit.Interest).Replace('，', ',').Replace('、', ',');
            userToEdit.Expertise = Request.Form.GetString("expertise", userToEdit.Expertise).Replace('，', ',').Replace('、', ',');

            userToEdit.PersonalHomepage = Request.Form.GetString("personalHomepage", userToEdit.PersonalHomepage);

            userToEdit.Stature = Request.Form.GetInt("stature", userToEdit.Stature);
            userToEdit.Weight = Request.Form.GetInt("weight", userToEdit.Weight);
            userToEdit.Shape = Request.Form.GetInt("shape", userToEdit.Shape);
            userToEdit.Blood = Request.Form.GetInt("blood", userToEdit.Blood);
            userToEdit.Smoke = Request.Form.GetInt("smoke", userToEdit.Smoke);
            userToEdit.Drink = Request.Form.GetInt("drink", userToEdit.Drink);
            userToEdit.Marriage = Request.Form.GetInt("marriage", userToEdit.Marriage);
            userToEdit.Health = Request.Form.GetInt("health", userToEdit.Health);
            userToEdit.NowSalary = Request.Form.GetInt("nowSalary", userToEdit.NowSalary);

            userToEdit.PublicEmail = Request.Form.GetString("publicEmail", userToEdit.PublicEmail);
            userToEdit.Mobile = Request.Form.GetString("mobile", userToEdit.Mobile);
            userToEdit.HomeTel = Request.Form.GetString("homeTel", userToEdit.HomeTel);
            userToEdit.HomeFax = Request.Form.GetString("homeFax", userToEdit.HomeFax);
            userToEdit.PersonAddress = Request.Form.GetString("personAddress", userToEdit.PersonAddress);
            userToEdit.PersonPostcode = Request.Form.GetString("personPostcode", userToEdit.PersonPostcode);
            userToEdit.MsnIM = Request.Form.GetString("msnIM", userToEdit.MsnIM);
            userToEdit.QQIM = Request.Form.GetString("qqIM", userToEdit.QQIM);
            userToEdit.YahooIM = Request.Form.GetString("yahooIM", userToEdit.YahooIM);
            userToEdit.SkypeIM = Request.Form.GetString("skypeIM", userToEdit.SkypeIM);

            userToEdit.Title = Request.Form.GetString("title", userToEdit.Title);
            userToEdit.TitleLevel = Request.Form.GetInt("titleLevel", userToEdit.TitleLevel);
            userToEdit.JobSort = Request.Form.GetString("jobSort", userToEdit.JobSort);
            userToEdit.CompanyName = Request.Form.GetString("companyName", userToEdit.CompanyName);
            userToEdit.CompanyTrade = Request.Form.GetInt("companyTrade", userToEdit.CompanyTrade);
            userToEdit.CompanyAreaCode = Request.Form.GetString("companyAreaCode", userToEdit.CompanyAreaCode);
            userToEdit.CompanyFax = Request.Form.GetString("companyFax", userToEdit.CompanyFax);
            userToEdit.CompanyAddress = Request.Form.GetString("companyAddress", userToEdit.CompanyAddress);
            userToEdit.CompanyPostcode = Request.Form.GetString("companyPostcode", userToEdit.CompanyPostcode);
            userToEdit.CompanyWebAddress = Request.Form.GetString("companyWebAddress", userToEdit.CompanyWebAddress);
            userToEdit.WorkTel = Request.Form.GetString("workTel", userToEdit.WorkTel);

            #region 编码

            switch (type)
            {
                case (int)PersonProfileType.BasicInfo:
                    userToEdit.CommonName = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.CommonName);
                    userToEdit.NickName = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.NickName);
                    userToEdit.SelfAppraisal = Formatter.FormatMultiLinePlainTextForStorage(userToEdit.SelfAppraisal, true);
                    userToEdit.Interest = Formatter.FormatMultiLinePlainTextForStorage(userToEdit.Interest, false);
                    userToEdit.Expertise = Formatter.FormatMultiLinePlainTextForStorage(userToEdit.Expertise, false);
                    userToEdit.PersonalHomepage = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.PersonalHomepage);
                    break;
                case (int)PersonProfileType.PersonContact:
                    userToEdit.PublicEmail = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.PublicEmail);
                    userToEdit.HomeTel = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.HomeTel);
                    userToEdit.MsnIM = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.MsnIM);
                    userToEdit.QQIM = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.QQIM);
                    userToEdit.YahooIM = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.YahooIM);
                    break;
                case (int)PersonProfileType.BusinessContact:
                    userToEdit.CardID = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.CardID);
                    userToEdit.WorkTel = Formatter.FormatSingleLinePlainTextForStorage(userToEdit.WorkTel);
                    break;
                default: break;
            }
            userToEdit.AvatarUrl = WebUtils.HtmlEncode(userToEdit.AvatarUrl);

            #endregion

            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.EnableSelectUserRankType)
                userToEdit.RankType = (UserRankTypes)Request.Form.GetInt("rankType", (int)userToEdit.RankType);

            // Update the user
            CreateUpdateUserStatuses status = CreateUpdateUserStatuses.Updated;
            status = Users.UpdateUser(userToEdit);
            if (status != CreateUpdateUserStatuses.Updated)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (status == CreateUpdateUserStatuses.DuplicateEmailAddress)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_DuplicateEmailAddress", applicationID);
                }
                else
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileFailed", applicationID);
                }
                return RedirectToAction("EditProfile");
            }
            if (prevAccountStatus.HasValue)
            {
                if (prevAccountStatus.Value == UserAccountStatuses.ApprovalPending &&
                    userToEdit.AccountStatus == UserAccountStatuses.Approved)
                {
                    UserEmailsProvider.Instance().UserAccountApproved(userToEdit);
                }

                if (prevAccountStatus.Value == UserAccountStatuses.ApprovalPending &&
                    userToEdit.AccountStatus == UserAccountStatuses.Disapproved)
                {
                    UserEmailsProvider.Instance().UserAccountRejected(userToEdit, CurrentUser);
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileSucceed", applicationID);

            return RedirectToAction("EditProfile", new RouteValueDictionary { { "userDomainName", userDomainName }, { "type", (PersonProfileType)type } });
        }

        #region 企业

        //企业用户账号信息
        private ActionResult EditCompanyUserProfile(User currentDomainUser)
        {
            SetPageTitle(currentDomainUser.UserName, GetResourceString("Title_EditCompanyUserProfile", applicationID));
            CompanyUser userToEdit = (CompanyUser)currentDomainUser;

            Dictionary<int, CodeItem> TimeZoneList = CodeItems.GetCodeItems(CodeSetCodes.Instance().TimeZone());
            ViewData["TimeZone"] = TimeZoneList;

            SiteSettings setting = SiteSettingsManager.GetSiteSettings();
            ViewData["SiteSetting"] = setting;

            Dictionary<string, Language> dictionary = Languages.GetEnabledLanguages();
            Dictionary<string, string> languages = dictionary.ToDictionary(n => n.Key, n => n.Value.NativeName);
            ViewData["languages"] = languages;
            return View("Pages/Settings/AccountInfo.aspx", "Masters/Settings.master", userToEdit);
        }

        //编辑企业用户基本信息
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditCompanyBasicInfo(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(currentDomainUser.UserName, GetResourceString("Title_EditCompanyUserProfile", applicationID));
            ViewData["CompanyPropertys"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().CompanyProperty());
            ViewData["Trades"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().Trade());
            ViewData["AreaDepth"] = 1;
            ViewData["Funds"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().Fund());
            ViewData["EmployeeCounts"] = CodeItems.GetCodeItems(CodeSetCodes.Instance().EmployeeCount());

            CompanyUser userToEdit = currentDomainUser as CompanyUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            SiteSettings setting = SiteSettingsManager.GetSiteSettingsForAdmin();
            ViewData["SiteSetting"] = setting;

            return View("Pages/Settings/BasicInfo.aspx", "Masters/Settings.master", userToEdit);
        }

        //编辑企业用户Logo
        public ActionResult EditCompanyAvatarFile(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(currentDomainUser.UserName, GetResourceString("Title_EditCompanyUserProfile", applicationID));
            CompanyUser userToEdit = currentDomainUser as CompanyUser;
            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            return View("Pages/Settings/AvatarFile.aspx", "Masters/Settings.master", userToEdit);
        }

        /// <summary>
        /// 更新企业用户资料
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateCompanyProfile(string userDomainName, UserAccountStatuses? prevAccountStatus)
        {
            User domainUser = Users.GetUser(userDomainName, true);
            CompanyUser userToEdit = domainUser as CompanyUser;
            string returnUrl = Request.Form.GetString("returnUrl", string.Empty);

            if (userToEdit == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            string privateEmail = Request.Form.GetString("privateEmail", userToEdit.PrivateEmail);
            if (!userToEdit.PrivateEmail.Equals(privateEmail, StringComparison.CurrentCultureIgnoreCase))
            {
                string message = string.Empty;
                if (Globals.ValidateEmail(privateEmail, out message))
                {
                    userToEdit.IsEmailVerified = false;
                    userToEdit.PrivateEmail = privateEmail;
                    UserEmailsProvider.Instance().UserCreateValidateEmail(userToEdit);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = message;
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = message;
                    return Redirect(returnUrl);
                }


            }


            #region 属性
            userToEdit.Timezone = Request.Form.GetInt("TimeZone", (int)userToEdit.Timezone);
            userToEdit.Language = Request.Form.GetString("languages", userToEdit.Language);
            userToEdit.EnableAvatar = Request.Form.GetBool("enableAvatar", userToEdit.EnableAvatar);
            userToEdit.EnableDisplayInMemberList = Request.Form.GetBool("enableDisplayInMemberList", userToEdit.EnableDisplayInMemberList);
            string Licence = Request.Form.GetString("Licence", userToEdit.Licence).Trim();
            int Trades = Request.Form.GetInt("Trades", userToEdit.Trade);
            string areaCode = Request.Form.GetString("areaCode", userToEdit.AreaCode);
            int CompanyPropertys = Request.Form.GetInt("CompanyPropertys", userToEdit.CompanyProperty);
            int Funds = Request.Form.GetInt("Funds", userToEdit.Fund);
            int EmployeeCounts = Request.Form.GetInt("EmployeeCounts", userToEdit.EmployeeNum);
            string Linkman = Request.Form.GetString("Linkman", userToEdit.Linkman).Trim();
            string LinkmanDuty = Request.Form.GetString("LinkmanDuty", userToEdit.LinkmanDuty).Trim();
            string LinkTel = Request.Form.GetString("LinkTel", userToEdit.LinkTel).Trim();
            string Fax = Request.Form.GetString("Fax", userToEdit.Fax).Trim();
            string Address = Request.Form.GetString("Address", userToEdit.Address).Trim();
            string Intro = Request.Form.GetString("Intro", userToEdit.Intro).Trim();
            string CommonName = Request.Form.GetString("CommonName", userToEdit.CommonName).Trim();
            string UserDomainDescription = Request.Form.GetString("UserDomainDescription", userToEdit.UserDomainDescription).Trim();
            userToEdit.IsShowLogo = Request.Form.GetBool("IsShowLogo", userToEdit.IsShowLogo);
            userToEdit.Licence = Licence;
            userToEdit.Trade = Trades;
            userToEdit.AreaCode = areaCode;
            userToEdit.CompanyProperty = CompanyPropertys;
            userToEdit.CommonName = CommonName;
            userToEdit.Fund = Funds;
            userToEdit.EmployeeNum = EmployeeCounts;
            userToEdit.Linkman = Linkman;
            userToEdit.LinkmanDuty = LinkmanDuty;
            userToEdit.LinkTel = LinkTel;
            userToEdit.Fax = Fax;
            userToEdit.Address = Address;
            userToEdit.Intro = Intro;
            userToEdit.UserDomainDescription = UserDomainDescription;
            #endregion

            #region 上传图像标示

            //if (Request.Files != null && Request.Files.Count > 0)
            //{
            //    System.Web.HttpPostedFileBase avatarFile = null;
            //    avatarFile = Request.Files["avatarFile"];
            //    if (avatarFile != null && !string.IsNullOrEmpty(avatarFile.FileName))
            //    {
            //        if (avatarFile.ContentLength > 512 * 1024)
            //        {
            //            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            //            TempData["StatusMessageContent"] = GetResourceString("Message_MaxAvatarFileLength", applicationID);
            //            return RedirectToAction("EditProfile", new RouteValueDictionary { { "userDomainName", userDomainName } });
            //        }
            //        if (avatarFile.ContentType.IndexOf("image") == -1)
            //        {
            //            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            //            TempData["StatusMessageContent"] = GetResourceString("Message_NoAvatarFile", applicationID);
            //            return RedirectToAction("EditProfile", new RouteValueDictionary { { "userDomainName", userDomainName } });
            //        }
            //        else
            //        {
            //            try
            //            {
            //                using (Bitmap bitmapOriginal = new Bitmap(avatarFile.InputStream))
            //                {
            //                    if (bitmapOriginal.Width > 480 || bitmapOriginal.Height > 60)
            //                    {
            //                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
            //                        TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_AvatarFileMaxHeiAndWit", applicationID), 480, 60);
            //                        return RedirectToAction("EditProfile", new RouteValueDictionary { { "userDomainName", userDomainName } });
            //                    }
            //                }
            //            }
            //            catch { }
            //        }
            //        Avatars.UploadCompanyLogo(domainUser.UserID, avatarFile.InputStream);
            //    }
            //}

            #endregion

            #region 更新操作

            // Update the user
            CreateUpdateUserStatuses status = CreateUpdateUserStatuses.Updated;
            status = Users.UpdateUser(userToEdit);
            if (status != CreateUpdateUserStatuses.Updated)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (status == CreateUpdateUserStatuses.DuplicateEmailAddress)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_DuplicateEmailAddress", applicationID);
                }
                else
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileFailed", applicationID);
                }
                return Redirect(returnUrl);
            }
            if (prevAccountStatus.HasValue)
            {
                // User Approval
                if (prevAccountStatus.Value == UserAccountStatuses.ApprovalPending &&
                    userToEdit.AccountStatus == UserAccountStatuses.Approved)
                {
                    UserEmailsProvider.Instance().UserAccountApproved(userToEdit);
                }

                if (prevAccountStatus.Value == UserAccountStatuses.ApprovalPending &&
                    userToEdit.AccountStatus == UserAccountStatuses.Disapproved)
                {
                    UserEmailsProvider.Instance().UserAccountRejected(userToEdit, CurrentUser);
                }
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdateProfileSucceed", applicationID);

            #endregion

            return Redirect(returnUrl);
        }

        #endregion

        /// <summary>
        /// 显示上传头像页
        /// </summary>
        public ActionResult UploadAvatar(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            if (currentDomainUser.HasAvatar)
                ViewData["canCropAvatar"] = true;

            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/Settings/UploadAvatar.aspx", "Masters/Settings.master", currentDomainUser);
        }

        /// <summary>
        /// 裁剪头像
        /// </summary>
        public ActionResult CropAvatar(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            StatusMessageData statusMessageData = null;
            if (!currentDomainUser.HasAvatar)
            {
                statusMessageData = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_CannotEditForNotUploadFile", applicationID));
                TempData["StatusMessageData"] = statusMessageData;
                return RedirectToAction("UploadAvatar", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }
            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/Settings/CropAvatar.aspx", "Masters/Settings.master");
        }

        /// <summary>
        /// 裁剪头像处理
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ContentResult CropAvatarAction(string userDomainName)
        {
            string jsoncall = Request.QueryString.GetString("jsoncallback", string.Empty);
            System.Web.Script.Serialization.JavaScriptSerializer serializer = new System.Web.Script.Serialization.JavaScriptSerializer();
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Content(string.Format("{0}({1})", jsoncall, serializer.Serialize(new { Status = "Error", Message = string.Format(GetResourceString("Exception_Pattern_UserNotExist", applicationID), userDomainName) })));

            }

            //StatusMessageData statusMessageData = null;

            bool autoCrop = Request.QueryString.GetBool("autoCrop", true);
            if (autoCrop)
            {
                Avatars.CropAvatar(currentDomainUser.UserID);
            }
            else
            {
                SiteSettings siteSettings = SiteSettingsManager.GetSiteSettings();
                float cropImageWidth = Request.QueryString.GetFloat("cropImageWidth", siteSettings.AvatarWidth);
                float cropImageHeight = cropImageWidth / siteSettings.AvatarWidth * siteSettings.AvatarHeight;
                float cropImageX = Request.QueryString.GetFloat("cropImageX", 0F);
                float cropImageY = Request.QueryString.GetFloat("cropImageY", 0F);

                Avatars.CropAvatar(currentDomainUser.UserID, cropImageWidth, cropImageHeight, cropImageX, cropImageY);

                //statusMessageData = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AvatarFileCutOffSucceed", applicationID));
                //TempData["StatusMessageData"] = statusMessageData;
                //return RedirectToAction("CropAvatar");
            }
            return Content(string.Format("{0}({1})", jsoncall, serializer.Serialize(new { Status = "Success", Message = GetResourceString("Message_AvatarFileCutOffSucceed", applicationID) })));
        }

        /// <summary>
        /// 显示修改密码页
        /// </summary>
        public ActionResult ChangePassword(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            if (CurrentUser != null && currentDomainUser.UserID != CurrentUser.UserID && CurrentUser.IsContentAdministrator)
            {
                ViewData["isShowOldPassword"] = false;//管理员修改其他人的密码
            }
            ViewData["UserType"] = currentDomainUser.UserType;
            return View("Pages/Settings/ChangePassword.aspx", "Masters/Settings.master");
        }

        /// <summary>
        /// 处理更新密码
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdatePassword(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }
            string errorMessage = "";
            string newPassword = Request.Form.GetString("newPassword1", string.Empty);
            string oldPassword = Request.Form.GetString("oldPassword", string.Empty);
            if (CurrentUser.UserID != currentDomainUser.UserID && CurrentUser.IsContentAdministrator)
            {
                oldPassword = string.Empty;
            }
            if (!Globals.ValidatePassword(newPassword, out errorMessage))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = errorMessage;
                return RedirectToAction("ChangePassword", new RouteValueDictionary { { "userDomainName", userDomainName } });
            }

            if (currentDomainUser.ChangePassword(oldPassword, newPassword, true))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdatePasswordSucceed", applicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdatePasswordFailed", applicationID);
            }
            return RedirectToAction("ChangePassword", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 账号绑定
        /// </summary>
        public ActionResult AccountBinding(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            Dictionary<int, AccountBinding> dictionary = AccountBindings.GetAccountBindings(currentDomainUser.UserID);
            return View("Pages/Settings/AccountBinding.aspx", "Masters/Settings.master", dictionary);
        }
        /// <summary>
        /// 取消绑定
        /// </summary>
        public ActionResult DeleteBinding(string userDomainName, int accountType)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            AccountBindings.DeleteAccountBinding(currentDomainUser.UserID, accountType);
            return RedirectToAction("AccountBinding", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #region 广告

        #region Show

        /// <summary>
        /// 管理广告
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="pageIndex"></param>
        /// <returns></returns>

        public ActionResult ManageAdvertisings(string userDomainName, int? pageIndex)
        {
            if (!SiteSettingsManager.GetSiteSettings().EnableUserAD)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            SetPageTitle(userDomainName, GetResourceString("Title_ManageAdvertisings", applicationID));
            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            PagingDataSet<Advertising> pds = Advertisings.GetAdvertisings(10, pageIndex.HasValue ? pageIndex.Value : 1, PresentAreaIDs.Instance().UserDomain(), currentDomainUser.UserID);
            return View("Pages/Settings/ManageAdvertisings.aspx", "Masters/Settings.master", pds);
        }

        /// <summary>
        /// 编辑广告
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <param name="advertisingID"></param>
        /// <returns></returns>

        public ActionResult EditAdvertising(string userDomainName, int? advertisingID)
        {
            if (!SiteSettingsManager.GetSiteSettings().EnableUserAD)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            Advertising advertising = new Advertising();

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                SetPageTitle(userDomainName, GetResourceString("Title_EditAdvertising", applicationID));
                advertising = Advertisings.GetAdvertising(advertisingID.Value, true);
                if (advertising == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format("广告ID为：“{0}”的广告实体不存在", advertisingID)));
                }
            }
            else
            {
                SetPageTitle(userDomainName, GetResourceString("Title_CreateAdvertising", applicationID));
                if (TempData["Advertising"] != null)
                {
                    advertising = TempData["Advertising"] as Advertising;
                }
                else
                {
                    advertising.EffectiveDateFrom = DateTime.Now;
                    advertising.EffectiveDateTo = advertising.EffectiveDateFrom.AddMonths(1);
                }
            }

            return View("Pages/Settings/EditAdvertising.aspx", "Masters/Settings.master", advertising);
        }

        /// <summary>
        /// 广告投放时显示的广告位列表
        /// </summary>
        /// <param name="advertisingID"></param>
        /// <param name="width"></param>
        /// <returns></returns>
        public ActionResult Control_ListAdvertisingPositions(string userDomainName, int? advertisingID, int? width)
        {
            if (!SiteSettingsManager.GetSiteSettings().EnableUserAD)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            List<AdvertisingPosition> position = Advertisings.GetAdvertisingPositions(PresentAreaIDs.Instance().UserDomain());

            List<int> widthList = Advertisings.GetAdvertisingPositionWidths();
            ViewData["widthList"] = widthList;

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                Advertising advertising = Advertisings.GetAdvertising(advertisingID.Value, false);
                if (advertising == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_AdvertisingNotExist", applicationID), advertisingID)));

                }

                ViewData["selectedPosition"] = advertising.UsedAdvertisingPositions;
            }

            if (width.HasValue && width.Value > 0)
            {
                ViewData["currentWidth"] = width.Value;
            }

            return PartialView("Controls/Settings/ListAdvertisingPositions.ascx", position);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建更新广告
        /// </summary>
        /// <param name="advertisingID"></param>
        /// <returns></returns>
        public ActionResult CreateUpdateAdvertising(string userDomainName, int? advertisingID)
        {
            if (!SiteSettingsManager.GetSiteSettings().EnableUserAD)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }


            Advertising advertising = new Advertising();

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                advertising = Advertisings.GetAdvertising(advertisingID.Value, true);
                if (advertising == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_AdvertisingNotExist", applicationID), advertisingID)));
                }
            }

            advertising.AdvertisingName = Request.Form.GetString("advertisingName", string.Empty).Trim();
            advertising.AdvertisingType = (AdvertisingTypes)Request.Form.GetInt("advertisingType", 0);
            advertising.IsEnabled = Request.Form.GetBool("isEnabled", true);
            advertising.PresentAreaID = PresentAreaIDs.Instance().UserDomain();
            advertising.OwnerID = currentDomainUser.UserID;
            advertising.Script = Request.Form.GetString("script", string.Empty).Trim();

            DateTime datetime;
            if (DateTime.TryParse(Request.Form.GetString("effectiveDateFrom", string.Empty).Trim(), out datetime))
            {
                advertising.EffectiveDateFrom = datetime;
            }
            else
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                {
                    TempData["Advertising"] = advertising;
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = GetResourceString("Message_AdCreateDateFromError", applicationID);
                return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "userDomainName", userDomainName }, { "advertisingID", advertisingID } });
            }

            if (DateTime.TryParse(Request.Form.GetString("effectiveDateTo", string.Empty).Trim(), out datetime))
            {
                advertising.EffectiveDateTo = datetime;
            }
            else
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                {
                    TempData["Advertising"] = advertising;
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = GetResourceString("Message_AdCreateDateToError", applicationID);
                return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "userDomainName", userDomainName }, { "advertisingID", advertisingID } });
            }

            if (advertising.EffectiveDateFrom > advertising.EffectiveDateTo)
            {
                if (!advertisingID.HasValue || advertisingID.Value <= 0)
                {
                    TempData["Advertising"] = advertising;
                }

                TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                TempData["StatusMessageContent"] = GetResourceString("Message_AdCreateDateFromShouldSmallerThanDateTo", applicationID);
                return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "userDomainName", userDomainName }, { "advertisingID", advertisingID } });
            }

            if (advertisingID.HasValue && advertisingID.Value > 0)
            {
                Advertisings.UpdateAdvertising(advertising);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            }
            else
            {
                try
                {
                    Advertisings.CreateAdvertising(advertising);
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                }
                catch (Exception)
                {
                    TempData["Advertising"] = advertising;
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_AdCreateError", applicationID);
                }
            }

            string[] positionCodes = null;
            List<string> positionCodeList = new List<string>();
            //bool isError = false;
            if (!string.IsNullOrEmpty(Request.Form.GetString("positionCheckBoxGroup", string.Empty)))
            {
                positionCodes = Request.Form.GetString("positionCheckBoxGroup", string.Empty).Split(',');
                if (positionCodes != null && positionCodes.Length > 0)
                {
                    foreach (var positionCode in positionCodes)
                    {
                        positionCodeList.Add(positionCode);
                    }
                }
            }

            Advertisings.UpdatePositionsOfAdvertising(advertising.AdvertisingID, positionCodeList);

            return RedirectToAction("EditAdvertising", new RouteValueDictionary { { "userDomainName", userDomainName }, { "advertisingID", advertisingID } });
        }

        /// <summary>
        /// 删除广告
        /// </summary>
        /// <returns></returns>
        public ActionResult DeleteAdvertisings(string userDomainName)
        {
            if (!SiteSettingsManager.GetSiteSettings().EnableUserAD)
            {
                throw new SPBException(ExceptionTypes.Instance().AccessDenied());
            }

            User currentDomainUser = Users.GetUser(userDomainName, true);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_UserNotExist", applicationID)));
            }

            string[] advertisingCodes = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("advertisingCheckBoxGroup", string.Empty)))
            {
                advertisingCodes = Request.Form.GetString("advertisingCheckBoxGroup", string.Empty).Split(',');
                if (advertisingCodes != null && advertisingCodes.Length > 0)
                {
                    List<int> advertisingIDs = new List<int>();
                    foreach (var id in advertisingCodes)
                    {
                        if (Convert.ToInt32(id) > 0)
                            advertisingIDs.Add(Convert.ToInt32(id));
                    }
                    Advertisings.DeleteAdvertisings(PresentAreaIDs.Instance().UserDomain(), advertisingIDs.ToArray());
                }
            }
            return RedirectToAction("ManageAdvertisings", new RouteValueDictionary(CurrentRouteValueDictionary));
        }

        #endregion
        #endregion

        #region 外观
        /// <summary>
        /// 显示菜单设置页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageUserNavigations(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            IList<OwnerNavigation> userNavigations = Navigations.GetOwnerNavigations(ThemeManager.PresentAreaID, currentDomainUser.UserID, true);
            return View("Pages/Settings/ManageUserNavigations.aspx", "Masters/Settings.master", userNavigations);
        }

        /// <summary>
        /// 更新菜单设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateUserNavigations(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            int count = Request.Form.GetInt("totalOptionCount", 0);
            List<OwnerNavigation> ownerNavigations = new List<OwnerNavigation>();
            for (int i = 0; i < count; i++)
            {
                OwnerNavigation ownerNavigation = null;
                int navID = Request.Form.GetInt(string.Format("navID_{0}", i), 0);
                int defaultNavID = Request.Form.GetInt(string.Format("DefaultNavID_{0}", i), 0);
                if (navID > 0)
                {
                    ownerNavigation = Navigations.GetOwnerNavigation(navID, true);
                    if (ownerNavigation != null)
                    {
                        ownerNavigation.NavigationText = Request.Form.GetString(string.Format("navigationText_{0}", i), ownerNavigation.NavigationText);
                        ownerNavigation.NavigationTarget = Request.Form.GetBool(string.Format("navigationTarget_{0}", i), false) ? "_blank" : "_self";
                        ownerNavigation.NavigationUrl = Request.Form.GetString(string.Format("navigationLink_{0}", i), ownerNavigation.NavigationUrl);
                        ownerNavigation.DisplayOrder = Request.Form.GetInt(string.Format("displayOrder_{0}", i), ownerNavigation.DisplayOrder);
                        if (!ownerNavigation.IsLocked)
                            ownerNavigation.IsEnabled = Request.Form.GetBool(string.Format("isEnabled_{0}", i), false);
                        ownerNavigation.NavigationText = Formatter.FormatSingleLinePlainTextForStorage(ownerNavigation.NavigationText);
                    }
                }
                else
                {
                    if (defaultNavID > 0)
                    {
                        Navigation navigation = Navigations.GetDefaultNavigation(defaultNavID, false);
                        if (navigation != null)
                            ownerNavigation = new OwnerNavigation(currentDomainUser.UserID, navigation);
                    }
                    else
                    {
                        ownerNavigation = new OwnerNavigation();
                        ownerNavigation.OwnerID = currentDomainUser.UserID;
                        ownerNavigation.NavigationType = NavigationTypes.HyperLink;
                        if (currentDomainUser.UserType == UserTypes.PersonUser)
                            ownerNavigation.PresentAreaID = ThemeManager.PresentAreaID;
                        else
                            ownerNavigation.PresentAreaID = PresentAreaIDs.Instance().CompanyUserDomain();
                    }

                    ownerNavigation.NavigationText = Request.Form.GetString(string.Format("navigationText_{0}", i), ownerNavigation.NavigationText);
                    ownerNavigation.NavigationUrl = Request.Form.GetString(string.Format("navigationLink_{0}", i), ownerNavigation.NavigationUrl);
                    if (string.IsNullOrEmpty(ownerNavigation.NavigationText))
                        continue;
                    ownerNavigation.NavigationTarget = Request.Form.GetBool(string.Format("navigationTarget_{0}", i), false) ? "_blank" : "_self";
                    ownerNavigation.DisplayOrder = Request.Form.GetInt(string.Format("displayOrder_{0}", i), ownerNavigation.DisplayOrder);
                    if (!ownerNavigation.IsLocked)
                        ownerNavigation.IsEnabled = Request.Form.GetBool(string.Format("isEnabled_{0}", i), ownerNavigation.IsEnabled);
                    ownerNavigation.NavigationText = Formatter.FormatSingleLinePlainTextForStorage(ownerNavigation.NavigationText);
                }
                ownerNavigations.Add(ownerNavigation);
            }
            Navigations.UpdateOwnerNavigations(currentDomainUser.UserType == UserTypes.PersonUser ? PresentAreaIDs.Instance().UserDomain() : PresentAreaIDs.Instance().CompanyUserDomain(), currentDomainUser.UserID, ownerNavigations);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            return RedirectToAction("ManageUserNavigations", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 显示编辑外链页
        /// </summary>
        public ActionResult Control_EditUserNavigation(string userDomainName, int? navigationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            OwnerNavigation userNavigation = null;
            if (navigationID.HasValue && navigationID.Value > 0)
            {
                userNavigation = Navigations.GetOwnerNavigation(navigationID.Value, true);
            }
            if (userNavigation == null)
                userNavigation = new OwnerNavigation();
            return PartialView("Controls/Settings/EditUserNavigation.ascx", userNavigation);
        }
        /// <summary>
        /// 创建或更新导航
        /// </summary>
        public ActionResult UpdateUserNavigation(string userDomainName, int? navigationID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            OwnerNavigation userNavigation = null;
            if (navigationID.HasValue && navigationID.Value > 0)
            {
                userNavigation = Navigations.GetOwnerNavigation(navigationID.Value, true);
            }
            else
            {
                userNavigation = new OwnerNavigation();
            }
            userNavigation.OwnerID = currentDomainUser.UserID;
            userNavigation.NavigationType = NavigationTypes.HyperLink;
            userNavigation.NavigationText = Request.Form.GetString("navigationText", string.Empty).Trim();
            userNavigation.NavigationUrl = Request.Form.GetString("navigationUrl", string.Empty).Trim();
            userNavigation.NavigationTarget = Request.Form.GetString("navigationTarget", string.Empty);
            userNavigation.IsEnabled = Request.Form.GetBool("isEnabled", false);
            if (navigationID.HasValue && navigationID.Value > 0)
            {
                Navigations.UpdateOwnerNavigation(userNavigation);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateUserNavigationSucceed", applicationID));
            }
            else
            {
                Navigations.CreateOwnerNavigation(userNavigation);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_CreateUserNavigationSucceed", applicationID));
            }
            return RedirectToAction("Control_EditUserNavigation", new RouteValueDictionary { { "userDomainName", userDomainName }, { "navigationID", userNavigation.OwnerNavigationID } });
        }
        /// <summary>
        /// 删除导航
        /// </summary>
        public ActionResult DeleteUserNavigation(string userDomainName, int ownerNavigationID)
        {
            if (ownerNavigationID > 0)
                Navigations.DeleteOwnerNavigation(ownerNavigationID);
            return new EmptyResult();
        }

        /// <summary>
        /// 空间装扮
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ModifyHome(string userDomainName)
        {
            User currentDomainUser = Users.GetUser(userDomainName, false);
            if (currentDomainUser.UserType != UserTypes.PersonUser)
                return Redirect(SiteUrls.Instance().ShowValidationMessage(ExceptionTypes.Instance().AccessDenied(), "当前用户不是个人用户"));

            SetPageTitle(userDomainName, GetResourceString("Title_ModifyHome", applicationID));
            return View("Pages/Settings/PersonUsers/ModifyHome.aspx");
        }

        #endregion

        #region 隐私

        #region 页面
        /// <summary>
        /// 显示基础隐私设置页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult PrivacySettings(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(userDomainName, GetResourceString("Title_PrivacySettings", applicationID));
            Dictionary<string, PrivacyItemSettings> userPrivacyItemSettingss = Privacys.GetUserPrivacyItemSettingss(currentDomainUser.UserID, true);
            List<PrivacyItem> items = Privacys.GetPrivacyItems(false);
            List<PrivacyItem> userProfileItems = new List<PrivacyItem>();
            List<PrivacyItem> actionItems = new List<PrivacyItem>();
            foreach (PrivacyItem item in items)
            {
                if (item.ItemKey.StartsWith("UserProfile"))
                    userProfileItems.Add(item);
                if (item.ItemKey.StartsWith("Action"))
                    actionItems.Add(item);
            }
            ViewData["UserProfileItems"] = userProfileItems;
            ViewData["ActionItems"] = actionItems;
            return View("Pages/Settings/PersonUsers/PrivacySettings.aspx", "Masters/Settings.master", userPrivacyItemSettingss);
        }
        /// <summary>
        /// 显示我阻止的用户列表页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageStopedUsers(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_ManageStopedUsers", applicationID));
            return View("Pages/Settings/PersonUsers/ManageStopedUsers.aspx", "Masters/Settings.master");
        }

        /// <summary>
        /// 显示我特许的用户列表页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageSpecialPrivacyUsers(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_ManageSpecialPrivacyUsers", applicationID));
            return View("Pages/Settings/PersonUsers/ManageSpecialPrivacyUsers.aspx", "Masters/Settings.master");
        }
        /// <summary>
        /// 显示特殊隐私设置页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult PrivacySpecialSettings(string userDomainName, int toUserID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            User toUser = Users.GetUser(toUserID, false);
            ViewData["ToUser"] = toUser;
            SetPageTitle(userDomainName, string.Format(GetResourceString("Title_PrivacySpecialSettings", applicationID), toUser.DisplayName));
            Dictionary<string, SpecialPrivacyItemSettings> dictionary = Privacys.GetUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUserID, true);
            List<PrivacyItem> items = Privacys.GetPrivacyItems(false);
            List<PrivacyItem> userProfileItems = new List<PrivacyItem>();

            List<PrivacyItem> actionItems = new List<PrivacyItem>();
            foreach (PrivacyItem item in items)
            {
                if (item.ItemKey.StartsWith("UserProfile"))
                    userProfileItems.Add(item);

                if (item.ItemKey.StartsWith("Action"))
                    actionItems.Add(item);
            }
            ViewData["UserProfileItems"] = userProfileItems;

            ViewData["ActionItems"] = actionItems;
            return View("Pages/Settings/PersonUsers/PrivacySpecialSettings.aspx", "Masters/Settings.master", dictionary);
        }

        /// <summary>
        /// 用户隐私权限设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DomainVisitPermission(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            PrivacyItemSettings privacyItemSettings = Privacys.GetPrivacyItemSettings(currentDomainUser.UserID, PrivacyItemKeys.Instance().UserDomainPrivacy(), true);
            ViewData["PrivacyItemSettings"] = privacyItemSettings;

            //选中用户
            List<User> selectedUsers = new List<User>();
            Dictionary<int, SpecialPrivacyUser> friendIds = Privacys.GetSpecialPrivacyUsers(currentDomainUser.UserID, true);
            if (friendIds != null && friendIds.Count > 0)
            {
                foreach (KeyValuePair<int, SpecialPrivacyUser> pair in friendIds)
                {
                    User user = Users.GetUser(pair.Key, false);
                    if (user != null)
                        selectedUsers.Add(user);
                }
            }

            //ViewData["selectedUsers"] = selectedUsers;

            return View("Pages/Settings/PersonUsers/DomainVisitPermission.aspx", "Masters/Settings.master");
        }

        /// <summary>
        /// 搜索用户
        /// </summary>
        public JsonResult SearchUsers()
        {
            string term = Request.QueryString.GetString("term", string.Empty);
            if (string.IsNullOrEmpty(term))
                return Json(new { }, JsonRequestBehavior.AllowGet);
            term = WebUtils.UrlDecode(term);
            PersonUserFullTextQuery personUserQuery = new PersonUserFullTextQuery();
            personUserQuery.Keyword = term;
            IList<PersonUser> users = PersonUserSearchManager.Instance().Search(CurrentUser, personUserQuery).Records;
            return Json(users.Select(n => n)
            .Where(
            n => (
            n != null
            && (n.CommonName.Contains(term)
            || n.NickName.Contains(term)
            || n.UserName.Contains(term))))
            .Select(n => new
            {
                userID = n.UserID,
                displayName = n.DisplayName,
                commonName = n.CommonName,
                nickName = n.NickName,
                userAvatarUrl = SiteUrls.Instance().UserAvatarUrl(n.UserID, AvatarSizeTypes.Small)
            }), JsonRequestBehavior.AllowGet);
        }

        #endregion

        #region 控件
        /// <summary>
        /// 我阻止的用户列表控件
        /// </summary>
        public ActionResult Control_ManageStopedUsers(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Dictionary<int, StopedUser> stopedUsers = Privacys.GetStopedUsers(currentDomainUser.UserID, true);
            List<User> listStopUsers = new List<User>();
            foreach (int toUserID in stopedUsers.Keys)
            {
                User toUser = Users.GetUser(toUserID, false);
                listStopUsers.Add(toUser);
            }
            return PartialView("Controls/Settings/ManageStopedUsers.ascx", listStopUsers);
        }
        /// <summary>
        /// 我特许的用户列表控件
        /// </summary>
        public ActionResult Control_ManageSpecialPrivacyUsers(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Dictionary<int, SpecialPrivacyUser> dictionary = Privacys.GetSpecialPrivacyUsers(currentDomainUser.UserID, true);
            List<User> specialPrivacyUsers = new List<User>();
            foreach (int toUserID in dictionary.Keys)
            {
                User toUser = Users.GetUser(toUserID, false);
                specialPrivacyUsers.Add(toUser);
            }
            return PartialView("Controls/Settings/ManageSpecialPrivacyUsers.ascx", specialPrivacyUsers);
        }

        /// <summary>
        /// 自定义设置隐私状态
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult Control_SetPrivacSettings(string userDomainName)
        {
            ViewData["PrivacyItemKey"] = Request.QueryString.GetString("privacyItemKey", string.Empty);
            ViewData["PrivacyDegree"] = Request.QueryString.GetString("privacyItemKey", string.Empty);
            return PartialView("Controls/PersonUsers/SetPrivacySettings.ascx");
        }
        #endregion

        #region 操作
        /// <summary>
        /// 更新空间者的基础隐私设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdatePrivacySettings(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            string reUrl = Request.Form.GetString("reUrl", string.Empty);

            Dictionary<string, PrivacyItemSettings> dictionary = Privacys.GetUserPrivacyItemSettingss(currentDomainUser.UserID, true);
            List<PrivacyItemSettings> list = new List<PrivacyItemSettings>();
            List<PrivacyItem> items = Privacys.GetPrivacyItems(false);
            foreach (var item in items)
            {
                if (item.ItemKey != PrivacyItemKeys.Instance().Action_ReceiveFriendInvitation())
                    dictionary[item.ItemKey].PrivacyDegree = (PrivacyDegree)Request.Form.GetInt(item.ItemKey, 1);
                else
                {
                    bool receiveFriendInvitationValue = Request.Form.GetBool(item.ItemKey, true);

                    dictionary[item.ItemKey].PrivacyDegree = receiveFriendInvitationValue ? PrivacyDegree.Public : PrivacyDegree.Private;
                }
                list.Add(dictionary[item.ItemKey]);
            }
            Privacys.UpdateUserPrivacySettingss(currentDomainUser.UserID, list);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdatePrivacySettingsSucceed", applicationID);
            if (!String.IsNullOrEmpty(reUrl) && reUrl.Length > 0)
            {
                return Redirect(reUrl);
            }
            return RedirectToAction("PrivacySettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }
        /// <summary>
        /// 在非隐私管理页设置隐私状态
        /// </summary>
        /// <param name="returnUrl"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SetPrivacySettings(string userDomainName)
        {

            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            string privacyItemKey = Request.QueryString.GetString("privacyItemKey", string.Empty);
            string selectedUserIDs = Request.QueryString.GetString("selectedUserIDs", string.Empty);
            string selectedStopUserIDs = Request.QueryString.GetString("selectedStopUserIDs", string.Empty);
            PrivacyDegree privacyDegree = (PrivacyDegree)Request.QueryString.GetInt("privacyDegree", 1);
            privacyDegree = privacyDegree > 0 ? privacyDegree : PrivacyDegree.Private;
            Dictionary<string, PrivacyItemSettings> userPrivacyItemSettingss = Privacys.GetUserPrivacyItemSettingss(currentDomainUser.UserID, true);
            if (!string.IsNullOrEmpty(privacyItemKey))
            {
                userPrivacyItemSettingss[privacyItemKey].PrivacyDegree = privacyDegree;
                Privacys.UpdateUserPrivacySettingss(currentDomainUser.UserID, new List<PrivacyItemSettings>() { userPrivacyItemSettingss[privacyItemKey] });

                if (!string.IsNullOrEmpty(selectedUserIDs))
                {
                    UpdateSpecialPrivacyItemSettings(selectedUserIDs, privacyItemKey, true, currentDomainUser);
                }

                if (!string.IsNullOrEmpty(selectedStopUserIDs))
                {
                    UpdateSpecialPrivacyItemSettings(selectedStopUserIDs, privacyItemKey, false, currentDomainUser);
                }

            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;

            return RedirectToAction("Control_SetPrivacSettings", new RouteValueDictionary { { "userDomainName", userDomainName } });

        }

        /// <summary>
        /// 更新特殊隐私设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdatePrivacySpecialSettings(string userDomainName, int toUserID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            User toUser = Users.GetUser(toUserID, false);

            Dictionary<string, SpecialPrivacyItemSettings> dictionary = Privacys.GetUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUserID, true);
            List<SpecialPrivacyItemSettings> list = new List<SpecialPrivacyItemSettings>();
            List<PrivacyItem> items = Privacys.GetPrivacyItems(false);
            foreach (var item in items)
            {
                dictionary[item.ItemKey].IsAllowed = Request.Form.GetBool(item.ItemKey, true);
                list.Add(dictionary[item.ItemKey]);
            }

            Privacys.UpdateUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUserID, toUser.DisplayName, list);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_UpdatePrivacySettingsSucceed", applicationID), toUser.DisplayName);
            return RedirectToAction("PrivacySpecialSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "toUserID", toUserID } });
        }

        /// <summary>
        /// 删除一个被阻止的用户
        /// </summary>
        public ActionResult DeleteStopedUser(string userDomainName, int toUserID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Privacys.DeleteStopedUser(currentDomainUser.UserID, toUserID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteStopedUserSucceed", applicationID);

            return RedirectToAction("Control_ManageStopedUsers", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 删除一个特许用户
        /// </summary>
        public ActionResult DeleteSpecialPrivacyUser(string userDomainName, int toUserID)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Privacys.DeleteSpecialPrivacyUser(currentDomainUser.UserID, toUserID);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSpecialPrivacyUserSucceed", applicationID);
            return RedirectToAction("Control_ManageSpecialPrivacyUsers", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateSpecialPrivacyUsers(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            string selectedUserIDs = Request.Form.GetString("selectedUserIDs", string.Empty);
            if (!string.IsNullOrEmpty(selectedUserIDs))
            {
                string[] toUserIDs = selectedUserIDs.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (toUserIDs != null && toUserIDs.Length > 0)
                {
                    foreach (string toUserID in toUserIDs)
                    {
                        int userID = Convert.ToInt32(toUserID.Trim());
                        if (userID > 0)
                        {
                            User toUser = Users.GetUser(userID, false);
                            if (toUser == null)
                                continue;
                            Dictionary<string, SpecialPrivacyItemSettings> dictionary = Privacys.GetUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUser.UserID, true);
                            List<SpecialPrivacyItemSettings> list = new List<SpecialPrivacyItemSettings>();

                            dictionary[PrivacyItemKeys.Instance().UserDomainPrivacy()].IsAllowed = true;
                            list.Add(dictionary[PrivacyItemKeys.Instance().UserDomainPrivacy()]);

                            Privacys.UpdateUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUser.UserID, toUser.DisplayName, list);

                            PrivacyItemSettings settings = Privacys.GetPrivacyItemSettings(currentDomainUser.UserID, PrivacyItemKeys.Instance().UserDomainPrivacy(), true);
                            settings.PrivacyDegree = PrivacyDegree.Private;
                            Privacys.UpdateUserPrivacySettingss(currentDomainUser.UserID, new List<PrivacyItemSettings>() { settings });
                        }
                    }
                }
            }
            return RedirectToAction("DomainVisitPermission", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }


        #endregion

        #endregion

        #region 通知、动态
        /// <summary>
        /// 显示通知设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult NoticeSettings(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            SetPageTitle(userDomainName, GetResourceString("Title_NoticeSettings", applicationID));
            ViewData["Items"] = NoticeSettingsManager.GetNoticeItems(currentDomainUser.UserType);
            Dictionary<string, Dictionary<int, NoticeItem>> dicCodeSettings = new Dictionary<string, Dictionary<int, NoticeItem>>();
            foreach (var itemKey in NoticeProcessorFactory.Instance().NoticeProcessorsNames.Keys)
            {
                Dictionary<int, NoticeItem> noticeItemSettings = NoticeSettingsManager.GetUserNoticeSettings(itemKey, currentDomainUser.UserID, true);
                dicCodeSettings.Add(itemKey, noticeItemSettings);
            }
            ViewData["isEmailVerified"] = currentDomainUser.IsEmailVerified;
            ViewData["privateEmail"] = currentDomainUser.PrivateEmail;
            if (currentDomainUser.UserType == UserTypes.CompanyUser)
                ViewData["isCompanyUser"] = true;
            return View("Pages/Settings/NoticeSettings.aspx", "Masters/Settings.master", dicCodeSettings);
        }

        /// <summary>
        /// 更新通知设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateNoticeSettings(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            Dictionary<string, NoticeItem> noticeItems = NoticeSettingsManager.GetNoticeItems(currentDomainUser.UserType);
            Dictionary<int, NoticeItem> noticeItemSettings = null;
            string strName = string.Empty;
            int itemID = -1;
            foreach (var itemCode in NoticeProcessorFactory.Instance().NoticeProcessorsNames.Keys)
            {
                noticeItemSettings = new Dictionary<int, NoticeItem>();
                foreach (KeyValuePair<string, NoticeItem> item in noticeItems)
                {
                    strName = itemCode + "_" + item.Value.ItemKey;
                    itemID = Request.Form.GetInt(strName, -1);
                    if (itemID > 0)
                        noticeItemSettings[itemID] = item.Value;
                }

                NoticeSettingsManager.UpdateUserNoticeSettings(itemCode, currentDomainUser.UserID, noticeItemSettings);
            }

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_UpdateNoticeSettingsSucceed", applicationID);
            return RedirectToAction("NoticeSettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 显示动态设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ActivitySettings(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(userDomainName, GetResourceString("Title_ActivitySettings", applicationID));
            UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(currentDomainUser.UserID, true);
            ViewData["Items"] = Activities.GetActivityItems();
            IList<Friend> pds = Friends.GetTopFriends(currentDomainUser.UserID, true, ValueHelper.GetSafeSqlInt(int.MaxValue));
            Dictionary<int, string> myOddFriends = new Dictionary<int, string>();//去除忽略好友后的剩余朋友集合
            Dictionary<int, string> ignoredFriendUsers = userActivitySettings.IgnoredFriendUsersForReceiveActivities;
            foreach (Friend friend in pds)
            {
                if (!ignoredFriendUsers.ContainsKey(friend.FriendUserID) && friend.FriendUser != null) //如果忽略好友中不包含当前朋友，则加到剩余朋友集合中
                    myOddFriends[friend.FriendUserID] = friend.FriendUser.DisplayName;
            }
            ViewData["myOddFriends"] = myOddFriends;
            if (currentDomainUser.UserType == UserTypes.CompanyUser)
                ViewData["isCompanyUser"] = true;
            return View("Pages/Settings/ActivitySettings.aspx", "Masters/Settings.master", userActivitySettings);
        }

        /// <summary>
        /// 更新动态设置
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateActivitySettings(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserActivitySettings userActivitySettings = Activities.GetUserActivitySettings(currentDomainUser.UserID, true);
            Dictionary<string, ActivityItem> items = Activities.GetActivityItems();

            string strName = string.Empty;
            string strName2 = string.Empty;
            foreach (var item in items)
            {
                ActivityItemSettings settings = new ActivityItemSettings(item.Value);
                strName = "Send" + item.Key;
                strName2 = "Receive" + item.Key;

                settings.IsAllowSending = Request.Form.GetBool(strName, false);
                settings.IsAllowReceiving = Request.Form.GetBool(strName2, false);
                userActivitySettings.ActivityItemSettingss[item.Key] = settings;
            }

            //忽略的好友
            string ignoreFriendUsers = Request.Form.GetString("ignoreFriendUser", string.Empty);
            userActivitySettings.IgnoredFriendUsersForReceiveActivities = ConvertIgnoreFriendsToDictionary(ignoreFriendUsers);

            Activities.UpdateUserActivitySettings(userActivitySettings);
            if (TempData["StatusMessageData"] == null)
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateActivitySettingsSucceed", applicationID));
            return RedirectToAction("ActivitySettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }




        /// <summary>
        /// 将忽略用户及其名字从string类型转换成string
        /// </summary>
        /// <param name="p">忽略用户字符串</param>
        /// <returns>字典类型忽略用户</returns>
        protected Dictionary<int, string> ConvertIgnoreFriendsToDictionary(string p)
        {
            Dictionary<int, string> ignoreFriends = new Dictionary<int, string>();
            if (!string.IsNullOrEmpty(p))
            {
                string[] ignoreFriendsPairs = p.Split(',');
                foreach (string friendPair in ignoreFriendsPairs)
                {

                    string[] pair = friendPair.Split('&');
                    int key = Convert.ToInt32(pair[0]);
                    string value = pair[1].Trim();
                    if (ignoreFriends.Count <= 50)//系统限制最多只能存放50个被忽略用户                    
                        ignoreFriends[key] = value;
                    else
                    {
                        TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_ConvertIgnoreFriendsToDictionarySucceed", applicationID));
                        break;
                    }
                }

            }
            return ignoreFriends;
        }

        #endregion

        #region 友情连接
        /// <summary>
        /// 显示友情链接
        /// </summary>
        /// <param name="userDomainName">空间用户DomainName</param>
        /// <returns></returns>

        public ActionResult ManageUserLinks(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_ManageUserLinks", applicationID));
            User u = GetCurrentDomainUser(userDomainName);
            ViewData["UserLinks"] = UserLinkManager.Instance().GetLinksForManage(u.UserID);
            return View("Pages/Settings/ManageUserLinks.aspx", "Masters/Settings.master");
        }
        /// <summary>
        /// 显示添加友情链接
        /// </summary>
        /// <param name="userDomainName">空间用户DomainName</param>
        /// <returns></returns>
        public ActionResult Control_AddLink(string userDomainName)
        {
            User u = GetCurrentDomainUser(userDomainName);
            List<LinkCategory> listLinkCate = UserLinkManager.Instance().GetLinkCategoriesByOwnerID(u.UserID, true);
            if (listLinkCate == null || listLinkCate.Count <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NoLinkCategory", applicationID);
                return RedirectToAction("Control_AddLinkCategory", "UserDomainSettings");
            }
            ViewData["LinkCategories"] = listLinkCate;

            int statusMessageType = Request.QueryString.GetInt("StatusMessageType", -2);
            string statusMessageContent = WebUtils.UrlDecode(Request.QueryString.GetString("StatusMessageContent", string.Empty));
            if (statusMessageType > -2)
            {
                TempData["StatusMessageType"] = statusMessageType;
            }

            if (!string.IsNullOrEmpty(statusMessageContent))
            {
                TempData["StatusMessageContent"] = statusMessageContent;
            }

            return PartialView("Controls/Settings/EditUserLink.ascx");
        }
        /// <summary>
        /// 添加友情链接
        /// </summary>
        /// <param name="userDomainName">空间用户DomainName</param>
        /// <returns></returns>
        public ActionResult Control_AddLinkAction(string userDomainName)
        {
            User u = GetCurrentDomainUser(userDomainName);
            Link l = new Link();
            l.OwnerID = u.UserID;
            l.Title = Request.Form.GetString("Title", string.Empty);
            l.LinkHref = Request.Form.GetString("linkUrl", string.Empty);
            l.LinkTitle = Request.Form.GetString("linkTitle", string.Empty);
            l.CategoryID = Request.Form.GetInt("linkcate", 0);
            l.LinkType = (LinkTypes)Request.Form.GetInt("UserLinkTypes", 1);
            l.IsEnabled = Request.Form.GetBool("isenable", false);
            l.DisplayOrder = Request.Form.GetInt("displayorder", 0);

            if (!string.IsNullOrEmpty(l.Title))
                l.Title = Formatter.FormatSingleLinePlainTextForStorage(l.Title);
            if (!string.IsNullOrEmpty(l.LinkTitle))
                l.LinkTitle = Formatter.FormatSingleLinePlainTextForStorage(l.LinkTitle);

            UserLinkManager ulm = UserLinkManager.Instance();
            if (l.LinkType == LinkTypes.Image)
            {
                System.Web.HttpPostedFileBase picFile = null;
                if (Request.Files != null && Request.Files.Count > 0)
                    picFile = Request.Files[0];
                if (picFile == null)
                {
                    return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_AddLink", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode(GetResourceString("Message_PleaseChooseOnePic", applicationID)) } }));
                }
                if (picFile.ContentType.IndexOf("image") == -1)
                {
                    return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_AddLink", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode(GetResourceString("Message_PleaseChooseOnePic", applicationID)) } }));
                }
                ulm.CreateLink(l, picFile);
            }
            else
            {
                ulm.CreateLink(l, null);
            }

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_AddLink", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "StatusMessageType", (int)StatusMessageType.Success }, { "StatusMessageContent", WebUtils.UrlEncode(GetResourceString("Message_AddLinkActionSucceed", applicationID)) } }));
        }
        /// <summary>
        /// 删除友情链接
        /// </summary>
        /// <param name="userDomainName">空间用户DomainName</param>
        /// <returns></returns>
        public ActionResult DeleteLinkAction(string userDomainName)
        {
            string strLinks = Request.Form.GetString("LinkGroup", string.Empty);
            string[] UserLinks = strLinks.Split(',');
            List<int> LinkIDs = new List<int>();
            foreach (var item in UserLinks)
            {
                try
                {
                    LinkIDs.Add(Convert.ToInt32(item));
                }
                catch { }
            }
            UserLinkManager um = UserLinkManager.Instance();
            um.DeleteLinks(LinkIDs);
            return RedirectToAction("ManageUserLinks", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }
        /// <summary>
        /// 显示编辑友情连接
        /// </summary>
        /// <param name="userDomainName">空间用户DomainName</param>
        /// <param name="linkID">连接id</param>
        /// <returns></returns>
        public ActionResult Control_EditLink(string userDomainName, int linkID)
        {
            User u = GetCurrentDomainUser(userDomainName);
            List<LinkCategory> listLinkCate = UserLinkManager.Instance().GetLinkCategoriesByOwnerID(u.UserID, true);
            if (listLinkCate == null || listLinkCate.Count <= 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NoLinkCategory", applicationID);
                return RedirectToAction("Control_AddLinkCategory", "UserDomainSettings");
            }
            ViewData["LinkCategories"] = listLinkCate;
            Link userlink = UserLinkManager.Instance().GetLink(linkID, true);
            ViewData["UserLink"] = userlink;

            int statusMessageType = Request.QueryString.GetInt("StatusMessageType", -2);
            string statusMessageContent = WebUtils.UrlDecode(Request.QueryString.GetString("StatusMessageContent", string.Empty));
            if (statusMessageType > -2)
            {
                TempData["StatusMessageType"] = statusMessageType;
            }

            if (!string.IsNullOrEmpty(statusMessageContent))
            {
                TempData["StatusMessageContent"] = statusMessageContent;
            }

            return PartialView("Controls/Settings/EditUserLink.ascx");
        }

        /// <summary>
        /// 更新编辑友情连接
        /// </summary>
        /// <param name="userDomainName">空间用户DomainName</param>
        /// <param name="linkID">连接id</param>
        /// <returns></returns>
        public ActionResult Control_EditLinkAction(string userDomainName, int linkID)
        {
            UserLinkManager ulm = UserLinkManager.Instance();
            Link l = UserLinkManager.Instance().GetLink(linkID, true);
            if (l == null)
            {
                return Redirect(SPBConfig.Instance().MainSiteRootUrl + SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserHasNoLink", applicationID), userDomainName, linkID)));
            }
            l.Title = Request.Form.GetString("Title", string.Empty);
            l.LinkHref = Request.Form.GetString("linkUrl", string.Empty);
            l.LinkTitle = Request.Form.GetString("linkTitle", string.Empty);
            l.CategoryID = Request.Form.GetInt("linkcate", 0);

            l.IsEnabled = Request.Form.GetBool("isenable", false);
            l.DisplayOrder = Request.Form.GetInt("displayorder", 0);

            if (!string.IsNullOrEmpty(l.Title))
                l.Title = Formatter.FormatSingleLinePlainTextForStorage(l.Title);
            if (!string.IsNullOrEmpty(l.LinkTitle))
                l.LinkTitle = Formatter.FormatSingleLinePlainTextForStorage(l.LinkTitle);

            if (l.LinkType == LinkTypes.Image)
            {
                l.LinkType = (LinkTypes)Request.Form.GetInt("UserLinkTypes", 1);
                if (l.LinkType == LinkTypes.Image)
                {
                    System.Web.HttpPostedFileBase picFile = null;
                    if (Request.Files != null && Request.Files.Count > 0)
                        picFile = Request.Files[0];
                    if (picFile != null && !string.IsNullOrEmpty(picFile.FileName))
                    {
                        if (picFile.ContentType.IndexOf("image") == -1)
                        {
                            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditLink", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "linkID", linkID }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode(GetResourceString("Message_PleaseChooseOnePic", applicationID)) } }));
                        }
                        ulm.UpdateLink(l, picFile);
                    }
                    else
                    {
                        ulm.UpdateLink(l, null);
                    }
                }
                else
                {
                    ulm.UpdateLink(l, null);
                }
            }
            else
            {
                l.LinkType = (LinkTypes)Request.Form.GetInt("UserLinkTypes", 1);
                if (l.LinkType == LinkTypes.Image)
                {
                    System.Web.HttpPostedFileBase picFile = null;
                    if (Request.Files != null && Request.Files.Count > 0)
                        picFile = Request.Files[0];
                    if (picFile != null)
                    {
                        if (picFile.ContentType.IndexOf("image") == -1)
                        {
                            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditLink", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "linkID", linkID }, { "StatusMessageType", (int)StatusMessageType.Hint }, { "StatusMessageContent", WebUtils.UrlEncode(GetResourceString("Message_PleaseChooseOnePic", applicationID)) } }));
                        }
                        ulm.UpdateLink(l, picFile);
                    }
                }
                else
                {
                    ulm.UpdateLink(l, null);
                }
            }

            return Redirect(SPBConfig.Instance().MainSiteRootUrl + SPBUrlHelper.Action("Control_EditLink", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName }, { "linkID", linkID }, { "StatusMessageType", (int)StatusMessageType.Success }, { "StatusMessageContent", WebUtils.UrlEncode(GetResourceString("Message_EditLinkActionSucceed", applicationID)) } }));
        }

        #region 链接类别
        /// <summary>
        /// 显示连接类别
        /// </summary>
        /// <param name="userDomainName">空间用户userDomainName</param>
        /// <returns></returns>

        public ActionResult ManageUserLinkCategories(string userDomainName)
        {
            SetPageTitle(userDomainName, GetResourceString("Title_ManageUserLinkCategories", applicationID));
            User u = GetCurrentDomainUser(userDomainName);
            UserLinkManager ULManager = UserLinkManager.Instance();
            ViewData["LinkCategories"] = ULManager.GetLinkCategoriesByOwnerID(u.UserID, true);
            return View("Pages/Settings/ManageUserLinkCategories.aspx", "Masters/Settings.master");
        }
        /// <summary>
        /// 显示添加链接类别
        /// </summary>
        /// <param name="userDomainName">空间用户userDomainName</param>
        /// <returns></returns>
        public ActionResult Control_AddLinkCategory(string userDomainName)
        {
            return PartialView("Controls/Settings/EditUserLinkCategory.ascx");
        }
        /// <summary>
        /// 显示编辑连接类别
        /// </summary>
        /// <param name="CateID">类别id</param>
        /// <returns></returns>
        public ActionResult Control_EditLinkCategory(int CateID)
        {
            UserLinkManager ulm = UserLinkManager.Instance();
            LinkCategory LinkCate = ulm.GetLinkCategory(CateID, true);
            ViewData["LinkCate"] = LinkCate;
            return PartialView("Controls/Settings/EditUserLinkCategory.ascx");
        }
        /// <summary>
        /// 添加链接类别
        /// </summary>
        /// <param name="userDomainName">空间用户userDomainName</param>
        /// <returns></returns>
        public ActionResult Control_AddLinkCategoryAction(string userDomainName)
        {
            User u = GetCurrentDomainUser(userDomainName);
            LinkCategory lc = new LinkCategory();
            lc.OwnerID = u.UserID;
            lc.CategoryName = Request.Form.GetString("category", string.Empty);
            lc.DisplayOrder = Request.Form.GetInt("displayorder", -1);
            lc.Description = Request.Form.GetString("decript", string.Empty);

            if (!string.IsNullOrEmpty(lc.CategoryName))
                lc.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(lc.CategoryName);
            if (!string.IsNullOrEmpty(lc.Description))
                lc.Description = Formatter.FormatMultiLinePlainTextForStorage(lc.Description, true);

            if (lc.DisplayOrder < 0 || string.IsNullOrEmpty(lc.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_CreateUserLinkCategoryNameOrDispalyOrderError", applicationID);
            }
            else
            {
                UserLinkManager ulm = UserLinkManager.Instance();
                ulm.CreateUserLinkCategory(lc);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_AddLinkCategoryActionSucceed", applicationID);
            }
            return RedirectToAction("Control_AddLinkCategory", "UserDomainSettings");
        }

        /// <summary>
        /// 编辑连接类别
        /// </summary>
        /// <param name="CateID">类别id</param>
        /// <returns></returns>
        public ActionResult Control_EditLinkCategoryAction(int CateID)
        {
            LinkCategory lc = UserLinkManager.Instance().GetLinkCategory(CateID, true);
            if (lc == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoUserLinkCategory", applicationID), CateID)));
            }

            lc.CategoryName = Request.Form.GetString("category", string.Empty);
            lc.DisplayOrder = Request.Form.GetInt("displayorder", -1);
            lc.Description = Request.Form.GetString("decript", string.Empty);

            if (!string.IsNullOrEmpty(lc.CategoryName))
                lc.CategoryName = Formatter.FormatSingleLinePlainTextForStorage(lc.CategoryName);
            if (!string.IsNullOrEmpty(lc.Description))
                lc.Description = Formatter.FormatMultiLinePlainTextForStorage(lc.Description, true);

            if (lc.DisplayOrder < 0 || string.IsNullOrEmpty(lc.CategoryName))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_UpdateUserLinkCategoryNameOrDispalyOrderError", applicationID);
            }
            else
            {
                UserLinkManager ulm = UserLinkManager.Instance();
                ulm.UpdateCategory(lc);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_EditLinkCategoryActionSucceed", applicationID);
            }
            return RedirectToAction("Control_EditLinkCategory", "UserDomainSettings", new RouteValueDictionary { { "CateID", CateID } });
        }
        /// <summary>
        /// 删除连接类别
        /// </summary>
        /// <param name="userDomainName">空间用户userDomainName</param>
        /// <returns></returns>
        public ActionResult DeleteLinkCategoryAction(string userDomainName)
        {
            string strCategroies = Request.Form.GetString("CategroyGroup", string.Empty);
            string[] Categroies = strCategroies.Split(',');
            List<int> CateIDs = new List<int>();
            foreach (var item in Categroies)
            {
                try
                {
                    CateIDs.Add(Convert.ToInt32(item));
                }
                catch { }
            }
            UserLinkManager um = UserLinkManager.Instance();
            um.DeleteCategories(CateIDs);
            return RedirectToAction("ManageUserLinkCategories", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }
        #endregion

        #endregion

        #region 应用管理

        /// <summary>
        /// 应用管理
        /// </summary>
        /// <returns></returns>
        public ActionResult ManageApplications(string userDomainName)
        {
            SetPageTitle(userDomainName, "应用管理");
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            IList<Application> apps = Applications.GetInstalledApplicationsOfOwner(ThemeManager.PresentAreaID, currentDomainUser.UserID).Where(n => n.IsEnabled).ToList();
            return View("Pages/Settings/ManageApplications.aspx", "Masters/Settings.master", apps);
        }

        public ActionResult UnInstallApplication(string userDomainName, int applicationID)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            try
            {
                Applications.UnInstallApplication(ThemeManager.PresentAreaID, domainUser.UserID, applicationID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "卸载应用成功！");
            }
            catch (SPBException e)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, e.Message);
            }
            return RedirectToAction("ManageApplications", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        /// <summary>
        /// 浏览应用
        /// </summary>
        /// <returns></returns>
        public ActionResult ListApplications(string userDomainName)
        {
            IList<Application> apps = Applications.GetAvailableApplications(ThemeManager.PresentAreaID).Where(n => n.IsEnabled).ToList();
            bool tipInstall = Request.QueryString.GetBool("tipInstall", false);
            if (tipInstall)
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Hint, GetResourceString("Message_PleaseInstallApplication"));

            return View("Pages/Settings/ListApplications.aspx", "Masters/Settings.master", apps);
        }

        public ActionResult InstallApplication(string userDomainName, int applicationID)
        {
            User domainUser = GetCurrentDomainUser(userDomainName);
            try
            {
                Applications.InstallApplication(ThemeManager.PresentAreaID, domainUser.UserID, applicationID);
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, "添加应用成功！");
            }
            catch (SPBException e)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, e.Message);
            }
            return RedirectToAction("ListApplications", "UserDomainSettings", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #region help Method

        public void UpdateSpecialPrivacyItemSettings(string selectedUserIDs, string privacyItemKey, bool isAllowed, User currentDomainUser)
        {

            string[] toUserIDs = selectedUserIDs.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
            if (toUserIDs != null && toUserIDs.Length > 0)
            {
                foreach (string toUserID in toUserIDs)
                {
                    int userID = Convert.ToInt32(toUserID.Trim());
                    if (userID > 0)
                    {
                        User toUser = Users.GetUser(userID, false);
                        if (toUser == null)
                            continue;
                        Dictionary<string, SpecialPrivacyItemSettings> userSpecialPrivacyItemSettingss = Privacys.GetUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUser.UserID, true);

                        userSpecialPrivacyItemSettingss[privacyItemKey].IsAllowed = isAllowed;

                        Privacys.UpdateUserSpecialPrivacyItemSettingss(currentDomainUser.UserID, toUser.UserID, toUser.DisplayName, new List<SpecialPrivacyItemSettings>() { userSpecialPrivacyItemSettingss[privacyItemKey] });

                    }
                }
            }
        }

        #endregion

        #region 个人标签

        #region Show

        /// <summary>
        /// 显示个人标签页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageUserTags(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            SetPageTitle(userDomainName, GetResourceString("Title_ManageUserTags", applicationID));

            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            ViewData["userTags"] = userTags;

            return View("Pages/Settings/ManageUserTags.aspx", "Masters/Settings.master");
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_UserSiteTags(string userDomainName, int? pageIndex)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //取站点标签，换页
            PagingDataSet<SiteTag> siteTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetTopSiteTagsSortBy(SiteTagsSortBy.UserCount, 100, 20, pageIndex ?? 1);
            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            siteTags.Records = siteTags.Records.Where(n => !userTags.Select(m => m.TagName).ToList().Contains(n.TagName)).ToList();
            return PartialView("Controls/Settings/UserSiteTags.ascx", siteTags);
        }

        [OutputCache(Location = OutputCacheLocation.None)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult Control_ListMyUserSiteTags(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            //取用户标签
            ICollection<UserTag> userTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            return PartialView("Controls/Settings/ListMyUserSiteTags.ascx", userTags);
        }

        #endregion

        #region Manage

        /// <summary>
        /// 创建个人标签
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUserTag(string userDomainName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 处理标签
            string selectedTagNames = Request.Form.GetString("userTags", string.Empty);
            List<UserTag> selectedTags = null;

            ICollection<UserTag> usersTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            if (usersTags.Count >= 10)
            {
                TempData["statusMessageData"] = new StatusMessageData(StatusMessageType.Error, "最多添加10个标签！");
                return RedirectToAction("Control_ListMyUserSiteTags");
            }

            if (!string.IsNullOrEmpty(selectedTagNames))
            {
                string[] tagNames = selectedTagNames.Split(UserTagManager.SplitCharacters, StringSplitOptions.RemoveEmptyEntries);
                if (tagNames != null && tagNames.Length > 0)
                {
                    selectedTags = new List<UserTag>();
                    foreach (string tagName in tagNames)
                    {
                        if (!string.IsNullOrEmpty(tagName.Trim()))
                        {
                            UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTag(tagName, currentDomainUser.UserID, true);
                            if (tag == null)
                            {
                                UserTag userTag = new UserTag();
                                userTag.UserID = currentDomainUser.UserID;
                                userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                                bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).CreateUserTag(userTag);
                                if (isSuccess)
                                    selectedTags.Add(userTag);
                            }
                        }
                    }
                }
            }

            UserTagManager.Instance(UserTagItemKeys.Instance().Person()).UpdateUserTagsOfItem(currentDomainUser.UserID, currentDomainUser.UserID, usersTags.Select(n => n.TagName).ToList());

            #endregion
            return RedirectToAction("Control_ListMyUserSiteTags");
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult CreateUserTag(string userDomainName, string tagName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);

            #region 处理标签
            List<UserTag> selectedTags = new List<UserTag>();
            UserTag tag = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTag(tagName, currentDomainUser.UserID, true);
            if (tag == null)
            {
                UserTag userTag = new UserTag();
                userTag.UserID = currentDomainUser.UserID;
                userTag.TagName = StringUtils.Trim(tagName, 8, string.Empty);
                bool isSuccess = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).CreateUserTag(userTag);
                if (isSuccess)
                    selectedTags.Add(userTag);
            }

            ICollection<UserTag> usersTags = UserTagManager.Instance(UserTagItemKeys.Instance().Person()).GetUserTags(currentDomainUser.UserID, true);
            UserTagManager.Instance(UserTagItemKeys.Instance().Person()).UpdateUserTagsOfItem(currentDomainUser.UserID, currentDomainUser.UserID, usersTags.Select(n => n.TagName).ToList());
            #endregion

            return RedirectToAction("Control_ListMyUserSiteTags", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        [UserDomainAuthorize(RequireOwnerOrAdministrator = true)]
        public ActionResult DeleteUserTag(string userDomainName, string tagName)
        {
            if (string.IsNullOrEmpty(userDomainName))
                throw new ArgumentException(GetResourceString("Exception_Pattern_UserDomainNameNoNull"));
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            UserTagManager.Instance(UserTagItemKeys.Instance().Person()).DeleteUserTag(tagName, currentDomainUser.UserID);

            return RedirectToAction("Control_ListMyUserSiteTags", new RouteValueDictionary { { "userDomainName", userDomainName } });
        }

        #endregion

        #endregion
    }

    /// <summary>
    /// 用户空间设置列表类型
    /// </summary>
    public enum UserDomainSettingsMenuType
    {
        /// <summary>
        /// 帐号
        /// </summary>
        Account,
        /// <summary>
        /// 更换皮肤
        /// </summary>
        ThemeSetting,
        /// <summary>
        /// 导航设置
        /// </summary>
        ManageUserNavigations,
        /// <summary>
        /// 隐私
        /// </summary>
        Privacy,
        /// <summary>
        /// 友情链接
        /// </summary>
        UserLink,
        /// <summary>
        /// 广告
        /// </summary>
        Advertising,
        /// <summary>
        /// 通知
        /// </summary>
        Notice,
        /// <summary>
        /// 动态
        /// </summary>
        Story,
        /// <summary>
        /// OpenSocial应用
        /// </summary>
        OSApplication,
        /// <summary>
        /// 应用管理
        /// </summary>
        ManageApplications
    }

    /// <summary>
    /// 个人档案类型
    /// </summary>
    public enum PersonProfileType
    {
        /// <summary>
        /// 帐号信息
        /// </summary>
        AccountInfo = 0,

        /// <summary>
        /// 基本信息
        /// </summary>
        BasicInfo = 1,

        /// <summary>
        /// 个人联络信息
        /// </summary>
        PersonContact = 2,

        /// <summary>
        /// 商务联络信息
        /// </summary>
        BusinessContact = 3,

        /// <summary>
        /// 私密信息
        /// </summary>
        PrivacyInfo = 4
    }

    public enum PersonProfileMenu
    {
        /// <summary>
        /// 帐号信息
        /// </summary>
        AccountInfo = 0,

        /// <summary>
        /// 基本信息
        /// </summary>
        BasicInfo = 1,

        /// <summary>
        /// 个人联络信息
        /// </summary>
        PersonContact = 2,

        /// <summary>
        /// 商务联络信息
        /// </summary>
        BusinessContact = 3,

        /// <summary>
        /// 私密信息
        /// </summary>
        PrivacyInfo = 4,

        /// <summary>
        /// 上传头像
        /// </summary>
        UploadAvatar = 5,

        /// <summary>
        /// 更改密码
        /// </summary>
        ChangePassword = 6,

        /// <summary>
        /// 论坛签名
        /// </summary>
        ForumSignature = 7,

        /// <summary>
        /// 账号绑定
        /// </summary>
        AccountBinding = 8,
        /// <summary>
        /// 个人标签
        /// </summary>
        PersonUserTags = 9
    }

    /// <summary>
    /// 企业资料类型
    /// </summary>
    public enum CompanyProfileTypes
    {
        /// <summary>
        /// 帐号信息
        /// </summary>
        CompanySetting = 0,

        /// <summary>
        /// 基本信息
        /// </summary>
        BasicMessage = 1,
        /// <summary>
        /// 企业标识图片
        /// </summary>
        CompanyLogo = 2
    }

    /// <summary>
    /// 企业资料导航
    /// </summary>
    public enum CompanyProfileMenu
    {
        /// <summary>
        /// 帐号信息
        /// </summary>
        CompanySetting = 0,

        /// <summary>
        /// 基本信息
        /// </summary>
        BasicMessage = 1,

        /// <summary>
        /// 企业标识图片
        /// </summary>
        CompanyLogo = 2,

        /// <summary>
        /// 上传头像
        /// </summary>
        UploadAvatar = 3,

        /// <summary>
        /// 更改密码
        /// </summary>
        ChangePassword = 4,

        /// <summary>
        /// 论坛签名
        /// </summary>
        ForumSignature = 5,

        /// <summary>
        /// 账号绑定
        /// </summary>
        AccountBinding = 6

    }

}
