﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Web;
using System.Web.Caching;

using SpaceBuilder.Rss;
using SpaceBuilder.Caching;
using SpaceBuilder.Utils;

namespace SpaceBuilder.Common.Handlers
{
    /// <summary>
    /// Rss输出 Handler 基类
    /// </summary>
    public abstract class BaseRssHandler : IHttpHandler
    {
        /// <summary>
        /// HttpContext
        /// </summary>
        protected HttpContext Context = null;

        /// <summary>
        /// 缓存的Feed
        /// </summary>
        protected CachedFeed Feed = null;

        /// <summary>
        /// The If-Modified-Since header of the request.
        /// </summary>
        protected string IfModifiedSinceHeader
        {
            get { return Context.Request.Headers["If-Modified-Since"] as string; }
        }

        /// <summary>
        /// The If-None-Match header of the request.
        /// </summary>
        protected string IfNoneMatchHeader
        {
            get { return Context.Request.Headers["If-None-Match"] as string; }
        }

        /// <summary>
        /// Feed是否允许缓存
        /// </summary>
        protected virtual bool FeedIsCacheable
        {
            get { return true; }
        }

        /// <summary>
        /// 缓存时间
        /// </summary>
        protected virtual int CacheTime { get { return 90; } }

        /// <summary>
        /// 缓存标识
        /// </summary>        
        protected abstract string CacheKey { get; }

        /// <summary>
        /// The Fully Qualifed part of the Url for the current request
        /// </summary>
        protected abstract string BaseUrl { get; }

        protected virtual string FormatUrl(string url)
        {
            if (!string.IsNullOrEmpty(url) && url.ToLower().StartsWith("http"))
                return url;
            else
                return string.Concat(BaseUrl, url);
        }

        /// <summary>
        /// xsl样式表
        /// </summary>
        protected virtual string StyleSheet
        {
            get
            {
                return string.Empty;
                //return "~/Utility/FeedStylesheets/rss.xsl"; 
            }
        }

        /// <summary>
        /// Write a 304 Not Modified response to the request.
        /// </summary>
        private void WriteNotModifiedResponse()
        {
            Context.Response.StatusCode = 304;
            Context.Response.Status = "304 Not Modified";
        }

        /// <summary>
        /// 创建一个feed.
        /// </summary>
        /// <returns>CachedFeed</returns>
        protected abstract CachedFeed BuildFeed();

        /// <summary>
        /// 在HttpResponse输出rss
        /// </summary>
        protected virtual void WriteFeed()
        {
            Context.Response.Clear();
            Context.Response.ContentEncoding = System.Text.Encoding.UTF8;
            Context.Response.ContentType = "text/xml";
            Context.Response.Cache.SetCacheability(HttpCacheability.Public);
            Context.Response.Cache.SetLastModified(Feed.LastModified);
            Context.Response.Cache.SetETag(Feed.Etag);
            Context.Response.Write(Feed.Xml);
        }

        /// <summary>
        /// 处理Feed并输出rss
        /// </summary>
        protected virtual void ProcessFeed()
        {
            if (FeedIsCacheable)
                Feed = CacheRepository.Get(this.CacheKey, false) as CachedFeed;

            if (Feed == null)
            {
                if ((Feed = BuildFeed()) == null)
                    return;

                if (FeedIsCacheable)
                    CacheRepository.Add(CacheKey, Feed, CachingExpirationTypes.UsualObjectCollection);
            }

            // Now check if the conditional nature of the GET indicates we can just
            // return a 304.
            if (Feed.IsCacheOK(IfNoneMatchHeader, IfModifiedSinceHeader))
                WriteNotModifiedResponse();
            else
                WriteFeed();
        }

        /// <summary>
        /// Process the request for the feed.
        /// </summary>
        /// <param name="context">The HTTP context for the request.</param>
        public virtual void ProcessRequest(HttpContext context)
        {
            if (!SiteSettingsManager.GetSiteSettings().EnableRSS)
                WebUtils.Return404(context);

            Context = context;
            ProcessFeed();
        }

        /// <summary>
        /// Is this HTTP handler reusable?
        /// </summary>
        public virtual bool IsReusable
        {
            get { return false; }
        }
    }
}
