﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using SpaceBuilder.Web;
using SpaceBuilder.Utils;
using System.IO;

namespace SpaceBuilder.Common.Handlers
{
    /// <summary>
    /// 友情链接图片
    /// </summary>
    public class LinkPhotoHandler : DownloadFileHandlerBase
    {
        public override void ProcessRequest(HttpContext context)
        {
            int linkID = context.Request.QueryString.GetInt("linkID", -1);
            int linkPosition = context.Request.QueryString.GetInt("linkPosition", 0);
            bool enableCaching = context.Request.QueryString.GetBool("enableCaching", true);

            if (linkID <= 0)
            {
                WebUtils.Return404(context);
                return;
            }

            LinkManager linkManager = null;
            LinkPosition linkPositionCurrent = (LinkPosition)linkPosition;
            switch (linkPositionCurrent)
            {
                case LinkPosition.Site:
                    linkManager = SiteLinkManager.Instance();
                    break;
                case LinkPosition.User:
                    linkManager = UserLinkManager.Instance();
                    break;
                case LinkPosition.Club:
                    linkManager = SpaceBuilder.Club.ClubLinkManager.Instance();
                    break;
            }

            IFile linkPhoto = linkManager.GetLinkPhoto(linkID);

            if (linkPhoto == null)
            {
                WebUtils.Return404(context);
                return;
            }
            else
            {
                DateTime lastModified = linkPhoto.LastModified.ToUniversalTime();
                if (enableCaching && IsCacheOK(context, lastModified))
                {
                    WebUtils.Return304(context);
                    return;
                }
                else
                {
                    //context.Response.Clear();
                    context.Response.ContentType = "image/jpeg";

                    FileSystemFile fileSystemFile = linkPhoto as FileSystemFile;
                    if (fileSystemFile != null && (!fileSystemFile.FullLocalPath.StartsWith(@"\")))
                    {
                        // Send files stored on UNC paths explicitly to avoid a bug with TransmitFile.
                        context.Response.TransmitFile(fileSystemFile.FullLocalPath);
                    }
                    else
                    {
                        context.Response.AddHeader("Content-Length", linkPhoto.ContentLength.ToString("0"));
                        using (Stream stream = linkPhoto.OpenReadStream())
                        {
                            if (stream == null)
                            {
                                WebUtils.Return404(context);
                                return;
                            }

                            int bufferLength = linkPhoto.ContentLength <= DownloadFileHandlerBase.BufferLength ? linkPhoto.ContentLength : DownloadFileHandlerBase.BufferLength;
                            byte[] buffer = new byte[bufferLength];
                            int readedSize;
                            while ((readedSize = stream.Read(buffer, 0, bufferLength)) > 0)
                            {
                                if (!context.Response.IsClientConnected)
                                    break;

                                context.Response.OutputStream.Write(buffer, 0, readedSize);
                            }

                            stream.Close();
                            stream.Dispose();

                            context.Response.OutputStream.Flush();
                            context.Response.Flush();
                        }
                    }

                    if (enableCaching)
                    {
                        // Browser cache settings
                        context.Response.Cache.SetCacheability(HttpCacheability.Private);
                        context.Response.Cache.SetLastModified(lastModified);
                        context.Response.Cache.SetETag(lastModified.Ticks.ToString());
                        context.Response.Cache.SetAllowResponseInBrowserHistory(true);
                        context.Response.Cache.SetValidUntilExpires(true);
                    }
                    else
                    {
                        context.Response.Cache.SetExpires(DateTime.Now.AddMonths(-1));
                    }
                }
            }

            context.Response.Cache.VaryByParams["linkID"] = true;
            context.Response.Cache.VaryByParams["linkPosition"] = true;
            context.Response.Cache.VaryByParams["enableCaching"] = true;
            context.Response.End();
        }
    }
}
