﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Web;
using System.IO;
using SpaceBuilder.Utils;

using SpaceBuilder.Web;

namespace SpaceBuilder.Common.Handlers
{
    /// <summary>
    /// 显示用户头像
    /// </summary>
    public class UserAvatarHandler : DownloadFileHandlerBase
    {
        public override void ProcessRequest(HttpContext context)
        {
            int userID = context.Request.QueryString.GetInt("UserID", -1);
            UserAvatarType avatarType = (UserAvatarType)context.Request.QueryString.GetInt("AvatarType", (int)UserAvatarType.Normal);
            AvatarSizeTypes avatarSizeType = (AvatarSizeTypes)context.Request.QueryString.GetInt("AvatarSizeType", (int)AvatarSizeTypes.Normal);
            bool enableCaching = context.Request.QueryString.GetBool("enableCaching", true);

            if (avatarType == UserAvatarType.Normal && userID <= 0)
            {
                WebUtils.Return404(context);
                return;
            }

            if (avatarType == UserAvatarType.Normal)
            {
                IFile avatarFile = Avatars.GetAvatar(userID, avatarSizeType);
                if (avatarFile == null)
                {
                    WebUtils.Return404(context);
                    return;
                }

                DateTime lastModified = avatarFile.LastModified.ToUniversalTime();
                if (enableCaching && IsCacheOK(context, lastModified))
                {
                    WebUtils.Return304(context);
                    return;
                }
                else
                {
                    // Send the file
                    // context.Response.Clear();
                    context.Response.ContentType = "image/jpeg";

                    FileSystemFile fileSystemFile = avatarFile as FileSystemFile;
                    if (fileSystemFile != null && (!fileSystemFile.FullLocalPath.StartsWith(@"\")))
                    {
                        // Send files stored on UNC paths explicitly to avoid a bug with TransmitFile.
                        context.Response.TransmitFile(fileSystemFile.FullLocalPath);
                    }
                    else
                    {
                        context.Response.AddHeader("Content-Length", avatarFile.ContentLength.ToString("0"));
                        using (Stream stream = avatarFile.OpenReadStream())
                        {
                            if (stream == null)
                            {
                                WebUtils.Return404(context);
                                return;
                            }

                            int bufferLength = avatarFile.ContentLength <= DownloadFileHandlerBase.BufferLength ? avatarFile.ContentLength : DownloadFileHandlerBase.BufferLength;
                            byte[] buffer = new byte[bufferLength];
                            int readedSize;
                            while ((readedSize = stream.Read(buffer, 0, bufferLength)) > 0)
                            {
                                if (!context.Response.IsClientConnected)
                                    break;

                                context.Response.OutputStream.Write(buffer, 0, readedSize);
                            }

                            stream.Close();
                            stream.Dispose();
                            context.Response.OutputStream.Flush();
                            context.Response.Flush();
                        }
                    }

                    if (enableCaching)
                    {
                        // Browser cache settings
                        context.Response.Cache.SetCacheability(HttpCacheability.Private);
                        context.Response.Cache.SetLastModified(lastModified);
                        context.Response.Cache.SetETag(lastModified.Ticks.ToString());
                        context.Response.Cache.SetAllowResponseInBrowserHistory(true);
                        context.Response.Cache.SetValidUntilExpires(true);
                    }
                    else
                    {
                        context.Response.Cache.SetExpires(DateTime.Now.AddMonths(-1));
                    }
                }
            }
            else
            {
                string avatarFileName;
                switch (avatarType)
                {
                    case UserAvatarType.HideAvatar:
                        avatarFileName = avatarSizeType == AvatarSizeTypes.Small ? "portrait_hide_small.gif" : "portrait_hide.gif";
                        break;
                    case UserAvatarType.NoAvatar_Default:
                        avatarFileName = avatarSizeType == AvatarSizeTypes.Small ? "portrait_default_small.gif" : "portrait_default.gif";
                        break;
                    case UserAvatarType.NoAvatar_Male:
                        avatarFileName = avatarSizeType == AvatarSizeTypes.Small ? "portrait_boy_small.gif" : "portrait_boy.gif";
                        break;
                    case UserAvatarType.NoAvatar_Female:
                        avatarFileName = avatarSizeType == AvatarSizeTypes.Small ? "portrait_girl_small.gif" : "portrait_girl.gif";
                        break;
                    case UserAvatarType.NoAvatar_DefaultForCompany:
                        avatarFileName = avatarSizeType == AvatarSizeTypes.Small ? "portrait_company_small.gif" : "portrait_company.gif";
                        break;
                    default:
                        avatarFileName = avatarSizeType == AvatarSizeTypes.Small ? "portrait_default_small.gif" : "portrait_default.gif";
                        
                        break;
                }

                string avatarFullPath = context.Server.MapPath("~/Themes/Shared/Styles/Images/" + avatarFileName);
                
                if (enableCaching)
                {
                    context.Response.Cache.SetExpires(DateTime.Now.AddHours(2));
                    context.Response.Cache.SetCacheability(HttpCacheability.Public);
                    context.Response.Cache.SetValidUntilExpires(true);
                }

                context.Response.ContentType = "image/gif";
                context.Response.TransmitFile(avatarFullPath);
                //context.Response.Flush();
            }

            context.Response.Cache.VaryByParams["UserID"] = true;
            context.Response.Cache.VaryByParams["AvatarType"] = true;
            context.Response.Cache.VaryByParams["AvatarSizeType"] = true;
            context.Response.Cache.VaryByParams["enableCaching"] = true;
            
            context.Response.End();
        }

    }
}
