﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

using System.Xml;
using System.IO;

using System.Web;
using SpaceBuilder.Caching;
using SpaceBuilder.Common;
using System.Web.Caching;

namespace SpaceBuilder.Common.Mail
{
    /// <summary>
    /// 对EmailTemplate进行访问的Provider
    /// </summary>
    public class EmailTemplateProvider
    {

        #region Instance

        private EmailTemplateProvider()
        { }

        private static volatile EmailTemplateProvider _defaultInstance = null;
        private static readonly object lockObject = new object();

        /// <summary>
        /// 获取EmailTemplateProvider实例
        /// </summary>
        /// <returns></returns>
        public static EmailTemplateProvider Instance()
        {
            if (_defaultInstance == null)
            {
                lock (lockObject)
                {
                    if (_defaultInstance == null)
                    {
                        _defaultInstance = new EmailTemplateProvider();
                    }
                }
            }

            return _defaultInstance;
        }

        #endregion

        /// <summary>
        /// 获取用户Email模板
        /// </summary>
        public EmailTemplate GetUserEmailTemplate(string emailType, User userEmailTo)
        {
            Dictionary<string, EmailTemplate> emailTemplates;

            // First, try to load the templates in the user language
            emailTemplates = LoadEmailTemplates(userEmailTo.Language, EmailTemplateTypes.UserEmail);

            // if the user language templates are not found, then load the system defaults
            if ((emailTemplates == null) || !emailTemplates.ContainsKey(emailType))
            {
                emailTemplates = LoadEmailTemplates(SPBConfig.Instance().DefaultLanguage, EmailTemplateTypes.UserEmail);

                // if they still are not found, then load the zh-CN templates
                if ((emailTemplates == null) || !emailTemplates.ContainsKey(emailType))
                    emailTemplates = LoadEmailTemplates("zh-CN", EmailTemplateTypes.UserEmail);
            }

            if (!emailTemplates.ContainsKey(emailType))
                return null;

            EmailTemplate email = emailTemplates[emailType].Clone();
            if (userEmailTo != null)
                email.IsBodyHtml = userEmailTo.EnableHtmlEmail;

            // Get the template;
            return email;
        }
        /// <summary>
        /// 获取通知Email模板
        /// </summary>
        public EmailTemplate GetNoticeEmailTemplate(string emailType, User userEmailTo)
        {
            Dictionary<string, EmailTemplate> emailTemplates;

            // First, try to load the templates in the user language
            emailTemplates = LoadEmailTemplates(userEmailTo.Language, EmailTemplateTypes.NoticeEmail);

            // if the user language templates are not found, then load the system defaults
            if ((emailTemplates == null) || !emailTemplates.ContainsKey(emailType))
            {
                emailTemplates = LoadEmailTemplates(SPBConfig.Instance().DefaultLanguage, EmailTemplateTypes.NoticeEmail);

                // if they still are not found, then load the zh-CN templates
                if ((emailTemplates == null) || !emailTemplates.ContainsKey(emailType))
                    emailTemplates = LoadEmailTemplates("zh-CN", EmailTemplateTypes.NoticeEmail);
            }

            if (!emailTemplates.ContainsKey(emailType))
                return null;

            EmailTemplate email = emailTemplates[emailType].Clone();
            if (userEmailTo != null)
                email.IsBodyHtml = userEmailTo.EnableHtmlEmail;

            // Get the template;
            return email;
        }


        /// <summary>
        /// 加载Email模板
        /// </summary>
        protected Dictionary<string, EmailTemplate> LoadEmailTemplates(string language, EmailTemplateTypes templateType)
        {
            Dictionary<string, EmailTemplate> emailTemplates;
            string cacheKey = string.Format("emailTemplates::language-{0}:templateType-{1}", language, templateType);

            emailTemplates = CacheRepository.Get(cacheKey,false) as Dictionary<string, EmailTemplate>;

            if (emailTemplates == null)
            {
                emailTemplates = new Dictionary<string, EmailTemplate>();

                // Read in the file
                //
                HttpContext httpContext = HttpContext.Current;
                string searchPattern = "UserEmails.xml";
                if (templateType == EmailTemplateTypes.NoticeEmail)
                    searchPattern = "NoticeEmails_*.xml";
                string directoryPath = httpContext.Server.MapPath(string.Format("~/Languages/" + language + "/emails/"));
                FileInfo fileInfo;
                string[] fileNames = Directory.GetFiles(directoryPath, searchPattern);
                foreach (string filePath in fileNames)
                {
                    try
                    {
                        fileInfo = new FileInfo(filePath);
                    }
                    catch
                    {
                        throw new SPBException(ExceptionTypes.Instance().EmailTemplateNotFound(), "No email templates found for language: " + language);
                    }

                    FileStream reader = fileInfo.OpenRead();
                    XmlDocument d = new XmlDocument();
                    d.Load(reader);
                    reader.Close();

                    string emailType;
                    foreach (XmlNode node in d.GetElementsByTagName("email"))
                    {
                        emailType = node.Attributes.GetNamedItem("emailType").InnerText;
                        EmailTemplate emailTemplate = new EmailTemplate(node);
                        emailTemplates[emailType] = emailTemplate;
                    }
                }

                //CacheDependency dep = new CacheDependency(fileNames);
                CacheRepository.Add(cacheKey, emailTemplates, CachingExpirationTypes.Stable);
            }

            return emailTemplates;
        }

    }
}
