﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Net.Mail;
using System.Text.RegularExpressions;
using SpaceBuilder.Common;

namespace SpaceBuilder.Common.Mail
{
    /// <summary>
    /// EmailTemplates
    /// </summary>
    public class EmailTemplates
    {
        #region Email Templates

        /// <overloads>获取Email模板</overloads>
        /// <summary>
        /// 获取UserEmail模板
        /// </summary>
        /// <param name="emailType">Email类型（在Email模板中是唯一的）</param>
        /// <param name="toUser">接收Email的用户</param>
        /// <returns>EmailTemplate</returns>
        public static EmailTemplate GetUserEmailTemplate(string emailType, User toUser)
        {
            EmailTemplate email = EmailTemplateProvider.Instance().GetUserEmailTemplate(emailType, toUser);
            if (email == null)
                return null;

            if (toUser != null)
                email.IsBodyHtml = toUser.EnableHtmlEmail;

            return email;
        }

        /// <summary>
        /// 获取Email模板，并设置Email的内容格式
        /// </summary>
        /// <param name="emailType">Email类型（在Email模板中是唯一的）</param>
        /// <param name="toUser">接收Email的用户</param>
        /// <param name="isHtml">是否设置Email内容格式为Html</param>
        /// <returns></returns>
        public static EmailTemplate GetUserEmailTemplate(string emailType, User toUser, bool isHtml)
        {
            EmailTemplate email = GetUserEmailTemplate(emailType, toUser);
            if (email != null)
                email.IsBodyHtml = isHtml;

            return email;
        }

        /// <overloads>获取NoticeEmail模板</overloads>
        /// <summary>
        /// 获取NoticeEmail模板
        /// </summary>
        /// <param name="emailType">Email类型（在Email模板中是唯一的）</param>
        /// <param name="toUser">接收Email的用户</param>
        /// <returns>EmailTemplate</returns>
        public static EmailTemplate GetNoticeEmailTemplate(string noticeItemKey, User toUser)
        {
            EmailTemplate email = EmailTemplateProvider.Instance().GetNoticeEmailTemplate(noticeItemKey, toUser);
            if (email == null)
                return null;

            if (toUser != null)
                email.IsBodyHtml = toUser.EnableHtmlEmail;

            return email;
        }

        /// <summary>
        /// 获取NoticeEmail模板，并设置Email的内容格式
        /// </summary>
        /// <param name="emailType">Email类型（在Email模板中是唯一的）</param>
        /// <param name="toUser">接收Email的用户</param>
        /// <param name="isHtml">是否设置Email内容格式为Html</param>
        /// <returns></returns>
        public static EmailTemplate GetNoticeEmailTemplate(string noticeItemKey, User toUser, bool isHtml)
        {
            EmailTemplate email = GetNoticeEmailTemplate(noticeItemKey, toUser);
            if (email != null)
                email.IsBodyHtml = isHtml;

            return email;
        }


        #endregion



        #region Helper Methods

        /// <summary>
        /// 转化EmailTemplate到MailMessage<see cref="System.Net.Mail.MailMessage"/>
        /// </summary>
        /// <param name="emailTemplate">待转换的EmailTemplate</param>
        /// <returns>MailMessage</returns>
        public static MailMessage ConvertEmailTemplateToMailMessage(EmailTemplate emailTemplate)
        {
            if (string.IsNullOrEmpty(emailTemplate.From) || string.IsNullOrEmpty(emailTemplate.To))
                return null;

            MailMessage email = new MailMessage();
            char[] separator = new char[2] { ',', ';' };
            string[] toArray = emailTemplate.To.Split(separator, StringSplitOptions.RemoveEmptyEntries);
            for (int i = 0; i < toArray.Length; i++)
            {
                try
                {
                    email.To.Add(toArray[i]);
                }
                catch{}
            }

            if (!string.IsNullOrEmpty(emailTemplate.Cc))
            {
                string[] ccArray = emailTemplate.Cc.Split(separator, StringSplitOptions.RemoveEmptyEntries);
                for (int i = 0; i < ccArray.Length; i++)
                {
                    try
                    {
                        email.CC.Add(ccArray[i]);
                    }
                    catch { }
                }
            }

            if (!string.IsNullOrEmpty(emailTemplate.Bcc))
            {
                string[] bccArray = emailTemplate.Bcc.Split(separator, StringSplitOptions.RemoveEmptyEntries);
                for (int i = 0; i < bccArray.Length; i++)
                {
                    try
                    {
                        email.Bcc.Add(bccArray[i]);
                    }
                    catch { }
                }
            }
            try
            {
                email.From = new MailAddress(emailTemplate.From);
            }
            catch { }

            email.Subject = emailTemplate.Subject;
            email.Priority = emailTemplate.Priority;
            //email.Body = emailTemplate.Body;
            if (emailTemplate.IsBodyHtml)
            {
                email.IsBodyHtml = true;
                //email.Body = "<html><body>" + FormatPlainTextAsHtml(emailTemplate.HtmlBody).Trim() + "</body></html>";
                email.Body = "<html><body>" + emailTemplate.HtmlBody.Trim() + "</body></html>";
            }
            else
            {
                email.IsBodyHtml = false;
                email.Body = FormatHtmlAsPlainText(emailTemplate.TextBody);
            }

            // Set the content base
            //email.Headers["Content-Base"] = hostPath + WebUtils.ApplicationPath;
            //email.Headers["Content-Location"] = hostPath + WebUtils.ApplicationPath;

            email.Headers["X-SPB-EmailID"] = emailTemplate.EmailID.ToString();
            email.Headers["X-SPB-Attempts"] = (emailTemplate.NumberOfTries + 1).ToString();
            //msEmail.Headers.Add("X-SPB-AppDomain", AppDomain.CurrentDomain.FriendlyName);

            return email;
        }


        /// <summary>
        /// 把stringToFormat格式化成Html格式
        /// </summary>
        /// <param name="stringToFormat">待格式的文本</param>
        /// <returns>格式化后的文本</returns>
        public static string FormatPlainTextAsHtml(string stringToFormat)
        {
            if (stringToFormat == null || stringToFormat == string.Empty)
                return string.Empty;

            // line breaks
            stringToFormat = Regex.Replace(stringToFormat, "\n", "<br />", RegexOptions.IgnoreCase | RegexOptions.Compiled);
            stringToFormat = stringToFormat.Replace("\r", string.Empty);

            // make urls clickable
            stringToFormat = Regex.Replace(stringToFormat, @"(http|ftp|https):\/\/[\w]+(.[\w]+)([\w\-\.,@?^=%&:/~\+#\$]*[\w\-\@?^=%&/~\+#])?", "<a href=\"$0\">$0</a>", RegexOptions.IgnoreCase | RegexOptions.Compiled);

            return stringToFormat;
        }

        /// <summary>
        /// 把stringToFormat格式化成纯文本格式
        /// </summary>
        /// <param name="stringToFormat">待格式的文本</param>
        /// <returns>格式化后的文本</returns>
        public static string FormatHtmlAsPlainText(string stringToFormat)
        {
            if (stringToFormat == null || stringToFormat == string.Empty)
                return string.Empty;

            // 将"<br/>"替换成"\n"
            stringToFormat = Regex.Replace(stringToFormat, @"<br[\s]*[/]?>", "\n", RegexOptions.IgnoreCase | RegexOptions.Compiled);
            //stringToFormat = stringToFormat.Replace("\r", string.Empty);

            // add linebreaks from HTML for <br>, <p>, <li>, and <blockquote> tags
            stringToFormat = Regex.Replace(stringToFormat, @"</?(br|p|li|blockquote)(\s/)?>", "\n", RegexOptions.IgnoreCase | RegexOptions.Compiled);

            // strip all remaining HTML
            stringToFormat = Regex.Replace(stringToFormat, @"</?(\w+)(\s*\w*\s*=\s*(""[^""]*""|'[^']'|[^>]*))*|/?>", "", RegexOptions.IgnoreCase | RegexOptions.Compiled);

            // replace special characters
            stringToFormat = stringToFormat.Replace("&nbsp;", " ");
            stringToFormat = stringToFormat.Replace("&lt;", "<");
            stringToFormat = stringToFormat.Replace("&gt;", ">");
            stringToFormat = stringToFormat.Replace("&amp;", "&");
            stringToFormat = stringToFormat.Replace("&quot;", "\"");

            return stringToFormat;
        }

        #endregion


    }
}
