﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;

namespace SpaceBuilder.Common.Mail
{
    /// <summary>
    /// Email模板标记处理类
    /// </summary>
    /// <remarks>
    /// 用于替换Email模板中的标记
    /// </remarks>
    /// <example>
    /// 这是一个账号注册成功后自动发送的Email模板，我们在使用该Email模板发送邮件时必须对该模板中的[SiteName]等标记进行替换。
    /// <code>
    ///&lt;email emailType="NewUserAccountCreated" priority="High"&gt;
    ///&lt;subject&gt;([SiteName]) 帐号注册成功&lt;/subject&gt;
    ///&lt;from&gt;[AdminEmailFrom]&lt;/from&gt;
    ///&lt;body&gt;
    ///[UserName],
    ///
    ///您在 [SiteName] 成功注册了一个新帐号。
    ///
    ///您的用户名和密码:
    ///
    ///  用户名: [UserName]
    ///  密  码: [Password]
    ///
    ///您可以通过这里进行登录: 
    ///[LoginUrl]
    ///
    ///登录后您可以在这里修改您的密码:
    ///[PasswordChange]
    ///
    ///
    ///谢谢,
    ///[SiteName]
    ///
    ///&lt;/body&gt;
    ///&lt;/email&gt;
    /// </code>
    /// </example>
    public sealed class EmailTextToken
    {
        /// <summary>
        /// EmailTextToken构造函数
        /// </summary>
        /// <param name="token">Email模板标记</param>
        /// <param name="text">Email模板标记需要被替换成的文本</param>
        public EmailTextToken(string token, string text)
        {
            this.token = token;
            this.text = text;
        }

        private string token = string.Empty;
        /// <summary>
        /// Email模板中的标记对应的正则表达式模式
        /// </summary>
        public string Pattern
        {
            get { return string.Format("\\[{0}\\]", token); }
        }


        private string text = string.Empty;
        /// <summary>
        /// Pattern对应的标记需要替换成的文本
        /// </summary>
        public string Text
        {
            get { return text; }
            set { text = value; }
        }

        /// <summary>
        /// 把textToFormat中含有Pattern的标记用Text替换
        /// </summary>
        /// <param name="textToFormat">待替换的文本</param>
        /// <returns>替换后的文本</returns>
        public string FormatText(string textToFormat)
        {
            return Replace(this.Pattern, textToFormat, this.Text);
        }


        /// <summary>
        /// 采用正则表达式替换sourceText
        /// </summary>
        /// <param name="_pattern">正则表达式模式</param>
        /// <param name="sourceText">待替换的文本</param>
        /// <param name="replaceText">符合_pattern模式内容要替换成的文本</param>
        /// <returns>替换后的文本</returns>
        private static string Replace(string _pattern, string sourceText, string replaceText)
        {
            if (string.IsNullOrEmpty(sourceText))
                return sourceText;

            StringBuilder resultText = new StringBuilder();
            int startat = 0;
            Regex regex = new Regex(_pattern, RegexOptions.Compiled | RegexOptions.IgnoreCase);
            for (Match match = regex.Match(sourceText); match.Value != string.Empty; match = regex.Match(sourceText, startat))
            {
                if (startat != match.Index)
                {
                    resultText.Append(sourceText.Substring(startat, match.Index - startat));
                }
                resultText.Append(replaceText);
                startat = match.Index + match.Length;
            }
            if (startat < sourceText.Length)
            {
                resultText.Append(sourceText.Substring(startat));
            }
            return resultText.ToString();
        }

    }
}
