﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------


using System;
using System.Collections.Generic;
using System.Text;

using SpaceBuilder.Common;
using SpaceBuilder.Utils;

namespace SpaceBuilder.Common.Mail
{
    /// <summary>
    /// 用户邮件提供类
    /// </summary>
    public class UserEmails : UserEmailsProvider
    {
        /// <summary>
        /// 构造函数
        /// </summary>
        public UserEmails() { }

        /// <summary>
        /// 用户注册成功
        /// </summary>
        public override void UserCreate(User user, string password)
        {
            if (user == null)
                return;

            User(UserEmailType.NewUserAccountCreated, user, password);
        }

        /// <summary>
        /// 用户注册后，验证用户Email
        /// </summary>
        public override void UserCreateValidateEmail(User user)
        {
            if (user == null)
                return;

            EmailTemplate emailTemplate = EmailTemplates.GetUserEmailTemplate(UserEmailType.NewUserAccountValidateEmail, user, false);
            if (emailTemplate == null)
                return;

            List<EmailTextToken> tokens = new List<EmailTextToken>();
            tokens.AddRange(EmailHelper.GetCommonFields());
            tokens.AddRange(EmailHelper.GetUserFields(user));

            string url = WebUtils.FullPath(SiteUrls.Instance().ValidateUserEmail(user));

            tokens.Add(new EmailTextToken("EmailValidateUrl", url));
            tokens.Add(new EmailTextToken("Name", user.DisplayName));

            emailTemplate.AddTo(user.PrivateEmail);

            emailTemplate.ProcessTokens(tokens);
            Emails.SendEmailMessage(EmailTemplates.ConvertEmailTemplateToMailMessage(emailTemplate));
        }

        /// <summary>
        /// 用户帐号等待批准
        /// </summary>
        public override void UserAccountPending(User user)
        {
            if (user == null)
                return;

            User(UserEmailType.NewUserAccountPending, user, null);
        }

        /// <summary>
        /// 用户帐号被批准
        /// </summary>
        public override void UserAccountApproved(User user)
        {
            if (user == null)
                return;
            User(UserEmailType.NewUserAccountApproved, user, null);
        }

        /// <summary>
        /// 用户帐号被拒绝
        /// </summary>
        public override void UserAccountRejected(User user, User moderatedBy)
        {
            if (user == null)
                return;
            User(UserEmailType.NewUserAccountRejected, user, null);
        }

        /// <summary>
        /// 用户更新密码
        /// </summary>
        public override void UserPasswordChanged(User user, string password)
        {
            if (user == null)
                return;

            if (!CanSendEmail(user))
                return;

            User(UserEmailType.ChangedPassword, user, password);
        }

        /// <summary>
        /// 忘记密码
        /// </summary>
        /// <remarks>点击收到的连接，才能对密码进行修改</remarks>
        public override void UserPasswordForgotten(User user)
        {
            if (user == null)
                return;

            EmailTemplate emailTemplate = EmailTemplates.GetUserEmailTemplate(UserEmailType.ForgottenPassword, user, false);
            if (emailTemplate == null)
                return;
            emailTemplate.IsBodyHtml = true;
            List<EmailTextToken> tokens = new List<EmailTextToken>();
            tokens.AddRange(EmailHelper.GetCommonFields());

            string url = WebUtils.FullPath(SiteUrls.Instance().UserResetPassword(user));

            tokens.Add(new EmailTextToken("ForgottenPasswordUrl", url));
            tokens.Add(new EmailTextToken("IP", WebUtils.IPAddress));
            tokens.Add(new EmailTextToken("Name", user.DisplayName));

            emailTemplate.AddTo(user.PrivateEmail);

            emailTemplate.ProcessTokens(tokens);
            Emails.SendEmailMessage(EmailTemplates.ConvertEmailTemplateToMailMessage(emailTemplate));
        }


        #region Private Methods

        private void User(string emailType, User user, string password)
        {
            if (string.IsNullOrEmpty(user.PrivateEmail))
                return;

            // Do we have a password?
            if (password != null)
                user.Password = password;

            EmailTemplate emailTemplate = EmailTemplates.GetUserEmailTemplate(emailType, user);
            if (emailTemplate == null)
                return;

            emailTemplate.AddTo(user.PrivateEmail);

            List<EmailTextToken> tokens = new List<EmailTextToken>();
            tokens.AddRange(EmailHelper.GetCommonFields());
            tokens.AddRange(EmailHelper.GetUserFields(user));

            emailTemplate.ProcessTokens(tokens);

            ////替换url 为 <a href="url">url</a>
            //emailTemplate.HtmlBody = EmailTemplates.FormatPlainTextAsHtml(emailTemplate.HtmlBody);

            Emails.EnqueuEmail(EmailTemplates.ConvertEmailTemplateToMailMessage(emailTemplate));
        }

        /// <summary>
        /// 是否向用户发送Email
        /// </summary>
        private bool CanSendEmail(User user)
        {
            return SiteSettingsManager.GetSiteSettings().EnableEmail && user.EnableEmail;
        }


        #endregion
    }
}