﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common;

namespace SpaceBuilder.LuceneSearch
{
    /// <summary>
    /// 个人用户全文检索查询类
    /// </summary>
    public class PersonUserFullTextQuery
    {
        /// <summary>
        /// 关键词
        /// </summary>
        /// <remarks>包括姓名、昵称、用户名、学校、公司、标签</remarks>
        public string Keyword = string.Empty;
        /// <summary>
        /// 名字关键字
        /// </summary>
        /// <remarks>包括CommonName、NickName、UserName</remarks>
        public string NameKeyword = string.Empty;

        /// <summary>
        /// 最小年龄限制
        /// </summary>
        public int MinAge { get; private set; }

        /// <summary>
        /// 最大年龄限制
        /// </summary>
        public int MaxAge { get; private set; }


        private AgeScopes? ageScope = null;
        /// <summary>
        /// 年龄范围
        /// </summary>
        public AgeScopes? AgeScope
        {
            get { return ageScope; }
            set
            {
                ageScope = value;
                switch (ageScope.Value)
                {
                    case AgeScopes.Year18:
                        MaxAge = 18;
                        break;
                    case AgeScopes.Year22:
                        MinAge = 19;
                        MaxAge = 22;
                        break;
                    case AgeScopes.Year29:
                        MinAge = 23;
                        MaxAge = 29;
                        break;
                    case AgeScopes.Year39:
                        MinAge = 30;
                        MaxAge = 39;
                        break;
                    case AgeScopes.Year40:
                        MinAge = 40;
                        break;
                    default:
                        break;
                }
            }
        }



        /// <summary>
        /// 性别
        /// </summary>
        public Genders Gender = Genders.All;

        /// <summary>
        /// 目前地区Code
        /// </summary>
        public string NowAreaCode = string.Empty;

        /// <summary>
        /// 家乡地区Code
        /// </summary>
        public string HomeAreaCode = string.Empty;

        /// <summary>
        /// 学历
        /// </summary>
        public int Degree = 0;

        /// <summary>
        /// 毕业院校关键字
        /// </summary>
        public string School = string.Empty;

        /// <summary>
        /// 兴趣爱好
        /// </summary>
        public string Interest = string.Empty;

        /// <summary>
        /// 领域专长
        /// </summary>
        public string Expertise = string.Empty;

        /// <summary>
        /// 所在公司名称
        /// </summary>
        public string CompanyName = string.Empty;

        /// <summary>
        /// 标签
        /// </summary>
        public string TagName = null;

        /// <summary>
        /// 公司所在行业
        /// </summary>
        public int CompanyTrade = 0;

        /// <summary>
        /// MSN Messager号码
        /// </summary>
        public string MsnIM = string.Empty;

        /// <summary>
        /// Yahoo Messager号码
        /// </summary>
        public string YahooIM = string.Empty;

        /// <summary>
        /// QQ号码
        /// </summary>
        public string QQIM = string.Empty;

        /// <summary>
        /// Skype号码
        /// </summary>
        public string SkypeIM = string.Empty;

        /// <summary>
        /// 手机号码
        /// </summary>
        public string Mobile = string.Empty;

        /// <summary>
        /// 公开Email
        /// </summary>
        public string PublicEmail = string.Empty;

        /// <summary>
        /// 即时通讯工具关键字
        /// </summary>
        public string IMKeyword = string.Empty;

        /// <summary>
        /// 用户搜索范围
        ///<list type="table">
        ///    <listheader>
        ///        <term>数值</term>
        ///        <description>描述</description>
        ///    </listheader>
        ///    <item>
        ///        <term>0</term>
        ///        <description>搜索所有用户</description>
        ///    </item>
        ///    <item>
        ///        <term>1</term>
        ///        <description>搜索一层朋友圈</description>
        ///    </item>
        ///    <item>
        ///        <term>2</term>
        ///        <description>搜索二层朋友圈</description>
        ///    </item>
        ///    <item>
        ///        <term>3</term>
        ///        <description>搜索三层朋友圈</description>
        ///    </item>
        ///</list>
        /// </summary>
        public int SearchRange = 0;

        /// <summary>
        /// 用户最后更新时间范围
        /// </summary>
        public DateScopes? UpdateDateScope = null;

        /// <summary>
        /// 排序字段
        /// </summary>
        public PersonUsersFullTextSortBy SortBy = PersonUsersFullTextSortBy.LastUpdatedDate;

        /// <summary>
        /// 排序方向（正序或倒序）
        /// </summary>
        public SortOrder SortOrder = SortOrder.Ascending;

        /// <summary>
        /// 当前显示页面页码
        /// </summary>
        private int pageIndex = 1;
        public int PageIndex
        {
            get
            {
                if (pageIndex < 1)
                    return 1;
                else
                    return pageIndex;
            }
            set { pageIndex = value; }
        }

        /// <summary>
        /// 每页显示记录数
        /// </summary>
        public int PageSize = 10;

        /// <summary>
        /// 最大记录数
        /// </summary>
        public int MaxRecords = 1000;

        /// <summary>
        /// 从QueryString获取查询条件
        /// </summary>
        public void ConvertFromQueryString(System.Web.HttpRequestBase request)
        {
            if (!string.IsNullOrEmpty(request.QueryString["Keyword"]))
                Keyword = request.QueryString["Keyword"];

            if (!string.IsNullOrEmpty(request.QueryString["NameKeyword"]))
                NameKeyword = request.QueryString["NameKeyword"];

            if (!string.IsNullOrEmpty(request.QueryString["Gender"]))
            {
                int genderValue = -1;
                int.TryParse(request.QueryString["Gender"], out genderValue);
                if (genderValue > -1)
                    this.Gender = (Genders)genderValue;
            }

            if (!string.IsNullOrEmpty(request.QueryString["AgeScope"]))
            {
                int ageScopeValue = -1;
                int.TryParse(request.QueryString["AgeScope"], out ageScopeValue);
                if (ageScopeValue > -1)
                    this.AgeScope = (AgeScopes)ageScopeValue;
            }

            if (!string.IsNullOrEmpty(request.QueryString["NowAreaCode"]))
                NowAreaCode = request.QueryString["NowAreaCode"];

            if (!string.IsNullOrEmpty(request.QueryString["HomeAreaCode"]))
                HomeAreaCode = request.QueryString["HomeAreaCode"];

            if (!string.IsNullOrEmpty(request.QueryString["Degree"]))
                int.TryParse(request.QueryString["Degree"], out Degree);

            if (!string.IsNullOrEmpty(request.QueryString["School"]))
                School = request.QueryString["School"];

            if (!string.IsNullOrEmpty(request.QueryString["Interest"]))
                Interest = request.QueryString["Interest"];

            if (!string.IsNullOrEmpty(request.QueryString["Expertise"]))
                Expertise = request.QueryString["Expertise"];

            if (!string.IsNullOrEmpty(request.QueryString["CompanyName"]))
                CompanyName = request.QueryString["CompanyName"];

            if (!string.IsNullOrEmpty(request.QueryString["TagName"]))
                this.TagName = request.QueryString["TagName"];

            if (!string.IsNullOrEmpty(request.QueryString["CompanyTrade"]))
                int.TryParse(request.QueryString["CompanyTrade"], out CompanyTrade);

            if (!string.IsNullOrEmpty(request.QueryString["MsnIM"]))
                MsnIM = request.QueryString["MsnIM"];

            if (!string.IsNullOrEmpty(request.QueryString["YahooIM"]))
                YahooIM = request.QueryString["YahooIM"];

            if (!string.IsNullOrEmpty(request.QueryString["QQIM"]))
                QQIM = request.QueryString["QQIM"];

            if (!string.IsNullOrEmpty(request.QueryString["SkypeIM"]))
                SkypeIM = request.QueryString["SkypeIM"];

            if (!string.IsNullOrEmpty(request.QueryString["PublicEmail"]))
                PublicEmail = request.QueryString["PublicEmail"];


            if (!string.IsNullOrEmpty(request.QueryString["IMKeyword"]))
                IMKeyword = request.QueryString["IMKeyword"];

            if (!string.IsNullOrEmpty(request.QueryString["SearchRange"]))
                int.TryParse(request.QueryString["SearchRange"], out SearchRange);

            if (!string.IsNullOrEmpty(request.QueryString["UpdateDateScope"]))
            {
                int updateDateScopeValue = -1;
                int.TryParse(request.QueryString["UpdateDateScope"], out updateDateScopeValue);
                if (updateDateScopeValue > -1)
                    this.UpdateDateScope = (DateScopes)updateDateScopeValue;
            }
        }


        public bool IsValid()
        {
            if (IsShowMoreCondition() || !string.IsNullOrEmpty(Keyword))
                return true;
            else
                return false;
        }

        /// <summary>
        /// 查询条件是否有效
        /// </summary>
        public bool IsShowMoreCondition()
        {
            bool hasCondition = false;

            if (!hasCondition && (MinAge > 0 || MaxAge > 0))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(NameKeyword))
                hasCondition = true;

            if (!hasCondition && Gender != Genders.All)
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(NowAreaCode))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(HomeAreaCode))
                hasCondition = true;

            if (!hasCondition && Degree > 0)
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(School))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(Interest))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(Expertise))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(CompanyName))
                hasCondition = true;

            if (!hasCondition && CompanyTrade > 0)
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(MsnIM))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(YahooIM))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(QQIM))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(SkypeIM))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(Mobile))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(PublicEmail))
                hasCondition = true;

            if (!hasCondition && !string.IsNullOrEmpty(IMKeyword))
                hasCondition = true;

            if (!hasCondition && SearchRange > 0)
                hasCondition = true;

            if (!hasCondition && AgeScope > 0)
                hasCondition = true;

            if (!hasCondition && TagName != null && TagName.Length > 0)
                hasCondition = true;

            return hasCondition;
        }


    }

    /// <summary>
    /// 年龄范围
    /// </summary>
    public enum AgeScopes
    {
        /// <summary>
        /// 18岁以下
        /// </summary>
        Year18 = 0,
        /// <summary>
        /// 19~22岁
        /// </summary>
        Year22 = 1,
        /// <summary>
        /// 23~29岁
        /// </summary>
        Year29 = 2,
        /// <summary>
        /// 30~39岁
        /// </summary>
        Year39 = 3,
        /// <summary>
        /// 40岁以上
        /// </summary>
        Year40 = 4
    }



}
