﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Data;
using System.Configuration;
using System.Linq;
using System.Web;
using System.Xml.Linq;
using System.Collections.Generic;
using System.Text;
using System.Web.Mvc;
using System.Web.Mvc.Html;
using System.Web.Routing;
using SpaceBuilder.Common;

namespace SpaceBuilder.Web
{
    /// <summary>
    /// Calendar for ASP.NET MVC
    /// </summary>
    /// <example>
    /// 默认样式设置：
    /// .calendarTable {border-collapse:collapse;}
    /// .calendarTable caption{padding:3px;text-align:center;}
    /// .calendarTable a {}
    /// .calendarTable th{font-weight:bold;background:#eee;}
    /// .calendarTable th,.calendarTable td{padding:3px;text-align:center;border:1px solid #ddd;border-collapse:collapse;}
    /// .calendarTable td{padding:3px 5px;}
    /// .calendarTable td.today{background:#a2adbc;font-weight:bold;color:#fff;}
    /// </example>
    public class Calendar
    {
        private string cssClass = "calendarTable";
        /// <summary>
        /// 日历所在Table的css class
        /// </summary>
        public string CssClass
        {
            get { return cssClass; }
            set { cssClass = value; }
        }

        /// <summary>
        /// Gets the name of current calendar, it's diaplayed as the "id" of the HTML element "table"
        /// </summary>
        public string Name { get; private set; }

        /// <summary>
        /// the selected month of current calendar
        /// </summary>
        public DateTime CalendarMonth { get; private set; }

        /// <summary>
        /// 获取选中的日期
        /// </summary>
        /// <remarks>
        /// key = day of month
        /// value = 选中日期生成链接的Title
        /// </remarks>
        public Dictionary<int, string> SelectedDays { get; private set; }

        /// <summary>
        /// Gets the controller name that to generate the daily links
        /// </summary>
        public string ControllerName { get; private set; }

        /// <summary>
        /// Gets the action name that to generate the daily links
        /// </summary>
        public string ActionName { get; private set; }

        private HtmlHelper helper;

        /// <summary>
        /// default calendar
        /// </summary>
        public Calendar()
        {
            this.CalendarMonth = DateTime.Now;
            this.Name = "MvcCalendar";
        }

        /// <summary>
        /// Create a calendar use the specified name
        /// </summary>
        /// <param name="name">the name of calendar</param>
        public Calendar(string name)
            : this()
        {
            this.Name = name;
        }

        /// <summary>
        /// Create a calendar use the specified name and selected date
        /// </summary>
        public Calendar(string name, DateTime calendarMonth)
            : this(name)
        {
            this.CalendarMonth = calendarMonth;
        }

        /// <summary>
        /// Create a calendar use data that generate links
        /// </summary>
        /// <param name="name">the name of calendar</param>
        /// <param name="calendarMonth">the selected month of current calendar</param>
        /// <param name="rawDates">the date array</param>
        /// <param name="helper">HtmlHelper object</param>
        /// <param name="controllerName">controller name</param>
        /// <param name="actionName">action name</param>
        public Calendar(string name, DateTime calendarMonth, Dictionary<int, string> selectedDays, HtmlHelper helper, string controllerName, string actionName)
            : this(name)
        {
            this.CalendarMonth = calendarMonth;
            this.SelectedDays = selectedDays;
            this.helper = helper;
            this.ControllerName = controllerName;
            this.ActionName = actionName;
        }

        private string GenerateHeader()
        {
            string prevLink = string.Empty;
            string nextLink = string.Empty;

            var header = new StringBuilder();
            header.AppendFormat("<table id=\"{0}\" class=\"{1}\">", this.Name, this.CssClass);

            if (this.helper != null && this.SelectedDays != null)
            {
                bool hasPrev = true;
                bool hasNext = true;

                //foreach (var date in SelectedDates)
                //{
                //    if (date.Year == CalendarMonth.Year && date.Month < CalendarMonth.Month) hasPrev = true;
                //    if (date.Year == CalendarMonth.Year && date.Month > CalendarMonth.Month) hasNext = true;
                //}

                if (hasPrev) prevLink = helper.ActionLink("<<", this.ActionName, this.ControllerName, new { year = CalendarMonth.AddMonths(-1).Year, month = CalendarMonth.AddMonths(-1).Month }, null).ToString();
                if (hasNext) nextLink = helper.ActionLink(">>", this.ActionName, this.ControllerName, new { year = CalendarMonth.AddMonths(1).Year, month = CalendarMonth.AddMonths(1).Month }, null).ToString();
            }

            header.AppendFormat("<caption>{0} {1} {2}</caption>", prevLink, this.CalendarMonth.ToString("y"), nextLink);

            header.Append("<tr>");

            //header.Append("<th scope=\"col\" abbr=\"Sunday\" title=\"Sunday\">Sun</th>");
            //header.Append("<th scope=\"col\" abbr=\"Monday\" title=\"Monday\">Mon</th>");
            //header.Append("<th scope=\"col\" abbr=\"Tuesday\" title=\"Tuesday\">Tue</th>");
            //header.Append("<th scope=\"col\" abbr=\"Wednesday\" title=\"Wednesday\">Wed</th>");
            //header.Append("<th scope=\"col\" abbr=\"Thursday\" title=\"Thursday\">Thu</th>");
            //header.Append("<th scope=\"col\" abbr=\"Friday\" title=\"Friday\">Fri</th>");
            //header.Append("<th scope=\"col\" abbr=\"Saturday\" title=\"Saturday\">Sat</th>");

            //header.Append("<th scope=\"col\" abbr=\"Sunday\" title=\"Sunday\">S</th>");
            //header.Append("<th scope=\"col\" abbr=\"Monday\" title=\"Monday\">M</th>");
            //header.Append("<th scope=\"col\" abbr=\"Tuesday\" title=\"Tuesday\">T</th>");
            //header.Append("<th scope=\"col\" abbr=\"Wednesday\" title=\"Wednesday\">W</th>");
            //header.Append("<th scope=\"col\" abbr=\"Thursday\" title=\"Thursday\">T</th>");
            //header.Append("<th scope=\"col\" abbr=\"Friday\" title=\"Friday\">F</th>");
            //header.Append("<th scope=\"col\" abbr=\"Saturday\" title=\"Saturday\">S</th>");

            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Sunday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Sunday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_Seven", Globals.GetCurrentUserLanguage()) + "</th>");
            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Monday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Monday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_One", Globals.GetCurrentUserLanguage()) + "</th>");
            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Tuesday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Tuesday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_Two", Globals.GetCurrentUserLanguage()) + "</th>");
            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Wednesday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Wednesday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_Three", Globals.GetCurrentUserLanguage()) + "</th>");
            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Thursday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Thursday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_Four", Globals.GetCurrentUserLanguage()) + "</th>");
            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Friday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Friday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_Five", Globals.GetCurrentUserLanguage()) + "</th>");
            header.Append("<th scope=\"col\" abbr=\"" + ResourceManager.GetString("Common_Saturday", Globals.GetCurrentUserLanguage()) + "\" title=\"" + ResourceManager.GetString("Common_Saturday", Globals.GetCurrentUserLanguage()) + "\">" + ResourceManager.GetString("Common_Six", Globals.GetCurrentUserLanguage()) + "</th>");

            header.Append("</tr>");

            return header.ToString();
        }

        private string GenerateFooter()
        {
            return "</table>";
        }

        private string GenerateDays()
        {
            var now = DateTime.Now;
            var date = new DateTime(this.CalendarMonth.Year, this.CalendarMonth.Month, 1);
            var days = DateTime.DaysInMonth(this.CalendarMonth.Year, this.CalendarMonth.Month);
            var weekDayOfFirstDay = (int)date.DayOfWeek;

            var daysXhtml = new StringBuilder();

            int day = 1;//first day of the date's month
            int weekDay = 0;//0 - 6

            if (weekDayOfFirstDay > 0)
            {
                daysXhtml.Append("<tr>");
                while (weekDay < weekDayOfFirstDay)
                {
                    daysXhtml.Append("<td>&nbsp;</td>");
                    weekDay++;
                }
            }

            while (day <= days)
            {
                if (weekDay == 0) daysXhtml.Append("<tr>");

                if (date.Year == now.Year && date.Month == now.Month && day == now.Day)
                    daysXhtml.Append("<td class=\"today\">" + day + "</td>");
                else
                    daysXhtml.Append("<td>" + day + "</td>");

                if (weekDay == 6)
                {
                    daysXhtml.Append("</tr>");
                    weekDay = -1;
                }

                day++;
                weekDay++;
            }

            if (weekDay > 0)
            {
                while (weekDay < 7)
                {
                    daysXhtml.Append("<td>&nbsp;</td>");
                    weekDay++;
                }
                daysXhtml.Append("</tr>");
            }

            return daysXhtml.ToString();
        }

        private string GenerateDays(Dictionary<int, string> selectedDays)
        {
            var now = DateTime.Now;
            var date = new DateTime(this.CalendarMonth.Year, this.CalendarMonth.Month, 1);
            var days = DateTime.DaysInMonth(this.CalendarMonth.Year, this.CalendarMonth.Month);
            var weekDayOfFirstDay = (int)date.DayOfWeek;

            var daysXhtml = new StringBuilder();

            int day = 1;//first day of the date's month
            int weekDay = 0;//0 - 6

            if (weekDayOfFirstDay > 0)
            {
                daysXhtml.Append("<tr>");
                while (weekDay < weekDayOfFirstDay)
                {
                    daysXhtml.Append("<td>&nbsp;</td>");
                    weekDay++;
                }
            }

            while (day <= days)
            {
                if (weekDay == 0) daysXhtml.Append("<tr>");

                string dayString = day.ToString();

                if (selectedDays.ContainsKey(day))
                    dayString = helper.ActionLink(day.ToString(), ActionName, ControllerName, new { year = this.CalendarMonth.Year, month = this.CalendarMonth.Month, day = day }, new { title = selectedDays[day] }).ToString();
                    //dayString = helper.ActionLink(day.ToString(), ActionName, ControllerName, null);


                if (date.Year == now.Year && date.Month == now.Month && day == now.Day)
                    daysXhtml.Append("<td class=\"today\">" + dayString + "</td>");
                else
                    daysXhtml.Append("<td>" + dayString + "</td>");

                if (weekDay == 6)
                {
                    daysXhtml.Append("</tr>");
                    weekDay = -1;
                }

                day++;
                weekDay++;
            }

            if (weekDay > 0)
            {
                while (weekDay < 7)
                {
                    daysXhtml.Append("<td>&nbsp;</td>");
                    weekDay++;
                }
                daysXhtml.Append("</tr>");
            }

            return daysXhtml.ToString();
        }

        /// <summary>
        /// Returns the built calendar
        /// </summary>
        public override string ToString()
        {
            var calendar = new StringBuilder();

            if (this.SelectedDays != null)
            {
                //Build a calendar with links
                calendar.AppendFormat("{0}{1}{2}",
                    GenerateHeader(),
                    GenerateDays(SelectedDays),
                    GenerateFooter()
                    );
            }
            else
            {
                //build an empty calendar for current month
                calendar.AppendFormat("{0}{1}{2}",
                    GenerateHeader(),
                    GenerateDays(),
                    GenerateFooter()
                    );
            }

            return calendar.ToString();
        }
    }

}