﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using System.Text.RegularExpressions;
using SpaceBuilder.Common;

namespace SpaceBuilder.Web
{
    /// <summary>
    /// 分页按钮控件
    /// </summary>
    public partial class SPBHtmlHelper
    {
        ///<overloads>呈现分页按钮</overloads>
        /// <summary>
        /// 呈现分页按钮
        /// </summary>
        /// <param name="paginationMode">分页按钮显示模式</param>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <returns>分页按钮html代码</returns>
        public string PagingButton(PaginationMode paginationMode, int totalRecords, int pageSize, int pageIndex)
        {
            return PagingButton(paginationMode, totalRecords, pageSize, pageIndex, 7, null);
        }


        ///<overloads>呈现分页按钮</overloads>
        /// <summary>
        /// 呈现分页按钮
        /// </summary>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <param name="numericPageButtonCount">数字分页按钮显示个数(建议填写奇数个)</param>
        /// <param name="pagingButtonUrlPattern">链接格式（例如：http://www.spacebuilder.cn/blog/list?pageIndex={0}）</param>
        /// <returns>分页按钮html代码</returns>
        public string PagingButton(int totalRecords, int pageSize, int pageIndex, int numericPagingButtonCount, string pagingButtonUrlPattern)
        {
            return PagingButton(PaginationMode.NumericNextPrevious, totalRecords, pageSize, pageIndex, numericPagingButtonCount, pagingButtonUrlPattern);
        }

        ///<overloads>呈现分页按钮</overloads>
        /// <summary>
        /// 呈现分页按钮
        /// </summary>
        /// <param name="paginationMode">分页按钮显示模式</param>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <param name="numericPageButtonCount">数字分页按钮显示个数</param>
        /// <param name="pagingButtonUrlPattern">链接格式（例如：http://www.spacebuilder.cn/blog/list?pageIndex={0}）</param>
        /// <returns>分页按钮html代码</returns>
        public string PagingButton(PaginationMode paginationMode, int totalRecords, int pageSize, int pageIndex, int numericPagingButtonCount, string pagingButtonUrlPattern)
        {
            return PagingButton(paginationMode, totalRecords, pageSize, pageIndex, numericPagingButtonCount, pagingButtonUrlPattern, false, null);
        }


        ///<overloads>呈现Ajax分页按钮</overloads>
        /// <summary>
        /// 呈现Ajax分页按钮
        /// </summary>
        /// <param name="paginationMode">分页按钮显示模式</param>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <returns>分页按钮html代码</returns>
        public string AjaxPagingButton(PaginationMode paginationMode, int totalRecords, int pageSize, int pageIndex, string pageIndexChangeCallback)
        {
            return AjaxPagingButton(paginationMode, totalRecords, pageSize, pageIndex, 7, null, pageIndexChangeCallback);
        }


        ///<overloads>呈现Ajax分页按钮</overloads>
        /// <summary>
        /// 呈现Ajax分页按钮
        /// </summary>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <param name="numericPageButtonCount">数字分页按钮显示个数</param>
        /// <param name="pagingButtonUrlPattern">链接格式（例如：http://www.spacebuilder.cn/blog/list?pageIndex={0}）</param>
        /// 
        /// <returns>分页按钮html代码</returns>
        public string AjaxPagingButton(int totalRecords, int pageSize, int pageIndex, int numericPagingButtonCount, string pagingButtonUrlPattern, string pageIndexChangeCallback)
        {
            return AjaxPagingButton(PaginationMode.NumericNextPrevious, totalRecords, pageSize, pageIndex, numericPagingButtonCount, pagingButtonUrlPattern, pageIndexChangeCallback);
        }

        ///<overloads>呈现分页按钮</overloads>
        /// <summary>
        /// 呈现分页按钮
        /// </summary>
        /// <param name="paginationMode">分页按钮显示模式</param>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <param name="numericPageButtonCount">数字分页按钮显示个数</param>
        /// <param name="pagingButtonUrlPattern">链接格式（例如：http://www.spacebuilder.cn/blog/list?pageIndex={0}）</param>
        /// <param name="pageIndexChangeCallback">点击分页按钮时客户端回调的JavaScript函数</param>
        /// <returns>分页按钮html代码</returns>
        public string AjaxPagingButton(PaginationMode paginationMode, int totalRecords, int pageSize, int pageIndex, int numericPagingButtonCount, string pagingButtonUrlPattern, string pageIndexChangeCallback)
        {
            return PagingButton(paginationMode, totalRecords, pageSize, pageIndex, numericPagingButtonCount, pagingButtonUrlPattern, true, pageIndexChangeCallback);
        }

        public string ContentPagingButton(int totalPage, int cPageIndex, int numericPagingButtonCount, string pagingButtonUrlPattern)
        {
            return PagingButton(PaginationMode.NumericNextPrevious, totalPage, 1, cPageIndex, numericPagingButtonCount, pagingButtonUrlPattern);
        }

        /// <summary>
        /// 呈现分页按钮
        /// </summary>
        /// <param name="paginationMode">分页按钮显示模式</param>
        /// <param name="totalRecords">总记录数</param>
        /// <param name="pageSize">每页显示记录数</param>
        /// <param name="pageIndex">当前页码（从1开始）</param>
        /// <param name="numericPagingButtonCount">数字分页按钮显示个数</param>
        /// <param name="pagingButtonUrlPattern">链接格式（例如：http://www.spacebuilder.cn/blog/list?pageIndex={0}）</param>
        /// <param name="enableAjax">是否使用ajax分页</param>
        /// <param name="pageIndexChangeCallback">点击分页按钮时客户端回调的JavaScript函数</param>
        /// <returns>分页按钮html代码</returns>
        private string PagingButton(PaginationMode paginationMode, int totalRecords, int pageSize, int pageIndex, int numericPagingButtonCount, string pagingButtonUrlPattern, bool enableAjax, string pageIndexChangeCallback)
        {
            if (totalRecords == 0 || pageSize == 0)
                return string.Empty;

            //计算总页数
            int totalPages = totalRecords / pageSize;
            if ((totalRecords % pageSize) > 0)
                totalPages++;

            //未超过一页时不显示分页按钮
            if (totalPages <= 1)
                return string.Empty;

            bool showFirst = false;
            if (paginationMode == PaginationMode.NextPreviousFirstLast)
                showFirst = true;

            bool showLast = false;
            if (paginationMode == PaginationMode.NextPreviousFirstLast)
                showLast = true;

            bool showPrevious = true;
            //if (paginationMode == PaginationMode.NextPrevious || paginationMode == PaginationMode.NextPreviousFirstLast || paginationMode == PaginationMode.NumericNextPrevious)
            //    showPrevious = true;

            bool showNext = true;
            //if (paginationMode == PaginationMode.NextPrevious || paginationMode == PaginationMode.NextPreviousFirstLast || paginationMode == PaginationMode.NumericNextPrevious)
            //    showNext = true;

            bool showNumeric = false;
            if (paginationMode == PaginationMode.NumericNextPrevious)
                showNumeric = true;

            //显示多少个数字分页按钮
            //int numericPageButtonCount = 10;

            //对pageIndex进行修正
            if ((pageIndex < 1) || (pageIndex > totalPages))
                pageIndex = 1;

            StringBuilder pagingButtonsHtml = new StringBuilder("<div class=\"paginator\">");

            //构建 "首页"
            if (showFirst)
            {
                if ((pageIndex > 1) && (totalPages > numericPagingButtonCount))
                {
                    pagingButtonsHtml.AppendLine();
                    if (enableAjax)
                        pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(ResourceManager.GetString("Common_PagerFirstButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(1, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-first tn-page-thumb", GetPagingNavigateUrl(1, null)));
                    else
                        pagingButtonsHtml.AppendFormat(BuildLinkWithHref(ResourceManager.GetString("Common_PagerFirstButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(1, pagingButtonUrlPattern), "tn-page-first tn-page-thumb"));
                }
                else if (paginationMode == PaginationMode.NextPreviousFirstLast)
                {
                    pagingButtonsHtml.AppendLine();
                    pagingButtonsHtml.AppendFormat("<span class=\"tn-page-first tn-page-thumb\">{0}</span>", ResourceManager.GetString("Common_PagerFirstButton", Globals.GetCurrentUserLanguage()));
                }
            }

            //构建 "上一页"
            if (showPrevious)
            {
                pagingButtonsHtml.AppendLine();
                if (pageIndex == 1)
                    pagingButtonsHtml.AppendFormat("<span class=\"tn-page-prev tn-page-thumb\">{0}</span>", ResourceManager.GetString("Common_PagerPreviousButton", Globals.GetCurrentUserLanguage()));
                else
                {
                    if (enableAjax)
                        pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(ResourceManager.GetString("Common_PagerPreviousButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(pageIndex - 1, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-prev tn-page-thumb", GetPagingNavigateUrl(pageIndex - 1, null)));
                    else
                        pagingButtonsHtml.AppendFormat(BuildLinkWithHref(ResourceManager.GetString("Common_PagerPreviousButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(pageIndex - 1, pagingButtonUrlPattern), "tn-page-prev tn-page-thumb"));
                }
            }

            //构建 数字分页部分
            if (showNumeric)
            {
                int startNumericPageIndex;
                if (numericPagingButtonCount > totalPages || pageIndex - (numericPagingButtonCount / 2) <= 0)
                    startNumericPageIndex = 1;
                else if (pageIndex + (numericPagingButtonCount / 2) > totalPages)
                    startNumericPageIndex = totalPages - numericPagingButtonCount + 1;
                else
                    startNumericPageIndex = pageIndex - (numericPagingButtonCount / 2);

                if (startNumericPageIndex < 1)
                    startNumericPageIndex = 1;

                int lastNumericPageIndex = startNumericPageIndex + numericPagingButtonCount - 1;
                if (lastNumericPageIndex > totalPages)
                    lastNumericPageIndex = totalPages;

                if (startNumericPageIndex > 1)
                {
                    for (int i = 1; i < startNumericPageIndex; i++)
                    {
                        pagingButtonsHtml.AppendLine();

                        if (i > 3)
                            break;
                        if (i == 3)
                            pagingButtonsHtml.Append("<span class=\"tn-page-break\">...</span>");
                        else
                        {
                            if (pageIndex == i)
                            {
                                pagingButtonsHtml.AppendFormat("<span class=\"tn-page-number tn-selected\">{0}</span>", i);
                            }
                            else
                            {
                                if (enableAjax)
                                    pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(i.ToString(), GetPagingNavigateUrl(i, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-number", GetPagingNavigateUrl(i, null)));
                                else
                                    pagingButtonsHtml.AppendFormat(BuildLinkWithHref(i.ToString(), GetPagingNavigateUrl(i, pagingButtonUrlPattern), "tn-page-number"));
                            }
                        }
                    }
                }

                for (int i = startNumericPageIndex; i <= lastNumericPageIndex; i++)
                {
                    pagingButtonsHtml.AppendLine();
                    if (pageIndex == i)
                    {
                        pagingButtonsHtml.AppendFormat("<span class=\"tn-page-number tn-selected\">{0}</span>", i);
                    }
                    else
                    {
                        if (enableAjax)
                            pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(i.ToString(), GetPagingNavigateUrl(i, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-number", GetPagingNavigateUrl(i, null)));
                        else
                            pagingButtonsHtml.AppendFormat(BuildLinkWithHref(i.ToString(), GetPagingNavigateUrl(i, pagingButtonUrlPattern), "tn-page-number"));
                    }
                }

                if (lastNumericPageIndex < totalPages)
                {
                    int lastStart = lastNumericPageIndex + 1;
                    if (totalPages - lastStart > 2)
                        lastStart = totalPages - 2;

                    for (int i = lastStart; i <= totalPages; i++)
                    {
                        pagingButtonsHtml.AppendLine();
                        if ((i == lastStart) && (totalPages - lastNumericPageIndex > 3))
                        {
                            pagingButtonsHtml.AppendLine();
                            pagingButtonsHtml.Append("<span class=\"tn-page-break\">...</span>");
                            continue;
                        }

                        if (pageIndex == i)
                        {
                            pagingButtonsHtml.AppendFormat("<span class=\"tn-page-number tn-selected\">{0}</span>", i);
                        }
                        else
                        {
                            if (enableAjax)
                                pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(i.ToString(), GetPagingNavigateUrl(i, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-number", GetPagingNavigateUrl(i, null)));
                            else
                                pagingButtonsHtml.AppendFormat(BuildLinkWithHref(i.ToString(), GetPagingNavigateUrl(i, pagingButtonUrlPattern), "tn-page-number"));
                        }
                    }
                }

            }

            if (showNext)
            {
                pagingButtonsHtml.AppendLine();
                if (pageIndex == totalPages)
                    pagingButtonsHtml.AppendFormat("<span class=\"tn-page-thumb tn-page-next\">{0}</span>", ResourceManager.GetString("Common_PagerNextButton", Globals.GetCurrentUserLanguage()));
                else
                {
                    if (enableAjax)
                        pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(ResourceManager.GetString("Common_PagerNextButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(pageIndex + 1, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-thumb tn-page-next", GetPagingNavigateUrl(pageIndex + 1, null)));
                    else
                        pagingButtonsHtml.AppendFormat(BuildLinkWithHref(ResourceManager.GetString("Common_PagerNextButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(pageIndex + 1, pagingButtonUrlPattern), "tn-page-thumb tn-page-next"));
                }
            }

            if (showLast)
            {
                if ((pageIndex < totalPages) && (totalPages > numericPagingButtonCount))
                {
                    pagingButtonsHtml.AppendLine();

                    if (enableAjax)
                        pagingButtonsHtml.AppendFormat(BuildLinkWithOnClick(ResourceManager.GetString("Common_PagerLastButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(totalPages, pagingButtonUrlPattern), pageIndexChangeCallback, "tn-page-thumb tn-page-last", GetPagingNavigateUrl(totalPages, null)));
                    else
                        pagingButtonsHtml.AppendFormat(BuildLinkWithHref(ResourceManager.GetString("Common_PagerLastButton", Globals.GetCurrentUserLanguage()), GetPagingNavigateUrl(totalPages, pagingButtonUrlPattern), "tn-page-thumb tn-page-last"));
                }
                else if (paginationMode == PaginationMode.NextPreviousFirstLast)
                {
                    pagingButtonsHtml.AppendLine();
                    pagingButtonsHtml.AppendFormat("<span class=\"tn-page-thumb tn-page-last\">{0}</span>", ResourceManager.GetString("Common_PagerLastButton", Globals.GetCurrentUserLanguage()));
                }
            }
            pagingButtonsHtml.Append("</div>");
            return pagingButtonsHtml.ToString();
        }



        /// <summary>
        /// 构建分页按钮的链接
        /// </summary>
        /// <param name="pageIndex">当前页码</param>
        /// <param name="pagerButtonUrlPattern">链接格式（例如：http://www.spacebuilder.cn/blog/list?pageIndex={0}）</param>
        /// <returns>分页按钮的url字符串</returns>
        private string GetPagingNavigateUrl(int pageIndex, string pagingButtonUrlPattern)
        {
            if (!string.IsNullOrEmpty(pagingButtonUrlPattern))
            {
                // pagingButtonUrlPattern = HttpContext.Current.Server.UrlDecode(pagingButtonUrlPattern);
                return string.Format(pagingButtonUrlPattern, pageIndex);
            }

            string currentUrl = SpaceBuilder.Utils.WebUtils.HtmlEncode(HttpContext.Current.Request.RawUrl);

            if (currentUrl.IndexOf("?") == -1)
            {
                return currentUrl + string.Format("?pageIndex={0}", pageIndex);
            }
            else
            {
                if (currentUrl.IndexOf("pageIndex=", StringComparison.InvariantCultureIgnoreCase) == -1)
                    return currentUrl + string.Format("&pageIndex={0}", pageIndex);
                else
                    return Regex.Replace(currentUrl, @"pageIndex=(\d+\.?\d*|\.\d+)", "pageIndex=" + pageIndex, RegexOptions.IgnoreCase | RegexOptions.Compiled);
            }
        }

        /// <summary>
        /// 生成带Href的链接
        /// </summary>
        private string BuildLinkWithHref(string linkText, string url, string cssClassName)
        {
            return string.Format("<a href=\"{0}\" {1}>{2}</a>", url, string.IsNullOrEmpty(cssClassName) ? string.Empty : string.Format("class=\"{0}\"", cssClassName), linkText);
        }

        /// <summary>
        /// 生成带onclick的链接
        /// </summary>
        private string BuildLinkWithOnClick(string linkText, string ajaxUrl, string pageIndexChangeCallback, string cssClassName, string completePageUrl)
        {
            return string.Format("<a onclick=\"{0}('{1}'); return false;\" href=\"{2}\" {3}>{4}</a>", pageIndexChangeCallback, ajaxUrl, completePageUrl, string.IsNullOrEmpty(cssClassName) ? string.Empty : string.Format("class=\"{0}\"", cssClassName), linkText);
        }

    }



    /// <summary>
    /// 分页按钮显示模式
    /// </summary>
    public enum PaginationMode
    {
        /// <summary>
        /// 上一页/下一页 模式
        /// </summary>
        NextPrevious,

        /// <summary>
        /// 首页/末页/上一页/下一页 模式
        /// </summary>
        NextPreviousFirstLast,

        /// <summary>
        /// 上一页/下一页 + 数字 模式，例如： 上一页 1 2 3 4 5 下一页
        /// </summary>
        NumericNextPrevious,
    }
}
