﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using System.Web.Mvc.Html;
using SpaceBuilder.Common;
using System.Web.Routing;
using System.Text.RegularExpressions;
using SpaceBuilder.Utils;
using SpaceBuilder.Club;

namespace SpaceBuilder.Web
{
    /// <summary>
    /// Widget
    /// </summary>
    public partial class SPBHtmlHelper
    {
        /// <summary>
        /// 在空间中呈现Widget
        /// </summary>
        /// <param name="htmlHelper"></param>
        /// <param name="isDesignMode">是否处于设计模式</param>
        /// <param name="widgetInstanceID"></param>
        /// <param name="owner"></param>
        public void RenderWidgetInstanceInUserDomain(HtmlHelper htmlHelper, bool isDesignMode, WidgetInstance widgetInstance, User owner)
        {
            if (widgetInstance != null)
            {
                int zoneID = widgetInstance.ZoneID;
                int displayOrder = widgetInstance.DisplayOrder;

                widgetInstance = WidgetInstances.GetWidgetInstance(widgetInstance.InstanceID, true);

                //DisplayOrder & ZoneID可能未来得及更新Single WidgetInstance
                widgetInstance.ZoneID = zoneID;
                widgetInstance.DisplayOrder = displayOrder;

                User currentUser = Globals.GetCurrentUser();

                //验证当前用户是否可以设置该Widget
                if (isDesignMode)
                {
                    if (!currentUser.IsContentAdministrator && owner.UserID != currentUser.UserID)
                        return;
                }
                else
                {
                    //验证当前用户是否具有浏览权限
                    if (!HasPermission(widgetInstance.VisibleRole, owner, currentUser))
                        return;
                }

                ViewDataDictionary viewDataDictionary = new ViewDataDictionary();
                if (!string.IsNullOrEmpty(widgetInstance.TitleUrl))
                {
                    string titleUrl = GenerateUrlFromWidgetUrlExpression(widgetInstance.TitleUrl, widgetInstance, owner.UserName, owner.UserID);
                    viewDataDictionary.Add("titleHyperLink", new HyperLinkEntity(widgetInstance.ProcessResourceToken(widgetInstance.Title, Globals.GetCurrentUserLanguage()), titleUrl, widgetInstance.TitleUrlTargetBlank, widgetInstance.TitleToolTips));
                }
                viewDataDictionary.Add("isDesignMode", isDesignMode);

                if (widgetInstance.WidgetType == WidgetTypes.Action)
                {
                    RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(widgetInstance.MvcAction.Parameters, widgetInstance, owner.UserName, owner.UserID);
                    viewDataDictionary.Add("routeValueDictionary", routeValueDictionary);

                    if (!string.IsNullOrEmpty(widgetInstance.ShortcutUrl) && HasPermission(widgetInstance.ShortcutVisibleRole, owner, currentUser))
                    {
                        string shortcutUrl = GenerateUrlFromWidgetUrlExpression(widgetInstance.ShortcutUrl, widgetInstance, owner.UserName, owner.UserID);
                        viewDataDictionary.Add("shortcutHyperLink", new HyperLinkEntity(widgetInstance.ProcessResourceToken(widgetInstance.ShortcutName, Globals.GetCurrentUserLanguage()), shortcutUrl, widgetInstance.ShortcutUrlTargetBlank, widgetInstance.ShortcutToolTips));
                    }

                    htmlHelper.RenderPartial("Controls/ActionWidgetInstance.ascx", widgetInstance, viewDataDictionary);
                }
                else if (widgetInstance.WidgetType == WidgetTypes.Html)
                {
                    htmlHelper.RenderPartial("Controls/HtmlWidgetInstance.ascx", widgetInstance, viewDataDictionary);
                }
            }
        }

        /// <summary>
        /// 在空间中呈现Widget
        /// </summary>
        /// <param name="htmlHelper"></param>
        /// <param name="widgetID"></param>
        /// <param name="owner"></param>
        public void RenderWidgetInUserDomain(HtmlHelper htmlHelper, string widgetID, User owner)
        {
            Widget widget = Widgets.GetWidget(widgetID, false);
            if (widget != null)
            {
                User currentUser = Globals.GetCurrentUser();

                //验证当前用户是否具有浏览权限
                if (!HasPermission(widget.VisibleRole, owner, currentUser))
                    return;

                ViewDataDictionary viewDataDictionary = new ViewDataDictionary();

                RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(widget.MvcAction.Parameters, widget, owner.UserName, owner.UserID);
                viewDataDictionary.Add("routeValueDictionary", routeValueDictionary);

                if (widget.HasBorder)
                {
                    if (!string.IsNullOrEmpty(widget.TitleUrl))
                    {
                        string titleUrl = GenerateUrlFromWidgetUrlExpression(widget.TitleUrl, widget, owner.UserName, owner.UserID);
                        viewDataDictionary.Add("titleHyperLink", new HyperLinkEntity(widget.ProcessResourceToken(widget.Title, Globals.GetCurrentUserLanguage()), titleUrl, widget.TitleUrlTargetBlank, widget.TitleToolTips));
                    }

                    if (!string.IsNullOrEmpty(widget.ShortcutUrl) && HasPermission(widget.ShortcutVisibleRole, owner, currentUser))
                    {
                        string shortcutUrl = GenerateUrlFromWidgetUrlExpression(widget.ShortcutUrl, widget, owner.UserName, owner.UserID);
                        viewDataDictionary.Add("shortcutHyperLink", new HyperLinkEntity(widget.ProcessResourceToken(widget.ShortcutName, Globals.GetCurrentUserLanguage()), shortcutUrl, widget.ShortcutUrlTargetBlank, widget.ShortcutToolTips));
                    }
                }

                htmlHelper.RenderPartial("Controls/ActionWidget.ascx", widget, viewDataDictionary);
            }
        }



        /// <summary>
        /// 在群组中呈现Widget
        /// </summary>
        /// <param name="htmlHelper"></param>
        /// <param name="isDesignMode">是否处于设计模式</param>
        /// <param name="widgetInstanceID"></param>
        /// <param name="owner"></param>
        public void RenderWidgetInstanceInClubDomain(HtmlHelper htmlHelper, bool isDesignMode, WidgetInstance widgetInstance, Club.Club owner)
        {
            if (widgetInstance != null)
            {
                int zoneID = widgetInstance.ZoneID;
                int displayOrder = widgetInstance.DisplayOrder;

                widgetInstance = WidgetInstances.GetWidgetInstance(widgetInstance.InstanceID, false);

                //DisplayOrder & ZoneID可能未来得及更新Single WidgetInstance
                widgetInstance.ZoneID = zoneID;
                widgetInstance.DisplayOrder = displayOrder;

                User currentUser = Globals.GetCurrentUser();

                //验证当前用户是否可以设置该Widget
                if (isDesignMode)
                {
                    if (!currentUser.IsContentAdministrator && !SpaceBuilder.Club.ClubMembers.HasClubManagerPermission(owner.ClubID, currentUser.UserID))
                        return;
                }
                else
                {
                    //验证当前用户是否具有浏览权限
                    if (widgetInstance.VisibleRole != WidgetVisibleRoles.All)
                    {
                        //暂时不处理Owner
                        //if (widgetInstance.VisibleRole == WidgetVisibleRoles.Other && SpaceBuilder.Club.ClubMembers.HasClubManagerPermission(owner.ClubID, currentUser.UserID))
                        //    return;

                        if (widgetInstance.VisibleRole == WidgetVisibleRoles.Anonymous && currentUser != null)
                            return;

                        if (widgetInstance.VisibleRole == WidgetVisibleRoles.RegisteredUsers && currentUser == null)
                            return;
                    }
                }

                ViewDataDictionary viewDataDictionary = new ViewDataDictionary();
                if (!string.IsNullOrEmpty(widgetInstance.TitleUrl))
                {
                    string titleUrl = GenerateUrlFromWidgetUrlExpression(widgetInstance.TitleUrl, widgetInstance, owner.DomainName, owner.ClubID);
                    viewDataDictionary.Add("titleHyperLink", new HyperLinkEntity(widgetInstance.ProcessResourceToken(widgetInstance.Title, Globals.GetCurrentUserLanguage()), titleUrl, widgetInstance.TitleUrlTargetBlank, widgetInstance.TitleToolTips));
                }
                viewDataDictionary.Add("isDesignMode", isDesignMode);

                if (widgetInstance.WidgetType == WidgetTypes.Action)
                {
                    RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(widgetInstance.MvcAction.Parameters, widgetInstance, owner.DomainName, owner.ClubID);
                    viewDataDictionary.Add("routeValueDictionary", routeValueDictionary);

                    if (!string.IsNullOrEmpty(widgetInstance.ShortcutUrl) && HasPermission(widgetInstance.ShortcutVisibleRole, owner, currentUser))
                    {
                        string shortcutUrl = GenerateUrlFromWidgetUrlExpression(widgetInstance.ShortcutUrl, widgetInstance, owner.DomainName, owner.ClubID);
                        viewDataDictionary.Add("shortcutHyperLink", new HyperLinkEntity(widgetInstance.ProcessResourceToken(widgetInstance.ShortcutName, Globals.GetCurrentUserLanguage()), shortcutUrl, widgetInstance.ShortcutUrlTargetBlank, widgetInstance.ShortcutToolTips));
                    }

                    htmlHelper.RenderPartial("Controls/ActionWidgetInstance.ascx", widgetInstance, viewDataDictionary);
                }
                else if (widgetInstance.WidgetType == WidgetTypes.Html)
                {
                    htmlHelper.RenderPartial("Controls/HtmlWidgetInstance.ascx", widgetInstance, viewDataDictionary);
                }
            }
        }


        /// <summary>
        /// 在群组中呈现Widget
        /// </summary>
        /// <param name="htmlHelper"></param>
        /// <param name="widgetID"></param>
        /// <param name="owner"></param>
        public void RenderWidgetInClubDomain(HtmlHelper htmlHelper, string widgetID, Club.Club owner)
        {
            Widget widget = Widgets.GetWidget(widgetID, false);
            if (widget != null)
            {
                User currentUser = Globals.GetCurrentUser();

                //验证当前用户是否具有浏览权限
                if (widget.VisibleRole != WidgetVisibleRoles.All)
                {
                    if (widget.VisibleRole == WidgetVisibleRoles.Anonymous && currentUser != null)
                        return;

                    if (widget.VisibleRole == WidgetVisibleRoles.RegisteredUsers && currentUser == null)
                        return;

                    if (widget.VisibleRole == WidgetVisibleRoles.Owner && (!currentUser.IsContentAdministrator && !ClubMembers.HasClubManagerPermission(owner.ClubID, currentUser.UserID)))
                        return;
                }

                ViewDataDictionary viewDataDictionary = new ViewDataDictionary();

                RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(widget.MvcAction.Parameters, widget, owner.DomainName, owner.ClubID);
                viewDataDictionary.Add("routeValueDictionary", routeValueDictionary);

                if (widget.HasBorder)
                {
                    if (!string.IsNullOrEmpty(widget.TitleUrl))
                    {
                        string titleUrl = GenerateUrlFromWidgetUrlExpression(widget.TitleUrl, widget, owner.DomainName, owner.ClubID);
                        viewDataDictionary.Add("titleHyperLink", new HyperLinkEntity(widget.ProcessResourceToken(widget.Title, Globals.GetCurrentUserLanguage()), titleUrl, widget.TitleUrlTargetBlank, widget.TitleToolTips));
                    }

                    if (!string.IsNullOrEmpty(widget.ShortcutUrl) && HasPermission(widget.ShortcutVisibleRole, owner, currentUser))
                    {
                        string shortcutUrl = GenerateUrlFromWidgetUrlExpression(widget.ShortcutUrl, widget, owner.DomainName, owner.ClubID);
                        viewDataDictionary.Add("shortcutHyperLink", new HyperLinkEntity(widget.ProcessResourceToken(widget.ShortcutName, Globals.GetCurrentUserLanguage()), shortcutUrl, widget.ShortcutUrlTargetBlank, widget.ShortcutToolTips));
                    }
                }

                htmlHelper.RenderPartial("Controls/ActionWidget.ascx", widget, viewDataDictionary);
            }
        }


        /// <summary>
        /// 在频道中呈现Widget
        /// </summary>
        /// <param name="htmlHelper"></param>
        /// <param name="isDesignMode">是否处于设计模式</param>
        /// <param name="widgetInstanceID"></param>
        public void RenderWidgetInstanceInChannel(HtmlHelper htmlHelper, bool isDesignMode, WidgetInstance widgetInstance)
        {
            if (widgetInstance != null)
            {
                int zoneID = widgetInstance.ZoneID;
                int displayOrder = widgetInstance.DisplayOrder;

                widgetInstance = WidgetInstances.GetWidgetInstance(widgetInstance.InstanceID, false);

                //DisplayOrder & ZoneID可能未来得及更新Single WidgetInstance
                widgetInstance.ZoneID = zoneID;
                widgetInstance.DisplayOrder = displayOrder;

                User currentUser = Globals.GetCurrentUser();

                //验证当前用户是否有权限
                if (isDesignMode && !currentUser.IsContentAdministrator)
                    return;

                //验证当前用户是否可以设置该Widget
                if (isDesignMode)
                {
                    if (!currentUser.IsContentAdministrator)
                        return;
                }
                else
                {
                    //验证当前用户是否具有浏览权限
                    if (widgetInstance.VisibleRole != WidgetVisibleRoles.All)
                    {
                        if (widgetInstance.VisibleRole == WidgetVisibleRoles.Anonymous && currentUser != null)
                            return;

                        if (widgetInstance.VisibleRole == WidgetVisibleRoles.RegisteredUsers && currentUser == null)
                            return;
                    }
                }

                ViewDataDictionary viewDataDictionary = new ViewDataDictionary();
                if (!string.IsNullOrEmpty(widgetInstance.TitleUrl))
                {
                    string titleUrl = GenerateUrlFromWidgetUrlExpression(widgetInstance.TitleUrl, widgetInstance);
                    viewDataDictionary.Add("titleHyperLink", new HyperLinkEntity(widgetInstance.ProcessResourceToken(widgetInstance.Title, Globals.GetCurrentUserLanguage()), titleUrl, widgetInstance.TitleUrlTargetBlank, widgetInstance.TitleToolTips));
                }
                viewDataDictionary.Add("isDesignMode", isDesignMode);

                if (widgetInstance.WidgetType == WidgetTypes.Action)
                {
                    RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(widgetInstance.MvcAction.Parameters, widgetInstance);
                    viewDataDictionary.Add("routeValueDictionary", routeValueDictionary);

                    if (!string.IsNullOrEmpty(widgetInstance.ShortcutUrl))
                    {
                        string shortcutUrl = GenerateUrlFromWidgetUrlExpression(widgetInstance.ShortcutUrl, widgetInstance);
                        viewDataDictionary.Add("shortcutHyperLink", new HyperLinkEntity(widgetInstance.ProcessResourceToken(widgetInstance.ShortcutName, Globals.GetCurrentUserLanguage()), shortcutUrl, widgetInstance.ShortcutUrlTargetBlank, widgetInstance.ShortcutToolTips));
                    }

                    htmlHelper.RenderPartial("Controls/ActionWidgetInstance.ascx", widgetInstance, viewDataDictionary);
                }
                else if (widgetInstance.WidgetType == WidgetTypes.Html)
                {
                    htmlHelper.RenderPartial("Controls/HtmlWidgetInstance.ascx", widgetInstance, viewDataDictionary);
                }
            }
        }

        /// <summary>
        /// 在频道中呈现Widget
        /// </summary>
        /// <param name="htmlHelper"></param>
        /// <param name="widgetID"></param>
        public void RenderWidgetInChannel(HtmlHelper htmlHelper, string widgetID)
        {
            Widget widget = Widgets.GetWidget(widgetID, false);
            if (widget != null)
            {
                User currentUser = Globals.GetCurrentUser();

                //验证当前用户是否具有浏览权限
                if (widget.VisibleRole != WidgetVisibleRoles.All)
                {
                    if (widget.VisibleRole == WidgetVisibleRoles.Anonymous && currentUser != null)
                        return;

                    if (widget.VisibleRole == WidgetVisibleRoles.RegisteredUsers && currentUser == null)
                        return;
                }

                ViewDataDictionary viewDataDictionary = new ViewDataDictionary();

                RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(widget.MvcAction.Parameters, widget);
                viewDataDictionary.Add("routeValueDictionary", routeValueDictionary);

                if (widget.HasBorder)
                {
                    if (!string.IsNullOrEmpty(widget.TitleUrl))
                    {
                        string titleUrl = GenerateUrlFromWidgetUrlExpression(widget.TitleUrl, widget);
                        viewDataDictionary.Add("titleHyperLink", new HyperLinkEntity(widget.ProcessResourceToken(widget.Title, Globals.GetCurrentUserLanguage()), titleUrl, widget.TitleUrlTargetBlank, widget.TitleToolTips));
                    }

                    if (!string.IsNullOrEmpty(widget.ShortcutUrl))
                    {
                        string shortcutUrl = GenerateUrlFromWidgetUrlExpression(widget.ShortcutUrl, widget);
                        viewDataDictionary.Add("shortcutHyperLink", new HyperLinkEntity(widget.ProcessResourceToken(widget.ShortcutName, Globals.GetCurrentUserLanguage()), shortcutUrl, widget.ShortcutUrlTargetBlank, widget.ShortcutToolTips));
                    }
                }

                htmlHelper.RenderPartial("Controls/ActionWidget.ascx", widget, viewDataDictionary);
            }
        }


        /// <summary>
        /// 检查currentUser是否符合visibleRoles要求
        /// </summary>
        /// <param name="visibleRoles"></param>
        /// <param name="owner"></param>
        /// <param name="currentUser"></param>
        /// <returns></returns>
        private static bool HasPermission(WidgetVisibleRoles visibleRoles, User owner, User currentUser)
        {
            bool result = true;
            switch (visibleRoles)
            {
                case WidgetVisibleRoles.Anonymous:
                    if (currentUser != null)
                        result = false;
                    break;
                case WidgetVisibleRoles.Other:
                    if (currentUser == null || owner.UserID == currentUser.UserID)
                        result = false;
                    break;
                case WidgetVisibleRoles.Owner:
                    if (currentUser == null || !currentUser.IsContentAdministrator && owner.UserID != currentUser.UserID)
                        result = false;
                    break;
                case WidgetVisibleRoles.RegisteredUsers:
                    if (currentUser == null)
                        result = false;
                    break;
            }
            return result;
        }

        /// <summary>
        /// 检查currentUser是否符合visibleRoles要求
        /// </summary>
        /// <param name="visibleRoles"></param>
        /// <param name="owner"></param>
        /// <param name="currentUser"></param>
        /// <returns></returns>
        private static bool HasPermission(WidgetVisibleRoles visibleRoles, SpaceBuilder.Club.Club currentClub, User currentUser)
        {
            bool result = true;
            switch (visibleRoles)
            {
                case WidgetVisibleRoles.Anonymous:
                    if (currentUser != null)
                        result = false;
                    break;
                case WidgetVisibleRoles.Other:
                    //无意义
                    break;
                case WidgetVisibleRoles.Owner:
                    if (currentUser == null || (!currentUser.IsContentAdministrator && !ClubMembers.HasClubManagerPermission(currentClub.ClubID, currentUser.UserID)))
                        result = false;
                    break;
                case WidgetVisibleRoles.RegisteredUsers:
                    if (currentUser == null)
                        result = false;
                    break;
            }
            return result;
        }

        /// <summary>
        /// 生成RouteValueDictionary(需要处理CONTEXT及UP变量)
        /// </summary>
        /// <param name="widget"></param>
        /// <param name="mvcActionParameters"></param>
        /// <returns></returns>
        private static RouteValueDictionary GenerateRouteValueDictionary(Dictionary<string, string> mvcActionParameters, Widget widget)
        {
            return GenerateRouteValueDictionary(mvcActionParameters, widget, string.Empty, -1);
        }

        /// <summary>
        /// 生成RouteValueDictionary(需要处理CONTEXT及UP变量)
        /// </summary>
        /// <param name="widget"></param>
        /// <param name="mvcActionParameters"></param>
        /// <param name="ownerDomainName"></param>
        /// <param name="ownerID"></param>
        /// <returns></returns>
        private static RouteValueDictionary GenerateRouteValueDictionary(Dictionary<string, string> mvcActionParameters, Widget widget, string ownerDomainName, int ownerID)
        {
            RouteValueDictionary routeValueDictionary = new RouteValueDictionary();
            foreach (var parameter in mvcActionParameters)
            {
                string parameterValue = parameter.Value;
                parameterValue = widget.ProcessUserPrefToken(parameterValue);

                if (ownerID > 0)
                {
                    parameterValue = parameterValue.Replace("__CONTEXT_OwnerID__", ownerID.ToString());
                    parameterValue = parameterValue.Replace("__CONTEXT_OwnerDomainName__", ownerDomainName);
                }

                routeValueDictionary[parameter.Key] = parameterValue;
            }
            return routeValueDictionary;
        }

        /// <summary>
        /// 生成Widget表达式的Url
        /// </summary>
        /// <param name="widgetUrlExpression"></param>
        /// <param name="widget"></param>
        /// <returns></returns>
        private static string GenerateUrlFromWidgetUrlExpression(string widgetUrlExpression, Widget widget)
        {
            return GenerateUrlFromWidgetUrlExpression(widgetUrlExpression, widget, string.Empty, -1);
        }

        /// <summary>
        /// 生成Widget表达式的Url
        /// </summary>
        /// <param name="widgetUrlExpression"></param>
        /// <param name="widget"></param>
        /// <param name="ownerDomainName"></param>
        /// <param name="ownerID"></param>
        /// <returns></returns>
        private static string GenerateUrlFromWidgetUrlExpression(string widgetUrlExpression, Widget widget, string ownerDomainName, int ownerID)
        {
            string url;
            if (widgetUrlExpression.StartsWith("MvcAction://", StringComparison.InvariantCultureIgnoreCase))
            {
                Widget.MvcActionEntity mvcAction = new Widget.MvcActionEntity(widgetUrlExpression);
                RouteValueDictionary routeValueDictionary = GenerateRouteValueDictionary(mvcAction.Parameters, widget, ownerDomainName, ownerID);
                url = SPBUrlHelper.Action(mvcAction.ActionName, mvcAction.ControllerName, routeValueDictionary);
            }
            else
            {
                url = widgetUrlExpression;
                url = widget.ProcessUserPrefToken(url);

                if (ownerID > 0)
                {
                    url = url.Replace("__CONTEXT_OwnerID__", ownerID.ToString());
                    url = url.Replace("__CONTEXT_OwnerDomainName__", ownerDomainName);
                }


                if (widgetUrlExpression.StartsWith("~/"))
                    url = WebUtils.ResolveUrl(url);
            }
            return url;
        }


    }
}
