﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.UI;
using System.Web;
using System.Collections.Specialized;

using SpaceBuilder.Utils;
using SpaceBuilder.Common;

namespace SpaceBuilder.Web
{
    [PersistChildren(true), ParseChildren(false), ToolboxData("<{0}:MobileHeader runat=\"server\"></{0}:MobileHeader>")]
    public class MobileHeader : Control
    {
        protected override void Render(HtmlTextWriter writer)
        {
            writer.WriteLine("<head>");
            writer.WriteLine("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />");
            if (DisableClientCache)
            {
                writer.WriteLine("<meta http-equiv=\"Pragma\" content=\"no-cache\" />\n");
                writer.WriteLine("<meta http-equiv=\"no-cache\" />\n");
                writer.WriteLine("<meta http-equiv=\"Expires\" content=\"-1\" />\n");
                writer.WriteLine("<meta http-equiv=\"Cache-Control\" content=\"no-cache\" />\n");
            }

            SiteSettings siteSettings = SiteSettingsManager.GetSiteSettings();
            RenderTitle(siteSettings, writer);
            RenderMetaTags(siteSettings, writer);
            RenderLinkTags(siteSettings, writer);

            RenderStyleSheets(writer);
            RenderAutoDiscoverySyndication(writer);

            RenderAdditionHealder(siteSettings, writer);
            RenderRawContent(writer);
            base.Render(writer);
            RenderLogoStyle(writer);
            writer.WriteLine("</head>");
        }

        /// <summary>
        /// Render favicon.ico
        /// </summary>
        protected virtual void RenderFavicon(HtmlTextWriter writer)
        {
            writer.WriteLine("<link rel=\"shortcut icon\" type=\"image/ico\" href=\"{0}\" />", Globals.GetFullUrlInResourceSite("~/favicon.ico"));
        }

        #region Render Methods

        /// <summary>
        /// Render AdditionHealder
        /// </summary>
        protected virtual void RenderAdditionHealder(SiteSettings settings, HtmlTextWriter writer)
        {
            if (!string.IsNullOrEmpty(settings.RawAdditionalHeader))
                writer.WriteLine(settings.RawAdditionalHeader);
        }

        /// <summary>
        /// Render rss(及其他服务)自动发现
        /// </summary>
        protected virtual void RenderAutoDiscoverySyndication(HtmlTextWriter writer)
        {
            List<string> autoDiscovery = Context.Items[autoDiscoveryKey] as List<string>;
            if (autoDiscovery != null && autoDiscovery.Count > 0)
            {
                foreach (string item in autoDiscovery)
                {
                    writer.WriteLine(item);
                }
            }
        }

        /// <summary>
        /// Render css样式
        /// </summary>
        /// <param name="writer"></param>
        protected virtual void RenderStyleSheets(HtmlTextWriter writer)
        {
            RenderStyleSheets(writer, StyleRelativePosition.First);
            RenderStyleSheets(writer, StyleRelativePosition.Unspecified);
            RenderStyleSheets(writer, StyleRelativePosition.Last);
        }
        protected virtual void RenderStyleSheets(HtmlTextWriter writer, StyleRelativePosition position)
        {
            Queue<StyleQueueItem> queueInContext = Context.Items[styleKey] as Queue<StyleQueueItem>;
            if (queueInContext != null && queueInContext.Count > 0)
            {
                Queue<StyleQueueItem>.Enumerator ie = queueInContext.GetEnumerator();
                while (ie.MoveNext())
                {
                    StyleQueueItem si = ie.Current;
                    if (si.Position == position)
                    {
                        if (this.EnableStyleHttpCompress)
                        {
                            string[] fileParts = si.Url.Split('?');

                            string queryString = string.Empty;
                            if (fileParts.Length > 1)
                                queryString = "?" + fileParts[1];

                            writer.WriteLine(styleCompressFormat, fileParts[0], queryString, si.Media);
                        }
                        else
                        {
                            writer.WriteLine(styleFormat, si.Url, si.Media);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Render meta(description,keywords)
        /// </summary>
        /// <param name="settings"></param>
        /// <param name="writer"></param>
        protected virtual void RenderMetaTags(SiteSettings settings, HtmlTextWriter writer)
        {
            NameValueCollection metaTags = Context.Items[metaKey] as NameValueCollection;
            if (metaTags == null)
                metaTags = new NameValueCollection();

            //如果未明确使用description及keywords则不生成该标签
            //if (string.IsNullOrEmpty(metaTags["description"]) && !string.IsNullOrEmpty(settings.SearchMetaDescription))
            //{
            //    metaTags["description"] = settings.SearchMetaDescription;
            //}

            //if (string.IsNullOrEmpty(metaTags["keywords"]) && !string.IsNullOrEmpty(settings.SearchMetaKeywords))
            //{
            //    metaTags["keywords"] = settings.SearchMetaKeywords;
            //}

            metaTags["GENERATOR"] = SPBConfig.Instance().SpaceBuilderVersionInfo;

            foreach (string key in metaTags.Keys)
            {
                writer.WriteLine(metaFormat, key, metaTags[key]);
            }
        }

        protected virtual void RenderLinkTags(SiteSettings settings, HtmlTextWriter writer)
        {
            NameValueCollection linkTags = Context.Items[linkKey] as NameValueCollection;
            if (linkTags == null)
                linkTags = new NameValueCollection();

            foreach (string key in linkTags.Keys)
            {
                writer.WriteLine(linkFormat, key, linkTags[key]);
            }
        }

        protected virtual void RenderTitle(SiteSettings settings, HtmlTextWriter writer)
        {
            string title = Context.Items[titleKey] as string;
            if (string.IsNullOrEmpty(title))
                title = settings.SiteName;

            writer.WriteLine("<title>{0}</title>", title);
        }

        protected virtual void RenderLogoStyle(HtmlTextWriter writer)
        {
            string logoUrl = Context.Items[logoUrlKey] as string;
            if (string.IsNullOrEmpty(logoUrl))
                logoUrl = "~/Images/logo.png";

            writer.WriteLine("<!-- InstanceBeginEditable name=\"head\" -->");
            writer.WriteLine("<!-- InstanceEndEditable -->");
            writer.WriteLine("<!-- InstanceParam name=\"spb-channel-page\" type=\"text\" value=\"\" -->");
            writer.WriteLine("<style type=\"text/css\">");
            writer.WriteLine("<!--");
            writer.WriteLine("#spb-logo a {");
            writer.WriteLine(string.Format("background: url({0}) no-repeat;", Globals.GetFullUrlInResourceSite(logoUrl)));
            writer.WriteLine("_background: none;");
            writer.WriteLine(string.Format("_filter: progid:DXImageTransform.Microsoft.AlphaImageLoader(sizingMethod=crop, src='{0}');", Globals.GetFullUrlInResourceSite(logoUrl)));
            writer.WriteLine("}");
            writer.WriteLine("-->");
            writer.WriteLine("</style>");
        }

        protected virtual void RenderRawContent(HtmlTextWriter writer)
        {
            List<string> rawContent = Context.Items[rawContentKey] as List<string>;
            if (rawContent == null)
                rawContent = new List<string>();

            foreach (string item in rawContent)
            {
                writer.WriteLine(item);
            }
        }

        #endregion

        #region Static Keys
        private static readonly string metaFormat = "<meta name=\"{0}\" content=\"{1}\" />";
        private static readonly string titleKey = "SpaceBuilder.Title.Value";
        private static readonly string logoUrlKey = "SpaceBuilder.LogoUrl.Value";
        private static readonly string metaKey = "SpaceBuilder.MetaTags";
        private static readonly string autoDiscoveryKey = "SpaceBuilder.AutoDiscovery";
        private static readonly string styleKey = "SpaceBuilder.StyleTags";

        private static readonly string styleFormat = "<link rel=\"stylesheet\" href=\"{0}\" type=\"text/css\" media=\"{1}\" />";
        //使用http压缩
        private static readonly string styleCompressFormat = "<link rel=\"stylesheet\" href=\"{0}.compress.axd{1}\" type=\"text/css\" media=\"{2}\" />";

        private static readonly string autoDiscoveryLinkFormat = "<link rel=\"{0}\" type=\"application/{1}+xml\" title=\"{2}\" href=\"{3}\"  />";
        private static readonly string linkFormat = "<link rel=\"{0}\" href=\"{1}\" />";
        private static readonly string linkKey = "SpaceBuilder.LinkTags";
        private static readonly string rawContentKey = "SpaceBuilder.RawHeaderContent";
        #endregion

        #region MetaTags
        /// <summary>
        /// Adds the description meta tag.
        /// </summary>
        public static void AddMetaDescription(string value)
        {
            AddMetaTag("description", value);
        }

        /// <summary>
        /// Adds the keywork meta tag
        /// </summary>
        public static void AddMetaKeywords(string value)
        {
            AddMetaTag("keywords", value);
        }

        /// <summary>
        /// Adds a new meta tag key and value
        /// </summary>
        public static void AddMetaTag(string key, string value)
        {
            if (!string.IsNullOrEmpty(key) && !string.IsNullOrEmpty(value))
            {
                NameValueCollection mc = HttpContext.Current.Items[metaKey] as NameValueCollection;
                if (mc == null)
                {
                    mc = new NameValueCollection();
                    HttpContext.Current.Items.Add(metaKey, mc);
                }
                mc[key] = MetaTagFormatter(value);
            }
        }
        #endregion

        #region Title
        public static void AddTitle(string title)
        {
            HttpContext.Current.Items[titleKey] = title;
        }

        /// <summary>
        /// 清空html页面的title
        /// </summary>
        /// <param name="context"><see cref="HttpContext"/></param>
        public static void ClearTitle(HttpContext context)
        {
            context.Items[titleKey] = string.Empty;
        }
        #endregion

        #region Style
        /// <summary>
        /// Adds a style to a Queue collection. Style items are always rendered first in first out
        /// Although some some control can be offered using StyleRelativePosition
        /// </summary>
        /// <param name="url">Fully qualified url to style sheet</param>
        /// <param name="media"></param>
        /// <param name="context"></param>
        public static void AddStyle(string url, string media)
        {
            AddStyle(url, media, StyleRelativePosition.Unspecified);
        }
        public static void AddStyle(string url, string media, StyleRelativePosition position)
        {
            Queue<StyleQueueItem> styleQueue = HttpContext.Current.Items[styleKey] as Queue<StyleQueueItem>;
            if (styleQueue == null)
            {
                styleQueue = new Queue<StyleQueueItem>();
                HttpContext.Current.Items[styleKey] = styleQueue;
            }
            styleQueue.Enqueue(new StyleQueueItem(url, media, position));
        }
        #endregion

        #region Logo
        public static void SetLogoUrl(string logoUrl)
        {
            HttpContext.Current.Items[logoUrlKey] = logoUrl;
        }
        #endregion

        #region Syndication AutoDiscovery

        /// <summary>
        /// Adds a RSS 2.0 autodiscoverable link to the header
        /// </summary>
        public static void AddRssAutoDiscovery(string url)
        {
            SetAutoDiscoverty(url, "RSS (RSS 2.0)", "alternate", "rss");
        }

        /// <summary>
        /// Adds an Atom 1.0 autodiscoverale link to the header
        /// </summary>
        public static void AddAtomDiscovery(string url)
        {
            SetAutoDiscoverty(url, " Atom (Atom 1.0)", "alternate", "atom");
        }

        /// <summary>
        /// RSD 自动发现
        /// </summary>
        public static void AddRSDDiscovery(string url)
        {
            SetAutoDiscoverty(url, "RSD", "rsd", "EditURI");
        }

        /// <summary>
        /// 用于Live writer 自动发现
        /// </summary>
        public static void AddWLWManifestDiscovery(string url)
        {
            SetAutoDiscoverty(url, "WLWManifest", "wlwmanifest", "wlwmanifest");
        }

        private static void SetAutoDiscoverty(string url, string title, string autoType, string rel)
        {
            HttpContext context = HttpContext.Current;
            List<string> list = context.Items[autoDiscoveryKey] as List<string>;
            if (list == null)
            {
                list = new List<string>();
                context.Items.Add(autoDiscoveryKey, list);
            }

            string autoDiscovertyLink = string.Format(autoDiscoveryLinkFormat, rel, autoType, title, WebUtils.FullPath(url));
            if (!list.Contains(autoDiscovertyLink))
                list.Add(autoDiscovertyLink);
        }

        #endregion

        #region Link Tags
        /// <summary>
        /// Adds a new link tag rel and href
        /// </summary>
        public static void AddLinkTag(string rel, string href, HttpContext context)
        {
            if (!string.IsNullOrEmpty(rel) && !string.IsNullOrEmpty(href))
            {
                NameValueCollection lc = context.Items[linkKey] as NameValueCollection;
                if (lc == null)
                {
                    lc = new NameValueCollection();
                    context.Items.Add(linkKey, lc);
                }
                lc[rel] = href;
            }
        }
        #endregion

        #region Raw Content

        /// <summary>
        /// Adds raw content to the HTML Header, such as script blocks or custom tags
        /// </summary>
        public static void AddRawContent(string content, HttpContext context)
        {
            if (!string.IsNullOrEmpty(content))
            {
                List<string> mc = context.Items[rawContentKey] as List<string>;
                if (mc == null)
                {
                    mc = new List<string>();
                    context.Items.Add(rawContentKey, mc);
                }
                mc.Add(content);
            }
        }

        #endregion


        public static void SetDisableClientCache(bool disableClientCache)
        {
            HttpContext.Current.Items["DisableClientCache"] = disableClientCache;
        }


        //是否禁止客户端缓存
        private bool disableClientCache = false;
        public bool DisableClientCache
        {
            get
            {
                if (Context.Items["DisableClientCache"] != null)
                {
                    try
                    {
                        disableClientCache = Convert.ToBoolean(Context.Items["DisableClientCache"]);
                    }
                    catch { }
                }

                return disableClientCache;
            }
            set { disableClientCache = value; }
        }

        private bool enableStyleHttpCompress = false;
        /// <summary>
        /// 是否启用样式表Http压缩
        /// </summary>
        public bool EnableStyleHttpCompress
        {
            get { return enableStyleHttpCompress; }
            //set { enableStyleHttpCompress = value; }
        }

        /// <summary>
        /// 格式化meta标签
        /// </summary>
        private static string MetaTagFormatter(string metaTagValue)
        {
            string returnValue = metaTagValue.Trim().Replace("\n", "");
            return Formatter.FormatMultiLinePlainTextForStorage(returnValue, false);
        }
    }
}
