﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Common.Controllers;
using System.Globalization;
using SpaceBuilder.Caching;

using SpaceBuilder.Common;
using SpaceBuilder.Web;
using SpaceBuilder.Club;
using System.Web.Hosting;

namespace SpaceBuilder.Web
{
    /// <summary>
    /// 用于SpaceBuilder支持Theme的ViewEngine
    /// </summary>
    public class ThemeViewEngine : IViewEngine
    {
        private static readonly VirtualPathProvider VPathProvider = HostingEnvironment.VirtualPathProvider;

        #region IViewEngine 成员

        public ViewEngineResult FindView(ControllerContext controllerContext, string viewName, string masterName, bool useCache)
        {
            if (controllerContext == null)
                throw new ArgumentNullException("controllerContext is required.", "controllerContext");
            if (string.IsNullOrEmpty(viewName))
                throw new ArgumentException("viewName is required.", "viewName");

            SPBControllerBase spbControllerBase = controllerContext.Controller as SPBControllerBase;
            if (spbControllerBase == null)
            {
                throw new ArgumentException("Controller 必须从 SPBControllerBase 派生.", controllerContext.Controller.GetType().FullName);
            }

            string selectedThemeName = spbControllerBase.ThemeManager.GetSelectedThemeName(controllerContext);
            string actualViewPath = GetViewPath(spbControllerBase.ThemeManager, selectedThemeName, viewName);

            string actualMasterPath = null;
            if (!string.IsNullOrEmpty(masterName))
                actualMasterPath = GetViewPath(spbControllerBase.ThemeManager, selectedThemeName, masterName);

            if (actualViewPath == null)
                throw new InvalidOperationException(String.Format("The view '{0}' could not be located at these paths: {1}", viewName, spbControllerBase.ThemeManager.PresentAreaID));

            return new ViewEngineResult(new WebFormView(actualViewPath, actualMasterPath), this);
        }


        public ViewEngineResult FindPartialView(ControllerContext controllerContext, string partialViewName, bool useCache)
        {
            return FindView(controllerContext, partialViewName, null, useCache);
        }


        public void ReleaseView(ControllerContext controllerContext, IView view)
        {
            IDisposable disposable = view as IDisposable;
            if (disposable != null)
                disposable.Dispose();
        }

        #endregion


        #region Help Methods

        /// <summary>
        /// 获取View文件路径
        /// </summary>
        /// <param name="themeName"></param>
        /// <param name="viewName"></param>
        public static string GetViewPath(PresentAreaThemeManager themeManager, string themeName, string viewName)
        {
            string cacheKey = "viewPath:" + themeManager.PresentAreaID + "|" + themeName + "|" + viewName;
            string path = CacheRepository.Get(cacheKey,false) as string;

            if (path == null)
            {
                path = WalkThemes(themeManager, themeName, viewName);
                if (path != null)
                {
                    CacheRepository.Add(cacheKey, path, CachingExpirationTypes.Stable);
                }
                else
                {
                    throw new SPBException(ExceptionTypes.Instance().SkinNotFound(), string.Format("The Skin {0} could not be found", viewName));
                }
            }
            return path;
        }

        /// <summary>
        /// 在ThemeName及其Parent查找View文件路径
        /// </summary>
        /// <param name="themeName">主题名称</param>
        /// <param name="viewName"></param>
        private static string WalkThemes(PresentAreaThemeManager themeManager, string themeName, string viewName)
        {
            string path = null;
            string themeLocation = themeManager.PresentArea.ThemeLocation;
            while (true)
            {
                Theme theme = Themes.GetTheme(themeManager.PresentAreaID, themeName, true, false);
                if (theme == null)
                    break;

                path = themeLocation + themeName + "/" + viewName;
                if (VPathProvider.FileExists(path))
                {
                    break;
                }
                else
                {
                    if (theme.HasParent)
                    {
                        themeName = theme.Parent;
                    }
                    else
                    {
                        path = null;
                        break;
                    }
                }
            }

            //如果还未找到view文件则从默认主题中查找
            if (path == null)
            {
                path = themeLocation + "default" + "/" + viewName;
                if (!VPathProvider.FileExists(path))
                    path = null;
            }
            //为了实现企业空间能共用个人空间的页面，作以下处理
            if (path == null && themeManager.PresentAreaID == PresentAreaIDs.Instance().CompanyUserDomain())
            {
                PresentArea presentArea = PresentAreas.GetPresentArea(PresentAreaIDs.Instance().UserDomain(), false);
                if (presentArea != null)
                {
                    path = presentArea.ThemeLocation + "default" + "/" + viewName;
                    if (!VPathProvider.FileExists(path))
                        path = null;
                }
            }
            //如果仍未找到则从公共区域Shared中查找
            if (path == null)
            {
                path = "~/Themes/Shared/" + viewName;
                if (!VPathProvider.FileExists(path))
                    path = null;
            }

            return path;
        }

        #endregion

    }
}
