﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text;
using System.Web.Mvc;
using SpaceBuilder.Utils;
using System.Web.Routing;
using System.Web;
using SpaceBuilder.Caching;

namespace SpaceBuilder.Web
{
    /// <summary>
    /// Url辅助类
    /// </summary>
    public class SPBUrlHelper
    {
        private static readonly string ControllerKey = "controller";
        private static readonly string ActionKey = "action";

        /// <summary>
        /// 通过routeName获取url
        /// </summary>
        /// <param name="routeName">routeName</param>
        /// <returns>url</returns>
        public static string RouteUrl(string routeName)
        {
            return RouteUrl(routeName, null);
        }

        /// <summary>
        /// 通过routeName获取url
        /// </summary>
        /// <param name="routeName">routeName</param>
        /// <param name="routeValueDictionary">路由数据</param>
        /// <returns>url</returns>
        public static string RouteUrl(string routeName, RouteValueDictionary routeValueDictionary)
        {
            string cacheKey = string.Format("RouteUrl::{0}", routeName);

            RouteValueDictionary routeParameters = new RouteValueDictionary();
            string[] values = null;
            if (routeValueDictionary != null)
            {
                if (routeValueDictionary.ContainsKey(ControllerKey))
                {
                    routeParameters[ControllerKey] = routeValueDictionary[ControllerKey];
                    routeValueDictionary.Remove(ControllerKey);
                    cacheKey += "-" + ControllerKey + ":" + routeParameters[ControllerKey];
                }
                if (routeValueDictionary.ContainsKey(ActionKey))
                {
                    routeParameters[ActionKey] = routeValueDictionary[ActionKey];
                    routeValueDictionary.Remove(ActionKey);
                    cacheKey += "-" + ActionKey + ":" + routeParameters[ActionKey];
                }

                values = new string[routeValueDictionary.Count];
                int index = 0;
                foreach (KeyValuePair<string, object> pair in routeValueDictionary)
                {
                    cacheKey += "-" + pair.Key + ":{" + index + "}";

                    if (pair.Value == null)
                        values[index] = string.Empty;
                    else
                        values[index] = pair.Value.ToString();

                    routeParameters[pair.Key] = "{" + index + "}";
                    index++;
                }
            }

            string url = CacheRepository.Get(cacheKey,false) as string;
            if (url == null)
            {
                url = GenerateRouteUrl(routeName, routeParameters);

                //替换UrlEncode编码
                url = url.Replace("%7B", "{").Replace("%7D", "}");

                CacheRepository.Add(cacheKey, url, CachingExpirationTypes.Stable);
            }

            if (values != null)
                return string.Format(url, values);
            else
                return url;
        }

        /// <summary>
        /// 通过routeName生成url
        /// </summary>
        /// <param name="routeName">routeName</param>
        /// <param name="routeValueDictionary">路由数据</param>
        /// <returns>url</returns>
        private static string GenerateRouteUrl(string routeName, RouteValueDictionary routeValueDictionary)
        {
            RouteBase route = RouteTable.Routes[routeName];
            if (route == null)
                return string.Empty;

            RequestContext requestContext = new RequestContext(new HttpContextWrapper(System.Web.HttpContext.Current), new RouteData());
            VirtualPathData pathData = route.GetVirtualPath(requestContext, routeValueDictionary);
            if (pathData == null)
                return string.Empty;


            string absolutePath = pathData.VirtualPath;

            if (!absolutePath.StartsWith("/"))
                absolutePath = "/" + absolutePath;

            string appPath = requestContext.HttpContext.Request.ApplicationPath;

            if (!absolutePath.StartsWith(appPath))
                absolutePath = appPath + absolutePath;

            return absolutePath;
        }

        /// <summary>
        /// 通过Action/Controller获取url
        /// </summary>
        /// <param name="actionName"></param>
        /// <param name="controllerName"></param>
        /// <returns></returns>
        public static string Action(string actionName, string controllerName)
        {
            return Action(actionName, controllerName, new RouteValueDictionary());
        }

        /// <summary>
        /// 通过Action/Controller获取url
        /// </summary>
        /// <param name="actionName"></param>
        /// <param name="controllerName"></param>
        /// <param name="routeValueDictionary"></param>
        /// <returns></returns>
        public static string Action(string actionName, string controllerName, RouteValueDictionary routeValueDictionary)
        {
            string cacheKey = string.Format("ActionUrl::c:{0}-a:{1}", controllerName, actionName);

            RouteValueDictionary routeParameters = new RouteValueDictionary();
            string[] values = null;
            if (routeValueDictionary != null)
            {
                values = new string[routeValueDictionary.Count];
                int index = 0;
                foreach (KeyValuePair<string, object> pair in routeValueDictionary)
                {
                    cacheKey += "-" + pair.Key + ":{" + index + "}";

                    if (pair.Value == null)
                        values[index] = string.Empty;
                    else
                        values[index] = pair.Value.ToString();

                    routeParameters[pair.Key] = "{" + index + "}";

                    index++;
                }
            }

            string url = CacheRepository.Get(cacheKey,false) as string;
            if (url == null)
            {
                url = GenerateActionUrl(actionName, controllerName, routeParameters);

                //替换UrlEncode编码
                url = url.Replace("%7B", "{").Replace("%7D", "}");

                CacheRepository.Add(cacheKey, url, CachingExpirationTypes.Stable);
            }

            if (values != null)
                return string.Format(url, values);
            else
                return url;
        }

        private static string GenerateActionUrl(string actionName, string controllerName, RouteValueDictionary routeValues)
        {
            if (actionName != null)
                routeValues["action"] = actionName;

            if (controllerName != null)
                routeValues["controller"] = controllerName;

            RequestContext requestContext = new RequestContext(new HttpContextWrapper(System.Web.HttpContext.Current), new RouteData());
            VirtualPathData pathData = RouteTable.Routes.GetVirtualPath(requestContext, routeValues);
            if (pathData == null)
                return string.Empty;


            string absolutePath = pathData.VirtualPath;

            if (!absolutePath.StartsWith("/"))
                absolutePath = "/" + absolutePath;

            string appPath = requestContext.HttpContext.Request.ApplicationPath;

            if (!absolutePath.StartsWith(appPath))
                absolutePath = appPath + absolutePath;

            return absolutePath;
        }

    }
}
