﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc.  All rights  reserved.
// </copyright> 
//------------------------------------------------------------------------------ 


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Web;
using SpaceBuilder.Event;
using SpaceBuilder.Common;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using System.IO;
using System.Collections;
using SpaceBuilder.Utils;
using SpaceBuilder.Club;
using System.Web.Routing;
using System.Net.Mail;
using System.Web.UI;

namespace SpaceBuilder.Event.Controllers
{
    [HttpCompress]
    public class ChannelEventController : ChannelControllerBase
    {
        private int EventApplicationID = ApplicationIDs.Instance().Event();
        #region 活动

        #region 页面
        /// <summary>
        /// 活动首页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Home()
        {
            SetPageTitle(GetResourceString("Title_Event", EventApplicationID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Event", EventApplicationID), string.Empty);
            ViewData["SiteCategories"] = SiteCategories.Instance(ApplicationIDs.Instance().Event()).GetRoots(false);
            ViewData["Areas"] = Areas.GetRootAreas(false);
            List<EventAreaStatistics> areaRank = EventAreaStatisticss.GetChildEventAreaStatisticss(EventConfiguration.Instance().RootAreaCode, EventAreaStatisticsSortBy.CumulateTotalCount, SortOrder.Descending);


            ViewData["Areas"] = areaRank;
            //活动达人方法 EventUserDatas.GetTopUserDatasSortBy
            ViewData["ExperiencePeople"] = EventUserDatas.GetTopUserDatasSortBy(EventUserDataSortBy.ExperiencePoints, 6);

            return View("Pages/Events/Home.aspx", "Masters/Event.master");
        }

        /// <summary>
        /// 活动回顾
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult ListEventsReview(EventThreadSortBy? sortBy, int? pageIndex)
        {
            SetPageTitle("活动回顾");
            BreadCrumb.AddBreadCrumbItem("活动回顾", string.Empty);
            PagingDataSet<EventThread> pds = EventThreads.GetThreadsBySecondary(sortBy ?? EventThreadSortBy.LastModified, null, string.Empty, null, null, null, EventStatuses.Finished, pageIndex ?? 1);

            return View("Pages/Events/ListEventsReview.aspx", "Masters/Event.master", pds);
        }

        public ActionResult Control_EventCard(int? EventID)
        {
            EventThread eventThread = EventThreads.GetEvent(EventID ?? 0, true);
            return PartialView("Controls/Events/EventCard.ascx", eventThread);
        }

        /// <summary>
        /// 显示活动详细信息
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowEvent(int eventID)
        {
            EventThread eventThread = null;
            if (HasManagePermission(eventID))
                eventThread = EventThreads.GetEvent(eventID, true);
            else
                eventThread = EventThreads.GetEvent(eventID, false);

            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindBrowseEvent", EventApplicationID)));
            }

            #region 处理审核
            int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Event()).AuditingStatusForPublicDisplay;
            if ((int)eventThread.AuditingStatus < auditingStatusValue)
            {
                if (eventThread.AuditingStatus == AuditingStatuses.Fail)
                    ViewData["errorMessage"] = GetResourceString("Message_EventNoPassAuditing", EventApplicationID);
                else
                    ViewData["errorMessage"] = GetResourceString("Message_EventAuditingOnTheMarch", EventApplicationID);
                ViewData["errorAuditingStatus"] = true;
            }
            #endregion

            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();

            #region 浏览器标题、计数、meta标签
            SetPageTitle(eventThread.EventName);
            CounterFactory.IncreaseCount<EventThreadCounter>(IncreasableCountTypes.WebView, eventID);
            SetPageMeta(eventThread);
            #endregion

            #region 面包屑
            //ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            //BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), string.Empty);
            BreadCrumb.AddBreadCrumbItem(eventThread.EventName, string.Empty);
            #endregion

            SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(eventThread.SiteCategoryID, false);
            ViewData["Category"] = category;
            if (category != null && category.ParentID > 0)
                ViewData["ParentCategory"] = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(category.ParentID, false);

            PagingDataSet<EventPhoto> photos = EventPhotos.GetEventPhotos(eventID, null, 10, 1);
            ViewData["EventPhotos"] = photos.Records;
            //显示十位活动成员
            PagingDataSet<EventMember> members = EventMembers.GetEventMembers(eventID, true, 12, 1, true);
            ViewData["EventMembers"] = members.Records;
            if (eventThread.IsClubEvent)
            {
                Club.Club club = Club.Clubs.GetClub(eventThread.ClubID, false);
                ViewData["Club"] = club;
            }
            ViewData["IsFavorited"] = FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).IsFavorite(CurrentUser != null ? CurrentUser.UserID : 0, eventID);

            EventMember eventMember = EventMembers.GetEventMember(CurrentUser != null ? CurrentUser.UserID : 0, eventID, true);
            ViewData["eventMember"] = eventMember;
            if (HasManagePermission(eventID))
            {
                ViewData["body"] = eventThread.GetBody(true);
            }
            else
            {
                ViewData["body"] = eventThread.GetBody(false);
            }
            return View("Pages/Events/ShowEvent.aspx", "Masters/Event.master", eventThread);
        }

        /// <summary>
        /// 显示创建新活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ChannelGuideInstallApplication(ApplicationID = 221)]
        public ActionResult NewEvent(int? clubID)
        {
            if (!Authorize(null, PermissionItemKeys.Instance().CreateEvent()))
                return new EmptyResult();

            SetPageTitle(GetResourceString("Title_ReleaseNewEvent", EventApplicationID));
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Event", EventApplicationID), SiteUrls.Instance().ChannelEventHome());
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_ReleaseNewEvent", EventApplicationID), string.Empty);

            EventThread eventThread = new EventThread();
            if (clubID.HasValue && clubID.Value > 0)
            {
                eventThread.ClubID = clubID.Value;
            }
            if (TempData["eventThread"] != null)
                eventThread = TempData["eventThread"] as EventThread;

            return View("Pages/Events/EditEvent.aspx", "Masters/Event.master", eventThread);
        }
        /// <summary>
        /// 显示活动设置页
        /// </summary> 
        /// <param name="isCreating">是否正在创建活动</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowEventSettings(int eventID, bool? isCreating)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, true);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedSetEvent", EventApplicationID)));
            }

            SetPageTitle(string.Format(GetResourceString("Title_Pattern_UserDefined", EventApplicationID), eventThread.EventName));

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EventSet", EventApplicationID), string.Empty);
            #endregion

            if (!Authorize(eventThread, PermissionItemKeys.Instance().EditEvent()))
                return new EmptyResult();

            if (!eventThread.IsClubEvent)
            {
                ViewData["myJoinedClubs"] = Clubs.GetMyClubs(CurrentUser.UserID, 30);
            }
            else
            {
                Club.Club club = Clubs.GetClub(eventThread.ClubID, false);
                ViewData["club"] = club;
            }
            //如果全局设置中不启用允许匿名用户报名，则不允许用户配置此项
            ViewData["EnableAnonymousUserSignUp"] = EventConfiguration.Instance().EnableAnonymousUserSignUp;
            ViewData["IsCreating"] = isCreating.HasValue ? isCreating.Value : false;
            return View("Pages/Events/ShowEventSettings.aspx", "Masters/Event.master", eventThread);
        }

        /// <summary>
        /// 显示编辑活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditEvent(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, true);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedEditEvent", EventApplicationID)));
            }
            if (TempData["eventThread"] != null)
                eventThread = TempData["eventThread"] as EventThread;
            if (!Authorize(eventThread, PermissionItemKeys.Instance().EditEvent()))
                return new EmptyResult();

            SetPageTitle(string.Format(GetResourceString("Title_Pattern_Editor", EventApplicationID), eventThread.EventName));

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditEvent", EventApplicationID), string.Empty);
            #endregion

            ViewData["body"] = eventThread.GetBody(true);
            return View("Pages/Events/EditEvent.aspx", "Masters/Event.master", eventThread);
        }

        #endregion

        #region 控件

        /// <summary>
        /// 总站首页显示的活动列表模块
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_ListEventsHomeBlock(EventThreadSortBy? sortBy, int? displayCount)
        {
            IList<EventThread> threads = EventThreads.GetTopThreadsSortBy(sortBy ?? EventThreadSortBy.HitTimes, EventStatuses.Published, displayCount ?? 6);
            return PartialView("Controls/Events/ListEventsHomeBlock.ascx", threads);
        }

        /// <summary>
        /// 显示活动操作列表以及统计信息
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EventAction(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindBrowseEvent", EventApplicationID)));
            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();

            if (CurrentUser != null && EventMembers.IsMember(CurrentUser.UserID, eventID, true))
                ViewData["IsMember"] = true;
            if (CurrentUser != null && EventMembers.IsManager(CurrentUser.UserID, eventID, true))
                ViewData["IsManager"] = true;
            if (CurrentUser != null && (CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
                ViewData["IsAdministrator"] = true;
            int userID = CurrentUser != null ? CurrentUser.UserID : 0;
            EventMember eventMember = EventMembers.GetEventMember(userID, eventID, false);
            ViewData["eventMember"] = eventMember;

            return PartialView("Controls/Events/EventAction.ascx", eventThread);
        }

        /// <summary>
        /// 显示具有活动内容块
        /// </summary>
        /// <param name="sortBy">排序依据</param>
        /// <param name="titleName">内容块标题</param>
        /// <param name="viewName">页面名称</param>
        /// <param name="status">活动状态</param>
        /// <param name="areaCode">活动地区</param>
        /// <param name="isClubEvent">是否圈子活动</param>
        /// <param name="isEssential">是否精品活动</param>
        /// <param name="topNumber">显示条数</param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_ListOrSummaryEventsBlock(EventThreadSortBy? sortBy, string titleName, string viewName, EventStatuses? status, string areaCode, bool? isClubEvent, bool? isEssential, int topNumber)
        {
            IList<EventThread> threads = null;
            if (isClubEvent.HasValue && isClubEvent.Value)
                threads = EventThreads.GetTopThreads(areaCode, isClubEvent, isEssential, status ?? EventStatuses.Published, topNumber);
            else
                threads = EventThreads.GetTopThreadsSortBy(sortBy ?? EventThreadSortBy.HitTimes, status ?? EventStatuses.Published, topNumber);

            ViewData["status"] = status;
            ViewData["sortBy"] = sortBy;
            ViewData["areaCode"] = areaCode;
            ViewData["moduleTitle"] = titleName;
            ViewData["isClubEvent"] = isClubEvent;
            ViewData["isEssential"] = isEssential;
            ViewData["displayCount"] = topNumber;

            return PartialView("Controls/Events/" + viewName + ".ascx", threads);
        }

        /// <summary>
        /// 显示活动类别列表
        /// </summary>
        /// <param name="siteCategoryID">指定类别ID，会根据它显示子类别列表，如果为最后一级类别，则会显示它的同一级的类别列表</param>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_ListEventSiteCategories(int? siteCategoryID)
        {
            List<SiteCategory> categories = null;
            if (siteCategoryID.HasValue && siteCategoryID.Value > 0)
            {
                SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(siteCategoryID.Value, false);
                SiteCategory parentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(currentCategory.ParentID, false);
                int depth = EventConfiguration.Instance().SiteCategoryLevel - 1;
                if (currentCategory.Depth == depth && currentCategory.ParentID > 0)
                {
                    if (parentCategory != null)
                        categories = parentCategory.Childs;
                }
                else if (currentCategory.Depth < depth)
                {
                    categories = currentCategory.Childs;
                    if (categories.Count == 0 && parentCategory != null)
                        categories = parentCategory.Childs;
                }
            }
            if (categories == null || categories.Count == 0)
                categories = SiteCategories.Instance(ApplicationIDs.Instance().Event()).GetRoots(false);
            ViewData["siteCategoryID"] = siteCategoryID;
            return PartialView("Controls/Events/ListEventSiteCategories.ascx", categories);
        }
        /// <summary>
        /// 地区排行
        /// </summary>
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_AreaRank(int displayItemCount)
        {
            List<EventAreaStatistics> areaRank = EventAreaStatisticss.GetChildEventAreaStatisticss(EventConfiguration.Instance().RootAreaCode, EventAreaStatisticsSortBy.CumulateTotalCount, SortOrder.Descending);



            ViewData["displayItemCount"] = displayItemCount;
            return PartialView("Controls/Events/AreaRank.ascx", areaRank);
        }
        #endregion

        #region 操作
        /// <summary>
        /// 创建新活动后持久化
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        [ChannelGuideInstallApplication(ApplicationID = 221)]
        public ActionResult CreateEvent(int? clubID)
        {
            EventThread eventThread = new EventThread();

            if (!Authorize(eventThread, PermissionItemKeys.Instance().CreateEvent()))
                return new EmptyResult();

            eventThread.AuditingStatus = AuditingStatuses.Pending;
            eventThread.EventStatus = EventStatuses.Published;
            eventThread.UserID = CurrentUser.UserID;
            eventThread.Sponsor = CurrentUser.DisplayName;
            eventThread = PopulateEventThreadFromRequest(eventThread);

            if (string.IsNullOrEmpty(eventThread.EventName) || eventThread.StartTime >= eventThread.EndTime || eventThread.SignEndTime >= eventThread.EndTime)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (string.IsNullOrEmpty(eventThread.EventName))
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_EventNameNotNull", EventApplicationID);
                }
                else if (eventThread.StartTime >= eventThread.EndTime)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_StartTimeNoExceedEndTime", EventApplicationID);
                }
                else if (eventThread.SignEndTime >= eventThread.EndTime)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_SignUpEndTimeNoExceedEndTime", EventApplicationID);
                }
                TempData["eventThread"] = eventThread;
                return RedirectToAction("NewEvent", new RouteValueDictionary { { "eventID", eventThread.EventID } });
            }
            if (clubID.HasValue && clubID.Value > 0)
            {
                eventThread.ClubID = clubID.Value;
            }
            EventThreads.CreateEvent(eventThread);
            if (eventThread.EventID > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                if (eventThread.AuditingStatus == AuditingStatuses.Success)
                    TempData["StatusMessageContent"] = GetResourceString("Message_EventCreateSuccess", EventApplicationID);
                else
                    TempData["StatusMessageContent"] = GetResourceString("Message_EventCreateSuccessButNeedAuditing", EventApplicationID);
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_EventCreateFailed", EventApplicationID);
            }
            return RedirectToAction("ShowEventSettings", new RouteValueDictionary { { "eventID", eventThread.EventID }, { "isCreating", true } });
        }
        /// <summary>
        /// 保存活动设置
        /// </summary>
        /// <param name="isCreating">是否正在创建活动</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SaveEventSettings(int eventID, bool isCreating)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, true);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedSetEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().EditEvent()))
                return new EmptyResult();
            eventThread.PayDescription = Request.Form.GetString("payDescription", string.Empty).Trim();
            eventThread.IsAuditingSigner = Request.Form.GetBool("isAuditingSigner", false);
            eventThread.IsAllowAnonymousUserSignUp = Request.Form.GetBool("isAllowAnonymousUserSignUp", false);
            eventThread.ClubID = Request.Form.GetInt("clubList", eventThread.ClubID);
            eventThread.IsOnlyAllowClubMemberSignUp = Request.Form.GetBool("isOnlyAllowClubMemberSignUp", false);
            eventThread.PayType = (PayType)Request.Form.GetInt("payType", 0);
            eventThread.PayDescription = Request.Form.GetString("payDescription", string.Empty).Trim();
            eventThread.LimitCount = Request.Form.GetInt("limitCount", 0);
            if (EventPermission.ValidateSetPermission(eventThread, CurrentUser, PermissionItemKeys.Instance().InviteFriendJoinEvent()))
                eventThread.AllowMemberInviteFriend = Request.Form.GetBool("allowMemberInviteFriend", false);
            if (EventPermission.ValidateSetPermission(eventThread, CurrentUser, PermissionItemKeys.Instance().UploadEventPhoto()))
                eventThread.AllowMemberUploadPhoto = Request.Form.GetBool("allowMemberUploadPhoto", false);

            EventThreads.UpdateEvent(eventThread);
            if (isCreating)
            {
                return RedirectToAction("EventInvitation", new RouteValueDictionary { { "eventID", eventThread.EventID }, { "isCreating", isCreating } });
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_EventSetAlreadySave", EventApplicationID);
                // return RedirectToAction("ShowEventSettings", new RouteValueDictionary { { "eventID", eventThread.EventID } });
                return RedirectToAction("ShowEvent", new RouteValueDictionary { { "eventID", eventThread.EventID } });
            }
        }

        /// <summary>
        /// 编辑活动后持久化
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateEvent(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, true);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedEditEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().EditEvent()))
                return new EmptyResult();

            eventThread = PopulateEventThreadFromRequest(eventThread);
            //验证时间
            if (string.IsNullOrEmpty(eventThread.EventName) || eventThread.StartTime >= eventThread.EndTime || eventThread.SignEndTime >= eventThread.EndTime)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                if (string.IsNullOrEmpty(eventThread.EventName))
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_EventNameNotNull", EventApplicationID);
                }

                else if (eventThread.StartTime >= eventThread.EndTime)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_StartTimeNoExceedEndTime", EventApplicationID);
                }
                else if (eventThread.SignEndTime >= eventThread.EndTime)
                {
                    TempData["StatusMessageContent"] = GetResourceString("Message_SignUpEndTimeNoExceedEndTime", EventApplicationID);
                }
                TempData["eventThread"] = eventThread;
                return RedirectToAction("EditEvent", new RouteValueDictionary { { "eventID", eventThread.EventID } });
            }

            EventThreads.UpdateEvent(eventThread);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_EventSetAlreadySave", EventApplicationID);
            return RedirectToAction("ShowEventSettings", new RouteValueDictionary { { "eventID", eventThread.EventID } });
        }
        /// <summary>
        /// 处理退出活动
        /// </summary>    
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ExitEvent(int userID, int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedSetEvent", EventApplicationID)));
            }
            if (CurrentUser != null && CurrentUser.UserID == eventThread.UserID)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "您不能退出自己发起的活动！";
                return RedirectToAction("ShowEvent", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }
            if (eventThread.EventStatus != EventStatuses.Published)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "必须是进行中的活动才能退出！";
                return RedirectToAction("ShowEvent", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }
            EventMember member = EventMembers.GetEventMember(userID, eventID, true);
            if (member != null)
            {
                EventMembers.DeleteEventMember(member.MemberID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_AlreadyExit", EventApplicationID), eventThread.EventName);
            }
            return RedirectToAction("ShowEvent", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
        }
        /// <summary>
        /// 处理更新活动精华状态
        /// </summary>   
        [AcceptVerbs(HttpVerbs.Get)]
        public JsonResult UpdateEssentialStatus(int eventID, bool? isEssential)
        {
            if (isEssential == null)
                return new JsonResult();
            EventThread eventThread = EventThreads.GetEvent(eventID, true);
            if (eventThread == null)
                return new JsonResult();
            if (!Authorize(eventThread, PermissionItemKeys.Instance().AdministerEvent()))
                return new JsonResult();

            EventThreads.UpdateEssentialStatus(eventID, isEssential.Value);
            eventThread = EventThreads.GetEvent(eventID, true);
            return Json(eventThread.IsEssential, JsonRequestBehavior.AllowGet);
        }
        /// <summary>
        /// 取消活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult CancelEvent(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                Response.Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedSetEvent", EventApplicationID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().CancelEvent()))
                return new EmptyResult();

            if (eventThread.EventStatus == EventStatuses.Published)
            {
                EventThreads.CancelEvent(eventID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = GetResourceString("Message_CancelOperationSuccess", EventApplicationID);
            }
            else
            {
                if (eventThread.EventStatus == EventStatuses.Finished)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_AlreadyEndAndNotCancel", EventApplicationID), eventThread.EventName);
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_AlreadyEndAndNotCancel", EventApplicationID), eventThread.EventName);
                }
            }
            return RedirectToAction("ListMyEvents", "UserDomainEvent", new RouteValueDictionary { { "userDomainName", CurrentUser.UserName } });
        }


        /// <summary>
        /// 搜索活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Search(string keyword, int? siteCategoryID, string areaCode, int? pageIndex)
        {
            if (!string.IsNullOrEmpty(keyword))
            {
                EventThreadFullQuery query1 = new EventThreadFullQuery();
                query1.ConvertFromQueryString(Request);
                PagingDataSet<EventThread> pds1 = EventThreads.GetThreadsForSearch(query1, pageIndex ?? 1);
                return GetSearchViewResult(query1, pds1);
            }
            SetPageTitle(GetResourceString("Title_EventSearch", EventApplicationID));
            EventThreadFullQuery query = new EventThreadFullQuery();
            query.SiteCategoryID = siteCategoryID ?? -1;
            query.AreaCode = areaCode;
            PagingDataSet<EventThread> pds = EventThreads.GetThreadsByPrimary(siteCategoryID, areaCode, pageIndex ?? 1);
            return GetSearchViewResult(query, pds);
        }

        /// <summary>
        /// 搜索活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult SecondarySearch(string keyword, EventThreadSortBy? sortBy, int? siteCategoryID, string areaCode, EventDateScopes? dateScope, bool? isEssential, bool? isClubEvent, EventStatuses? status, int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_EventSearch", EventApplicationID));
            if (!string.IsNullOrEmpty(keyword))
            {
                EventThreadFullQuery query1 = new EventThreadFullQuery();
                query1.ConvertFromQueryString(Request);
                query1.SortBy = sortBy ?? EventThreadSortBy.StartTime;
                query1.SiteCategoryID = siteCategoryID ?? -1;
                query1.AreaCode = areaCode;
                query1.DateScope = dateScope;
                query1.IsEssential = isEssential;
                query1.IsClubEvent = isClubEvent;
                query1.Status = status ?? EventStatuses.Published;
                query1.IsPublic = true;
                PagingDataSet<EventThread> pds1 = EventThreads.GetThreadsForSearch(query1, pageIndex ?? 1);
                return GetSearchViewResult(query1, pds1);
            }

            EventThreadFullQuery query = new EventThreadFullQuery();
            query.SortBy = sortBy ?? EventThreadSortBy.StartTime;
            query.SiteCategoryID = siteCategoryID ?? -1;
            query.AreaCode = areaCode;
            query.DateScope = dateScope;
            query.IsEssential = isEssential;
            query.IsClubEvent = isClubEvent;
            query.Status = status ?? EventStatuses.Published;
            query.IsPublic = true;
            PagingDataSet<EventThread> pds = EventThreads.GetThreadsBySecondary(query.SortBy, siteCategoryID, areaCode, dateScope, isEssential, isClubEvent, status, pageIndex ?? 1);
            return GetSearchViewResult(query, pds);
        }

        /// <summary>
        /// 需要根据关键词搜索活动的方法
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult AdvancedSearch(int? pageIndex)
        {
            SetPageTitle(GetResourceString("Title_EventSearch", EventApplicationID));

            EventThreadFullQuery query = new EventThreadFullQuery();
            query.ConvertFromQueryString(Request);
            query.IsPublic = true;
            query.Status = EventStatuses.Published;
            PagingDataSet<EventThread> pds = EventThreads.GetThreadsForSearch(query, pageIndex ?? 1);
            return GetSearchViewResult(query, pds);
        }

        /// <summary>
        /// 获取Serch视图
        /// </summary>
        private ActionResult GetSearchViewResult(EventThreadFullQuery query, PagingDataSet<EventThread> pds)
        {
            if (query == null)
                return new EmptyResult();
            ViewData["Query"] = query;

            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EventSearch", EventApplicationID), string.Empty);
            Stack<SiteCategory> categoryStack = null;
            List<SiteCategory> siteCategoties = GetChildCategories(query.SiteCategoryID, out categoryStack);
            Stack<EventAreaStatistics> areaStatisticsStack = null;
            List<EventAreaStatistics> areaStatisticss = GetChildAreas(query.AreaCode, out areaStatisticsStack);

            ViewData["siteCategoties"] = siteCategoties;
            ViewData["categoryStack"] = categoryStack;
            ViewData["areas"] = areaStatisticss;
            ViewData["areaStack"] = areaStatisticsStack;
            if (query.SiteCategoryID > 0)
            {
                SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(query.SiteCategoryID, false);
                if (currentCategory != null)
                    ViewData["CurrentCategoryName"] = currentCategory.GetMultilingualCategoryName(CurrentUserLanguage);
            }
            if (!string.IsNullOrEmpty(query.AreaCode))
            {
                Area area = Areas.GetArea(query.AreaCode, false);
                if (area != null)
                    ViewData["CurrentAreaName"] = area.GetMultilingualAreaName(CurrentUserLanguage);
            }
            return this.View("Pages/Events/Search.aspx", "Masters/Event.master", pds);
        }


        /// <summary>
        /// 可根据关键词搜索活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Search()
        {
            string keyWord = Request.Form.GetString("Keyword", string.Empty).Trim();
            return RedirectToAction("AdvancedSearch", new RouteValueDictionary(CurrentRouteValueDictionary) { { "Keyword", keyWord } });
        }

        /// <summary>
        /// 获取当前类别的子类别，如果是最后一级，则获取同辈类别
        /// </summary>
        /// <param name="siteCategoryID">当前类别</param>
        private List<SiteCategory> GetChildCategories(int? currentCagoryID, out Stack<SiteCategory> categoryStack)
        {
            categoryStack = new Stack<SiteCategory>();
            SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(currentCagoryID ?? 0, false);

            List<SiteCategory> siteCategoties = null;

            if (currentCategory == null)
                siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Event()).GetRoots(false);
            else if (currentCategory.ChildCount > 0)
                siteCategoties = currentCategory.Childs;
            else
            {
                if (currentCategory.ParentID <= 0)
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Event()).GetRoots(false);
                else
                    siteCategoties = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(currentCategory.ParentID, false).Childs;
            }

            while (currentCategory != null)
            {
                if (currentCategory.ChildCount > 0)
                    categoryStack.Push(currentCategory);
                currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(currentCategory.ParentID, false);
            }
            return siteCategoties;
        }
        /// <summary>
        /// 获取当前地区的子地区，如果是最后一级，则获取同辈地区
        /// </summary>
        /// <param name="siteCategoryID">当前类别</param>
        private List<EventAreaStatistics> GetChildAreas(string currentAreaCode, out Stack<EventAreaStatistics> eventAreaStatisticsStack)
        {
            List<EventAreaStatistics> eventAreaStatisticss = null;
            eventAreaStatisticsStack = new Stack<EventAreaStatistics>();
            string rootAreaCode = EventConfiguration.Instance().RootAreaCode;
            EventAreaStatistics rootAreaStatistics = EventAreaStatisticss.GetEventAreaStatistics(rootAreaCode);
            if (!string.IsNullOrEmpty(currentAreaCode))
            {
                int maxDepth = EventConfiguration.Instance().AreaLevel;
                if (rootAreaStatistics != null)
                    maxDepth += rootAreaStatistics.Depth;

                EventAreaStatistics currentAreaStatistics = EventAreaStatisticss.GetEventAreaStatistics(currentAreaCode);
                if (currentAreaStatistics != null)
                {
                    if (currentAreaStatistics.Depth == maxDepth)
                    {
                        EventAreaStatistics parentAreaStatistics = EventAreaStatisticss.GetEventAreaStatistics(currentAreaStatistics.ParentCode);
                        if (parentAreaStatistics != null)
                        {
                            eventAreaStatisticss = parentAreaStatistics.Childs;
                        }
                        else
                            eventAreaStatisticss = EventAreaStatisticss.GetRootEventAreaStatisticss();

                    }
                    else if (currentAreaStatistics.Depth < maxDepth)
                    {
                        eventAreaStatisticss = currentAreaStatistics.Childs;
                    }
                }
                while (currentAreaStatistics != null)
                {
                    if (currentAreaStatistics.ChildCount > 0)
                        eventAreaStatisticsStack.Push(currentAreaStatistics);
                    currentAreaStatistics = EventAreaStatisticss.GetEventAreaStatistics(currentAreaStatistics.ParentCode);
                }
            }
            else
            {
                if (rootAreaStatistics != null)
                {
                    eventAreaStatisticss = rootAreaStatistics.Childs;
                }
                else
                    eventAreaStatisticss = EventAreaStatisticss.GetRootEventAreaStatisticss();
            }
            return eventAreaStatisticss;
        }


        #endregion

        #endregion

        #region 收藏
        /// <summary>
        /// 收藏活动
        /// </summary>
        public ActionResult CreateFavorite(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedSetEvent", EventApplicationID)));
            }

            Favorite favorite = new Favorite();
            favorite.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            favorite.DisplayName = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);
            favorite.ItemID = eventID;

            favorite.PrivacyStatus = (PrivacyStatuses)Request.Form.GetInt("PrivacyStatus", (int)PrivacyStatuses.Public);
            if (CurrentUser != null && !FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).IsFavorite(CurrentUser.UserID, eventID))
                FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).CreateFavorite(favorite);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AlreadyAddAttention", EventApplicationID));
            return RedirectToAction("ShowEvent", new RouteValueDictionary { { "eventID", eventID } });
        }
        /// <summary>
        /// 删除收藏
        /// </summary>
        public ActionResult DeleteFavorite(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindNeedSetEvent", EventApplicationID)));
            }
            //throw new Exception("没有找到相关活动");
            bool isFavorite = FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).IsFavorite(CurrentUser.UserID, eventID);
            if (isFavorite == false)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_AlreadyCancelAttention", EventApplicationID));
                return RedirectToAction("ShowEvent", new RouteValueDictionary { { "eventID", eventID } });
            }

            FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).DeleteFavorite(CurrentUser.UserID, eventID);
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_AlreadyCancelAttention", EventApplicationID));

            return RedirectToAction("ShowEvent", new RouteValueDictionary { { "eventID", eventID } });
        }

        #endregion

        #region 活动成员、报名、邀请

        #region 页面

        /// <summary>
        /// 邀请好友加入活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EventInvitation(int eventID, bool? isCreating)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Message_NoFindEvent", EventApplicationID)));
                // throw new Exception("没有找到相关活动");
            }

            SetPageTitle(GetResourceString("Action_InvitationFriend"), eventThread.EventName);

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_InvitationFriend"), string.Empty);
            #endregion

            if (!Authorize(eventThread, PermissionItemKeys.Instance().InviteFriendJoinEvent()))
                return new EmptyResult();

            if (CurrentUser != null && (CurrentUser.IsContentAdministrator || EventMembers.IsManager(CurrentUser.UserID, eventID, false)))
                ViewData["IsManagerOrAdministrator"] = true;

            ViewData["EventThread"] = eventThread;
            ViewData["IsCreating"] = isCreating.HasValue ? isCreating.Value : false;
            return View("Pages/Events/EventInvitation.aspx", "Masters/Event.master");
        }

        /// <summary>
        /// 管理活动成员列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageEventMembers(int eventID)
        {
            if (eventID <= 0)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindManageEvent", EventApplicationID)));
            }
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindManageEvent", EventApplicationID)));
            }

            SetPageTitle(GetResourceString("Title_ManageEventMemberList", EventApplicationID), eventThread.EventName);

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Action_ManageEventMember", EventApplicationID), string.Empty);
            #endregion

            if (!Authorize(eventThread, PermissionItemKeys.Instance().ManageEventMembers()))
                return new EmptyResult();

            ViewData["EventThread"] = eventThread;
            return View("Pages/Events/ManageEventMembers.aspx", "Masters/Event.master");
        }
        /// <summary>
        /// 显示活动成员列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListEventMembers(int eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEventMembers()))
                return new EmptyResult();

            SetPageTitle(GetResourceString("Title_EventMemberList", EventApplicationID), eventThread.EventName);

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EventMember", EventApplicationID), string.Empty);
            #endregion

            return View("Pages/Events/ListEventMembers.aspx", "Masters/Event.master");
        }

        /// <summary>
        /// 活动成员列表
        /// </summary>      
        public ActionResult Controls_ListEventMembers(int eventID, int? pageIndex)
        {
            string displayNameKeyWord = Request.QueryString.GetString("displayNameKeyWord", string.Empty);
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            PagingDataSet<EventMember> members = EventMembers.GetEventMembers(displayNameKeyWord, eventID, true, 20, pageIndex ?? 1, false);
            ViewData["SiteName"] = SiteSettingsManager.GetSiteSettings().SiteName;
            return PartialView("Controls/Events/ListEventMembers.ascx", members);
        }
        /// <summary>
        /// 关注的人列表
        /// </summary>  
        public ActionResult Controls_ListEventAttentionMembers(int eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            PagingDataSet<Favorite> favouritemembers = FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).GetFavoritesOfItem(eventID, PrivacyStatusesForDisplay.Public, pageIndex ?? 1, false);
            ViewData["SiteName"] = SiteSettingsManager.GetSiteSettings().SiteName;
            return PartialView("Controls/Events/ListEventAttentionMembers.ascx", favouritemembers);
        }
        #endregion

        #region 控件
        /// <summary>
        /// 已邀请过的好友列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_EventInvitationOutBox(int eventID, int? pageIndex)
        {
            IList<Invitation> invitations = Invitations.GetInvitationsOfRelativeItem(eventID, ApplicationIDs.Instance().Event());

            Dictionary<int, PersonUser> personUsers = new Dictionary<int, PersonUser>();
            foreach (Invitation invitation in invitations)
            {
                User user = Users.GetUser(invitation.ReceiverUserID, false);
                PersonUser pu = user as PersonUser;
                if (pu == null)
                    continue;
                personUsers[pu.UserID] = pu;
            }

            ViewData["personUsers"] = personUsers;
            ViewData["EventID"] = eventID;
            return PartialView("Controls/Events/EventInvitationOutBox.ascx", invitations);
        }

        /// <summary>
        /// 显示活动成员列表控件
        /// </summary>
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageEventMembers(int eventID, bool? isApproved, int? pageIndex)
        {
            string keyWord = Request.QueryString.GetString("keyWord", string.Empty);

            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindEventID", EventApplicationID), eventID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ManageEventMembers()))
                return new EmptyResult();

            PagingDataSet<EventMember> members = EventMembers.GetEventMembers(keyWord, eventID, isApproved, 10, pageIndex ?? 1, true);
            ViewData["IsApproved"] = isApproved;
            ViewData["EventID"] = eventID;
            ViewData["UserID"] = eventThread.UserID;
            return PartialView("Controls/Events/ManageEventMembers.ascx", members);
        }

        /// <summary>
        /// 显示未处理的活动邀请成员
        /// </summary>
        public ActionResult Control_NotReplyMembers(int eventID)
        {

            //IList<Invitation> invitations = Invitations.GetInvitationsOfRelativeItem(eventID);

            //Dictionary<int, PersonUser> personUsers = new Dictionary<int, PersonUser>();
            //foreach (Invitation invitation in invitations)
            //{
            //    User user = Users.GetUser(invitation.ReceiverUserID, false);
            //    if (user != null && user.UserType == UserTypes.PersonUser)
            //    {
            //        PersonUser pu = user as PersonUser;
            //        if (pu != null)
            //        {
            //            personUsers[pu.UserID] = pu;
            //        }
            //    }
            //}
            ViewData["EventID"] = eventID;

            return PartialView("Controls/Events/MemberNotReply.ascx");
        }
        /// <summary>
        /// 显示活动报名页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EventSignUp(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }
            string statusMessageContent = string.Empty;
            if (!EventPermission.ValidateSignUp(eventThread, CurrentUser, ref statusMessageContent) && TempData["StatusMessageType"] == null)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = statusMessageContent;
            }

            EventMember member = EventMembers.GetEventMember(CurrentUser != null ? CurrentUser.UserID : 0, eventID, false);//如果是匿名用户将会返回空
            if (CurrentUser != null && member != null)
            {
                ViewData["StatusMessageType"] = (int)StatusMessageType.Hint;
                ViewData["StatusMessageContent"] = GetResourceString("Message_AlreadySignUpNotAgainSignUp", EventApplicationID);
            }
            if (CurrentUser != null)
            {
                member = EventMembers.GetLastSignUpMember(CurrentUser != null ? CurrentUser.UserID : 0);
                if (member != null)
                    ViewData["member"] = member;
            }
            ViewData["SurplusCount"] = eventThread.SurplusCount;
            ViewData["EventID"] = eventID;
            return PartialView("Controls/Events/EventSignUp.ascx");
        }
        #endregion

        #region 操作
        /// <summary>
        /// 更新活动成员的是否为管理员属性
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult UpdateEventMemberIsManager(int memberID, bool isManager, bool? isApproved, int? pageIndex)
        {
            EventMember member = EventMembers.GetEventMember(memberID, true);
            if (member == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindMemberID", EventApplicationID), memberID)));

            }
            EventThread eventThread = EventThreads.GetEvent(member.EventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (member.UserID == eventThread.UserID && !isManager)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NotAllowCancelEventCreatorAdministratorDegree", EventApplicationID);
                return RedirectToAction("Control_ManageEventMembers", "ChannelEvent", new RouteValueDictionary { { "eventID", member.EventID }, { "isApproved", isApproved }, { "pageIndex", pageIndex } });
            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ManageEventMembers()))
                return new EmptyResult();
            //清除该活动成员权限集合缓存
            EventPermission.RemoveEventPermissionCache(member.UserID, eventThread.EventID);
            EventMembers.UpdateEventMemberIsManager(memberID, isManager);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            if (isManager)
                TempData["StatusMessageContent"] = GetResourceString("Message_SetAdministratorOperationSuccess", EventApplicationID);
            else
                TempData["StatusMessageContent"] = GetResourceString("Message_CancelAdministratorOperationSuccess", EventApplicationID);
            return RedirectToAction("Control_ManageEventMembers", "ChannelEvent", new RouteValueDictionary { { "eventID", member.EventID }, { "isApproved", isApproved }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 处理删除活动成员
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEventMember(int memberID, bool? isApproved, int? pageIndex)
        {
            EventMember member = EventMembers.GetEventMember(memberID, true);
            if (member == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindMemberID", EventApplicationID), memberID)));

            }
            EventThread eventThread = EventThreads.GetEvent(member.EventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (member.UserID == eventThread.UserID)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_NotAllowDeleteEventCreator", EventApplicationID);
                return RedirectToAction("Control_ManageEventMembers", "ChannelEvent", new RouteValueDictionary { { "eventID", member.EventID }, { "isApproved", isApproved }, { "pageIndex", pageIndex } });
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().ManageEventMembers()))
                return new EmptyResult();
            //清除该活动成员权限集合缓存
            EventPermission.RemoveEventPermissionCache(member.UserID, eventThread.EventID);
            EventMembers.DeleteEventMember(memberID);
            //TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            //TempData["StatusMessageContent"] = GetResourceString("Message_DeleteEventMemberOperationSuccess", EventApplicationID);

            return RedirectToAction("Control_ManageEventMembers", "ChannelEvent", new RouteValueDictionary { { "eventID", member.EventID }, { "isApproved", isApproved }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 处理批准活动成员
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ApproveEventMember(int memberID, bool? isApproved, int? pageIndex)
        {
            EventMember member = EventMembers.GetEventMember(memberID, true);
            if (member == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindMemberID", EventApplicationID), memberID)));
            }
            EventThread eventThread = EventThreads.GetEvent(member.EventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ManageEventMembers()))
                return new EmptyResult();
            //清除该活动成员权限集合缓存
            EventPermission.RemoveEventPermissionCache(member.UserID, eventThread.EventID);
            EventMembers.UpdateEventMemberIsApproved(eventThread.EventID, memberID, true);
            //TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            //TempData["StatusMessageContent"] = GetResourceString("Message_ApproveEventMemberOperationSuccess", EventApplicationID);
            return RedirectToAction("Control_ManageEventMembers", "ChannelEvent", new RouteValueDictionary { { "eventID", member.EventID }, { "isApproved", isApproved }, { "pageIndex", pageIndex } });
        }
        /// <summary>
        /// 处理注册用户报名参加活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult RegisteredUserSignUp(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            string statusMessageContent = string.Empty;
            if (!EventPermission.ValidateSignUp(eventThread, CurrentUser, ref statusMessageContent))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = statusMessageContent;
                return RedirectToAction("Control_EventSignUp", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }

            if (Request.Form.GetInt("bringCount", 0) + 1 > eventThread.SurplusCount && eventThread.SurplusCount > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "您携带的人超限了！";
                return RedirectToAction("Control_EventSignUp", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }

            EventMember member = new EventMember();
            member.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            member.DisplayName = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);
            member.EventID = eventID;
            PersonUser pu = CurrentUser as PersonUser;
            if (pu != null)
            {
                member.Address = pu.PersonAddress;
                member.Company = pu.CompanyName;
                member.Gender = pu.Gender;
                member.Job = pu.Title;
            }
            member.BringCount = Request.Form.GetInt("bringCount", 0);
            member.Email = Request.Form.GetString("email", string.Empty).Trim();
            member.Moblie = Request.Form.GetString("moblie", string.Empty).Trim();
            member.Telephone = Request.Form.GetString("telephone", string.Empty).Trim();
            member.RequestBody = Request.Form.GetString("body", string.Empty).Trim();
            member.Remark = Request.Form.GetString("Remark", string.Empty).Trim();
            member.IsApproved = !eventThread.IsAuditingSigner;
            member.IsManager = false;
            EventMembers.CreateEventMember(member);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            if (member.IsApproved)
                TempData["StatusMessageContent"] = GetResourceString("Message_SuccessJoinEvent", EventApplicationID);
            else
                TempData["StatusMessageContent"] = GetResourceString("Message_SignUpApplicationWaitAdministratorApprove", EventApplicationID);
            return RedirectToAction("Control_EventSignUp", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
        }

        /// <summary>
        /// 处理匿名用户报名参加活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult AnonymousSignUp(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            string statusMessageContent = string.Empty;
            if (!EventPermission.ValidateSignUp(eventThread, CurrentUser, ref statusMessageContent))
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = statusMessageContent;
                return RedirectToAction("Control_EventSignUp", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }
            if (Request.Form.GetInt("eventbringCount", 0) + 1 > eventThread.SurplusCount && eventThread.SurplusCount > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "您携带的人超限了！";
                return RedirectToAction("Control_EventSignUp", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }
            EventMember member = new EventMember();
            member.UserID = 0;
            member.EventID = eventID;
            member.Address = Request.Form.GetString("Address", string.Empty).Trim();
            member.Company = Request.Form.GetString("Company", string.Empty).Trim();
            member.DisplayName = Request.Form.GetString("DisplayName", string.Empty).Trim();
            member.Email = Request.Form.GetString("Email", string.Empty).Trim();
            member.Gender = (Genders)Request.Form.GetInt("Gender", 0);
            member.Job = Request.Form.GetString("Job", string.Empty).Trim();
            member.Moblie = Request.Form.GetString("Moblie", string.Empty).Trim();
            member.BringCount = Request.Form.GetInt("eventbringCount", 0);
            member.Telephone = Request.Form.GetString("Telephone", string.Empty).Trim();
            member.RequestBody = Request.Form.GetString("RequestBody", string.Empty).Trim();
            member.Remark = Request.Form.GetString("Remark", string.Empty).Trim();
            member.IsApproved = !eventThread.IsAuditingSigner;
            member.IsManager = false;
            EventMembers.CreateEventMember(member);

            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            if (member.IsApproved)
                TempData["StatusMessageContent"] = GetResourceString("Message_SuccessJoinEvent", EventApplicationID);
            else
                TempData["StatusMessageContent"] = GetResourceString("Message_SignUpApplicationWaitAdministratorApprove", EventApplicationID);
            return RedirectToAction("Control_EventSignUp", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
        }

        /// <summary>
        /// 处理发送活动邀请
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateJoinEventInvitation(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindManageEvent", EventApplicationID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().InviteFriendJoinEvent()))
                return new EmptyResult();

            string[] receivers;
            string receiverIDs = Request.Form.GetString("receiverIDs", string.Empty);
            if (!string.IsNullOrEmpty(receiverIDs))
            {
                //收件人集合
                receivers = receiverIDs.Split(',');
                //每个消息最多允许500个接收人
                if (receivers.Length > 100)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_SelectAddresseeExceedMaxLimit", EventApplicationID);
                    TempData["body"] = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("body", string.Empty).Trim(), true);
                    return RedirectToAction("EventInvitation", new RouteValueDictionary { { "eventID", eventID } });
                }
                //发送成功的接收人集合
                List<string> successReceivers = new List<string>();
                //重复发送的接受人集合
                List<string> duplicateReceivers = new List<string>();
                //为每个收件人建立邀请
                for (int i = 0; i < receivers.Length; i++)
                {
                    //查找收件人
                    int receiversID;
                    if (int.TryParse(receivers[i], out receiversID))
                    {
                        User user = Users.GetUser(int.Parse(receivers[i]), false);
                        if (user == null)
                            continue;

                        //判断该收件人是否接受当前登录用户的邀请
                        if (CurrentUser != null && (Privacys.Validate(user.UserID, CurrentUser, PrivacyItemKeys.Instance().Action_ReceiveJoinEventInvitation()) && !EventMembers.IsMember(user.UserID, eventID, true)))
                        {
                            Invitation invitation = new Invitation();
                            invitation.Subject = string.Format(GetResourceString("Description_Pattern_InviteJoinEvent", EventApplicationID), SiteUrls.Instance().ShowEvent(eventID), eventThread.EventName);
                            invitation.Body = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("body", string.Empty).Trim(), true);
                            invitation.SenderUserID = CurrentUser.UserID;
                            invitation.Sender = CurrentUser.DisplayName;
                            invitation.ApplicationID = ApplicationIDs.Instance().Event();

                            invitation.ReceiverUserID = user.UserID;
                            invitation.Receiver = user.DisplayName;

                            invitation.RelativeItemID = eventID;
                            invitation.RelativeItemName = eventThread.EventName;

                            CreateInvitationStatuses status = Invitations.CreateInvitation(invitation);
                            if (status == CreateInvitationStatuses.Duplicate)
                                duplicateReceivers.Add(user.DisplayName);
                            else
                                successReceivers.Add(user.DisplayName);
                        }
                    }
                }
                //如果有成功发送的消息则显示提示信息
                if (successReceivers.Count > 0)
                {
                    string strReceivers = string.Empty;
                    foreach (string dispalyname in successReceivers)
                    {
                        strReceivers += "," + dispalyname;
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_EventInviteSendSuccess", EventApplicationID), strReceivers.Substring(1));
                    string strDuplicateReceivers = string.Empty;
                    if (duplicateReceivers.Count > 0)
                    {
                        foreach (string dispalyname in duplicateReceivers)
                        {
                            strDuplicateReceivers += "," + dispalyname;
                        }
                        TempData["StatusMessageContent"] += string.Format(GetResourceString("Message_Pattern_InviteAlreadySendNotAgainSend", EventApplicationID), strDuplicateReceivers.Substring(1));
                    }
                }
                else if (duplicateReceivers.Count > 0)
                {
                    string strDuplicateReceivers = string.Empty;
                    foreach (string dispalyname in duplicateReceivers)
                    {
                        strDuplicateReceivers += "," + dispalyname;
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Hint;
                    TempData["StatusMessageContent"] += string.Format(GetResourceString("Message_Pattern_InviteAlreadySendNotAgainSend", EventApplicationID), strDuplicateReceivers.Substring(1));
                    TempData["body"] = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("body", string.Empty).Trim(), true);
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_InputFriendRefuseInvite", EventApplicationID);
                    TempData["body"] = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("body", string.Empty).Trim(), true);
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = GetResourceString("Message_InputFriendIsNull", EventApplicationID);
                TempData["body"] = Formatter.FormatMultiLinePlainTextForStorage(Request.Form.GetString("body", string.Empty).Trim(), true);
            }
            return RedirectToAction("EventInvitation", new RouteValueDictionary { { "eventID", eventID } });
        }

        /// <summary>
        /// 从被邀请人列表中删除一条邀请
        /// </summary>
        public ActionResult DeleteOutBoxInvitation(int invitationID, int eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_NoFindEvent", EventApplicationID));
                return RedirectToAction("Control_EventInvitationOutBox", new RouteValueDictionary { { "eventID", eventID }, { "pageIndex", pageIndex } });
            }
            Invitation invitation = Invitations.GetInvitation(invitationID);
            if (invitation == null)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_NoFindNeedDeleteInvitation", EventApplicationID));
                return RedirectToAction("Control_EventInvitationOutBox", new RouteValueDictionary { { "eventID", eventID }, { "pageIndex", pageIndex } });
            }
            if (invitation.SenderUserID != CurrentUser.UserID && CurrentUser.IsContentAdministrator)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_NotDeleteInvitationPopedom", EventApplicationID));
                return RedirectToAction("Control_EventInvitationOutBox", new RouteValueDictionary { { "eventID", eventID }, { "pageIndex", pageIndex } });
            }
            try
            {
                Invitations.DeleteInvitationFromRelativeItem(invitationID, eventID, ApplicationIDs.Instance().Event());
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_DeleteInvitationOperationSuccess", EventApplicationID));
            }
            catch (Exception)
            {
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, GetResourceString("Message_DeleteInvitationOperationFailed", EventApplicationID));
            }
            return RedirectToAction("Control_EventInvitationOutBox", new RouteValueDictionary { { "eventID", eventID }, { "pageIndex", pageIndex } });
        }

        #endregion

        #endregion

        #region 活动留言

        #region 控件
        /// <summary>
        /// 活动留言列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListEventPosts(int eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }
            PagingDataSet<EventPost> posts = new PagingDataSet<EventPost>();
            if (CurrentUser != null && (eventThread.UserID == CurrentUser.UserID || CurrentUser.IsAdministrator || CurrentUser.IsContentAdministrator))
            {
                posts = EventPosts.GetEventPosts(eventID, 10, pageIndex ?? 1, true);
                ViewData["IsAdministrator"] = true;
            }
            else
                posts = EventPosts.GetEventPosts(eventID, 10, pageIndex ?? 1, false);
            ViewData["EventID"] = eventID;
            if (CurrentUser != null && EventMembers.IsManager(CurrentUser.UserID, eventID, false))
                ViewData["IsManager"] = true;

            return PartialView("Controls/Events/ListEventPosts.ascx", posts);
        }
        #endregion

        #region 操作
        /// <summary>
        /// 处理发表活动留言
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateEventPost(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().CreateEventPost()))
                return new EmptyResult();

            string body = Request.Form.GetString("postBody", string.Empty).Trim();
            int parentID = Request.Form.GetInt("parentID", 0);

            EventPost post = new EventPost();
            post.UserID = CurrentUser != null ? CurrentUser.UserID : 0; ;
            post.ParentID = parentID;
            post.Author = CurrentUser != null ? SPBHtmlHelper.Instance().UserDisplayName(CurrentUser) : post.Author;
            post.Body = body;
            post.EventID = eventID;



            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                //检查验证码是否录入正确
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (!string.IsNullOrEmpty(verifyCode))
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["postBody"] = post.Body;
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    return RedirectToAction("Control_ListEventPosts", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
                }
            }

            if (post.ParentID > 0)
            {
                EventPost parentPost = EventPosts.GetEventPost(post.ParentID, false);
                if (parentPost != null)
                {
                    post.ParentAuthor = parentPost.Author;
                    post.ParentBody = parentPost.Body;
                    post.ParentUserID = parentPost.UserID;
                    post.ParentPostDate = parentPost.PostDate;


                    if (post.Body.StartsWith(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author), StringComparison.CurrentCultureIgnoreCase))
                    {
                        if (post.Body.Length > string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author).Length)
                        {
                            post.Body = post.Body.Substring(string.Format("{0}{1}：", GetResourceString("Common_Reply"), parentPost.Author).Length);
                        }
                        else
                        {
                            TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                            TempData["StatusMessageContent"] = "留言内容不能为空！";
                            TempData["Body"] = body;
                            TempData["ParentID"] = parentID;
                            return RedirectToAction("Control_ListEventPosts", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
                        }
                    }
                }
            }

            try
            {
                #region 检查发帖时间间隔,防灌水
                Globals.CheckPostDate();
                #endregion

                EventPosts.CreateEventPost(post);
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Event());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ListEventPosts", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
            }

            if (post.PostID > 0)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                if (post.AuditingStatus == AuditingStatuses.Success)
                    TempData["StatusMessageContent"] = GetResourceString("Message_LeaveMessageSuccess", EventApplicationID);
                else
                    TempData["StatusMessageContent"] = GetResourceString("Message_LeaveMessageSuccessNeedAdministratorAuditingDisplay", EventApplicationID);
            }
            return RedirectToAction("Control_ListEventPosts", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID } });
        }

        /// <summary>
        /// 处理删除活动留言
        /// </summary>
        /// <param name="postID">留言ID</param>
        /// <param name="eventID">活动ID</param>
        /// <param name="pageIndex">当前页号</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEventPost(int postID, int eventID, int pageIndex)
        {
            EventPost eventPost = EventPosts.GetEventPost(postID, false);
            if (eventPost == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }

            if (!EventPermission.ValidateEventPost(eventPost, CurrentUser))
            {
                if (Request.IsAuthenticated)
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                else
                    Response.Redirect(SiteUrls.Instance().Login(true), true);
                return new EmptyResult();
            }
            EventPosts.DeleteEventPost(postID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_DeleteSucecssful", EventApplicationID);
            return RedirectToAction("Control_ListEventPosts", "ChannelEvent", new RouteValueDictionary { { "eventID", eventID }, { "pageIndex", pageIndex } });
        }
        #endregion

        #endregion

        #region 活动照片

        #region 页面
        /// <summary>
        /// 显示活动照片
        /// </summary>
        /// <param name="AttachmentID">活动照片ID</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ShowEventPhoto(int attachmentID)
        {
            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, false);
            if (eventPhoto == null)
                return Redirect(SiteUrls.Instance().Error("您访问的活动相片不存在！"));

            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();


            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EventPhoto", EventApplicationID), SPBUrlHelper.Action("ListEventPhotos", "ChannelEvent", new RouteValueDictionary { { "eventID", eventThread.EventID } }));
            BreadCrumb.AddBreadCrumbItem(eventPhoto.FriendlyFileName, string.Empty);
            #endregion

            ViewData["PageIndex"] = EventPhotos.GetPageIndex(eventThread.EventID, eventPhoto.AttachmentID, CurrentUser != null && CurrentUser.IsContentAdministrator);
            ViewData["EventID"] = eventThread.EventID;

            return View("Pages/Events/ShowEventPhoto.aspx", "Masters/Event.master");
        }
        /// <summary>
        /// 显示活动图片列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ListEventPhotos(int? eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID ?? 0, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();

            SetPageTitle(string.Format(GetResourceString("Title_Pattern_PhotoList", EventApplicationID), eventThread.EventName));

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EventPhoto", EventApplicationID), string.Empty);
            #endregion

            ViewData["EventID"] = eventID;
            ViewData["PageIndex"] = pageIndex.HasValue ? pageIndex.Value : 1;
            return View("Pages/Events/ListEventPhotos.aspx", "Masters/Event.master");
        }
        /// <summary>
        /// 显示上传活动图片页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult UploadEventPhotos(int eventID)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().UploadEventPhoto()))
                return new EmptyResult();

            SetPageTitle(string.Format(GetResourceString("Title_Pattern_UpLoadPhoto", EventApplicationID), eventThread.EventName));

            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_UpLoadPhoto", EventApplicationID), string.Empty);
            #endregion

            ViewData["EventID"] = eventID;
            ViewData["EventAttachmentSettings"] = EventConfiguration.Instance().AttachmentSettings;
            string extensions = EventConfiguration.Instance().AttachmentSettings.Extensions;
            ViewData["Extensions"] = extensions.Insert(0, ",").Replace(",", ";*.").Substring(1);
            return View("Pages/Events/UploadEventPhotos.aspx", "Masters/Event.master");
        }
        /// <summary>
        /// 显示编辑多图片信息页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult EditEventPhotos(int eventID, int number)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().UploadEventPhoto()))
                return new EmptyResult();

            SetPageTitle(string.Format(GetResourceString("Title_Pattern_EditUpLoadPhoto", EventApplicationID), eventThread.EventName));
            #region 面包屑
            ShowCategoryBreadCrumb(eventThread.SiteCategoryID);
            BreadCrumb.AddBreadCrumbItem(StringUtils.Trim(eventThread.EventName, 8), SiteUrls.Instance().ShowEvent(eventThread.EventID));
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_EditPhoto", EventApplicationID), string.Empty);
            #endregion
            BlockPagingID blockPagingID = EventDataProvider.Instance().GetEventPhotoIDs(eventID, CurrentUser != null ? CurrentUser.UserID : 0, null, number, number, 1);
            List<EventPhoto> eventPhotos = new List<EventPhoto>();
            foreach (int photoID in blockPagingID.GetAll())
            {
                EventPhoto eventPhoto = EventPhotos.GetEventPhoto(photoID, true);
                if (eventPhoto != null)
                    eventPhotos.Add(eventPhoto);
            }
            if (CurrentUser != null && EventMembers.IsManager(CurrentUser.UserID, eventID, true) || CurrentUser.IsContentAdministrator)
                ViewData["IsEventManager"] = true;
            return View("Pages/Events/EditEventPhotos.aspx", "Masters/Event.master", eventPhotos);
        }

        #endregion

        #region 控件
        /// <summary>
        /// 显示最新前100张中随机抽取的照片列表
        /// </summary>
        /// <param name="eventID"></param>
        /// <param name="displayItemCount">需要显示的个数</param>
        public ActionResult Control_RandomPhotos(int? displayItemCount)
        {
            IList<EventPhoto> photos = EventPhotos.GetRandomLastestPhotos(displayItemCount.HasValue ? displayItemCount.Value : 10);

            return PartialView("Controls/Events/RandomPhotos.ascx", photos);
        }

        /// <summary>
        /// 显示单个图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ShowEventPhoto(int eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }
            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();
            PagingDataSet<EventPhoto> pds = EventPhotos.GetEventPhotos(eventID, true, 1, pageIndex ?? 1);
            if (pds.Records != null && pds.Records.Count > 0)
            {
                if (EventPermission.ValidateEventPhoto(pds.Records[0], CurrentUser))
                {
                    ViewData["CanUpdateDeleteThisEventPhoto"] = true;
                }

                #region 处理审核
                int auditingStatusValue = (int)Applications.GetApplication(ApplicationIDs.Instance().Event()).AuditingStatusForPublicDisplay;
                if ((int)pds.Records[0].AuditingStatus < auditingStatusValue)
                {
                    if (pds.Records[0].AuditingStatus == AuditingStatuses.Fail)
                        ViewData["errorMessage"] = string.Format(GetResourceString("Title_Pattern_Point_EventPhotoAuditingFail", ApplicationIDs.Instance().Event()), eventThread.EventName, pds.Records[0].FriendlyFileName);
                    else
                        ViewData["errorMessage"] = "活动相片正在审核";
                    ViewData["errorAuditingStatus"] = true;
                }
                #endregion
            }



            ViewData["EventID"] = eventThread.EventID;
            ViewData["EventName"] = eventThread.EventName;

            return PartialView("Controls/Events/ShowEventPhoto.ascx", pds);
        }
        /// <summary>
        /// 显示图片列表控件
        /// </summary>
        /// <remarks>用于显示某个活动的图片列表</remarks>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListEventPhotos(int eventID, int? pageIndex)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();
            ViewData["EventName"] = eventThread.EventName;
            ViewData["EventID"] = eventThread.EventID;
            PagingDataSet<EventPhoto> photos = EventPhotos.GetEventPhotos(eventID, null, 20, pageIndex ?? 1);

            return PartialView("Controls/Events/ListEventPhotos.ascx", photos);
        }
        /// <summary>
        /// 显示活动照片编辑模式框
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_EditEventPhoto(int attachmentID)
        {
            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, true);
            if (eventPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEventPhoto", EventApplicationID)));
            }
            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            if (!EventPermission.ValidateEventPhoto(eventPhoto, CurrentUser))
            {
                if (Request.IsAuthenticated)
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                else
                    Response.Redirect(SiteUrls.Instance().Login(true), true);
                return new EmptyResult();
            }
            if (eventThread.FeaturedPhotoID == attachmentID)
                ViewData["IsFeaturedPhoto"] = true;
            return PartialView("Controls/Events/EditEventPhoto.ascx", eventPhoto);
        }

        #endregion

        #region 操作
        /// <summary>
        /// 更新单个活动照片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateEventPhoto(int attachmentID)
        {
            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, true);
            if (eventPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEventPhoto", EventApplicationID)));
            }
            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            if (!EventPermission.ValidateEventPhoto(eventPhoto, CurrentUser))
            {
                if (Request.IsAuthenticated)
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                else
                    Response.Redirect(SiteUrls.Instance().Login(true), true);
                return new EmptyResult();
            }

            string friendlyFileName = Request.Form.GetString("friendlyFileName", string.Empty).Trim();
            string description = Request.Form.GetString("description", string.Empty).Trim();
            if (!string.IsNullOrEmpty(friendlyFileName))
                eventPhoto.FriendlyFileName = friendlyFileName;

            eventPhoto.Description = description;
            EventPhotos.UpdateEventPhoto(eventPhoto);

            if (EventMembers.IsManager(CurrentUser.UserID, eventThread.EventID, true) || CurrentUser.IsContentAdministrator)
            {
                int featuredPhotoID = Request.Form.GetInt("FeaturedPhoto", 0);
                if (featuredPhotoID > 0)
                {
                    eventThread.FeaturedPhotoID = featuredPhotoID;
                    EventThreads.UpdateEvent(eventThread);
                }
            }
            TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Success, GetResourceString("Message_UpdateEventPhotoSuccess", EventApplicationID));
            return RedirectToAction("Control_EditEventPhoto", new RouteValueDictionary { { "attachmentID", eventPhoto.AttachmentID } });
        }
        /// <summary>
        /// 更新多图片信息
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateEventPhotos(int eventID, int number)
        {
            EventThread eventThread = EventThreads.GetEvent(eventID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));

            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().UploadEventPhoto()))
                return new EmptyResult();
            PagingDataSet<EventPhoto> pds = EventPhotos.GetEventPhotos(eventID, CurrentUser != null ? CurrentUser.UserID : 0, null, number, 1);
            IList<EventPhoto> eventPhotos = pds.Records;
            string auditingMessage = string.Empty;
            foreach (EventPhoto eventPhoto in eventPhotos)
            {
                string friendlyFileName = Request.Form.GetString(string.Format("friendlyFileName{0}", eventPhoto.AttachmentID), string.Empty).Trim();
                string description = Request.Form.GetString(string.Format("description{0}", eventPhoto.AttachmentID), string.Empty).Trim();
                if (!string.IsNullOrEmpty(friendlyFileName))
                    eventPhoto.FriendlyFileName = friendlyFileName;

                eventPhoto.Description = description;
                EventPhotos.UpdateEventPhoto(eventPhoto);

            }
            if (EventMembers.IsManager(CurrentUser != null ? CurrentUser.UserID : 0, eventID, true) || CurrentUser.IsContentAdministrator)
            {
                int featuredPhotoID = Request.Form.GetInt("FeaturedPhoto", 0);
                if (featuredPhotoID > 0)
                {
                    eventThread.FeaturedPhotoID = featuredPhotoID;
                    EventThreads.UpdateEvent(eventThread);
                }
            }
            if (!Auditings.IsInAllowScope(eventPhotos[0].AuditingStatus, ApplicationIDs.Instance().Event()))
            {
                auditingMessage = "，您上传的图片需要通过审核后才可以显示！";
            }
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = GetResourceString("Message_BatchUpdateEventPhotoSuccess", EventApplicationID) + auditingMessage; ;
            return RedirectToAction("ListEventPhotos", new RouteValueDictionary { { "eventID", eventID } });
        }

        /// <summary>
        /// 删除活动图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEventPhoto(int attachmentID)
        {
            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, true);
            if (eventPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEventPhoto", EventApplicationID)));
            }
            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_NoFindEvent", EventApplicationID)));
            }

            if (!EventPermission.ValidateEventPhoto(eventPhoto, CurrentUser))
            {
                if (Request.IsAuthenticated)
                    throw new SPBException(ExceptionTypes.Instance().AccessDenied());
                else
                    Response.Redirect(SiteUrls.Instance().Login(true), true);
                return new EmptyResult();
            }
            string photoName = eventPhoto.FriendlyFileName;
            EventPhotos.DeleteEventPhoto(eventPhoto.AttachmentID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = string.Format(GetResourceString("Message_Pattern_SuccessDeletePhoto", EventApplicationID), photoName);

            return RedirectToAction("ListEventPhotos", new RouteValueDictionary { { "eventID", eventThread.EventID } });
        }

        #endregion

        #endregion

        #region 活动照片评论
        /// <summary>
        /// 活动照片评论列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ListEventPhotoComments(int attachmentID, int? pageIndex)
        {
            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, false);
            if (eventPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindPhoto", EventApplicationID), attachmentID)));
            }

            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindEvent", EventApplicationID), eventPhoto.AssociateID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().ViewEvent()))
                return new EmptyResult();

            PagingDataSet<EventPhotoComment> pds = EventPhotoComments.GetEventPhotoComments(attachmentID, pageIndex ?? 1, EventPermission.Validate(eventThread, CurrentUser, PermissionItemKeys.Instance().ManageEventPhotoComments()));
            ViewData["eventID"] = eventThread.EventID;
            ViewData["attachmentID"] = attachmentID;
            ViewData["hasManagePermission"] = EventPermission.Validate(eventThread, CurrentUser, PermissionItemKeys.Instance().ManageEventPhotoComments());
            ViewData["hasCreatePermission"] = EventPermission.Validate(eventThread, CurrentUser, PermissionItemKeys.Instance().CreateEventPhotoComment());
            return PartialView("Controls/Events/ListEventPhotoComments.ascx", pds);
        }
        /// <summary>
        /// 创建活动照片评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult CreateEventPhotoComment(int attachmentID, int? pageIndex)
        {
            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, false);
            if (eventPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindPhoto", EventApplicationID), attachmentID)));
            }

            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindEvent", EventApplicationID), eventPhoto.AssociateID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().CreateEventPhotoComment()))
                return new EmptyResult();

            string author = Request.Form.GetString("author", string.Empty);
            string body = Request.Form.GetString("body", string.Empty);
            SiteSettings settings = SiteSettingsManager.GetSiteSettings();
            if (settings.VerifyCodeOption == VerifyCodeOption.Enabled || CurrentUser == null && settings.VerifyCodeOption == VerifyCodeOption.Anonymous)
            {
                //检查验证码是否录入正确
                string verifyCodeFromCookie = Globals.GetVerifyCodeFromCookie(this.HttpContext);
                string verifyCode = Request.Form.GetString("verifyCode", string.Empty);
                if (!string.IsNullOrEmpty(verifyCode))
                    verifyCode = verifyCode.Trim();

                if (!verifyCode.Equals(verifyCodeFromCookie, StringComparison.CurrentCultureIgnoreCase))
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = GetResourceString("Message_IdentifyingCodeError");
                    TempData["author"] = author;
                    TempData["body"] = body;
                    return RedirectToAction("Control_ListEventPhotoComments", new RouteValueDictionary { { "attachmentID", attachmentID }, { "pageIndex", pageIndex } });
                }
            }

            EventPhotoComment eventPhotoComment = new EventPhotoComment();
            eventPhotoComment.AttachmentID = attachmentID;
            eventPhotoComment.Author = SPBHtmlHelper.Instance().UserDisplayName(CurrentUser);// Request.Form.GetString("author", SPBHtmlHelper.Instance().UserDisplayName(CurrentUser));
            eventPhotoComment.UserID = CurrentUser != null ? CurrentUser.UserID : 0;
            eventPhotoComment.ParentID = Request.Form.GetInt("parentID", 0);
            eventPhotoComment.Body = Request.Form.GetString("body", string.Empty);
            try
            {
                #region 检查发帖时间间隔,防灌水
                Globals.CheckPostDate();
                #endregion

                EventPhotoComments.CreateEventPhotoComment(eventPhotoComment);
                if (eventPhotoComment.CommentID > 0)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    if (eventPhotoComment.AuditingStatus == AuditingStatuses.Success)
                        TempData["StatusMessageContent"] = GetResourceString("Message_LeaveMessageSuccess", EventApplicationID);
                    else
                        TempData["StatusMessageContent"] = GetResourceString("Message_LeaveMessageSuccessNeedAdministratorAuditingDisplay", EventApplicationID);
                }
            }
            catch (SPBException e)
            {
                ExceptionMessage exceptionMessage = ResourceManager.GetExceptionMessage(e.ExceptionType, ApplicationIDs.Instance().Event());
                if (exceptionMessage != null)
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = exceptionMessage.Title + "，" + exceptionMessage.Body;
                }
                return RedirectToAction("Control_ListEventPhotoComments", new RouteValueDictionary { { "attachmentID", attachmentID }, { "pageIndex", pageIndex } });
            }

            return RedirectToAction("Control_ListEventPhotoComments", new RouteValueDictionary { { "attachmentID", attachmentID }, { "pageIndex", pageIndex } });
        }

        /// <summary>
        /// 删除活动照片评论
        /// </summary>
        public ActionResult DeleteEventPhotoComment(int eventID, int commentID, int? pageIndex)
        {
            EventPhotoComment photoComment = EventPhotoComments.GetEventPhotoComment(commentID, EventPermission.Validate(eventID, CurrentUser, PermissionItemKeys.Instance().ManageEventPhotoComments()));
            if (photoComment == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindEventPhotoComment", EventApplicationID), commentID)));
            }

            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(photoComment.AttachmentID, false);
            if (eventPhoto == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindPhoto", EventApplicationID), photoComment.AttachmentID)));
            }

            EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, false);
            if (eventThread == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_NoFindEvent", EventApplicationID), eventPhoto.AssociateID)));
            }

            if (!Authorize(eventThread, PermissionItemKeys.Instance().ManageEventPhotoComments()) && photoComment.UserID != CurrentUser.UserID)
                return new EmptyResult();
            EventPhotoComments.DeleteEventPhotoComment(commentID);
            return RedirectToAction("Control_ListEventPhotoComments", new RouteValueDictionary { { "attachmentID", photoComment.AttachmentID }, { "pageIndex", pageIndex } });
        }

        #endregion

        #region 活动地区选择
        /// <summary>
        /// 返回某活动地区统计下的所有子地区统计列表,使用json格式传输
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public JsonResult GetArea_StatisticsNavigations(string parentCode)
        {
            List<EventAreaStatistics> list = null;
            if (!string.IsNullOrEmpty(parentCode))
            {
                EventAreaStatistics parentArea = EventAreaStatisticss.GetEventAreaStatistics(parentCode);
                list = parentArea.Childs;
            }
            else
            {
                EventConfiguration config = EventConfiguration.Instance();
                if (!string.IsNullOrEmpty(config.RootAreaCode))
                {
                    EventAreaStatistics rootArea = EventAreaStatisticss.GetEventAreaStatistics(config.RootAreaCode);
                    list = rootArea.Childs;
                }
                else
                    list = EventAreaStatisticss.GetRootEventAreaStatisticss();
            }
            if (list == null)
                list = new List<EventAreaStatistics>();
            List<StatisticsNavigation> navs = new List<StatisticsNavigation>();
            foreach (EventAreaStatistics statistics in list)
            {
                StatisticsNavigation navigation = new StatisticsNavigation();
                navigation.ID = statistics.AreaCode.ToString();
                navigation.Name = statistics.GetMultilingualAreaName(CurrentUserLanguage);
                navigation.Url = SiteUrls.Instance().ListEventsByAreaCode(statistics.AreaCode);
                navigation.Count = statistics.CumulateTotalCount;
                navs.Add(navigation);
            }
            return Json(navs, JsonRequestBehavior.AllowGet);
        }


        #endregion

        /// <summary>
        /// 返回某站点类别统计下的所有子类别统计列表,使用json格式传输
        /// </summary>
        /// <param name="parentID">父类别ID</param>
        public JsonResult GetEventSiteCategories_StatisticsNavigations(int? parentID)
        {
            List<SiteCategory> list = null;
            if (parentID.HasValue && parentID.Value > 0)
                list = SiteCategories.Instance(ApplicationIDs.Instance().Event()).GetAllChilds(parentID.Value, false);
            else
                list = SiteCategories.Instance(ApplicationIDs.Instance().Event()).GetRoots(false);
            List<StatisticsNavigation> navs = new List<StatisticsNavigation>();
            foreach (SiteCategory siteCategory in list)
            {
                StatisticsNavigation navigation = new StatisticsNavigation();
                navigation.ID = siteCategory.CategoryID.ToString();
                navigation.Name = siteCategory.GetMultilingualCategoryName(CurrentUserLanguage);
                navigation.Url = SiteUrls.Instance().ListEventsByCategoryID(siteCategory.CategoryID);
                navigation.Count = siteCategory.CumulateItemCount;
                navs.Add(navigation);
            }
            return Json(navs, JsonRequestBehavior.AllowGet);
        }

        #region Help methods

        /// <summary>
        /// 活动权限验证
        /// </summary>
        /// <param name="permissionItemKey">PermissionItemKey</param>
        private bool Authorize(EventThread eventThread, string permissionItemKey)
        {
            string exceptionType = ExceptionTypes.Instance().AccessDenied();
            bool isAllowed = false;
            if (!EventPermission.Validate(eventThread, CurrentUser, permissionItemKey, ref exceptionType))
            {
                if (permissionItemKey == PermissionItemKeys.Instance().ViewEvent() || permissionItemKey == PermissionItemKeys.Instance().CreateEventPost())
                {
                    string token = Request.QueryString.GetString("token", string.Empty);
                    if (!string.IsNullOrEmpty(token))
                    {
                        int ticketEventID = 0;
                        int ticketUserID = 0;
                        EventThreads.DecryptInvitationTicket(token, out ticketEventID, out ticketUserID);
                        if (eventThread.EventID == ticketEventID && CurrentUser.UserID == ticketUserID)
                        {
                            isAllowed = true;
                        }
                    }
                }
            }
            else
                isAllowed = true;

            if (!isAllowed)
            {
                if (Request.IsAuthenticated)
                    Response.Redirect(SiteUrls.Instance().Error("没有浏览当前活动的权限!"));
                else
                    Response.Redirect(SiteUrls.Instance().Login(true), true);
            }
            return isAllowed;
        }


        /// <summary>
        /// 从表单中获取新数据更新活动实体
        /// </summary>
        private EventThread PopulateEventThreadFromRequest(EventThread eventThread)
        {
            eventThread.EventName = Request.Form.GetString("eventName", string.Empty).Trim();
            if (eventThread.EventID <= 0)
                eventThread.IsPublic = Request.Form.GetInt("isPublic", 1) == 1 ? true : false;
            eventThread.AreaCode = Request.Form.GetString("areaCode", string.Empty);
            if (eventThread.IsPublic)
                eventThread.SiteCategoryID = Request.Form.GetInt("siteCategoryID", eventThread.SiteCategoryID);
            eventThread.IsWholeDay = Request.Form.GetBool("isWholeDay", false);
            eventThread.FeaturedPhotoID = Request.Form.GetInt("featuredPhotoID", 0);

            try
            {
                if (eventThread.IsWholeDay)
                    eventThread.StartTime = DateTime.Parse(Request.Form.GetString("startTimeIsNotWholeDay", string.Empty));
                else
                    eventThread.StartTime = DateTime.Parse(Request.Form.GetString("startTime", string.Empty));

            }
            catch
            {
                eventThread.StartTime = new DateTime(DateTime.Now.Year, DateTime.Now.Month, DateTime.Now.Day, 8, 0, 0);
            }
            try
            {
                if (eventThread.IsWholeDay)
                {
                    eventThread.EndTime = DateTime.Parse(Request.Form.GetString("endTimeIsNotWholeDay", string.Empty));
                    if (eventThread.EndTime == eventThread.StartTime)
                        eventThread.EndTime = eventThread.EndTime.AddDays(1).AddSeconds(-1);
                }
                else
                    eventThread.EndTime = DateTime.Parse(Request.Form.GetString("endTime", string.Empty));
            }
            catch
            {
                eventThread.EndTime = new DateTime(DateTime.Now.Year, DateTime.Now.Month, DateTime.Now.Day, 18, 0, 0);
            }
            try
            {
                eventThread.SignEndTime = DateTime.Parse(Request.Form.GetString("signEndTime", string.Empty));
            }
            catch
            {
                eventThread.SignEndTime = new DateTime(DateTime.Now.Year, DateTime.Now.Month, DateTime.Now.Day, 8, 0, 0);
            }
            eventThread.BodyForDataTransmission = Request.Form.GetString("body", string.Empty).Trim();
            eventThread.Location = Request.Form.GetString("location", string.Empty).Trim();
            eventThread.Moblie = Request.Form.GetString("moblie", string.Empty).Trim();
            eventThread.Email = Request.Form.GetString("email", string.Empty).Trim();
            eventThread.Telephone = Request.Form.GetString("telephone", string.Empty).Trim();
            return eventThread;
        }

        /// <summary>
        /// 生成活动类别面包屑
        /// </summary>
        private void ShowCategoryBreadCrumb(int siteCategoryID)
        {
            BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_Event", EventApplicationID), SiteUrls.Instance().ChannelEventHome());
            //BreadCrumb.AddBreadCrumbItem(GetResourceString("Title_CategoryBrowse", EventApplicationID), SPBUrlHelper.Action("ListEventsByCategory", "ChannelEvent"));
            //Stack<SiteCategory> stack = new Stack<SiteCategory>();
            //SiteCategory currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(siteCategoryID, false);
            //while (currentCategory != null)
            //{
            //    stack.Push(currentCategory);
            //    currentCategory = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(currentCategory.ParentID, false);
            //}
            //while (stack.Count > 0)
            //{
            //    currentCategory = stack.Pop();
            //    BreadCrumb.AddBreadCrumbItem(currentCategory.GetMultilingualCategoryName(CurrentUserLanguage), SiteUrls.Instance().ListEventsByCategoryID(currentCategory.CategoryID));
            //}
        }

        /// <summary>
        /// 设置活动详细显示页面的meta标签
        /// </summary>
        private void SetPageMeta(EventThread thread)
        {
            string metaKeyWords = string.Empty;
            string metaDescription = string.Empty;

            //站点分类
            if (thread.SiteCategoryID > 0)
            {
                SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(thread.SiteCategoryID, false);
                if (category != null)
                    metaKeyWords = string.Format("{0},{1}", metaKeyWords, category.GetMultilingualCategoryName(CurrentUserLanguage));
            }

            if (!string.IsNullOrEmpty(thread.AreaCode))
            {
                metaKeyWords = string.Format("{0},{1}", metaKeyWords, Formatter.FormatArea(thread.AreaCode, true));
            }

            //标题
            metaKeyWords = string.Format("{0},{1}", metaKeyWords, thread.EventName);
            metaDescription = string.Format("{0},{1}", thread.EventName, thread.Excerpt);

            Header.AddMetaDescription(metaDescription);
            Header.AddMetaKeywords(metaKeyWords.Substring(1));
        }

        /// <summary>
        /// 是否具有管理活动权限
        /// </summary>
        protected bool HasManagePermission(int eventID)
        {
            return (CurrentUser != null && (EventPermission.Validate(eventID, CurrentUser, PermissionItemKeys.Instance().CreateEvent()) || CurrentUser.IsContentAdministrator || CurrentUser.IsAdministrator));
        }


        #endregion

        #region 更改活动审核状态

        public ActionResult updateEventAuditingStatuses(int eventID)
        {
            EventThreads.UpdateEventAuditingStatus(eventID, AuditingStatuses.Success);
            return RedirectToAction("ShowEvent", new RouteValueDictionary { { "eventID", eventID } });
        }

        #endregion

        #region 推荐活动
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.UsualObjectCollection)]
        public ActionResult Control_RecommendEventBlock(int commendTypeID, int? displayCount)
        {
            #region 推荐活动
            IList<RecommendedItem> pds = RecommendedItems.GetTopRecommendedItems(ApplicationIDs.Instance().Event(), commendTypeID, displayCount ?? 4);

            //List<EventThread> recommendedEvents = new List<EventThread>();
            //foreach (RecommendedItem recommendItem in recommendedClubItems)
            //{
            //    EventThread recommendedEvent = EventThreads.GetEvent(recommendItem.ItemID, true);
            //    if (recommendedEvent != null && recommendedEvent.IsPublic)
            //        recommendedEvents.Add(recommendedEvent);
            //}
            //ViewData["recommendedEvents"] = recommendedEvents;
            ViewData["RecommendationType"] = RecommendationTypes.GetRecommendationType(commendTypeID, false);
            #endregion
            return PartialView("Controls/Events/RecommendEventBlock.ascx", pds);
        }

        #endregion

        #region  向活动成员发送消息

        /// <summary>
        /// 显示群发消息页面
        /// </summary>
        public ActionResult MassMessages(int eventID)
        {
            ViewData["eventID"] = eventID;
            return PartialView("Controls/Events/MassMessages.ascx");
        }
        /// <summary>
        /// 处理群发消息
        /// </summary>
        public ActionResult MassMessageAction(int eventID)
        {
            string subject = Request.Form.GetString("subject", string.Empty);
            string body = Request.Form.GetString("body", string.Empty);
            TransportModes mode = (TransportModes)Request.Form.GetInt("transportMode", 0);
            UserPointItemRole role = null;
            string userPointRecordDescription = string.Empty;

            role = Points.GetUserPointItemRole(UserPointItemKeys.Instance().MassEventMessage());

            if (CurrentUser != null && CurrentUser.TradePoints < role.BasicPoints)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageData"] = "您的积分不足！";

                return RedirectToAction("MassMessages", new RouteValueDictionary { { "eventID", eventID } });
            }

            if (string.IsNullOrEmpty(subject) || string.IsNullOrEmpty(body))
            {
                TempData["subject"] = subject;
                TempData["body"] = body;
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageData"] = new StatusMessageData(StatusMessageType.Error, "标题不允许为空");
                return RedirectToAction("MassMessages", new RouteValueDictionary { { "eventID", eventID } });
            }
            Dictionary<int, int> members = EventMembers.GetUserID2MemberIDDictionaryOfEvent(eventID);
            List<UserIdentifier> users = new List<UserIdentifier>();
            foreach (int userID in members.Keys)
            {
                UserIdentifier user = Users.GetUserIdentifier(userID);
                if (user == null || user.UserID <= 0)
                    continue;
                users.Add(user);

            }
            try
            {
                SendMessage(subject, body, users, mode);



                List<UserPointRecord> records = Points.PopulateUserPointRecords(role, CurrentUser.UserID, role.GetMultilingualItemName(Globals.GetCurrentUserLanguage()), userPointRecordDescription);
                Points.CreateUserPointRecords(records);

                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageData"] = "群发消息成功";
            }
            catch
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageData"] = "群发消息失败";
            }
            return RedirectToAction("MassMessages", new RouteValueDictionary { { "eventID", eventID } });
        }
        /// <summary>
        /// 批量发送消息
        /// </summary>
        /// <param name="subject">标题</param>
        /// <param name="body">内容</param>
        /// <param name="receivers">接受人集合</param>
        /// <param name="mode">传输方式</param>
        private void SendMessage(string subject, string body, IList<UserIdentifier> receivers, TransportModes mode)
        {
            foreach (UserIdentifier receiver in receivers)
            {
                if (mode == TransportModes.PrivateMessage)
                {
                    PrivateMessage pm = new PrivateMessage();
                    pm.Subject = Formatter.FormatSingleLinePlainTextForStorage(subject);
                    pm.Body = Formatter.FormatMultiLinePlainTextForStorage(body, true);
                    pm.SenderUserID = CurrentUser.UserID;
                    pm.ReceiverUserID = receiver.UserID;
                    pm.Receiver = receiver.DisplayName;
                    pm.MessageType = PrivateMessageType.Serve;

                    PrivateMessages.SendMessage(pm);
                }
                else if (mode == TransportModes.Email)
                {
                    MailMessage mail = new MailMessage();
                    mail.Subject = subject;
                    mail.Body = body;
                    try
                    {
                        mail.From = new MailAddress(SiteSettingsManager.GetSiteSettings().AdminEmailAddress, Globals.SiteName);
                        mail.To.Add(new MailAddress(receiver.Email, receiver.DisplayName));
                    }
                    catch (Exception)
                    {
                        continue;
                    }
                    Emails.EnqueuEmail(mail);
                }
            }
        }
        #endregion

        #region EventAttachment
        /// <summary>
        /// 管理活动主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_UploadEventAttachment(int? threadID, string queueCompleteCallback)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            UploadAttachmentSettings settings = new UploadAttachmentSettings();
            EventAttachmentSettings attachmentSettings = (EventAttachmentSettings)EventAttachmentManager.Instance().Settings;
            string uploadPhotoTypes = EventAttachmentManager.Instance().Settings.Extensions;
            IList<string> imageExtensions = MimeTypeConfiguration.GetImageExtensions();
            StringBuilder newUploadPhotoTypes = new StringBuilder();
            string[] types = uploadPhotoTypes.Split(',');
            foreach (string type in types)
            {
                if (!string.IsNullOrEmpty(type))
                    newUploadPhotoTypes.AppendFormat("*.{0};", type);
            }
            settings.AssociateID = threadID ?? 0;
            settings.FileTypes = newUploadPhotoTypes.ToString();
            settings.UploadUrl = Globals.GetFullUrlInFileHandlerSite(EventAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Services/Event/UploadEventPhotos.ashx"));
            settings.FlashUrl = Globals.GetFullUrlInFileHandlerSite(EventAttachmentManager.Instance().StorageProviderName, WebUtils.ResolveUrl("~/Utility/UploadFiles/swfupload.swf"));
            settings.FileSizeLimit = attachmentSettings.MaxAttachmentSize.ToString();
            settings.QueueCompleteCallback = queueCompleteCallback;
            settings.IsImage = true;

            ViewData["UploadAttachmentSettings"] = settings;
            return View("Controls/UploadAttachmentInEditer.ascx", settings);
        }

        /// <summary>
        /// 管理活动主题附件
        /// </summary>
        /// <param name="userDomainName">用户名</param>
        [AcceptVerbs(HttpVerbs.Get)]
        [OutputCache(Location = OutputCacheLocation.None)]
        public ActionResult Control_ManageThreadAttachments(int? threadID)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            ManageThreadAttachmentsModel model = new ManageThreadAttachmentsModel();
            model.IsImage = true;

            if (threadID.HasValue && threadID.Value > 0)
            {
                EventThread eventThread = EventThreads.GetEvent(threadID.Value, true);
                if (eventThread == null)
                {
                    return Redirect(SiteUrls.Instance().Error404(GetResourceString("Exception_Pattern_VisitEventArticleNoFound", EventApplicationID)));
                }

                IList<SpaceBuilder.Common.Attachment> attachments = EventAttachmentManager.Instance().GetAttachmentsByAssociateID(threadID.Value, true);
                model.Attachments = attachments;
                model.UserFeaturedImageAttachmentID = eventThread.FeaturedPhotoID;
            }
            else
            {
                IList<SpaceBuilder.Common.Attachment> attachments = EventAttachmentManager.Instance().GetTemporaryAttachmentsOfUser(CurrentUser.UserID);
                model.Attachments = attachments;
            }
            model.FeaturePicAttachmentIDControlName = "featuredPhotoID";
            model.GetAttachmentUrl = SiteUrls.Instance().EventPhotoUrl;
            model.DeleteAttachmentUrl = SiteUrls.Instance().DeleteEventAttachmentUrl;
            model.ShowAttachmentThumbnail = SPBHtmlHelper.Instance().ShowEventAttachmentThumbnail;
            return PartialView("Controls/ManageThreadAttachmentsInEditer.ascx", model);
        }

        public ActionResult UserContentSelector_UpdateEventAttachments(int? threadID)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            return PartialView("Controls/Events/UserContentSelector_UpdateEventAttachments.ascx");
        }

        #region Manage

        /// <summary>
        /// 删除活动附件
        /// </summary>
        public ActionResult DeleteEventAttachment(int attachmentID)
        {
            if (CurrentUser == null)
                return new EmptyResult();

            EventPhoto eventPhoto = EventPhotos.GetEventPhoto(attachmentID, true);
            if (eventPhoto == null)
                return new EmptyResult();
            if (eventPhoto.AssociateID > 0)
            {
                EventThread eventThread = EventThreads.GetEvent(eventPhoto.AssociateID, true);
                if (eventThread == null || eventThread.UserID != eventPhoto.UserID)
                    return new EmptyResult();
            }

            if (CurrentUser.UserID != eventPhoto.UserID)
                return new EmptyResult();
            EventPhotos.DeleteEventPhoto(attachmentID);
            User attachmentOwner = Users.GetUser(eventPhoto.UserID, false);
            if (attachmentOwner == null)
                return new EmptyResult();

            return RedirectToAction("Control_ManageThreadAttachments", new RouteValueDictionary { { "threadID", eventPhoto.AssociateID } });
        }

        #endregion


        #endregion


    }

    /// <summary>
    /// 频道中的活动子菜单
    /// </summary>
    public enum ChannelEventSubMenu
    {
        /// <summary>
        /// 活动首页
        /// </summary>
        Home = 1,
        /// <summary>
        /// 搜索活动
        /// </summary>
        Search = 2,
        /// <summary>
        /// 我的活动
        /// </summary>
        MyEvents = 3,
        /// <summary>
        /// 活动回顾
        /// </summary>
        Review = 4
    }

}
