﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;


using SpaceBuilder.Web;
using System.Web.Routing;

namespace SpaceBuilder.Event.Controllers
{
    /// <summary>
    /// 后台管理活动的Controller
    /// </summary>
    [ManageAuthorize]
    [HttpCompress]
    public class ManageEventController : ManageControllerBase
    {
        #region 活动
        /// <summary>
        /// 显示管理活动页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageEvents(AuditingStatusesForDisplay? auditingStatusForDisplay)
        {
            SetPageTitle("管理活动");
            EventThreadFullQuery query = new EventThreadFullQuery();
            query.SortBy = EventThreadSortBy.LastModified;
            query.SortOrder = SortOrder.Descending;
            query.ConvertFromQueryString(Request);
            query.AuditingStatusForDisplay = auditingStatusForDisplay.HasValue ? auditingStatusForDisplay.Value : AuditingStatusesForDisplay.Pending;
            query.IsPublic = null;
            PagingDataSet<EventThread> pds = EventThreads.GetThreadsForSearch(query, 20, Request.QueryString.GetInt("PageIndex", 1));

            Dictionary<int, string> eventSiteCategories = new Dictionary<int, string>();
            foreach (EventThread thread in pds.Records)
            {
                SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(thread.SiteCategoryID, true);
                if (category != null)
                    eventSiteCategories[thread.EventID] = category.CategoryName;
                else
                    eventSiteCategories[thread.EventID] = string.Empty;
            }

            ViewData["EventSiteCategories"] = eventSiteCategories;

            ViewData["SiteCategoryDepth"] = EventConfiguration.Instance().SiteCategoryLevel - 1;
            ViewData["AuditingStatusForDisplay"] = auditingStatusForDisplay;

            return View("Pages/Events/ManageEvents.aspx", "Masters/Event.master", pds);
        }
        #region 操作
        /// <summary>
        /// 搜索活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchEvents(AuditingStatusesForDisplay auditingStatusForDisplay)
        {
            string eventName = Request.Form.GetString("EventName", string.Empty).Trim();
            int siteCategoryID = Request.Form.GetInt("SiteCategoryID", -1);
            string areaCode = Request.Form.GetString("AreaCode", string.Empty);
            int userID = Request.Form.GetInt("UserID", -1);
            int statusValue = Request.Form.GetInt("Status", -1);
            int sortByValue = Request.Form.GetInt("SortBy", 1);
            int sortOrderValue = Request.Form.GetInt("SortOrder", 1);
            return RedirectToAction("ManageEvents", new RouteValueDictionary { { "auditingStatusForDisplay", auditingStatusForDisplay }, { "Keyword", eventName }, { "siteCategoryID", siteCategoryID }, { "areaCode", areaCode }, { "userID", userID }, { "status", statusValue }, { "sortBy", sortByValue }, { "sortOrder", sortOrderValue } });
        }
        /// <summary>
        /// 设置活动的审核状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetEventAuditingStatus(bool isPass, string returnUrl)
        {
            string selectedIDsString = Request.Form["ManageEventsCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    List<int> eventIDs = new List<int>();
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int eventID = int.Parse(IDString);
                        if (eventID > 0)
                            eventIDs.Add(eventID);
                    }
                    if (isPass)
                        EventThreads.UpdateEventAuditingStatus(eventIDs, AuditingStatuses.Success);
                    else
                        EventThreads.UpdateEventAuditingStatus(eventIDs, AuditingStatuses.Fail);

                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "批量审核活动操作成功";
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "请选中要审核的活动";
                }
            }
            return Redirect(returnUrl);
        }
        /// <summary>
        /// 批量删除活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteEvents(string returnUrl)
        {
            string selectedIDsString = Request.Form["ManageEventsCheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int eventID = int.Parse(IDString);
                        if (eventID > 0)
                            EventThreads.DeleteEvent(eventID);
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "批量删除操作成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的活动";
            }
            return Redirect(returnUrl);
        }
        /// <summary>
        /// 删除单个活动
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEvent(int eventID, string returnUrl)
        {
            EventThread thread = EventThreads.GetEvent(eventID, true);
            if (thread == null)
                throw new ArgumentException(string.Format("不存在EventID为{0}的活动", eventID));
            try
            {
                EventThreads.DeleteEvent(eventID);
                TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                TempData["StatusMessageContent"] = "删除活动操作成功";
            }
            catch (Exception)
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "删除活动操作失败，请与管理员联系";
            }
            return Redirect(returnUrl);
        }
        #endregion

        #endregion

        #region 活动类别
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageEventCategories()
        {
            SetPageTitle("管理活动类别");
            ViewData["SiteCategoryDepth"] = EventConfiguration.Instance().SiteCategoryLevel - 1;
            return View("Pages/Events/ManageEventCategories.aspx", "Masters/Event.master");
        }
        #endregion

        #region 活动照片

        /// <summary>
        /// 管理活动照片页
        /// </summary>
        public ActionResult ManageEventPhotos(AuditingStatusesForDisplay? auditingStatusForDisplay, int? pageIndex)
        {
            SetPageTitle("管理活动照片");
            int userID = Request.QueryString.Get("userID", -1);
            string SubjectKeywords = Request.QueryString.GetString("subjectKeywords", string.Empty).Trim();
            PagingDataSet<EventPhoto> pds = EventPhotos.GetEventPhotosForAdmin(userID, SubjectKeywords, auditingStatusForDisplay ?? AuditingStatusesForDisplay.Pending, 10, pageIndex ?? 1);
            ViewData["AuditingStatusForDisplay"] = auditingStatusForDisplay.HasValue ? auditingStatusForDisplay : AuditingStatusesForDisplay.Pending;
            return View("Pages/Events/ManageEventPhotos.aspx", "Masters/Event.master", pds);
        }


        /// <summary>
        /// 删除单个图片
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEventPhoto(int attachmentID, string returnUrl)
        {
            EventPhoto photo = EventPhotos.GetEventPhoto(attachmentID, true);
            if (photo == null)
                throw new ArgumentException(string.Format("attachmentID为{0}的图片不存在", attachmentID));
            EventPhotos.DeleteEventPhoto(attachmentID);
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量删除图片
        /// </summary>
        /// <param name="returnUrl">返回的url</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteEventPhotos(string returnUrl)
        {
            string selectedAttachmentIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedAttachmentIDsString))
            {
                string[] selectedAttachmentIDsStringArray = selectedAttachmentIDsString.Split(',');
                if (selectedAttachmentIDsStringArray != null && selectedAttachmentIDsStringArray.Length > 0)
                {
                    int attachmentID;
                    foreach (string Id in selectedAttachmentIDsStringArray)
                    {
                        try
                        {
                            attachmentID = int.Parse(Id);
                            EventPhoto photo = EventPhotos.GetEventPhoto(attachmentID, true);
                            if (photo != null)
                                EventPhotos.DeleteEventPhoto(attachmentID);
                        }
                        catch { }
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "图片批量删除成功";
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的图片";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量更新图片状态
        /// </summary>
        /// <param name="returnUrl">返回url</param>
        /// <param name="auditingStatus">AuditingStatuses枚举</param>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateEventPhotoAuditingStatuses(string returnUrl, AuditingStatuses auditingStatus)
        {
            string selectedAttachmentIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedAttachmentIDsString))
            {
                string[] selectedAttachmentIDsStringArray = selectedAttachmentIDsString.Split(',');
                if (selectedAttachmentIDsStringArray != null && selectedAttachmentIDsStringArray.Length > 0)
                {
                    int attachmentID;
                    List<int> listAttachmentID = new List<int>();
                    foreach (string Id in selectedAttachmentIDsStringArray)
                    {
                        try
                        {
                            attachmentID = int.Parse(Id);
                            listAttachmentID.Add(attachmentID);
                        }
                        catch { }
                    }

                    try
                    {
                        EventPhotos.UpdateEventPhotoAuditingStatus(listAttachmentID, auditingStatus);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = "图片批量操作成功";
                    }
                    catch
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = "图片批量操作失败";
                    }
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的图片";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 搜索活动留言
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchEventPhotos(AuditingStatusesForDisplay? auditingStatusForDisplay)
        {
            SetPageTitle("搜索活动留言");
            string subjectKeywords = Request.Form.GetString("subjectKeywords", string.Empty).Trim();
            int userID = Request.Form.GetInt("userID", 0);
            return RedirectToAction("ManageEventPhotos", new RouteValueDictionary { { "auditingStatusForDisplay", auditingStatusForDisplay }, { "subjectKeywords", subjectKeywords }, { "userID", userID } });
        }

        #endregion

        #region 活动照片评论

        /// <summary>
        /// 管理活动照片评论
        /// </summary>       
        public ActionResult ManageEventPhotosComments(AuditingStatusesForDisplay? auditingStatusForDisplay, int? pageIndex)
        {
            SetPageTitle("管理活动照片评论");
            int userID = Request.QueryString.Get("userID", -1);
            string keyword = Request.QueryString.GetString("Keyword", string.Empty).Trim();
            ViewData["AuditingStatusForDisplay"] = auditingStatusForDisplay.HasValue ? auditingStatusForDisplay : AuditingStatusesForDisplay.Pending;
            int intStatus = Request.QueryString.GetInt("AuditingStatusForDisplay", -1);
            if (intStatus > 0)
                ViewData["AuditingStatusForDisplay"] = (AuditingStatusesForDisplay)intStatus;
            PagingDataSet<EventPhotoComment> pds = EventPhotoComments.GetEventPhotoCommentsForAdmin(userID, keyword, auditingStatusForDisplay ?? AuditingStatusesForDisplay.Pending, 30, pageIndex ?? 1);
            return View("Pages/Events/ManageEventPhotoComments.aspx", "Masters/Event.master", pds);
        }
        /// <summary>
        /// 删除照片评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEventPhotoComment(int commentID, string returnUrl)
        {
            EventPost post = EventPosts.GetEventPost(commentID, true);
            if (post == null)
                throw new ArgumentException(string.Format("commentID为{0}的照片评论不存在", commentID));
            EventPhotoComments.DeleteEventPhotoComment(commentID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除成功！";
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 批量删除照片评论
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteEventPhotoComments(string returnUrl)
        {
            string[] deleteCommentIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("CheckBoxGroup", string.Empty)))
            {
                deleteCommentIDs = Request.Form.GetString("CheckBoxGroup", string.Empty).Split(',');
                if (deleteCommentIDs != null && deleteCommentIDs.Length > 0)
                {
                    int id = 0;
                    foreach (string commnetID in deleteCommentIDs)
                    {
                        int.TryParse(commnetID, out id);
                        if (id > 0)

                            EventPhotoComments.DeleteEventPhotoComment(id);
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "照片评论批量删除成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的照片评论";
            }

            return Redirect(returnUrl);
        }
        /// <summary>
        /// 活动照片评论审核状态管理
        /// </summary>        
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SetEventPhotoCommentsAuditingStatus(bool isPass, string returnUrl)
        {
            string selectedIDsString = Request.Form["CheckBoxGroup"];
            if (!string.IsNullOrEmpty(selectedIDsString))
            {
                string[] selectedIDsStringArray = selectedIDsString.Split(',');
                if (selectedIDsStringArray != null && selectedIDsStringArray.Length > 0)
                {
                    List<int> CommentIDs = new List<int>();
                    foreach (string IDString in selectedIDsStringArray)
                    {
                        int CommentID = int.Parse(IDString);
                        if (CommentID > 0)
                            CommentIDs.Add(CommentID);
                    }
                    if (isPass)
                        EventPhotoComments.UpdateEventPhotoCommentAuditingStatus(CommentIDs, AuditingStatuses.Success);
                    else
                        EventPhotoComments.UpdateEventPhotoCommentAuditingStatus(CommentIDs, AuditingStatuses.Fail);

                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "批量审核活动照片评论操作成功";
                }
                else
                {
                    TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                    TempData["StatusMessageContent"] = "请选中要审核的活动照片评论";
                }
            }
            return Redirect(returnUrl);
        }

        #endregion

        #region 活动留言

        #region 页面

        /// <summary>
        /// 管理活动留言页
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult ManageEventPosts(AuditingStatusesForDisplay? auditingStatusForDisplay, int? pageIndex)
        {
            SetPageTitle("管理活动留言");
            int userID = Request.QueryString.Get("userID", -1);
            string bodyKeywords = Request.QueryString.Get("bodyKeywords", string.Empty).Trim();
            PagingDataSet<EventPost> pds = EventPosts.GetEventPostsForAdmin(userID, bodyKeywords, auditingStatusForDisplay ?? AuditingStatusesForDisplay.Pending, 15, pageIndex ?? 1);

            ViewData["auditingStatusForDisplay"] = auditingStatusForDisplay ?? AuditingStatusesForDisplay.Pending;
            return View("Pages/Events/ManageEventPosts.aspx", "Masters/Event.master", pds);
        }

        #endregion

        #region 操作

        /// <summary>
        /// 删除留言
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult DeleteEventPost(int postID, string returnUrl)
        {
            EventPost post = EventPosts.GetEventPost(postID, true);
            if (post == null)
                throw new ArgumentException(string.Format("postID为{0}的留言不存在", postID));
            EventPosts.DeleteEventPost(postID);
            TempData["StatusMessageType"] = (int)StatusMessageType.Success;
            TempData["StatusMessageContent"] = "删除成功！";
            return Redirect(returnUrl);
        }


        /// <summary>
        /// 批量删除留言
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult DeleteEventPosts(string returnUrl)
        {
            string[] deletePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("CheckBoxGroup", string.Empty)))
            {
                deletePostIDs = Request.Form.GetString("CheckBoxGroup", string.Empty).Split(',');
                if (deletePostIDs != null && deletePostIDs.Length > 0)
                {
                    int id = 0;
                    foreach (string postID in deletePostIDs)
                    {
                        int.TryParse(postID, out id);
                        if (id > 0)
                            EventPosts.DeleteEventPost(id);
                    }
                    TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                    TempData["StatusMessageContent"] = "留言批量删除成功";

                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要删除的留言";
            }

            return Redirect(returnUrl);
        }

        /// <summary>
        /// 更新审核状态
        /// </summary>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult UpdateEventPostAuditingStatuses(string returnUrl, AuditingStatuses auditingStatus)
        {
            string[] updatePostIDs = null;
            if (!string.IsNullOrEmpty(Request.Form.GetString("CheckBoxGroup", string.Empty)))
            {
                updatePostIDs = Request.Form.GetString("CheckBoxGroup", string.Empty).Split(',');
                if (updatePostIDs != null && updatePostIDs.Length > 0)
                {
                    List<int> list = new List<int>();
                    int id = 0;
                    foreach (string postID in updatePostIDs)
                    {
                        int.TryParse(postID, out id);
                        if (id > 0)
                            list.Add(id);
                    }


                    try
                    {
                        EventPosts.UpdateEventPostAuditingStatus(list, auditingStatus);
                        TempData["StatusMessageType"] = (int)StatusMessageType.Success;
                        TempData["StatusMessageContent"] = "审核状态批量操作成功";
                    }
                    catch
                    {
                        TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                        TempData["StatusMessageContent"] = "审核状态批量操作失败";
                    }
                }
            }
            else
            {
                TempData["StatusMessageType"] = (int)StatusMessageType.Error;
                TempData["StatusMessageContent"] = "请选中要操作的项";
            }
            return Redirect(returnUrl);
        }

        /// <summary>
        /// 搜索活动留言
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult SearchEventPosts(AuditingStatusesForDisplay? auditingStatusForDisplay)
        {
            SetPageTitle("搜索活动留言");
            string bodyKeywords = Request.Form.GetString("bodyKeywords", string.Empty).Trim();
            int userID = Request.Form.GetInt("userID", 0);
            return RedirectToAction("ManageEventPosts", new RouteValueDictionary { { "auditingStatusForDisplay", auditingStatusForDisplay }, { "bodyKeywords", bodyKeywords }, { "userID", userID } });
        }


        #endregion

        #endregion

        /// <summary>
        /// 获取活动统计数据
        /// </summary>

        public JsonResult GetManageableCounts()
        {
            Dictionary<EventManageableCounts, int> dictonary = EventThreads.GetManageableCounts();
            Application eventApplication = Applications.GetApplication(ApplicationIDs.Instance().Event());
            List<ManageableCount> list = new List<ManageableCount>();
            list.Add(new ManageableCount { Name = "新增活动", Value = eventApplication.CountInPast24Hours });
            if (dictonary.ContainsKey(EventManageableCounts.PendingThreadCount))
                list.Add(new ManageableCount { Name = "待审核活动", Value = dictonary[EventManageableCounts.PendingThreadCount], Url = SPBUrlHelper.Action("ManageEvents", "ManageEvent", new RouteValueDictionary { { "auditingStatusForDisplay", AuditingStatusesForDisplay.Pending } }) });
            if (dictonary.ContainsKey(EventManageableCounts.AgainThreadCount))
                list.Add(new ManageableCount { Name = "需再审核活动", Value = dictonary[EventManageableCounts.AgainThreadCount], Url = SPBUrlHelper.Action("ManageEvents", "ManageEvent", new RouteValueDictionary { { "auditingStatusForDisplay", AuditingStatusesForDisplay.Again } }) });
            if (dictonary.ContainsKey(EventManageableCounts.PendingPhotoCount))
                list.Add(new ManageableCount { Name = "待审核活动照片", Value = dictonary[EventManageableCounts.PendingPhotoCount], Url = SPBUrlHelper.Action("ManageEventPhotos", "ManageEvent", new RouteValueDictionary { { "auditingStatusForDisplay", AuditingStatusesForDisplay.Pending } }) });
            if (dictonary.ContainsKey(EventManageableCounts.AgainPhotoCount))
                list.Add(new ManageableCount { Name = "需再审核活动照片", Value = dictonary[EventManageableCounts.AgainPhotoCount], Url = SPBUrlHelper.Action("ManageEventPhotos", "ManageEvent", new RouteValueDictionary { { "auditingStatusForDisplay", AuditingStatusesForDisplay.Again } }) });

            if (dictonary.ContainsKey(EventManageableCounts.PendingPhotoCommentCount))
                list.Add(new ManageableCount { Name = "待审核活动照片评论", Value = dictonary[EventManageableCounts.PendingPhotoCommentCount], Url = SPBUrlHelper.Action("ManageEventPhotosComments", "ManageEvent", new RouteValueDictionary { { "auditingStatusForDisplay", AuditingStatusesForDisplay.Pending } }) });

            if (dictonary.ContainsKey(EventManageableCounts.PendingPostCount))
                list.Add(new ManageableCount { Name = "待审核活动留言", Value = dictonary[EventManageableCounts.PendingPostCount], Url = SPBUrlHelper.Action("ManageEventPosts", "ManageEvent", new RouteValueDictionary { { "auditingStatusForDisplay", AuditingStatusesForDisplay.Pending } }) });
            list.Add(new ManageableCount { Name = "总活动", Value = eventApplication.TotalCount });
            return Json(list, JsonRequestBehavior.AllowGet);

        }
    }

    public enum EventSubMenu
    {
        /// <summary>
        /// 管理活动
        /// </summary>
        ManageEvents = 2101,
        /// <summary>
        /// 管理活动照片
        /// </summary>
        ManageEventPhotos = 2102,
        /// <summary>
        /// 管理活动留言
        /// </summary>
        ManageEventPosts = 2103,

        /// <summary>
        /// 管理活动类别
        /// </summary>
        ManageEventCategories = 2106,
        //管理活动照片评论
        ManageEventPhotoComment = 2109
    }

}
