﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
// Copyright (c) Tunynet Inc. All rights reserved.
// </copyright> 
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SpaceBuilder.Common.Controllers;
using System.Web.Mvc;
using SpaceBuilder.Common;

namespace SpaceBuilder.Event.Controllers
{
    [UserDomainGuideInstallApplication(ApplicationID = 221)]
    [HttpCompress]
    public class UserDomainEventController : UserDomainControllerBase
    {
        private int EventApplicationID = ApplicationIDs.Instance().Event();
        /// <summary>
        /// 显示我的活动详细列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListMyEvents(string userDomainName, UserEventScopes? eventScope)
        {
            if (!eventScope.HasValue)
                eventScope = UserEventScopes.Joined;
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserDomainNotFind"), userDomainName)));
            };
            string genderName = GetResourceString("Navigation_MyEvent", EventApplicationID);
            string genderattentionName = GetResourceString("Navigation_MyAttentionEvent", EventApplicationID);
            if (CurrentUser != null && currentDomainUser.UserID != CurrentUser.UserID)
            {
                PersonUser pu = currentDomainUser as PersonUser;
                if (pu != null)
                {
                    genderName = GetResourceString("Navigation_HisEvent", EventApplicationID);
                    genderattentionName = GetResourceString("Navigation_HisAttentionEvent", EventApplicationID);
                    if (pu.Gender == Genders.Female)
                    {
                        genderName = GetResourceString("Navigation_HerEvent", EventApplicationID);
                        genderattentionName = GetResourceString("Navigation_HerAttentionEvent", EventApplicationID);
                    }
                }
            }
            SetPageTitle(genderName);
            ViewData["eventScope"] = eventScope;
            ViewData["CurrentDomainUser"] = currentDomainUser;
            ViewData["GenderName"] = genderName;
            ViewData["GenderAttentionName"] = genderattentionName;
            return View("Pages/Events/ListMyEvents.aspx", "Masters/Event.master");
        }

        /// <summary>
        /// 显示我的活动详细列表
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        [ActionOutputCache(ActionOutputCacheAttribute.CachingExpirationType.RelativelyStable)]
        public ActionResult Control_ListMyEvents(string userDomainName, UserEventScopes? eventScope, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            PagingDataSet<EventThread> pds = null;
            if (HasManagePermission(userDomainName))
                pds = EventThreads.GetMyThreads(currentDomainUser.UserID, eventScope ?? UserEventScopes.Joined, pageIndex ?? 1);
            else
                pds = EventThreads.GetThreads(currentDomainUser.UserID, eventScope ?? UserEventScopes.Joined, pageIndex ?? 1, false);

            Dictionary<int, string> eventSiteCategories = new Dictionary<int, string>();
            foreach (EventThread thread in pds.Records)
            {
                SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(thread.SiteCategoryID, false);
                if (category != null)
                    eventSiteCategories[thread.EventID] = category.CategoryName;
                else
                    eventSiteCategories[thread.EventID] = string.Empty;
            }
            if (IsUserDomainOwner(userDomainName))
                ViewData["IsUserDomainOwner"] = true;
            ViewData["eventScope"] = eventScope;
            ViewData["EventSiteCategories"] = eventSiteCategories;
            return PartialView("Controls/Events/ListMyEvents.ascx", pds);
        }
        /// <summary>
        /// 显示我的朋友的最新活动
        /// </summary>
        /// <param name="userDomainName"></param>
        /// <returns></returns>
        public ActionResult ListMyFriendEvents(string userDomainName)
        {
            #region 标题
            string title = GetResourceString("Title_FriendsEvent", ApplicationIDs.Instance().Event());
            SetPageTitle(userDomainName, title);
            #endregion
            return View("Pages/Events/ListMyFriendCreateEvents.aspx", "Masters/Event.master");
        }
        /// <summary>
        /// 显示我的朋友的最新活动
        /// </summary>
        public ActionResult Control_ListMyFriendCreateEvents(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
                throw new ArgumentException(string.Format(GetResourceString("Exception_Pattern_UserDomainNotFind"), userDomainName));

            PagingDataSet<EventThread> pds = null;
            pds = EventThreads.GetFriendThreads(currentDomainUser.UserID, pageIndex.Value);
            Dictionary<int, string> eventSiteCategories = new Dictionary<int, string>();
            foreach (EventThread thread in pds.Records)
            {
                SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(thread.SiteCategoryID, false);
                if (category != null)
                    eventSiteCategories[thread.EventID] = category.CategoryName;
                else
                    eventSiteCategories[thread.EventID] = string.Empty;
            }
            ViewData["eventSiteCategories"] = eventSiteCategories;
            return PartialView("Controls/Events/ListMyFriendCreateEvents.ascx", pds);
        }

        /// <summary>
        /// 关注的活动列表
        /// </summary>
        [UserDomainPrivacy(IsAuthentication = true)]
        public ActionResult ListFavoriteEvents(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
            {
                return Redirect(SiteUrls.Instance().Error404(string.Format(GetResourceString("Exception_Pattern_UserDomainNotFind"), userDomainName)));
            }
            string genderName = GetResourceString("Navigation_MyEvent", EventApplicationID);
            string genderattentionName = GetResourceString("Navigation_MyAttentionEvent", EventApplicationID);
            if (CurrentUser != null && currentDomainUser.UserID != CurrentUser.UserID)
            {
                PersonUser pu = currentDomainUser as PersonUser;
                if (pu != null)
                {
                    genderName = GetResourceString("Navigation_HisEvent", EventApplicationID);
                    genderattentionName = GetResourceString("Navigation_HisAttentionEvent", EventApplicationID);
                    if (pu.Gender == Genders.Female)
                    {
                        genderName = GetResourceString("Navigation_HerEvent", EventApplicationID);
                        genderattentionName = GetResourceString("Navigation_HerAttentionEvent", EventApplicationID);
                    }
                }
            }
            SetPageTitle(genderName);
            ViewData["CurrentDomainUser"] = currentDomainUser;
            ViewData["GenderName"] = genderName;
            ViewData["GenderAttentionName"] = genderattentionName;
            return View("Pages/Events/ListFavoriteEvents.aspx", "Masters/Event.master");
        }
        /// <summary>
        /// 我关注的活动列表用户控件
        /// </summary>
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Control_ListFavoriteEvents(string userDomainName, int? pageIndex)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            if (currentDomainUser == null)
            {
                throw new ArgumentException(string.Format(GetResourceString("Exception_Pattern_UserDomainNotFind"), userDomainName));
            }
            PrivacyStatusesForDisplay? privacyStatusForDisplay = PrivacyStatusesForDisplay.Public;

            if (CurrentUser != null && Friends.IsFriend(CurrentUser.UserID, currentDomainUser.UserID))
                privacyStatusForDisplay = PrivacyStatusesForDisplay.OnlyFriend_GreaterThanOrEqual;

            if (CurrentUser != null && (currentDomainUser.UserID == CurrentUser.UserID || CurrentUser.IsContentAdministrator))
                privacyStatusForDisplay = null;
            PagingDataSet<Favorite> pds = FavoriteManager.Instance(FavoriteItemKeys.Instance().EventThread()).GetFavoritesOfUser(currentDomainUser.UserID, privacyStatusForDisplay, pageIndex ?? 1, false);
            IList<Favorite> favorites = pds.Records;

            Dictionary<int, EventThread> events = new Dictionary<int, EventThread>();
            Dictionary<int, string> eventSiteCategories = new Dictionary<int, string>();
            foreach (Favorite favorite in favorites)
            {
                //if ((favorite.PrivacyStatus == PrivacyStatuses.Privacy || favorite.PrivacyStatus == PrivacyStatuses.OnlyFriend && (CurrentUser == null || CurrentUser != null && !Friends.IsFriend(currentDomainUser.UserID, CurrentUser.UserID))) && !IsUserDomainOwner(userDomainName))
                //{
                //    pds.Records.Remove(favorite);
                //}

                EventThread thread = EventThreads.GetEvent(favorite.ItemID, false);
                if (thread != null)
                {
                    events[favorite.ItemID] = thread;
                    SiteCategory category = SiteCategories.Instance(ApplicationIDs.Instance().Event()).Get(thread.SiteCategoryID, false);
                    if (category != null)
                        eventSiteCategories[thread.EventID] = category.CategoryName;
                    else
                        eventSiteCategories[thread.EventID] = string.Empty;
                }
            }
            ViewData["eventThreads"] = events;
            ViewData["EventSiteCategories"] = eventSiteCategories;
            return View("Controls/Events/ListFavoriteEvents.ascx", pds);
        }

        /// <summary>
        /// 统计信息
        /// </summary>     
        public ActionResult Control_EventCountBlock(string userDomainName)
        {
            User currentDomainUser = GetCurrentDomainUser(userDomainName);
            EventUserData userdate = EventUserDatas.GetUserData(currentDomainUser.UserID, false);
            return PartialView("Controls/Events/EventCountBlock.ascx", userdate);
        }
    }


    /// <summary>
    /// 空间博客二级菜单
    /// </summary>
    public enum EventUserDomainSubMenus
    {
        /// <summary>
        /// 我的活动
        /// </summary>
        myEvents = 1,

        /// <summary>
        /// 朋友的活动
        /// </summary>
        friendEvents = 10,

        /// <summary>
        /// 我关注的活动
        /// </summary>
        myFavoriteEvents = 20,

        /// <summary>
        /// 站点中的活动
        /// </summary>
        siteEvents = 30




    }

}